#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
This module contains code generation tools for the ufc::form class.
"""

# Copyright (C) 2008-2009 Martin Sandve Alnes and Simula Resarch Laboratory
#
# This file is part of SyFi.
#
# SyFi is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#
# SyFi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with SyFi. If not, see <http://www.gnu.org/licenses/>.
#
# First added:  2008-08-13
# Last changed: 2009-03-05

from sfc.codegeneration.codeformatting import indent, CodeFormatter

class FormCG:
    "Code generator for an ufc::form."
    def __init__(self, formrep):
        self.formrep = formrep

    def hincludes(self):
        l = []
        return l

    def cincludes(self):
        l = []
        return l

    def generate_code_dict(self):
        form_vars = {
             "classname"                       : self.formrep.classname,
             "constructor"                     : indent(self.gen_constructor()),
             "constructor_arguments"           : indent(self.gen_constructor_arguments()),
             "initializer_list"                : indent(self.gen_initializer_list()),
             "destructor"                      : indent(self.gen_destructor()),
             "members"                         : indent(self.gen_members()),
             "rank"                            : indent(self.gen_rank()),
             "num_coefficients"                : indent(self.gen_num_coefficients()),
             "signature"                       : indent(self.gen_signature()),
             "create_dofmap"                   : indent(self.gen_create_dofmap()),
             "create_finite_element"           : indent(self.gen_create_finite_element()),
             "create_cell_integral"            : indent(self.gen_create_cell_integral()),
             "create_exterior_facet_integral"  : indent(self.gen_create_exterior_facet_integral()),
             "create_interior_facet_integral"  : indent(self.gen_create_interior_facet_integral()),
             "num_cell_domains"                : indent(self.gen_num_cell_domains()),
             "num_exterior_facet_domains"      : indent(self.gen_num_exterior_facet_domains()),
             "num_interior_facet_domains"      : indent(self.gen_num_interior_facet_domains()),
            }
        return form_vars
    
    def generate_support_code(self):
        return ""

    def gen_constructor(self):
        return ""

    def gen_constructor_arguments(self):
        return ""

    def gen_initializer_list(self):
        return ""

    def gen_destructor(self):
        return ""

    def gen_members(self):
        return ""

    def gen_rank(self):
        return 'return %d;'   % self.formrep.rank

    def gen_num_coefficients(self):
        return 'return %d;'   % self.formrep.num_coefficients

    def gen_signature(self):
        return 'return "%s";' % self.formrep.signature
    
    def _gen_num_domains(self, names):
        if names:
            n = 1 + max(list(names.keys()))
        else:
            n = 0
        return 'return %d;' % n

    def gen_num_cell_domains(self):
        return self._gen_num_domains(self.formrep.citg_names)

    def gen_num_exterior_facet_domains(self):
        return self._gen_num_domains(self.formrep.eitg_names)

    def gen_num_interior_facet_domains(self):
        return self._gen_num_domains(self.formrep.iitg_names)

    def _gen_create_switch(self, classnames, name, indices = None, return_null=False):
        "Utility function for creating factory functions."
        code = CodeFormatter()
        #if options.add_debug_code: # FIXME
        #    code.begin_debug()
        #    code += 'std::cout << "Creating object in %s, index is " << i << std::endl;' % name
        #    code.end_debug()
        if classnames:
            if isinstance(classnames, dict):
                assert indices is None
            else:
                if indices is None:
                    indices = range(len(classnames))
                classnames = dict((i,c) for (i,c) in zip(indices, classnames))

            code.begin_switch("i")
            for i, c in classnames.iteritems():
                code.begin_case(i)
                code += "return new %s();" % c
                code.end_case()
            code.end_switch()

        if return_null:
            code += 'return 0;'
        else:
            code += 'throw std::runtime_error("Invalid index in %s()");' % name
        return str(code)

    def gen_create_dofmap(self):
        return self._gen_create_switch(self.formrep.dm_names, "create_dofmap")

    def gen_create_finite_element(self):
        return self._gen_create_switch(self.formrep.fe_names, "create_finite_element")

    def gen_create_cell_integral(self):
        return self._gen_create_switch(self.formrep.citg_names, "create_cell_integral", return_null=True)

    def gen_create_exterior_facet_integral(self):
        return self._gen_create_switch(self.formrep.eitg_names, "create_exterior_facet_integral", return_null=True)

    def gen_create_interior_facet_integral(self):
        return self._gen_create_switch(self.formrep.iitg_names, "create_interior_facet_integral", return_null=True)

if __name__ == '__main__':
    print "form.py, no test implemented"

