///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __CAMERA_OBJECT_H
#define __CAMERA_OBJECT_H

#include <core/Core.h>
#include <core/scene/objects/AbstractCameraObject.h>
#include <core/scene/animation/controller/Controller.h>
#include <core/gui/properties/PropertiesEditor.h>
#include <mesh/tri/TriMesh.h>
#include "../StdObjects.h"

namespace StdObjects {

/******************************************************************************
* The default free camera scene object.
******************************************************************************/
class STDOBJECTS_DLLEXPORT CameraObject : public AbstractCameraObject
{
public:
	/// Default constructor.
	CameraObject(bool isLoading = false);

	// Property access methods:

	/// Returns the controller that controls the field of view of the camera.
	FloatController* fovController() const { return _fov; }

	/// Replaces the controller that controls the field of view of the camera.
	void setFovController(FloatController* ctrl) { _fov = ctrl; }

	// from AbstractCameraObject

	/// Fills in the missing fields of the camera view descriptor structure.
	/// The follwing fields of the CameraViewDescription structure must have been filled in before
	/// the method is called:
	///   - CameraViewDescription::aspectRatio (The aspect ratio (height/width) of the viewport)
	///   - CameraViewDescription::viewAffineTransformation (The world to view space transformation)
	///   - CameraViewDescription::znear (The distance to the bounding box of the scene in view space)
	///   - CameraViewDescription::zfar (The distance to the back side of the bounding box of the scene in view space)
	/// The validity interval field is reduced by this method to the time interval with a constant view.   
	virtual void getCameraDescription(TimeTicks time, CameraViewDescription& descriptor);

	// From SceneObject:

	/// Asks the object for its validity interval at the given time.
	virtual TimeInterval objectValidity(TimeTicks time);

	/// Makes the object render itself into the viewport. 
	/// The viewport transformation is already set up, when this method is called by the
	/// system. The object has to be rendered in the local object coordinate system.
	virtual void renderObject(TimeTicks time, ObjectNode* contextNode, Viewport* vp);

	/// Returns the bounding box of the object in local object coordinates. 
	virtual Box3 boundingBox(TimeTicks time, ObjectNode* contextNode);

	/// Do not show selection marker for cameras.
	virtual bool showSelectionMarker() { return false; }

public:

	Q_PROPERTY(FloatController* fovController READ fovController WRITE setFovController) 

protected:

	// From RefTarget:

	/// This method is called when a reference target changes. 
	virtual bool onRefTargetMessage(RefTarget* source, RefTargetMessage* msg) { 
		if(msg->type() == REFTARGET_CHANGED) {
			// Invalidate mesh cache when a parameter has changed.
			meshValidity.setEmpty();
		}
		return SceneObject::onRefTargetMessage(source, msg);
	}

	/// Is called when the value of a reference field of this RefMaker changes.
	virtual void onRefTargetReplaced(const PropertyFieldDescriptor& field, RefTarget* oldTarget, RefTarget* newTarget) {
		// Invalidate mesh cache when a parameter controller has been replaced.
		meshValidity.setEmpty();
        SceneObject::onRefTargetReplaced(field, oldTarget, newTarget);
	}
	
protected:

	/// Builds the icon mesh for the representation of this camera object in the viewports.
	void buildMesh(TimeTicks time);

	/// The validity interval of the cached camera icon mesh.
	TimeInterval meshValidity;

	/// The cached camera icon mesh that represents the camera in the viewports.
	TriMesh mesh;

	/// This controller stores the field of view of the camera.
	ReferenceField<FloatController> _fov;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(CameraObject)
	DECLARE_REFERENCE_FIELD(_fov)
};


/******************************************************************************
* A properties editor for the CameraObject class.
******************************************************************************/
class CameraObjectEditor : public PropertiesEditor
{
protected:
	
	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);
	
private:

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(CameraObjectEditor)
};

};

#endif // __CAMERA_OBJECT_H
