////////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file CustomAttributesContainer.h 
 * \brief Contains the definition of the Core::CustomAttributesContainer class. 
 */

#ifndef __OVITO_CUSTOM_ATTRIBUTES_CONTAINER_H
#define __OVITO_CUSTOM_ATTRIBUTES_CONTAINER_H

#include <core/Core.h>
#include "RefTarget.h"

namespace Core {

/**
 * \brief Holds a list of custom data objects associated with a RefMaker.
 *
 * \author Alexander Stukowski 
 * \sa RefMaker::customAttributes()
 */
class CORE_DLLEXPORT CustomAttributesContainer : public RefMaker
{
public:
	/// \brief Default constructor that constructs an empty container.
	/// \param isLoading Indicates whether the object is being deserialized.
	/// \param owner The RefMaker that owns this container.
	/// \note Plugin developers should not use this constructor. There is a one-to-one
	///       correspondence between the RefMaker and the CustomAttributesContainer class.
	///       The attribute container that belongs to a RefMaker can be accessed using the
	///       RefMaker::customAttributes() method.
	CustomAttributesContainer(bool isLoading = false, RefMaker* owner = NULL);

	/// \brief Inserts the an object into the attribute container. 
	/// \param data The object to be added to the container. 
	///        It will be saved along with the owner RefMaker object of the attribute container.
	///
	/// Data objects inserted into the container can be retrieved at a later time using
	/// the methods attributeOfType(), attributesOfType() and attribute().
	///
	/// \undoable
	void addAttribute(RefTarget* data);

	/// \brief Inserts the an object into the attribute container. 
	/// \param data The object to be added to the container. 
	///        It will be saved along with the owner RefMaker object of the attribute container.
	///
	/// This is the same method as above but takes a smart pointer instead of a raw pointer.
	/// \undoable
	void addAttribute(const RefTarget::SmartPtr& data) { addAttribute(data.get()); }
	
	/// \brief Removes the given object from the attribute container.
	/// \param data An object stored in the container that is to be removed from the container.
	///
	/// After this method returns the container will no longer
	/// hold a reference to the object \a data and it will be automatically deleted
	/// if there are no other references left to it.
	///
	/// \undoable
	void removeAttribute(RefTarget* data);

	/// \brief Retrieves an attribute data object from the container that is an instance of a specific class.
	/// \param type Specifies the type of object to retrieve. The returned object is either an instance of
	///             this class or any derived class.
	/// \return The first object stored in the container that matches the given class. 
	///         If there is no such attribute stored in this container then \c NULL is returned.
	///
	/// The returned object can safely be cast to the given class \a type.
	/// \sa attributes(), attributesOfType()
	RefTarget* attributeOfType(PluginClassDescriptor* type) const;

	/// \brief Retrieves an attribute data object from the container that is an instance of a specific class.
	/// \return The first object stored in the container that matches the class given by the template parameter T. 
	///         If there is no such attribute stored in this container then \c NULL is returned.
	///
	/// This is the templated version of the function above.
	template<class T>
	T* attributeOfType() const { return static_object_cast<T>(attributeOfType(PLUGINCLASSINFO(T))); }

	/// \brief Retrieves all attribute objects from the container that are an instance of a specific class.
	/// \param type Specifies the type of objects to retrieve. The returned objects are either instances of
	///             this class or any derived class.
	/// \return All objects stored in the container that match the given class. 
	///
	/// The returned objects can safely be cast to the given class \a type.
	/// \sa attributes(), attributeOfType()
	QVector<RefTarget*> attributesOfType(PluginClassDescriptor* type) const;

	/// \brief Returns all attribute data objects stord in this container.
	/// \return The stored objects.	
	const QVector<RefTarget*>& attributes() const { return _attributes.targets(); }
	
	/// \brief Returns the owner of this custom attribute container.
	/// \return The RefMaker whose custom attributes are stored in this container.
	RefMaker* owner() const { return _owner; }

protected:

	/***************************** Overriden methods *******************************/

	/// This method is called when a reference target changes. To be implemented by sub-classes.
	virtual bool onRefTargetMessage(RefTarget* source, RefTargetMessage* msg);

	/// Is called when the value of a reference field of this RefMaker changes.
	virtual void onRefTargetReplaced(const PropertyFieldDescriptor& field, RefTarget* oldTarget, RefTarget* newTarget) {
		OVITO_ASSERT(false);	// Should not happen.
	}

	/// Is called when a reference target has been added to a list reference field of this RefMaker.
	virtual void onRefTargetInserted(const PropertyFieldDescriptor& field, RefTarget* newTarget, int listIndex);

	/// Is called when a reference target has been removed from a list reference field of this RefMaker.
	virtual void onRefTargetRemoved(const PropertyFieldDescriptor& field, RefTarget* oldTarget, int listIndex);

	/// Saves the class' contents to the given stream. 
	virtual void saveToStream(ObjectSaveStream& stream);

	/// Loads the class' contents from the given stream. 
	virtual void loadFromStream(ObjectLoadStream& stream);

private:

	/// The RefMaker that owns this container.
	RefMaker* _owner;

	/// Contains the custom data objects.
	VectorReferenceField<RefTarget> _attributes;

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(CustomAttributesContainer)
	DECLARE_VECTOR_REFERENCE_FIELD(_attributes)
};

};

#endif // __OVITO_CUSTOM_ATTRIBUTES_CONTAINER_H
