//----------------------------------------------------------------------------
//
// License:  See top level LICENSE.txt file.
//
// Author:  David Burken
//
// Description:
//
// Command line application "img2rr" to build overviews.
//
//----------------------------------------------------------------------------
// $Id: img2rr.cpp 14814 2009-07-01 13:54:56Z dburken $
#include <iostream>
#include <iterator>

#include <tiff.h> /* for compression defines */
#include <ossim/ossimConfig.h>
#include <ossim/base/ossimConstants.h>
#include <ossim/base/ossimRefPtr.h>
#include <ossim/base/ossimProperty.h>
#include <ossim/base/ossimStringProperty.h>
#include <ossim/imaging/ossimTiffOverviewBuilder.h>
#include <ossim/imaging/ossimOverviewBuilderFactoryRegistry.h>
#include <ossim/imaging/ossimCibCadrgTileSource.h>
#include <ossim/imaging/ossimImageHandler.h>
#include <ossim/imaging/ossimImageHandlerRegistry.h>
#include <ossim/base/ossimFilename.h>
#include <ossim/init/ossimInit.h>
#include <ossim/base/ossimArgumentParser.h>
#include <ossim/base/ossimApplicationUsage.h>
#include <ossim/base/ossimKeywordNames.h>
#include <ossim/parallel/ossimMpi.h>

static void outputWriterTypes();


int main(int argc, char* argv[])
{
#if OSSIM_HAS_MPI
   ossimMpi::instance()->initialize(&argc, &argv);
#endif
   
   double start_time = 0.0;
   if (ossimMpi::instance()->getRank() == 0)
   {
      ossimNotify(ossimNotifyLevel_INFO)
         << "MPI running with "
         << ossimMpi::instance()->getNumberOfProcessors()
         << " processors..." << std::endl;
      start_time = ossimMpi::instance()->getTime();
   }
 
   ossimString tempString;
   ossimArgumentParser::ossimParameter stringParam(tempString);
   ossimArgumentParser argumentParser(&argc, argv);
   ossimInit::instance()->addOptions(argumentParser);
   ossimInit::instance()->initialize(argumentParser);
   argumentParser.getApplicationUsage()->setApplicationName(argumentParser.getApplicationName());

   argumentParser.getApplicationUsage()->setDescription(argumentParser.getApplicationName()+" creates overviews for the passed in image");

   argumentParser.getApplicationUsage()->setCommandLineUsage(argumentParser.getApplicationName()+" [options] <input file>");

   argumentParser.getApplicationUsage()->addCommandLineOption("-a or --include-fullres", "Wants to include full res dataset as well as reduced res sets.");

   argumentParser.getApplicationUsage()->addCommandLineOption("-h or --help", "Shows help");

   argumentParser.getApplicationUsage()->addCommandLineOption("--version", "Outputs version information.");

   argumentParser.getApplicationUsage()->addCommandLineOption("-e or --entry","Give the entry(zero based) to build an overview for.");
   
   argumentParser.getApplicationUsage()->addCommandLineOption("--list-entries","Lists the entries within the image");

   argumentParser.getApplicationUsage()->addCommandLineOption("--compression-quality","compression quality varies from 0 to 100, where 100 is best.  Currently only for JPEG comporession");

   argumentParser.getApplicationUsage()->addCommandLineOption("--compression-type","compression type can be: NONE, JPEG, PACKBITS, or DEFLATE");

   argumentParser.getApplicationUsage()->addCommandLineOption("--set-property","key:value NOTE: separate key value by a colon.");

   argumentParser.getApplicationUsage()->addCommandLineOption("--tile-size", "Defines the tile size for the supported overview handler. Tiff only right now");
  
   argumentParser.getApplicationUsage()->addCommandLineOption("-o", "Write overview to file specified.  If used on a multi-entry file, given \"foo.ovr\" you will get: \"foo_e0.ovr\"");

   argumentParser.getApplicationUsage()->addCommandLineOption("-s", "Stop dimension for overviews.  This controls how \nmany layers will be built. If set to 64 then the builder will stop when height and width for current level are less than or equal to 64.  Note a default can be set in the ossim preferences file, setting the keyword \"overview_stop_dimension\".");
   
   argumentParser.getApplicationUsage()->addCommandLineOption(
      "-t or --type",
      "see list at bottom for valid types. (default = ossim_tiff_box)");

   argumentParser.getApplicationUsage()->addCommandLineOption(
      "-r or --rebuild",
      "Rebuild overviews even if they are already present.");
   
   //---
   // Extract optional arguments.
   //---
   bool copyAllFlag      = false;
   bool entryListSetFlag = false;
   bool listEntriesFlag  = false;
   bool tileSizeFlag     = false;
   bool rebuildFlag      = false;

   ossimFilename inputFile;
   ossimFilename outputFile;
   
   std::vector<ossim_uint32> entryList;

   ossimIpt tileSize(64, 64);
   ossim_uint32 overviewStopDimension(0);


   
   //---
   // Temporary way to pass generic things to builder.
   //---
   std::vector< ossimRefPtr<ossimProperty> > propertyList(0);

   ossimString overviewType = "ossim_tiff_box";
   
   // Compression typedefs in tiff.h
   ossimString compressionType = "";
   ossim_uint32 cQuality = 75;
   if(argumentParser.read("-h") || argumentParser.read("--help"))
   {
      argumentParser.getApplicationUsage()->write(std::cout);
      outputWriterTypes();
      ossimMpi::instance()->finalize();
       exit(0);
   }

   if ( argumentParser.read("--version") )
   {
      ossimNotify(ossimNotifyLevel_NOTICE)
         << argumentParser.getApplicationName().c_str() << " " 
         << ossimInit::instance()->instance()->version().c_str()
         << std::endl;
      exit(0);
   }
   
   if(argumentParser.read("--compression-quality", stringParam))
   {
      cQuality = tempString.toUInt32();
   }
   if(argumentParser.read("--compression-type", stringParam))
   {
      compressionType = tempString;      
   }
   if(argumentParser.read("--tile-size", stringParam))
   {
      tileSize.x = tempString.toInt32();
      tileSize.y = tileSize.x;
      tileSizeFlag = true;
   }
   if( argumentParser.read("--list-entries"))
   {
      listEntriesFlag = true;
   }

   if( argumentParser.read("-o", stringParam) )
   {
      outputFile = tempString.trim();
   }

   if( argumentParser.read("-t", stringParam) ||
       argumentParser.read("--type", stringParam))
   {
      overviewType = tempString.trim();
   }

   if( argumentParser.read("-s", stringParam) )
   {
      overviewStopDimension = tempString.toUInt32();
   }
    
   if(argumentParser.read("-a") || argumentParser.read("--include-fullres"))
   {
      copyAllFlag = true;
   }

   if(argumentParser.read("-r") || argumentParser.read("--rebuild"))
   {
      rebuildFlag = true;
   }

   while(argumentParser.read("-e", stringParam) ||
         argumentParser.read("--entry", stringParam))
   {
      entryList.push_back(ossimString(tempString).toUInt32());
      entryListSetFlag = true;
   }

   while( argumentParser.read( "--set-property", stringParam) )
   {
      if (tempString.size())
      {
         ossimString propertyName;
         ossimString propertyValue;
         std::vector<ossimString> v = tempString.split(":");
         if (v.size() == 2)
         {
            propertyName = v[0];
            propertyValue = v[1];
         }
         else
         {
            propertyName = tempString;
         }
         
         ossimRefPtr<ossimProperty> p =
            new ossimStringProperty(propertyName, propertyValue);
         propertyList.push_back(p);
      }
      
   }
   
   argumentParser.reportRemainingOptionsAsUnrecognized();
   if (argumentParser.errors())
   {
      argumentParser.writeErrorMessages(std::cout);
      ossimMpi::instance()->finalize();
      exit(1);
   }

   if(argumentParser.argc() < 2)
   {
      argumentParser.getApplicationUsage()->write(std::cout);
      outputWriterTypes();
      ossimMpi::instance()->finalize();
      exit(1);
   }
   for (int i=1 ; i < argumentParser.argc(); i++)
   {
      inputFile = argumentParser.argv()[i];
      ossimRefPtr<ossimImageHandler> ih =
         ossimImageHandlerRegistry::instance()->open(inputFile);
      
      if(ih.valid())
      {
         // User wants to see entries.
         if(listEntriesFlag)
         {
            ih->getEntryList(entryList);
            std::cout << "Entries: ";
            std::copy(entryList.begin(),
                      entryList.end(),
                      std::ostream_iterator<ossim_uint32>(std::cout, " "));
            std::cout << std::endl;
            exit(0);
         }

         // Get the list of entries
         if (entryList.size() == 0)
         {
            ih->getEntryList(entryList);
         }

         // Force a rebuild.
         if (rebuildFlag)
         {
            for(ossim_uint32 idx = 0; idx < entryList.size(); ++idx)
            {
               ih->setCurrentEntry(entryList[idx]);
               ossimFilename f = ih->getOverviewFile();
               ih->closeOverview();
               if (f.exists())
               {
                  f.remove();
               }
            }
         }

         ossimRefPtr<ossimOverviewBuilderBase> ob =
            ossimOverviewBuilderFactoryRegistry::instance()->
            createBuilder(overviewType);
         if ( ob.valid() == false )
         {
            std::cout << "img2rr ERROR:"
                      << "\nCould not create builder for:  "
                      << overviewType
                      << endl;
            argumentParser.getApplicationUsage()->write(std::cout);
            outputWriterTypes();
            ossimMpi::instance()->finalize();
            exit(1);
         }
         propertyList.push_back(
            new ossimStringProperty("copy_all_flag",
                                    ossimString::toString(copyAllFlag)));
         propertyList.push_back(
            new ossimStringProperty(ossimKeywordNames::COMPRESSION_TYPE_KW,
                                    compressionType));

         if(tileSizeFlag)
         {
            propertyList.push_back(
               new ossimStringProperty(ossimKeywordNames::OUTPUT_TILE_SIZE_KW,
                                       tileSize.toString()));
         }
         
         // Generic property setting.
         if (propertyList.size())
         {
            ob->setProperties(propertyList);
         }

         // Get the list of entries
         if (entryList.size() == 0)
         {
            ih->getEntryList(entryList);
         }

         if (overviewStopDimension)
         {
            ob->setOverviewStopDimension(overviewStopDimension);
         }

         //---
         // This code block will put an entry string in the resulting
         // overview filename.  We only use this if the user used the -e or
         // --entry option, or we have multiple (more than one) entries.
         //---
         if(entryListSetFlag || entryList.size() > 1)
         {
            for(ossim_uint32 idx = 0; idx < entryList.size(); ++idx)
            {
               ih->setCurrentEntry(entryList[idx]);
               ossimFilename f;
               if (outputFile.size())
               {
                  f = outputFile.noExtension();
                  f += "_e";
                  f += ossimString::toString(entryList[idx]);
                  f.setExtension(".ovr");
               }
               else
               {
                  f = ih->getFilenameWithThisExtension(
                     ossimString(".ovr"), true);
               }

               std::cout << "Creating overviews for entry number: "
                         << entryList[idx]
                         << std::endl;

               ob->setOutputFile(f);
               ob->setInputSource(ih.get(), false);
               ob->execute();
            }
         }
         else
         {
            std::cout << "Creating overviews for file: "
                      << inputFile << std::endl;

            if (outputFile.size())
            {
               ob->setOutputFile(outputFile);
            }
            ob->setInputSource(ih.get(), false);
            ob->execute();
         }
         
      } // end of "if(ih.valid())"
      else
      {
         ossimNotify(ossimNotifyLevel_WARN)
            << "WARNING: Unable to open image file " << inputFile
            << std::endl;
      }
      
   } // end of "for (int i=1 ; i < argumentParser.argc(); i++)"

   if(ossimMpi::instance()->getRank() == 0)
   {
      double stop_time = ossimMpi::instance()->getTime();
      std::cout << "Elapsed time: " << (stop_time-start_time)
                << std::endl;
   }
   ossimMpi::instance()->finalize();

   exit(0);
}

void outputWriterTypes()
{
   ossimNotify(ossimNotifyLevel_NOTICE)
      << "\nValid overview types: " << std::endl;
 
   std::vector<ossimString> outputType;
   
   ossimOverviewBuilderFactoryRegistry::instance()->getTypeNameList(outputType);
   std::copy(outputType.begin(),
             outputType.end(),
             std::ostream_iterator<ossimString>(ossimNotify(ossimNotifyLevel_NOTICE), "\t\n"));
}
