/*
 * Copyright (c) 2015, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

import org.testng.annotations.DataProvider;
import org.testng.annotations.Test;

import static jdk.testlibrary.SerializationUtils.*;
import static org.testng.Assert.assertEquals;

/*
 * @test
 * @bug 8077559
 * @library /lib/testlibrary
 * @build jdk.testlibrary.SerializationUtils
 * @summary Tests Compact String. This one is testing String serialization
 *          among -XX:+CompactStrings/-XX:-CompactStrings/LegacyString
 * @run testng/othervm -XX:+CompactStrings SerializationTest
 * @run testng/othervm -XX:-CompactStrings SerializationTest
 */

public class SerializationTest {
    @DataProvider
    public Object[][] provider() {
        return new Object[][] {
                // every byte array is serialized from corresponding String object
                // by previous JDK(build 1.8.0_45-b14).
                new Object[] { "", new byte[] { -84, -19, 0, 5, 116, 0, 0 } },
                new Object[] { "A", new byte[] { -84, -19, 0, 5, 116, 0, 1, 65 } },
                new Object[] { "AB", new byte[] { -84, -19, 0, 5, 116, 0, 2, 65, 66 } },
                new Object[] { "abcdefghijk",
                        new byte[] {-84, -19, 0, 5, 116, 0, 11, 97, 98, 99, 100, 101,
                        102, 103, 104, 105, 106, 107 } },
                new Object[] { "\uff21", new byte[] { -84, -19, 0, 5, 116, 0, 3, -17, -68, -95 } },
                new Object[] { "\uff21\uff22", new byte[] { -84, -19, 0, 5, 116, 0, 6, -17, -68,
                        -95, -17, -68, -94 } },
                new Object[] { "\uff21A\uff21A\uff21A\uff21A\uff21A",
                        new byte[] { -84, -19, 0, 5, 116, 0, 20, -17, -68, -95, 65, -17, -68,
                        -95, 65, -17, -68, -95, 65, -17, -68, -95, 65, -17, -68, -95, 65 } },
                new Object[] { "A\uff21B\uff22C\uff23D\uff24E\uff25F\uff26G\uff27H\uff28",
                        new byte[] { -84, -19, 0, 5, 116, 0, 32, 65, -17, -68, -95, 66, -17, -68,
                        -94, 67, -17, -68, -93, 68, -17, -68, -92, 69, -17, -68, -91, 70, -17,
                        -68, -90, 71, -17, -68, -89, 72, -17, -68, -88 } },
                new Object[] { "\uff21A\uff22B\uff23C\uff24D\uff25E\uff26F\uff27G\uff28H",
                        new byte[] { -84, -19, 0, 5, 116, 0, 32, -17, -68, -95, 65, -17, -68,
                        -94, 66, -17, -68, -93, 67, -17, -68, -92, 68, -17, -68, -91, 69, -17,
                        -68, -90, 70, -17, -68, -89, 71, -17, -68, -88, 72 } },
                new Object[] { "\ud801\udc00\ud801\udc01\uff21A",
                        new byte[] { -84, -19, 0, 5, 116, 0, 16, -19, -96, -127, -19, -80, -128,
                        -19, -96, -127, -19, -80, -127, -17, -68, -95, 65 } },
                new Object[] { "\uff21\uff22\uff21\uff22\uff21\uff22\uff21\uff22\uff21\uff22",
                        new byte[] { -84, -19, 0, 5, 116, 0, 30, -17, -68, -95, -17, -68, -94, -17,
                        -68, -95, -17, -68, -94, -17, -68, -95, -17, -68, -94, -17, -68, -95, -17,
                        -68, -94, -17, -68, -95, -17, -68, -94 } } };
    }

    /*
     * Verify serialization works between Compact String/Legacy String
     */
    @Test(dataProvider = "provider")
    public void test(String strContent, byte[] baInJDK8) throws Exception {
        // Serialize a String object into byte array.
        byte[] ba = serialize(strContent);
        assertEquals(ba, baInJDK8);
        // Deserialize a String object from byte array which is generated by previous JDK(build 1.8.0_45-b14).
        Object obj = deserialize(ba);
        assertEquals(obj.getClass(), String.class);
        assertEquals((String)obj, strContent);
    }
}
