// avcodec_sample.cpp

// A small sample program that shows how to use libavformat and libavcodec to
// read video from a file.
//
// Use
//
// g++ -o avcodec_sample avcodec_sample.cpp -lavformat -lavcodec -lz
//
// to build (assuming libavformat and libavcodec are correctly installed on
// your system).
//
// Run using
//
// avcodec_sample myvideofile.mpg
//
// to write the first five frames from "myvideofile.mpg" to disk in PPM
// format.

#include "ffmpeg_access.h"
#include "avcodec.h"
#include "avformat.h"

#include <stdio.h>


#include <tiffio.h>

#include <errno.h>

typedef int bool;
#define true 1
#define false 0;

static int _ffmpegErrorOccuredFlag;
static char* _ffmpegLastErrorString;


int ffmpegNextFrame(FfmpegMovie* movie){
  AVFormatContext *pFormatCtx;
  AVCodecContext *pCodecCtx;
  int videoStream;
  AVFrame *pFrame;

  pFormatCtx = movie->pFormatCtx;
  pCodecCtx = movie->pCodecCtx;
  videoStream = movie->videoStream;
  pFrame = movie->pCurrentFrame;

  return GetNextFrame(pFormatCtx, pCodecCtx,videoStream,pFrame);
}

int ffmpegConvertCurrentFrame(FfmpegMovie* movie){
  img_convert((AVPicture *)movie->pConvertedFrame, PIX_FMT_RGB24, (AVPicture*)movie->pCurrentFrame,movie->pCodecCtx->pix_fmt,movie->pCodecCtx->width,movie->pCodecCtx->height);
}

int GetNextFrame(AVFormatContext *pFormatCtx, AVCodecContext *pCodecCtx, 
    int videoStream, AVFrame *pFrame)
{
    static AVPacket packet;
    static int      bytesRemaining=0;
    static uint8_t  *rawData;
    static bool     fFirstTime=true;
    int             bytesDecoded;
    int             frameFinished;

    //printf("Getnextframe called \n");

    // First time we're called, set packet.data to NULL to indicate it
    // doesn't have to be freed
    if(fFirstTime)
    {
        fFirstTime=false;
        packet.data=NULL;
    }

    // Decode packets until we have decoded a complete frame
    while(true)
    {
        // Work on the current packet until we have decoded all of it
        while(bytesRemaining > 0)
        {
          // Decode the next chunk of data
          bytesDecoded=avcodec_decode_video(pCodecCtx, pFrame,
                                            &frameFinished, rawData, bytesRemaining);

          //printf("Bytes Remaining: %d    Bytes Decoded: %d \n",bytesRemaining,bytesDecoded);

          // Was there an error?
          if(bytesDecoded < 0)
            {
              fprintf(stderr, "Error while decoding frame\n");
              return false;
            }
          
          bytesRemaining = bytesRemaining - bytesDecoded;
          rawData+=bytesDecoded;
          
          // Did we finish the current frame? Then we can return
          if(frameFinished)
            return true;
        }
        //printf("next packet \n");
        // Read the next packet, skipping all packets that aren't for this
        // stream
        do
        {
            // Free old packet
            if(packet.data!=NULL)
                av_free_packet(&packet);

            // Read new packet
            if(av_read_packet(pFormatCtx, &packet)<0)
                goto loop_exit;
        } while(packet.stream_index!=videoStream);

        bytesRemaining=packet.size;
        rawData=packet.data;
    }

loop_exit:

    // Decode the rest of the last frame
    bytesDecoded=avcodec_decode_video(pCodecCtx, pFrame, &frameFinished, 
        rawData, bytesRemaining);

    // Free last packet
    if(packet.data!=NULL)
        av_free_packet(&packet);

    return frameFinished!=0;
}

void SaveFrame(AVFrame *pFrame, int width, int height, int iFrame)
{
    FILE *pFile;
    char szFilename[32];
    int  y;

    // Open file
    sprintf(szFilename, "frame%d.ppm", iFrame);
    pFile=fopen(szFilename, "wb");
    if(pFile==NULL)
        return;

    // Write header
    fprintf(pFile, "P6 %d %d 255\n", width, height);

    // Write pixel data
    for(y=0; y<height; y++)
        fwrite(pFrame->data[0]+y*pFrame->linesize[0], 1, width*3, pFile);

    // Close file
    fclose(pFile);
}



void ffmpegSetLastErrorOccured(char* msg, int errorCode){
    _ffmpegErrorOccuredFlag = errorCode;
    _ffmpegLastErrorString = msg;
}

int ffmpegInit(){
    // Register all formats and codecs
    av_register_all();
    ffmpegSetLastErrorOccured("No error.",0);
}

int ffmpegErrorOccured(){
  return _ffmpegErrorOccuredFlag;
}
char* ffmpegLastErrorOccured(){
  return _ffmpegLastErrorString;
}


//TODO
void ffmpegMovieBackToStart(FfmpegMovie* movie){

  FfmpegMovie newMovie;

  ffmpegLoadMovie((movie->pFormatCtx->filename),&newMovie);

  av_close_input_file(movie->pFormatCtx);
  avcodec_close(movie->pCodecCtx);
  av_free(((AVPicture *)(movie->pConvertedFrame))->data[0]);
  av_free(movie->pCurrentFrame);
  free(movie->ppmData);

  movie->pFormatCtx = newMovie.pFormatCtx;
  movie->pCodecCtx = newMovie.pCodecCtx;
  movie->videoStream = newMovie.videoStream;
  movie->pCurrentFrame = newMovie.pCurrentFrame;
  movie->pConvertedFrame = newMovie.pConvertedFrame;
  movie->ppmData = newMovie.ppmData;
}


int ffmpegLoadMovie(char* movieName,FfmpegMovie * resultMovie){
    AVFormatContext *pFormatCtx;
    int             i, videoStream;
    AVCodecContext  *pCodecCtx;
    AVCodec         *pCodec;
    int             numBytes;
    uint8_t         *buffer;
    AVFrame         *pFrame; 
    AVFrame         *pFrameRGB;
    char* ppmHeader;
    int size;
    int ret;

    // Open video file
    ret = av_open_input_file(&pFormatCtx, movieName, NULL, 0, NULL);
    if(ret != 0){
      printf("ERROR: Could not open file ");
      printf(movieName);
      printf("    return : %d \n",ret);
      //return -1; // Couldn't open file
      ffmpegSetLastErrorOccured("Couldn't open file.",-1);
      printf("toto 2\n");
      return _ffmpegErrorOccuredFlag;
    }
     

    // Retrieve stream information
    if(av_find_stream_info(pFormatCtx)<0){
      printf("ERROR: Could not retrieve stream operation");
      printf("\n");
      av_close_input_file(pFormatCtx);
      //return -1; // Couldn't find stream information
      ffmpegSetLastErrorOccured("Couldn't find stream information.",-1);
      return _ffmpegErrorOccuredFlag;
    }
     

    // Dump information about file onto standard error
    //dump_format(pFormatCtx, 0, movieName, false);

    // Find the first video stream
    videoStream=-1;
    for(i=0; i<pFormatCtx->nb_streams; i++)
        if(pFormatCtx->streams[i]->codec.codec_type==CODEC_TYPE_VIDEO)
        {
            videoStream=i;
            break;
        }
    if(videoStream==-1){
      printf("ERROR: I didn't find a video stream");
      printf("\n");
      av_close_input_file(pFormatCtx);
      ffmpegSetLastErrorOccured("Didn't find a video stream.",-1);
      //return -1; // Didn't find a video stream
      return _ffmpegErrorOccuredFlag;
    }

    // Get a pointer to the codec context for the video stream
    pCodecCtx=&pFormatCtx->streams[videoStream]->codec;

    // Find the decoder for the video stream
    pCodec=avcodec_find_decoder(pCodecCtx->codec_id);
    if(pCodec==NULL){
      printf("ERROR: Codec not found");
      printf("\n");
      av_close_input_file(pFormatCtx);
      //return -1; // Codec not found
      ffmpegSetLastErrorOccured("Codec not found.",-1);
      return _ffmpegErrorOccuredFlag;
    }

    // Inform the codec that we can handle truncated bitstreams -- i.e.,
    // bitstreams where frame boundaries can fall in the middle of packets
    if(pCodec->capabilities & CODEC_CAP_TRUNCATED){
        pCodecCtx->flags|=CODEC_FLAG_TRUNCATED;
    }

    // Open codec
    if(avcodec_open(pCodecCtx, pCodec)<0){
      printf("ERROR: Can't open the codec");
      printf("\n");
      av_close_input_file(pFormatCtx);
      //return -1; // Could not open codec
      ffmpegSetLastErrorOccured("Could not open codec.",-1);
      return _ffmpegErrorOccuredFlag;
    }

    // Hack to correct wrong frame rates that seem to be generated by some 
    // codecs
    if(pCodecCtx->frame_rate>1000 && pCodecCtx->frame_rate_base==1){
        pCodecCtx->frame_rate_base=1000;
    }

    // Allocate video frame
    pFrame=avcodec_alloc_frame();

    // Allocate an AVFrame structure
    pFrameRGB=avcodec_alloc_frame();

    // Determine required buffer size and allocate buffer
    numBytes=avpicture_get_size(PIX_FMT_RGB24, pCodecCtx->width,pCodecCtx->height);
    buffer = (uint8_t*) malloc(sizeof(uint8_t)*numBytes);
    // Assign appropriate parts of buffer to image planes in pFrameRGB
    avpicture_fill((AVPicture *)pFrameRGB, buffer, PIX_FMT_RGB24,pCodecCtx->width, pCodecCtx->height);
    
    resultMovie->pFormatCtx = pFormatCtx;
    resultMovie->pCodecCtx = pCodecCtx;
    resultMovie->videoStream = videoStream;
    resultMovie->pCurrentFrame = pFrame;
    resultMovie->pConvertedFrame = pFrameRGB;

    size = asprintf(&ppmHeader,"P6 %d %d 255\n",pCodecCtx->width, pCodecCtx->height);
    size = size + (((pCodecCtx->width)*3)*(pCodecCtx->height));
    resultMovie->ppmData = malloc(size);

    free(ppmHeader);
    ffmpegSetLastErrorOccured("No error.",0);
    return _ffmpegErrorOccuredFlag;
}



int saveTIFFPictureToFile(char* fileName,char* buffer,unsigned int width, unsigned int height){
 // Define an image
  TIFF *image;

  // Open the TIFF file
  if((image = TIFFOpen(fileName, "w")) == NULL){
    printf("Could not open output.tif for writing\n");
   return -1;
  }

  // We need to set some values for basic tags before we can add any data
  TIFFSetField(image, TIFFTAG_IMAGEWIDTH, width);
  TIFFSetField(image, TIFFTAG_IMAGELENGTH, height);
  TIFFSetField(image, TIFFTAG_BITSPERSAMPLE, 16);
  TIFFSetField(image, TIFFTAG_SAMPLESPERPIXEL, 3);
  TIFFSetField(image, TIFFTAG_ROWSPERSTRIP, height);

  TIFFSetField(image, TIFFTAG_COMPRESSION, COMPRESSION_NONE);
  TIFFSetField(image, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_RGB);
  TIFFSetField(image, TIFFTAG_FILLORDER, FILLORDER_MSB2LSB);
  TIFFSetField(image, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);

  TIFFSetField(image, TIFFTAG_XRESOLUTION, 150.0);
  TIFFSetField(image, TIFFTAG_YRESOLUTION, 150.0);
  TIFFSetField(image, TIFFTAG_RESOLUTIONUNIT, RESUNIT_INCH);
  
  // Write the information to the file
  printf("Writting file ... %d \n",width*height);
  TIFFWriteEncodedStrip(image, 0, buffer, width*height*6);
  printf("done ...\n");

  // Close the file
  TIFFClose(image);
  return 0;
}
