/*
 * SimoComboBox is meant to be an alternative combobox to the
 * standard GtkComboBox.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * Author:
 * Todd A. Fisher (toddf@simosoftware.com)
 *
 */

#ifdef HAVE_CONFIG_H
#  include <kcconfig.h>
#endif

#include <string.h>
#include <libintl.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#ifdef HAVE_GTKSPELL
#  include <gtkspell/gtkspell.h>
#endif

#include "simocombobox.h"

#if defined(GDK_WINDOWING_X11) && defined(USE_SHADOWS)
#include "dropshadow.h"
#endif
#ifdef GDK_WINDOWING_WIN32
#include <windows.h>
#include "gdk/gdkwin32.h"
#endif
#include "util_cursor.h"
//#define WITHOUT_BUTTON

#define CHILD_SPACING     1
#define COMBO_MIN_WIDTH   50
#define COMBO_MIN_HEIGHT  10


enum {
  CLICKED,
  POPUP_DISPLAYED,
  POPUP_CLOSED,
  CHANGED,
  ACTIVATE,
  SIZE_CHANGED,
  LAST_SIGNAL
};

enum {
  PROP_0,
  PROP_MODEL,
  PROP_MAX_HEIGHT,
  PROP_MAX_WIDTH,
  PROP_EDIT,
  PROP_MAX_LINES
};

static const char *combo_box_seperator = "------------";

struct _SimoComboBoxPrivate{

  GtkWidget *parent;
  GtkWidget *popup_window;	/* the window that is dropped down when
                                   the button is clicked */
  GtkWidget *popup_scroll;	/* the scrolled window for the treeview */

  GtkTreePath *current_path;    /* path to the active cell, can be NULL
                                   if no items are currently selected */

  gint button_press_x, button_press_y;
#if defined(GDK_WINDOWING_X11) && defined(USE_SHADOWS)
  DropShadow *ds;
#endif
  SimoCursor hand;

  GtkEntryCompletion *completion;

#ifdef HAVE_GTKSPELL
  GtkSpell *gtkspell;
#endif

  gboolean scrolls_visible;
  gint     n_lines;
  gint     max_lines;

  guint editable_signal;
  guint drag_signal;

  guint arrow_sig;              /* arrow button sig */
  guint bl_sig;                 /* button label sig */
  GtkAllocation entry_alloc;
  gint fixed_entry_size;

  GtkWidget *label_button;

  GtkWidget *scrolled_window;
  GtkWidget *vbox_arrows;
  /* Return value of get_text for multi-line widget - so that the
     user does not need to free it. */
  gchar *ml_text;
  /* IDs of signal handlers */
  gulong single_changed_id;
  gulong multi_changed_id;
  /* Line height for size calculation */
  gint cached_line_height;
};

static GtkBinClass *parent_class = NULL;
static guint combo_box_signals[LAST_SIGNAL] = {0,};


static void     simo_combo_box_class_init(SimoComboBoxClass *klass);
static void     simo_combo_box_init(SimoComboBox *combo_box);
static void     simo_combo_box_size_request(GtkWidget      *widget,
                                            GtkRequisition *requisition);
static void     simo_combo_box_finalize(GObject *object);
static void     simo_combo_box_get_property(GObject    *object,
                                            guint       param_id,
                                            GValue     *value,
                                            GParamSpec *pspec);
static void     simo_combo_box_set_property(GObject      *object,
                                            guint         param_id,
                                            const GValue *value,
                                            GParamSpec   *pspec);
static void     simo_combo_box_size_allocate(GtkWidget     *widget,
                                             GtkAllocation *allocation );
static gint     simo_combo_box_get_line_height(SimoComboBox *combo_box);

static gboolean simo_combo_box_update_pressed(GtkWidget      *widget,
                                              GdkEventButton *event,
                                              SimoComboBox   *combo_box);

static void     simo_combo_box_row_activated(GtkTreeView *treeview,
                                             GtkTreePath *arg1,
                                             GtkTreeViewColumn *arg2,
                                             SimoComboBox *combo);

static gboolean simo_combo_box_treeview_press_event(GtkWidget *widget,
                                                    GdkEventButton *event,
                                                    SimoComboBox *combo_box);

static gboolean simo_combo_box_treeview_release_event(GtkWidget *widget,
                                                      GdkEventButton *event,
                                                      SimoComboBox *combo_box);
static gboolean simo_combo_box_key_release_event(GtkWidget *widget,
                                                 GdkEventKey *event,
                                                 SimoComboBox *combo_box);

static gboolean simo_combo_box_treeview_motion_event(GtkWidget *widget,
                                                     GdkEventMotion *event,
                                                     SimoComboBox *combo_box);

static gboolean simo_combo_box_is_seperator_row(SimoComboBox *combo_box,
                                                GtkTreeIter *iter);

static void simo_combo_box_row_deleted(GtkTreeModel *model,
                                       GtkTreePath  *path,
                                       SimoComboBox *combo);
/*static void simo_combo_box_row_inserted(GtkTreeModel *model,
                                        GtkTreePath  *path,
                                        GtkTreeIter  *iter,
                                        SimoComboBox *combo);*/

static void simo_combo_box_popup_close(SimoComboBox *combo_box);


static void      simo_combo_box_grab_focus_signal(GtkWidget *widget);
static void      simo_combo_box_text_changed_cb(GtkEditable *editable,
                                                gpointer     data);
static void      simo_combo_box_entry_activate_cb(GtkEditable *editable,
                                                  gpointer     data);

static gboolean  simo_combo_box_txtview_keypress_cb(GtkWidget   *widget,
                                                    GdkEventKey *evt,
                                                    gpointer     data);
static gboolean  simo_combo_box_arrow_up_cb(GtkWidget      *widget,
                                            GdkEventButton *event,
                                            gpointer        data);
static gboolean  simo_combo_box_arrow_down_cb(GtkWidget      *widget,
                                              GdkEventButton *event,
                                              gpointer        data);




GType
simo_combo_box_get_type(void)
{
  static GType combo_box_type = 0;

  if (!combo_box_type) {
    static const GTypeInfo combo_box_info = {
      sizeof (SimoComboBoxClass),
      NULL, /* base_init */
      NULL, /* base_finalize */
      (GClassInitFunc) simo_combo_box_class_init,
      NULL, /* class_finalize */
      NULL, /* class_data */
      sizeof (SimoComboBox),
      0,
      (GInstanceInitFunc) simo_combo_box_init
    };

    combo_box_type = g_type_register_static(GTK_TYPE_HBOX,
                                            "SimoComboBox",
                                            &combo_box_info,
                                            (GTypeFlags) 0);
  }

  return combo_box_type;
}

/*static
void
simo_combo_box_show(SimoComboBox *combo_box)
{
  gtk_widget_show(combo_box->button);
  gtk_widget_show(combo_box->entry);
  gtk_widget_show(combo_box);
}
static
void
simo_combo_box_hide(SimoComboBox *combo_box)
{
  gtk_widget_hide(combo_box->button);
  gtk_widget_hide(combo_box->entry);
  //gtk_widget_hide(combo_box);
} */


static
void
simo_combo_box_class_init(SimoComboBoxClass *klass)
{
  GObjectClass      *g_object_class;
  GtkObjectClass    *gtk_object_class;
  GtkWidgetClass    *widget_class;

  g_object_class   = G_OBJECT_CLASS (klass);
  gtk_object_class = (GtkObjectClass*) klass;
  widget_class     = (GtkWidgetClass*) klass;

  //widget_class->show = simo_combo_box_show;
  //widget_class->hide = simo_combo_box_hide;

  parent_class             = g_type_class_peek_parent((GObjectClass*) klass);
  g_object_class->finalize     = simo_combo_box_finalize;
  //gtk_object_class->destroy = simo_combo_box_destroy;
  //widget_class->realize = simo_combo_box_realize;
  g_object_class->set_property = simo_combo_box_set_property;
  g_object_class->get_property = simo_combo_box_get_property;
  widget_class->size_allocate  = simo_combo_box_size_allocate;
  widget_class->size_request   = simo_combo_box_size_request;
  widget_class->grab_focus     = simo_combo_box_grab_focus_signal;

  klass->clicked = NULL;
  klass->popup_displayed = NULL;

  /* signals */
  combo_box_signals[CLICKED] = g_signal_new("clicked",
                                            G_OBJECT_CLASS_TYPE(gtk_object_class),
                                            G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                                            G_STRUCT_OFFSET(SimoComboBoxClass, clicked),
                                            NULL, NULL,
                                            g_cclosure_marshal_VOID__VOID,
                                            G_TYPE_NONE, 0);
  combo_box_signals[POPUP_DISPLAYED] = g_signal_new("popup-displayed",
                                                    G_OBJECT_CLASS_TYPE(gtk_object_class),
                                                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                                                    G_STRUCT_OFFSET(SimoComboBoxClass, popup_displayed),
                                                    NULL, NULL,
                                                    g_cclosure_marshal_VOID__VOID,
                                                    G_TYPE_NONE, 0);
  combo_box_signals[POPUP_CLOSED] = g_signal_new("popup-closed",
                                                 G_OBJECT_CLASS_TYPE(gtk_object_class),
                                                 G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                                                 G_STRUCT_OFFSET(SimoComboBoxClass, popup_closed),
                                                 NULL, NULL,
                                                 g_cclosure_marshal_VOID__VOID,
                                                 G_TYPE_NONE, 0);
  combo_box_signals[CHANGED] = g_signal_new("changed",
                                            G_OBJECT_CLASS_TYPE(gtk_object_class),
                                            G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                                            G_STRUCT_OFFSET(SimoComboBoxClass, changed),
                                            NULL, NULL,
                                            g_cclosure_marshal_VOID__VOID,
                                            G_TYPE_NONE, 0);
  combo_box_signals[ACTIVATE] = g_signal_new("activate",
                                            G_OBJECT_CLASS_TYPE(gtk_object_class),
                                            G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                                            G_STRUCT_OFFSET(SimoComboBoxClass, activate),
                                            NULL, NULL,
                                            g_cclosure_marshal_VOID__VOID,
                                            G_TYPE_NONE, 0);
  combo_box_signals[SIZE_CHANGED] = g_signal_new("size_changed",
                                            G_OBJECT_CLASS_TYPE(gtk_object_class),
                                            G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                                            G_STRUCT_OFFSET(SimoComboBoxClass, size_changed),
                                            NULL, NULL,
                                            g_cclosure_marshal_VOID__VOID,
                                            G_TYPE_NONE, 0);

  /* properties */
  g_object_class_install_property( g_object_class,
                                   PROP_MODEL,
                                   g_param_spec_object( "model",
                                                        "ComboBox model",
                                                        "The model for the combo box",
                                                        GTK_TYPE_TREE_MODEL,
                                                        G_PARAM_READWRITE));

  g_object_class_install_property( g_object_class,
                                   PROP_MAX_WIDTH,
                                   g_param_spec_int( "max_width",
                                                     "Max width",
                                                     "Max width the popup window is allowed to grow to before displaying a scroll bar",
                                                     0,
                                                     G_MAXINT,
                                                     150,
                                                     G_PARAM_READWRITE));

  g_object_class_install_property( g_object_class,
                                   PROP_MAX_HEIGHT,
                                   g_param_spec_int( "max_height",
                                                     "Max height",
                                                     "Max height the popup window is allowed to grow to before displaying a scroll bar",
                                                     0,
                                                     G_MAXINT,
                                                     150,
                                                     G_PARAM_READWRITE));

  g_object_class_install_property( g_object_class,
                                   PROP_EDIT,
                                   g_param_spec_boolean( "edit",
                                                         "Editable",
                                                         "Controls whether or not the combobox entry is editable",
                                                         TRUE,
                                                         G_PARAM_READWRITE));

  g_object_class_install_property( g_object_class,
                                   PROP_MAX_LINES,
                                   g_param_spec_int( "max_lines",
                                                     "Maximum lines",
                                                     "Max number of lines allowed in the multi-line text editing widget",
                                                     1,
                                                     G_MAXINT,
                                                     10,
                                                     G_PARAM_READWRITE));
}

static gboolean simo_combo_box_is_seperator_row( SimoComboBox *combo_box, GtkTreeIter *iter )
{
  gchar *text = 0;

  gtk_tree_model_get( combo_box->model, iter, combo_box->visible_column, &text, -1 );

  if( text != 0 && !strcmp( text, combo_box_seperator ) ){
    g_free( text );
    return TRUE;
  }
  g_free( text );
  return FALSE;
}

static
gboolean allow_row_selection( GtkTreeSelection *selection,
                              GtkTreeModel *model,
                              GtkTreePath *path,
                              gboolean path_currently_selected,
                              SimoComboBox *combo_box )
{
  GtkTreeIter iter;

  if( gtk_tree_model_get_iter( model, &iter, path ) ){
    if( simo_combo_box_is_seperator_row( combo_box, &iter ) )
      return FALSE;
  }
  return TRUE;
}


static
void
simo_combo_box_init(SimoComboBox *combo_box)
{
  int mask;
  GtkWidget *arrow;
  GtkWidget *arrow_evtbox;
  GtkWidget *button_align;
  GtkWidget *frame;
  SimoComboBoxPrivate *mthis = combo_box->priv = g_new0(SimoComboBoxPrivate, 1);
  GtkCellRenderer *cell;
  GtkTreeSelection *selection;
  GtkWidget *align;

  combo_box->priv = mthis;
  combo_box->max_popup_height = 350;
  combo_box->max_popup_width = 450;
  combo_box->visible_column = 0;

  mthis->n_lines = 1;
  mthis->max_lines = 10;
  mthis->hand = simo_cursor_new( SIMO_HAND );
  // build the popup window
  mthis->popup_window = gtk_window_new( GTK_WINDOW_POPUP );
  align = gtk_alignment_new( 0, 0, 1, 1 );
  mthis->popup_scroll = gtk_scrolled_window_new( NULL, NULL );
  frame = gtk_frame_new( NULL );
  combo_box->model = (GtkTreeModel*)gtk_tree_store_new( 1, G_TYPE_STRING );
  combo_box->treeview = (GtkTreeView*) gtk_tree_view_new_with_model(combo_box->model);

  gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( mthis->popup_scroll ), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC );

  g_object_set( G_OBJECT( combo_box->treeview ), "fixed-height-mode", FALSE,
                "headers-visible", FALSE,
                "rules-hint", FALSE,
                "reorderable", FALSE, NULL );
  combo_box->column = gtk_tree_view_column_new();
  g_object_set( combo_box->column, "expand", TRUE, NULL );

  cell = gtk_cell_renderer_text_new();

  gtk_tree_view_column_pack_start( combo_box->column, cell, TRUE );
  gtk_tree_view_column_add_attribute( combo_box->column, cell, "text", 0 );

  //	gtk_tree_view_column_set_cell_data_func( combo_box->column, cell, simo_combo_box_visible_column_render, NULL, NULL );
  gtk_tree_view_column_set_resizable( combo_box->column, TRUE );
  gtk_tree_view_column_set_sizing( combo_box->column, GTK_TREE_VIEW_COLUMN_AUTOSIZE );

  gtk_tree_view_append_column( combo_box->treeview, combo_box->column );
  //#ifdef NDEBUG // only enable for release until warnnigs are fixed
#ifdef GDK_WINDOWING_WIN32 /* once Linux version becomes 2.6 we'll remove the macro */
  g_object_set( cell, "ellipsize", PANGO_ELLIPSIZE_END, NULL );
#endif
  //#endif
  selection = gtk_tree_view_get_selection( combo_box->treeview );

  gtk_tree_selection_set_select_function(selection,
                                         (GtkTreeSelectionFunc) allow_row_selection,
                                         combo_box, NULL);

  gtk_tree_selection_set_mode( selection, GTK_SELECTION_SINGLE );

  //gtk_tree_selection_unselect_all( selection );

  // set up the popup window with the appropriate attributes so that it works correctly as a popup window
  //gtk_window_set_type_hint( GTK_WINDOW( mthis->popup_window ), GDK_WINDOW_TYPE_HINT_MENU );
  gtk_window_set_keep_above( GTK_WINDOW( mthis->popup_window ), TRUE );
  gtk_window_set_decorated( GTK_WINDOW( mthis->popup_window ), FALSE );
  gtk_window_set_resizable( GTK_WINDOW( mthis->popup_window ), TRUE );
  gtk_widget_realize( mthis->popup_window );
  simo_cursor_set_delay(gtk_widget_get_window(mthis->popup_window),
                        mthis->hand );
  mask = gdk_window_get_events(gtk_widget_get_window(mthis->popup_window));
  gdk_window_set_events(gtk_widget_get_window(mthis->popup_window),
                        mask | GDK_BUTTON_PRESS_MASK
                        | GDK_BUTTON_RELEASE_MASK | GDK_KEY_PRESS_MASK
                        | GDK_KEY_RELEASE_MASK);

  //	gtk_container_add( GTK_CONTAINER( mthis->popup_window ), frame );
  gtk_frame_set_shadow_type( GTK_FRAME( frame ), GTK_SHADOW_OUT );
  gtk_container_set_border_width( GTK_CONTAINER( mthis->popup_scroll ), 0 );
  gtk_container_add( GTK_CONTAINER( mthis->popup_window ), align );
  gtk_container_add( GTK_CONTAINER( align ), frame );
  gtk_container_add( GTK_CONTAINER( frame ), mthis->popup_scroll );
  gtk_container_set_border_width( GTK_CONTAINER( mthis->popup_window ), 0 );
  gtk_widget_set_name( mthis->popup_window, "white" );
#if defined(GDK_WINDOWING_X11) && defined(USE_SHADOWS) /* only needed for unix dropshadows */
  gtk_alignment_set_padding( GTK_ALIGNMENT( align ), 0, 0, 0, 5 );
#endif
  //gtk_widget_set_name( mthis->popup_scroll, "white" );

  gtk_container_add(GTK_CONTAINER( mthis->popup_scroll ),
                    (GtkWidget*)combo_box->treeview);

  //	g_object_get( G_OBJECT( frame ), "border-width", &(mthis->frame_border ), NULL );

  // build the widget face
  arrow = gtk_arrow_new( GTK_ARROW_DOWN, GTK_SHADOW_IN );
  combo_box->button = gtk_toggle_button_new();//gtk_button_new();

  combo_box->entry = gtk_entry_new();

  mthis->scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(mthis->scrolled_window),
                                 GTK_POLICY_NEVER,
                                 GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(mthis->scrolled_window),
                                      GTK_SHADOW_IN);
  combo_box->txtview = gtk_text_view_new();
  gtk_text_view_set_pixels_above_lines(GTK_TEXT_VIEW(combo_box->txtview),
                                       2);
  gtk_text_view_set_left_margin(GTK_TEXT_VIEW(combo_box->txtview),
                                2);
  gtk_text_view_set_right_margin(GTK_TEXT_VIEW(combo_box->txtview),
                                 2);
  gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(combo_box->txtview),
                              GTK_WRAP_WORD_CHAR);
  combo_box->txtbuffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(combo_box->txtview));
  gtk_container_add(GTK_CONTAINER(mthis->scrolled_window), combo_box->txtview);

  mthis->label_button = gtk_toggle_button_new();//gtk_button_new();
  button_align = gtk_alignment_new( 0, (gfloat)0.5, 1, 1 );
  combo_box->label = gtk_label_new("");
#ifdef GDK_WINDOWING_WIN32 // FIXME: should check gtk+ version number for 2.5+
  gtk_label_set_ellipsize( GTK_LABEL( combo_box->label ), PANGO_ELLIPSIZE_END );
#endif
  //	hbox										= gtk_hbox_new( FALSE, 0 );
  combo_box->editable = TRUE;

  // give an initial allocation size to the entry box
  // Removed by Eduardo Kalinowski. Seems not to be necessary, and
  // does not compile when GSEAL_ENABLE is defined.
  //combo_box->priv->entry_alloc.width = COMBO_MIN_WIDTH - 25;
  //combo_box->entry->allocation.width = COMBO_MIN_WIDTH - 25;
  //combo_box->entry->allocation.height = COMBO_MIN_HEIGHT;
  //	g_print( "setting fixed entry size to default\n" );
  //combo_box->priv->fixed_entry_size = -1;
  //mthis->label_button->allocation.width = COMBO_MIN_HEIGHT;
  //mthis->label_button->allocation.height = COMBO_MIN_HEIGHT;

  //gtk_container_remove( GTK_CONTAINER( combo_box->button ),
  //                     gtk_bin_get_child( GTK_BIN( combo_box->button ) ) );
  //	gtk_box_pack_start( GTK_BOX( hbox ), combo_box->label, 1, 1, 0 );
  //	gtk_box_pack_start( GTK_BOX( hbox ) , arrow, 0, 1, 0 );
  //	gtk_widget_show( hbox );
  gtk_widget_show( arrow );

  gtk_container_add( GTK_CONTAINER( combo_box->button ), arrow );
  gtk_container_add( GTK_CONTAINER( mthis->label_button ), button_align );
  gtk_container_add( GTK_CONTAINER( button_align ), combo_box->label );

  gtk_widget_set_name( mthis->label_button, "combo_label" );
  gtk_widget_set_name( combo_box->label, "combo_label" );
  gtk_widget_set_name( arrow, "combo_arrow" );
  gtk_widget_set_name( combo_box->button, "combo_button" );
  gtk_widget_set_name( combo_box->entry, "combo_entry" );

  /* Create the completion object */
  mthis->completion = gtk_entry_completion_new();
  /* Assign the completion to the entry */
  gtk_entry_set_completion( GTK_ENTRY( combo_box->entry ), mthis->completion );
  g_object_unref( mthis->completion );

  gtk_entry_completion_set_model( mthis->completion, combo_box->model );

  /* Use model column 0 as the text column */
  /*gtk_entry_completion_set_text_column( mthis->completion, 0 );*/

#if defined(GDK_WINDOWING_X11) && defined(USE_SHADOWS) /* windows xp does not need this */
  // create the drop shadow object
  mthis->ds = drop_shadow_new();
#endif

  gtk_button_set_focus_on_click( GTK_BUTTON( combo_box->button ), FALSE );

  gtk_widget_show( arrow );
  gtk_widget_show( combo_box->button );
  gtk_widget_show( combo_box->entry );
  gtk_widget_show( combo_box->label );
  gtk_widget_show( button_align );
  gtk_widget_show( mthis->label_button );
  gtk_widget_hide( mthis->label_button );
  //	gtk_widget_show( frame );
  gtk_widget_show( mthis->popup_scroll );
  gtk_widget_show( GTK_WIDGET( combo_box->treeview ) );

  //	gtk_box_pack_start( GTK_BOX( combo_box ), mthis->label_button, TRUE, TRUE, 0 );
  gtk_box_pack_start( GTK_BOX( combo_box ), combo_box->entry, TRUE, TRUE, 0 );
  g_object_set(G_OBJECT(mthis->scrolled_window), "no-show-all", TRUE, NULL);
  gtk_box_pack_start(GTK_BOX(combo_box),
                     mthis->scrolled_window, TRUE, TRUE, 0);

  mthis->vbox_arrows = gtk_vbox_new(TRUE, 0);
  arrow = gtk_arrow_new(GTK_ARROW_UP, GTK_SHADOW_OUT);
  gtk_misc_set_alignment(GTK_MISC(arrow), .5, 0);
  arrow_evtbox = gtk_event_box_new();
  g_signal_connect(G_OBJECT(arrow_evtbox), "button-press-event",
                   G_CALLBACK(simo_combo_box_arrow_up_cb), combo_box);
  gtk_container_add(GTK_CONTAINER(arrow_evtbox), arrow);
  gtk_box_pack_start(GTK_BOX(mthis->vbox_arrows), arrow_evtbox,
                     TRUE, TRUE, 0);

  arrow = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_OUT);
  gtk_misc_set_alignment(GTK_MISC(arrow), .5, 1);
  arrow_evtbox = gtk_event_box_new();
  g_signal_connect(G_OBJECT(arrow_evtbox), "button-press-event",
                   G_CALLBACK(simo_combo_box_arrow_down_cb), combo_box);
  gtk_container_add(GTK_CONTAINER(arrow_evtbox), arrow);
  gtk_box_pack_start(GTK_BOX(mthis->vbox_arrows), arrow_evtbox,
                     TRUE, TRUE, 0);

  gtk_box_pack_end(GTK_BOX(combo_box), mthis->vbox_arrows, FALSE, FALSE, 0);



  gtk_alignment_set(GTK_ALIGNMENT(button_align),
                    (gfloat) 0.01, (gfloat) 0.5,
                    (gfloat) 1.0,  (gfloat) 0.5);
  gtk_misc_set_alignment(GTK_MISC(combo_box->label),
                         (gfloat) 0.01, (gfloat) 0.5);
  //	gtk_button_set_alignment( GTK_BUTTON( combo_box->label ), (gfloat)0.01, (gfloat)0.5 );
  //	gtk_box_pack_start( GTK_BOX( combo_box ), button, TRUE, TRUE, 0 );
  gtk_box_pack_end( GTK_BOX( combo_box ), combo_box->button, FALSE, FALSE, 0 );

  g_object_ref( G_OBJECT( combo_box->entry ) );
  g_object_ref( G_OBJECT( mthis->label_button ) );

  mthis->arrow_sig = g_signal_connect_swapped( G_OBJECT( combo_box->button ), "toggled", G_CALLBACK( simo_combo_box_popup ), combo_box );
  mthis->bl_sig = g_signal_connect_swapped( G_OBJECT( mthis->label_button ), "toggled", G_CALLBACK( simo_combo_box_popup ), combo_box );
  g_signal_connect( G_OBJECT( combo_box->treeview ), "row-activated", G_CALLBACK( simo_combo_box_row_activated ), combo_box );
  g_signal_connect( G_OBJECT( combo_box->treeview ), "button-press-event", G_CALLBACK( simo_combo_box_treeview_press_event ), combo_box );
  g_signal_connect( G_OBJECT( combo_box->treeview ), "button-release-event", G_CALLBACK( simo_combo_box_treeview_release_event ), combo_box );
  g_signal_connect( G_OBJECT( combo_box->treeview ), "key-release-event", G_CALLBACK( simo_combo_box_key_release_event ), combo_box );
  g_signal_connect( G_OBJECT( combo_box->treeview ), "motion-notify-event", G_CALLBACK( simo_combo_box_treeview_motion_event ), combo_box );

  mthis->single_changed_id
    = g_signal_connect(G_OBJECT(combo_box->entry), "changed",
                       G_CALLBACK(simo_combo_box_text_changed_cb), combo_box);
  g_signal_connect(G_OBJECT(combo_box->entry), "activate",
                   G_CALLBACK(simo_combo_box_entry_activate_cb), combo_box);

  mthis->multi_changed_id =
    g_signal_connect(G_OBJECT(combo_box->txtbuffer), "changed",
                     G_CALLBACK(simo_combo_box_text_changed_cb), combo_box);
  g_signal_connect(G_OBJECT(combo_box->txtview), "key-press-event",
                   G_CALLBACK(simo_combo_box_txtview_keypress_cb), combo_box);
}


static
gboolean
simo_combo_box_key_release_event(GtkWidget    *widget,
                                 GdkEventKey  *event,
                                 SimoComboBox *combo_box)
{
  if( event->keyval == GDK_Escape )
    simo_combo_box_popup_close( combo_box );

  return FALSE;
}


static
void
simo_combo_box_finalize(GObject *object)
{
  SimoComboBoxPrivate *mthis;
  //	g_print( "ENTER: simo_combo_box_finalize\n" );
  mthis = SIMO_COMBO_BOX( object )->priv;

  //	g_object_unref( G_OBJECT( SIMO_COMBO_BOX( object )->entry ) );
  //	g_object_unref( G_OBJECT( SIMO_COMBO_BOX( object )->label ) );

  // run parent finalize
  G_OBJECT_CLASS(parent_class)->finalize(object);

  if( mthis->current_path )
    gtk_tree_path_free( mthis->current_path );

#if defined(GDK_WINDOWING_X11) && defined(USE_SHADOWS)
  drop_shadow_destroy( mthis->ds );
#endif
  simo_cursor_free( mthis->hand );

  g_free(mthis->ml_text);

  g_free( mthis );
  //	g_print( "LEAVE: simo_combo_box_finalize\n" );
  //	g_free( GTK_WIDGET( object )->name );
}


static
void
simo_combo_box_get_property(GObject    *object,
                            guint       prop_id,
                            GValue     *value,
                            GParamSpec *pspec)
{
  SimoComboBox *combo_box = SIMO_COMBO_BOX( object );

  switch( prop_id ){
  case PROP_MODEL:
    g_value_set_object( value, combo_box->model );
    break;

  case PROP_MAX_WIDTH:
    g_value_set_int( value, combo_box->max_popup_width );
    break;

  case PROP_MAX_HEIGHT:
    g_value_set_int( value, combo_box->max_popup_height );
    break;

  case PROP_EDIT:
    g_value_set_boolean( value, gtk_editable_get_editable( GTK_EDITABLE( combo_box->entry ) ) );
    break;

  case PROP_MAX_LINES:
    g_value_set_int(value, combo_box->priv->max_lines);
    break;

  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID( object, prop_id, pspec );
    break;
  }
}


static
void
simo_combo_box_set_property( GObject *object,
                             guint prop_id,
                             const GValue	*value,
                             GParamSpec *pspec )
{
  SimoComboBox *combo_box = SIMO_COMBO_BOX( object );

  switch( prop_id ){
  case PROP_MODEL:{
    GtkTreeModel *model = g_value_get_object( value );
    simo_combo_box_set_model( combo_box, model );
    if( model )
      gtk_entry_completion_set_model( combo_box->priv->completion, model );
    //		else
    //			gtk_entry_completion_set_model( combo_box->priv->completion, 0 );
  }break;

  case PROP_MAX_WIDTH:
    combo_box->max_popup_width = g_value_get_int( value );
    break;

  case PROP_MAX_HEIGHT:
    combo_box->max_popup_height = g_value_get_int( value );
    break;

  case PROP_EDIT:
    gtk_editable_set_editable( GTK_EDITABLE( combo_box->entry ), g_value_get_boolean( value ) );
    break;

  case PROP_MAX_LINES:
    combo_box->priv->max_lines = g_value_get_int(value);
    break;

  default:
    break;
  }
}


static
void
simo_combo_box_size_request( GtkWidget *widget, GtkRequisition *req )
{
  SimoComboBox *combo_box = SIMO_COMBO_BOX(widget);

  //gint width  = req->width;
  //gint height = req->height;
  GtkRequisition breq, ereq, lreq, areq;
  //	g_print( "ENTER: simo_combo_box_size_request\n" );

  //	g_print( "before entry width: %d, button width: %d. total width: %d height: %d\n",
  //						combo_box->priv->entry_alloc.width,  combo_box->button->allocation.width, width, height );

  if (combo_box->priv->n_lines > 1) {
    gtk_widget_size_request(combo_box->priv->scrolled_window, &ereq);
  } else {
    gtk_widget_size_request(combo_box->entry, &ereq);
  }
  gtk_widget_size_request(combo_box->button, &breq);
  gtk_widget_size_request(combo_box->priv->label_button, &lreq);
  gtk_widget_size_request(combo_box->priv->vbox_arrows, &areq);

  if (lreq.width < ereq.width) {
    ereq.width = lreq.width;
  } else {
    lreq.width = ereq.width;
  }

  lreq.height = breq.height;
  ereq.height = breq.height;

  if (combo_box->priv->n_lines > 1) {
    req->height = simo_combo_box_get_line_height(combo_box) * combo_box->priv->n_lines + 8;
  } else {
    req->height = breq.height;
  }
  req->width  = ereq.width + breq.width + areq.width;

  if (req->width
      < (combo_box->priv->fixed_entry_size + breq.width + areq.width)) {
    req->width = combo_box->priv->fixed_entry_size + breq.width + areq.width;
    ereq.width = combo_box->priv->fixed_entry_size;
    lreq.width = combo_box->priv->fixed_entry_size;
  }

  gtk_widget_set_size_request(combo_box->priv->label_button,
                              lreq.width, lreq.height);
  //	gtk_widget_size_request( combo_box->priv->label_button, &lreq );
  //gtk_widget_size_request( combo_box->entry, &ereq );
  gtk_widget_set_size_request(combo_box->entry, ereq.width, ereq.height);
  gtk_widget_set_size_request(combo_box->priv->scrolled_window, ereq.width, ereq.height);

  //width  = req->width;
  //height = req->height;
  //	g_print( "after entry width: %d, button width: %d. total width: %d height: %d\n",
  //						combo_box->priv->entry_alloc.width,  combo_box->button->allocation.width, width, height );

  //	g_print( "LEAVE: simo_combo_box_size_request\n" );
}

static
void
simo_combo_box_size_allocate(GtkWidget *widget, GtkAllocation *allocation)
{
  GtkAllocation ealloc;
  SimoComboBox *combo = SIMO_COMBO_BOX( widget );
  GtkAllocation combobtnAlloc;
  GtkAllocation vboxarrowsAlloc;

  //	g_print( "ENTER: simo_combo_box_size_allocate\n" );

  g_return_if_fail( SIMO_IS_COMBO_BOX( widget ) );
  g_return_if_fail( allocation != NULL );

  GTK_WIDGET_CLASS( parent_class )->size_allocate( widget, allocation );
  //		g_print( "combo box allocate: %d:%d:%d:%d:%d\n", allocation->x, allocation->y, allocation->width, allocation->height, combo->priv->fixed_entry_size );

#if GTK_MINOR_VERSION < 18
  combobtnAlloc   = combo->button->allocation;
  vboxarrowsAlloc = combo->priv->vbox_arrows->allocation;
#else
  gtk_widget_get_allocation(combo->button, &combobtnAlloc);
  gtk_widget_get_allocation(combo->priv->vbox_arrows, &vboxarrowsAlloc);
#endif

  ealloc.x = allocation->x;
  ealloc.y = allocation->y;
  ealloc.width = allocation->width
    - combobtnAlloc.width
    - vboxarrowsAlloc.width;
  ealloc.height = allocation->height;

  if (combo->priv->n_lines > 1) {
    gtk_widget_size_allocate( combo->priv->scrolled_window, &ealloc );
  } else {
    gtk_widget_size_allocate( combo->entry, &ealloc );
  }
  //	gtk_widget_size_allocate( combo->button, &combo->button->allocation.width );
  gtk_widget_size_allocate( combo->priv->label_button, &ealloc );

  //	g_print( "LEAVE: simo_combo_box_size_allocate\n" );

  //	}
}


static
gint
simo_combo_box_get_line_height(SimoComboBox *combo_box)
{
  if (combo_box->priv->cached_line_height == 0) {
    PangoLayout    *layout;
    PangoRectangle  logical_rect;

    layout = gtk_widget_create_pango_layout(GTK_WIDGET(combo_box->txtview), "W");
    pango_layout_get_pixel_extents(layout, NULL, &logical_rect);
    combo_box->priv->cached_line_height = logical_rect.height;
    g_object_unref(layout);
  }

  return combo_box->priv->cached_line_height;
}


GtkWidget *
simo_combo_box_new()
{
  return g_object_new( simo_combo_box_get_type(), NULL );
}


GtkWidget *
simo_combo_box_new_with_model( GtkTreeModel *model )
{
  SimoComboBox *cb = (SimoComboBox*)g_object_new( simo_combo_box_get_type(), NULL );

  simo_combo_box_set_model( cb, model );
  gtk_entry_completion_set_model( cb->priv->completion, model );

  return (GtkWidget*)cb;
}


void
simo_combo_box_set_model( SimoComboBox *combo_box, GtkTreeModel *model)
{
  if( combo_box->model )
    g_object_unref( combo_box->model );

  combo_box->model = model;

  gtk_tree_view_set_model( combo_box->treeview, combo_box->model );
  if( combo_box->model ){
    g_object_ref( combo_box->model );
    g_signal_connect( combo_box->model, "row-deleted", G_CALLBACK( simo_combo_box_row_deleted ), combo_box );
    gtk_entry_set_completion( GTK_ENTRY( combo_box->entry ), NULL );
    //		g_object_unref( combo_box->priv->completion );
    combo_box->priv->completion = gtk_entry_completion_new();

    gtk_entry_set_completion( GTK_ENTRY( combo_box->entry ), combo_box->priv->completion );
    g_object_unref( combo_box->priv->completion );
    gtk_entry_completion_set_model( combo_box->priv->completion, combo_box->model );
    /*gtk_entry_completion_set_text_column( combo_box->priv->completion, combo_box->visible_column );*/
  }

  //	g_signal_connect( combo_box->model, "row-inserted", G_CALLBACK( simo_combo_box_row_inserted ), combo_box );
}


void
simo_combo_box_set_visible_column( SimoComboBox *combo_box, int col_id )
{
  combo_box->visible_column = col_id;
  gtk_entry_completion_set_text_column( combo_box->priv->completion, col_id );
}


void
simo_combo_box_set_combo_column(SimoComboBox      *combo_box,
                                GtkTreeViewColumn *column)
{
  gtk_tree_view_remove_column(combo_box->treeview,
                              combo_box->column);
  gtk_tree_view_append_column(combo_box->treeview,
                              column);
  combo_box->column = column;
}


GtkEntryCompletion *
simo_combo_box_get_completion(SimoComboBox *combo_box)
{
  return combo_box->priv->completion;
}


void
simo_combo_box_set_completion_renderer(SimoComboBox    *combo_box,
                                       GtkCellRenderer *renderer)
{
  gtk_cell_layout_clear(GTK_CELL_LAYOUT(combo_box->priv->completion));
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(combo_box->priv->completion),
                             renderer,
                             TRUE);
}


void
simo_combo_box_set_completion_cell_func(SimoComboBox          *combo_box,
                                        GtkCellRenderer       *renderer,
                                        GtkCellLayoutDataFunc  func,
                                        gpointer               user_data,
                                        GDestroyNotify         destroy)
{
  gtk_cell_layout_set_cell_data_func(GTK_CELL_LAYOUT(combo_box->priv->completion),
                                     renderer,
                                     func,
                                     user_data,
                                     destroy);
}


GtkTreeModel
*simo_combo_box_get_model( SimoComboBox *combo_box)
{
  return combo_box->model;
}


static
void
simo_combo_box_row_deleted( GtkTreeModel *model,
                            GtkTreePath *path,
                            SimoComboBox *combo )
{
  if( combo->priv->current_path ){
    if( !gtk_tree_path_compare( path, combo->priv->current_path ) ){
      gtk_tree_path_free( combo->priv->current_path );
      combo->priv->current_path = NULL;
    }
  }
}


static
gboolean
simo_combo_box_update_pressed( GtkWidget *widget,
                               GdkEventButton *event,
                               SimoComboBox *combo_box )

{
  gdouble x, y;
  gint x1, x2, y1, y2;
  gint xoffset, yoffset;
  GtkAllocation alloc;
  //	GtkWidget *event_widget;

  if( event->button != 1 ){
    return FALSE;
  }
  x = event->x_root;
  y = event->y_root;
  gdk_window_get_root_origin( gtk_widget_get_window(widget),
                              &xoffset, &yoffset );

#if GTK_MINOR_VERSION < 18
  alloc = widget->allocation;
#else
  gtk_widget_get_allocation(widget, &alloc);
#endif

  xoffset += alloc.x;
  yoffset += alloc.y;

  x1 = alloc.x + xoffset;
  y1 = alloc.y + yoffset;
  x2 = x1 + alloc.width;
  y2 = y1 + alloc.height;

  if( x > x1 && x < x2 && y > y1 && y < y2 ){
    return TRUE;
  }
  simo_combo_box_popup_close( combo_box );

  return TRUE;
}


static
void
simo_combo_box_popup_close( SimoComboBox *combo_box )
{
  g_signal_emit( combo_box, combo_box_signals[POPUP_CLOSED], 0 );

  gtk_grab_remove( combo_box->priv->popup_window );
  gdk_keyboard_ungrab( GDK_CURRENT_TIME );
  gdk_pointer_ungrab( GDK_CURRENT_TIME );

  g_signal_handlers_disconnect_by_func( G_OBJECT( combo_box->priv->popup_window ),
                                        G_CALLBACK( simo_combo_box_update_pressed ),
                                        combo_box );
#if defined(GDK_WINDOWING_X11) && defined(USE_SHADOWS)
  drop_shadow_detach( combo_box->priv->ds );
#endif
#ifdef GDK_WINDOWING_WIN32
  {
    RECT rcFrom,rcTo;
    HWND from_hwnd =GDK_WINDOW_HWND(combo_box->priv->popup_window->window);
    HWND to_hwnd = GDK_WINDOW_HWND(GTK_WIDGET(combo_box)->window);

    GetWindowRect(from_hwnd,&rcFrom);
    GetWindowRect(to_hwnd,&rcTo);

    DrawAnimatedRects(from_hwnd,1,&rcFrom,&rcTo);
    ShowWindow(from_hwnd,SW_HIDE);
  }
#endif

  // set toggle state on buttons
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( combo_box->priv->label_button ), FALSE );
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( combo_box->button ), FALSE );
  g_signal_handler_unblock( G_OBJECT( combo_box->button ), combo_box->priv->arrow_sig );
  g_signal_handler_unblock( G_OBJECT( combo_box->priv->label_button ), combo_box->priv->bl_sig );

  gtk_widget_hide(GTK_WIDGET(combo_box->priv->popup_window));
  // the label should have focus after the combobox is clicked
  // make sure our widget has focus after being clicked on
#if GTK_MINOR_VERSION < 18
  if (GTK_WIDGET_CAN_FOCUS(combo_box)) {
    //    gtk_widget_grab_focus( GTK_WIDGET( combo_box ) );
    if (GTK_WIDGET_CAN_FOCUS(combo_box->entry)) {
      gtk_widget_grab_focus(GTK_WIDGET(combo_box->entry));
    } else if (GTK_WIDGET_CAN_FOCUS(combo_box->priv->label_button)) {
      gtk_widget_grab_focus(GTK_WIDGET( combo_box->priv->label_button));
    }
  }
#else
  if (gtk_widget_get_can_focus(GTK_WIDGET(combo_box))) {
    //    gtk_widget_grab_focus( GTK_WIDGET( combo_box ) );
    if (gtk_widget_get_can_focus(combo_box->entry)) {
      gtk_widget_grab_focus(GTK_WIDGET(combo_box->entry));
    } else if (gtk_widget_get_can_focus(combo_box->priv->label_button)) {
      gtk_widget_grab_focus(GTK_WIDGET( combo_box->priv->label_button));
    }
  }
#endif
}


typedef struct _DimWin {
  gint count;
  gint max_width;
  SimoComboBox *combo;
  PangoFontDescription *desc;
} DimWin;


static
gboolean
compute_max_width( GtkTreeModel *model,
                   GtkTreePath *path,
                   GtkTreeIter *iter,
                   DimWin *dim )
{
  GValue value = {0};
  PangoLayout *layout;
  PangoRectangle pango_rect;

  gtk_tree_model_get_value( model, iter, dim->combo->visible_column, &value );

  if( G_VALUE_TYPE( &value ) == G_TYPE_STRING ){
    const gchar *text = g_value_get_string( &value );

    if( text && strlen( text ) > 0 ){
      if( dim->desc ){
        layout = gtk_widget_create_pango_layout( GTK_WIDGET( dim->combo->treeview ), text );
        pango_layout_set_font_description( layout, dim->desc );
        pango_layout_get_pixel_extents( layout, NULL, &pango_rect );
        g_object_unref( layout );

        if( pango_rect.width > dim->max_width )
          dim->max_width = pango_rect.width;

      }
    }

    g_value_unset( &value );
  }
  dim->count++;

  return FALSE;
}


static
void
simo_combo_box_compute_tree_size( SimoComboBox *combo,
                                  gint *max_cell_width,
                                  gint *max_cell_height,
                                  gint *number_rows )
{
  GList *i, *j;
  gint max_height = 0;
  gint height;
  gint spacing = 0;
  DimWin dim = { 0, 0, combo, 0 };
  PangoFontDescription *desc = 0;


  // get the max height i.e. the tallest cell renderer
  // loop over each column
  for( i = gtk_tree_view_get_columns( combo->treeview ); i; i = g_list_next( i ) ){
    spacing += gtk_tree_view_column_get_spacing( GTK_TREE_VIEW_COLUMN( i->data ) );
    // loop over each cell renderer
    for( j = gtk_cell_layout_get_cells( GTK_CELL_LAYOUT( i->data ) );
         j; j = g_list_next( j ) ){

      gtk_cell_renderer_get_size( GTK_CELL_RENDERER( j->data ), GTK_WIDGET( combo->treeview ),
                                  NULL, NULL, NULL, NULL, &height );
      if( gtk_cell_renderer_text_get_type() == G_OBJECT_TYPE( j->data ) ){
        if( desc ){
          pango_font_description_free( desc );
          desc = 0;
        }
        g_object_get( G_OBJECT( j->data ), "font-desc", &desc, NULL );
      }
      height += gtk_tree_view_column_get_spacing( GTK_TREE_VIEW_COLUMN( i->data ) );
      if( height > max_height ){
        max_height = height;
      }
    }
  }
  dim.desc = desc;
  // calculate the max width required based on the text in each cell this is expansive and hence we should consider
  // an alternative.  Esspecially since it assumes text!!
  gtk_tree_model_foreach( combo->model, (GtkTreeModelForeachFunc)compute_max_width, &dim );
#ifdef GDK_WINDOWING_WIN32 // NOTE: this should be conditional if gtk+2.5 > then don't add 80 and use ellipsize
  *max_cell_width  = dim.max_width + spacing + 30; // for padding we add 80 this can't be right there must be a value we can get from somewhere else
#else
  *max_cell_width  = dim.max_width + spacing + 80; // for padding we add 80 this can't be right there must be a value we can get from somewhere else
#endif
  *max_cell_height = max_height;
  *number_rows = dim.count;
  if( desc ){
    pango_font_description_free( desc );
    desc = 0;
  }
}


void
simo_combo_box_popup( SimoComboBox *combo_box )
{
  gint bx,by;
  gint width, height;
  gint screen_height, screen_width;
  GtkWidget *widget;
  GtkAllocation alloc;
  SimoComboBoxPrivate *mthis;
  //	GtkTreeViewColumn *column;
  gint tree_req_height, cell_width, cell_height, count;

  g_return_if_fail( SIMO_IS_COMBO_BOX( combo_box ) );
  /*
    if( combo_box->priv->editable_signal > 0 &&
    g_signal_handler_is_connected( combo_box->entry, combo_box->priv->editable_signal ) ){
    g_signal_handler_block( combo_box->entry, combo_box->priv->editable_signal );
    g_print( "signal blocked\n" );
    }
  */
  widget = GTK_WIDGET( combo_box );
  mthis = combo_box->priv;

  if (mthis->parent == NULL) {
    mthis->parent = gtk_widget_get_toplevel(widget);
#if GTK_MINOR_VERSION < 18
   if (!GTK_WIDGET_TOPLEVEL(mthis->parent)) {
      return;
    }
#else
   if (!gtk_widget_is_toplevel(mthis->parent)) {
      return;
    }
#endif
  }

  g_signal_emit( combo_box, combo_box_signals[POPUP_DISPLAYED], 0 );

  g_signal_connect( G_OBJECT( mthis->popup_window ), "button-release-event",
                    G_CALLBACK( simo_combo_box_update_pressed ), combo_box );
  //	g_signal_connect( G_OBJECT( mthis->popup_window ), "event_after",
  //										G_CALLBACK( simo_combo_box_button_after ), combo_box );

#if GTK_MINOR_VERSION < 18
  alloc = widget->allocation;
#else
  gtk_widget_get_allocation(widget, &alloc);
#endif

  // begin determining the window position

  gdk_window_get_origin( gtk_widget_get_window(widget), &bx, &by );
  bx += alloc.x;
  by += alloc.y + alloc.height;
  screen_height = gdk_screen_height() - by;
  screen_width = gdk_screen_width();

#if GTK_MINOR_VERSION < 18
  alloc = mthis->popup_window->allocation;
#else
  gtk_widget_get_allocation(mthis->popup_window, &alloc);
  height = alloc.height;
  width = alloc.width;
#endif

#if GTK_MINOR_VERSION < 18
  alloc = widget->allocation;
#else
  gtk_widget_get_allocation(widget, &alloc);
#endif

  gtk_window_move( GTK_WINDOW( mthis->popup_window ), -500, -500 );
  gtk_widget_set_size_request( mthis->popup_window, alloc.width, 20 );

  gtk_window_resize( GTK_WINDOW( mthis->popup_window ), 500, 500 );

  gtk_widget_show( mthis->popup_window );
  // begin determining the window dimensions
  //	column = gtk_tree_view_get_column( combo_box->treeview, 0 );
  simo_combo_box_compute_tree_size( combo_box, &cell_width, &cell_height, &count );

  gtk_widget_hide( mthis->popup_window );

  //	g_print( "max width: %d max height: %d\n", cell_width, cell_height );

  // we pad by 1/2 a cell height to avoid showing scrollbars when they're not needed
  tree_req_height = (count * cell_height) + (cell_height / 2);

  if( count > 0 && tree_req_height > 0 && tree_req_height < (gint)combo_box->max_popup_height ){

    //		gtk_widget_set_size_request( mthis->popup_window, cell_width, tree_req_height );

    //		gtk_window_resize( GTK_WINDOW( mthis->popup_window ), cell_width, tree_req_height );


    // check if the scroll bars are visible
    tree_req_height += 15;//bar_height;
    cell_width += 15;//bar_width;
    /*		if( GTK_SCROLLED_WINDOW( mthis->popup_scroll )->hscrollbar_visible || mthis->scrolls_visible ){
                tree_req_height += 15;//bar_height;
                mthis->scrolls_visible = TRUE;
                }
                else{
                mthis->scrolls_visible = FALSE;
                }
                if( GTK_SCROLLED_WINDOW( mthis->popup_scroll )->vscrollbar_visible || mthis->scrolls_visible ){
                cell_width += 15;//bar_width;
                mthis->scrolls_visible = TRUE;
                }
                else{
                mthis->scrolls_visible = FALSE;
                }*/

    height = tree_req_height;
  }
  else if( count > 0 && tree_req_height > 0 ){
    height = combo_box->max_popup_height;
  }
  else{
    height = 20;
  }
  // TODO: we need to have a horizontal scroll bar apppear when the width exceeds our max width
  /*	if( cell_width > widget->allocation.width + combo_box->max_popup_width ){
        width = widget->allocation.width + combo_box->max_popup_width;
        }
        else*/
  width = alloc.width;   /* Never wider than the widget */

  /* Always show popup below widget */
  if( height > screen_height ){
    height = screen_height - 15;
  }
  //	g_print( "width: %d, height: %d\n", width, height );

  gtk_widget_set_size_request( mthis->popup_window, width, height );

  gtk_window_resize( GTK_WINDOW( mthis->popup_window ), width, height );
  // end determining the window dimensions

  // do edge detection to ensure window is not clipped by the edge of the screen


  // ensure things are not hanging over the left edge
  if( bx < 0 ){
    bx = 0;
  }
  // ensure things are not hanging over the right edge
  if( ( bx + width ) > screen_width ){
    bx -= bx+width - screen_width;
  }


  gtk_window_move( GTK_WINDOW( mthis->popup_window ), bx, by );
  gtk_widget_show_all( mthis->popup_window );

  //	gtk_widget_set_sensitive( combo_box->button, 0 );
  //	gtk_widget_set_sensitive( mthis->label_button, 0 );
  g_signal_handler_block( G_OBJECT( combo_box->button ), combo_box->priv->arrow_sig );
  g_signal_handler_block( G_OBJECT( combo_box->priv->label_button ), combo_box->priv->bl_sig );

  //gtk_widget_grab_default( GTK_WIDGET( combo_box ) );

  //grab_on_window( mthis->popup_window->window, gtk_get_current_event_time() );
  //	grab_on_window( GTK_WIDGET(combo_box->treeview)->window, gtk_get_current_event_time() );
#if GTK_MINOR_VERSION < 18
  if (!GTK_WIDGET_HAS_FOCUS(combo_box->treeview)) {
    GTK_WIDGET_SET_FLAGS(combo_box->treeview, GTK_CAN_DEFAULT);
    gdk_keyboard_grab(gtk_widget_get_window(mthis->popup_window),
                      FALSE, GDK_CURRENT_TIME);
    gtk_widget_grab_focus(GTK_WIDGET(combo_box->treeview));
  }
#else
  if (!gtk_widget_has_focus(GTK_WIDGET(combo_box->treeview))) {
    gtk_widget_set_can_default(GTK_WIDGET(combo_box->treeview), TRUE);
    gdk_keyboard_grab(gtk_widget_get_window(mthis->popup_window),
                      FALSE, GDK_CURRENT_TIME);
    gtk_widget_grab_focus(GTK_WIDGET(combo_box->treeview));
  }
#endif

  gtk_grab_add( mthis->popup_window );
  gdk_pointer_grab( gtk_widget_get_window(mthis->popup_window), TRUE,
                    GDK_BUTTON_PRESS_MASK |
                    GDK_BUTTON_RELEASE_MASK |
                    GDK_POINTER_MOTION_MASK,
                    NULL, NULL, GDK_CURRENT_TIME );
  //	gtk_grab_add( combo_box->treeview );

  //	gtk_widget_grab_focus( mthis->parent );
  //	GTK_WIDGET_SET_FLAGS (combo_box->treeview, GTK_CAN_DEFAULT);
  //	gtk_widget_grab_default( GTK_WIDGET( combo_box->treeview ) );
  //gtk_button_released( GTK_BUTTON( combo_box->button ) );
  //gtk_button_leave( GTK_BUTTON( combo_box->button ) );
#ifdef G_OS_WIN32
  // workaround for bug: 152566 & 144269
  //gtk_window_present( GTK_WINDOW( mthis->popup_window ) );
  /*{// special case code to get focus on the combobox window
    HWND hwnd = (HWND)GDK_WINDOW_HWND( GTK_WIDGET( combo_box->treeview )->window );
    SetFocus( hwnd );
    SendMessage( tophwnd, WM_PARENTNOTIFY, (WPARAM)WM_LBUTTONDOWN, MAKELPARAM( pt.x, pt.y ) );
    SendMessage( tophwnd, WM_MOUSEACTIVATE, (WPARAM)focus_handler->moz_hwnd, 1 )
    }*/
  //BringWindowToTop( (HWND)GDK_WINDOW_HWND( mthis->popup_window->window ) );
#endif
#if defined(GDK_WINDOWING_X11) && defined(USE_SHADOWS)
  drop_shadow_attach( mthis->ds, mthis->popup_window, bx, by );
#endif
  // set toggle state on buttons
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( mthis->label_button ), TRUE );
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( combo_box->button ), TRUE );
  //
}


static
gchar
*get_gvalue_as_string( GValue *value )
{
  GType type = G_VALUE_TYPE( value );
  switch( type ){
  case G_TYPE_BOOLEAN:
    return g_strdup_printf( "%d", (int)g_value_get_boolean( value ) );
    break;
  case G_TYPE_INT:
    return g_strdup_printf( "%d", g_value_get_int( value ) );
    break;
  case G_TYPE_FLOAT:
    return g_strdup_printf( "%f", g_value_get_float( value ) );
    break;
  case G_TYPE_DOUBLE:
    return g_strdup_printf( "%f", g_value_get_double( value ) );
    break;
  case G_TYPE_STRING:
    return g_value_dup_string( value );
    break;
  default:
    break;
  }
  return NULL;
}


static
gboolean
simo_combo_box_treeview_press_event( GtkWidget *widget,
                                     GdkEventButton *event,
                                     SimoComboBox *combo_box )
{

  //SIMO_COMBO_BOX( widget )->priv->button_press_x = event->x;
  //SIMO_COMBO_BOX( widget )->priv->button_press_y = event->y;

  return FALSE;
}


static
gboolean
simo_combo_box_treeview_release_event( GtkWidget *widget,
                                       GdkEventButton *event,
                                       SimoComboBox *combo_box )
{
  GtkTreePath *path;
  GtkTreeViewColumn *col;
  GtkAllocation      alloc;

#if GTK_MINOR_VERSION < 18
  alloc = widget->allocation;
#else
  gtk_widget_get_allocation(widget, &alloc);
#endif

  if( (event->x > alloc.x) &&
      (event->x < (alloc.x + alloc.width)) &&
      (event->y > alloc.y) &&
      (event->y < (alloc.y + alloc.height)) ){

    if( gtk_tree_view_get_path_at_pos( combo_box->treeview, (gint)event->x, (gint)event->y,
                                       &path, &col, NULL, NULL ) ){
      GtkTreeIter iter;
      if( gtk_tree_model_get_iter( combo_box->model, &iter, path ) ){

        if( simo_combo_box_is_seperator_row( combo_box, &iter ) )
          return FALSE;

        gtk_tree_view_row_activated( combo_box->treeview, path, col );
        g_signal_emit( combo_box, combo_box_signals[CLICKED], 0 );
      }
    }
  }

  return FALSE;
}


static
gboolean
simo_combo_box_treeview_motion_event( GtkWidget *widget,
                                      GdkEventMotion *event,
                                      SimoComboBox *combo )
{
  GtkTreePath *path;

  if( gtk_tree_view_get_path_at_pos( combo->treeview, (gint)event->x, (gint)event->y,
                                     &path, NULL, NULL, NULL ) ){
    GtkTreeIter iter;
    if( gtk_tree_model_get_iter( combo->model, &iter, path ) ){
      GtkTreeSelection *selection;
      selection = gtk_tree_view_get_selection( combo->treeview );
      gtk_tree_selection_select_iter( selection, &iter );
    }
    gtk_tree_path_free( path );
  }
  return FALSE;
}

static
void
simo_combo_box_row_activated( GtkTreeView *treeview,
                              GtkTreePath *path,
                              GtkTreeViewColumn *col,
                              SimoComboBox *combo )
{
  GtkTreeIter iter;
  //	g_print( "row activated\n" );

  if( gtk_tree_model_get_iter( combo->model, &iter, path ) ){
    if( simo_combo_box_is_seperator_row( combo, &iter ) )
      return;
    simo_combo_box_set_active_iter( combo, &iter );
    simo_combo_box_popup_close( combo );
  }
}


gboolean
simo_combo_box_get_active_iter( SimoComboBox *combo_box, GtkTreeIter *iter )
{
  if( combo_box->priv->current_path )
    return gtk_tree_model_get_iter( combo_box->model, iter, combo_box->priv->current_path );
  else
    return FALSE;
}


void simo_combo_box_set_active_iter( SimoComboBox *combo_box,
                                     GtkTreeIter *iter )
{
  SimoComboBoxPrivate *mthis = combo_box->priv;
  GValue value = {0};
  gchar *text = 0;

  if( mthis->current_path )
    gtk_tree_path_free( mthis->current_path );

  mthis->current_path = gtk_tree_model_get_path( combo_box->model, iter );
  if( mthis->current_path )
    gtk_tree_model_get_value( combo_box->model, iter, combo_box->visible_column, &value );

  if( G_IS_VALUE( &value ) )
    text = get_gvalue_as_string( &value );
  if( text ){

    gtk_entry_set_text( GTK_ENTRY( combo_box->entry ), text );
    gtk_label_set_text( GTK_LABEL( combo_box->label ), text );
    // TODO: elipsize text here when it does not fit in the button
    //gtk_button_set_label( GTK_BUTTON( combo_box->label ), text );

    g_free( text );
  }
  else{
    gtk_entry_set_text( GTK_ENTRY( combo_box->entry ), "" );
    gtk_label_set_text( GTK_LABEL( combo_box->label ), "" );
    //gtk_button_set_label( GTK_BUTTON( combo_box->label ), "" );
  }
  g_value_unset( &value );
}


void
simo_combo_box_insert_seperator( SimoComboBox *combo_box,
                                 GtkTreeIter *iter )
{
  gtk_tree_store_set( GTK_TREE_STORE( combo_box->model ), iter,
                      combo_box->visible_column, combo_box_seperator, -1 );
}


//EXPORT
GtkWidget *
create_simo_combo_box( gchar *widget_name,
                       gchar *button_name,
                       gchar *entry_name,
                       gint editable,
                       gint entry_alloc_width )
{
  GtkWidget *widget = simo_combo_box_new();
  //	g_print( "input: s1(%s), s2(%s), i1(%d), i2(%d)\n", string1, string2, int1, int2 );

  //	construct_simple_combo( SIMO_COMBO_BOX( widget ) );


  gtk_widget_set_name( widget, widget_name );

  simo_combo_box_set_editable( SIMO_COMBO_BOX( widget ), editable );

  if( button_name )
    gtk_widget_set_name( SIMO_COMBO_BOX( widget )->button, button_name );
  //	else
  //		gtk_widget_set_name( SIMO_COMBO_BOX( widget )->button, "combo_button" );

  if( entry_name )
    gtk_widget_set_name( SIMO_COMBO_BOX( widget )->entry, entry_name );
  //	else
  //		gtk_widget_set_name( SIMO_COMBO_BOX( widget )->button, "combo_entry" );


  if( entry_alloc_width > 25 ){
    //		g_print( "setting fixed entry size: %d\n", entry_alloc_width );
    SIMO_COMBO_BOX( widget )->priv->entry_alloc.width = entry_alloc_width;
    SIMO_COMBO_BOX( widget )->priv->fixed_entry_size = entry_alloc_width;
  }

  /*
    if( string1 && string2 ){
    if( !strcmp( string1, "editable" ) ){
    simo_combo_box_set_editable( SIMO_COMBO_BOX( widget ), !strcmp( string2, "TRUE" ) );
    }
    }
  */
  gtk_widget_show( widget );

  return widget;
}


void
simo_combo_box_set_editable( SimoComboBox *combo_box, gboolean editable_state )
{
  GtkWidget *widget = GTK_WIDGET( combo_box );
  GtkWidget *button = combo_box->priv->label_button;

  //	g_print( "ENTER: simo_combo_box_set_editable\n" );
  combo_box->editable = editable_state;
  //	gtk_editable_set_editable( GTK_EDITABLE( combo_box->entry ), editable_state );
  //	gtk_widget_set_sensitive( combo_box->entry, editable_state );
  //	g_object_set( G_OBJECT( combo_box->entry ), "can-focus", editable_state, NULL );

  if( editable_state ){
    if( widget == gtk_widget_get_parent(button) ){
      const gchar *text = gtk_label_get_text( GTK_LABEL( combo_box->label ) );

      g_object_ref( G_OBJECT( button ) );
      gtk_container_remove( GTK_CONTAINER( combo_box ), button );

      gtk_box_pack_start( GTK_BOX( combo_box ), combo_box->entry, TRUE, TRUE, 0 );
      g_object_unref( G_OBJECT( combo_box->entry ) );

      gtk_widget_hide( button );
      gtk_widget_show( combo_box->entry );

      if( text ){
        gtk_entry_set_text( GTK_ENTRY( combo_box->entry ), text );
      }
      //gtk_widget_set_size_request( GTK_WIDGET( combo_box->button ), 22, 22 );
    }
  }
  else
    if( widget == gtk_widget_get_parent(combo_box->entry) ){
      const gchar *text = gtk_entry_get_text( GTK_ENTRY( combo_box->entry ) );
      // save the entry allocation
      //memcpy( &(combo_box->priv->entry_alloc), &(combo_box->entry->allocation), sizeof( GtkAllocation ) );

      g_object_ref( G_OBJECT( combo_box->entry ) );
      gtk_container_remove( GTK_CONTAINER( combo_box ), combo_box->entry );
      //
      gtk_box_pack_start( GTK_BOX( combo_box ), button, TRUE, TRUE, 0 );
      g_object_unref( G_OBJECT( button ) );

      //gtk_widget_size_allocate( button, &combo_box->entry->allocation );
      gtk_widget_hide( combo_box->entry );
      gtk_widget_show( button );

      if( text ){
        gtk_label_set_text( GTK_LABEL( combo_box->label ), text );
      }
    }

  //	g_print( "LEAVE: simo_combo_box_set_editable\n" );
}


gboolean
simo_combo_box_get_editable( SimoComboBox *combo_box )
{
  return combo_box->editable;//gtk_editable_get_editable( GTK_EDITABLE( combo_box->entry ) );
}



/* Text access functions */
const gchar *
simo_combo_box_get_text(SimoComboBox *combo_box)
{
  if (combo_box->priv->n_lines > 1) {
    GtkTextIter start;
    GtkTextIter end;

    g_free(combo_box->priv->ml_text);
    gtk_text_buffer_get_bounds(combo_box->txtbuffer, &start, &end);
    combo_box->priv->ml_text = gtk_text_buffer_get_text(combo_box->txtbuffer,
                                                        &start,
                                                        &end,
                                                        FALSE);
    return combo_box->priv->ml_text;
  } else {
    return gtk_entry_get_text(GTK_ENTRY(combo_box->entry));
  }
}


void
simo_combo_box_set_text(SimoComboBox *combo_box,
                        const gchar  *text)
{
  if (combo_box->priv->n_lines > 1) {
    gtk_text_buffer_set_text(combo_box->txtbuffer, text, -1);
  } else {
    gtk_entry_set_text(GTK_ENTRY(combo_box->entry), text);
  }
}


void
simo_combo_box_set_visibility(SimoComboBox *combo_box,
                              gboolean      visible)
{
  gtk_entry_set_visibility(GTK_ENTRY(combo_box->entry), visible);
}


void
simo_combo_box_clear_text(SimoComboBox *combo_box)
{
  simo_combo_box_set_text(combo_box, "");
}


void
simo_combo_box_set_position(SimoComboBox *combo_box,
                            gint          position)
{
  if (combo_box->priv->n_lines > 1) {
    GtkTextIter iter;

    gtk_text_buffer_get_iter_at_offset(combo_box->txtbuffer, &iter, position);
    gtk_text_buffer_place_cursor(combo_box->txtbuffer, &iter);
  } else {
    gtk_editable_set_position(GTK_EDITABLE(combo_box->entry), position);
  }
}


gboolean
simo_combo_box_get_selection_bounds(SimoComboBox *combo_box,
                                    gint         *start,
                                    gint         *end)
{
  if (combo_box->priv->n_lines > 1) {
    GtkTextIter start_iter;
    GtkTextIter end_iter;
    gboolean    retval;

    retval = gtk_text_buffer_get_selection_bounds(combo_box->txtbuffer,
                                                  &start_iter, &end_iter);
    if (start) {
      *start = gtk_text_iter_get_offset(&start_iter);
    }
    if (end) {
      *end = gtk_text_iter_get_offset(&end_iter);
    }
    return retval;
  } else {
    return gtk_editable_get_selection_bounds(GTK_EDITABLE(combo_box->entry),
                                             start, end);
  }
}


void
simo_combo_box_select_region(SimoComboBox *combo_box,
                             gint          start,
                             gint          end)
{
  if (combo_box->priv->n_lines > 1) {
    GtkTextIter start_iter;
    GtkTextIter end_iter;

    gtk_text_buffer_get_iter_at_offset(combo_box->txtbuffer,
                                       &start_iter, start);
    gtk_text_buffer_get_iter_at_offset(combo_box->txtbuffer,
                                       &end_iter, end);
    gtk_text_buffer_select_range(combo_box->txtbuffer, &start_iter, &end_iter);
  } else {
    gtk_editable_select_region(GTK_EDITABLE(combo_box->entry), start, end);
  }
}


void
simo_combo_box_delete_selection(SimoComboBox *combo_box)
{
  if (combo_box->priv->n_lines > 1) {
    gtk_text_buffer_delete_selection(combo_box->txtbuffer, FALSE, TRUE);
  } else {
    gtk_editable_delete_selection(GTK_EDITABLE(combo_box->entry));
  }
}


void
simo_combo_box_cut_clipboard(SimoComboBox *combo_box)
{
  if (combo_box->priv->n_lines > 1) {
    GdkDisplay   *display;
    GtkClipboard *clipboard;

    display = gtk_widget_get_display(GTK_WIDGET(combo_box));

    clipboard = gtk_clipboard_get_for_display(display,
                                              GDK_SELECTION_CLIPBOARD);
    gtk_text_buffer_cut_clipboard(combo_box->txtbuffer, clipboard, TRUE);
  } else {
    gtk_editable_cut_clipboard(GTK_EDITABLE(combo_box->entry));
  }
}


void
simo_combo_box_copy_clipboard(SimoComboBox *combo_box)
{
  if (combo_box->priv->n_lines > 1) {
    GdkDisplay   *display;
    GtkClipboard *clipboard;

    display = gtk_widget_get_display(GTK_WIDGET(combo_box));

    clipboard = gtk_clipboard_get_for_display(display,
                                              GDK_SELECTION_CLIPBOARD);
    gtk_text_buffer_copy_clipboard(combo_box->txtbuffer, clipboard);
  } else {
    gtk_editable_copy_clipboard(GTK_EDITABLE(combo_box->entry));
  }
}


void
simo_combo_box_paste_clipboard(SimoComboBox *combo_box)
{
  if (combo_box->priv->n_lines > 1) {
    GdkDisplay   *display;
    GtkClipboard *clipboard;

    display = gtk_widget_get_display(GTK_WIDGET(combo_box));

    clipboard = gtk_clipboard_get_for_display(display,
                                              GDK_SELECTION_CLIPBOARD);
    gtk_text_buffer_paste_clipboard(combo_box->txtbuffer, clipboard,
                                    NULL, TRUE);
  } else {
    gtk_editable_paste_clipboard(GTK_EDITABLE(combo_box->entry));
  }
}


void
simo_combo_box_set_entry_font(SimoComboBox         *combo_box,
                              PangoFontDescription *fontDesc)
{
  gtk_widget_modify_font(combo_box->entry,   fontDesc);
  gtk_widget_modify_font(combo_box->txtview, fontDesc);
  combo_box->priv->cached_line_height = 0;
}



static
void
simo_combo_box_grab_focus_signal(GtkWidget *widget)
{
  SimoComboBox *combo_box = SIMO_COMBO_BOX(widget);

  if (combo_box->priv->n_lines > 1) {
    gtk_widget_grab_focus(combo_box->txtview);
  } else {
    gtk_widget_grab_focus(combo_box->entry);
  }
}


static
void
simo_combo_box_text_changed_cb(GtkEditable *editable,
                                gpointer     data)
{
  g_signal_emit(SIMO_COMBO_BOX(data), combo_box_signals[CHANGED], 0);
}


static
void
simo_combo_box_entry_activate_cb(GtkEditable *editable,
                                 gpointer     data)
{
  g_signal_emit(SIMO_COMBO_BOX(data), combo_box_signals[ACTIVATE], 0);
}


static
gboolean
simo_combo_box_txtview_keypress_cb(GtkWidget   *widget,
                                   GdkEventKey *evt,
                                   gpointer    data)
{
  if ((evt->keyval == GDK_Return || evt->keyval == GDK_KP_Enter)
      && !(evt->state & GDK_MOD1_MASK)
      && !(evt->state & GDK_SHIFT_MASK)
      && !(evt->state & GDK_CONTROL_MASK)) {
    g_signal_emit(SIMO_COMBO_BOX(data), combo_box_signals[ACTIVATE], 0);
    return TRUE;
  }

  return FALSE;
}


static
gboolean
simo_combo_box_arrow_up_cb(GtkWidget      *widget,
                           GdkEventButton *event,
                           gpointer        data)
{
  SimoComboBox *combo_box = (SimoComboBox *) data;

  if (event->button != 1) {
    return FALSE;
  }

  if (combo_box->priv->n_lines == combo_box->priv->max_lines) {
    return TRUE;
  }

  simo_combo_box_set_n_lines(combo_box, combo_box->priv->n_lines + 1);
  if (combo_box->priv->n_lines == 2) {
    gtk_widget_grab_focus(combo_box->txtview);
  }

  return TRUE;
}


static
gboolean
simo_combo_box_arrow_down_cb(GtkWidget      *widget,
                             GdkEventButton *event,
                             gpointer        data)
{
  SimoComboBox *combo_box = (SimoComboBox *) data;

  if (event->button != 1) {
    return FALSE;
  }

  if (combo_box->priv->n_lines == 1) {
    return TRUE;
  }

  simo_combo_box_set_n_lines(combo_box, combo_box->priv->n_lines - 1);
  if (combo_box->priv->n_lines == 1) {
    gtk_widget_grab_focus(combo_box->entry);
  }

  return TRUE;
}


/* Multi-line support */
gint
simo_combo_box_get_n_lines(SimoComboBox *combo_box)
{
  return combo_box->priv->n_lines;
}


void
simo_combo_box_set_n_lines(SimoComboBox *combo_box,
                           gint          n_lines)
{
  gint        start;
  gint        end;
  GtkTextIter start_iter;
  GtkTextIter end_iter;

  if (n_lines == combo_box->priv->n_lines
      || n_lines <= 0) {
    return;
  }

  if (combo_box->priv->n_lines == 1
      || (combo_box->priv->n_lines == 2 && n_lines == 1)) {
    if (n_lines > 1) {
      g_object_set(G_OBJECT(combo_box->entry), "no-show-all", TRUE, NULL);
      g_object_set(G_OBJECT(combo_box->priv->scrolled_window),
                   "no-show-all", FALSE, NULL);

      gtk_widget_hide(combo_box->entry);
      gtk_widget_show_all(combo_box->priv->scrolled_window);

      /* Copy text and selected part */
      g_signal_handler_block(G_OBJECT(combo_box->entry),
                             combo_box->priv->single_changed_id);
      gtk_text_buffer_set_text(combo_box->txtbuffer,
                               gtk_entry_get_text(GTK_ENTRY(combo_box->entry)),
                               -1);

      gtk_editable_get_selection_bounds(GTK_EDITABLE(combo_box->entry),
                                        &start, &end);
      gtk_text_buffer_get_iter_at_offset(combo_box->txtbuffer,
                                         &start_iter, start);
      gtk_text_buffer_get_iter_at_offset(combo_box->txtbuffer,
                                         &end_iter, end);
      gtk_text_buffer_select_range(combo_box->txtbuffer,
                                   &start_iter, &end_iter);

      g_signal_handler_unblock(G_OBJECT(combo_box->entry),
                               combo_box->priv->single_changed_id);
    } else {
      gchar *text;

      g_object_set(G_OBJECT(combo_box->entry), "no-show-all", FALSE, NULL);
      g_object_set(G_OBJECT(combo_box->priv->scrolled_window),
                   "no-show-all", TRUE, NULL);

      gtk_widget_hide(combo_box->priv->scrolled_window);
      gtk_widget_show(combo_box->entry);

      /* Copy text and selected part */
      g_signal_handler_block(G_OBJECT(combo_box->txtbuffer),
                             combo_box->priv->multi_changed_id);
      gtk_text_buffer_get_bounds(combo_box->txtbuffer, &start_iter, &end_iter);
      text = gtk_text_buffer_get_text(combo_box->txtbuffer,
                                      &start_iter, &end_iter,
                                      FALSE);
      gtk_entry_set_text(GTK_ENTRY(combo_box->entry), text);
      g_free(text);

      gtk_text_buffer_get_selection_bounds(combo_box->txtbuffer,
                                           &start_iter, &end_iter);
      start = gtk_text_iter_get_offset(&start_iter);
      end   = gtk_text_iter_get_offset(&end_iter);
      gtk_editable_select_region(GTK_EDITABLE(combo_box->entry), start, end);
      g_signal_handler_unblock(G_OBJECT(combo_box->txtbuffer),
                               combo_box->priv->multi_changed_id);
    }
  }

  combo_box->priv->n_lines = n_lines;
  gtk_widget_queue_resize(GTK_WIDGET(combo_box));
  g_signal_emit(combo_box, combo_box_signals[SIZE_CHANGED], 0);
}


/* Spell checking */
gboolean simo_combo_box_set_spell(SimoComboBox  *combo,
                                  gboolean       use_spell,
                                  gchar         *language,
                                  GError       **error)
{
#ifdef HAVE_GTKSPELL
  if (combo->priv->gtkspell) {   /* We already have one */
    if (!use_spell) {
      gtkspell_detach(combo->priv->gtkspell);
      combo->priv->gtkspell = NULL;
      return TRUE;
    } else {
      return gtkspell_set_language(combo->priv->gtkspell, language, error);
    }
  } else { /* We don't have one */
    if (use_spell) {
      combo->priv->gtkspell = gtkspell_new_attach(GTK_TEXT_VIEW(combo->txtview),
                                                  language,
                                                  error);
      if (!combo->priv->gtkspell) {
        return FALSE;
      }
      return TRUE;
    } else {
      return TRUE;
    }
  }


#else /* ! defined HAVE_GTKSPELL */
  return FALSE;
#endif
}
