/***************************************************************************
 * C++ Implementation:                                                     *
 * Copyright (C) 2012-2014 by Eduard Kalinowski                            *
 * Germany, Lower Saxony, Hanover                                          *
 * eduard_kalinowski@yahoo.de                                              *
 *                                                                         *
 * HTTraQt is free software; you can redistribute it and/or                *
 * modify it under the terms of the GNU Lesser General Public              *
 * License as published by the Free Software Foundation; either            *
 * version 3 of the License, or (at your option) any later version.        *
 *                                                                         *
 * HTTraQt is distributed in the hope that it will be useful, but          *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                    *
 *                                                                         *
 * See the GNU Lesser General Public License for more details.             *
 *                                                                         *
 * You should have received a copy of the GNU Lesser General Public        *
 * License along with HTTraQt. If not, see  http://www.gnu.org/licenses    *
 ***************************************************************************/

#include "../version.h"


#include <QtGui>
#include <QSystemTrayIcon>
#include <QApplication>
#include <QMenu>
#include <QDir>
#include <QFileDialog>
#include <QIcon>
#include <QTreeView>
#include <QSound>
#include <QDBusInterface>

#if USE_QT_VERSION == 5
// Qt5 part
#else
// Qt4 part
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <exception>
#include <assert.h>


#include "htinterface.h"
#include "htsstrings.h"

#include "includes/translator.h"

#include "includes/httraqt.h"

#include "ui_mainForm.h"

#include "includes/buttonPanel.h"
#include "../options/includes/OptionsDialog.h"
#include "includes/AboutDialog.h"
#include "includes/NewProjTab.h"
#include "includes/OptionsTab.h"
#include "includes/StartTab.h"
#include "includes/ConfirmTab.h"
#include "includes/ProgressTab.h"
#include "includes/FinalTab.h"


class OptionsDialog;

class ProgressTab;


bool writeWIniFile(QIODevice &device, const QSettings::SettingsMap &map)
{
    QDataStream out(&device);

    if ( device.isOpen () && ! map.empty() ) {
        QMap<QString, QVariant>::const_iterator it;

        for ( it = map.begin(); it != map.end(); ++it ) {
            QString buffer;
            QString val = it.value().toString();

            if (val.length() > 0) {
                if (val.indexOf("\"") == 0) {
                    val.remove(0, 1);
                }

                if (val.lastIndexOf("\"") == val.length() - 1) {
                    val.remove(val.length() - 1, 1);
                }

                val.remove("%0d%0a");
            }

            buffer = QString ( "%1=%2\n" ).arg ( it.key(), val );

            if ( ! buffer.isEmpty() ) {
                device.write ( buffer.toStdString().c_str() );
            }

            buffer.clear();
        }

        return true;
    }

    return false;
}


bool readWIniFile(QIODevice &device, QSettings::SettingsMap &map)
{
    QDataStream in(&device);

    if ( device.isOpen () ) {
        if ( !map.empty() ) {
            map.clear();
        }

        while ( ! device.atEnd() ) {
            QByteArray line = device.readLine().trimmed();

            if ( line.startsWith ( "#" )) {
                continue;
            }

            if ( !line.contains ( "=" )) {
                continue;
            }

            int pos = line.indexOf("=");
            QString key = line.left(pos);
            QString val = line.mid(pos + 1);

            if (val.length() > 0) {
                if (val.indexOf("\"") == 0) {
                    val.remove(0, 1);
                }

                if (val.lastIndexOf("\"") == val.length() - 1) {
                    val.remove(val.length() - 1, 1);
                }

                val.remove("%0d%0a");
            }

            if ( ! key.isEmpty() ) {
                if (val.length() > 0) {
                    if ( val.contains ( QRegExp ( "^[0-9]+$" ) ) ) {
                        map.insert ( key, QVariant ( val ).toInt() );
                    } else {
                        map.insert ( key, QVariant ( val ).toString() );
                    }
                }
            }
        }

        return true;
    }

    return false;
}

HTTraQt* mainWidget;

static const QSettings::Format WIniFormat = QSettings:: registerFormat ("ini", readWIniFile, writeWIniFile);

using namespace std;


void MessageTimerBox::showEvent ( QShowEvent * event )
{
    currentTime = 0;

    if (autoClose) {
        this->startTimer(1000);
    }
}


void MessageTimerBox::setDefaultText(const QString &t)
{
    defaultText = t;
}


void MessageTimerBox::setAutoClose(bool b)
{
    autoClose = b;
}


void MessageTimerBox::setTimeout(int t)
{
    timeout = t;
    QString tx;

    tx = defaultText;
    tx.replace("%d", QString::number(timeout));
    setText(tx);
}


void MessageTimerBox::timerEvent(QTimerEvent *event)
{
    QString t;
    currentTime++;
    t = defaultText;
    t.replace("%d", QString::number(timeout - currentTime));

    setText(t);

    if (currentTime >= timeout) {
        this->done(0);
    }
}


int MessageBox::exec(const QString &title, const QString &text, int ticon)
{
    int ret;
    QMessageBox* msgBox = NULL;

    if (ticon == QMessageBox::Question) {
        msgBox = new QMessageBox(QMessageBox::Question, title, text, QMessageBox::Yes | QMessageBox::No);

        msgBox->setButtonText(QMessageBox::Yes, translate(_YES));
        msgBox->setButtonText(QMessageBox::No, translate(_NO));
    }

    if (ticon == QMessageBox::Critical || ticon == QMessageBox::Warning || QMessageBox::Information) {
        if (ticon == QMessageBox::Critical) {
            msgBox = new QMessageBox(QMessageBox::Critical, title, text, QMessageBox::Ok);
        }

        if (ticon == QMessageBox::Warning) {
            msgBox = new QMessageBox(QMessageBox::Warning, title, text, QMessageBox::Ok);
        }

        if (ticon == QMessageBox::Information) {
            msgBox = new QMessageBox(QMessageBox::Information, title, text, QMessageBox::Ok);
        }

        msgBox->setButtonText(QMessageBox::Ok, translate(_OK));
    }

    if (msgBox == NULL) {
        //         qDebug() << "not found dialog" << ticon;
        return 0;
    }

    if (mainWidget->programStyleSheet.length() > 0) {
        msgBox->setStyleSheet(mainWidget->programStyleSheet);
    }

    ret = msgBox->exec();

    delete msgBox;
    //     qDebug() << "mydialog" << ret;
    return ret;
}


void MyThread::run()
{
    httraq_main();
}


QVariant DirModel::headerData ( int section, Qt::Orientation orientation, int role ) const
{
    if (orientation == Qt::Horizontal) {
        if (role != Qt::DisplayRole) {
            return QVariant();
        }

        switch (section) {
            case 0:
                return mainWidget->currentWorkDir;

            default:
                return QVariant();
        }
    }
}


void HTTraQt::launch()
{
    cmdArgumentsOptions();

    writeSettings(false);

    initSInfo();

    // wait time, if entered
    QString hh, mm, ss;
    bool okHH, okMM, okSS;
    int  intHH, intMM, intSS;
    intHH = intMM = intSS = 0;

    hh = (static_cast<ConfirmTab*>(widgets[3]))->labelHH->text(); // HH
    mm = (static_cast<ConfirmTab*>(widgets[3]))->labelMM->text(); // MM
    ss = (static_cast<ConfirmTab*>(widgets[3]))->labelSS->text(); // SS

    pcShutdown = false;
    pcHibernate = false;

    if ((static_cast<ConfirmTab*>(widgets[3]))->groupPCoff->isChecked() == true) {
        pcShutdown = (static_cast<ConfirmTab*>(widgets[3]))->radioShutdown->isChecked();
        pcHibernate = (static_cast<ConfirmTab*>(widgets[3]))->radioHibernate->isChecked();
    }

    if (hh.length() > 0) {
        intHH = hh.toInt(&okHH);
    }

    if (intHH < 0 || intHH > 23) {
        okHH = false;
        intHH = 0;
    }

    if (mm.length() > 0) {
        intMM = mm.toInt(&okMM);
    }

    if (intMM < 0 || intMM > 59) {
        okMM = false;
        intMM = 0;
    }

    if (ss.length() > 0) {
        intSS = ss.toInt(&okSS);
    }

    if (intSS < 0 || intSS > 59) {
        okSS = false;
        intSS = 0;
    }

    if (intHH > 0 || intMM > 0 || intSS > 0) {
        int mSec = 1000 * (intHH * 3600 + intMM * 60 + intSS);
        QTimer::singleShot(mSec, this, SLOT(onStopAll()));
    }

    int result = 0;
    {
        QString projectDir;
        projectDir =  currentWorkDir + "/" + currentProject + "/";

        // on efface le doit.log, pour annuler les parametres anciens et en redéfinir de nouveaux
        // c'est ici une logique qui diffère de la version en ligne de commande
        if (QFile::exists(projectDir + "hts-cache/new.zip") || QFile::exists(projectDir + "hts-cache/new.ndx")) {    // un cache est présent
            QFile fl(projectDir + "hts-cache/doit.log");

            if (fl.exists()) {
                fl.remove();
            }

            if (fl.open(QFile::WriteOnly) == true) {
                fl.close();
            }
        }
    }


    //     if (global_opt != NULL) {
    //         hts_free_opt(global_opt);
    //         global_opt = NULL;
    //     }

    termine = 0;
    termine_requested = 0;

    //     global_opt = hts_create_opt();

    timerProgressUpdate->start(1000); //progress tab ones per second to update

    timerDirRefresh->start(10000); // ones per 10 seconds


    mth = new MyThread;

    connect(mth, SIGNAL(finished ()), this, SLOT(threadFinished()));
    connect(mth, SIGNAL(terminated ()), this, SLOT(threadFinished()));

    mth->start();
}


void HTTraQt::refreshDirModel()
{
    dModel.refresh();
    statusBar()->showMessage("Update dir list", 1000);
}


HTTraQt::HTTraQt(QWidget* parent, Qt::WindowFlags flags)
    : QMainWindow(parent, flags)
{
    setupUi(this);

    mainWidget = this;
    
    currentAppDir = qApp->applicationDirPath();

    if (currentAppDir.lastIndexOf("/build") > 0) { // build dir detection
        currentAppDir.remove("/build");
    }

    currentLang = "English";

    _singular = new QSharedMemory("HTTraQt", this);


    QFont sysFont = qApp->font();
    sysFont = sysFont;

    fontSize = sysFont.pointSize();

    mth = NULL;

    verbose = true; // for hibernate and shutdown debug messages only!

    readGUISettings();

    programStyleSheet = QString().sprintf("font-size: %dpt", fontSize);

    if ( fontSize == -1) {
        fontSize = sysFont.pixelSize();
        programStyleSheet = QString().sprintf("font-size: %dpx", fontSize);
    }

    if (programStyleSheet.length() > 0) {
        setStyleSheet(programStyleSheet);
    }


    if (readLangDir() == false) { // init from langFiles variable in format "filename:language"
        MessageBox::exec(translate(_ERR),
                         "Directory with other languages not found\nDefault GUI language is english", QMessageBox::Critical);
    }

    currentTab = 0;

    groupMain->setFont(sysFont);
    groupButtons->setFont(sysFont);
    dirView->setFont(sysFont);

    // this function i need only for converting! normally is disabled
    // old help file format -> new
    //     convertTranslateFile();

    buttonsWidget = new buttonPanel(this);
    gridLayoutB->addWidget(buttonsWidget);

    widgets[0] = new StartTab(this);
    gridLayoutM->addWidget(widgets[0]);

    widgets[1] = new NewProjTab(this);
    gridLayoutM->addWidget(widgets[1]);

    widgets[2] = new OptionsTab(this);
    gridLayoutM->addWidget(widgets[2]);

    widgets[3] = new ConfirmTab(this);
    gridLayoutM->addWidget(widgets[3]);

    widgets[4] = new ProgressTab(this);
    gridLayoutM->addWidget(widgets[4]);

    widgets[5] = new FinalTab(this);
    gridLayoutM->addWidget(widgets[5]);

    setFontForWidgets();

    timerProgressUpdate = new QTimer(this);
    timerDirRefresh =  new QTimer(this);

    connect(timerProgressUpdate, SIGNAL(timeout()), widgets[4], SLOT(update()));
    connect(timerDirRefresh, SIGNAL(timeout()), this, SLOT(refreshDirModel()));

    // flag de termination

    termine = 0;
    process = NULL;
    termine_requested = 0;
    shell_terminated = 0;
    soft_term_requested = 0;

#if USE_QT_VERSION == 5
//     QImage();
    setWindowIcon(QIcon(QPixmap::fromImage(QImage(httraqt_icon))));
#else
    setWindowIcon(QIcon(httraqt_icon));
#endif
    
    dModel.setReadOnly(true);

    dirView->setModel(&dModel);

    setCentralWidget(centralwidget);

    dirView->setColumnHidden(1, true);
    dirView->setColumnHidden(2, true);
    dirView->setColumnHidden(3, true);
    dirView->setAnimated(false);
    dirView->setIndentation(20);
    dirView->setSortingEnabled(true);

    connect(dirView, SIGNAL(expanded(const QModelIndex &)), this, SLOT(treeItemClicked(const QModelIndex &)));
    connect(dirView, SIGNAL(collapsed(const QModelIndex &)), this, SLOT(treeItemClicked(const QModelIndex &)));

    resetDefaultOptions();

    readSettings(true); // global settings

    statusBar()->showMessage("Current directory: " + currentWorkDir);

    createActions();
    createToolBars();
    createStatusBar();

    createTrayIcon();

    if (getLangTable() == false) {
        MessageBox::exec(translate(_ERR),
                         "Can't open language file!\nDefault GUI language is english", QMessageBox::Critical);

        currentLang = "English";
    }

    for (QVector<QAction*>::iterator itL = actLangSelect.begin(); itL != actLangSelect.end(); ++itL) {
        if ((*itL)->text() == currentLang) {
            (*itL)->setChecked(true);
            break;
        }
    }

#if 0

    if(checkInstanceRuns() == true) {
        int res = QMessageBox::question(0, "Warning", "Application already running\nDo you want to continue?", QMessageBox::Yes | QMessageBox::No);

        if (res == QMessageBox::No) {
            exit(-1);
        }
    }

#endif
    setLangGUI();

    activatePage(0);
}


// void HTTraQt::resizeTabs(QResizeEvent* se)
// {
//     qDebug() << "resize";
// }

void HTTraQt::setFontForWidgets()
{
    buttonsWidget->setStyleSheet(programStyleSheet);

    for (int i = 0; i < 6; i++) {
        widgets[i]->setStyleSheet(programStyleSheet);
    }
}


void HTTraQt::clearStatsBuffer(void)
{
    // SInfo.refresh
    for (int i = 0; i < NStatsBuffer; i++) {
        StatsBuffer[i].strStatus = ""; // etat
        StatsBuffer[i].name = "";  // nom
        StatsBuffer[i].file = "";  // fichier
        StatsBuffer[i].url_sav = "";
        StatsBuffer[i].back = NULL;
        StatsBuffer[i].size = 0;
        StatsBuffer[i].sizeTotal = 0;
    }
}


void HTTraQt::threadFinished(void)
{
    if (global_opt != NULL) {
        hts_free_opt(global_opt);
        global_opt = NULL;
    }

    buttonsWidget->onNext(); // to log page
    onEndMirror();

    timerProgressUpdate->stop();
    timerDirRefresh->stop();

    /* Aborted mirror or finished? */
    {
        QString projectDir;
        projectDir =  currentWorkDir + "/" + currentProject + "/";

        if (soft_term_requested || termine_requested) {
            QFile fl(projectDir + "hts-cache/interrupted.lock");

            if (fl.open(QFile::WriteOnly) == true) {
                fl.close();
            }
        } else {
            QFile::remove(projectDir + "hts-cache/interrupted.lock");
        }
    }

    delete mth;
    mth = NULL;

    if (pcShutdown == true) {
        shutdown();
    }

    if (pcHibernate == true) {
        hibernate();
    }
}


//
bool HTTraQt::treeItemClicked(const QModelIndex &m)
{
    //     dirView->resizeColumnToContents(0);
}


// true if new project
bool HTTraQt::rebuildWorkDirView()
{
    bool newProj = false;

    if (currentWorkDir.length() == 0) {
        //         currentWorkDir = QDir::homePath() + "/My Web Sites";
        readSettings(true);
    }

    QDir wDir(currentWorkDir);

    if (wDir.exists() == false) {
        wDir.mkpath(currentWorkDir);
        newProj = true;
    }

    dirView->reset();
    dirView->setRootIndex(dModel.index(currentWorkDir));
    dModel.refresh();

    // something in hts-cache?
    if (checkContinue(false) == false) {
        newProj = true;
    }

    return newProj;
}


void HTTraQt::initSInfo()
{
    //     SInfo.ask_refresh = 0;
    SInfo.refresh = 0;
    SInfo.stat_bytes = 0;
    SInfo.stat_time = 0;
    SInfo.lien_n = 0;
    SInfo.lien_tot = 0;
    SInfo.stat_nsocket = 0;
    SInfo.rate = 0;
    SInfo.irate = 0;
    SInfo.ft = 0;
    SInfo.stat_written = 0;
    SInfo.stat_updated = 0;
    SInfo.stat_errors = 0;
    SInfo.stat_warnings = 0;
    SInfo.stat_infos = 0;
    SInfo.stat_timestart = mtime_local();
    SInfo.stat_back = 0;
}


// reprise possible?
bool HTTraQt::checkContinue(bool msg)
{
    QString projectDir;
    projectDir =  currentWorkDir + "/" + currentProject + "/";

    if (QFile::exists(projectDir + "hts-cache/new.zip")  ||
            ((QFile::exists(projectDir + "hts-cache/new.dat")) && (QFile::exists(projectDir + "hts-cache/new.ndx")))) {  // il existe déja un cache précédent.. renommer
        return true;
    }

    if (QFile::exists(projectDir + "hts-cache/old.zip") ||
            ((QFile::exists(projectDir + "hts-cache/old.dat")) && (QFile::exists(projectDir + "hts-cache/old.ndx")))) {  // il existe déja un ancien cache précédent.. renommer
        return true;
    }

    if ( msg == true) {
        MessageBox::exec(translate(_ERR), translate(_CANTFIND), QMessageBox::Critical);
    }

    return false;
}


void HTTraQt::afterChangepathlog()
{
    QString st = "";
    char tempo[8192];
    bool modify;
    QString projectDir;
    projectDir =  currentWorkDir + "/" + currentProject + "/";
#if 0

    if (fexist(fconcat(catbuff, tempo, "winprofile.ini"))) {  // un cache est présent
        if (fsize(fconcat(catbuff, tempo, "winprofile.ini")) > 0) { // taille log contrôle>0
            int i;

            for(i = 0; i < (int) strlen(tempo); i++) {
                if (tempo[i] == '/') {
                    tempo[i] = '\\';
                }
            }

            Read_profile(fconcat(catbuff, tempo, "winprofile.ini"), 0);

            // peut on modifier?
            int pos = m_ctl_todo.GetCurSel();

            if ((pos == LAST_ACTION) || (pos == LAST_ACTION - 1) || (pos == 0)) {
                modify = true;
            } else {
                modify = false;
            }

            // existe: update
            if (fexist(fconcat(catbuff, tempo, "hts-cache/new.zip")) ||
                    (fexist(fconcat(catbuff, tempo, "hts-cache/new.dat"))) && (fexist(fconcat(catbuff, tempo, "hts-cache/new.ndx")))
               ) {  // il existe déja un cache précédent.. renommer
                if (modify) {
                    if ((!fexist(fconcat(catbuff, tempo, "hts-in_progress.lock"))) &&
                            (!fexist(fconcat(catbuff, tempo, "hts-cache/interrupted.lock")))
                       ) {
                        m_ctl_todo.SetCurSel(LAST_ACTION);
                    } else {
                        m_ctl_todo.SetCurSel(LAST_ACTION - 1);
                    }
                }

                log_flip = 1;
            } else if (log_flip) {
                if (modify) {
                    m_ctl_todo.SetCurSel(0);
                }

                log_flip = 0;
            }

            OnSelchangetodo();
        }
    } else if (log_flip) {
        m_ctl_todo.SetCurSel(0);
        log_flip = 0;
    }

#endif
}



void HTTraQt::renameOldToNew()
{
    QString projectDir;
    projectDir =  currentWorkDir + "/" + currentProject + "/";

    if (QFile::exists(projectDir + "hts-cache/old.dat") && QFile::exists(projectDir + "hts-cache/old.ndx")) {
        if (QFile::remove(projectDir + "hts-cache/new.dat") == false) {
            MessageBox::exec(translate(_ERR), translate(_ERRDELETING), QMessageBox::Warning);
        }

        if ( QFile::remove(projectDir + "hts-cache/new.ndx") == false) {
            MessageBox::exec(translate(_ERR), translate(_ERRDELETING), QMessageBox::Warning);
        }
    }

    if (QFile::remove(projectDir + "hts-cache/new.lst") == false ) {
        MessageBox::exec(translate(_ERR), translate(_ERRDELETING), QMessageBox::Warning);
    }

    if (QFile::exists(projectDir + "hts-cache/old.zip")) {
        if (QFile::remove(projectDir + "hts-cache/new.zip") == false ) {
            MessageBox::exec(translate(_ERR), translate(_ERRDELETING), QMessageBox::Warning);
        }
    }

    QFile::remove(projectDir + "hts-cache/new.txt");
    QFile::rename(projectDir + "hts-cache/old.zip", projectDir + "hts-cache/new.zip");
    QFile::rename(projectDir + "hts-cache/old.dat", projectDir + "hts-cache/new.dat");
    QFile::rename(projectDir + "hts-cache/old.ndx", projectDir + "hts-cache/new.ndx");
    QFile::rename(projectDir + "hts-cache/old.lst", projectDir + "hts-cache/new.lst");
    QFile::rename(projectDir + "hts-cache/old.txt", projectDir + "hts-cache/new.txt");
}


QString HTTraQt::sizeToText(long long int s)
{
    QString t;

    if (s < 0) {
        return QString(" size is wrong!");
    }

    if (s > 1024 && s < 1024 * 1024) {
        t = QString::number(s / 1024) + " kB";
    } else if (s > (1024 * 1024)) {
        t = QString::number(s / (1024 * 1024)) + " MB";
    } else {
        t = QString::number(s / 1024) + " B";
    }

    return t;
}


void HTTraQt::onEndMirror()
{
    QStringList ext;
    QString oldName, oldShort;
    QString newName, newShort;
    long int oldSize = -1;
    long int newSize = -1;

    //     endInProgress();

    ext << "zip" << "dat" << "ndx";

    QString projectDir;
    projectDir =  currentWorkDir + "/" + currentProject + "/";

    for (QStringList::iterator iExt = ext.begin(); iExt != ext.end(); ++iExt) {
        QFile oldF;
        QFile newF;
        oldName = projectDir + "hts-cache/old." + (*iExt);
        oldShort = "old."  + (*iExt);
        newName = projectDir + "hts-cache/new." + (*iExt);
        newShort = "new."  + (*iExt);

        oldF.setFileName (oldName);
        newF.setFileName (newName);

        if (oldF.exists() == true && newF.exists() == true ) {
            oldSize = oldF.size();
            newSize = newF.size();
        }

        if (oldSize > 0 && newSize > 0) {
            break;
        }
    }

    if (oldSize == -1) { // nothing to overwrite, project was new
        return;
    }

    if (newSize == -1) {
        MessageBox::exec(translate(_ERR), translate(_MIRROR_ERR), QMessageBox::Warning);

        return; // nothing was found
    }
}

void HTTraQt::contextMenuEvent(QContextMenuEvent *)
{
#if 0
    QMenu* contextMenu = new QMenu("boo", this);
    Q_CHECK_PTR(contextMenu);
    QString caption = QString("<font color=darkblue><u><b>"
                              "Context Menu</b></u></font>");
    //caption->setAlignment( Qt::AlignCenter );
    // contextMenu->setTitle( "boo" );
    contextMenu->addMenu(QString("&New"));            //,  SLOT(news()), QKeySequence(Qt::CTRL+Qt::Key_N) );
    contextMenu->addMenu(QString("&Open..."));            //, this, SLOT(open()), CTRL+Key_O );
    contextMenu->addMenu(QString("&Save"));            //, this, SLOT(save()), CTRL+Key_S );

    QMenu *submenu = new QMenu(this);
    Q_CHECK_PTR(submenu);
    submenu->addMenu(QString("&Print to printer"));            //, this, SLOT(printer()) );
    submenu->addMenu(QString("Print to &file"));            //, this, SLOT(file()) );
    submenu->addMenu(QString("Print to fa&x"));            //, this, SLOT(fax()) );
    contextMenu->addMenu(QString("&Print"));            //, submenu );
    contextMenu->exec(QCursor::pos());
    delete contextMenu;
#endif
}


void HTTraQt::displayProgressMsg()
{
    QByteArray msg;
    QString result;
    //   QProcess* actualP = NULL;

    process->setReadChannel(QProcess::StandardError);
    msg = process->readAllStandardError();
    //     if ( msg.length() > 1 ) {
    result = msg.data();
    //       actualP = process;
    //     qDebug() << "displayProgressMsg:" << result ;
    //     }

    //   if ( result.length() < 1 )
    //     return;
}

void HTTraQt::displayOutputMsg()
{
    QString line;

    process->setReadChannel(QProcess::StandardOutput);

    while (process->canReadLine()) {
        line = process->readLine().trimmed();
    }

    //     qDebug() << "displayOutputMsg:" << line ;
}


void HTTraQt::processFinished(int exitCode, QProcess::ExitStatus exitStatus)
{
}


void HTTraQt::createActions()
{
    //     newProjAct->setShortcut ( tr ( "Ctrl+N" ) );
    newProjAct->setStatusTip(translate(_OPENGUI));
    connect(newProjAct, SIGNAL(triggered()), this, SLOT(newProject()));

    //     openProjAct->setShortcut ( tr ( "Ctrl+O" ) );
    //     openProjAct->setStatusTip(translate("Open project"));
    //     connect(openProjAct, SIGNAL(triggered()), this, SLOT(openProject()));

    //     saveProjAct->setShortcut ( tr ( "Ctrl+S" ) );
    //     saveProjAct->setStatusTip(translate("Save project"));
    //     connect(saveProjAct, SIGNAL(triggered()), this, SLOT(saveProject()));

    //     delProjAct->setShortcut ( tr ( "Ctrl+D" ) );
    //     delProjAct->setStatusTip(translate("Delete a project"));
    //     connect(delProjAct, SIGNAL(triggered()), this, SLOT(deleteProject()));


    browseAct->setShortcut ( tr ( "Ctrl+B" ) );
    browseAct->setStatusTip(translate(_BROWSEEXISTS));
    connect(browseAct, SIGNAL(triggered()), this, SLOT(browseSites()));

    //     loadDefOptAct->setShortcut ( tr ( "Ctrl+L" ) );
    loadDefOptAct->setStatusTip(translate(_LOADDEF));
    connect(loadDefOptAct, SIGNAL(triggered()), this, SLOT(loadDefaultOptions()));

    saveOptAct->setStatusTip(translate(_SAVEDEF));
    connect(saveOptAct, SIGNAL(triggered()), this, SLOT(saveDefaultOptions()));

    //     resetOptToDefAct->setShortcut ( tr ( "Ctrl+R" ) );
    resetOptToDefAct->setStatusTip(translate(_RESETTODEF));
    connect(resetOptToDefAct, SIGNAL(triggered()), this, SLOT(resetToDefault()));

    loadOptAct->setStatusTip(translate(_LOADOPT));
    connect(loadOptAct, SIGNAL(triggered()), this, SLOT(loadOptionsFromFile()));

    //     recentAct->setShortcut ( tr ( "Ctrl+R" ) );
    //     recentAct->setStatusTip(translate("Recent file"));
    //     connect(recentAct, SIGNAL(triggered()), this, SLOT(recent()));

    //     exitAct->setShortcut ( tr ( "Ctrl+X" ) );
    exitAct->setStatusTip(translate(_EXIT));
    connect(exitAct, SIGNAL(triggered()), this, SLOT(quit()));

    saveOptAsAct->setStatusTip(translate(_SAVEOPTAS));
    connect(saveOptAsAct, SIGNAL(triggered()), this, SLOT(saveOptionsAs()));

    //     selectFontAct->setStatusTip(translate(_SEL_FONT));
    //     connect(selectFontAct, SIGNAL(triggered()), this, SLOT(selectFontSize()));

    // fontSizePrefAct->setStatusTip(translate(_SEL_FONT));
    langPrefAct->setStatusTip(translate(_LANGPREF));
    //     fontPrefAct->setStatusTip(translate(_SEL_FONT));
    //     connect(langPrefAct, SIGNAL(triggered()), this, SLOT(langPreference()));

    //     modifyOptAct->setShortcut ( tr ( "Ctrl+M" ) );
    modifyOptAct->setStatusTip(translate(_MODIFYOPT));
    connect(modifyOptAct, SIGNAL(triggered()), this, SLOT(modifyOptions()));

    //     pauseOptAct->setShortcut ( tr ( "Ctrl+P" ) );
    pauseOptAct->setStatusTip(translate(_PAUSETRANSF));
    connect(pauseOptAct, SIGNAL(triggered()), this, SLOT(pauseTransfer()));

    viewLogAct->setStatusTip(translate(_VIEW_LOG));
    connect(viewLogAct, SIGNAL(triggered()), this, SLOT(viewLog()));

    viewErrLogAct->setStatusTip(translate(_VIEWERRLOG));
    connect(viewErrLogAct, SIGNAL(triggered()), this, SLOT(viewErrorLog()));

    //     viewTransferAct->setStatusTip(translate(_VIEWFILETRANSFERS));
    //     connect(viewTransferAct, SIGNAL(triggered()), this, SLOT(viewTransfers()));

    //     hideAct->setShortcut ( tr ( "Ctrl+H" ) );
    hideAct->setStatusTip(translate(_HIDE));
    connect(hideAct, SIGNAL(triggered()), this, SLOT(hideWindow()));

    restAct = new QAction(translate(_OPEN), this);
    //     restAct->setStatusTip(translate(_RESTORE));
    connect(restAct, SIGNAL(triggered()), this, SLOT(restoreWindow()));

    //     winStatusAct->setStatusTip(translate("Status Bar"));
    //     connect(winStatusAct, SIGNAL(triggered()), this, SLOT(windowStatusBar()));

    //     checkAct->setShortcut ( tr ( "Ctrl+U" ) );
    checkAct->setStatusTip(translate(_TOPROJECTPAGE));
    connect(checkAct, SIGNAL(triggered()), this, SLOT(checkUpdates()));

    contentAct->setStatusTip(translate(_CONTENTS));
    connect(contentAct, SIGNAL(triggered()), this, SLOT(contens()));

    stepByStepAct->setStatusTip(translate(_STEPBYSTEP));
    connect(stepByStepAct, SIGNAL(triggered()), this, SLOT(stepByStep()));

    //     aboutAct->setShortcut ( tr ( "Ctrl+A" ) );
    aboutAct->setStatusTip(translate(_ABOUTPROG));
    connect(aboutAct, SIGNAL(triggered()), this, SLOT(about()));

    aboutQtAct->setStatusTip(translate(_ABOUTQT));
    aboutQtAct->setFont(sysFont);
    connect(aboutQtAct, SIGNAL(triggered()), this, SLOT(aboutQt()));
}


void HTTraQt::createTrayIcon()
{
    if (!QSystemTrayIcon::isSystemTrayAvailable()) {
        hideAct->setEnabled(false);
        return;
    }

    QApplication::setQuitOnLastWindowClosed(false);

    trayIconMenu = new QMenu(this);

    trayIconMenu->addAction(restAct);
    trayIconMenu->addSeparator();

    trayIconMenu->addAction(modifyOptAct);
    trayIconMenu->addAction(viewLogAct);
    trayIconMenu->addAction(viewErrLogAct); // view err log
    trayIconMenu->addSeparator();

    trayIconMenu->addAction(pauseOptAct); // pause
    trayIconMenu->addAction(aboutAct); // about
    trayIconMenu->addAction(exitAct);

    trayIcon = new QSystemTrayIcon(this);

    connect(trayIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)),
            this, SLOT(iconActivated(QSystemTrayIcon::ActivationReason)));

    trayIcon->setContextMenu(trayIconMenu);

#if USE_QT_VERSION == 5
//     QImage();
    trayIcon->setIcon(QIcon(QPixmap::fromImage(QImage(httraqt_icon))));
#else
    trayIcon->setIcon(QIcon(httraqt_icon));
#endif
    
  
}


void HTTraQt::iconActivated(QSystemTrayIcon::ActivationReason reason)
{
    switch (reason) {
        case QSystemTrayIcon::Trigger:
            //             qDebug() << "trigger";
            break;

        case QSystemTrayIcon::DoubleClick:
            //             qDebug() << "double";
            emit restoreWindow();
            //          iconComboBox->setCurrentIndex((iconComboBox->currentIndex() + 1)
            //                                        % iconComboBox->count());
            break;

        case QSystemTrayIcon::MiddleClick:
            //             qDebug() << "middle";
            //          showMessage();
            break;

        default:
            ;
    }
}


void HTTraQt::hideWindow()
{
    // to system tray
    trayIcon->show();
    hide();
}


void HTTraQt::restoreWindow()
{
    // to normal
    trayIcon->hide();
    showNormal();
}


void HTTraQt::newProject()
{
    QString app;
    app = qApp->applicationDirPath() + "/httraqt";

    QProcess *myProcess = new QProcess();
    myProcess->start(app);
}


void HTTraQt::browseSites()
{
    QString name = currentWorkDir + "/" + currentProject + "/index.html";

    if (QFile::exists(name) == true ) {
        QDesktopServices::openUrl(QUrl::fromLocalFile(qPrintable( name )));
    }
}


void HTTraQt::loadDefaultOptions()
{
    int ans = MessageBox::exec(translate(_QUEST), translate(_LOADDEFAULT), QMessageBox::Question);

    if (ans == QMessageBox::Yes) {
        readSettings(true);
    }
}


void HTTraQt::saveDefaultOptions()
{
    int ans = MessageBox::exec(translate(_QUEST), translate(_SAVEDEFAULT), QMessageBox::Question);

    if (ans == QMessageBox::Yes) {
        writeSettings(true); // global options
    }
}


void HTTraQt::resetToDefault()
{
    int ans = MessageBox::exec(translate(_QUEST), translate(_RESETDEFAULT), QMessageBox::Question);

    if (ans == QMessageBox::Yes) {
        resetDefaultOptions();
    }
}


void HTTraQt::saveOptions(QSettings *s, bool gl)
{
    QString os, st;
    GetProfile("BrowserID", st);
    GetProfile("osID", os);

    if (st.length() > 0) {
        if (st.contains("%s")) {
            st.replace("%s", os);
        }

        SetProfile("UserID", st);
    }

    for (QMap<QString, OptStruct>::const_iterator iopt = prOptions.constBegin(); iopt != prOptions.constEnd(); ++iopt) {
        if (gl == true) { // do not save as global
            if (iopt.key() == "CurrentUrl" || iopt.key() == "CurrentURLList") { // only in project settings
                continue;
            }
        }

        if ((*iopt).type == TEXT) {
            QString v = (*iopt).value.toString();
            profile_code(v);

            s->setValue(iopt.key(), v);
        }

        if ((*iopt).type == FLOAT) {
            if ((*iopt).value.toFloat() >= 0) {
                float fTmp = (*iopt).value.toFloat();
                QString fTxt;
                fTxt = QString().sprintf("%6.4f", fTmp);
                //qDebug() << "save" << (*iopt).name << fTxt;
                s->setValue(iopt.key(), (qint64)( fTxt.toFloat() * 1024.0 * 1024.0));
            } else {
                s->setValue(iopt.key(), "");
            }
        }

        if ((*iopt).type == INTEGER) {
            if ((*iopt).value.toInt() != -1) {
                s->setValue(iopt.key(), (*iopt).value.toInt());
            } else {
                s->setValue(iopt.key(), "");
            }
        }
    }

    s->sync();
}


void HTTraQt::loadOptions(QSettings *s)
{
    resetDefaultOptions();
    //     qDebug() << "load";

    for (QMap<QString, OptStruct>::iterator iopt = prOptions.begin(); iopt != prOptions.end(); ++iopt) {
        if ((*iopt).type == TEXT) {
            QString tStr;
            tStr = s->value(iopt.key(), (*iopt).value).toString();
            profile_decode(tStr);
            (*iopt).value = tStr;

            continue;
        }

        if ((*iopt).type == FLOAT) {
            float tFloat;
            bool iOk;
            tFloat = s->value(iopt.key(), (*iopt).value).toFloat(&iOk);

            if (iOk == true) {
                if (tFloat >= 0) {
                    QString tTxt;
                    tTxt = QString().sprintf("%6.4f", tFloat / (1024.0 * 1024.0));
                    tFloat = tTxt.toFloat();
                }

                //qDebug() << "load" << (*iopt).name << tFloat << (qint64)tFloat;
                (*iopt).value = tFloat; // overwrite the default value if right
            } else {
                (*iopt).value = -1.0;
            }

            continue;
        }

        if ((*iopt).type == INTEGER) {
            int tInt;
            bool iOk;
            tInt = s->value(iopt.key(), (*iopt).value).toInt(&iOk);

            if (iOk == true) {
                (*iopt).value = tInt; // overwrite the default value if right
            }

            continue;
        }
    }

    // to implement the checking of options file!
}


void HTTraQt::loadOptionsFromFile()
{
    QSettings* s;

    QString fileName = QFileDialog::getOpenFileName(NULL, translate(_OPENFILE),
                       currentWorkDir, "HTTrack Settings (linprofile.ini winprofile.ini)");

    if (fileName.length() == 0) {
        return;
    }

    s = new QSettings(fileName, WIniFormat);

    loadOptions(s);
}


bool HTTraQt::checkInstanceRuns()
{
    if(_singular->attach(QSharedMemory::ReadOnly)) {
        //         _singular->detach();
        return false;
    }

    if(_singular->create(1)) {
        return true;
    }

    return false;
}



void HTTraQt::saveOptionsAs()
{
    QSettings* s;

    QString fileName = QFileDialog::getSaveFileName(NULL, translate(_SAVEFILE),
                       currentWorkDir, "winprofile.ini");

    if (fileName.length() == 0) {
        return;
    }

    s = new QSettings(fileName, WIniFormat);

    saveOptions(s, false);
}


void HTTraQt::pauseTransfer()
{
    if (pauseOptAct->isChecked() == true) {
        hts_setpause(global_opt, 1);
    } else {
        hts_setpause(global_opt, 0);
    }
}


void HTTraQt::onStopAll()
{
    pauseOptAct->setChecked(false);

    // qDebug() << "onStopAll";
    if (soft_term_requested) {
        termine_requested = 1;
    } else {
        soft_term_requested = 1;
        hts_request_stop(mainWidget->global_opt, 0);
    }
}


void HTTraQt::viewTransfers()
{
}



// this is only possible, if programm installed in home directory, like firefox
// in other case only display information about update,
// normal updates only from repository
void HTTraQt::checkUpdates()
{
    QDesktopServices::openUrl(QUrl("http://httraqt.sourceforge.net/", QUrl::TolerantMode));
}


// zeige hilfe
void HTTraQt::contens()
{
    QDesktopServices::openUrl(QUrl::fromLocalFile(qPrintable( helpDir + "/index.html")));
}


void HTTraQt::stepByStep()
{
    QDesktopServices::openUrl(QUrl::fromLocalFile(qPrintable( helpDir + "/step.html")));
}


void HTTraQt::createToolBars()
{
    //     fileToolBar()->addToolBar(translate("File"));
    //     fileToolBar->addAction(newLetterAct);
    //     fileToolBar->addAction(saveAct);
    //     fileToolBar->addAction(printAct);
    //
    //     editToolBar = addToolBar(translate("Edit"));
    //     editToolBar->addAction(undoAct);
}


void HTTraQt::createStatusBar()
{
    statusBar()->showMessage(translate(_READY), 2000);
}


void HTTraQt::getOptions()
{
    currentOptionsFile = "";

    if (currentProject.length() > 0) {
        QString linname, winname;
        linname = currentWorkDir + "/" + currentProject + "/linprofile.ini";
        winname = currentWorkDir + "/" + currentProject + "/hts-cache/winprofile.ini";

        if (QFile::exists(linname) == true) {
            currentOptionsFile = linname;
        }

        if (QFile::exists(winname) == true) {
            currentOptionsFile = winname;
        }
    }

    if (currentOptionsFile.length() > 0) {
        readSettings(false); // project
    } else {
        readSettings(true); // global
    }
}


void HTTraQt::resetDefaultOptions()
{
    //     prOptions.clear();

    prOptions[ "Near" ] = (OptStruct) {
        INTEGER, 1
    };
    prOptions[ "ParseAll" ] = (OptStruct) {
        INTEGER, 1
    };
    prOptions[ "Test" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "HTMLFirst" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "Dos" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "ISO9660" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "NoErrorPages" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "NoExternalPages" ] = (OptStruct) {
        INTEGER, 1
    };
    prOptions[ "NoPwdInPages" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "NoQueryStrings" ] = (OptStruct) {
        INTEGER, 1
    };
    prOptions[ "NoPurgeOldFiles" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "Build" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "WildCardFilters" ] = (OptStruct) {
        TEXT, "+*.png +*.gif +*.jpg +*.css +*.js -ad.doubleclick.net/*"
    };
    prOptions[ "Depth" ] = (OptStruct) {
        INTEGER, -1
    };
    prOptions[ "ExtDepth" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "CurrentAction" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "MaxHtml" ] = (OptStruct) {
        FLOAT, -1
    };
    prOptions[ "MaxOther" ] = (OptStruct) {
        FLOAT, -1
    };
    prOptions[ "MaxAll" ] = (OptStruct) {
        FLOAT, -1
    };
    prOptions[ "MaxWait" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "MaxTime" ] = (OptStruct) {
        INTEGER, -1
    };
    prOptions[ "MaxRate" ] = (OptStruct) {
        FLOAT, -1
    };
    prOptions[ "MaxConn" ] = (OptStruct) {
        INTEGER, -1
    };
    prOptions[ "MaxLinks" ] = (OptStruct) {
        INTEGER, -1
    };
    prOptions[ "Proxy" ] = (OptStruct) {
        TEXT, ""
    };
    prOptions[ "CurrentUrl" ] = (OptStruct) {
        TEXT, ""
    };
    prOptions[ "CurrentURLList" ] = (OptStruct) {
        TEXT, ""
    };
    prOptions[ "Category" ] = (OptStruct) {
        TEXT, ""
    };
    prOptions[ "BuildString" ] = (OptStruct) {
        TEXT, "%h%p/%n%q.%t"
    };
    prOptions[ "UseHTTPProxyForFTP" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "PrimaryScan" ] = (OptStruct) {
        INTEGER, 4
    };
    prOptions[ "Debug" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "UserID" ] = (OptStruct) {
        TEXT, "Mozilla/5.0 (X11; U; Linux i686; I; rv:17.0.1) Gecko/20100101 Firefox/17.0.1"
    };
    prOptions[ "BrowserID" ] = (OptStruct) {
        TEXT, "Mozilla/5.0 (%s; rv:17.0.1) Gecko/20100101 Firefox/17.0.1"
    };
    prOptions[ "osID" ] = (OptStruct) {
        TEXT, "X11; U; Linux i686; I"
    };
    prOptions[ "Travel" ] = (OptStruct) {
        INTEGER, 1
    };
    prOptions[ "GlobalTravel" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "RewriteLinks" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "Footer" ] = (OptStruct) {
        TEXT, "<!-- Mirrored from %s%s by HTTraQt Website Copier/1.x [Karbofos 2010-2014] %s -->"
    };
    prOptions[ "Cache" ] = (OptStruct) {
        INTEGER, 1
    };
    prOptions[ "TimeOut" ] = (OptStruct) {
        INTEGER, -1
    };
    prOptions[ "Sockets" ] = (OptStruct) {
        INTEGER, 8
    };
    prOptions[ "Retry" ] = (OptStruct) {
        INTEGER, -1
    };
    prOptions[ "KeepAlive" ] = (OptStruct) {
        INTEGER, 1
    };
    prOptions[ "RemoveTimeout" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "RemoveRateout" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "RateOut" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "StoreAllInCache" ] = (OptStruct) {
        INTEGER, 0
    };

    for (int j = 1; j <= 8; ++j) {
        QString n;
        n = "MIMEDefsMime" + QString::number(j);
        prOptions[ n ] = (OptStruct) {
            TEXT, ""
        };
        n = "MIMEDefsExt" + QString::number(j);
        prOptions[ n ] = (OptStruct) {
            TEXT, ""
        };
    }

    prOptions[ "NoRecatch" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "Log" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "Index" ] = (OptStruct) {
        INTEGER, 1
    };
    prOptions[ "BuildTopIndex" ] = (OptStruct) {
        INTEGER, 1
    };
    prOptions[ "WordIndex" ] = (OptStruct) {
        INTEGER, 1
    };
    prOptions[ "LogType" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "URLHack" ] = (OptStruct) {
        INTEGER, 1
    };
    prOptions[ "Cookies" ] = (OptStruct) {
        INTEGER, 1
    };
    prOptions[ "UpdateHack" ] = (OptStruct) {
        INTEGER, 1
    };
    prOptions[ "ParseJava" ] = (OptStruct) {
        INTEGER, 1
    };
    prOptions[ "TolerantRequests" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "HTTP10" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "CheckType" ] = (OptStruct) {
        INTEGER, 0
    };
    prOptions[ "FollowRobotsTxt" ] = (OptStruct) {
        INTEGER, 2
    };
    prOptions[ "ProxyPort" ] = (OptStruct) {
        INTEGER, 21
    };
}


void HTTraQt::writeGUISettings(void)
{
    QSettings* s;
    s = new QSettings(QSettings::UserScope, "KarboSoft", "HTTraQt");
    s->setValue("pos", pos());
    s->setValue("size", size());
    s->setValue("WorkDir", currentWorkDir);
    s->setValue("LANGUAGE", currentLang);
    s->setValue("LASTPROJ", currentProject);
    //     s->setValue("FontSize", fontSize);
    //     s->setValue("GUIFont", sysFont);
    // qDebug() << "writeGUISettings";
    int i = 0;

    for (QStringList::Iterator iDir = lastDirs.begin(); iDir != lastDirs.end(); iDir++, i++) {
        if (i > 8) { // max last dirs
            break;
        }

        s->setValue("LASDIR" + QString::number(i), (*iDir));
    }

    s->sync();
}


void HTTraQt::writeSettings(bool global)
{
    QSettings* s;
    QString fname;

    if (global == true) {
        s = new QSettings(QSettings::UserScope, "KarboSoft", "HTTraQt");
        s->setValue("WorkDir", currentWorkDir);
        s->setValue("LANGUAGE", currentLang);
        //                 qDebug() << "writesettings global";
    } else {
        if (currentProject.length() == 0) {
            return;
        }

        fname = currentWorkDir + "/" + currentProject/* + "/hts-cache"*/;
        QDir d(fname);

        if (d.exists() == false) {
            d.mkpath((const QString) fname);
        }

        if (d.mkpath(fname + "/hts-cache") == false) {
            qDebug() << "can not create";
        }

        fname += "/hts-cache/winprofile.ini";
        //         qDebug() << "writesettings local" << fname;
        s = new QSettings(fname, WIniFormat);
    }

    saveOptions(s, global);
}

// to get locale and convert to internal string
QString HTTraQt::getLocaleString()
{
    QString res;
    QLocale lSys = QLocale::system();

    switch (lSys.language()) {
        case QLocale::Bulgarian:
            res = "Bulgarian";
            break;

        case QLocale::Spanish:
            res = "Castellano";
            break;

        case QLocale::Czech:
            res = "Česky";
            break;

        case QLocale::Chinese:
            res = "Chinese-Simplified";
            break;

        case QLocale::Danish:
            res = "Dansk";
            break;

        case QLocale::German:
            res = "Deutsch";
            break;

        case QLocale::Estonian:
            res = "Eesti";
            break;

        case QLocale::C:
            res = "English";
            break;

        case QLocale::Finnish:
            res = "Finnish";
            break;

        case QLocale::French:
            res = "Français";
            break;

        case QLocale::Greek:
            res = "Greek";
            break;

        case QLocale::Italian:
            res = "Italiano";
            break;

        case QLocale::Japanese:
            res = "Japanese";
            break;

        case QLocale::Hungarian:
            res = "Magyar";
            break;

        case QLocale::Netherlands:
            res = "Nederlands";
            break;

        case QLocale::NorwegianNynorsk:
            res = "Norsk";
            break;

        case QLocale::Polish:
            res = "Polski";
            break;

        case QLocale::Brazil:
            res = "Português-Brasil";
            break;

        case QLocale::Portuguese:
            res = "Português";
            break;

        case QLocale::Romanian:
            res = "Romanian";
            break;

        case QLocale::Russian:
            res = "Russian";
            break;

        case QLocale::Slovak:
            res = "Slovak";
            break;

        case QLocale::Slovenian:
            res = "Slovenian";
            break;

        case QLocale::Swedish:
            res = "Svenska";
            break;

        case QLocale::Turkish:
            res = "Turkish";
            break;

        case QLocale::Ukrainian:
            res = "Ukrainian";
            break;

        default:
            res = "English";
            break;
    }

    return res;
}


void HTTraQt::readGUISettings()
{
    QSettings* s;
    s = new QSettings(QSettings::UserScope, "KarboSoft", "HTTraQt");
    QPoint pos = s->value("pos", QPoint(200, 200)).toPoint();
    QSize size = s->value("size", QSize(840, 640)).toSize();
    resize(size);
    move(pos);

    QString l;
    l = getLocaleString();

    currentLang = s->value("LANGUAGE", l).toString();
    currentProject = s->value("LASTPROJ").toString();
    sysFont = sysFont.toString();

    int sz = sysFont.pointSize();

    if ( sz == -1) {
        sz = sysFont.pixelSize();
    }

    fontSize = sz;

    for (int i = 0; i < 8; i++) {
        QString d = s->value("LASDIR" + QString::number(i)).toString();
        QDir dr;

        if (d.length() == 0) {
            break;
        }

        if (dr.exists(d) == true) {
            lastDirs << d;
        }
    }

    QDir dir;
    QStringList dirsLang;
    dirsLang << "/usr/share/httraqt/" << "/usr/local/share/httraqt/" << currentAppDir;

    foreach(QString entry, dirsLang) {
        helpDir = entry + "/help/";

        dir = QDir(helpDir);

        if (dir.exists() == true) {
            break;
        } else {
            helpDir = "";
        }
    }

    foreach(QString entry, dirsLang) {
        langDir = entry + "/lang/";

        dir = QDir(langDir);

        if (dir.exists() == true) {
            break;
        } else {
            langDir = "";
        }
    }
}


void HTTraQt::readSettings(bool global)
{
    QSettings* s;
    QString fname;

    QString text;

    if (global == true) {
        s = new QSettings(QSettings::UserScope, "KarboSoft", "HTTraQt");

        currentWorkDir = s->value("WorkDir", QDir::homePath() + "/My Web Sites").toString();
    } else {
        fname = currentWorkDir + "/" + currentProject + "/linprofile.ini";

        if (QFile::exists(fname) == false) { // exists
            fname = currentWorkDir + "/" + currentProject + "/hts-cache/winprofile.ini";

            if (QFile::exists(fname) == false) { // exists
                fname = "";
            }
        }
    }

    if (fname.length() == 0) { // file not found, global settings
        s = new QSettings(QSettings::UserScope, "KarboSoft", "HTTraQt");
    } else {
        s = new QSettings(fname, WIniFormat);
    }

    currentOptionsFile = fname;

    loadOptions(s);
}


void HTTraQt::onCancelAll()
{
}


void HTTraQt::onQuit()
{
    quit();
}


void HTTraQt::activatePage(int pageNum)
{
    switch (pageNum) {
        case 0: { // start tab
            currentTab = 0;

            // actions
            pcShutdown = false;
            pcHibernate = false;

            browseAct->setEnabled(false);

            viewLogAct->setEnabled(false);
            viewErrLogAct->setEnabled(false);
            pauseOptAct->setEnabled(false);
            modifyOptAct->setEnabled(false);

            break;
        }

        case 1: { // new project tab or select project
            //             qDebug() << "seite 1: " << currentWorkDir;
            (static_cast<NewProjTab*>(widgets[1]))->editBasePath->setText(currentWorkDir);
            (static_cast<NewProjTab*>(widgets[1]))->selectProj(currentProject);

            currentTab = 1;

            //actions
            browseAct->setEnabled(false);

            viewLogAct->setEnabled(false);
            viewErrLogAct->setEnabled(false);
            pauseOptAct->setEnabled(false);
            modifyOptAct->setEnabled(false);

            break;
        }

        case 2: { // options tab
            if (currentTab == 1) {   // prüfe, ob die parameter i.o. sind: aus NewProjTab
                if ((static_cast<NewProjTab*>(widgets[1]))->testIt() == false) {   // projektname muss vorhanden sein
                    return;
                }
            }

            bool nProj = rebuildWorkDirView();

            QString url;
            GetProfile("CurrentUrl", url);
            (static_cast<OptionsTab*>(widgets[2]))->textEdit->setText(url);

            QString urlFile;
            GetProfile("CurrentURLList", urlFile);
            (static_cast<OptionsTab*>(widgets[2]))->label1286->setText(urlFile);

            if (nProj == false) {
                (static_cast<OptionsTab*>(widgets[2]))->updateLastAction();
                getOptions();
            } else {
                m_todo = 0;
                (static_cast<OptionsTab*>(widgets[2]))->comboAction->setCurrentIndex(m_todo);
                readSettings(true); // global
                m_todo = 0;
            }

            // wenn projekt existierte, einlesen von diversen settings und url liste.

            if ((m_todo == LAST_ACTION) || (m_todo == LAST_ACTION - 1)) {
                if (checkContinue(true) == false) {
                    return;
                }
            }

            currentTab = 2;

            //actions
            browseAct->setEnabled(false);

            viewLogAct->setEnabled(false);
            viewErrLogAct->setEnabled(false);
            pauseOptAct->setEnabled(false);
            modifyOptAct->setEnabled(false);

            break;
        }

        case 3: { // confirm tab with delay time
            if (currentTab == 2) {   // prüfe, ob die parameter i.o. sind: aus OptionsTab
                if ((static_cast<OptionsTab*>(widgets[2]))->testIt() == false) {    // die url liste muss vorhanden sein
                    MessageBox::exec(translate(_NOURL), translate(_URLNOTCOMP), QMessageBox::Critical);

                    return;
                }
            }

            (static_cast<OptionsTab*>(widgets[2]))->updateLastAction();

            QString url = (static_cast<OptionsTab*>(widgets[2]))->textEdit->toPlainText();
            SetProfile("CurrentUrl", url);

            QString urlFile = (static_cast<OptionsTab*>(widgets[2]))->label1286->text();
            SetProfile("CurrentURLList", urlFile);

            (static_cast<ConfirmTab*>(widgets[3]))->groupPCoff->setChecked(false);

            (static_cast<ConfirmTab*>(widgets[3]))->labelHH->clear(); // HH
            (static_cast<ConfirmTab*>(widgets[3]))->labelMM->clear(); // MM
            (static_cast<ConfirmTab*>(widgets[3]))->labelSS->clear(); // SS

            currentTab = 3;

            getMainOptionsFromGUI();
            writeSettings(false); // write project settings

            // check current action, if
            QString prDir;
            prDir =  currentWorkDir + "/" + currentProject + "/";

            //actions
            browseAct->setEnabled(false);

            viewLogAct->setEnabled(false);
            viewErrLogAct->setEnabled(false);
            pauseOptAct->setEnabled(false);
            modifyOptAct->setEnabled(false);

            menuPreferences->setEnabled(true);
            break;
        }

        case 4: { // progress tab

            currentTab = 4;

            //actions
            browseAct->setEnabled(true);

            viewLogAct->setEnabled(true);
            viewErrLogAct->setEnabled(true);
            pauseOptAct->setEnabled(true);
            modifyOptAct->setEnabled(true);

            menuPreferences->setEnabled(false);
            for (int i = 1; i < NStatsBuffer; i++){
                (static_cast<ProgressTab*>(widgets[4]))->tableWidget->setRowHidden(i, true);
            }

            launch();

            break;
        }

        case 5: { // log tab
            currentTab = 5;

            // actions
            browseAct->setEnabled(true);

            viewLogAct->setEnabled(false);
            viewErrLogAct->setEnabled(false);
            pauseOptAct->setEnabled(false);
            modifyOptAct->setEnabled(false);

            menuPreferences->setEnabled(true);
            break;
        }

        default:
            break;
    }

    for (int i = 0; i <= 5; i++) {
        if (i == pageNum) {
            if (widgets[i]->isHidden() == true) {
                widgets[i]->show();
            }
        } else {
            if (widgets[i]->isHidden() == false) {
                widgets[i]->hide();
            }
        }
    }

    buttonsWidget->onButtons(pageNum);
}


HTTraQt::~HTTraQt()
{
    if(_singular->isAttached()) {
        _singular->detach();
    }

    quit();
}


int HTTraQt::removeFolder(QDir & dir)
{
    int res = 0;
    //list of dirs
    QStringList lstDirs  = dir.entryList(QDir::Dirs | QDir::AllDirs | QDir::NoDotAndDotDot);
    //file list
    QStringList lstFiles = dir.entryList(QDir::Files);
    //rm files
    foreach(QString entry, lstFiles) {
        QString entryAbsPath = dir.absolutePath() + "/" + entry;
        QFile::remove(entryAbsPath);
    }

    //for directories recursive
    foreach(QString entry, lstDirs) {
        QString entryAbsPath = dir.absolutePath() + "/" + entry;
        QDir dr;
        dr = QDir(entryAbsPath);
        removeFolder(dr);
    }

    //deleting of directory
    if (!QDir().rmdir(dir.absolutePath())) {
        res = 1;
    }

    return res;
}


void HTTraQt::viewLog()
{
    QString name = currentWorkDir + "/" + currentProject + "/"  "hts-log.txt";

    // fichier log existe ou on est télécommandé par un !
    if (QFile::exists(name) == true ) {
        QDesktopServices::openUrl(QUrl::fromLocalFile(qPrintable( name )));
        return;
    }

    //     name = currentWorkDir + "/" + currentProject + "/"  "hts-err.txt";
    //
    //     if (QFile::exists(name) == true ) {
    //         QDesktopServices::openUrl(QUrl::fromLocalFile(qPrintable( name )));
    //         return;
    //     }

    QString l = translate(_NOLOGFILES);
    l.replace("%s", currentProject);

    MessageBox::exec(translate(_ERROR), l, QMessageBox::Warning);
}


void HTTraQt::viewErrorLog()
{
    QString name = currentWorkDir + "/" + currentProject + "/"  "hts-err.txt";

    if (QFile::exists(name) == true ) {
        QDesktopServices::openUrl(QUrl::fromLocalFile(qPrintable( name )));
        return;
    }

    QString l = translate(_NOLOGFILES);
    l.replace("%s", currentProject);
    MessageBox::exec(translate(_ERROR), l, QMessageBox::Warning);
}

#if 0
void HTTraQt::createFontSizeMenu()
{
    fontSizeMenu = menuPreferences->addMenu(translate(_SEL_FONT));
    fsizeGroup = new QActionGroup(this);
    QStringList szList;
    szList << "9" << "10" << "12" << "14" << "16" << "18" << "20";

    foreach(QString entry, szList) {
        QAction *tmpAction = new QAction(entry, fontSizePrefAct);
        tmpAction->setCheckable(true);

        fontSizeMenu->addAction(tmpAction);
        fsizeGroup->addAction(tmpAction);

        if (fontSize == entry.toInt()) {
            tmpAction->setChecked(true);
        }

        actFSizeSelect.push_back(tmpAction);
    }

    connect(fsizeGroup, SIGNAL(triggered(QAction*)), this, SLOT(selectFontSize(QAction*)));
}
#endif



bool HTTraQt::readLangDir()
{
    bool found = false;
    QString lngDirName;
    QStringList dirsLang;
    QDir dir;
    dirsLang << "/usr/share/httraqt/" << "/usr/local/share/httraqt/" << currentAppDir;

    dictFormat = 0;

    foreach(QString entry, dirsLang) {
        lngDirName = entry + "/lang/";

        dir = QDir(lngDirName);

        if (dir.exists() == true) {
            found = true;
            break;
        }

        lngDirName = entry + "/language/";
        dir = QDir(lngDirName);

        if (dir.exists() == true) {
            found = true;
            break;
        }
    }

    if(found == false) {
        return false;
    }

    langFiles.clear();

    QStringList fList = dir.entryList(QStringList("*.utf"));

    found = false;

    langMenu = menuPreferences->addMenu(translate(_LANGUAGE));

    langGroup = new QActionGroup(this);


    for (QStringList::Iterator iL = fList.begin(); iL != fList.end(); iL++) {
        QFile fLang(lngDirName + *iL);

        if (fLang.exists() == false) {
            continue;
        }

        QString iconName;

        if (fLang.open(QIODevice::ReadOnly)) {      //wird eingelesen
            QTextStream stream(&fLang);
            stream.setCodec("UTF-8");
            QString line, nm;

            int lines = 0;

            while (!stream.atEnd()) {
                line = stream.readLine(); // line of text excluding '\n'
                lines++;

                if (line == "LANGUAGE_NAME") {
                    line = stream.readLine();
                    lines++;
                    nm = line;
                    continue;
                }

                if (line == "LANGUAGE_ISO") {
                    line = stream.readLine();
                    selectedLang = line;
                    lines++;

                    iconName = lngDirName + "flags/" + line + ".png";

                    found = true;

                    langFiles += (*iL) + ":" + nm;
                    QAction *tmpAction = new QAction(nm, langPrefAct);
                    tmpAction->setCheckable(true);

                    if (QFile::exists(iconName) == true) {
                        QFile flIcon(iconName);

                        if (flIcon.size() < 1024 ) { // checking of filesize
                            tmpAction->setIcon(QIcon(iconName));
                        }
                    }

                    langGroup->addAction(tmpAction);
                    langMenu->addAction(tmpAction);

                    if (currentLang == nm) {
                        tmpAction->setChecked(true);
                    }

                    actLangSelect.push_back(tmpAction);
                    break;
                }

                if (lines > 8) {
                    break;
                }
            }

            fLang.close();

        } else {
            continue;
        }
    }

    connect(langGroup, SIGNAL(triggered(QAction*)), this, SLOT(setLang(QAction*)));

    return (found);
}


void HTTraQt::selectFontSize(QAction* mnu)
{
    QString lngStr;

    lngStr = mnu->text();
    fontSize = lngStr.toInt();

    mnu->setChecked(true);

    int sz = sysFont.pointSize();

    // for lang menu and for fontsize menu
    if ( sz == -1) {
        programStyleSheet = QString().sprintf("font-size: %dpx", fontSize);
    } else {
        programStyleSheet = QString().sprintf("font-size: %dpt", fontSize);
    }

    QString sSheet2 = QString("QMenu { %1; }").arg( programStyleSheet );

    setStyleSheet(programStyleSheet);

    langMenu->setStyleSheet(sSheet2);

    setFontForWidgets();
}


void HTTraQt::setLang(QAction* mnu)
{
    QString lngStr;
    mnu = langGroup->checkedAction();

    lngStr = mnu->text();
    currentLang = lngStr;

    if (getLangTable() == false) {
        qDebug() << "setLang" << false;
    }

    QVector<QAction*>::iterator itL;

    disconnect(langGroup, SIGNAL(triggered(QAction*)), this, SLOT(setLang(QAction*)));

    mnu->setChecked(true);

    connect(langGroup, SIGNAL(triggered(QAction*)), this, SLOT(setLang(QAction*)));

    setLangGUI();
}


void HTTraQt::convertTranslateFile()
{
    QVector<hInfo> hList;

    QFile hdrFile(currentAppDir + "/sources/main/includes/lang.h.orig");

    if (hdrFile.open(QIODevice::ReadOnly)) {
        QTextStream stream(&hdrFile);
        stream.setCodec("UTF-8");
        int num = 1;

        while (!stream.atEnd()) {
            QString ll = stream.readLine();

            if (ll.indexOf("#define") != 0 || ll.indexOf("\"") <= 0) {
                continue;
            }

            int pos1 = ll.indexOf(QRegExp("[_A-Z]"));
            int pos2 = ll.indexOf(" ", pos1);
            QString name = ll.mid(pos1, pos2 - pos1);

            pos1 = ll.indexOf("\"", pos2);
            QString txt = ll.mid(pos1);
            bool found = false;

            for (QVector<hInfo>::iterator il = hList.begin(); il != hList.end(); ++il) {
                if ((*il).tInfo == txt) {
                    found = true;
                    break;
                }
            }

            if (found == false) {
                hInfo tmp;
                tmp.name = name;
                tmp.num = num;
                txt.replace("\t", "\\t");
                txt.replace("\n", "\\n");
                tmp.tInfo = txt.mid(1, txt.length() - 2);

                hList << tmp;
                num++;
            }
        }

        hdrFile.close();

        QFile hdrOutFile(currentAppDir + "/sources/main/includes/lang.h.new");

        if (hdrOutFile.open(QIODevice::WriteOnly)) {
            QTextStream stream(&hdrOutFile);
            stream.setCodec("UTF-8");
            stream << "#ifndef LANGUAGE_HEADER\n";
            stream << "#define LANGUAGE_HEADER\n\n\n";

            stream << "struct engText { \n";

            for (QVector<hInfo>::iterator il = hList.begin(); il != hList.end(); ++il) {
                stream << "\"" << (*il).tInfo << "\", // " << QString::number((*il).num) << "\n";
            }

            stream << "};\n\n";

            for (QVector<hInfo>::iterator il = hList.begin(); il != hList.end(); ++il) {
                stream << "#define " << (*il).name << " "  << QString::number((*il).num) << "\n";
            }

            stream << "\n\n#endif\n";
            hdrOutFile.close();
        } else {
            qDebug() << "cannot open" << hdrOutFile.fileName();
            return;
        }
    } else {
        qDebug() << "cannot open" << hdrFile.fileName();
        return;
    }


    //     bool found = false;
    QString lngDirName = currentAppDir + "/lang/";
    QDir dir(lngDirName);

    QStringList fList = dir.entryList(QStringList("*.utf"));

    for (QStringList::Iterator iL = fList.begin(); iL != fList.end(); iL++) {
        QStringList dictionary;
        dictionary.clear();

        QFile fLang(lngDirName + *iL);

        if (fLang.open(QIODevice::ReadOnly)) {      //wird eingelesen
            QTextStream stream(&fLang);
            stream.setCodec("UTF-8");
            QString line;

            while (!stream.atEnd()) {
                line = stream.readLine(); // line of text excluding '\n'
                dictionary << line;
            }

            fLang.close();
        } else {
            qDebug() << "not found" << *iL;
        }

        qDebug() << (*iL) << "lines: " << dictionary.size();
        // #ifdef TRANSLATE_FILE
        QFile langTrFile(lngDirName + (*iL) + ".new");
        bool trBeg = false;

        if (langTrFile.open(QIODevice::WriteOnly)) {
            QTextStream stream(&langTrFile);
            stream.setCodec("UTF-8");

            for (QStringList::iterator it = dictionary.begin(); it != dictionary.end(); ++it) {
                if (trBeg == false) {
                    if ((*it).length() == 0) {
                        stream << (*it) << "\r\n";
                    }

                    if ((*it).indexOf("OK") == 0) { // beg of translations detected
                        trBeg = true;
                    }
                }

                if (trBeg == false) {
                    stream << (*it) << "\r\n";
                } else {
                    QString origText;
                    origText = (*it);
                    ++it; // jede zweite

                    if (it == dictionary.end()) {
                        break;
                    }

                    QString ll = *it;

                    //                 ll.replace("\\r\\n", "\n");
                    if (ll.length() > 0) {
                        ll.replace("winhttrack", "httraqt");
                        ll.replace(" (B/s)", "");
                        ll.replace("C:", "%HOME%");
                        ll.replace("c:", "%HOME%");
                        //                         ll.replace("\\n", "\n");
                        //                         ll.replace("\\t", "\t");
                        ll.replace("\\r\\n", "\\n");
                        ll.replace( "\\\\", "/");
                        //                         ll.replace("\\t", "\t");
                        ll.replace( "&gt;", ">");
                        ll.replace( "&lt;", "<");
                    } else {
                        continue;
                    }

                    if (origText.length() > 0) {
                        origText.replace("winhttrack", "httraqt");
                        origText.replace(" (B/s)", "");
                        //                         origText.replace("\\n", "\n");
                        origText.replace("C:", "%HOME%");
                        origText.replace("c:", "%HOME%");
                        //                          origText.replace("\\t", "\t");
                        origText.replace("\\r\\n", "\\n");
                        origText.replace( "\\\\", "/");
                        origText.replace( "&amp;", "&");
                        origText.replace( "&gt;", ">");
                        origText.replace( "&lt;", "<");
                    } else {
                        continue;
                    }

                    if (ll.length() > 0) {
                        ll.replace( "\\\\", "/");
                        ll.replace( "&amp;", "&");
                    } else {
                        continue;
                    }

                    //                     if (origText.length() > 0) {
                    //                         origText.replace( "/", "\\\\");
                    //                     }
                    //                     else{
                    //                         continue;
                    //                     }

                    if (ll.length() > 0) {
                        bool found = false;

                        for (QVector<hInfo>::iterator itr = hList.begin(); itr != hList.end(); ++itr) {
                            //                             origText = "\"" + origText + "\"";
                            if ( origText == (*itr).tInfo) {
                                stream << QString().sprintf("t%03d=", (*itr).num) << ll << "\r\n";
                                found = true;
                                break;
                            }

                            //                         stream << (*itr).tInfo << ", // " << QString::number((*itr).num) << "\n";
                        }

                        if (found == false) {
                            qDebug() << "file" << *iL << "line not found" << origText;
                        }
                    }

                    //                     n++;
                }
            }

            langTrFile.close();
        } else {
            qDebug() << "cannot open" <<  langTrFile.fileName();
        }
    }
}

// thanks to Leon
// https://qt-project.org/forums/viewthread/5957
//HIBERNATE
void HTTraQt::hibernate()
{
    int ret;
    MessageTimerBox msgBox;

    msgBox.setDefaultText("Hibernate PC?\nAutoclose in %d seconds...");
    msgBox.setIcon(QMessageBox::Question);
    msgBox.setStandardButtons(QMessageBox::Yes | QMessageBox::No);
    msgBox.setAutoClose(true);
    msgBox.setTimeout(30); //Closes after three seconds
    ret = msgBox.exec();

    if (ret == QMessageBox::No) {
        return;
    }

#ifdef __WIN32__
    // code for win systems
#endif

#ifdef __linux__
    bool gnome_power1 = false;
    bool gnome_power2 = false;
    bool hal_works = false;
    QDBusMessage response;

    gnome_power1 = QProcess::startDetached("gnome-power-cmd.sh hibernate");
    gnome_power2 = QProcess::startDetached("gnome-power-cmd hibernate");

    if(!gnome_power1 && !gnome_power2 && verbose) {
        qWarning() << "W: gnome-power-cmd and gnome-power-cmd.sh didn't work";
    }

    if(!gnome_power1 && !gnome_power2) {
        QDBusInterface powermanagement("org.freedesktop.Hal",
                                       "/org/freedesktop/Hal/devices/computer",
                                       "org.freedesktop.Hal.Device.SystemPowerManagement",
                                       QDBusConnection::systemBus());
        response = powermanagement.call("Hibernate");

        if(response.type() == QDBusMessage::ErrorMessage) {
            if(verbose) {
                qWarning() << "W: " << response.errorName() << ":" << response.errorMessage();
            }
        } else {
            hal_works = true;
        }
    }

    if(!hal_works && !gnome_power1 && !gnome_power2) {
        QDBusInterface powermanagement("org.freedesktop.DeviceKit.Power", "/org/freedesktop/DeviceKit/Power",
                                       "org.freedesktop.DeviceKit.Power", QDBusConnection::systemBus());

        if(response.type() == QDBusMessage::ErrorMessage) {
            if(verbose) {
                qWarning() << "W: " << response.errorName() << ":" << response.errorMessage();
            }
        }
    }

#endif // LINUX

#ifdef __APPLE__

#endif
}


//SHUTDOWN
void HTTraQt::shutdown()
{
    int ret;
    MessageTimerBox msgBox;

    msgBox.setDefaultText("Shutdown PC?\nAutoclose in %d seconds...");
    msgBox.setIcon(QMessageBox::Question);
    msgBox.setStandardButtons(QMessageBox::Yes | QMessageBox::No);
    msgBox.setAutoClose(true);
    msgBox.setTimeout(30); //Closes after three seconds
    ret = msgBox.exec();

    if (ret == QMessageBox::No) {
        return; // no shutdown
    }

#ifdef __WIN32__
    // code for win systems
#endif

#ifdef __linux__
    bool shutdown_works = false;
    bool gnome_power1 = false;
    bool gnome_power2 = false;
    bool hal_works = false;
    QDBusMessage response;

    QDBusInterface gnomeSessionManager("org.gnome.SessionManager",
                                       "/org/gnome/SessionManager", "org.gnome.SessionManager",
                                       QDBusConnection::sessionBus());
    response = gnomeSessionManager.call("RequestShutdown");

    if(response.type() == QDBusMessage::ErrorMessage) {
        if(verbose) {
            qWarning() << "W: " << response.errorName() << ":" << response.errorMessage();
        }

        gnome_power1 = QProcess::startDetached("gnome-power-cmd.sh shutdown");
        gnome_power2 = QProcess::startDetached("gnome-power-cmd shutdown");

        if(verbose && !gnome_power1 && !gnome_power2) {
            qWarning() << "W: gnome-power-cmd and gnome-power-cmd.sh didn't work";
        }
    } else {
        shutdown_works = true;
    }

    QDBusInterface kdeSessionManager("org.kde.ksmserver", "/KSMServer",
                                     "org.kde.KSMServerInterface", QDBusConnection::sessionBus());
    response = kdeSessionManager.call("logout", 0, 2, 2);

    if(response.type() == QDBusMessage::ErrorMessage) {
        if(verbose) {
            qWarning() << "W: " << response.errorName() << ":" << response.errorMessage();
        }
    } else {
        shutdown_works = true;
    }

    if(!shutdown_works && !gnome_power1 && !gnome_power2) {
        QDBusInterface powermanagement("org.freedesktop.Hal",
                                       "/org/freedesktop/Hal/devices/computer",
                                       "org.freedesktop.Hal.Device.SystemPowerManagement",
                                       QDBusConnection::systemBus());
        response = powermanagement.call("Shutdown");

        if(response.type() == QDBusMessage::ErrorMessage) {
            if(verbose) {
                qWarning() << "W: " << response.errorName() << ":" << response.errorMessage();
            }
        } else {
            hal_works = true;
        }
    }

    if(!hal_works && !shutdown_works && !gnome_power1 && !gnome_power2) {
        QDBusInterface powermanagement("org.freedesktop.ConsoleKit",
                                       "/org/freedesktop/ConsoleKit/Manager", "org.freedesktop.ConsoleKit.Manager",
                                       QDBusConnection::systemBus());
        response = powermanagement.call("Stop");

        if(response.type() == QDBusMessage::ErrorMessage) {
            if(verbose) {
                qWarning() << "W: " << response.errorName() << ":" << response.errorMessage();
            }

            QProcess::startDetached("sudo shutdown -P now");
        }
    }

#endif // UNIX

#ifdef __APPLE__

#endif
}


bool HTTraQt::getLangTable()
{
    QString ll;
    QString lang = currentLang;
    QString fileLang = "";

    for (QStringList::Iterator iLang = langFiles.begin(); iLang != langFiles.end(); iLang++) {
        if ((*iLang).contains(":" + lang) > 0) {
            fileLang = *iLang;
            fileLang.remove(":" + lang);
            break;
        }
    }

    if (fileLang == "") {
        return (false);
    }

    QFile langFile(langDir + fileLang);

    if (langFile.exists() == false) {
        MessageBox::exec(translate(_ERROR), "Language file not exists!\n\n"
                         + langDir + "\n\n" + fileLang, QMessageBox::Warning);

        return (false);
    }

    translateTable.clear();

    if (langFile.open(QIODevice::ReadOnly)) {
        QTextStream stream(&langFile);
        stream.setCodec("UTF-8");

        ll = stream.readLine();

        if (ll.length() > 0) {
            ll.remove("\n");
        }

        if (ll == "LANGUAGE_NAME") {
            translateTable << ll;
        } else {
            MessageBox::exec(translate(_FILEERR), "File is not a language file!", QMessageBox::Warning);

            langFile.close();
            return (false);
        }

        while (!stream.atEnd()) {
            ll = stream.readLine();

            if (ll.length() > 0) {
                ll.remove("\n");
            }

            if (ll.length() == 0) { // english original text not found
                ll = stream.readLine();

                if (stream.atEnd() == true) {
                    break;
                } else {
                    continue;
                }
            }

            if (ll.length() > 0) {
                ll.replace( "\\r\\n", "\n");
                ll.replace( "\\n", "\n");
                ll.replace( "\\\\", "/");
                ll.replace( "&amp;", "&");
                ll.replace( "&gt;", ">");
                ll.replace( "&lt;", "<");
            }

            if (ll.indexOf(QRegExp("t[0-9]{3}=")) == 0) {
                qDebug() << "neues format";
                dictFormat = 2; // new format
            }

            if (ll.length() > 0) {
                if (ll.indexOf("C:/") >= 0) {
                    ll.replace("C:/", QString(QDir::homePath() + "/"));
                }

                if (ll.indexOf("c:/") >= 0) {
                    ll.replace("c:/", QString(QDir::homePath() + "/"));
                }

                if (ll.indexOf("%HOME%") >= 0) {
                    ll.replace("%HOME%", QString(QDir::homePath() + "/"));
                }
            }

            translateTable << ll;
        }

        langFile.close();

        return (true);
    }


    return (false);
}


void HTTraQt::setLangGUI()
{
    (static_cast<StartTab*>(widgets[0]))->translateTab();
    (static_cast<NewProjTab*>(widgets[1]))->translateTab();
    (static_cast<OptionsTab*>(widgets[2]))->translateTab();
    (static_cast<ConfirmTab*>(widgets[3]))->translateTab();
    (static_cast<ProgressTab*>(widgets[4]))->translateTab();
    (static_cast<FinalTab*>(widgets[5]))->translateTab();

    // actions translate
    translateActions();

    rebuildWorkDirView();
}


void HTTraQt::translateActions(QAction* act, int id)
{
    QString tmp;
    QString str = translate(id);
    int posTab;

    if (str.length() == 0) {
        return;
    }

    posTab = str.indexOf("\\t");
    tmp = translate(id);

    if (posTab > 0) {
        QString shrtcut = str.mid(posTab + 2);
        QStringList tmpC = tmp.split("\\t");
        act->setText(tmpC.at(0));
        act->setFont(sysFont);
        act->setShortcut(shrtcut);
    } else {
        act->setText(tmp);
        act->setFont(sysFont);
    }
}


void HTTraQt::translateActions()
{
    buttonsWidget->translateButtons();

    menuFile->setTitle(translate(_FILE));

    translateActions(newProjAct, _P18);

    translateActions(browseAct, _P23);
    browsProjAct->setStatusTip ( translate ( _BROWSEEXISTS ) );

    translateActions(exitAct, _EXIT);
    //     exitAct->setStatusTip ( tr ( "Exit" ) );

    menuPreferences->setTitle(translate(_PREF));
    //     menuPreferences->setStyleSheet(sSheet);
    translateActions(loadDefOptAct, _LOADDEF);
    //     loadDefOptAct->setStatusTip ( tr ( "Load default options" ) );

    translateActions(saveOptAct, _SAVEDEF);
    //     saveOptAct->setStatusTip ( tr ( "Save default options" ) );

    translateActions(resetOptToDefAct, _RESETTODEF);
    //     resetOptToDefAct->setStatusTip ( tr ( "Reset to default options" ) );

    translateActions(loadOptAct,  _LOADOPT);
    //     loadOptAct->setStatusTip ( tr ( "Load options..." ) );

    //     translateActions(recentAct, "Recent file");
    //     recentAct->setStatusTip ( tr ( "Recent file" ) );

    translateActions(saveOptAsAct, _SAVEOPTAS);
    //     translateActions(selectFontAct, _SEL_FONT);
    //     saveOptAsAct->setStatusTip ( tr ( "Save options as..." ) );
    langMenu->setTitle(translate(_LANGPREF));
    //     fontSizeMenu->setTitle(translate(_SEL_FONT));

    menuMirror->setTitle(translate(_MIRROR));
    // menuMirror->setStyleSheet(sSheet);
    translateActions(modifyOptAct, _MODIFYOPT);
    //     modifyOptAct->setStatusTip ( tr ( "Modify options" ) );

    pauseOptAct->setCheckable(true);
    translateActions(pauseOptAct, _PAUSETRANSF);
    //     pauseOptAct->setStatusTip ( tr ( "Pause transfer" ) );

    menuLog->setTitle(translate(_LOG));
    //     menuLog->setStyleSheet(sSheet);
    translateActions(viewLogAct, _VIEW_LOG);
    //     viewLogAct->setStatusTip ( translate("View log" ) );

    translateActions(viewErrLogAct, _VIEWERRLOG);
    //     viewErrLogAct->setStatusTip ( tr ( "View error log" ) );

    menuWindow->setTitle(translate(_WINDOW));
    //  menuWindow->setStyleSheet(sSheet);
    translateActions(hideAct, _HIDE);
    translateActions(restAct, _OPEN);
    //     hideAct->setStatusTip ( tr ( "Hide" ) );

    menuHelp->setTitle(translate(_HELP));
    //     menuHelp->setStyleSheet(sSheet);
    translateActions(checkAct, _TOPROJECTPAGE);
    translateActions(stepByStepAct, _STEPBYSTEP);
    translateActions(contentAct, _CONTENTS);
    translateActions(aboutAct, _ABOUTPROG);
}



void HTTraQt::quit()
{
    int ans;

    if (pcHibernate == false && pcShutdown == false ) {
        ans = MessageBox::exec(translate(_QUIT), translate(_REALLYQUIT), QMessageBox::Question);

        if (ans == QMessageBox::No) {
            return;
        }
    }

    writeGUISettings();

    qApp->quit();
}


void HTTraQt::closeEvent(QCloseEvent* ce)
{
    int ans;

    if (pcHibernate == false && pcShutdown == false ) {
        ans = MessageBox::exec(translate(_QUIT), translate(_REALLYQUIT), QMessageBox::Question);

        if (ans == QMessageBox::No) {
            ce->ignore();
            return;
        }
    }

    writeGUISettings();

    ce->accept();
    return;
}


void HTTraQt::aboutQt()
{
    QMessageBox::aboutQt(this, PROGRAM_NAME);
}


void HTTraQt::about()
{
    AboutDialog *ad;
    ad = new AboutDialog(this);

    ad->exec();
    delete ad;
}


// modif RX 10/10/98 pour gestion des , et des tabs
void HTTraQt::StripControls(QString &st)
{
    //     st.replace(0x09, " ");
    //     st.replace(0x0a, " ");
    //     st.replace(0x0d, " ");
    st.simplified();
    st.replace("  ", " ");
}


// modifs RX 10/10/98: gestion des ,
QString HTTraQt::change(char* chaine, char c)
{
    int comma = 1;
    int first = 1;
    QString chaine1;

    for (int i = 0; i < (int) strlen(chaine); i++) {
        switch (chaine[i]) {
            case 10:
            case 13:
            case 9:
            case ' ':
            case ',':
                comma = 1;
                break;

            default:

                if (comma) {
                    if (!first) {
                        chaine1 += ' ';
                    } else {
                        first = 0;
                    }

                    chaine1  += c;

                    comma = 0;
                }

                chaine1 += chaine[i];

                break;
        }
    }

    return chaine1;
}


// Ecriture profiles
void HTTraQt::profile_code(QString& from)
{
    from.replace("%%", "%");    // delete double entries
    from.replace("\n\n", "\n");    // delete double entries
    from.replace("%", "%%");
    from.replace("\r", "%0a");
    from.replace("\n", "%0d%0a");
    from.replace("\t", "%09");
}


void HTTraQt::profile_decode(QString& from)
{
    from.replace("%0d%0a", "\n");
    from.replace("%0d", "\n");
    from.replace("%0a", "\r");
    from.replace("%09", "\t");
    from.replace("%%", "%");
}


void HTTraQt::SetProfile(const QString& variName, float val)
{
    if (variName.length() < 1) {
        return;
    }

    OptStruct tmp = prOptions[variName];

    if (tmp.type != FLOAT) {
        qDebug() << "Wrong type, setProfile()" << variName << "was expected FLOAT!";
        return;
    }

    tmp.value = val;
    prOptions[variName] = tmp;
}


void HTTraQt::SetProfile(const QString& variName, int val)
{
    if (variName.length() < 1) {
        return;
    }

    OptStruct tmp = prOptions[variName];

    if (tmp.type != INTEGER) {
        qDebug() << "Wrong type, setProfile()" << variName << "was expected INTEGER!";
        return;
    }

    tmp.value = val;
    prOptions[variName] = tmp;
}


void  HTTraQt::SetProfile(const QString& variName, const QString &val)
{
    if (variName.length() < 1) {
        return;
    }

    OptStruct tmp = prOptions[variName];

    if (tmp.type != TEXT) {
        qDebug() << "Wrong type, setProfile()" << variName << "was expected TEXT!";
        return;
    }

    tmp.value = val;
    prOptions[variName] = tmp;
}


void HTTraQt::GetProfile(const QString& variName, float &val)
{
    if (variName.length() < 1) {
        return;
    }

    OptStruct tmp = prOptions[variName];

    if (tmp.type != FLOAT) {
        qDebug() << "Wrong type, getProfile()" << variName << "was expected FLOAT!";
        return;
    }

    val = tmp.value.toFloat();
}


void HTTraQt::GetProfile(const QString& variName, int &val)
{
    if (variName.length() < 1) {
        return;
    }

    OptStruct tmp = prOptions[variName];

    if (tmp.type != INTEGER) {
        qDebug() << "Wrong type, getProfile()" << variName << "was expected INTEGER!";
        return;
    }

    val = tmp.value.toInt();
}


void HTTraQt::GetProfile(const QString& variName, QString &val)
{
    if (variName.length() < 1) {
        return;
    }

    OptStruct tmp = prOptions[variName];

    if (tmp.type != TEXT) {
        qDebug() << "Wrong type, getProfile()" << variName << "was expected TEXT!";
        return;
    }

    val = tmp.value.toString();
}


void HTTraQt::getMainOptionsFromGUI()
{
    QString str;
    QStringList sList;

    // str = firstWidget->label1027->currentText();
    // SetProfileString(mainOptions, "Category", str);
    str = (static_cast<OptionsTab*>(widgets[2]))->textEdit->toPlainText();
    SetProfile("CurrentUrl", str);

    //end of combobox
    str = (static_cast<OptionsTab*>(widgets[2]))->label1286->text();
    SetProfile("CurrentURLList", str);
}


void HTTraQt::gOptions(bool dialog)
{
    if (mth == NULL ) {
        return;
    }

    OptionsDialog *oDia;
    httrackp *opt = hts_create_opt();

    opt->log = opt->errlog = NULL;

    // dévalider champs (non modifiés)
    opt->maxsite = -1;
    opt->maxfile_nonhtml = -1;
    opt->maxfile_html = -1;
    opt->maxsoc = -1;
    opt->nearlink = -1;
    opt->timeout = -1;
    opt->rateout = -1;
    opt->maxtime = -1;
    //     opt->check_type = 0;
    opt->mms_maxtime = -1;
    opt->maxrate = -1;
    StringClear(opt->user_agent);
    opt->retry = -1;
    opt->hostcontrol = -1;
    opt->errpage = -1;
    opt->travel = -1;
    opt->external = -1;
    opt->delete_old = -1;
    opt->parseall = -1;
    opt->delete_old = -1;
    opt->travel = 0;       // NOTE: NE SERA PRIS EN COMPTE QUE LE BIT 8

    if (dialog == true) {
        oDia = new OptionsDialog(this);
        oDia->exec();
    }

    int n;
    float nf;
    GetProfile("Test", n);

    if (n) {
        opt->travel |= 256;
    }

    //     GetProfile("CheckType", n);
    //     opt->check_type = n;

    GetProfile("ParseAll", n);

    if (n) {
        opt->parseall = 1;
    } else {
        opt->parseall = 0;
    }

    // near link,err page
    GetProfile("Near", n);

    if (n) {
        opt->nearlink = 1;
    } else {
        opt->nearlink = 0;
    }

    GetProfile("NoErrorPages", n);

    if (n) {
        opt->errpage = 1;
    } else {
        opt->errpage = 0;
    }

    GetProfile("NoExternalPages", n);

    if (n) {
        opt->external = 1;
    } else {
        opt->external = 0;
    }

    GetProfile("NoPurgeOldFiles", n);

    if (n) {
        opt->delete_old = 1;
    } else {
        opt->delete_old = 0;
    }


    // host control
    {
        int a = 0;
        GetProfile("RemoveTimeout", n);

        if (n > 0) {
            a += 1;
        }

        GetProfile("RemoveRateout", n);

        if (n > 0) {
            a += 2;
        }

        opt->hostcontrol = a;
    }

    // sockets
    GetProfile("Sockets", n);

    if (n >= 0) {
        opt->maxsoc = n;
        maxProgressRows = n;
    } else {
        maxProgressRows = 8;
        opt->maxsoc = 8;
    }

    // maxfile_nonhtml
    GetProfile("MaxOther", nf);

    if (nf > 0) {
        opt->maxfile_nonhtml = (qint64)(nf * 1024.0 * 1024.0);
    } else {
        opt->maxfile_nonhtml = -1;
    }

    // maxfile_html
    GetProfile("MaxHtml", nf);

    if (nf > 0) {
        opt->maxfile_html = (qint64)nf * 1024.0 * 1024.0;
    } else {
        opt->maxfile_html = -1;
    }

    // maxsite
    GetProfile("MaxAll", nf);

    if (nf > 0) {
        opt->maxsite = (qint64)(nf * 1024.0 * 1024.0);
    } else {
        opt->maxsite = -1;
    }

    // fragment
    GetProfile("MaxWait", n);

    if (n > 0) {
        opt->fragment = (qint64)n;
    } else {
        opt->fragment = -1;
    }

    // timeout
    GetProfile("TimeOut", n);

    if (n >= 0) {
        opt->timeout = n;
    } else {
        opt->timeout = -1;
    }

    // rateout
    GetProfile("RateOut", n);

    if (n != 0) {
        opt->rateout = n;
    } else {
        opt->rateout = -1;
    }

    // maxtime
    GetProfile("MaxTime", n);

    if (n > 0) {
        opt->maxtime = n;
    } else {
        opt->maxtime = -1;
    }

    // maxrate
    GetProfile("MaxRate", nf);

    if (nf > 0) {
        opt->maxrate = (int)(nf * 1024.0 * 1024.0);
    } else {
        opt->maxrate = -1;
    }

    // max. connect
    GetProfile("MaxConn", n);

    if (n > 0) {
        opt->maxconn = n;
    } else {
        opt->maxconn = -1;
    }

    // retry
    GetProfile("Retry", n);

    if (n > 0) {
        opt->retry = n;
    } else {
        opt->retry = -1;
    }

    // user_agent
    QString os, st;
    GetProfile("BrowserID", st);
    GetProfile("osID", os);

    if (st.length() > 0) {
        if (st.contains("%s")) {
            st.replace("%s", os);
        }

        SetProfile("UserID", st);
        StringCopy(opt->user_agent, st.toLatin1().data());
    }

    if (global_opt != NULL) {
        copy_htsopt(opt, global_opt);
    }

    hts_free_opt(opt);

    if (dialog == true) {
        delete oDia;
        writeSettings(false);
    }
}


void HTTraQt::setOptions()
{
    gOptions(false); // without dialog
}


void HTTraQt::modifyOptions()
{
    gOptions(true);
}



void HTTraQt::cmdArgumentsOptions()
{
    QString st;
    int n;
    float nf;
    int n1;
    char choixdeb;
    int action;

    cmdOpt = "";
    action = (static_cast<OptionsTab*>(widgets[2]))->comboAction->currentIndex(); //GetProfile("CurrentAction", action);

    m_todo = action;

    QString array;
    array = "wWgY!/i";

    choixdeb = array.at(action).toLatin1();

    if (choixdeb != 'W') {
        cmdOpt += "-q ";
    }

    GetProfile("BuildTopIndex", n);

    if (n == 0) {
        cmdOpt += "-%i0 ";
    } else {
        cmdOpt += "-%i ";
    }

    if (choixdeb == '/') {
        cmdOpt += "-i ";
    } else if (choixdeb != '!') {
        cmdOpt += ("-" + QString(choixdeb) + " ");
    }

    QString url, urllist;
    GetProfile("CurrentUrl", url);
    url.replace("\n", " ");
    url.simplified();

    GetProfile("CurrentURLList", urllist);

    if (url.length() == 0 && urllist.length() == 0) {
        // nothong to download
        return;
    }

    if (url.length() > 0) { // ersetzen!
        cmdOpt += " " + url + " ";//+"\"";
    }

    if (urllist.length() > 0) {
        cmdOpt += (" -%L \"" + urllist + "\" ");
    }

    GetProfile("Depth", n);

    if (n >= 0) {
        cmdOpt += ("-r" + QString::number(n) + " ");
    }

    GetProfile("ExtDepth", n);

    if (n >= 0) {
        cmdOpt += ("-%e" + QString::number(n) + " ");
    }

    if (choixdeb == '/') {
        cmdOpt += "-C1 ";
    } else {
        GetProfile("Cache", n);

        if (n == 0) {
            cmdOpt += "-C0 ";
        } else {
            cmdOpt += "-C2 ";
        }
    }

    GetProfile("NoRecatch", n);

    if (n != 0) {
        cmdOpt += "-%n ";
    }

    GetProfile("Test", n);

    if (n == 1) {
        cmdOpt += "-t ";
    }

    GetProfile("ParseAll", n);

    if (n == 1) {
        cmdOpt += "-%P ";
    } else {
        cmdOpt += "-%P0 ";
    }

    GetProfile("Near", n);

    if (n == 1) {
        cmdOpt += "-n ";
    }

    GetProfile("NoExternalPages", n);

    if (n != 0) {
        cmdOpt += "-x ";
    }

    GetProfile("NoPurgeOldFiles", n);

    if (n != 0) {
        cmdOpt += "-X0 ";
    }

    GetProfile("NoPwdInPages", n);

    if (n != 0) {
        cmdOpt += "-%x ";
    }

    GetProfile("NoQueryStrings", n);

    if (n != 1) { // include-query-string
        cmdOpt += "-%q ";
    }

    GetProfile("FollowRobotsTxt", n);

    if (n >= 0) {
        cmdOpt += ("-s" + QString::number(n) + " ");
    }

    GetProfile("Cookies", n);

    if (n == 0) {
        cmdOpt += "-b0 ";
    }

    GetProfile("CheckType", n);

    if (n > 0) {
        cmdOpt += ("-u" + QString::number(n) + " ");
    }

    GetProfile("ParseJava", n);

    if (n == 0) {
        cmdOpt += "-j0 ";
    }

    GetProfile("StoreAllInCache", n);

    if (n != 0) {
        cmdOpt += "-k ";
    }

    GetProfile("LogType", n);

    if (n == 1) {
        cmdOpt += "-z ";
        GetProfile("Debug", n);

        if (n == 1) {
            cmdOpt += "-%H ";
        }
    } else if (n == 2) {
        cmdOpt += "-Z ";
        GetProfile("Debug", n);

        if (n == 1) {
            cmdOpt += "-%H ";
        }
    }

    GetProfile("HTTP10", n);

    if (n != 0) {
        cmdOpt += "-%h ";
    }

    GetProfile("TolerantRequests", n);

    if (n > 0) {
        cmdOpt += "-%B ";
    }

    GetProfile("UpdateHack", n);

    if (n > 0) {
        cmdOpt += "-%s ";
    }

    GetProfile("URLHack", n);

    if (n > 0) {
        cmdOpt += "-%u ";
    } else {
        cmdOpt += "-%u0 ";
    }

    if (choixdeb != 'g') {
        QString bld;

        GetProfile("Build", n);

        if (n == 14) {   // i.O.
            GetProfile("BuildString", st);
            bld = ("-N \"" + st + "\"");
            //             cmdOpt += bld;
        } else {
            if (n >= 0 && n <= 5) {
                bld = "-N" + QString::number(n);
            } else if (n >= 6 && n <= 11) {
                bld = "-N10" + QString::number(n - 6);
            } else if (n == 12) {
                bld = "-N99";
            } else if (n == 13) {
                bld = "-N199";
            }
        }

        if (bld.length() > 0) {
            cmdOpt += (bld + " ");
        }
    }

    GetProfile("Dos", n);
    GetProfile("ISO9660", n1);

    if (n > 0) {
        cmdOpt += "-L0 ";
    } else if (n1 > 0) {
        cmdOpt += "-L2 ";
    }

    GetProfile("Index", n);

    if (n == 0) {
        cmdOpt += "-I0 ";
    }

    GetProfile("WordIndex", n);

    if (n == 0) {
        cmdOpt += "-%I0 ";
    } /*else {

        cmdOpt += "-%I ";
    }*/
    /*
        GetProfile("HTMLFirst", n);

        if (n == 1) {
            cmdOpt += "-p7 ";
        }
    */
    QString filter;
    GetProfile("PrimaryScan", n);      // filter

    if (n >= 0 && n <= 2) {
        filter = "-p" + QString::number(n) + " ";
    } else if (n == 3) {   /* default */
        GetProfile("HTMLFirst", n1);

        if (!n1) {
            filter = "-p3 ";
        }
    } else {
        if (n == 4) {
            filter = "-p7 ";
        }
    }


    GetProfile("Travel", n);

    if (n == 0) {
        filter += "-S ";
    } else if (n == 1) {
        filter += "-D ";
    } else if (n == 2) {
        filter += "-U ";
    } else if (n == 3) {
        filter += "-B ";
    }

    //
    GetProfile("GlobalTravel", n);

    if (n == 0) {
        filter += "-a ";
    } else if (n == 1) {
        filter += "-d ";
    } else if (n == 2) {
        filter += "-l ";
    } else if (n == 3) {
        filter += "-e ";
    }

    //
    GetProfile("RewriteLinks", n);

    if (n == 0) {
        filter += "-K0 ";
    } else if (n == 1) {
        filter += "-K ";
    } else if (n == 2) {
        filter += "-K3 ";
    } else if (n == 3) {
        filter += "-K4 ";
    }

    cmdOpt += filter;

    //sizemax
    GetProfile("MaxAll", nf);

    if (n > 0) {
        cmdOpt += ("-M" + QString::number((qint64)(nf * 1024.0 * 1024.0)) + " ");
    }

    //pausebytes
    GetProfile("MaxWait", n);

    if (n > 0) {
        cmdOpt += ("-G" + QString::number(n) + " ");
    }

    float nf1;
    GetProfile("MaxHtml", nf);
    GetProfile("MaxOther", nf1);

    if (nf >= 0 ||  nf1 >= 0) {
        cmdOpt += "-m";

        if (nf1 > 0) {
            cmdOpt += QString::number((qint64)(nf1 * 1024.0 * 1024.0));
        } else {
            cmdOpt += "0";
        }

        if (nf > 0) {
            cmdOpt += ("," + QString::number((qint64)(nf * 1024.0 * 1024.0)));
        } else {
            cmdOpt += ",0";
        }

        cmdOpt += " ";
    }

    GetProfile("Sockets", n);

    if (n >= 0) {
        maxProgressRows = n;
    } else {
        maxProgressRows = 8;
    }

    cmdOpt += ("-c" + QString::number(n) + " ");

    GetProfile("TimeOut", n);

    if (n > 0) {
        cmdOpt += ("-T" + QString::number(n) + " ");
    }

    GetProfile("RateOut", n);

    if (n > 0) {
        cmdOpt += ("-J" + QString::number(n) + " ");
    }

    GetProfile("Retry", n);

    if (n > 0) {
        cmdOpt += ("-R" + QString::number(n) + " ");
    }

    int a = 0;
    GetProfile("RemoveTimeout", n);

    if (n > 0) {
        a += 1;
    }

    GetProfile("RemoveRateout", n);

    if (n > 0) {
        a += 2;
    }

    if (a > 0 ) {
        cmdOpt += ("-H" + QString::number(a) + " ");
    }

    GetProfile("KeepAlive", n);

    if (n > 0) {
        cmdOpt += "-%k ";
    } else {
        cmdOpt += "-%k0 ";
    }

    GetProfile("Log", n);

    if (n != 0) {
        cmdOpt += "-f2 ";
    } else {
        cmdOpt += "-Q ";
    }

    GetProfile("NoErrorPages", n);

    if (n > 0) {
        cmdOpt += "-o0 ";
    }

    //
    GetProfile("MaxTime", n);

    if (n > 0) {
        cmdOpt += ("-E" + QString::number(n) + " ");         // max timeout
    }

    GetProfile("MaxRate", nf);

    if (nf > 0.0) {
        cmdOpt += ("-A" + QString::number((int)(nf * 1024.0 * 1024.0)) + " ");     // max rate
    }

    //     maxConnections = 0;

    GetProfile("MaxConn", n);

    if (n > 0) {
        cmdOpt += ("-%c" + QString::number(n) + " ");      // max connections
        //         maxConnections = n;
    }

    GetProfile("MaxLinks", n);

    if (n > 0) {
        cmdOpt += ("-#L" + QString::number(n) + " ");         // max links
    }

    GetProfile("UseHTTPProxyForFTP", n);

    if (n > 0) {
        cmdOpt += "-%f ";
    } /*else {

        cmdOpt += "-%f0 ";
    }*/

    //     cmdOpt += "#f";  // flush

    QString os;
    GetProfile("BrowserID", st);
    GetProfile("osID", os);

    if (st.length() > 0) {
        if (st.contains("%s")) {
            st.replace("%s", os);
        }

        // qDebug() << "cmd" << st;
        cmdOpt += (" -F \"" + st + "\" ");
    }

    GetProfile("Footer", st);

    if (st.length() > 0) {
        QString stTmp = st;
        cmdOpt += (" -%F \"" + stTmp + "\" ");
    }

    GetProfile("Proxy", st);

    if (st.length() > 0) {
        GetProfile("ProxyPort", n);
        cmdOpt += (" -P " + st + ":" + QString::number(n) + " ");
    }

    // lang iso
    QString lng_iso = selectedLang;//translate("LANGUAGE_ISO");

    if (lng_iso.length() > 0) {
        cmdOpt +=  " -%l \"" + lng_iso;

        if (lng_iso != "en") {
            cmdOpt += ",en";
        }

        cmdOpt += "\" ";
    }

    // end of lang
    if (choixdeb == '!') {
        cmdOpt += " --testlinks ";
    } else if (choixdeb == 'Y') {
        cmdOpt += " --mirrorlinks ";
    }

    if (currentProject.length() > 0) {
        cmdOpt += " -O \"" + currentWorkDir + "/" + currentProject + "/\" ";
    }

    GetProfile("WildCardFilters", st);

    if (st.length() > 0) {
        cmdOpt += (" " + st);
    }

    for (int an = 1; an < 9; an++) {
        QString st1, st2;
        QString n1, n2;
        n1 = "MIMEDefsMime" + QString::number(an);
        n2 = "MIMEDefsExt" + QString::number(an);
        GetProfile(n1, st1);
        GetProfile(n2, st2);

        if (st1.length() > 0 && st2.length() > 0) {
            cmdOpt += (" --assume " + st2 + "=" + st1 + " ");
        }
    }

    // clean
    StripControls(cmdOpt);
}


void HTTraQt::setMainOptionsToGUI()
{
    QString str;
    QStringList sList;

    readSettings(true);
    // GetProfileString(mainOptions, "Category", str);
    //   str = firstWidget->label1027->Text();
    GetProfile("CurrentUrl", str);
    str.replace(" ", "\n");

    (static_cast<OptionsTab*>(widgets[2]))->textEdit->setText(str);

    str = (static_cast<OptionsTab*>(widgets[2]))->comb01;
    sList = str.split("\n");

    GetProfile("CurrentURLList", str);
    (static_cast<OptionsTab*>(widgets[2]))->label1286->setText(str);
}


bool HTTraQt::RemoveEmptyDir(QString path)
{
    int ans = MessageBox::exec(translate(_REMOVE), translate(_SURE), QMessageBox::Question);

    if (ans == QMessageBox::No) {
        return false;
    }

    QDirIterator it(path, QDirIterator::Subdirectories);

    while (it.hasNext()) {
        QString n = it.next();

        if (n == "." || n == "..") {
            continue;
        }

        QString fullPath = path + "/" + n;

        QDir subd(fullPath);

        if (subd.rmdir(fullPath) == false) {
            MessageBox::exec(translate(_ERR), translate(_ANYDIRNOTEMPTY), QMessageBox::Warning);

            return false;
        }
    }

    QStringList names;

    QString maske = path + "/*.*";
    QDir d(path);
    names = d.entryList(QStringList("*.*"));

    for (QStringList::Iterator ist = names.begin(); ist != names.end(); ++ist) {
        if (*ist == "." || *ist == "..") {   // is dir
            continue;
        } else {
            d.remove(path + "/" + (*ist));
        }
    }

    if (d.remove(maske) == true) {
        d.rmdir(path);
    }

#ifdef WIN32
    QDir::setCurrent("C:\\");

#else
    QString home = QDir::homePath();

    QDir::setCurrent(home);

#endif

    return true;
}

