/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file ffgtk/audio.c
 * \brief Play sounds
 */

#include <ffgtk.h>
#include <faxophone/isdn-convert.h>
#include <sndfile.h>

#define QUANT_MASK	( 0xF )
#define SEG_SHIFT	( 4 )
#define SEG_MASK	( 0x70 )
#define SIGN_BIT	( 0x80 )

#define RING_PERIOD			4
#define RING_SHORT_PERIOD	0.044
#define RING_FADE_LENGTH	0.003
#define RING_LENGTH			1.3
#define RING_SHORT_LENGTH	(RING_SHORT_PERIOD/2)
#define RING_FREQUENCY		1300
#define RINGING_FREQUENCY	425

static short anSegEnd[ 8 ] = { 0xFF, 0x1FF, 0x3FF, 0x7FF, 0xFFF, 0x1FFF, 0x3FFF, 0x7FFF };
static unsigned char *pnAudioLutIn = NULL;
static unsigned char *pnAudioLutGenerate = NULL;
static int nStopEffect = 0;

/**
 * \brief Small index search
 * \return index
 */
static int search( int nVal, short *pnTable, int nSize ) {
	int nIndex;

	for ( nIndex = 0; nIndex < nSize; nIndex++ ) {
		if ( nVal <= *pnTable++ ) {
			return nIndex;
		}
	}

	return nSize;
}

/**
 * \brief Convert linear to alaw value
 * \param nVal linear value
 * \return alaw value
 */
static unsigned char linear2alaw( int nVal ) {
	int nMask;
	int nSeg;
	unsigned char nValA;

	if ( nVal >= 0 ) {
		nMask = 0xD5;
	} else {
		nMask = 0x55;
		nVal = -nVal - 8;
	}

	nSeg = search( nVal, anSegEnd, 8 );

	if ( nSeg >= 8 ) {
		return ( 0x7F ^ nMask );
	} else {
		nValA = nSeg << SEG_SHIFT;
		if ( nSeg < 2 ) {
			nValA |= ( nVal >> 4 ) & QUANT_MASK;
		} else {
			nValA |= ( nVal >> ( nSeg + 3 ) ) & QUANT_MASK;
		}

		return ( nValA ^ nMask );
	}
}

/**
 * \brief Convert alaw to linear
 * \param nValA alaw value we want to convert
 * \return linear value
 */
static short alaw2linear( unsigned char nValA ) {
	short nT;
	int nSeg;

	nValA ^= 0x55;

	nT = ( nValA & QUANT_MASK ) << 4;
	nSeg = ( ( unsigned ) nValA & SEG_MASK ) >> SEG_SHIFT;

	switch ( nSeg ) {
		case 0:
			nT += 8;
			break;
		case 1:
			nT += 0x108;
			break;
		default:
			nT += 0x108;
			nT <<= nSeg - 1;
	}

	return ( ( nValA & SIGN_BIT ) ? nT : -nT );
}

/**
 * \brief Generate fx sound
 * \param nType sound type
 * \param nIndex index
 * \param fSeconds seconds
 * \return current byte
 */
unsigned char fxGenerate( eCallType nType, int nIndex, double fSeconds ) {
	double fRest;
	double fRest2;
	double fFactor;
	double fFactor2;
	unsigned char nX = 0;

	switch ( nType ) {
		case CALL_TYPE_INCOMING:
			/* 4 sec period */
			fRest = fmod( fSeconds, RING_PERIOD );
			/* short period */
			fRest2 = fmod( fSeconds, RING_SHORT_PERIOD );

			if ( fRest < RING_FADE_LENGTH ) {
				/* Fade in */
				fFactor = -cos( fRest * M_PI / RING_FADE_LENGTH ) / 2 + 0.5;
			} else if ( fRest > RING_LENGTH - RING_FADE_LENGTH && fRest < RING_LENGTH ) {
				/* Fade out */
				fFactor = -cos( ( RING_LENGTH - fRest ) * 2 * M_PI / ( RING_FADE_LENGTH * 2 ) ) / 2 + 0.5;
			} else if ( fRest >= RING_LENGTH ) {
				/* Pause */
				fFactor = 0;
			} else {
				/* Beep */
				fFactor = 1;
			}

			if ( fRest2 > RING_SHORT_PERIOD - 0.5 * RING_FADE_LENGTH ) {
				/* Fade in short (1/1) */
				fFactor2 = -sin( ( RING_SHORT_PERIOD - fRest2 ) * 2 * M_PI / ( RING_FADE_LENGTH * 2 ) ) / 2 + 0.5;
			} else if ( fRest2 < 0.5 * RING_FADE_LENGTH ) {
				/* Fade in short (2/2) */
				fFactor2 = sin( fRest2 * 2 * M_PI / ( RING_FADE_LENGTH * 2 ) ) / 2 + 0.5;
			} else if ( fRest2 > RING_SHORT_LENGTH - 0.5 * RING_FADE_LENGTH && fRest2 < RING_SHORT_LENGTH + 0.5 * RING_FADE_LENGTH ) {
				fFactor2 = -sin( ( fRest2 - RING_SHORT_LENGTH ) * 2 * M_PI / ( RING_FADE_LENGTH * 2 ) ) / 2 + 0.5;
			} else if ( fRest2 <= RING_SHORT_LENGTH ) {
				/* Beep */
				fFactor2 = 1;
			} else {
				/* Pause */
				fFactor2 = 0;
			}

			fFactor = fFactor * fFactor2;

			nX = pnAudioLutGenerate[ ( int )( sin( fSeconds * 2 * M_PI * RING_FREQUENCY ) * fFactor * 127.5 + 127.5 ) ];
			break;
		case CALL_TYPE_OUTGOING:
			/* waiting for the other end to pick up the phone */
			fRest = fmod( fSeconds, 5 );

			if ( fRest >= 2 && fRest < 3 ) {
				/* beep */
				nX = pnAudioLutGenerate[ ( int )( sin( fSeconds * 2 * M_PI * RINGING_FREQUENCY ) * 127.5 + 127.5 ) ];
			} else {
				/* pause */
				nX = pnAudioLutGenerate[ 128 ];
			}
			break;
		default:
			Debug( KERN_WARNING, "Unknown effect %d\n", nType );
			nX = 0;
			break;
	}

	return nX;
}

/**
 * \brief Handle effects
 * \param pUserData user data
 * \return error code
 */
static gpointer handleEffect( gpointer pUserData ) {
	eCallType nType = GPOINTER_TO_INT( pUserData );
	const gchar *pnFile = NULL;
	int nJustRead;
	short anBuffer[ 2048 ];
	SNDFILE *psSndFile = NULL;
	SF_INFO sSfInfo;
	unsigned int nIndex;
	int nSample;
	unsigned char anAlawBuffer[ 4096 ];
	unsigned int nAlawCount = 0;
	unsigned char anSndBuffer[ 12 * 4096 ];
	unsigned int nSndCount = 0;
	unsigned long nEffectPos = 0;
	void *pPriv = NULL;
	struct sAudio *psAudio = getDefaultAudioPlugin( getActiveProfile() );
	int nCounter = 0;

	if ( psAudio == NULL ) {
		return NULL;	
	}

	pPriv = psAudio -> Open();
	if ( pPriv == NULL ) {
		return NULL;
	}

	Debug( KERN_DEBUG, "nType: %d\n", nType );
	pnFile = nType == CALL_TYPE_INCOMING ? callMonitorGetIncomingSound( getActiveProfile() ) : callMonitorGetOutgoingSound( getActiveProfile() );

	psSndFile = sf_open( pnFile, SFM_READ, &sSfInfo );
	
	while ( !nStopEffect ) {
		if ( psSndFile != NULL ) {
			nJustRead = sf_readf_short( psSndFile, anBuffer, sizeof( anBuffer ) / 4 );
			if ( nJustRead > 0 ) {
				for ( nIndex = 0; nIndex < ( unsigned int ) nJustRead; ++nIndex ) {
					//nSample = ( ( int ) anBuffer[ 2 * nIndex ] ) + ( ( int ) anBuffer[ 2 * nIndex + 1 ] );
					nSample = anBuffer[ nIndex ];
					if ( nSample < -32768 ) {
						nSample = -32768;
					} else if ( nSample > 32767 ) {
						nSample = 32767;
					}
#ifdef HAVE_FAXOPHONE
					anAlawBuffer[ nIndex ] = bitinverse( linear2alaw( nSample ) );
#endif
				}
				nAlawCount = nJustRead;
			} else {
				nAlawCount = 0;
			}
		} else {
			if ( nType == CALL_TYPE_INCOMING || nType == CALL_TYPE_OUTGOING ) {
				for ( nIndex = 0; nIndex < sizeof( anAlawBuffer ) / 4; ++nIndex ) {
#ifdef HAVE_FAXOPHONE
					anAlawBuffer[ nIndex ] = bitinverse( fxGenerate( nType, 0, nEffectPos++ / 8000.0 ) );
#else
					nEffectPos++;
#endif
				}
				nAlawCount = sizeof( anAlawBuffer ) / 4;
			}
			nCounter++;
			if ( nCounter == 0x20 ) {
				nAlawCount = 0;
			}
		}

		if ( nAlawCount == 0 ) {
			Debug( KERN_DEBUG, "End-Of-File reached, stopping playback\n" );

			break;
		}

#ifdef HAVE_FAXOPHONE
		convertIsdnToAudio( NULL, anAlawBuffer, nAlawCount, anSndBuffer, &nSndCount, NULL );
#endif

		psAudio -> Write( pPriv, anSndBuffer, nSndCount );
	}

	if ( psSndFile != NULL ) {
		sf_close( psSndFile );
		psSndFile = NULL;
	}

	psAudio -> Close( pPriv, FALSE );

	return NULL;
}

/**
 * \brief Create look-up table
 * \param ppnLutIn input lut
 * \param ppnLutGenerate fx generate lut
 * \return error code
 */
static int makeLut( unsigned char **ppnLutIn, unsigned char **ppnLutGenerate ) {
	int nSampleSizeIn = 2;
	int nBufSizeIn;
	int nIndex;
	int nSample;

	if ( nSampleSizeIn == 0 || !( *ppnLutIn = ( unsigned char * ) malloc( nBufSizeIn = nSampleSizeIn * 256 ) ) ) {
		return -1;
	}

	if ( !( *ppnLutGenerate = ( unsigned char * ) malloc( 256 ) ) ) {
		return -1;
	}

	/* Calculation */
	for ( nIndex = 0; nIndex < nBufSizeIn; nIndex += nSampleSizeIn ) {
		nSample = alaw2linear( ( unsigned char )( nIndex / 2 ) );

		( *ppnLutIn )[ nIndex + 0 ] = ( unsigned char )( nSample & 0xFF );
		( *ppnLutIn )[ nIndex + 1 ] = ( unsigned char )( ( nSample >> 8 & 0xFF ) );
	}

	for ( nIndex = 0; nIndex < 256; nIndex++ ) {
		( *ppnLutGenerate )[ nIndex ] = linear2alaw( ( nIndex - 128 ) * 256 );
	}

	return 0;
}

/**
 * \brief Play sounds
 * \param nType call type
 */
void PlayRingtone( eCallType nType ) {
	/* Check if the user wants to play sounds */
	if ( callMonitorGetPlaySounds( getActiveProfile() ) == 0 || callMonitorGetMute( getActiveProfile() ) == TRUE ) {
		return;
	}

	if ( pnAudioLutIn == NULL ) {
		makeLut( &pnAudioLutIn, &pnAudioLutGenerate );
	}

	nStopEffect = 0;
	CREATE_THREAD( "effect", handleEffect, GINT_TO_POINTER(nType) );
}

/**
 * \brief Stop effect handler
 */
void StopRingtone( void ) {
	nStopEffect = 1;
}
