/*
 *   This file is part of Auralquiz
 *   Copyright 2011-2012  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "optionsdialog.h"

/*
 *   Constructor, called on startup even if options dialog is never shown
 *
 */
OptionsDialog::OptionsDialog()
{
    QSettings settings;
    this->setWindowTitle(tr("Game options"));
    this->setWindowIcon(QIcon::fromTheme("configure"));
    this->setWindowModality(Qt::ApplicationModal);

    this->resize(700, 380);

    playMode = false;

    this->musicDirectory = settings.value("musicDirectory",
                                    QDesktopServices::storageLocation(QDesktopServices::MusicLocation)).toString();
    this->reload = false;

    // TOP layout
    topLayout = new QHBoxLayout();
    musicFromLabel = new QLabel(tr("Using music from:") + " ");
    topLayout->addWidget(musicFromLabel);
    currentMusicDirectoryLabel = new QLabel("");
    currentMusicDirectoryLabel->setText(this->musicDirectory);
    topLayout->addWidget(currentMusicDirectoryLabel);
    topLayout->addSpacing(32);
    selectMusicDirectoryButton = new QPushButton(QIcon::fromTheme("folder-sound"),
                                                 tr("Select your &music folder..."));
    connect(selectMusicDirectoryButton, SIGNAL(clicked()),
            this, SLOT(selectMusicDirectory()));
    topLayout->addWidget(selectMusicDirectoryButton);


    // MIDDLE layout
    middleLayout = new QHBoxLayout();

    // left side
    difficultyTopLayout = new QHBoxLayout();

    //// difficulty sub-layout
    difficultyEasy = new QLabel(tr("Easier"));
    difficultyTopLayout->addWidget(difficultyEasy);

    difficultyLevel = new QSlider(Qt::Horizontal);
    difficultyLevel->setRange(0, 5); // 0 = very easy, 5 = hardcore
    difficultyLevel->setValue(settings.value("difficultyLevel", 2).toInt());
    difficultyLevel->setTickInterval(1); // One tick mark for every difficulty level
    difficultyLevel->setTickPosition(QSlider::TicksBothSides);
    connect(difficultyLevel, SIGNAL(valueChanged(int)), this, SLOT(updateDifficultyName(int)));
    difficultyTopLayout->addWidget(difficultyLevel);

    difficultyHard = new QLabel(tr("Harder"));
    difficultyTopLayout->addWidget(difficultyHard);

    QFont difficultyFont;
    difficultyFont.setPointSize(14);
    difficultyFont.setBold(true);
    difficultyName = new QLabel();
    difficultyName->setFont(difficultyFont);
    difficultyName->setAlignment(Qt::AlignCenter);

    difficultyLayout = new QVBoxLayout();
    difficultyLayout->setAlignment(Qt::AlignCenter);
    difficultyLayout->addLayout(difficultyTopLayout);
    difficultyLayout->addWidget(difficultyName);

    middleLayout->addLayout(difficultyLayout);

    updateDifficultyName(this->difficultyLevel->value()); // update difficulty label initially

    middleLayout->addSpacing(32);



    // right side
    rightSideLayout = new QGridLayout();

    numQuestionsLabel = new QLabel(tr("How many &questions?"));
    rightSideLayout->addWidget(numQuestionsLabel, 0, 0);

    numQuestions = new QSpinBox();
    numQuestions->setRange(1, 99);
    numQuestions->setValue(settings.value("numQuestions", 25).toInt());
    numQuestions->setSuffix(" " + tr("questions"));
    rightSideLayout->addWidget(numQuestions, 0, 1);

    numQuestionsLabel->setBuddy(this->numQuestions);

    numPlayersLabel = new QLabel(tr("How many &players?"));
    rightSideLayout->addWidget(numPlayersLabel, 1, 0);

    numPlayers = new QSpinBox();
    numPlayers->setRange(1, 8); // up to 8 players
    numPlayers->setValue(settings.value(("numPlayers"), 1).toInt());
    numPlayers->setSuffix(" " + tr("players"));
    rightSideLayout->addWidget(numPlayers, 1, 1);

    numPlayersLabel->setBuddy(this->numPlayers);

    useOwnColors = new QCheckBox(tr("Use own color theme"));
    useOwnColors->setChecked(settings.value("useOwnColorTheme", false).toBool());
    rightSideLayout->addWidget(useOwnColors, 2, 0);

    middleLayout->addLayout(this->rightSideLayout);

    // PLAYER NAMES layout

    playerNamesLayout = new QGridLayout();


    playerNames = settings.value("playerNames",
                                 QStringList() << "Prime"
                                               << "Jazz"
                                               << "Wheeljack"
                                               << "Skyfire"
                                               << "Megatron"
                                               << "Starscream"
                                               << "Soundwave"
                                               << "Astrotrain").toStringList();

    player1Number = new QLabel("  1");
    player1Name = new QLineEdit(playerNames.at(0));
    player2Number = new QLabel("  2");
    player2Name = new QLineEdit(playerNames.at(1));
    player3Number = new QLabel("  3");
    player3Name = new QLineEdit(playerNames.at(2));
    player4Number = new QLabel("  4");
    player4Name = new QLineEdit(playerNames.at(3));
    player5Number = new QLabel("  5");
    player5Name = new QLineEdit(playerNames.at(4));
    player6Number = new QLabel("  6");
    player6Name = new QLineEdit(playerNames.at(5));
    player7Number = new QLabel("  7");
    player7Name = new QLineEdit(playerNames.at(6));
    player8Number = new QLabel("  8");
    player8Name = new QLineEdit(playerNames.at(7));

    playerNamesLayout->addWidget(player1Number, 2, 0);
    playerNamesLayout->addWidget(player1Name,   2, 1);
    playerNamesLayout->addWidget(player2Number, 2, 2);
    playerNamesLayout->addWidget(player2Name,   2, 3);
    playerNamesLayout->addWidget(player3Number, 3, 0);
    playerNamesLayout->addWidget(player3Name,   3, 1);
    playerNamesLayout->addWidget(player4Number, 3, 2);
    playerNamesLayout->addWidget(player4Name,   3, 3);
    playerNamesLayout->addWidget(player5Number, 4, 0);
    playerNamesLayout->addWidget(player5Name,   4, 1);
    playerNamesLayout->addWidget(player6Number, 4, 2);
    playerNamesLayout->addWidget(player6Name,   4, 3);
    playerNamesLayout->addWidget(player7Number, 5, 0);
    playerNamesLayout->addWidget(player7Name,   5, 1);
    playerNamesLayout->addWidget(player8Number, 5, 2);
    playerNamesLayout->addWidget(player8Name,   5, 3);

    playerNamesGroup = new QGroupBox(tr("Player names"));
    playerNamesGroup->setLayout(playerNamesLayout);



    // BOTTOM layout
    bottomLayout = new QHBoxLayout();

    saveButton = new QPushButton("SAVE");
    connect(saveButton, SIGNAL(clicked()), this, SLOT(saveOptions()));
    bottomLayout->addWidget(saveButton);

    reloadMusicInfoButton = new QPushButton(QIcon::fromTheme("view-refresh"),
                                            tr("Save and &reload music"));
    connect(reloadMusicInfoButton, SIGNAL(clicked()),
            this, SLOT(saveAndReload()));
    bottomLayout->addWidget(reloadMusicInfoButton);


    cancelButton = new QPushButton(QIcon::fromTheme("dialog-cancel"), tr("&Cancel"));
    connect(cancelButton, SIGNAL(clicked()), this, SLOT(close()));
    bottomLayout->addWidget(cancelButton);



    mainLayout = new QVBoxLayout();
    mainLayout->addLayout(topLayout);
    mainLayout->addLayout(middleLayout);
    mainLayout->addWidget(playerNamesGroup);
    mainLayout->addLayout(bottomLayout);
    this->setLayout(mainLayout);


    qDebug() << "Options window created";
}


OptionsDialog::~OptionsDialog()
{
    qDebug() << "Options window destroyed";
}


void OptionsDialog::showEvent(QShowEvent *event)
{
    event->accept();
    qDebug() << "Options window shown";

    this->reload = false; // make forceReload false again
                          // until Reload button is clicked
}


void OptionsDialog::closeEvent(QCloseEvent *event)
{
    event->accept();
    qDebug() << "Options dialog closed";
}



/*
 *   Open dialog to select directory where music is located
 *
 */
void OptionsDialog::selectMusicDirectory()
{
    QString newMusicDir;
    newMusicDir = QFileDialog::getExistingDirectory(this,
                                                    tr("Select where your music is located"),
                                                    this->musicDirectory,
                                                    QFileDialog::ShowDirsOnly);
    if (!newMusicDir.isEmpty()) // if user DID NOT cancel the "select directory" dialog
    {
        musicDirectory = newMusicDir;
        if (!musicDirectory.endsWith("/"))
        {
            // Adding final "/" if it's not present in chosen path
            musicDirectory.append("/");
        }
        // update label indicating current path for music
        currentMusicDirectoryLabel->setText(this->musicDirectory);
    }
}


/*
 *   Update the label with the description of the selected difficulty level
 *
 */
void OptionsDialog::updateDifficultyName(int newValue)
{
    qDebug() << "new difficulty level:" << newValue;
    QStringList levels = QStringList(tr("Piece of cake"))
                         << tr("Let's rock")
                         << tr("Come get some")
                         << tr("Damn I'm good")
                         << tr("HARD!!")
                         << tr("Hardcore! Type the answer");

    this->difficultyName->setText("[ " + levels.at(newValue) + " ]");

    if (newValue == 5) // Hardcore level
    {
        difficultyName->setToolTip(tr("In this level you'll have to type the answer.\n"
                                      "It's not too strict, so you don't have to worry about caps, commas and such."));
    }
    else
    {
        difficultyName->setToolTip(tr("Is this level you'll have to click the correct button,\n"
                                      "or press the 1, 2, 3 and 4 keys in your keyboard."));
    }
}



/*
 *   Emit signal with new option values to AuralWindow
 *
 */
void OptionsDialog::saveOptions()
{
    playerNames.clear();
    playerNames << player1Name->text()
                << player2Name->text()
                << player3Name->text()
                << player4Name->text()
                << player5Name->text()
                << player6Name->text()
                << player7Name->text()
                << player8Name->text();

    // If a player name from the list is empty, replace with number
    for (int counter=0; counter < playerNames.length(); ++counter)
    {
        if (playerNames.at(counter).isEmpty())
        {
            playerNames[counter] = QString("[%1]").arg(counter+1);
        }
    }



    // FIXME: this IF code is duplicated 3 times
    if (!musicDirectory.endsWith("/"))
    {
        // Adding final "/" if it's not present in chosen path
        musicDirectory.append("/");
    }


    emit optionsChanged(this->playMode,
                        this->musicDirectory,
                        this->reload,
                        this->difficultyLevel->value(),
                        this->numQuestions->value(),
                        this->numPlayers->value(),
                        this->playerNames,
                        this->useOwnColors->isChecked());


    qDebug() << "Options saved";
    qDebug() << "playMode:" << this->playMode;
    this->close();
}


void OptionsDialog::saveAndReload()
{
    this->reload = true;
    qDebug() << "Save and reload music pressed";
    saveOptions();
}


/*
 *  Show Options window in full configuration mode
 *  Used from the "Options" button
 */
void OptionsDialog::showConfigMode()
{
    playMode = false;

    this->musicFromLabel->show();
    this->currentMusicDirectoryLabel->show();
    this->selectMusicDirectoryButton->show();
    this->reloadMusicInfoButton->show();

    this->useOwnColors->show();

    bottomLayout->setAlignment(Qt::AlignBottom | Qt::AlignRight);

    this->saveButton->setIcon(QIcon::fromTheme("document-save"));
    this->saveButton->setText(tr("&Save options"));
    this->saveButton->setMinimumSize(1, 1);
    this->cancelButton->show();

    this->show();
}


/*
 *  Show Options window in a "light mode"
 *  This will be used from Start Game button, to set players and difficulty only
 *  (music config and theme options will be hidden)
 */
void OptionsDialog::showPlayMode()
{
    playMode = true;

    this->musicFromLabel->hide();
    this->currentMusicDirectoryLabel->hide();
    this->selectMusicDirectoryButton->hide();
    this->reloadMusicInfoButton->hide();

    this->useOwnColors->hide();

    bottomLayout->setAlignment(Qt::AlignBottom | Qt::AlignCenter);

    this->saveButton->setIcon(QIcon::fromTheme("media-playback-start"));
    this->saveButton->setText(tr("Start!"));
    this->saveButton->setShortcut(Qt::Key_Return);
    this->saveButton->setMinimumSize(240, 60);
    this->cancelButton->hide();

    this->show();
}
