/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* libe-book
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libebook.sourceforge.net
 */

#include <libwpd/WPXDocumentInterface.h>
#include <libwpd/WPXString.h>

#include <libwpd-stream/libwpd-stream.h>

#include "libebook_utils.h"
#include "EBOOKZlibStream.h"
#include "ZTXTParser.h"

using boost::scoped_ptr;

namespace libebook
{

static const uint32_t ZTXT_TYPE = PDX_CODE("zTXT");
static const uint32_t ZTXT_CREATOR = PDX_CODE("GPlm");

ZTXTParser::ZTXTParser(WPXInputStream *const input, WPXDocumentInterface *const document)
  : PDXParser(input, document)
  , m_recordCount(0)
  , m_size(0)
  , m_recordSize(0)
{
}

bool ZTXTParser::isFormatSupported(const unsigned type, const unsigned creator)
{
  return (ZTXT_TYPE == type) && (ZTXT_CREATOR == creator);
}

void ZTXTParser::readAppInfoRecord(WPXInputStream *)
{
}

void ZTXTParser::readSortInfoRecord(WPXInputStream *)
{
}

void ZTXTParser::readIndexRecord(WPXInputStream *const record)
{
  readU16(record, true);
  m_recordCount = readU16(record, true);
  m_size = readU32(record, true);
  m_recordSize = readU16(record, true);
  // the rest is not interesting...
}

void ZTXTParser::readDataRecord(WPXInputStream *const record, bool)
{
  WPXString text;
  while (!record->atEOS())
  {
    const uint8_t c = readU8(record);
    if ('\n' == c)
    {
      handleText(text);
      text.clear();
    }
    else
      text.append(c);
  }

  if (0 < text.len())
    handleText(text);
}

void ZTXTParser::readDataRecords()
{
  const scoped_ptr<WPXInputStream> block(getDataRecords(0, m_recordCount - 1));
  EBOOKZlibStream input(block.get());
  openDocument();
  readDataRecord(&input);
  closeDocument();
}

void ZTXTParser::openDocument()
{
  WPXPropertyList metadata;
  metadata.insert("dc:title", WPXString(getName()));

  getDocument()->startDocument();
  getDocument()->setDocumentMetaData(metadata);
  getDocument()->openPageSpan(WPXPropertyList());
}

void ZTXTParser::closeDocument()
{
  getDocument()->closePageSpan();
  getDocument()->endDocument();
}

void ZTXTParser::handleText(const WPXString &text)
{
  getDocument()->openParagraph(WPXPropertyList(), WPXPropertyListVector());
  getDocument()->openSpan(WPXPropertyList());
  getDocument()->insertText(text);
  getDocument()->closeSpan();
  getDocument()->closeParagraph();
}

} // namespace libebook

/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
