# Filename: solar.tcl
# Purpose:  standard scripts for use with solar
# Authors:  Charles Peterson and Tom Dyer
# Date:     February 20, 1998 (started)
#
# Copyright (C) Southwest Foundation for Biomedical Research, 1998-2011.
# Copyright (C) Texas Biomedical Research Institute, 2011-present
#
# All rights reserved.

proc solar_tcl_version {} {
    return "8.1.1 (General)"
}

proc solar_up_date {} {
    global env
    global auto_path
    global SOLAR_Date
    set fullpath [lindex $auto_path 0]/solar.tcl
    set SOLAR_Date [file mtime $fullpath]
    set bins [glob $env(SOLAR_BIN)/*]
    foreach bin $bins {
	if {[file mtime $bin] > $SOLAR_Date} {
	    set SOLAR_Date [file mtime $bin]
	}
    }
    set fdate [clock format $SOLAR_Date -format "%B %d"]
    return $fdate
}

proc solar_up_year {} {
    global SOLAR_Date
    set year [clock format $SOLAR_Date -format %Y]
    return $year
}

# solar::about --
#
# Purpose:  Copyright, authors, and disclaimers
#
# SOLAR is Copyright (c) 1995-2014 Southwest Foundation for Biomedical
# Research.  All rights reserved.
#
# The authors are John Blangero, Kenneth Lange, Laura Almasy, Harald Goring,
# Jeff Williams, Tom Dyer, Michael Boehnke, and Charles Peterson.  Parts of
# SOLAR consist of software developed by others; a complete list is provided
# in Appendix Four of the documentation included with this package (use the
# "doc" command to find out where that is).
#
# Use of this program should be acknowledged in scientific publications.
# 
# Commands, features, performance, and availability are subject to change.
# There is absolutely no warranty, express or implied.
# There is no committment to support scripts written using current commands
#   in future releases.
# -

proc about {} {
    return [helpscript about]
}

# solar::register --
#
# Purpose:  Create registration key file
#
# Usage:    register <key>
#
# Notes:    This creates a file ~/.solar_reg containing the key.  Do
#           not delete this file.  You may copy this file to your
#           home directory on other systems to run SOLAR if the
#           same username is used.  (Each key matches only one
#           username.)
#
#           To obtain a key, please send a request to solar@txbiomedgenetics.org.
#           specifing the username(s) under which you will be using the
#           program, the email addresses of the users, and the NIH grant
#           numbers (if any) that apply to the work for which SOLAR may
#           be used.
# -

proc register {key} {
    if {[key -check]} {
	puts "You already have a valid ~/.solar_reg key file installed."
	puts "Are you sure you want to do this?  Y or N:"
	gets stdin yorn
	if {"y" != [string tolower $yorn]} {
	    return ""
	}
    }
    exec rm -f ~/.solar_reg
    exec echo $key >~/.solar_reg
    set status [key -validate]
    if {!$status} {
	error "register: Incorrect key was entered.  Try again."
    }
    return "Registration Successful!"
}

# solar::please_register -- private
#
# Please register SOLAR by sending an email to solar@txbiomedgenetics.org
# asking for a solar key(s) and providing the following information:
#
#  1.  login name(s) (on the machine(s) on which SOLAR will be run).  We need
#      the short form of your login name, usually less than 8 characters.
#-

# solar::please_register_2 -- private
#  2.  email address
#
#  3.  if used in research funded by the NIH, please specify grant number(s)
#
# If you have not gotten your registration key in a few days, please ask again.
# Meanwhile, even without registration you can use all SOLAR commands except
# those which maximize models (such as the polygenic and multipoint commands).
# -

proc please_register {} {
    helpscript please_register
    catch {
	set loginname [exec whoami]
	puts "     Your short login name appears to be: $loginname\n"
    }
    helpscript please_register_2
    return ""
}

# solar::doc --
#
# Purpose:  Find the SOLAR documentation
#
# Usage:    doc [-whereis]
#
#           doc               show official documentation URL and
#                             location of documentation files on this system
#
# Notes:    This command now tells you the URL where SOLAR may be seen with
#           any available browser.  Previously, it would run Netscape.
# -

proc doco {args} {
    return [eval doc -old $args]
}

proc doc {args} {

    global env
    set dirname [file dirname $env(SOLAR_LIB)]/doc
    
    set whereis 0
    set chapter 0
    set appendix 0
    set preface 0
    set contents 0
    set oldsession 0
    set badargs [read_arglist $args \
        -chapter chapter \
	-appendix appendix \
	-old {set oldsession 1} \
        -whereis {set whereis 1} \
    ]

    return [docwhere $dirname]
}


proc docwhere {dirname} {
    return "Point your browser to http://solar.txbiomedgenetics.org/doc/index.html\n\nOn this computer documentation files are located at $dirname"
}


# solar::example --
#
# Purpose:  Copy the SOLAR example to the current working directory
#
# Usage:    example
#
# Notes:    The example may be used in conjunction with the SOLAR tutorial
#           in Chapter 3.  To read that, give the command "doc -chapter 3"
#
#           The example files are actually located in the doc/Example
#           subdirectory of the SOLAR installation.  To find the "doc"
#           subdirectory, give the command "doc -whereis"
# -

proc example {} {

    global env
    set docdir "[file dirname $env(SOLAR_LIB)]/doc/Example"
    puts "Copying example files to working directory"
    set files [glob $docdir/*]
    foreach fi $files {
	if {![file isdirectory $fi]} {
	    file copy -force $fi .
	}
    }
    newtcl
}

proc untagify {infile outfile} {
    set ifile [open $infile r]
    set ofile [open $outfile w]
    while {-1 != [gets $ifile line]} {
	if {[string length $line] == 0} {
	    puts $ofile $line
	} else {
	    while {-1 != [string first < $line]} {
		if {-1 == [string first > $line]} {
		    break
		}
		set beginb [string first < $line]
		set endb [string first > $line]
		if {0 == $beginb} {
		    set line [string range $line [expr $endb + 1] end]
		    if {0 == [string length $line]} {
			break
		    }
		} else {
		    set prefix [string range $line 0 [expr $beginb - 1]]
		    set postfix [string range $line [expr $endb + 1] end]
		    set line [catenate $prefix $postfix]
		}
	    }
	    if {[string length $line] != 0} {  ;# if wasn't only tags
		puts $ofile $line
	    }
	}
    }
    close $ifile
    close $ofile
}


proc dominance-notes {} {
    return [helpscript dominance-notes]
}

# solar::dominance-notes --
#
# Purpose:  Find dominance documentation
#
# Dominance analysis is documented in section 9.4 of the SOLAR
# manual.  Use the "doc" command or point your browser to
# the "doc" directory of your SOLAR directory, then select
# "Go to full SOLAR manual", then select Chapter 9.
#
# Dominance analysis is made possible by the "delta7" and "d7" columns
# in SOLAR phi2.gz and ibd matrices.  For polygenic models, the delta7
# matrix column is loaded, a d2r parameter is created and added to the
# "e2" constraint, then a delta7*d2r term is added to the omega.  The
# commands required to do this are described in Section 9.4
# -


proc discrete-notes {} {
    return [helpscript discrete-notes]
}

# solar::discrete-notes --
#
# Purpose:  Describe support for discrete traits
#
# Usage:    discrete-notes
#
# Discrete traits are detected automatically by SOLAR.  They must be
# coded as two integer values separated by exactly 1.  Typical codings
# are 0,1 or 1,2.  If you specify two values that are not separated by
# exactly 1, this will be detected as an error.  If you specify more than
# two values, your trait will not be determined to be discrete.  For
# this reason, DO NOT specify missing values with a third number. 
# Missing values should always be coded as blank (" ") or null with no
# number or character.  DO NOT use "0" to signify missing values.  See
# toward the bottom of this note for advice regarding analyzing traits
# with two values quantitatively.
#
# Discrete traits having more than 2 values are not supported by SOLAR.
#
# (This is also true for discrete phenotypic covariates: if discrete, they
#  should not have more than 2 values.  If you have such data, they
#  should be recoded into N-1 discrete binary covariates or recast into
#  "household groups."  See the documentation for the "house" command.)
#
# Models with discrete traits may be used with any command in SOLAR such as
# polygenic, twopoint, multipoint, maximize, etc.  Sometimes the
# information returned by SOLAR differs.  For example, while the
# "polygenic" command normally returns "proportion of variance due to all
# covariates" when used with a quantitative trait, it instead returns the
# "Kullback-Leibler R-squared" when used with a discrete trait.  (For
# technical reasons, the proportion of variance due to all covariates is
# not available for discrete trait models in SOLAR.)
#
# By examining the maximization output files you can determine unambiguously
# whether discrete or quantitative methods were used.  (An example of
# a maximization output file is "null0.out" saved in the maximization
# output directory after running "polygenic".)  In this file, immediately
# after the "Descriptive Statistics" and immediately before the "Model
# Parameter Starting Points and Boundaries" there will be one of two
# comments, either:
#
#                Using SOLAR Quantitative Trait Modeling
#
# or
#
#                  Using SOLAR Discrete Trait Modeling
#
# When a model with a discrete trait is maximized, special discrete trait
# algorithms are used. Unfortunately, these methods are much more prone
# to numerical problems than the usual "quantiative trait" methods.
# Numerical problems lead to faulty parameter estimates and convergence
# failures.
#
# The default descrete method is relatively robust and only infrequently
# has the problem where the heritability erroneously gets close to 1.0.
#
# Even if the polygenic heritability (h2r) goes to 1.0, you may still be
# able to run a "multipoint" linkage analysis to find important locii.
# The heritibilities will be wrong, and the LOD scores will be wrong,
# but the "peaks" may be at or near the correct locations.
#
# It is not recommended to use the optional second discrete method set by
# giving the command "option DiscreteMethod 2" prior to running
# polygenic.  Although it was intended to be more accurate, it more
# frequently fails with convergence errors or having the heritability go
# to 1.0, and at this time it is not recommended.
#
# Some people also try analyzing their discrete trait as quantitative.
# This can be done by giving the command "option EnableDiscrete 0".
# The likelihoods, LODS, and parameter estimates may be inaccurate, but the
# LOD peaks should be in the correct places.  Better convergence is
# sometimes obtained, however, than when using the discrete method.
#
# Beware that there is a fundamental error when analyzing a discrete trait
# as quantitative.  There are not truly two degrees of freedom for the
# mean and SD.  Therefore, convergence failure is still more common with these
# models than with true quantitative models.
#
# Also beware that if you had previously analyzed the trait as discrete,
# and then changed the EnableDiscrete option to 0 without exiting SOLAR
# or giving the "model new" command, you will still have parameter SD
# constrained to 1.0, which is probably NOT what you need to do.  SD is
# properly constrained to 1.0 only when you are analyzing a discrete trait
# as discrete (and, perhaps, in a few other esoteric cases).
#
# Because of all the pitfalls in using discrete traits, we try to find and
# use relevant quantitative traits whenever possible.
#-

# solar::file-matrix --
#
# Purpose:  Describe csv matrix file format requirements
#
# CSV matrix files were introduced in SOLAR version 7.5.0, and it is
# conditionally recommended that all users who are creating their own matrix
# files use this format, as it is more easily understood and created than the
# previous format.  Matrix writers should beware that there are many possible
# pitfalls in hand writing a matrix file, and matrix files should at minimum
# be checked with the "matrix debug" command after loading the first time.
#
# The original space column delimited matrix file is still created and used by
# SOLAR itself and documented in the manual, and if you merely wish to modify
# an existing matrix file, it may still be easiest to use the original format.
# The format of the phi2.gz kinship matrix file is described in Section 8.3
# of the manual, and the same rules would apply to any original format matrix
# file.
#
# The csv matrix file is an ordinary comma separated variable file with the
# first line being a header which names all the fields used, as is common
# with CSV files.  However SOLAR requires that all matrix files, including
# csv matrix files, be compressed using the gzip program.  Thus all matrix
# files will have the final filename extension ".gz".
#
# All csv matrix files must have the following three required fields:
# id1, id2, matrix1.  id1 and id2 are user assigned ID's, as used in pedigree
# and phenotypes files.  matrix1 is typically the primary matrix in the file,
# for example the primary matrix in the phi2.gz file is the "phi2" matrix.  A
# csv matrix file may also have a second matrix named matrix2.  If FAMID's are
# required to disambiguate ID's in your dataset, you must also include
# famid1 and famid2, the famid's corresponding to id1 and id2.  SOLAR will
# determine whether famid's are needed or not from the pedigree file.  If
# famid's are not needed, famid1 and famid2 will be ignored if they are
# present.  If famid's are needed and not present in the matrix file, the
# load matrix command will generate an error and the matrix will not be loaded.
#
# Note that is the "load matrix" command which actually assigns meaningful
# names to matrixes for use in the SOLAR omega.  For example, the phi2
# matrix is usually loaded with the following command:
#
#   load matrix phi2.gz phi2
#
# Alternatively, an analysis examining dominance would require a second
# matrix, delta7:
#
#   load matrix phi2.gz phi2 delta7
#
# In both cases the first matrix (called matrix1 in a CSV matrix file) will
# be associated with the name phi2 in the omega.  The second command will
# also associate the second matrix (matrix2) with the name delta7.  Neither
# command shown let you directly use the names matrix1 and matrix2 in the
# omega, unless those names were also specified in the load matrix command.
# The CSV naming scheme is intended to allow the "load matrix" command to be
# used exactly as it was before.
#
# Variable names other than id1, id2, matrix1, matrix2, famid1, and famid2
# in a CSV matrix file will be (in this version) ignored.
#
# There are other semantic requirements for matrix files, depending
# on the type of matrix involved.  One is that every individual in an analysis
# must be included in the matrix file regardless of whether they have pairwise
# relationships with others in the file.  At minimum every individual has a
# self relationship of value 1.0.  With version 7.5.0 it is required that you
# include these "diagonal" matrix values of 1.0 for every individual in your
# sample, and possibly everyone in your pedigree file.  Otherwise, diagonal
# values will default to -1 which could have bad consequences.  (Note: in
# version 7.5.0 it is not possible to check this with matrix debug because
# it only checks values in the input file, not defaulted values in the
# matrix itself.)
#
# It is not permitted to have individuals in the CSV matrix file who are not
# defined in the pedigree file.
#
# Historically matrix files were dependent on the pedindex.out files created
# when the pedigree was loaded.  This is because the very IBDID's used in
# original format matrix files might be assigned to different actual ID's
# if the pedigree is changed.  CSV format matrix files are less dependent,
# but it is still likely that changes to a pedigree file would require
# corresponding changes in the matrix file.  And sometimes this may be
# overlooked, causing disasterous results.  SOLAR has long prepended a
# pedigree checksum record to all the matrix files it creates which are checked
# against the pedigree file when the matrix is loaded.  Changes to the
# pedigree after the matrix file was created will cause an error to
# be raised when attempting to load that matrix file.
#
# The same checksum checking features can optionally be used in CSV matrix
# files.  Once the matrix has been created and compressed using gzip, the
# procedure "matcrc" can be run on them, for example:
#
# solar> matcrc phi2.csv.gz
#
# This will determine a checksum value from the currently loaded pedigree
# file, and prepend this to the matrix file in a record with id1=checksum
# and id2=checksum.  It also will gunzip the matrix file to perform this
# change, and gzip after the change has been made.
# 
# If the names assigned to matrixes (in the load matrix command) begin with
# "ibd" or "mibd" a special defaulting rule applies.  -1 values found in these
# matrixes mean that the actual value should be taken from the phi2 matrix
# (for the first matrix) or delta7 (for the second matrix).  Furthermore, if
# -1 is READ FROM THE FILE for a diagonal matrix entry, the default is at
# that time applied to every pair including that individual.  (On the other
# hand, if a -1 occurs on the diagonal only because the diagonal entry was
# missing from the file, defaulting would occur for the missing diagonal but
# not for every other pair including that individual, the default value for
# those other pairs would be zero.)  This feature had historical importance
# but is considered obsolescent now and is not recommended for use in new
# matrix files.
#- 

proc file-matrix {} {
    return [helpscript file-matrix]
}

# solar::file-phenotypes --
#
# Purpose:  Describe phenotypes data file requirements
#
# The phenotypes file may be in either PEDSYS or Comma Delimited format.
#
# The phenotypes file consists of one record for each individual.
# Each record must include an ego ID and one or more phenotypic
# values (which may be blank to signify missing data).
#
#    ego ID, phen 1, phen 2, ...
#
# (The phenotypes file may also contain other data, such as pedigree
# data.  You could use one file as both your phenotype and your
# pedigree file, though that is not necessarily recommended.  There
# are fewer possible problems with separate files.)
#
# Just as with the pedigree file, a field name FAMID is required when
# IDs are not unique across the entire data set.  (If your ego IDs
# are unique, it is probably better _not_ to include a family ID,
# as it just complicates things slightly.)
#
# If your data has probands and you wish to employ ascertainment
# correction, the phenotypes file must have a proband field.  In this
# field, blank ( ) or zero (0) signifies non-proband, and anything
# else signifies proband.  A decimal point is _not_ permitted after
# the zero.  The presence of a proband field automatically turns on
# ascertainment correction.
#
# The default field names are ID, FAMID, and PROBND.  You can set up
# SOLAR to use different field names by using the field command.
#
# The phenotype field names may be anything within certain rules.
# (no spaces, tabs, or slashes; also certain special characters such
# as *#,^/-+ can cause problems in the names of phenotypes used
# as covariates).  If you stick with alphabetic characters, numeric
# characters, and underscores you will be safe.
#
# The phenotype data fields must be numbers, either with or without
# decimal points.  Zero (0) is always considered a permissible value;
# blank ( ) or null (e.g. no value in between the commas ",," in a
# comma delimited file) must be used to signify missing values.
#
# Floating or fixed point numbers must always include a decimal
# point; numbers without a decimal point are assumed to be integers.
# Binary, discrete or categorical values should be indicated with
# consecutive integers (e.g. 0,1 or 1,2 or 2,3).  SOLAR checks all
# phenotype fields to see if they contain only two consecutive
# integers and judges them "binary" if they do.  Binary traits are
# automatically handled by the SOLAR discrete trait liability
# threshold modeling code; you don't need to do anything special.
# See Chapter 9 for discussion on what to do with "categorical"
# data that has more than two categories.
#
# Without special scripting, categorical phenotypes with more than two
# categories should not be used in SOLAR.  (SOLAR will not identify
# categorical phenotypes with more than two categories and instead
# treat them as quantitative phenotypes.)
#
# The 'load phenotypes' command creates a file named phenotypes.info
# in the working directory.  Once a phenotypes file has been loaded,
# it need not be loaded again in the same working directory, unless
# you change the file itself.
#
# SOLAR automatically removes pedigrees in which no non-proband has
# all required phenotypic data from the analysis.  You need not
# remove these pedigrees yourself.  You will get a full accounting of
# pedigrees and individuals included and excluded in the maximization
# output files (described below) , by running the 'maximize' command,
# or giving the 'verbosity max' command prior to other commands.
#-

proc file-phenotypes {} {
    return [helpscript file-phenotypes]
}

# solar::file-pedigree --
#
# Purpose:  Describe pedigree data file requirements
#
# The pedigree file consists of one record for each individual in the data
# set.  Each record must include the following fields:
#
#     ego ID, father ID, mother ID, sex
#
# In addition, a family ID is required when ego IDs are not unique across
# the entire data set.  If the data set contains genetically identical
# individuals, an MZ-twin ID must be present (as described below).  If an
# analysis of household effects is planned, a household ID can be included
# (also described below).
#
# The default field names are ID, FA, MO, SEX, FAMID, MZTWIN, and HHID.
# EGO, SIRE, and DAM are also accepted by default.  You can set up SOLAR to
# use different field names by using the field command (see 'help field').
# You do not necessarily need to change your names to match ours.
#
# A blank parental ID or a parental ID of 0 (zero) signifies a missing
# parent.  SOLAR requires that either both parents are unknown, i.e. the
# individual is a founder, or both parents are known.
#
# If the pedigree data consists of unrelated individuals with no parental
# data, then the father ID and mother ID fields are not required. If there
# are parents for whom pedigree file records do not exist, then records
# are created internally for those parents, who are assumed to be founders.
#
# Sex may be encoded as M, m, or 1 for males and F, f, or 2 for females.
# The missing value for sex is 0, U, u, or blank.
#
# The MZ-twin ID is used to designate genetically identical individuals,
# e.g. monozygotic twins or triplets.  Each member of a group of identical
# individuals should be assigned the same MZ-twin ID.  Twin IDs must be
# unique across the entire data set.  If there are no genetically identical
# individuals in the data set, this field need not be present in the
# pedigree file.
#
# The household ID, if present, will be used to generate a matrix file
# (house.gz) that can be used later to include a variance component for
# household effects.  Household IDs must be unique across the entire data
# set.
#
# The family ID field is required only when ego IDs are not unique across
# the entire data set.  For example, if a data set consists of nuclear
# families, and the same ego ID may appear in more than one family, then
# the family ID must be included.  Or if, for example, IDs are sequential
# integers unique only within pedigrees, then the pedigree ID must be
# included.
#
# At the time the pedigree file is loaded, SOLAR indexes the data set.
# This indexing is internal and should not be confused with any external
# indexing the user may have imposed upon the data set.  This indexing
# information is stored in a file named 'pedindex.out' in the directory
# where SOLAR is running when the pedigree data is loaded.  Be careful
# about deleting files unless you are sure they are not needed by SOLAR!  
#
# Once a pedigree file has been loaded, it is not necessary to load
# it again in subsequent SOLAR runs from the same working directory.
# -

proc file-pedigree {} {
    return [helpscript file-pedigree]
}

# solar::file-marker --
#
# Purpose:  Describe marker data file requirements
#
# The marker file contains genotype data for one or more marker loci.
# The file consists of one record for each individual who has been typed
# for one or more of these markers.  Each record must contain the following
# fields:
#
#     ego ID, genotype1, genotype2, ...
#
# In addition, a family ID field must be included when ego IDs are not
# unique across the entire data set.  If, however, each ego ID is unique
# to an individual and an individual may appear multiple times in the
# data set, then the family ID should not be included.  The same genotypic
# data is then associated with every occurrence of an individual.
#
# The default field names are ID and FAMID.  EGO is also accepted by
# default.  You can set up SOLAR to use different field names by using
# the field command (see 'help field').  You do not necessarily need to
# change your names to match ours.
#
# Fields with names other than ID and FAMID are assumed to contain marker
# data, with the exception of the following names: FA, MO, SEX, MZTWIN,
# HHID, AGE, PEDNO, and GEN. Fields having one of these names are ignored.
#
# The scheme used to encode genotypes may vary from field to field.
# SOLAR recognizes many standard coding schemes, but the safest way to
# code genotypes is with the forward slash to separate the alleles.
#
# Ex: AB
#     E1 E3
#     123/456
#
# A blank genotype field denotes missing data, as do the genotypes 0/0
# and -/-.  SOLAR requires that either both alleles are typed or both
# alleles are missing, except for male genotypes at X-linked marker loci.
# In that case, either a single allele is specified (the other allele is
# blank, 0, or -), or the genotype is coded as a "homozygote".
#
# Ex: 237/243   valid female X-linked marker genotype
#        /251   valid male X-linked marker genotype
#       251/0   valid male X-linked marker genotype
#       -/251   valid male X-linked marker genotype
#     251/251   valid male X-linked marker genotype
#
# The marker loci in the marker file must all be autosomal or all be
# X-linked.  By default, SOLAR assumes that the markers are autosomal.
# If the markers are X-linked, then either the XLinked option must be
# set with the ibdoption command prior to loading the marker file, or
# the -xlinked option must be given in the load marker command.
#
# Once a marker file has been loaded, it is not necessary to load it
# again in subsequent SOLAR runs from the same working directory.
# -


proc file-marker {} {
    return [helpscript file-marker]
}

# solar::file-freq --
#
# Purpose:  Describe frequency data file requirements
#
# The freq file contains allele frequency data for a set of marker loci,
# one line per marker.  Each line consists of the following space-delimited
# fields:
#
#     marker name, all_1 name, all_1 freq, all_2 name, all_2 freq, ...
#
# The allele frequencies for a marker must sum to 1 (a small roundoff error
# is tolerated.)
#
# Allele frequency information is used when IBDs are computed for a marker
# that is not completely typed, i.e. there are individuals for whom genotype
# data is not available.
#
# Example:
#
# D20S101 123 0.2457 127 0.1648 133 0.5895
# IGF1 A 0.4 B 0.3 C 0.1 F 0.2
# ApoE E1 .125 E2 .25 E3 .625
#
# Once a freq file has been loaded, it is not necessary to load it again
# in subsequent SOLAR runs from the same working directory.
# -

proc file-freq {} {
    return [helpscript file-freq]
}


# solar::file-map --
#
# Purpose:  Describe map data file requirements
#
# The map file contains chromosomal locations for a set of marker loci
# on a single chromosome.  Typically, marker locations are given in cM
# and a mapping function is used to convert inter-marker distances to
# recombination fractions.  Currently, the Kosambi and Haldane mapping
# functions are allowed.  Marker locations can also be specified in
# basepairs.  While cM locations can be floating point numbers, basepair
# locations must be integers; non-integer locations are truncated to
# integers.  When basepair locations are used, the mapping function is
# called "basepair" rather than Kosambi or Haldane, but in fact there
# is no mapping provided from basepairs to recombination fractions and
# such maps cannot be used to compute multipoint IBDs.  The first line
# of the map file contains the chromosome number, and (optionally) the
# name of the mapping function.  If no mapping function is specified,
# the mapping is assumed to be Kosambi.  The chromosome number can be
# any character string not containing a blank or a forward slash (/),
# although the use of integers is recommended.  For example, the strings
# '01' and '10q' are allowed.  Each line after the first line consists
# of the following space-delimited fields:
#
#     marker name, marker location
#
# Examples:
#
# 20
# D20S101         0.0
# D20S202        34.2
# D20S303        57.5
#
# TCF basepair
# 2448b   19828941
# 380659  19829489
#
# -

proc file-map {} {
    return [helpscript file-map]
}


# Check phenotypes file for some errors:
#   File not found
#   Missing FAMID when required

proc check_phenotypes {args} {

# If pedigree file not loaded, the safer assumption is that it does
# have FAMID field.  If it doesn't, ID's must be unique anyway

    set pedfilefam 1
    catch {
	if {[file exists pedindex.out]} {
	    set pedfile [tablefile open pedindex.out]
	    set pedfilefam [tablefile $pedfile test_name famid]
	    tablefile $pedfile close
	}
    }

# See if each phenotypes file exists, then see if it has famid
# If any phenotypes files are missing famid, can't use famid

    set phenfilefam 1
    foreach phenfilename $args {

	if {![file exists $phenfilename]} {
	    error "phenotypes: File $phenfilename not found"
	}
	set phenfile [solarfile open $phenfilename]
	set testfilefam [solarfile $phenfile test_name famid]
	if {!$testfilefam} {
	    set phenfilefam 0
	}
	solarfile $phenfile close
    }

    ifdebug puts "Using famid: pedfile: $pedfilefam    phenfile: $phenfilefam"

    set outfilefam $phenfilefam
    if {$phenfilefam != $pedfilefam} {
	set outfilefam 0
	foreach phenfilename $args {
	    set ids {}
	    set phenfile [solarfile open $phenfilename]
	    solarfile $phenfile start_setup
	    solarfile $phenfile setup id
	    while {{} != [set record [solarfile $phenfile get]]} {
		lappend ids [lindex $record 0]
	    }
	    set idss [lsort $ids]
	    set index 0
	    while {{}  != [set next [lindex $idss [expr 1 + $index]]]} {
		set this [lindex $idss $index]
		if {![string compare $this $next]} {
		    solarfile $phenfile close
		    error \
	"Duplicate ID's in $phenfilename; have you mislabeled FAMID field?"
		}
		incr index
	    }
	    solarfile $phenfile close
	}
    }
    return $outfilefam
}

# solar::ibddir --
#
# Purpose:  Set directory in which IBD matrix files are stored
#            (twopoint only; use mibddir to set up multipoint)
#
# Usage:    ibddir <dirname>     ; set director for IBD files
#           ibddir               ; show current ibddir
#           ibddir -session      ; show ibddir entered in this session
#
# Notes:    The ibddir selected is saved in file ibddir.info for
#           future SOLAR sessions.  Once a midddir is selected, it
#           need not be selected again within the same working directory,
#           EXCEPT for the purposes of writing out ibd files.  To
#           prevent accidentally overwriting pre-existing ibd files,
#           it is necessary to explicitly enter the ibddir
#           command before using the ibd command or other commands
#           which write files into the ibddir.
# -

# solar::mibddir --
#
# Purpose:  Set directory in which MIBD matrix files are stored
#            (multipoint only; use ibddir to set up twopoint)
#
# Usage:    mibddir <dirname>     ; set directory for MIBD files
#           mibddir               ; show current mibddir
#           mibddir -session      ; show mibddir entered in this session
#
# Notes:    The mibddir selected is saved in file mibddir.info for
#           future SOLAR sessions.  Once a midddir is selected, it
#           need not be selected again within the same working directory,
#           EXCEPT for the purposes of writing out mibd files.  To
#           prevent accidentally overwriting pre-existing mibd files,
#           it is necessary to explicitly enter the mibddir
#           command before using the mibd command or other commands
#           which write files into the mibddir.
# -

# solar::snpdir -- private
#
# Purpose:  Set directory in which snp related files are to be put or found
#           (See "snp".)  NOTE: This command is not supported for any
#           useful purpose.
#
# Usage:    snpdir <dirname>     ; set directory for MIBD files
#           snpdir               ; show current mibddir
#           snpdir -session      ; show mibddir entered in this session
#
# Notes:    The snpdir selected is saved in file snpdir.info for
#           future SOLAR sessions.  Once a midddir is selected, it
#           need not be selected again within the same working directory,
#           EXCEPT for the purposes of writing out mibd files.  To
#           prevent accidentally overwriting pre-existing mibd files,
#           it is necessary to explicitly enter the snpdir
#           command before using the mibd command or other commands
#           which write files into the snpdir.
# -

proc ibddir {args} {
    return [eval ibdmibddir ibddir $args]
}

proc mibddir {args} {
    return [eval ibdmibddir mibddir $args]
}

proc snpdir {args} {
    return [eval ibdmibddir snpdir $args]
}


proc ibdmibddir {dirt args} {
    set save 1
    set session 0
    set args [read_arglist $args -nosave {set save 0} \
		  -session {set session 1}]

    global Solar_$dirt
    if {$args == {}} {
	if {![if_global_exists Solar_$dirt]} {
	    error "No $dirt specification has been given"
	}
	if {$session && ![if_global_exists Solar_Session_$dirt]} {
	    error "No $dirt specification has been given in this session"
	}
	eval return \$Solar_$dirt
    }

# IMPORTANT!!! BEYOND THIS POINT WE DELETE $dirt.info for all errors!!!
# (unless -nosave option)

    purge_global Solar_$dirt
    if {$save} {
	global Solar_Session_$dirt
	set Solar_Session_$dirt 1
    }
    set dir [lindex $args 0]

    global Solar_Forbidden_Ibd_Root
    if {[if_global_exists Solar_Forbidden_Ibd_Root]} {
	set fsir $Solar_Forbidden_Ibd_Root
	set fsirend [expr [string length $fsir] - 1]
	if {-1 < $fsirend} {
	    if {![string compare [string range $dir 0 $fsirend] $fsir]} {
		set froot [string range $fsir 0 [expr $fsirend - 1]]
		if {$save} {catch {file delete $dirt.info}}
		error "$dirt must not be a subdirectory of $froot"
	    }
	}
    }
    if {![file isdirectory $dir]} {
	if {$save} {
	    catch {file delete $dirt.info}
	    error "No such directory: $dir"
	}
    }
    set absolute_dir [make_absolute_pathname $dir]
    set Solar_$dirt $absolute_dir

# Now, save mibddir to a state file, but only if different
# from previous state file

    if {$save} {
	set changed 1
	if {0==[catch {set testfile [open $dirt.info]}]} {
	    if {-1 != [gets $testfile line]} {
		if {0 == [string compare $line $absolute_dir]} {
		    set changed 0
		}
	    }
	    close $testfile
	}
	if {$changed} {
	    exec echo $absolute_dir > $dirt.info
	}
    }
    return ""
}

# Invoked by solar.cc to initialize ibddir and mibddir

proc Start_Dirs {} {

    foreach dir {ibddir mibddir snpdir} {
	if {[file exists $dir.info]} {
	    set dfile [open $dir.info r]
	    if {[gets $dfile line]} {
		if {[catch {$dir -nosave $line} errmes]} {
#		    puts "\nError initializing $dir: $errmes"
		}
	    }
	    close $dfile
	}
    }
    return ""
}


# solar::chromosome --
#
# Purpose:  Select chromosome(s) for multipoint scan
#
# Usage:   chromosome [<number>|<name>|<low>-<high>|all|*]+    ;select
#          chromosome                  ; show currently selected chromosomes
#          chromosome show             ; show all available chromosomes
#          chromosome showm            ; show mibd's in pass (see note 2)
#
# Examples: 
#          chromosome 10
#          chromosome 10-13 15-17 20
#          chromosome 11 11p
#          chromosome all              ; select all available chromosomes
#          chromosome *                ; select all available chromosomes
#
# Notes:   Use in conjunction with mibddir, interval, multipoint commands.
#
#  (2)  The showm option lists the mibds's that will be selected by
#       the current "chromosome" and "interval" commands.
#
#  (3)  Alphanumeric chromosomes may not be in <low>-<high> ranges, but may
#       be selected individually (for example, 11p), or with "all" or *.
#
#  (4)  The chromosome specification is not saved from one solar session
#       to the next unless put in a .solar file.
#
#  (5)  For convenience, you may specify a chromosome or range of
#       chromosomes whose mibds are not actually present, and
#       the gap is ignored silently, as long as there are some mibds
#       available for other specified chromosomes.  The chromosome
#       command acts as a filter applied to the mibd data actually
#       available.
# -

proc chromosome {args} {
    if {$args == {}} {
	if {0 == [llength [info globals Solar_Chromosomes]]} {
	    error "No chromosome specification has been given"
	}
	global Solar_Chromosomes
	return $Solar_Chromosomes
    }
    if {$args == "show"} {
	return [get_all_chromos [mibddir]]
    }
    if {$args == "showm"} {
	return [global_mibd_list]
    }
    purge_global Solar_Chromosomes
    expand_ranges $args    ;# this is only to test for errors now
    global Solar_Chromosomes
    set Solar_Chromosomes $args
    return {}
}

# solar::finemap
#
# Purpose:  Set fine mapping threshold for multipoint
#
# Usage:    finemap <LOD> [<LOD> ...]
#           finemap default
#           finemap off
#           finemap                  {displays current finemap setting}
#
# Example:  finemap 0.588
#
# Notes:    After each multipoint pass when the interval is greater than 1
#           SOLAR will examine all points in the regions around points
#           higher than some threshold.  This threshold is set with the
#           finemap command.
#
#           The default is 0.588.
#
#           Finemapping can also be turned off.  The finemap setting is
#           unimportant when the interval is 1.  (Note: versions of SOLAR
#           prior to 1.1.0 did finemapping only around the single highest
#           peak by default.)
#           
# -

proc finemap {args} {
    if {$args == {}} {
	if {0 == [if_global_exists Solar_Fine_Map]} {
	    return 0.588
	} else {
	    global Solar_Fine_Map
	    return $Solar_Fine_Map
	}
    }
    global Solar_Fine_Map
    if {0==[string compare $args default]} {
	if {[if_global_exists Solar_Fine_Map]} {
	    unset Solar_Fine_Map
	}
    } elseif {0==[string compare $args off]} {
	set Solar_Fine_Map off
    } else {
	foreach arg $args {
	    ensure_float $arg
	}
	set Solar_Fine_Map $args
    }
    return ""
}


# solar::interval --
#
# Purpose:  Set cM interval and range for multipoint scanning each chromosome
#
# Usage:      interval <count> <range> ; set increment count and range
#             interval <count>         ; default range is 0-* (* means last)
#             interval                 ; displays current setting
#
# Examples:   interval 5               ; Check every 5 cM
#             interval 1 101-109       ; Check every 1 cM between 101 and 109
#             interval 10 200-*        ; Check every 10 cM after <200 cM>
#             interval 0 100           ; Check at position <100 cM>
#             interval -5 *-100        ; Check every 5 cM from last to 100
# -

proc interval {args} {
    if {$args == {}} {
	if {0 == [llength [info globals Solar_Interval]] || \
		0 == [llength [info globals Solar_Interval_Range]]} {
	    error "No interval specification has been given"
	}
	global Solar_Interval
	global Solar_Interval_Range

	return [format "%d %s-%s" \
	        $Solar_Interval \
		[lindex $Solar_Interval_Range 0] \
		[lindex $Solar_Interval_Range 1]]
    }

# Clear out old values
	
    purge_global Solar_Interval
    purge_global Solar_Interval_Range

# Get increment_count

    set argc [llength $args]
    if {1 != $argc && 2 != $argc} {
	error "Invalid interval command"
    }
    set increment_count [lindex $args 0]
    ensure_integer $increment_count

# Get first and last

    if {$argc == 1} {
	if {$increment_count >= 0} {
	    set first 0
	    set last *
	} else {
	    set first *
	    set last 0
	}
    } else {
	set range [lindex $args 1]
	set hyphen [string first "-" $range]
	if {-1 == $hyphen} {
	    ensure_integer $range
	    set first $range
	    set last $range
	    if {$increment_count != 0} {
		error "Non-zero count with single marker"
	    }
	} else {
	    set first_and_last [split $range "-"]
	    if {2 != [llength $first_and_last]} {
		error "Range should be first-last"
	    }
	    set first [lindex $first_and_last 0]
	    set last [lindex $first_and_last 1]
	    if {0 == [string compare $first *]} {
		ensure_integer $last
		if {$increment_count >= 0} {
		    error "Interval direction inconsistent"
		}
	    } elseif {0 == [string compare $last *]} {
		ensure_integer $first
		if {$increment_count <= 0} {
		    error "Interval direction inconsistent"
		}
	    } else {
		ensure_integer $first
		ensure_integer $last
		if {$increment_count > 0} {
		    if {$first >= $last} {
			error "Interval direction inconsistent"
		    }
		} elseif {$increment_count < 0} {
		    if {$first <= $last} {
			error "Interval direction inconsistent"
		    }
		} else {
		    if {$first != $last} {
			error "Interval direction inconsistent"
		    }
		}
	    }
	}
    }
    global Solar_Interval
    global Solar_Interval_Range
    set Solar_Interval $increment_count
    set Solar_Interval_Range [list $first $last]
    return {}
}

proc global_mibd_list {} {
    if {[catch {mibddir}]} {
#	puts "Info level is [info level]"
	if {2<[info level]} {
	    error "First use mibddir command to set mibd directory"
	}
	puts -nonewline "Enter mibddir: "
        flush stdout
	gets stdin m
	mibddir $m
    }
    if {0 == [llength [info globals Solar_Chromosomes]]} {
	if {2<[info level]} {
	    error "Use chromosome command to select chromosomes"
	}
	puts -nonewline "Enter chromosomes: "
        flush stdout
	gets stdin c
	eval chromosome $c
    }
    if {0 == [llength [info globals Solar_Interval]]} {
	if {2<[info level]} {
	    error "Use interval command to set interval(s)"
	}
	puts -nonewline "Specify interval (also range if desired): "
        flush stdout
	gets stdin i
	eval interval $i
    }
    if {0 == [llength [info globals Solar_Interval_Range]]} {
	error "Use interval command to set interval(s)"
    }
    global Solar_Chromosomes
    global Solar_Interval
    global Solar_Interval_Range
    global Solar_Mibd_List
    set chromolist [expand_ranges $Solar_Chromosomes]
    set begin_range [lindex $Solar_Interval_Range 0]
    set end_range [lindex $Solar_Interval_Range 1]

    return [mibd_list [mibddir] $chromolist $Solar_Interval \
	    $begin_range $end_range]
}

proc get_all_chromos {mdir} {
    set chromolist {}
    set all_mibds [glob -nocomplain $mdir/mibd.*.*.gz]
    set plength [string length "$mdir/mibd."]
    foreach mibd $all_mibds {
	set tail [string range $mibd $plength end]
	set dotpos [string first . $tail]
	if {!$dotpos} {
	    error "Invalid mibd name $mibd"
	}
	set c [string range $tail 0 [expr $dotpos - 1]]
	set chromolist [setappend chromolist $c]
    }
    return $chromolist
}

proc mibd_list {mibddir chromolist increment begin_range end_range} {

    set mlist {}
    if {[catch {set mdir [glob $mibddir]}]} {
	error "mibddir $mibddir is empty or unavailable"
    }

    if {-1 != [lsearch $chromolist all] || \
	    -1 != [lsearch -exact $chromolist *]} {
	set chromolist [get_all_chromos $mdir]
    }

    foreach chromo $chromolist {
	set wildcard [format "%s/mibd.%s.*.gz" $mdir $chromo]
	set full_vector [glob -nocomplain $wildcard]
	
	set flength [llength $full_vector]
	if {0 == $flength} continue
	
# Find/Set begin and end numbers

	set marker_list {}
	set file_format [format "%s/mibd.%s.%s.gz" $mdir $chromo "%d"]
	foreach file $full_vector {
	    if {0 < [scan $file $file_format marker_number]} {
		lappend marker_list $marker_number
	    } else {
		error "Invalid loc name in mibddir filename [file tail $file]"
	    }
	}
	set sorted_list [lsort -integer $marker_list]
	set highest_marker_found [lindex $sorted_list [expr $flength - 1]]

	if {0 == [string compare $end_range *]} {
	    set end_marker $highest_marker_found
	} else {
	    set end_marker $end_range
	}

	if {0 == [string compare $begin_range *]} {
	    set begin_marker $highest_marker_found
	} else {
	    set begin_marker $begin_range
	}

	if {$increment >= 0} {
	    set test {$marker <= $end_marker}
	} else {
	    set test {$marker >= $end_marker}
	}

	if {$increment == 0} {set increment 1}

	for {set marker $begin_marker } $test {incr marker $increment} {
	    set testname [format "%s/mibd.%s.%d.gz" $mdir $chromo \
	                  $marker]
	    if {[file exists $testname]} {
		lappend mlist $testname
	    }
	}
    }
    return $mlist
}

# expand ranges permits alphanumeric singletons (e.g. 1,2p,2q,3-23)
proc expand_ranges {ranges} {
    set final_list {}
    foreach range $ranges {
	set hyphen [string first "-" $range]
	if {-1 == $hyphen} {
#	    ensure_integer $range
	    lappend final_list $range
	} else {
	    set first_and_last [split $range "-"]
	    if {2 != [llength $first_and_last]} {
		error "Invalid range specification"
	    }
	    set first [lindex $first_and_last 0]
	    set last [lindex $first_and_last 1]
	    ensure_integer $first
	    ensure_integer $last
	    if {$first > $last} {error "Invalid range specification"}
	    for {set i $first} {$i <= $last} {incr i} {
		lappend final_list $i
	    }
	}
    }
    return $final_list
}

# solar::deputy --
#
# Purpose:  Make limited user key (for deputy registrars)
#
# Usage::   deputy register <deputy-key>
#           deputy make <access-code> <username>
#
# Notes:
#
# 1) Deputy registrar must obtain deputy-key and access-code from
#    solar@txbiomedgenetics.org.  Key is granted for critical collaborators
#    only for use in cluster systems where normal registration process
#    is unwieldy.
#
# 2) Deputy registrar uses "deputy register" command to register as
#    deputy.  This creates a file named .solar_deputy in home directory.
#    (Note: It does not move the .solar_deputy file to SOLAR_DEPUTY_HOME
#    if that is different from the deputy's home directory.)
#
# 3) The .solar_deputy file must be copied to a user to a deputy
#    directory on all systems.  This can be done in one of two ways.
#    The default way is to access the .solar_deputy file in the home
#    directory of the deputy, which must be found in a pathname
#    with the deputy's username replacing the current username.  For
#    example if deputy registrar jsmith has registered the name pmiller,
#    and the home directory for pmiller is:
#
#        /home/pmiller
#
#    Then the .solar_deputy file must be found in directory named:
#
#        /home/jsmith
#
#    If this default method cannot be used, there is an alternate
#    method involving creating a shell variable SOLAR_DEPUTY_HOME
#    giving the path to the .solar_deputy file.  For example, the
#    following line could be added to the "solar" startup script:
#
#        export SOLAR_DEPUTY_HOME=/home/admin/jsmith
#    
# 4) The deputy registrar can now make a limited range key for each
#    user using the "deputy make" command.  The user uses the
#    normal "register" command to install the key into a file named
#    .solar_reg in the user's home directory.  The .solar_reg file
#    AND the .solar_deputy file (located as described in note 3)
#    must be found on each system where SOLAR is to be run because
#    both are used in the validation process for keys created by
#    deputy registrars.
#
# 5) The "deputy make" command adds the usernames registered to a file
#    named "solar_registrations" in your home directory.  The contents
#    of this file should be sent to solar@txbiomedgenetics.org on at least
#    a biannual basis.
#
# 6) Username must be 2 characters or longer.
# -

proc deputy {args} {
    if {"register" == [lindex $args 0]} {
	if {[llength $args] < 2} {
	    error "deputy: must specify <deputy-key>"
	}
	if {[file exists ~/.solar_deputy]} {
	    puts ""
	    puts "Note: Use the 'deputy make' command to make user keys."
	    puts "You already have a .solar_deputy file in your home directory"
	    puts "Changing the file will invalidate existing user keys !!!"
	    error "Error! You must delete or rename the old .solar_deputy first.\n"
	}
	exec echo [lindex $args 1] >~/.solar_deputy
	return ""
    }
    if {"make" == [lindex $args 0]} {
	if {[llength $args] < 3} {
	    error "deputy: must specify access code and username"
	}
	set access_code [lindex $args 1]
	set foruser [lindex $args 2]
	exec echo $foruser >>~/solar_registrations
	return [key deputy $access_code $foruser]
    }
    error "invalid arguments to deputy command"
}


# solar::outdir --
#
# Purpose:  Set maximization output directory (overriding default)
#
# Usage:    outdir <dirname>
#           outdir                   ; shows current outdir
#           outdir -default          ; restore default: (trait name)
#
# Notes:   By default, solar models and related output are written to
#          the maximization output directory.  By default, that directory
#          is named after the trait.*  For bivariate models, the trait
#          names are separated by a period (".").
#
#          The default output directory can be overridden by this command.
#          Once set, it stays overridden until the "outdir -default" command
#          is given, or a new SOLAR session is started.
#
#          (*The directory will be named after the trait as entered in the
#           trait command, rather than as it exists in the phenotypes file.
#           For example, it will be named 'foo' if the command 'trait foo'
#           has been given, even if the variable is actually named FOO.)
#
#          To prepend the name of the maximization output directory to
#          any filename, use the "full_filename" command.
# -

proc outdir {args} {
    if {{} == $args} {
	if {0 == [llength [info globals Solar_Out_Dir]]} {
	    if {![catch {trait}]} {
		error \
	"Maximization output directory is defaulting to trait name: [full_filename ""]"
	    } else {
		error \
   "Maximization output directory is defaulting to trait name (none selected)."
	    }
	}
	global Solar_Out_Dir
	return "outdir $Solar_Out_Dir"
    } elseif {"-default" == $args} {
	purge_global Solar_Out_Dir
	return ""
    }
	
    purge_global Solar_Out_Dir
    set dir [lindex $args 0]
    if {[file exists $dir] && ![file isdirectory $dir]} {
	error "File $dir exists but is not directory"
    }
    global Solar_Out_Dir
    set Solar_Out_Dir $dir
    return ""
}


# solar::allsnp
#
# Purpose:  Include all snps as covariates in current model
#
# Usage:    allsnp
#
# Notes:    allsnp includes all the phenotypes prefixed with snp_ or
#           hap_ as covariates in the current model.  This is often the
#           first step in a qtn analysis.  Afterwards, you can remove
#           some snps using the "covariate delete" command.
#
#           It is OK if you have already selected other covariates,
#           including some of the snps.  Every covariate is only added
#           once no matter what.
#
#           allsnp looks at all currently loaded phenotype files.
# -

proc allsnp {} {

    set phens [phenotypes]
    foreach phen $phens {

	if {[string_imatch snp_ [string range $phen 0 3]] || \
		[string_imatch hap_ [string range $phen 0 3]]} {

	    if  {![string_imatch ":" [string range $phen end end]]} {
		covariate $phen
	    }
	}
    }
}


# solar::automodel --
#
# Purpose:  Default model setup
#
# Usage:    automodel <phenotypes> <trait>
#              phenotypes is the name of phenotype file
#              trait is the name of trait
#                  (all other variables will be used as covariates)
#
# Notes:   1.  Automodel will create a new model, with all non-trait and
#              non-pedigree variables as covariates (see note 2).
#
#          2.  The pedigree-related fields listed by the 'field' command
#              will not be used as covariates (except for SEX, which will be).
#              Certain other standard PEDSYS names are in the default exclude
#              list.  You can add additional items to the exclude list with
#              the exclude command.  See 'exclude' and 'allcovar' help.
#
#          3.  Boundaries and starting points are set automatically by the
#              maximize command.
#
#          4.  You can pick and choose from the commands that automodel uses
#              if you want to do things differently. Here is the body of 
#              automodel:
#
#                  model new                   ;# Start a new model
#                  phenotypes load filename    ;# load phenotypes file
#                  trait traitname             ;# assign trait variable
#                  allcovar                    ;# assign covariates
#                  polymod                     ;# set polygenic model type
# -

proc automodel {phenotypesname traitname} {
    model new
    phenotypes load $phenotypesname
    trait $traitname
    allcovar
    polymod
    return {}
}

proc get_chromosome {mibdfilename} {
    set mibdfilename [file tail $mibdfilename]
    set strings [split $mibdfilename .]
    if {4>[llength $strings]} {
	error "Invalid mibdfilename $mibdfilename"
    }
    return [lindex $strings 1]
}


proc get_locus {mibdfilename} {
    set mibdfilename [file tail $mibdfilename]
    set strings [split $mibdfilename .]
    set length [llength $strings]
    if {4>$length || 5<$length} {
	error "Invalid mibdfilename $mibdfilename"
    }
    set locus [lindex $strings 2]
    if {5 == $length} {
	set locus $locus.[lindex $strings 3]
    }
    return $locus
}


# solar::multipoint --
#
# Purpose:  Perform a multipoint analysis.
#             Scan loci on selected chromosomes at selected interval
#             (use chromosome, interval, and finemap commands beforehand)
#
# Usage:   multipoint [<LOD1> [<LOD2> [<LOD3> ...]]] [-overwrite] [-restart]
#                     [-renew mod] [-nullbase] [-plot] [-score]
#                     [-cparm <plist>] [-rhoq <fixed value>] [-saveall]
#                     [-ctparm <plist>] [-se]
#
#          Zero or more criterion LOD scores may be specified.  If none are
#          specified, multipoint will make one full scan and then stop.  If
#          one LOD score is specified, multipoint will continue scanning
#          until the highest LOD found in the last scan is no longer
#          greater than or equal to the LOD score specified.  If more than
#          one LOD score is specified, each LOD will apply after one scan
#          has been completed.  Then the last LOD specified will remain in
#          effect.
#
#          -overwrite  (or -ov) Overwrite existing multipoint output files.
#
#          -restart    (or -r) Restart previous multipoint run
#
#          -nose       Don't bother computing standard errors in best
#                      models (S.E.'s are not normally computing while
#                      scanning anyway).  This should not be combined with
#                      -slod option.
#
#          -plot       plot each point while scanning (uses plot -quick)
#                        Shows the current chromosome in the current pass,
#                        ending with the last chromosome in the last pass.
#                        To view previous passes, or for best quality plot,
#                        use the plot command.  The plot command may be run
#                        simultaneously in other SOLAR sessions plotting the
#                        same data from the multipoint*.out files.  For more
#                        information, see help for the plot command.
#
#          -score        Use Score based LOD (S-LOD) defined as:
#                        SLOD (score(i)^2 * SE(i))/(2 ln (10)) (where i is 
#                        the index of the new parameter).
#
#          -cparm <plist>  Custom parameters.  (See also -ctparm.)  This is
#                          discussed in Section 9.5 of the manual.  Scanning
#                          will consist of replacing one matrix with another
#                          matrix, everything else is unchanged.  The
#                          starting model MUST be a "prototype" linkage model
#                          will all the desired parameters, omega, and
#                          constraints.  Starting points and boundaries for
#                          all variance parameters must be explicitly
#                          specified.  Following the -cparm tag, there must be
#                          a list of parameters in curly braces that you want
#                          printed out for each model.  The list can be empty
#                          as indicated by an empty pair of curly braces {}.
#                          The matrix to be replaced must have name mibd or
#                          mibd1, mibd2, etc.  The highest such mibd will be
#                          replaced.  If the loaded matrix has two columns,
#                          each succeeding matrix will also be loaded with two
#                          columns.  There must be a model named null0 in
#                          the maximization output directory for LOD
#                          computation.  See section 9.5 for an example of
#                          custom parameterization.  Note: the user's
#                          starting model is saved in the output directory
#                          as multipoint.template.mod.  Any or all parameters
#                          in the <plist> may also be multiple-term
#                          expressions.  See second example below.
#
#                          After revision in version 4.1.5, -cparm now
#                          reloads the prototype model at the beginning of
#                          each chromosome or during finemapping if there is
#                          a gap greater than 11cm.  This provides much more
#                          stable operation of -cparm and fixes the problems
#                          that led most people to use -ctparm.  However,
#                          -ctparm may be preferable in some cases where
#                          there are convergence errors.  Or vice versa.
#                          Another strategy is to set the interval to 1.
#                        
#          -ctparm <plist> Custom parameters, as -cparm, but rebuilding each
#                          model from the "prototype" linkage model.  This
#                          might be slower, but it has the advantage of
#                          greater reliability.  If any model ends up with
#                          parameter(s) on boundaries, it has no ill effect
#                          on the remaining models.
#
#          -se           Calculate standard errors in all linkage models.
#                        Otherwise, they are always NOT calculated.  This
#                        is mainly useful in conjunction with -cparm
#                        and -ctparm.  See second example below.
#                        
#
#          -link <proc>  Use specified (by name) procedure instead of
#                        linkmod (default) to move to the next locus.
#                        The procedure requires 1 argument which is the
#                        full relative or absolute pathname to the mibd file.
#                        For now, it should ignore additional arguments
#                        (use a trailing "args" argument to do this).
#
#          -nullbase     Reload null model as base for each linkage model.  
#                        The default is to start from the previous linkage
#                        model if on the same chromosome.
#
#          -epistasis N   Use current loaded model as the base for a one-pass
#                         epistasis scan.  N is the index of the mibdN to
#                         be included in epistatic interactions (e.g. 1 for
#                         mibd1).  An additional parameter H2qE1 will be
#                         added for the interaction term (for mibdN and
#                         mibd<scan>).  Output files will be named
#                         multipointe.out and multipointe1.out.  Only one
#                         epistasis pass is currently supported; if
#                         oligogenic scanning is desired that should be
#                         done first before running an epistasis scan.  At the
#                         one QTL where mibdN and mibd<scan> are the same,
#                         h2q<scan> is constrained to zero (it is not and
#                         should not be constrained to zero elsewhere).
#
#          -rhoq <value>  Constrain rhoq parameters to <value>.
#
#          -saveall       Save the multipoint linkage models for every locus
#                         tested, not just the best ones.  The filenames look
#                         like this: multi.pass1.2.3.mod for pass 1, chromosome
#                         2, locus 3.  The maximization output files are saved
#                         also following the same naming convention but with
#                         a .out suffix.  Warning!  This can fill up a lot of
#                         harddrive space quickly.  It is recommended to
#                         restrict this to a chromosome and/or range (set with
#                         the interval command) of interest.
#                         
# Examples: multipoint 3 1.9
#
#          This will first do a full scan and zero-in scan once, then, if the
#          highest LOD >= 3, it will scan again.  If the next highest 
#          LOD >= 1.9, it will continue scanning until the last highest
#          LOD < 1.9.
#
#          trait q4
#          polymod
#          maximize
#          save model q4/null0
#          linkmod gaw10mibd/mibd.9.1.gz
#          option standerr 1
#          multipoint -ctparm {h2r h2q1 {par h2q1 se}} -se
#
#          This illustrates a simple use of the "custom parameterization"
#          option.  Note that unlike the typical use of the multipoint
#          command, it is necessary to create a "prototype" linkage model
#          first (here it is done with the linkmod command, but one might
#          also use linkqsd or build the model "by hand" setting up the
#          parameters and omega).  The list of parameters following -ctparm
#          may also include commands enclosed in a second level of braces.
#          The command must include more than one element as it is not
#          the braces but the element length that determines whether the
#          element is interpreted as a parameter or a command.
#          In this example, a command extracts the standard error of h2q1.
#          
# Requires:    mibddir, chromosome, and interval commands must have been
#              given to select mibd files to use.  finemap may be adjusted
#              with finemap command.
#
#              There must be a null0.mod model in the trait or outdir
#              directory.  This can be created with the polygenic command
#              prior to running multipoint.  (This model may include
#              household and covariate effects.  See the help for the
#              polygenic command for more information.)
#
# IMPORTANT NOTE:  In most cases, multipoint starts by loading a model named
#                  null0.mod from the current output directory.  The
#                  model currently in memory is ignored.  This is done
#                  because it is absolutely essentially that the null0
#                  model be the basis to build all multipoint models.  However,
#                  some options, such as -ctparm, use the model currently
#                  in memory when multipoint is invoked because all
#                  models are derived from a custom linkage model that
#                  the multipoint command does not necessarily know how
#                  to build.
#
# Notes:   1.  Summary output is written to multipoint.out in a subdirectory
#              named after the trait variable.  You can set another output
#              directory with the outdir command.  Contents of the output
#              directory will be purged of previous files at the beginning
#              of each invocation if -overwrite is used.
#
#          2.  The final "best" linkage model is link.mod.  In addition,
#              a series of additional "null" models is produced, starting
#              with null1.mod (containing 1 QTL), null2.mod, etc.  These
#              models are produced only if a LOD criterion is specified
#              and satisfied (so there is more than one pass).
#
#          3.  If a LOD adjustment is in effect (see lodadj command) it
#              is applied here.
#
#          4.  If models have two traits, the 2df LOD scores will be
#              converted to 1df effective LOD scores.  To override this, 
#              use the lodp command (see).  This feature
#              was first included with SOLAR beta version 2.0.1.
#
#          5.  At the beginning of each pass through the selected genome,
#              multipoint calls a user script named multipoint_user_start_pass
#              which takes one argument, the pass number (which starts at 1
#              for the first pass).  Within this routine, the user can change
#              the selected chromosomes or interval.
# -

proc multipoint {args} {

    set verbose 0
    set qu -q
    ifverbplus set verbose 1
    ifverbplus set qu ""

    set ts [trait]
    if {[llength $ts] == 1} {
	set multi 0
    } else {
	set multi 1
    }
    set nts [llength $ts]

# Process arguments

    set plist \'
    set ptlist \'
    set mplot 0
    set reuse_null 0
    set force_overwrite 0
    set re_start 0
    set lod_criteria {}
    set restart_model ""
    set restart_model_required 0
    set lod_format [use_global_if_defined Solar_Lod_Format "%.4f"]
    set slod 0
    set lod_type LOD
    set scoredebug 0
    set epistasis 0
    set save_all_models 0
    set nose 0
    set FLOAT_RHOQ 2
    set con_rhoq $FLOAT_RHOQ
    set linkproc linkmod
    set atemplate 0
    set searg ""
    set nullifneg 0

    set lod_criteria [read_arglist $args \
	    -overwrite {set force_overwrite 1} -ov {set force_overwrite 1} \
	    -restart {set re_start 1} -r {set re_start 1} \
	    -renew restart_model \
	    -plot {set mplot 1} \
	    -epistasis epistasis \
	    -score {set slod 1; set lod_type S-LOD} \
	    -scoredebug  {set slod 1; set lod_type S-LOD; set scoredebug 1} \
	    -saveall {set save_all_models 1} \
	    -nose {set nose 1} \
	    -parm plist \
            -cparm plist \
            -ctparm ptlist \
	    -rhoq con_rhoq \
	    -link linkproc \
	    -se {set searg -se} \
	    -nullifneg {set nullifneg 1} \
	    -nullbase {set reuse_null 1}]

    ensure_integer $epistasis

    foreach lod $lod_criteria {
	ensure_float $lod
    }

    if {0<[llength $restart_model]} {
	set re_start 1
	if {"." != $restart_model} {
	    load model $restart_model
	}
    }

    if {$re_start && $force_overwrite} {
		error "Arguments -restart and -overwrite are incompatible."
    }

    if {"\'" != $plist} {
	set noparama "-cparm"
	set aparama 1
    } elseif {"\'" != $ptlist} {
	set plist $ptlist
	set noparama "-cparm"
	set aparama 1
	set atemplate 1
    } else {
	set noparama ""
	set aparama 0
	set plist ""
    }

# Check for existing multipoint output files

    if {$force_overwrite} {
	eval delete_files_forcibly [glob -nocomplain [full_filename multipoint*.out]]
	eval delete_files_forcibly [glob -nocomplain [full_filename multipoint*.save]]
	purge_multipoint_output_directory
    } elseif {!$re_start && $epistasis} {
	eval delete_files_forcibly [glob -nocomplain [full_filename multipointe*.out]]
    } elseif {!$re_start && !$epistasis} {
	if {0<[llength [glob -nocomplain [full_filename multipoint*.out]]]} {
    error "Multipoint output files exist.  Use -overwrite or -restart option."
	}
    }

# If restarting, check previous lodadj (if any) with current lodadj

    if {$re_start} {
	if {[file exists [full_filename multipoint.out]]} {
	    set mfile [open [full_filename multipoint.out] r]
	    set found 1.0
	    while {-1 != [gets $mfile line]} {
		if {-1 != [string first "*** Using LOD Adjustment:" $line]} {
		    set found [lindex $line 4]
		    break
		}
	    }
	    close $mfile
	    set format_newlodadj [lodadj -query]
	    catch {set format_newlodadj [format %.5f $format_newlodadj]}
	    if {$found != $format_newlodadj} {
		puts "old lodadj: $found"
		puts "new lodadj: $format_newlodadj"
		error \
 "lodadj has changed.  Use 'madj' command to readjust old multipoint files."
	    }
	}
    }

    global Solar_Mibd_List
    set Solar_Mibd_List [global_mibd_list]
    global Solar_Fixed_Loci
    set Solar_Fixed_Loci 0
    global Solar_Interval
    set major_chromosome_list {}
    set major_locus_list {}
    set major_lod_list {}
    set last_chromosome -1
    global Solar_Plot_Last_Chrom
    set Solar_Plot_Last_Chrom -1

# Start from current model (including linkage) if epistasis

    set epiarg ""
    set el ""
    if {$epistasis} {
	set Solar_Fixed_Loci [h2qcount]
	set rootname nulle$Solar_Fixed_Loci
	maximize_quietly $rootname.out
	save model [full_filename $rootname.mod]
	outheader multipointe.out 1 $lod_type 0
	outresults multipointe.out $rootname "" [loglike] 1
	lodadj -query -inform stdout
	lodadj -query -inform multipoint.out
	set epiarg "-epistasis $epistasis"
	set el e

# start from existing model if arbitrary parameterization

    } elseif {$aparama} {
	set basemodel [full_filename multipoint.template.mod]
	save model $basemodel
	set Solar_Fixed_Loci 0

# Setup minimum resultfile and report null model

	load model [full_filename null0]
	set headings "Model LOD Loglike"
	set formats "%19s %11s %12.3f"
	set expressions "\$modelname \$lodscore \[loglike\]"
	set modelname polygenic
	set lodscore ""
	foreach par $plist {
	    lappend formats %9.6f
	    lappend headings $par
	    if {1<[llength $par]} {
		lappend expressions \[$par\]
	    } else {
		lappend expressions "\[parameter $par =\]"
	    }

# Parameters which do not exist in null model are forced to 0

	    if {![if_parameter_exists $par]} {
		parameter $par = 0
	    }
	}
	set open_option -create
	if {$re_start} {
	    set open_option -append
	    putsout multipoint.out "\n    *** Restarting multipoint scan"
	}
	set resultf [resultfile $open_option [full_filename multipoint.out] \
			 -headings $headings \
			 -formats $formats \
			 -expressions $expressions -display]
	if {!$re_start} {
	    resultfile $resultf -header
	    resultfile $resultf -write
	}
	load model $basemodel

    } else {

# No epistasis, must have previously stored null model

	if {![file exists [full_filename null0.mod]]} {
	    error "Model [full_filename null0] not found.\
\nThis can be created with polygenic command."
        }

# Load null model and display its results
# First, for univariate case (uses older output routines)

	if {!$multi} {
	outheader multipoint.out 1 $lod_type 0
	load model [full_filename null0]
	set modelname polygenic
	if {[check_house]} {
	    set modelname "household polygenic"
	}
	outresults multipoint.out $modelname "" [loglike] 0

# Check for certain warning conditions related to the null model

	catch {
	set as_quantitative 0
	set constrained_to_1 0
        if {{} != [find_string [full_filename null0.out] "quantitative!"]} {
	    set as_quantitative 1
	}
	if {![catch {set cval [find_simple_constraint sd]}] && $cval == 1} {
	    set constrained_to_1 1
	}
	if {$constrained_to_1} {
	    set discrete_trait 0
	    if {!$as_quantitative} {
		set tstats [stats -return -q]
		set discrete_trait [stats_get $tstats discrete]
	    }
	    if {!$discrete_trait || $as_quantitative} {
		putsout multipoint.out \
"\n\tWARNING!  YOU HAVE PARAMETER SD CONSTRAINED TO 1.0 !"
		putsout multipoint.out \
"\tThis is probably not what you intended."
		putsout multipoint.out \
"\tYou need to use command \"model new\" before changing from discrete to"
		putsout multipoint.out \
"\tquantitative trait analysis."
	    }
	}
	if {$as_quantitative} {
	    putsout multipoint.out \
"\n\tWARNING!  You are analyzing a discrete trait as quantitative!"
	    putsout multipoint.out \
"\tSee \"help discrete-notes\" for discussion."
	}
	}

	} else {
#	    countmax

# Set up resultfile for bivariate output

	    load model [full_filename null0]
	    set headings "Model LOD Loglike"
	    set formats "%19s %11s %12.3f"
	    set expressions "\$modelname \$lodscore \[loglike\]"

# Set modelname of base model, and, if house, set upper c2 bounds also

	    set modelname polygenic
	    if {[check_house]} {
		set modelname "household polygenic"
	    }

# OK, back to resultfile

	    set lodscore ""
	    foreach tr $ts {
		lappend headings H2r($tr)
		lappend formats %9.6f
		lappend expressions "\[parameter H2r($tr) =\]"
	    }
	    lappend headings RhoG
	    lappend formats %9.6f
	    lappend expressions "\[parameter rhog =\]"

	    set error_status ""
	    lappend headings ""
	    lappend formats "%s"
	    lappend expressions "\$error_status"

	    set open_option -create
	    if {$re_start} {
		set open_option -append
		putsout multipoint.out "\n    *** Restarting multipoint scan"
	    }
	    set resultf [resultfile $open_option \
			     [full_filename multipoint.out] \
			     -headings $headings \
			     -formats $formats \
			     -expressions $expressions -display]
	    if {!$re_start} {
		resultfile $resultf -header
		resultfile $resultf -write
	    }
	}
	lodadj -query -inform stdout
	lodadj -query -inform multipoint.out

    }  ;# End if _not_ epistasis

# Continue scanning while current lod_criteria is satisfied
#   (This is a "do-while"...test is near bottom...one pass assured)

    while {1} {
	
# Define scanning procedure for each linkage model

# (Keep sub-scripts un-indented for proper autoloading
proc scanpass {mibdlist} {
            upvar mplot mplot
	    upvar highest_lod hlod
	    upvar highest_mibdfile hmibd
            upvar verbose verbose
	    global Solar_Fixed_Loci
            upvar last_chromosome last_chromosome
            upvar reuse_null reuse_null
            upvar finemap_level finemap_level
            upvar finemap_list finemap_list
            upvar re_start re_start
            upvar bnd_or_conv_err_in_pass bnd_or_conv_err_in_pass
            upvar slod slod
            upvar scoredebug scoredebug
            upvar previous_h2q_value previous_h2q_value
            upvar epiarg epiarg
            upvar noparama noparama
            upvar aparama aparama
            upvar atemplate atemplate
            upvar el el
            upvar save_all_models save_all_models
            upvar multi multi
            upvar resultf resultf
            upvar con_rhoq con_rhoq
            upvar FLOAT_RHOQ FLOAT_RHOQ
            upvar linkproc linkproc
            upvar searg searg

            set h2q_index [expr $Solar_Fixed_Loci + 1]
            set minlike ""
	    foreach mibdfile $mibdlist {

# Set new_chromosome and locus; loading null model if necessary

		set retry_mode 3
		set new_chromosome [get_chromosome $mibdfile]
		set locus [get_locus $mibdfile]

		if {!$atemplate && !$aparama && ($reuse_null || \
		     $last_chromosome != $new_chromosome || $slod==1)} {
		    set retry_mode 1
		    set last_chromosome $new_chromosome
		    model load [full_filename null$Solar_Fixed_Loci]
		    set minlike [loglike]
		}
# New template reloading rules for -cparm (4.1.5)
#   If chromosome is different, or if locus is > 11 + previous locus
#   Reload template model
# Previously, and very badly, template was never reloaded, leading to frequent
#   convergence errors
		if {!$atemplate && $aparama} {
# Determine chrom,locus of currently loaded chromosome by inspection
# This is really "a better way" of finding out that trying to keep track of
#   all changes through a "last" variable, and the "last" variable used here
#   isn't reliable anyway, it's reset to -1 on each finemap segment.
		    set current_mibd [get_current_mibd]
		    if {$current_mibd == ""} {
			set current_chrom -1
			set current_locus -1
		    } else {
			set current_chrom [get_chromosome $current_mibd]
			set current_locus [get_locus $current_mibd]
		    }
		    if 	{$current_chrom != $new_chromosome || \
			     ![is_integer $locus] || \
			     ![is_integer $current_locus] || \
			     $current_locus + 11 < $locus || \
			     $current_locus > $locus + 11} {
			ifdebug puts "Reloading template model"
			load model [full_filename multipoint.template.mod]
			set minlike ""
			set last_chromosome $new_chromosome
		    }
		}

# If restarting, see if this locus already done

		if {$re_start && ![catch {set lodlist \
			[get_prev_lod_and_h2q $h2q_index $new_chromosome \
			$locus]}]} {
		    set h2q_value [lindex $lodlist 1]
		    set olod [lindex $lodlist 0]
		    if {![is_nan $olod]} {
			if {$olod > $hlod} {
			    set hlod $olod
			    set hmibd $mibdfile
			}
			if {$olod >= $finemap_level} {
			    lappend finemap_list [list $mibdfile $h2q_value]
			}
		    }
		    if {$verbose} {
			puts \
       "\n    *** Using previous results for Chrom $new_chromosome  Loc $locus"
		    }
		} else {
		    if {$verbose} {
			puts \
                        "\n    *** Analyzing Chrom $new_chromosome  Loc $locus"
		    }
		    if {$atemplate} {
			load model [full_filename multipoint.template.mod]
			set minlike ""
		    }
		    if {!$slod} {
			eval $linkproc $mibdfile $noparama $epiarg -lasth2q \
				$previous_h2q_value $searg
			set previous_h2q_value none
		    } else {
			eval $linkproc $mibdfile $noparama $epiarg -zerostart \
			    $searg
			option StandErr 1
			option ScoreOnlyIndex $h2q_index
			option MaxIter 1
		    }
		    if {$con_rhoq != $FLOAT_RHOQ} {
			parameter rhoq1 = $con_rhoq
			constraint rhoq1 = $con_rhoq
		    }
		    set boundary_error 0
		    if {[catch {set newchf [format "%02d" $new_chromosome]}]} {
			set newchf [format "%2s" $new_chromosome]
			set char0 [string range $new_chromosome 0 0]
			set char1 [string range $new_chromosome 1 1]
			if {[is_integer $char0] && ![is_integer $char1]} {
			    set newchf [format "%2s" 0$new_chromosome]
			}
		    }
		    set locf [format "%5s" $locus]
		    if {$save_all_models} {
			set outfilename \
			  multi.pass$h2q_index.$new_chromosome.$locus
		    } else {
			set outfilename temp
		    }
		    set max_status [maxtry $retry_mode $h2q_index $mibdfile \
			    [full_filename $outfilename.out] $verbose $minlike]
		    if {$save_all_models} {
			save model [full_filename $outfilename]
		    }
		    if {"" != $max_status && "ConsRhoq" != $max_status && \
			"ZeroLodX" != $max_status} {
			set bnd_or_conv_err_in_pass 2
			set last_chromosome -1
			delete_files_forcibly [full_filename temp.out]
			if {!$multi && !$aparama} {
			    outresults multipoint$el$h2q_index.out \
				"chrom $newchf  loc $locf" \
				"" NaN $h2q_index 2 none -none
			} else {
			    set modelname "chrom $newchf  loc $locf"
			    set lodscore ""
			    set error_status "ConvrgErr"
			    resultfile $resultf -write
			}
			continue
		    }
		    delete_files_forcibly [full_filename temp.out]
		    if {[catch {set olod [lod_or_slod_n $slod \
			    $Solar_Fixed_Loci]}]} {
			set boundary_error 3
			set bnd_or_conv_err_in_pass 3
			puts "catch failed, now setting olod NAN"
			set olod NaN
		    } else {
			if {$olod > $hlod} {
			    set hlod $olod
			    set hmibd $mibdfile
			    model save [full_filename best$h2q_index]
			}
			if {$olod >= $finemap_level} {
			    if {!$multi && !$aparama} {
				set h2q_value [parameter h2q$h2q_index start]
			    } else {
				set h2q_value 0.01  ;# lots of work req. for bi
			    }
			    lappend finemap_list [list $mibdfile $h2q_value]
			}
			if {$save_all_models} {
			    save model [full_filename \
			      multi.pass$h2q_index.$new_chromosome.$locus]
			}
		    }
		    if {0==[string compare [verbosity] "verbosity max"]} {
			scanputs "    *** Total process memory is [memory]"
		    }
		    if {$slod && $scoredebug} {
			set score [parameter h2q$h2q_index score]
			set lcse [parameter h2q$h2q_index se]
		    } else {
			set score none
			set lcse none
		    }
		    if {!$multi && !$aparama} {
		    outresults multipoint$el$h2q_index.out \
			    "chrom $newchf  loc $locf" \
		      $olod [loglike] $h2q_index $boundary_error none -none \
		      $score $lcse
		    } else {
			set modelname "chrom $newchf  loc $locf"
			if {[catch {set lodscore [format %11.4f $olod]}]} {
			    set lodscore [format %11s $olod]
			}
			set error_status $max_status
			resultfile $resultf -write
		    }
		}
		if {$mplot} {
		    catch {plot -quick -chrom $new_chromosome -pass $h2q_index}
		    ifverbplus puts \
		     "plot -quick -chrom $new_chromosome -pass $h2q_index"
		}
	    }

# Do replot without "-quick" option now that chromosome is done
#   (if any chromosomes were done!)

	    if {$mplot && 0<[llength $mibdlist]} {
		catch {plot -chrom $new_chromosome -pass $h2q_index}
		ifverbplus puts \
		 "plot -chrom $new_chromosome -pass $h2q_index"
	    }
        }


# Do regular scan through using established interval and range

	set previous_h2q_value none
	set bnd_or_conv_err_in_pass 0
	set h2q_index [expr $Solar_Fixed_Loci + 1]
	set highest_lod -10000
	set highest_mibdfile ""
	set finemap_list {}
	set finemap_level 1000
	if {0!=[string compare [finemap] off] && \
		0!=[string compare [finemap] default]} {
	    set rindex [expr [lowest [llength [finemap]] $h2q_index] - 1]
	    set finemap_level [lindex [finemap] $rindex]
	}
	if {![file exists [full_filename multipoint$el$h2q_index.out]]} {
	    if {!$verbose} {puts ""}
	    if {!$multi && !$aparama} {
		outheader multipoint$el$h2q_index.out $h2q_index $lod_type 0 \
		    $epistasis
	    } else {
		set open_option -create
	    }
	} 

# Multivariate uses generic "resultfile"

        if {$multi && !$aparama} {
# Remove previous error_status

	    set headings [lrange $headings 0 [expr [llength $headings] - 2]]
	    set expressions [lrange $expressions 0 \
				 [expr [llength $expressions] - 2]]
	    set formats [lrange $formats 0 [expr [llength $formats] - 2]]

# Add values for new traits
	    
	    foreach tr $ts {
		set newterm [catenate H2q$h2q_index ($tr)]
		lappend headings $newterm
		lappend expressions "\[parameter $newterm =\]"
		lappend formats %9.6f
	    }
	    if {$nts == 2} {
		lappend headings RhoQ$h2q_index
		lappend expressions "\[parameter rhoq$h2q_index =\]"
		lappend formats %9.6f
	    } else {
		for {set iti 1} {$iti < $nts} {incr iti} {
		    for {set itj [expr $iti + 1]} {$itj <= $nts} {incr itj} {
			set pname [catenate RhoQ$h2q_index _ $iti$itj]
			lappend headings $pname
			lappend expressions "\[parameter $pname =\]"
			lappend formats %9.6f
		    }
		}
	    }
			

# Add error_status

	    lappend headings ""
	    lappend expressions "\$error_status"
	    lappend formats "%s"

# Update resultfile

	    set resultf [resultfile $open_option \
			     [full_filename multipoint$h2q_index.out] \
			     -headings $headings \
			     -formats $formats \
			     -expressions $expressions -display]
	} elseif {$aparama} {
	    set resultf [resultfile $open_option \
			     [full_filename multipoint1.out] \
			     -headings $headings \
			     -formats $formats \
			     -expressions $expressions -display]
	}
        if {$multi || $aparama} {
            if {"-create" == $open_option} {
		resultfile $resultf -header
	    } else {
		resultfile $resultf -header -displayonly
	    }
	}
	    
        catch {multipoint_user_start_pass [expr $Solar_Fixed_Loci + 1]}
	set chrom [chromosome]
	set int [interval]
	scanputs \
"\n    *** Pass $h2q_index:  Chrom $chrom;  Interval $int"
	scanpass $Solar_Mibd_List

# Do finemapping to find the real peak

	if {$highest_lod > -10000} {
	    if {0!=[string compare [finemap] off]} {
		if {$Solar_Interval > 1} {
		    if {[llength $finemap_list]} {
			set last_chrom -1
			set last_locus -1
			set test_list {}

# Scan all points around entries in finemap_list

                        foreach elem $finemap_list {
			    set mibdfile [lindex $elem 0]
			    set h2q_value [lindex $elem 1]
			    set cnum [get_chromosome $mibdfile]
			    set lnum [get_locus $mibdfile]
			    if {$last_chrom == $cnum && \
			      [expr $last_locus + $Solar_Interval] == $lnum} {
			        set lm [expr $lnum + 1]
		            } else {
				set lm [expr $lnum + 1 - $Solar_Interval]
			    }
			    set last_chrom $cnum
			    set last_locus $lnum
			    set hm [expr $lnum + $Solar_Interval - 1]
			    set sublist [mibd_list [mibddir] $cnum 1 $lm $hm]
			    set center [lsearch $sublist $mibdfile]
			    if {$center != -1} {
			      set sublist [lreplace $sublist $center $center]
			    }

# For each sublist of new mibd's, a previous_h2q_value is included

			    set tlist [list $h2q_value $sublist]
			    lappend test_list $tlist
			}
			set last_chrom -1
			set col 1
                        scanputs -nonewline \
			"    *** Fine Map over the following locations:"
			foreach tlist $test_list {
			    set mlist [lindex $tlist 1]
			    foreach m $mlist {
				set cnum [get_chromosome $m]
				set lnum [get_locus $m]
				if {$cnum != $last_chrom} {
				    set last_chrom $cnum
			            scanputs -nonewline \
					    "\n        Chrom $cnum  Loc"
				    set col 22
				}
				if {$col > 71} {
				    scanputs -nonewline \
					    "\n                      "
				    set col 22
				}
				scanputs -nonewline " $lnum"
				set col [expr $col + [string length " $lnum"]]
			    }
			}
			scanputs ""
			foreach tlist $test_list {
			    set last_chromosome -1
			    set previous_h2q_value [lindex $tlist 0]
			    set mlist [lindex $tlist 1]
			    scanpass $mlist
			}
		    } else {
			scanputs -nonewline \
		"    *** No LOD met finemap criterion of $finemap_level"
		    }
		}
	    }

# Re-do final linkage model with standard errors and saving output
	    
	    set last_chromosome -1
	    set got_se 0
	    set h2q_index [expr $Solar_Fixed_Loci + 1]

# *** This section applies to restart only

	    if {$re_start} {
		if {[file exists [full_filename null$h2q_index.mod]]} {
		    set got_se 1
		    if {$verbose} {
			puts "    *** Re-using previous null$h2q_index.mod"
		    }
		} elseif {!$slod && \
		    ![file exists [full_filename best$h2q_index.mod]]} {
		    if {$verbose} {
			puts \
	         "    *** Maximizing model with highest lod found in prior run"
		    }
		    if {$aparama || $atemplate} {
			load model [full_filename multipoint.template.mod]
		    } else {
			load model [full_filename null$Solar_Fixed_Loci]
		    }
		    eval $linkproc $highest_mibdfile $noparama $epiarg $searg
		    if {$con_rhoq != $FLOAT_RHOQ} {
			parameter rhoq1 = $con_rhoq
			constraint rhoq1 = $con_rhoq
		    }
		    model save [full_filename best$h2q_index.mod]
		    if {"" != [maxtry 1 $h2q_index $highest_mibdfile \
			    [full_filename null$h2q_index] $verbose $minlike]} {
			if {$aparama || $atemplate} {
			    load model [full_filename multipoint.template.mod]
			} else {
			    load model [full_filename null$Solar_Fixed_Loci]
			}
			eval $linkproc $noparama $epiarg $highest_mibdfile \
			    $searg
			if {$con_rhoq != $FLOAT_RHOQ} {
			    parameter rhoq1 = $con_rhoq
			    constraint rhoq1 = $con_rhoq
			}
               error "Can't maximize highest lod model: best$h2q_index.mod"
		    }
		    model save best$h2q_index
		}
	    }

#  OK, now we've got bestX.mod, regardless of restart or regular
#  If we're not restarting, or we haven't got S.E. model,
#  compute S.E.

	    if {!$got_se} {
		if {$nose && !$slod} {
		    model load [full_filename best$h2q_index]
		    model save [full_filename null$h2q_index]
		} else {

		if {$verbose} {
		    puts \
"    *** Reloading best linkage model in this pass to compute standard errors"
                }
		if {$slod} {
		    if {$aparama || $atemplate} {
			load model [full_filename multipoint.template.mod]
		    } else {
			load model [full_filename null$Solar_Fixed_Loci]
		    }
		    eval $linkproc $highest_mibdfile $searg
		    if {$con_rhoq != $FLOAT_RHOQ} {
			parameter rhoq1 = $con_rhoq
			constraint rhoq1 = $con_rhoq
		    }
#		    option maxiter 1000
#                   option ScoreOnlyIndex -1
		} else {
		    model load [full_filename best$h2q_index]
		    perturb
		}
		option standerr 1
		if {0 == [catch {set maxmsg [maxtry 3 $h2q_index \
			$highest_mibdfile \
			[full_filename null$h2q_index] $verbose $minlike]}] \
			&& $maxmsg == ""} {
		    model save [full_filename null$h2q_index]
		    if {!$slod} {
			set hlodf [format %.2f $highest_lod]
			set lodf 0.0
			catch {set lodf [format %.2f \
				[lod_or_slod_n $slod $Solar_Fixed_Loci]]}
			if {$hlodf !=  $lodf} {
			    set bnd_or_conv_err_in_pass -1
			}
		    }
		} else {

# Maximization of best model with S.E. failed
		    if {$verbose} {
			puts "    *** Couldn't compute standard errors."
		    }

# If slod, try again but w/o S.E.
# For slod, we NEED this for next pass

		    if {$slod} {
			if {$aparama || $atemplate} {
			    load model [full_filename multipoint.template.mod]
			} else {
			    load model [full_filename null$Solar_Fixed_Loci]
			}
			eval $linkproc $highest_mibdfile $searg
			if {$con_rhoq != $FLOAT_RHOQ} {
			    parameter rhoq1 = $con_rhoq
			    constraint rhoq1 = $con_rhoq
			}
#			option maxiter 1000
#                       option ScoreOnlyIndex -1
			option standerr 0
			if {0 == [catch {set maxmsg [maxtry 3 $h2q_index \
				$highest_mibdfile \
				[full_filename null$h2q_index] $verbose \
							$minlike]}] && \
				$maxmsg == ""} {
			    model save [full_filename null$h2q_index]
			} else {
			    set bnd_or_conv_err_in_pass -2
			}
		    } else {


# If not slod, we'll just have to go with previously saved bestX model
# without standard errors

			model load [full_filename best$h2q_index]
			model save [full_filename null$h2q_index]
		    }
		}
		if {$bnd_or_conv_err_in_pass >= 0} {
		    delete_files_forcibly [full_filename best$h2q_index.mod]
		}
	    }   
	}
	}
	
# Sort output file

        set multipoint_index [expr $Solar_Fixed_Loci + 1]
        if {0 != [string length [usort]]} {
	    set mname [full_filename multipoint$multipoint_index.out]
	    if {$verbose} {puts "\n    *** Sorting $mname"}
	    catch {
		exec head -2 $mname >$mname.tmp
		if {[exec uname] == "SunOS"} {
		    exec tail +3 $mname | [usort] -k 2,2 -k 4,4n >>$mname.tmp
		} else {
		    exec tail -n +3 $mname | [usort] -k 2,2 -k 4,4n >>$mname.tmp
		}		    
		file rename -force $mname.tmp $mname
	    }
	}

# If restart, rescan output file and remove duplicates

        if {$re_start} {
	    set resinfilename [full_filename multipoint$multipoint_index.out]
	    set resinfile [open $resinfilename]
	    set resoutfile [open $resinfilename.tmp w]
	    gets $resinfile resline ;# copy header
	    puts $resoutfile $resline
	    gets $resinfile resline
	    puts $resoutfile $resline
	    set last_reschrom ""
	    set last_resloc ""
	    set lastresline ""
	    while {-1 != [gets $resinfile resline]} {
		set reschrom [lindex $resline 1]
		set resloc [lindex $resline 3]
		if {[string compare $reschrom $last_reschrom] || \
			[string compare $resloc $last_resloc]} {
		    if {"" != $lastresline} {
			puts $resoutfile $lastresline
		    }
		    set lastresline $resline
		} else {
		    if {[is_float [lindex $resline end]] || \
			    ![is_float [lindex $lastresline end]]} {
			set lastresline $resline  ;# only change if not worse
		    }
		}
		set last_reschrom $reschrom
		set last_resloc $resloc
	    }
	    if {"" != $lastresline} {
		puts $resoutfile $lastresline
	    }
	    close $resinfile
	    close $resoutfile
	    file rename -force $resinfilename.tmp $resinfilename
	}

# Report best location found (regardless of verbosity)
# UNLESS there was a convergence error

	if {[llength $highest_mibdfile] && \
		$bnd_or_conv_err_in_pass == 0} {
	    set chromosome [get_chromosome $highest_mibdfile]
	    set locus [get_locus $highest_mibdfile]
	    set old_verbose $verbose
	    set verbose 1
	    set flod [format $lod_format $highest_lod]
	    set pno [expr $Solar_Fixed_Loci + 1]
	    scanputs \
"\n    *** Highest $lod_type in pass $pno was $flod at Chrom $chromosome Loc $locus"
            set verbose $old_verbose
        }

# See if another scan is required

	if {$bnd_or_conv_err_in_pass > 1 && \
	    $Solar_Fixed_Loci <=  0} {
	    set old_verbose $verbose
	    set verbose 1
            scanputs \
"\n    *** Exiting because convergence errors occurred in first pass"
            scanputs \
"    *** Use 'help boundary' for help resolving such errors"
            set verbose $old_verbose
            break
        }

	if {0 == [llength $lod_criteria]} {
	    scanputs \
"\n    *** No $lod_type criterion specified...exiting after one pass"
            break
        }

	set lod_criterion [lindex $lod_criteria 0]
	if {1 < [llength $lod_criteria]} {
	    set lod_criteria [lrange $lod_criteria 1 end]
	}

# We exit here either if LOD criterion not satisfied or if bnd/conv error

 	if {$highest_lod < $lod_criterion || \
	    $bnd_or_conv_err_in_pass} {

	    if {!$bnd_or_conv_err_in_pass && $verbose} {
	        puts "\n    *** Current LOD criterion is $lod_criterion"
	        puts "    *** LOD criterion was not satisfied in last scan"
	    }
	    if {$Solar_Fixed_Loci > 0} {
		set old_verbose $verbose
		set verbose 1
		if {$Solar_Fixed_Loci > 1} {
		    scanputs \
		    "\n    *** $Solar_Fixed_Loci loci have satisfied criteria:"
		} else {
		    scanputs \
		    "\n    *** $Solar_Fixed_Loci locus has satisfied criteria:"
		}

		for {set i 0} {$i < $Solar_Fixed_Loci} {incr i} {
		    set chromosome [lindex $major_chromosome_list $i]
		    set locus [lindex $major_locus_list $i]
		    set lod [lindex $major_lod_list $i]
		    set flod [format $lod_format $lod]
		    scanputs \
              "        Chromosome: $chromosome   Loc: $locus   $lod_type: $flod"
		}
		set verbose $old_verbose
		catch {eval exec ln [full_filename null$Solar_Fixed_Loci.mod] \
			[full_filename link.mod]}
		scanputs \
                    "\n    *** Best model is link.mod, now being re-loaded..."
		model load [full_filename link.mod]
	    }
	    scanputs \
		    "    *** Pedigree:    [topline pedigree.info]"
	    scanputs \
		    "    *** Phenotypes:  [phenotypes -files]"
	    set rfilename [full_filename multipoint*.out]
	    if {!$verbose} {puts ""}
	    puts "    *** Additional information is in files named $rfilename"
	    if {$bnd_or_conv_err_in_pass > 0} {
		set old_verbose $verbose
		set verbose 1
		scanputs \
"\n    *** Exiting because convergence errors occurred in last pass"
                scanputs \
"    *** Use 'help boundary' for help resolving such errors"
                set verbose $old_verbose
            } elseif {$bnd_or_conv_err_in_pass == -1} {
		set old_verbose $verbose
		set verbose 1
		scanputs \
"\n    *** Exiting because LOD changed when computing Std Errors on best model"
                set h2q_ind [expr $Solar_Fixed_Loci + 1]
                scanputs \
"    *** Compare null$h2q_ind.mod and best$h2q_ind.mod"
                scanputs \
"    *** This indicates poor convergence"
                set verbose $old_verbose
            } elseif {$bnd_or_conv_err_in_pass == -2} {
		set old_verbose $verbose
		set verbose 1
		scanputs \
"\n    *** Exiting because we failed to get MLE for best model in pass"
                set verbose $old_verbose
            }
	    break
	}

# Add results to lists

	set highest_chromosome [get_chromosome $highest_mibdfile]
	set highest_locus [get_locus $highest_mibdfile]
    	lappend major_lod_list $highest_lod
	lappend major_chromosome_list $highest_chromosome
	lappend major_locus_list $highest_locus

# Linkage model becomes new null model

	set Solar_Fixed_Loci [expr 1 + $Solar_Fixed_Loci]
	if {$verbose} {puts \
          "    *** Repeating scan with $Solar_Fixed_Loci fixed loci\n"}
    }
}

proc scanputs {args} {
    upvar el el
    set nonewline 0
    set message [lindex $args 0]
    if {0==[string compare [lindex $args 0] -nonewline]} {
	set nonewline 1
	set message [lindex $args 1]
    }
    set sfile [open [full_filename multipoint$el.out] a]
    if {$nonewline} {
	puts -nonewline $sfile $message
    } else {
	puts $sfile $message
    }
    close $sfile

    upvar verbose verbose
    if {$verbose} {
	if {$nonewline} {
	    puts -nonewline $message
	} else {
	    puts $message
	}
    }
}

proc get_current_mibd {} {
    set matlist [matrix]
    set len [llength $matlist]
    for {set i [expr $len - 1]} {$i > 0} {incr i -1} {
	set test [lindex $matlist $i]
	if {[file extension $test] == ".gz"} {
	    set filename [file tail $test]
	    if {[string range $filename 0 4] == "mibd."} {
		return $test
	    }
	}
    }
    return ""
}
	

# solar::madj --
#
# Purpose:  Apply current lodadj to a previous multipoint run
#
# Usage:    madj
#           madj -restore    ;# restore ORIGINAL multipoint files
#           madj -undo       ;# restore previous multipoint files
#
# Notes: trait or outdir must already have been selected.
#
#        madj applies loadadj from lodadj.info file in trait/outdir.
#
#        madj may be used on incomplete multipoint runs prior to restarting.
#
#        It is not necessary to -restore before applying another lodadj.
#        Some roundoff errors occur in last decimal place, but do not
#        "accumulate" over multiple runs because LOD's are calculated
#        from loglikelihood values, not previous LOD's.  NULL models must
#        be present.
#
#        If there is an error, there should either be a "no files modified"
#        or "restoring previous files" message.  If not, or if it is
#        desired to restore the ORIGINAL multipoint files for any
#        reason, use the command "madj -restore."  The FIRST time madj is
#        run, those files were saved as multipoint*.save.  (The PREVIOUS
#        set of multipoint files were also saved as multipoint*.tmp, and
#        may also be restored with the "madj -undo" command.)
# -

proc madj {args} {

# ensure trait/outdir

    full_filename foo
#
# recalc is now a fixed option; LOD's are recalculated using null models
#     and stored loglikelihood values.
#   If recalc is 0, LOD's would be scaled from previous LOD scores.
#

    set recalc 1 

# Process arguments: -restore and -undo options are separate procs

    set restore 0
    set undo 0
    set badargs [read_arglist $args -restore {set restore 1} \
	    -undo {set undo 1} -* foo]
    if {"" != $badargs} {
	puts "No files modified"
	error "Invalid arguments: $badargs"
    }
    if {$restore} {
	return [madj_restore]
    }
    if {$undo} {
	return [madj_undo]
    }


# Check for all the files we need and saved files

    set backup_already_done 0
    if {![file exists [full_filename multipoint.out]]} {
	puts "No files modified"
	error "Missing file [full_filename multipoint.out]"
    }
    if {[file exists [full_filename multipoint.save]]} {
	set backup_already_done 1
    }
    set t_file [open [full_filename multipoint.out] r]
    set last_pass 0
    while {-1 != [gets $t_file line]} {
	if {-1 != [string first "*** Highest LOD in pass" $line]} {
	    if {![file exists [full_filename null$last_pass.mod]]} {
		puts "No files modified"
		error "Missing model [full_filename null$last_pass.mod]"
	    }
	    incr last_pass
	    if {![file exists [full_filename multipoint$last_pass.out]]} {
		puts "No files modified"
		error "Missing file [full_filename multipoint$last_pass.out]"
	    }

# Check that save file exists or does not exist as expected

	    if {[file exists [full_filename multipoint$last_pass.save]]} {
		if {!$backup_already_done} {
		    puts "No files modified"
		    puts \
		      "Found multipoint$last_pass.save but not multipoint.save"
		    error "Missing multipoint.save file?"
		}
	    } elseif {$backup_already_done} {
		puts "No files modified"
		error "Missing multipoint$last_pass.save file"
	    }
	}
    }
    close $t_file

# Now check for additional multipointN.out files not yet in multipoint.out

    while {[file exists [full_filename multipoint[expr $last_pass+1].out]]} {
	if {![file exists [full_filename null$last_pass.mod]]} {
	    puts "No files modified"
	    error "Missing model [full_filename null$last_pass.mod]"
	}
	incr last_pass
	if {[file exists [full_filename multipoint$last_pass.save]]} {
	    if {!$backup_already_done} {
		puts "No files modified"
		puts \
		  "Found multipoint$last_pass.save but not multipoint.save"
		error "Missing multipoint.save file?"
	    }
	} elseif {$backup_already_done} {
	    puts "No files modified"
	    error "Missing multipoint$last_pass.save file"
	}
    }

# Backup if backup not done before

    if {!$backup_already_done} {
	file copy -force [full_filename multipoint.out] [full_filename multipoint.save]
	for {set i 1} {$i <= $last_pass} {incr i} {
	    file copy -force [full_filename multipoint$i.out] \
		    [full_filename multipoint$i.save]
	}
    }


# Start reading and copying multipoint.out file

    set new_adj [lodadj -query]
    file rename -force [full_filename multipoint.out] [full_filename multipoint.tmp]
    set m_file [open [full_filename multipoint.tmp] r]
    set o_file [open [full_filename multipoint.out] w]
    set found ""
    set old_adj 1.0

# Find and replace LOD Adj line (if any) in multipoint.out file

    while {-1 != [gets $m_file line]} {
	if {-1 != [string first "*** Using LOD Adjustment:" $line]} {
	    set found [lindex $line 4]
	    break
	} elseif {-1 != [string first "*** Pass 1:" $line]} {
	    break
	}
	puts $o_file $line
    }
    if {"" == $found} {
	puts $o_file "    *** Using LOD Adjustment:  [format %.5f $new_adj]\n"
	puts $o_file $line
    } else {
	set old_adj $found
	puts $o_file "    *** Using LOD Adjustment:  [format %.5f $new_adj]"
    }

# Determine adj_factor (not actually used unless recalc is 0)

    set adj_factor [expr $new_adj / $old_adj]

# Now, update all multipointN.out files
# Keep tabs of highest lods for final update of multipoint.out file

    set highest_lods {}

    for {set N 1} {$N <= $last_pass} {incr N} {

	set null_loglike [oldmodel null[expr $N - 1] loglike]
	file rename -force [full_filename multipoint$N.out] \
		[full_filename multipoint$N.tmp]
	set mfile [open [full_filename multipoint$N.tmp] r]
	set ofile [open [full_filename multipoint$N.out] w]

# Copy "header" lines

	for {set i 0} {$i < 2} {incr i} {
	    if {-1 != [gets $mfile line]} {
		puts $ofile $line
	    }
	}

# Copy rest of file, adjusting LOD all the way

	set highest_lod 0.0
	while {-1 != [gets $mfile line]} {
	    set old_lod [lindex $line 4]
	    set current_loglike [lindex $line 5]

# New LOD calculated from loglikelihoods (lodadj built-in to lod)

	    if {$recalc} {
		if {![is_nan $current_loglike] && ![is_nan $null_loglike]} {
		    set new_lod [lod $current_loglike $null_loglike]
		    if {$new_lod < 0.0} {
			set new_lod 0.0
		    }
		    if {$new_lod > $highest_lod} {
			set highest_lod $new_lod
		    }
		    set new_lod [format %.4f $new_lod]
		} else {
		    set new_lod NaN
		}
	    } else {

# LOD scaled from old lod

		if {![is_nan $old_lod]} {
		    set new_lod [format %.4f [expr $old_lod * $adj_factor]]
		} else {
		    set new_lod NaN
		}
	    }

# output new LOD to file

	    set line [format "%s%12s%s" \
		    [string range $line 0 18] $new_lod \
		    [string range $line 31 end]]
	    puts $ofile $line
	}
	close $mfile
	close $ofile
	lappend highest_lods $highest_lod
    }

# Now copy rest of multipoint.out file
# Updating LOD scores as we go...

    set final_index 0
    set ending 0
    while {-1 != [gets $m_file line]} {
	if {-1 != [string first "*** Highest LOD in pass" $line]} {
	    set passno [lindex $line 5]
	    set chromno [lindex $line 10]
	    set locno [lindex $line 12]
	    set newlod [lindex $highest_lods [expr $passno - 1]]
	    puts $o_file [format \
              "    *** Highest LOD in pass %d was %.4f at Chrom %d Loc %d" \
              $passno $newlod $chromno $locno]
	} elseif {-1 != [string first "satisfied criter" $line]} {
	    set ending 1
	    puts $o_file $line
	} elseif {-1 != [string first "Chromosome:" $line]} {
	    if {!$ending} {
		close $o_file
		close $m_file
		madj_undo "Parsing error: Chromosome: in multipoint.out "
	    }
	    set chromosome [lindex $line 1]
	    set locus [lindex $line 3]
	    set flod [format %.4f [lindex $highest_lods $final_index]]
	    incr final_index
	    set lod_type LOD
	    puts $o_file \
             "        Chromosome: $chromosome   Loc: $locus   $lod_type: $flod"
        } else {
	    puts $o_file $line
	}
    }
    close $o_file
    close $m_file
    return "DONE:  $old_adj  ->  $new_adj"
}

proc madj_restore {args} {
    puts "Restoring original files..."
    file copy -force [full_filename multipoint.save] [full_filename multipoint.out]
    for {set i 1} {[file exists [full_filename multipoint$i.save]]} {incr i} {
	file copy -force [full_filename multipoint$i.save] \
		[full_filename multipoint$i.out]
    }
    if {"" != $args} {
	error $args
    }
}

proc madj_undo {args} {
    puts "Restoring previous files..."
    file copy -force [full_filename multipoint.tmp] [full_filename multipoint.out]
    for {set i 1} {[file exists [full_filename multipoint$i.tmp]]} {incr i} {
	file copy -force [full_filename multipoint$i.tmp] \
		[full_filename multipoint$i.out]
    }
    if {"" != $args} {
	error $args
    }
}


# solar::stringplot
#
# Purpose:  String plot of entire genome scan
#
# Usage:    multipoint
#           stringplot [-pass pass] [-allpass] [-title] [-lod <lod>] [-lodmark]
#                      [-color <name>] [-noconv] [-date] [-nomark]
#                      [-font <X-font-spec>] [-titlefont <X-font-spec>]
#                      [-dash <dash spec>] [-linestyle <dash spec>]
#                      [-mibddir <mibddir>]
#
# Notes:    You can also use the command "plot -string" which has the
#           same options and works identically.  For further information on
#           the options, see "help plot", where all the options are
#           described.  Here are the more important ones.  No options are
#           usually needed, they are usually for fine-tuning the display.
#
#           -pass       Multipoint oligogenic pass number, "1" is default
#           -allpass    Plot all multipoint passes (in separate plots)
#           -title      Title of plot
#           -lod <lod>  Show LOD scale for lods this high (default is highest)
#           -lodmark    Put marker ticks ON TOP of LOD curve (default is axis)
#           -color      Takes standard names like "blue" and "red"
#           -noconv     Do not mark convergence errors
#           -date       Datestamp plot
#           -nomark     Do not show marker ticks (very useful for GWAS)
#           -font       X font for text (see xlsfonts | more)
#           -titlefont  X font for title only
#           -dash       Line style (see "help plot" for description of spec)
#           -linestyle  Line style (same as -dash)
#           -mibddir    specify mibddir (default is set with mibddir command)
#           -mapfile    User mapfile
#           -layers     Method of using multiple colors.  See help plot.
#
#           mibddir and trait (or outdir) must have been specified previously.
#
#           String plot graph will be both displayed on screen and written
#           to file.  If you are running on a remote system, you will
#           need to enable X window forwarding by setting DISPLAY variable
#           to point back to X display, and enabling acceptance of X
#           protocol with xhost + command, as described in section
#           3.8.3.1 of the SOLAR documentation.  Sorry, there is no possible
#           way to write the the file without displaying the plot, the
#           underlying "tk/wish" program does not allow that.
#
#           An encapsulated postscript file is written to the trait/outdir
#           with the name str.passN.ps where N is the pass number,
#           such as str.pass01.ps
#
#           If a copy of the string plot script, which is named
#           "stringplotk", is found in the current working directory, that
#           will be used in place of the standard version in the
#           SOLAR bin directory.  You can customize stringplotk as you
#           wish.  (It is a "wish" script, after all.)  Good luck!
# -

proc stringplot {args} {

# Check for outdir and mibddir

    full_filename foo
    mibddir

    set main_title ""
    if {[catch {set main_title [trait]}]} {
	catch {set main_title [lindex [outdir] 1]}
    }

# Capitalize first letter

    if {0<[string length $main_title]} {
	set main_title \
"[string toupper [string range $main_title 0 0]][string range $main_title 1 end]"
    }

# See if there is a copy of stringplotk in local directory

    set program stringplotk
    if {[file exists stringplotk]} {
        puts "\nUsing stringplotk in working directory...\n"
        set program ./stringplotk
    }


# Read arguments

    set no_such_title "!>/?*&%@^qawitmbp<zv239485721029751238941236743@"

    set pass 1
    set usertitle $no_such_title
    set layername ""
    set coloropt ""
    set moreargs [read_arglist $args -pass pass -allpass {set pass -1} \
	    -title usertitle -name layername -color coloropt -* foo]

# Tk intercepts -name so we need to use -layername

    if {"" != $layername} {
        set layername "-layername $layername"
    }

    if {"" != $coloropt} {
        set coloropt "-layercolor $coloropt"
    }

    if {$pass == -1} {
	set hightest 0
	for {set i 1} {[file exists [full_filename multipoint$i.out]]} \
		{incr i} {
	    set highest $i
	}
	for {set i $highest} {$i >= 1} {incr i -1} {
	    set infile [full_filename multipoint$i.out]
	    set outfile [full_filename pass[format %02d $i].str.ps]
	    if {$usertitle != $no_such_title} {
		set use_title $usertitle
	    } else {
		set use_title "$main_title  (Pass $i)"
	    }
	    eval exec $program -mibddir [mibddir] -in $infile \
		    -o $outfile -title \"$use_title\" $layername $coloropt \
                    $moreargs &
	    after 1000
	}
    } else {
	set infile [full_filename multipoint$pass.out]
	set outfile [full_filename pass[format %02d $pass].str.ps]
	if {$usertitle != $no_such_title} {
	    set use_title $usertitle
	} else {
	    set use_title "$main_title  (Pass $pass)"
	}
	eval exec $program -mibddir [mibddir] -in $infile \
		-o $outfile -title \"$use_title\" $layername $coloropt \
                $moreargs &
    }
    
    return ""
}


# solar::plot --
#
# Purpose:  Plot multipoint LOD scores, empirical LOD adjustments, or power
#
# Usage:    plot [<chromnum>] [-pass <passnum>] [-write]
#                     [-color <colornum>] [-overlay]
#                     [-title <title>] [-subtitle <subtitle>]
#                     [-yscale <maxy>] [-map <user_map>] [-lodadj]
#                     [-min x] [-max x] [-nomark] [-nomarklab]
#                     [-all | -allpass [-nodisplay] [-nomini]]
#                     [-string [-allpass] [-lod <lod>] [-lodmark] [-lodscale]
#                       [-color <colorname>] [-noconv] [-date] [-name <name>]
#                       [-font <X-font-spec>] [-titlefont <X-font-spec]
#                       [-layers {{<layername> [-color <colorname>]} ... }
#                       [-replay {{<layername> [-color <colorname>]} ... }
#                       [-title <title>] [-dash 1/2] [-linestyle 1/2]
#                     [-liability [-model <name>]]
#                     [-power [-title <plot_title>]]
#
#           plot -purge
#           plot -close
#
# Examples: plot                    plot chromosome with highest LOD in pass 1
#           plot 9                  plot chromosome 9 in pass 1
#           plot 9 -pass 2          plot chromosome 9 in pass 2
#           plot -all               plot all chromosomes in pass 1
#           plot -all -pass 2       plot all chromosomes in pass 2
#           plot -allpass           plot all chromosomes in all passes
#           plot -string            plot all chromosomes in pass 1 using 
#                                     "string" plot format
#           plot -string -allpass   plot all chromosomes in all passes using
#                                     "string" plot format
#
# If postscript output files are saved, they are written to the current
# trait or outdir directory with names like these:
#
#   chr01.ps              chromosome 1 (pass 1)
#   chr01.pass02.ps       chromosome 1 (pass 2)
#   pass01.ps             Miniplot of chromosomes in pass 1 (plot -all -pass 1)
#   pass01.str.ps         String plot of pass 1
#
#
#           chromnum  [1-29] Set chromosome number for plotting.  The default
#                     is to plot chromosome with highest LOD score.
#
#           -pass     Set multipoint pass number for plotting.  "1" would
#                     mean the first pass in which all models have one
#                     QTL.  1 is the default.
#
#           -close  Close the XMGR plot window.  The miniplot and string plot
#                   display windows must be closed with their close buttons,
#                   but it is better if you close XMGR from the SOLAR
#                   command line.  Otherwise, on your next plot, there will
#                   be a delay until SOLAR determines that it cannot
#                   communicate with the old XMGR session.  Then, it will
#                   time-out and tell you to use the "tclgr close" command,
#                   which does the same thing as "plot -close".
#
#           -write    Write postscript output file for plot(s).  If there are
#                     no other arguments and if a plot was done previously,
#                     the output file for the previous plot is written.  
#                     Miniplot and stringplot files are always written
#                     by default.  For plots drawn using XMGR, you can
#                     also choose to write the postscript file from the
#                     XMGR graphical interface, which give you more options.
#                     See note 8 below.
#
#           -nomark     Do not show ticks or labels for markers.  (This works
#                       for both regular and -string plots.)  Unless this
#                       option is selected, there must be a mibddir
#                       selection in the current directory so that SOLAR
#                       can find the map files.
#
#           -nomarklab  Do not show labels for markers (still show ticks).
#
#           -title      Set plot title.  Title may be blanked with
#                       -title "" or -title " ".  This is supported
#                       by regular plots, string plots, and power
#                       plots only.  Plots made through XMGR may also
#                       have title set through graphical interface or
#                       by editing .gr file such as multipoint.gr.
#
#           -subtitle   Set plot subtitle.  Supported by regular
#                       multipoint plots only.  Subtitle may be blanked
#                       with -subtitle "" or -subtitle " ".
#
#           -color  Use this color for curve (overrides multipoint.gr default)
#
#                   For regular plots, this must be integer from 1-15; 
#                   colors are defined by XMGR:
#
#                   0:White 1:Black 2:Red 3:Green 4:Blue 5:Yellow
#                   6:Brown 7:Gray 8:Violet 9:Cyan 10:Magenta 11:Orange
#                   12:Indigo 13:Maroon 14:Turquoise 15:Green4
#
#                   For string plots, the X11 color names are used.  Typical
#                   color names are:
#
#                   black white blue red green grey orange purple brown violet
#                   magenta yellow cyan
#
#                   Many mixtures and shades are also available.  Find the
#                   rgb.txt file in your X11 installation for a complete list.
#
#           -overlay  Plot this curve on top of the current graph, which may
#                     already include more than one curve.  (Each curve
#                     corresponds to a distinct XMGR set, of which 30 are
#                     available in the custom version of XMGR used by SOLAR.
#                     To control order of sets in Legend, use the -set
#                     option for every plot.)
#
#           -purge    Delete all previously created plotfiles (not valid with
#                     other options; only valid for multipoint plots).
#
#           -string     Plot all chromosomes (in pass 1 unless otherwise
#                       specified) using "string plot" format.  (An
#                       alternative page of plots in xmgr format can be
#                       produced by with plot -all command.)
#
#           -name       Name this plot for later use (-string plots only).
#
#           -layers <layerlist>  Add one or more previous plots to this plot.
#                       This is either a simple list of previous names, or a
#                       nested list of names with other options, in either case
#                       each element of <layerlist> specifies a single layer.
#                       See extended example below under replay.
#                      (-string plots only).
#
#           -replay <layerlist>  Draw previous plots only, otherwise this is
#                       the same as -layers.  (-string plots only) Example:
#
#     trait q1
#     plot -string -name A1
#     trait q2
#     plot -string -name A2 -layers {{A1 -color green}}
#     trait q3
#     plot -string -name A3 -layers {{A2 -color blue} {A1 -color green}}
#     plot -string -replay {{A3 -color grey} {A2 -color blue} {A1 -color red}}
#     plot -string -replay {A3 A2 A1}  ;# just default colors
#
#                  Note that spaces between close and open braces, as
#                  shown above, is required.
#
#                  You can specify -color for the new top level plot and/or
#                  for layers in the -layers or -replay list.  Any unspecified
#                  colors will default to a built-in set of defaults.
#
#           -lod lod    Add horizontal scales above this lodscore (for string
#                         plot only)
#
#           -noconv     Do not mark convergence errors (string plot only)
#
#           -date       Datestamp (string plot only)
#
#           -lodmark    Put marker ticks ON TOP of LOD curve (default is to the
#                         left of the plot axis)  String plot only.
#
#           -lodscale   Show the LOD scale when this LOD is exceeded (default
#                       is for the scale only to appear for the highest LOD).
#                       String plot only.
#
#           -font       (String plot only!) Specify X font to use for title.
#                       Use command "xlsfonts | more" to list X fonts.
#                       Wildcards may be used, however, results are
#                       sometimes unexpected.  For example *bold-r* will
#                       match first bold,roman font in list, whatever
#                       that happens to be.
#
#           -dash 5/5   (String plot only!) Specify line style dot and
#                        dash.  Two or more integers are specified separated
#                        by slash.  The first and all odd position numbers
#                        specify the length of drawn segments, whereas the
#                        second and all even position numbers specify
#                        the transparent segments.  Results are approximate
#                        depending on system used.
#
#           -linestyle 5/5  (String plot only!) Same as -dash (see above).
#
#                        Note that for regular plots, linestyle can be
#                        changed by editing the linestyle parameter in the
#                        applicable .gr file such as multipoint.gr.
#
#           -titlefont  (String plot only!) Same as -font, except applies
#                       to title only.  Supercedes -font for title only.
#
#           -all        Plot all chromosomes in xmgr postscript format (in
#                       the first pass only unless -allpass specified). A page
#                       of miniature chromosome plots in postscript is
#                       created (if a python interpreter is available).  The
#                       names of all postscript files are listed, and any
#                       of them may be printed with the lp command.  Nothing
#                       is displayed on your desktop with this command. An
#                       alternative genome plot is available with
#                       "plot -string".
#
#           -allpass    Plot all chromosomes in all passes, producing either
#                       miniplots or "string" plots (if -string).
#
#           -nodisplay  Used with -all or -allpass to skip displaying 
#                       miniplots on-screen (has no effect on xmgr graphs).
#
#           -nomini     Used with -all or -allpass to skip making miniplots.
#                       Automatically sets the "-write" option to write all
#                       individual chromosome plots.  Miniplots can always
#                       be made later, and with more options, using the 
#                       separate "miniplot" command.
#
#           -yscale  [NOTE: Ordinarily you do not need to use this.]
#                    This sets the smallest LOD scaling if there is no
#                    LOD above 4.99.  Autoscaling will not apply for smaller
#                    values to prevent confusion (e.g. seeing what looks
#                    like a large peak but isn't because the y scale is
#                    is so small).  The default value is 4.99.  You can
#                    set this to 0 if you really want to look at tiny LOD
#                    curves.  Larger scaling is applied automatically, as
#                    is the adjustment to allow space for marker labels.
#
#           -map    Use user_map file (in user map format).  By default,
#                   the map information processed by the 'load map' command
#                   (and saved in the mibddir) is used to display marker
#                   labels and locations.  However, you can substitute a user
#                   map file (set file-map) by using this argument.  This 
#                   allows you to delete unimportant markers and add QTL's of
#                   particular interest.
#                
#           -min m  location at which to start plotting
#           -max m  location at which to end plotting
#                   (min and/or max may be used to restrict interval.
#                    These apply to ordinary chromosome plots only.)
#
#           -quick  Save time when plotting by re-using marker ticks and
#                   labels from previous plot.  This option is used
#                   automatically when you are plotting from the multipoint
#                   command using the "-plot" argument.
#
#           -lodadj  Plot empirical LOD adjustment scores.  None of the above
#                    arguments except -close and -color are applicable in
#                    this case.  The format file lodadj.gr is used instead
#                    of multipoint.gr, but the rules are applied in the
#                    same way (see notes below).
#
#           -liability   Plot discrete trait liability function (a different
#                        kind of plot, not related to above).  "polygenic"
#                        command must have been run first, and the following
#                        covariates must have been included:
#
#                          sex age age*sex age^2 age^2*sex
#
#                        The xmgr parameter file for -liability is liability.gr
#
#           -model name  Specify a different modelname for the -liability
#                        option.  There must be a companion maximization
#                        output file (maximize -output name.out) so, for
#                        example, there is name.mod and name.out.  The
#                        default is poly (poly.mod and poly.out are created
#                        by the polygenic command).
#
#           -power  Plot power versus QTL heritability.  Only the -title
#                   argument is applicable in this case.  The format file
#                   power.gr is used instead of multipoint.gr.
#
# Notes:
#
#          1.  The trait or outdir must have previously been specified so
#              the applicable multipoint file can be found.
#
#          2.  Marker labels and ticks are taken from the mibdchrN.loc file
#              (where N is the chromosome number) created from the user's
#              map file during mibd file creation.  These files should be
#              stored in the mibddir (and the mibddir should be specified
#              before running plot).  If the mibdchrN.loc file(s) cannot
#              be found, marker ticks and labels will not be shown.  In
#              SOLAR releases 1.1.2-1.2.0 the 'load map' command will create
#              a mibdchrN.loc file in the current directory.
#
#              There will be a tick for each marker, and up to 42 marker
#              labels will be displayed.  If there are more than 42
#              markers, some labels will not be displayed.  Labels are
#              prioritized based on LOD score and distance from nonzero
#              value.  By plotting after the multipoint session has
#              completed, one gets the best benefit from this label
#              prioritization.  Marker ticks are always drawn vertically;
#              add additional line (which might be diagonal) joins the
#              label to its tick.
#
#              You can eliminate the need for the map file by using the
#              -nomark option.
#
#          3.  XMGR (ACE/gr) is used for most plotting, using tclgr command.
#              Each SOLAR process can have only one tclgr session open.  You
#              can change the plot command used with the 'tclgr syscommand'
#              command (it must be XMGR pipe command compatible).
#              If SOLAR is exited before closing the plot session, the plot
#              session may remain active (however, it may print a warning
#              about not being able to access the named pipe).  If the user
#              terminates the XMGR session through its graphical interface, 
#              the command 'plot -close' must be given to reset it before
#              another plot command can be given.
#
#           4. The XMGR parameter setup file multipoint.gr is loaded.
#              First, the multipoint.gr from SOLAR_LIB is loaded, then the
#              multipoint.gr from ~/lib (if any), then the multipoint.gr
#              from the working directory (if any).  You need only include
#              the parameters you want to change in your local copy.
#
#           5. When SOLAR exits, the XMGR session will be terminated.  If
#              the automatic termination of XMGR should fail, the user
#              should terminate XMGR anyway to prevent it from hogging CPU.
#              (The custom XMGR in SOLAR_BIN prevents CPU hogging.)
#
#           6. NaN's are indicated by X's on the curve.  Areas of the curve
#              in between multiple X's may be invalid.  (NaN's are Not A Number
#              which means maximization failed to arrive at a valid likelihood
#              estimate.
#
#           7. There are two additional options, -set and -graph, whose usage
#              is discouraged except under exceptional circumstances.  They
#              might force the set and graph numbers to specific values.
#              By default, the set number is 1 (changed in version 1.6.0)
#              except for overlays.  Overlays use the first available set
#              number counting backwards from 29.  The graph number (except
#              for overlays) is the same as the set number (overlays must use
#              the preceding graph number).  Fooling with these can get you
#              into trouble, but under difficult circumstances they might
#              help.
#
#          8.  Standard postscript landscape mode is used in all output files.
#              If you want to choose any other output features, such as
#              Encapsulated Postscript (EPS), portrait mode, etc., for
#              those plots made by XMGR, you can open the "Printer Setup"
#              dialog (under the "File" menu).  There you can select
#              portrait output in a pulldown menu, check a "Generate EPS"
#              check box, etc.  Then, to write the file, select the
#              "File" option in the "Print to:" pulldown, and then press
#              the "Print" button at the bottom of the dialog box.  You
#              need not go to the separate "Print" option in the file menu,
#              and sometimes it seems to work better to print directly
#              from the Printer Setup dialog anyway.  All postscript files
#              can be printed using "lp" command.  Displaying postscript
#              or editing on screen depends on locally available software.
# -

# Obsolescent command name plotmulti handled through make_solar_aliases

proc plot {args} {

# -liability is special

    if {-1 != [lsearch -exact $args -liability]} {
	set largs [read_arglist $args -liability {set foo 1} -* foo]
	return [eval plot_liability $largs]
    }

# -close is special

    if {-1 != [lsearch -exact $args -close]} {
	if {[string compare -close $args]} {
	    error "-close is not valid with other arguments"
	}
	tclgr close
	return ""
    }

# -lodadj is special

    if {-1 != [lsearch -exact $args -lodadj]} {
	return [eval plot_lodadj $args]
    }

# -power is special

    if {-1 != [lsearch -exact $args -power]} {
	return [eval plot_power $args]
    }

# -h2power is special

    if {-1 != [lsearch -exact $args -h2power]} {
	return [eval plot_h2power $args]
    }


# -purge is special

    if {-1 != [lsearch -exact $args -purge]} {
	if {[string compare -purge $args]} {
	    error "-purge is not valid with other arguments"
	}
	full_filename foo
	catch {eval delete_files_forcibly [eval glob [full_filename chr*.ps \
		pass??.ps pass??.str.ps]]}
	return ""
    }

# -write is special if by itself and if a plot was done previously

    if {![string compare -write $args] && \
	    [if_global_exists Solar_Plot_Last_Chrom]} {
	return [plotwrite]
    }

# If "-string", this is a string plot

    if {-1 != [lsearch -exact $args -string]} {
	set args [read_arglist $args -string {set foo 1} -all {set foo 1} \
		-* foo]
	eval stringplot $args
	return ""
    }

# If -all or -allpass, this is a plot_all* of some kind

    if {-1 != [lsearch -exact $args -all] || \
	    -1 != [lsearch -exact $args -allpass]} {

# Remove -all or -allpass from arg list; also check for nomini and nodisplay

	set mini 1
	set display ""
        set pass 1
	set saveplots 0
	set args [read_arglist $args -all {set foo 1} -allpass {set pass -1} \
		-nodisplay {set display -nodisplay} \
		-write {set saveplots 1} \
		-nomini {set mini 0; set saveplots 1} -* foo]
	read_arglist $args -pass pass -* foo

# -allpass

	if {$pass == -1} {

# Delete earlier files

	    full_filename foo
	    if {$mini} {
		catch {eval delete_files_forcibly [eval glob [full_filename pass??.ps \
			chr*.ps]]}
	    } else {
		catch {eval delete_files_forcibly [eval glob [full_filename \
			chr*.ps]]}
	    }
# Plot
	    eval plot_all_pass $args
	    if {$mini} {
		eval miniplot -allpass $display
	    }

# Delete output files, unless -write requested

	    if {!$saveplots} {
		ifdebug puts "deleting all chr*.ps files"
		catch {eval delete_files_forcibly [eval glob [full_filename chr*.ps]]}
	    }
	} else {

# -all

# Delete earlier files

	    full_filename foo
	    if {$mini} {
		set oname [format "pass%02d.ps" $pass]
		delete_files_forcibly [full_filename $oname]
	    }
	    if {$pass == 1} {
		set testonames [glob -nocomplain [full_filename chr*.ps]]
		set onames {}
		foreach testoname $testonames {
		    if {-1 == [string first pass $testoname]} {
			lappend onames $testoname
		    }
		}
	    } else {
		set onames [glob -nocomplain [full_filename \
			[format "chr*.pass%02d.ps" $pass]]]
	    }
	    if {{} != $onames} {
		catch {eval delete_files_forcibly $onames}
	    }
# Plot
	    eval plot_all $args
	    if {$mini} {
		eval miniplot -pass $pass $display
	    }

# OBSOLETE: Delete output files unless -write requested

	    if {0} {
		if {$pass == 1} {
		    set tfilenames [glob -nocomplain [full_filename chr*.ps]]
		    set dfilenames {}
		    foreach tfilename $tfilenames {
			if {-1 == [string first pass $tfilename]} {
			    lappend dfilenames $tfilename
			}
		    }
		} else {
		    set dfilenames [eval glob -nocomplain [full_filename \
			    "chr*.pass[format %02d $pass].ps"]]
		}
		ifdebug puts "Deleting files: $dfilenames"
		if {{} != $dfilenames} {
		    catch {eval delete_files_forcibly $dfilenames}
		}
	    }
	}
	return ""
    }

# ensure trait/outdir has been specified
    full_filename test_trait_or_outdir

# set defaults and parse arguments (-1 will mean not specified)
    set write_flag 0
    set user_map ""
    set color -1
    set min_y_range 4.99
    set quick 0
    set num_points 0
    set passnum 1
    set chromnum -1
    set setnum -1      
    set graphnum -1
    set overlay 0
    set marker_ticks {}
    set marker_labels {}
    set max_locnum -1.0
    set max_olod 0
    set max_mrk_location 0
    set user_min 0
    set user_max 0
    set no_markers 0
    set no_marker_labels 0
    set no_such_title "!>/?*&%@^qawitmbp<zv239485721029751238941236743@"
    set user_title $no_such_title
    set user_subtitle $no_such_title

    set max_mrk_name_len  \
	    [use_global_if_defined Solar_Plot_Last_Max_Chars 0]
    set last_chrom_plotted \
	    [use_global_if_defined Solar_Plot_Last_Chrom -1]
    set last_max_loc \
	    [use_global_if_defined Solar_Plot_Last_Max_Loc -1]

    set moreargs [read_arglist $args \
	    -pass passnum \
	    -chrom chromnum \
	    -set setnum \
	    -graph graphnum \
	    -overlay {set overlay 1} \
	    -quick {set quick 1} \
	    -yscale min_y_range \
	    -color color \
	    -write {set write_flag 1} \
	    -min user_min \
	    -max user_max \
	    -nomark {set no_markers 1} \
	    -nomarker {set no_markers 1} \
	    -nomarkers {set no_markers 1} \
	    -nomarklab {set no_marker_labels 1} \
	    -title user_title \
            -subtitle user_subtitle \
	    -map user_map]
    
    if {$moreargs != ""} {
	if {1 != [llength $moreargs]} {
	    error "Invalid arguments or more than one chromosome: $moreargs"
	}
	set chromnum $moreargs
    }
    ensure_integer $passnum
    ensure_integer $setnum
    ensure_integer $graphnum
    ensure_integer $color
    ensure_float $min_y_range
    ensure_float $user_min
    ensure_float $user_max

# Ensure mibddir available if required

    if {!$no_markers && 0==[string length $user_map]} {
	mibddir
    }

# Determine if modified xmgr is available

    set modified_xmgr 0
    set max_ticks_modified_xmgr 1000
    set u_name [string tolower [exec uname]]
    if {![string compare $u_name sunos] || \
	    ![string compare $u_name osf1] || \
	    [string match *linux* $u_name] || \
	    [string match *irix* $u_name]} {
	set modified_xmgr 1
    }

    if {$user_min < 0 || $user_max < 0} {
	error "Negative -min or -max not possible"
    }
    if {$user_min != 0} {
	if {$user_max != 0} {
	    if {$user_min >= $user_max} {
		error "-min must be less than -max"
	    }
	}
    }

    if {0.0001 > $min_y_range} {set min_y_range 0.0001}

    if {$color != -1} {
	if {$color < 1 || $color > 15} {
	    error "Color must be 1..15"
	}
    }

    if {$passnum < 0} {
	error "Invalid pass number"
    }

# Choose setnum and graphnum:
#   If this is an overlay, choose an unused set
#     starting from end (29) going backwards to not interfere with chromosomes
#     unless very daring user is using -set command to set set
#   If this is not an overlay, we default setnum to 1 and graphnum to setnum

    if {$overlay} {
	set sets_in_use \
		[use_global_if_defined Solar_Plot_Sets_In_Use {}]
	if {$setnum == -1} {
	    for {set i 29} {$i > 0} {set i [expr $i - 1]} {
		if {-1==[lsearch $sets_in_use $i]} {
		    set setnum $i
		    break
		}
	    }
	    if {$setnum == -1} {
		error "All 30 possible sets are in use; overlay not possible"
	    }
	} elseif {$setnum < 1 || $setnum > 29} {
	    error \
		    "Invalid setnum chosen:  1-29 allowed"
	}
	set graphnum [use_global_if_defined Solar_Plot_Last_Graphnum -1]
	if {$graphnum == -1} {
	    error "Overlay adds to an existing plot; apparently there is none"
	}
    } else {

# No overlay, set set number to chromosome number
#   unless very daring user is using -set command

	set sets_in_use {}
	if {$setnum == -1} {
	    set setnum 1
	} elseif {$setnum < 1 || $setnum > 29} {
	    error "Invalid setnum chosen:  1-29 allowed"
	}
    }
    if {$graphnum == -1} {
	set graphnum $setnum
    }

#
# Read multipointN.out file
#
    set nans {}
    set all_points {}
    set notkilled 1
    set infilename [full_filename multipoint$passnum.out]
    set infile [open $infilename r]
    gets $infile line
    set ylabel [lindex $line 1]
    gets $infile 


    if {$chromnum == -1} {

# We need to scan file for chromosome with highest LOD score first
#   to set defaulted chromosome number

	set highest_lod -1
	set chrom_with_highest_lod -1
	while {-1 != [gets $infile line]} {
	    if {5 != [scan $line "%s %s %s %d %s" cid chrom lid locnum olod]} {
		error "Invalid line in multipoint$passnum.out: \n$line"
	    }
	    if {[string compare $cid "chrom"] || \
		    [string compare $lid "loc"]} {
		error "Invalid line in multipoint$passnum.out: \n$line"
	    }
	    if {![is_nan $olod]} {
		if {$olod > $highest_lod} {
		    set highest_lod $olod
		    set chrom_with_highest_lod $chrom
		}
	    }
	}
	set chromnum $chrom_with_highest_lod
	seek $infile 0 start
	gets $infile
	gets $infile
    }

# Remove leading zero from chromnum

    if {![string compare "0" [string index $chromnum 0]]} {
	set chromnum [string range $chromnum 1 end]
    }

# Now we can read the file for real

    set sort_points 0
    set highest_chromosome 0
    while {-1 != [gets $infile line]} {
	if {5 != [scan $line "%s %s %s %d %s" cid chrom lid locnum olod]} {
	    error "Invalid line in multipoint$passnum.out: \n$line"
	}
	if {![string compare "0" [string index $chrom 0]]} {
	    set chrom [string range $chrom 1 end]
	}
	if {[string compare $cid "chrom"] || \
		[string compare $lid "loc"]} {
	    error "Invalid line in multipoint$passnum.out: \n$line"
	}
	if {$chrom > $highest_chromosome} {
	    set highest_chromosome $chrom
	}
	if {[string compare $chromnum $chrom]} {
	    continue
	}
	if {[is_nan $olod]} {
	    lappend nans  $locnum
	} else {
	    lappend all_points [list $locnum $olod]
	    incr num_points
	    if {$locnum > $max_locnum} {
		set max_locnum $locnum
	    } else {
		set sort_points 1
	    }
	    if {$olod > $max_olod} {set max_olod $olod}
	}
    }
    close $infile

    if {$num_points < 1} {
	error "Chromosome $chromnum not found in $infilename"
    }

    set num_nans [llength $nans]
    global Solar_Plot_If_Nans
    if {![if_global_exists Solar_Plot_If_Nans]} {
	set Solar_Plot_If_Nans 0
    }
    if {$num_nans > 0 || \
	    ($overlay && $Solar_Plot_If_Nans)} {
	set if_nans 1
    } else {
	set if_nans 0
    }

# Now, we've read all the points
#   we also now know the selected or defaulted setnum and graphnum
#   so we can see now if graph can be reused
#   puts "setnum: $setnum  graphnum: $graphnum  chromnum: $chromnum"

    set reuse_graph 0
    if {$quick && ![string compare $last_chrom_plotted $chromnum] && \
	    $max_locnum <= $last_max_loc} {
	set reuse_graph 1
	set max_locnum $last_max_loc
    }
    if {$overlay && [string compare $last_chrom_plotted $chromnum]} {
	error "Cannot overlay a different chromosome"
    }

    if {$overlay} {
	set test_max $max_locnum
	if {$user_max != 0} {
	    set test_max $user_max
	}
	if {$last_max_loc < $test_max} {
	    error "Cannot overlay when previous graph had smaller width (maximum chrom position)\nTry putting widest graph first"
	}
    }

# Read the map file
    
    if {!$no_markers} {
	set sort_markers 0
	set last_location -100.0
	if {!$reuse_graph && !$overlay} {
	    set max_mrk_name_len 0
	    if {"" != $user_map} {
		set map_name $user_map
	    } else {
		set map_name1 [mibddir]/mibdchr$chromnum.loc
		set map_name2 [mibddir]/mibdchr0$chromnum.loc
		set exists1 [file exists $map_name1]
		set exists2 [file exists $map_name2]
		if {$exists1 && $exists2} {
		    puts \
                  "For chromosome $chromlabel found both $exists1 and $exists2"
		    break
		}
		if {$exists2} {
		    set map_name $map_name2
		} else {
		    set map_name $map_name1
		}
	    }
	    if {![catch {set mapfile [open  $map_name r]}]}  {
		set old_format 0
		set instring [gets $mapfile]
		set word [string tolower [lindex $instring 0]]
		if {$word == "nloci"} {
		    gets $mapfile
		}

# Regardless of format (2 or 3 columns) marker name is first column, location
#   is last column.  Assume at least one space between columns.

		while {-1 != [gets $mapfile mapline]} {
		    set mrkname [lindex $mapline 0]
		    set mrk_location [lindex $mapline [expr \
			    [llength $mapline] -1]]
		    if {![is_float $mrk_location]} {
			error "Invalid line in map file: $mapline"
		    }

# Round marker location to nearest 0.000000001 cM
# (Actually, it is probably in 1/10 anyway, but we must be certain of
#  level of precision in order for marker moving algorithm to work)

		    set mrk_location [format "%.9f" $mrk_location]
		    if {$mrk_location < $last_location} {set sort_markers 1}
		    set last_location $mrk_location
		    lappend marker_ticks $mrk_location
		    lappend marker_labels \
			    [list $mrk_location $mrkname $mrk_location 0.0]

# marker_labels structure: name_pos, name, tick_pos, interpolated LOD

		    if {$mrk_location > $max_mrk_location} {
			set max_mrk_location $mrk_location
		    }
		    set mrk_name_len [string length $mrkname]
		    if {$mrk_name_len > $max_mrk_name_len} {
			set max_mrk_name_len $mrk_name_len
		    }
		}
		close $mapfile
	    } else {
		if {"" != $user_map} {
		    error "Error opening map file: $user_map"
		}
	    }
	}
    }

# Open new or existing tclgr session

    if {[catch {tclgr open} errmsg]} {
	if {[string compare $errmsg \
		"tclgr session already opened from this solar session"]} {
	    error $errmsg
	}
    }

# Kill previous set, and graph if not overlaying

    if {$overlay || $reuse_graph} {
	if {!$overlay} {
	    tclgr send kill s$setnum saveall
	    tclgr send kill s0 saveall
	}
	tclgr send focus g$graphnum
    } else {
	tclgr send kill graphs
	tclgr send clear line
	tclgr send clear string
	tclgr send focus g$graphnum
    }

# Calculate scaling for x and y

    set max_x_unscaled $max_locnum
    set max_mrk_location [expr ceil($max_mrk_location)]
    if {$max_mrk_location > $max_x_unscaled} {
	set max_x_unscaled $max_mrk_location
    }
    if {$max_x_unscaled < 5} {set max_x_unscaled 5}


# Apply user-specified range limits

    set min_x_unscaled 0.0
    set restricted_range 0
    if {$user_min > 0} {
	set min_x_unscaled $user_min
	set restricted_range 1
    }
    if {$user_max > 0} {
	set max_x_unscaled $user_max
	set restricted_range 1
    }
	
# min_x and max_x allow extra 5% margin on inside of plot "window" so
# curve is fully exposed

    set delta_x_unscaled [expr $max_x_unscaled - $min_x_unscaled]
    set max_x [expr $max_x_unscaled + (0.05 * $delta_x_unscaled)]
    set min_x [expr $min_x_unscaled - (0.05 * $delta_x_unscaled)]

# Create y expansion factor to permit showing markers at top
# Also to permit showing legend above graph but below markers
#   (Legend is shown only if there are nan's)

    set chars_that_fit_vertically 58.0  ;# using default font size
    set tick_len_in_chars 6.6           ;# This is padded to allow margin
    set legend_height_in_chars 5.0
    set legend_margin 1.5               ;# add below legend only
    if {$if_nans} {
	set allow_for_legend [expr $legend_height_in_chars+$legend_margin]
    } else {
	set allow_for_legend 0.0
    }
    set unavailable_length [expr $max_mrk_name_len + $tick_len_in_chars + \
	    $allow_for_legend]

    set y_expand [expr 1.0/(1.0-($unavailable_length / \
	    $chars_that_fit_vertically))]

    set max_y [expr $max_olod * $y_expand]
    if {$max_y < $min_y_range} {set max_y $min_y_range}

#  Legend y is calculated, but used only if we need it

    set size_of_frame 0.7
    set frame_top 0.85
    set use_tick_len $tick_len_in_chars
    if {$max_mrk_name_len < 1} {
	set use_tick_len [expr $tick_len_in_chars - 1.0]
    }
    set legend_y [expr $frame_top - (($max_mrk_name_len+$use_tick_len+1) \
	    * $size_of_frame / \
	    $chars_that_fit_vertically)]
    
    if {$overlay} {
	set last_max_y [use_global_if_defined Solar_Plot_Last_Max_Y 0]
	if {$max_y < $last_max_y} {set max_y $last_max_y}
    }

# y axis set anew whether reusing graph or not
#   Marker positioning depends on x axis, not y axis

    tclgr send world ymin 0
    tclgr send world ymax $max_y

# Note that neither max_x or max_y are allowed to go to zero
# This is required for later code to work

# Set up basic frame and then read multipoint.gr

    if {!$overlay && !$reuse_graph} {
	tclgr send world xmin $min_x
	tclgr send world xmax $max_x
	if {$user_title != $no_such_title} {
	    tclgr send title \"$user_title\"
	} else {
	    tclgr send title \"Chromosome $chromnum\"
	}
	if {$user_subtitle != $no_such_title} {
	    tclgr send subtitle \"$user_subtitle\"
	} else {
	    tclgr send subtitle \"SOLAR Multipoint Pass # $passnum\"
	}
	if {4.0 < $max_y && $max_y < 10.0} {
	    tclgr send yaxis label place spec
	    tclgr send yaxis label place 0.043,0
	}
	tclgr send yaxis label \"$ylabel\"
	tclgr send xaxis tick op bottom
	tclgr send xaxis ticklabel op bottom
	tclgr send xaxis ticklabel start type spec
	tclgr send xaxis ticklabel start $min_x_unscaled
	global env
	if {[file exists $env(SOLAR_LIB)/multipoint.gr]} {
	    set mpathname [glob $env(SOLAR_LIB)/multipoint.gr]
	    tclgr send read \"$mpathname\"
	}
	if {[file exists ~/lib/multipoint.gr]} {
	    set mpathname [glob ~/lib/multipoint.gr]
	    tclgr send read \"$mpathname\"
	} 
	if {[file exists multipoint.gr]} {
	    tclgr send read \"multipoint.gr\"
	} 
    }
    if {$color != -1} {	
	tclgr send s$setnum color $color
    }

# Sort points if required 

    proc sort_by_first_num {a b} {
	set a0 [lindex $a 0]
	set b0 [lindex $b 0]
	if {$b0 > $a0} {
	    return -1
	} elseif {$b0 < $a0} {
	    return 1
	}
	return 0
    }

    if {$sort_points} {
	set all_points [lsort -command sort_by_first_num $all_points]
    }

# Sort marker ticks and labels too

    if {!$no_markers && $sort_markers} {
	set marker_ticks [lsort $marker_ticks]
	set marker_labels [lsort -command sort_by_first_num $marker_labels]
    }

# Plot points

    foreach point $all_points {
	set x [lindex $point 0]
	set y [lindex $point 1]

	if {$x >= $min_x_unscaled && $x <= $max_x_unscaled} {
	    tclgr send g$graphnum.s$setnum point $x,$y
	    ifverbmax puts "graphing g$graphnum.s$setnum point $x,$y"
	} else {
	    ifdebug puts "Removing point $x,$y because outside range"
	}
    }

# Plot NaN's (if there are any) using set 0 with symbols and linestyle 0
#   (no lines connecting symbols)

# Assign a LOD to each 'new' NaN using interpolation
#   Note: NaN's more than one past last point get last value

    if {0 < $num_nans} {
	set nan_points {}
	set j 0
	set previously_went_beyond_points 0
	set previous_x 0.0
	set previous_y 0.0
	set point_x 0.0
	set point_y 0.0
	for {set i 0} {$i < $num_nans} {incr i} {
	    set found_point_after 0
	    if {!$previously_went_beyond_points} {
		set nan_x [lindex $nans $i]
		set nan_y 0.0
		for {} {$j < $num_points} {incr j} {
		    set point [lindex $all_points $j]
		    set point_x [lindex $point 0]
		    set point_y [lindex $point 1]
		    if {$point_x >= $nan_x} {
			set found_point_after 1
			break
		    }
		    set previous_x $point_x
		    set previous_y $point_y
		}
	    }
	    if {$previously_went_beyond_points} {
		set nan_y $point_y
	    } elseif {!$found_point_after} {
		set previously_went_beyond 1
		set nan_y $point_y
	    } elseif {$point_x == $nan_x} {
		set nan_y $point_y
	    } else {
		set slope [expr ($point_y - $previous_y) / \
			double($point_x - $previous_x)]
		set delta_x [expr $nan_x - $previous_x]
		set nan_y [expr $previous_y + $slope * $delta_x]
	    }
	    if {$nan_y < 0.0} {set nan_y 0.0}
	    lappend nan_points [list $nan_x $nan_y]
	}
	foreach point $nan_points {
	    set x [lindex $point 0]
	    set y [lindex $point 1]
	    
	    tclgr send g$graphnum.s0 point $x,$y
	    ifverbmax puts "graphing NAN g$graphnum.s0 point $x,$y"
	}
    }

# Position legend box for nans
    
    if {$if_nans} {
	tclgr send legend loctype view
	tclgr send legend X1 0.295
	tclgr send legend Y1 $legend_y
	tclgr send legend on
	tclgr send legend box on
    }

# Set up x and y tick marks (major and minor)

    if {$max_x > 499} {
	tclgr send xaxis tick major 100
	tclgr send xaxis tick minor 50
    } elseif {$max_x > 100} {
	tclgr send xaxis tick major 50
	tclgr send xaxis tick minor 10
    } else {
	tclgr send xaxis tick major 10
	tclgr send xaxis tick minor 5
    }

# The design maximum LOD is 2000
#   (highest tick range is 750-1500)
#   (I don't like limits, but this is way beyond anything I can imagine)
# maxy must be 4*10^N or 1*10^N
# the other values are related

    set maxy  1000
    set major 200
    set major_digit 2
    set minor 100

    set maxdiv 2.5
    set majdiv 2
    set mindiv 2

# Since we mess with global variable tcl_precision
# We must catch errors to be sure it gets restored

    global tcl_precision
    set save_tcl_precision $tcl_precision
    set tcl_precision 6
    set not_ok [catch {
    while {1} {
	if {$max_y > $maxy} {
	    tclgr send yaxis tick major $major
	    tclgr send yaxis tick minor $minor
	    break
	}
	set maxy [expr double($maxy) / $maxdiv]
	set major [expr double($major) / $majdiv]
	set minor [expr double($minor) / $mindiv]

	if {$major_digit == 2} {
	    set major_digit 1
	    set maxdiv 2
	    set majdiv 2
	    set mindiv 2
	} elseif {$major_digit == 1}  {
	    set major_digit 5
	    set maxdiv 2
	    set majdiv 2.5
	    set mindiv 2.5
	} else {
	    set major_digit 2
	    set maxdiv 2.5
	    set majdiv 2
	    set mindiv 2
	}
    }
    } caught_error]
    set tcl_precision $save_tcl_precision
    if {$not_ok} {
	error $caught_error
    }
#
#            ***     PLOT MARKERS    ***
#
    if {!$no_markers && !$reuse_graph && !$overlay} {
#
# Remove markers out-of-range
#
	if {$restricted_range} {
	    ifdebug puts "Restricting marker range"
	    set new_marker_labels {}
	    foreach marker $marker_labels {
		set location [lindex $marker 0]
		if {$location < $min_x_unscaled || \
			$location > $max_x_unscaled } {
		    ifdebug puts "    Removing marker at $location"
		    continue
		}
		lappend new_marker_labels $marker
	    }
	    set marker_labels $new_marker_labels
#
# Remove tics out-of-range also (2.1.5)
#
	    set new_marker_ticks ""
	    foreach marker $marker_ticks {
		set location [lindex $marker 0]
		if {$location < $min_x_unscaled || \
			$location > $max_x_unscaled } {
		    ifdebug puts "    Removing marker tick at $location"
		    continue
		}
		lappend new_marker_ticks $marker
	    }
	    set marker_ticks $new_marker_ticks
	}
#
#       puts "[llength $marker_labels] markers within range"
#
# See if we can handle this number of markers in any way
#
	set num_markers [llength $marker_labels]
	set max_markers 42
	if {$modified_xmgr && $num_markers> $max_ticks_modified_xmgr} {
	    set no_markers 1
	    puts "Warning. Cannot display more than $max_ticks_modified_xmgr marker ticks"
	}
    }
    if {!$no_markers && !$reuse_graph && !$overlay} {
#
# Remove marker labels down to maximum we can handle
#
	if {!$no_marker_labels && $num_markers > $max_markers} {
#
# Assign a LOD to each marker using interpolation
#   Note: markers more than one past last point get value 0
#   
	    set new_marker_labels {}
	    set j 0
	    set previously_went_beyond_points 0
	    set previous_x 0.0
	    set previous_y 0.0
	    set point_x 0.0
	    set point_y 0.0
	    for {set i 0} {$i < $num_markers} {incr i} {
		set found_point_after 0
		if {!$previously_went_beyond_points} {
		    set marker [lindex $marker_labels $i]
		    set marker_x [lindex $marker 2]
		    for {} {$j < $num_points} {incr j} {
			set point [lindex $all_points $j]
			set point_x [lindex $point 0]
			set point_y [lindex $point 1]
			if {$point_x >= $marker_x} {
			    set found_point_after 1
			    break
			}
			set previous_x $point_x
			set previous_y $point_y
		    }
		}
		if {$previously_went_beyond_points} {
		    set marker_y 0.0
		} elseif {!$found_point_after} {
		    set previously_went_beyond 1
		    set marker_y $point_y
		} elseif {$point_x == $marker_x} {
		    set marker_y $point_y
		} else {
		    set slope [expr ($point_y - $previous_y) / \
			    double($point_x - $previous_x)]
		    set delta_x [expr $marker_x - $previous_x]
		    set marker_y [expr $previous_y + $slope * $delta_x]
		}
		if {$marker_y < 0.0} {set marker_y 0.0}
		set new_marker [lreplace $marker 3 3 $marker_y]
		lappend new_marker_labels $new_marker
	    }
	    set marker_labels $new_marker_labels
#
# Set zero valued marker labels to negative based on distance to nearest
#   non-zero  (we pass from both directions to do this)
#
	    set distance_to_zero 0
	    for {set i 0} {$i < $num_markers} {incr i} {
		set marker [lindex $marker_labels $i]
		set value [lindex $marker 3]
		if {$value <= 0.0} {
		    if {$distance_to_zero > 0} {
			set marker [lreplace $marker 3 3 \
				[expr 0.0 - $distance_to_zero]]
		      set marker_labels [lreplace $marker_labels $i $i $marker]
		    }
		    incr distance_to_zero
		} else {
		    set distance_to_zero 0
		}
	    }
	    set distance_to_zero 0
	    for {set i [expr $num_markers-1]} {$i >= 0} {set i [expr $i - 1]} {
		set marker [lindex $marker_labels $i]
		set value [lindex $marker 3]
		if {$value <= 0.0} {
		    if {$distance_to_zero >= 0} {
			if {$distance_to_zero < 0.0 - $value} {
			    set marker [lreplace $marker 3 3 \
				    [expr 0.0 - $distance_to_zero]]
			    set marker_labels [lreplace $marker_labels $i $i \
				    $marker]
			}
		    }
		    incr distance_to_zero
		} else {
		    set distance_to_zero 0
		}
	    }

	    while {$num_markers > $max_markers} {
		set lowest_value 10000.0
		set lowest_index 1
		for {set i 1} {$i < $num_markers} {incr i} {
		    set marker [lindex $marker_labels $i]
		    set value [lindex $marker 3]
		    if {$value < $lowest_value} {
			set lowest_value $value
			set lowest_index $i
		    }
		}
		set marker_labels [lreplace $marker_labels \
			$lowest_index $lowest_index]
		set num_markers [expr $num_markers - 1]

		if {$num_markers > $max_markers} {
		    set lowest_value 10000.0
		    set lowest_index [expr $num_markers - 2]
		    for {set i 1} {$i < $num_markers} {incr i} {
			set marker [lindex $marker_labels \
				[expr $num_markers - (1+$i)]]
			set value [lindex $marker 3]
			if {$value < $lowest_value} {
			    set lowest_value $value
			    set lowest_index $i
			}
		    }
		    set lowest_index [expr $num_markers - (1+$lowest_index)]
		    set marker_labels [lreplace $marker_labels $lowest_index \
			    $lowest_index]
		    set num_markers [expr $num_markers - 1]
		}
	    }
	}
	if {!$no_marker_labels} {
#
# Move remaining marker labels around to fit
# This should only take 1.5 passes with current algorithm
#   (old algorithm repeated until no more moves)

	    set min_distance [format %.9f [expr 0.0241 * $delta_x_unscaled]]
	    set markers_moved_left 0
	    set markers_moved_right 0
	    set pass_count 0
	    set max_pass_count 1.5
	    while {$pass_count<1 || \
		    $markers_moved_left>0|| $markers_moved_right>0} {
		incr pass_count
		set markers_moved_left 0
		set markers_moved_right 0

# Scan left to right, moving "next_marker" right if necessary

		for {set i -1} {$i < $num_markers - 1} {incr i} {
		    if {$i == -1} {
			set nloc [format %.10f [expr $min_x_unscaled - $min_distance]]
		    } else {
			set nloc [lindex [lindex $marker_labels $i] 0]
		    }
		    set next_marker [lindex $marker_labels [expr $i + 1]]
		    set next_nloc [lindex $next_marker 0]
		    set distance [format %.10f [expr $next_nloc - $nloc]]
		    if {$distance < $min_distance} {

# Move next marker by 1/2 distance required if this is the first pass
#  (makes offsets more symmetrical)
# Move next marker by entire distance required if this is subsequent pass
# Must maintain rounding to 0.001 precision

			set move_distance [expr $min_distance - $distance]
			if {$pass_count == 1} {
			    set move_distance [expr $move_distance / 2]
			}
			set move_distance [format %.10f $move_distance]
			set new_position [expr $next_nloc + $move_distance]
			set new_position [format %.10f $new_position]
			set j [expr $i + 1]
			set next_marker \
				[lreplace $next_marker 0 0 $new_position]
			set marker_labels \
				[lreplace $marker_labels $j $j $next_marker]
			incr markers_moved_right
		    }
		}
#
# This has been proven to take no more than 1.5 passes
#
		if {$pass_count > $max_pass_count} {
		    break
		}
#
# Scan right to left, moving next_marker right if necessary
#
		for {set i $num_markers} {$i > 0} {set i [expr $i - 1]} {
		    if {$i == $num_markers} {
			set nloc [format %.10f [expr $max_x_unscaled + \
				$min_distance]]
		    } else {
			set nloc [lindex [lindex $marker_labels $i] 0]
		    }
		    set next_marker [lindex $marker_labels [expr $i - 1]]
		    set next_nloc [lindex $next_marker 0]
		    set distance [format %.10f [expr $nloc - $next_nloc]]
		    if {$distance < $min_distance} {
			set move_distance [expr $min_distance - $distance]
			set move_distance [format %.10f $move_distance]
			set new_position [expr $next_nloc - $move_distance]
			set new_position [format %.10f $new_position]
			set j [expr $i - 1]
			set next_marker \
				[lreplace $next_marker 0 0 $new_position]
			set marker_labels [lreplace $marker_labels $j $j \
				$next_marker]
			incr markers_moved_left
		    }
		}
	    }
        }


# Draw marker ticks

	proc view_cx {worldc} {
	    upvar min_x min_x
	    upvar max_x max_x
	    return [expr 0.15 + (0.7 * ($worldc-double($min_x)) / \
		    ($max_x-double($min_x)))]
	}

	set vertical_ticks 0

# Vertical ticks only available for Solaris and Alpha versions
#   requires a recompile of xmgr for linux...not done yet

	if {$modified_xmgr} {
	    set vertical_ticks 1
	    foreach mloc $marker_ticks {
		tclgr send with line
		tclgr send line loctype view
		tclgr send line [view_cx $mloc], 0.85, [view_cx $mloc], 0.8325
		tclgr send line def
	    }
        }

# Write marker labels (when they fit)

	if {!$no_marker_labels} {
	    set last_nloc -100
	    set num_markers [llength $marker_labels]
	    for {set i 0} {$i < $num_markers} {incr i} {
		set marker_label [lindex $marker_labels $i]
		set mname [lindex $marker_label 1]
		set mloc [lindex $marker_label 2]
		set nloc [lindex $marker_label 0]

# setup leader line for label 

		tclgr send with line
		tclgr send line loctype view
		if {$vertical_ticks} {
		   tclgr send line [view_cx $mloc],0.8325,[view_cx $nloc],0.815
		} else {
		   tclgr send line [view_cx $mloc],0.85,[view_cx $nloc],0.815
		}
		tclgr send line def
		
		# setup marker label
		
		tclgr send with string
		tclgr send string on
		tclgr send string loctype view
		tclgr send string g$graphnum
		tclgr send string [view_cx $nloc], 0.8090
		tclgr send string rot 270
		tclgr send string char size 0.60
		tclgr send string def \"$mname\"
		ifdebug puts "Marker $mname at $nloc mapped to [view_cx $nloc]"
	    }
	}
    }

# Done with markers

# DRAW !!!

    tclgr send redraw

# Save certain variables to permit overlays

    global Solar_Plot_Last_Chrom
    global Solar_Plot_Last_Max_Loc 
    global Solar_Plot_Last_Max_Chars
    global Solar_Plot_Sets_In_Use	
    global Solar_Plot_Last_Max_Y
    global Solar_Plot_Last_Graphnum
    global Solar_Plot_If_Nans
    global Solar_Plot_Pass	
    global Solar_Plot_Highest_Chromosome
	
    set Solar_Plot_Last_Chrom $chromnum
    set Solar_Plot_Last_Max_Loc $max_x_unscaled
    set Solar_Plot_Last_Max_Chars $max_mrk_name_len
    set Solar_Plot_Sets_In_Use [lappend sets_in_use $setnum]
    set Solar_Plot_Last_Max_Y $max_y	
    set Solar_Plot_Last_Graphnum $graphnum	
    set Solar_Plot_Nans $if_nans
    set Solar_Plot_Pass $passnum
    set Solar_Plot_Highest_Chromosome $highest_chromosome

    if {$write_flag} {
	plotwrite
    }

    return ""
}


proc tclgrd {args} {
    puts "tclgr $args"
    eval tclgr $args
}

proc plot_all_pass {args} {

# remove -allpass from arglist

    set args [read_arglist $args -allpass {set foo 1} -* foo]

# Do a plot_all for every pass

    for {set i 1} {1} {incr i} {
	if {[file exists [full_filename multipoint$i.out]]} {
	    eval plot_all $args -pass $i
	} else {
	    break
	}
    }
    return ""
}

# solar::miniplot --
#
# Purpose:  Arrange miniature plots on a single page
#
# Usage:    miniplot [-pass <pass>] [-allpass] [-plots <number>] 
#                    [-port] [-land]
#
#             See also "plot -all"
#
#           -pass       Do this pass number (default is 1)
#           -allpass    Do all passes, each on a separate page
#           -plots      Put this many plots on a page
#           -port       Portrait layout
#           -land       Landscape layout
#           -nodisplay  Generate postscript, but don't display
#
#          Output file named passN.out (pass01.out for pass 1) in trait/outdir
#          is created.  The trait or outdir must have been specified previously
#          and the plots must have been created previously (see usage for
#          example).
#
#          The individual chromosome plots should have been created previously
#          using the "plot" command.  In fact, "plot -all" or "plot -allpass"
#          will invoke miniplot automatically.
#
#          This requires that Python (1.5.2 and later works, maybe earlier)
#          be installed.  If you do not have python, use "plot -string"
#          instead.
# -

proc miniplot {args} {

# Read arguments

    set display 1
    set pass 1
    set allpass 0
    set portland ""
    set plots -1
    set lessverbose 0
    set moreargs [read_arglist $args -pass pass -allpass {set allpass 1} \
	    -* foo]
    set badargs [read_arglist $moreargs \
	    -plots plots -port {set portland -P} -land {set portland -L} \
		     -lessverbose {set lessverbose 1} \
	    -nodisplay {set display 0}]
    if {"" != $badargs} {
	error "arrange: invalid arguments: $badargs"
    }

# Translate args to foobar getopt style required by arrangeps.py

    if {$plots == -1} {
	set plots ""
    } else {
	if {![is_integer $plots]} {
	    error "-plots must be followed by integer number of plots per page"
	}
	set plots "-p $plots "
    }
    set foobargs "$plots$portland"

# If -allpass, loop and recurse

    if {$allpass} {
	for {set i 1} {[file exists [full_filename multipoint$i.out]]} \
		{incr i} {
	    eval miniplot -pass $i $moreargs -nodisplay -lessverbose
	}
#	if {$display} {
#	    for {set j [expr $i - 1]} {$j > 0} {incr j -1} {
#		after 2000
#		exec pageview -right [full_filename pass[format %02d $j].ps] &
#	    }
#	}
    } else {
	if {$pass == 1} {
	    set pass 01
	    after 2000
	    set testfilenames [glob [full_filename chr*.ps]]
	    set filenames {}
	    foreach tname $testfilenames {
		if {-1 == [string first pass $tname]} {
		    lappend filenames $tname
		}
	    }
	} else {
	    set pass [format %02d $pass]
	    after 2000
	    set filenames [glob [full_filename chr*.pass$pass.ps]]
	}
	set filenames [lsort $filenames]
	puts "\nChromosome plot filenames for pass $pass are:\n$filenames"
	if {[llength $filenames] < 2} {
	    if {!$lessverbose} {
		puts "Only one chromosome plot is available,"
		puts "  so no page of miniplots will be produced."
	    }
	} else {
	    eval exec arrangeps.py $foobargs -o \
		[full_filename pass$pass.ps] $filenames >/dev/null
#	    if {$display} {
#	        set displayed 0
#
# After all, ghostscript does a useless job of rendering apparently due to
# lousy fonts so this section is currently inactive.
#
#	        catch {
#		    exec gs [full_filename pass$pass.ps] 2>/dev/null &
#		    set displayed 1
#	        }
#	        if {!$displayed} {
#		    puts "gs (ghostscript) not found to display miniplots"
#		    puts "Add gs directory to PATH if available on this system"
#	        }
#	    }
	    puts "\nMiniplots for pass $pass written to file [full_filename pass$pass.ps]"
	}

    }
    if {!$lessverbose} {
	puts "\nNote: plot -all does not display anything on your screen"
	puts "But these postscript files may be sent to printer with lp command"
	puts "An alternative genome plot can be produced with plot -string"
    }
    return ""
}

proc plot_all {args} {

    ifdebug puts "Entering plot_all"

# remove -all from arglist

    set pass 1
    set args [read_arglist $args -all {set foo 1} -pass pass -* foo]

    if {$pass != 1} {
	set args "$args -pass $pass"
    }

# Make list of all chromosomes for this pass

    set chrom_list {}
    set last_chrom {}
    set infilename [full_filename multipoint$pass.out]
    set infile [open $infilename r]
    gets $infile
    gets $infile
    while {-1 != [gets $infile line]} {
	if {2 == [scan $line "%s %s" cid chrom]} {
	    if {[is_integer $chrom]} {   ;# remove leading zero if any
		scan $chrom %d chrom
	    }
	    if {[string compare $chrom $last_chrom]} {
		lappend chrom_list $chrom
		set last_chrom $chrom
	    }
	}
    }
    close $infile

# Loop over all chromosomes, plot, and write files

    foreach chrom $chrom_list {
	ifdebug puts "Plotting chromosome $chrom with $args"
	eval plot $chrom $args
	ifdebug puts "Calling plotwrite"
	plotwrite
    }
    plot -close    ;# Ensure output files have been written
    return ""
}

proc plotwrite {args} {

# Ensure last chrom variable is present; if not, no plot had been done

    ifdebug puts "Starting plotwrite"

    global Solar_Plot_Last_Chrom
    global Solar_Plot_Pass
    if {![if_global_exists Solar_Plot_Last_Chrom]} {
	error "Must create plot first, or use -allplot"
    }

# Set default output filename

    if {[catch {set lchrom [format %02d $Solar_Plot_Last_Chrom]}]} {
	set lchrom $Solar_Plot_Last_Chrom
	if {[is_integer [string range $lchrom 0 0]]} {
	    if {![is_integer [string range $lchrom 1 1]]} {
		set lchrom 0$lchrom
	    }
	}
    }
    set ppass [format %02d $Solar_Plot_Pass]
    set filename [full_filename chr$lchrom.ps]
    if {$ppass != 1} {
	set filename [full_filename chr$lchrom.pass$ppass.ps]
    }

    ifdebug puts "plotwrite filename: $filename"

# Only optional argument allowed is "-filename"

    set badargs [read_arglist $args -write {set write 1} -filename filename]
    if {"" != $badargs} {
	error "plotwrite does not allow arguments: $badargs"
    }

# Send commands to xmgr

    tclgr send print to psmonol
    tclgr send print to file \"$filename\"
    tclgr send hardcopy
    return ""
}

proc plot_lodadj {args} {
    full_filename test_trait_or_outdir

    set color -1
    read_arglist $args -color color -lodadj {set ignore_this 0}
    if {$color != -1} {
	if {$color < 1 || $color > 15} {
	    error "Color must be 1..15"
	}
    }

    if {[catch {tclgr open} errmsg]} {
	if {[string compare $errmsg \
		"tclgr session already opened from this solar session"]} {
	    error $errmsg
	}
    }
    set all_points {}
    set last_point "0 0"
    set num_points 0
    set infile [open [full_filename lodadj.lods]]
    while {-1 != [gets $infile line]} {
	lappend all_points $line
	set last_point $line
	incr num_points
    }
    close $infile

    set last_x [lindex $last_point 0]
    if {$last_x == 0} {
	error "Last point estimated is 0,0...can't plot"
    }

    tclgr send kill graphs
    tclgr send clear line
    tclgr send clear string
    tclgr send focus g0

    tclgr send world ymin 0
    tclgr send world ymax $last_x
    tclgr send world xmin 0
    tclgr send world xmax $last_x

# Get number of reps from lodadj.info

    catch {
	set infile [open [full_filename lodadj.info] r]
	gets $infile line
	set nreps [lindex $line 3]
	close $infile
	tclgr send subtitle \"Repetitions:  $nreps\"
    }

    global env
    if {[file exists $env(SOLAR_LIB)/lodadj.gr]} {
	set mpathname [glob $env(SOLAR_LIB)/lodadj.gr]
	tclgr send read \"$mpathname\"
    }
    if {[file exists ~/lib/lodadj.gr]} {
	set mpathname [glob ~/lib/lodadj.gr]
	tclgr send read \"$mpathname\"
    } 
    if {[file exists lodadj.gr]} {
	tclgr send read \"lodadj.gr\"
    } 
    if {$color != -1} {	
	tclgr send s0 color $color
    }
    foreach point $all_points {
	tclgr send g0.s0 point "[lindex $point 0],[lindex $point 1]"
    }

    tclgr send redraw
}


# solar::plot_liability -- private
#
# Purpose:  Implements "plot -liability"
#
# Usage:    plot_liability [-model <modelname>]
#
#           Default (no arguments) assumes "polygenic" command has just been
#           given; uses poly.mod and poly.out in current trait/outdir.
#
#           -model modelname  Use this model.  Maximization output file
#                             must have been saved as *.out where *.mod
#                             is modelname.
# -

proc plot_liability {args} {

# Get arguments

    set modelname poly
    set badargs [read_arglist $args -model modelname]
    if {"" != $badargs} {
	error "Invalid -liability arguments: $badargs"
    }

# Setup modname and outputname

    set usemodel [append_mod $modelname]
    set useoutput [append_extension [string range $usemodel 0 \
	    [expr [string length $usemodel] - 5]] .out]
    set modname [full_filename $usemodel]
    set outputname [full_filename $useoutput]


# See if model and maximization output files exist

    if {"poly" == $modelname && ![file exists $modname]} {
	error "Missing poly.mod.  Run polygenic command first, or specify -model"
    }
    if {"poly" == $modelname && ![file exists $outputname]} {
	error "Missing poly.out.  Run polygenic command first, or specify -model"
    }

    if {![file exists $modname]} {
	error "Model $modname not found"
    }

    if {![file exists $outputname]} {
	error "Can't find maximization output file $outputname"
    }

# Get maximized parameter values

    set noage 0
    set nosex 0
    set tailname [file tail $modname]
    if {[catch {set bsex [oldmodel $tailname bsex]}]} {
	set bsex 0.0
	incr nosex
    }
    if {[catch {set bage [oldmodel $tailname bage]}]} {
	set bage 0.0
	incr noage
    }
    if {[catch {set bagesex [oldmodel $tailname bage*sex]}]} {
	set bagesex 0.0
	incr noage
	incr nosex
    }
    if {[catch {set bage2 [oldmodel $tailname bage^2]}]} {
	set bage2 0.0
	incr noage
    }
    if {[catch {set bage2sex [oldmodel $tailname bage^2*sex]}]} {
	set bage2sex 0.0
	incr noage
	incr nosex
    }
    set mu [oldmodel $tailname mean]

# Determine whether we found any age or sex covariates

    set found_age 0
    set found_sex 0
    if {$noage < 4} {
	set found_age 1
    }
    if {$nosex < 3} {
	set found_sex 1
    }

# Get mean, min, max age

    if {![catch {set meanage [getvar -mean $useoutput age]}]} {
	set minage [getvar -min $useoutput age]
	set maxage [getvar -max $useoutput age]
    } else {
	if {$found_age} {
	    error "Outdated maximization output file missing age variable"
	}
	set meanage 50.0
	set minage 0.0
	set maxage 100.0
    }

# Determine proper plotting range for age

    set minx [expr round (floor ($minage / 5.0) * 5)]
    set maxx [expr round (ceil ($maxage / 5.0) * 5)]


# Keep min,max for middle range (35% - 65%) so that position of
#   legend may be determined

    set middle_min 1.0
    set middle_max 0.0
    set begin_middle [expr ($maxx - $minx)*0.35 + $minx]
    set end_middle [expr $maxx - ($maxx - $minx)*0.35]

# Create MALE curve (leave out "*sex" terms since male=0)

    set male_curve {}
    for {set x $minx} {$x <= $maxx} {incr x} {
	set z [expr $mu \
		+ $bage * ($x - $meanage) \
		+ $bage2 * (($x - $meanage)*($x - $meanage))]

	set liability [alnorm $z t]


	lappend male_curve [list $x $liability]

	if {$x >= $begin_middle && $x <= $end_middle} {
	    if {$liability > $middle_max} {
		set middle_max $liability
	    }
	    if {$liability < $middle_min} {
		set middle_min $liability
	    }
	}
    }

# Create FEMALE curve (include all terms, female=1)

    if {$found_sex} {
	set female_curve {}
	for {set x $minx} {$x <= $maxx} {incr x} {
	    set z [expr $mu \
		    + $bage * ($x - $meanage) \
		    + $bage2 * (($x - $meanage) * ($x - $meanage)) \
		    + $bsex \
		    + $bagesex * ($x - $meanage) \
		    + $bage2sex * (($x - $meanage) * ($x - $meanage))] 
	    set liability [alnorm $z t]
	    lappend female_curve [list $x $liability]

	    if {$x >= $begin_middle && $x <= $end_middle} {
		if {$liability > $middle_max} {
		    set middle_max $liability
		}
		if {$liability < $middle_min} {
		    set middle_min $liability
		}
	    }
	}
    }

# Open new or existing tclgr session and clear out

    if {[catch {tclgr open} errmsg]} {
	if {[string compare $errmsg \
		"tclgr session already opened from this solar session"]} {
	    error $errmsg
	}
    }
    tclgr send kill graphs
    tclgr send clear line
    tclgr send clear string
    tclgr send focus g0
    tclgr send flush

# Setup default xmgr parameters

    tclgr send world ymin 0
    tclgr send world ymax 1.1

    set graph_x_min [expr $minx - 5]
    set graph_x_max [expr $maxx + 5]
    tclgr send world xmin $graph_x_min
    tclgr send world xmax $graph_x_max

    set captraitname \
      "[string toupper [string range [trait] 0 0]][string range [trait] 1 end]"
    tclgr send title \"$captraitname\"
    tclgr send xaxis tick op bottom

    tclgr send xaxis ticklabel start type spec
    set xstart [expr round (ceil ($minx / 10.0) * 10)]
    tclgr send xaxis ticklabel start $xstart

    set xstop [expr round (floor ($maxx / 10.0) * 10)]
    tclgr send xaxis ticklabel stop type spec
    tclgr send xaxis ticklabel stop $xstop

# Determine whether legend goes at top or bottom

    tclgr send legend loctype view
    if {$middle_min > 0.45} {
	tclgr send legend Y1 0.295
    } else {
	tclgr send legend Y1 0.705
    }

# Load xmgr parameter file liability.gr

    global env
    if {[file exists $env(SOLAR_LIB)/liability.gr]} {
	set mpathname [glob $env(SOLAR_LIB)/liability.gr]
	tclgr send read \"$mpathname\"
    }
    if {[file exists ~/lib/liability.gr]} {
	set mpathname [glob ~/lib/liability.gr]
	tclgr send read \"$mpathname\"
    } 
    if {[file exists liability.gr]} {
	tclgr send read \"liability.gr\"
    } 

# Plot male curve (set 1, so blue is on top of overlap)

    if {$found_sex} {
	set use_set 1
    } else {
	set use_set 2
    }
    foreach point $male_curve {
	set x [lindex $point 0]
	set y [lindex $point 1]
	tclgr send g0.s$use_set point $x,$y
    }

# Plot female curve (set 0)

    if {$found_sex} {
	foreach point $female_curve {
	    set x [lindex $point 0]
	    set y [lindex $point 1]
	    tclgr send g0.s0 point $x,$y
	}

# Turn on legend box

	tclgr send legend on
	tclgr send legend box on
    }

# DRAW !

    tclgr send redraw
}


# solar::plot_power -- private
#
# Purpose:  Implements "plot -power"
#
# Usage:    plot_power [-title <plot_title>]
#
# -

proc plot_power {args} {
    set title "Power"
    read_arglist $args -title title -power {set ignore_this 0}

    if {[catch {tclgr open} errmsg]} {
	if {[string compare $errmsg \
		"tclgr session already opened from this solar session"]} {
	    error $errmsg
	}
    }

    if {[catch {set inf [open power.info r]}]} {
        error "Cannot open power.info"
    }
    gets $inf line
    if {[lindex $line 0] == "data"} {
        gets $inf line
    }
    set nreps [lindex $line 2]
    gets $inf line
    if {[lindex $line 0] == "h2t"} {
        set h2t [lindex $line 2]
        set h2r ""
    } else {
        set h2r [lindex $line 2]
        set h2t ""
    }
    gets $inf line
    set lods [lrange $line 7 [expr [llength $line] - 1]]
    set nlods [llength $lods]
    close $inf

    if {[catch {set inf [open power.out r]}]} {
        error "Cannot open power.out"
    }

    set x {}
    for {set i 0} {$i < $nlods} {incr i} {
        set y($i) {}
    }
    while {-1 != [gets $inf line]} {
        lappend x [lindex $line 0]
        set i1 1
        for {set i 0} {$i < $nlods} {incr i} {
            lappend y($i) [lindex $line $i1]
            incr i1
        }
    }
    close $inf

    set npts [llength $x]
    if {$npts == 0} {
	error "No ELODs in power.out ... can't plot"
    }

    tclgr send kill graphs
    tclgr send clear line
    tclgr send clear string
    tclgr send focus g0

    tclgr send title \"$title\"

    tclgr send world xmin 0
    set max_x [lindex $x [expr [llength $x] - 1]]
    set max_x [expr ceil(10*$max_x)/10]
    tclgr send world xmax $max_x
    tclgr send xaxis tick major 0.1
    tclgr send xaxis tick minor 0.05

    tclgr send world ymin 0
    tclgr send world ymax 1
    tclgr send yaxis tick major 0.1
    tclgr send yaxis tick minor 0.05

    tclgr send legend loctype view
    tclgr send legend X1 0.58
    tclgr send legend Y1 [expr 0.21 + $nlods*0.04]
    tclgr send legend on
    tclgr send legend box on

    global env
    if {[file exists $env(SOLAR_LIB)/power.gr]} {
	set mpathname [glob $env(SOLAR_LIB)/power.gr]
	tclgr send read \"$mpathname\"
    }
    if {[file exists ~/lib/power.gr]} {
	set mpathname [glob ~/lib/power.gr]
	tclgr send read \"$mpathname\"
    } 
    if {[file exists power.gr]} {
	tclgr send read \"power.gr\"
    } 

    for {set i 0} {$i < $npts} {incr i} {
        for {set j 0} {$j < $nlods} {incr j} {
	    tclgr send s$j color [expr $j + 2]
	    tclgr send g0.s$j point "[lindex $x $i],[lindex $y($j) $i]"
            tclgr send legend string $j \"LOD = [lindex $lods $j]\"
        }
    }

    tclgr send redraw
}

proc purge_multipoint_output_directory {} {
    eval delete_files_forcibly \
    [glob -nocomplain [full_filename link*.mod]] \
    [glob -nocomplain [full_filename link*.out]] \
    [glob -nocomplain [full_filename link*.smp]] \
    [full_filename temp.out] \
    [full_filename temp.smp]

    set nullmods [glob -nocomplain [full_filename null*.mod]]
    foreach mod $nullmods {
	if {[string compare $mod [full_filename null0.mod]]} {
	    delete_files_forcibly $mod
	}
    }

    set nullmods [glob -nocomplain [full_filename null*.out]]
    foreach mod $nullmods {
	if {[string compare $mod [full_filename null0.out]]} {
	    delete_files_forcibly $mod
	}
    }
}

proc purge_sporadic_output_directory {} {
    eval delete_files_forcibly [full_filename s0.mod] \
    [full_filename s0.out] \
    [full_filename s0.smp] \
    [glob -nocomplain [full_filename no*.mod]] \
    [glob -nocomplain [full_filename no*.out]] \
    [glob -nocomplain [full_filename no*.smp]] \
    [full_filename spor.mod] \
    [full_filename spor.out] \
    [full_filename spor.smp] \
    [full_filename temp.out] \
    [full_filename temp.smp]
}

proc purge_polygenic_output_directory {} {
    eval delete_files_forcibly [full_filename s0.mod] \
    [full_filename s0.out] \
    [full_filename s0.smp] \
    [full_filename p0.mod] \
    [full_filename p0.out] \
    [full_filename p0.smp] \
    [glob -nocomplain [full_filename no*.mod]] \
    [glob -nocomplain [full_filename no*.out]] \
    [glob -nocomplain [full_filename no*.smp]] \
    [full_filename spor.mod] \
    [full_filename spor.out] \
    [full_filename spor.smp] \
    [full_filename poly.mod] \
    [full_filename poly.out] \
    [full_filename poly.smp] \
    [full_filename temp.out] \
    [full_filename temp.smp]
}

proc bayesavg_purge {args} {
    set testonly 0
    set outname bayesavg
    set prefix c
    read_arglist $args -testonly {set testonly 1} \
	    -outname outname -prefix prefix

    set baylist [concat \
	    [glob -nocomplain [full_filename tmp.*]] \
	    [glob -nocomplain [full_filename $outname.*]] \
	    [glob -nocomplain [full_filename $prefix.spor.*]] \
	    [glob -nocomplain [full_filename $prefix.sat.*]] \
	    [glob -nocomplain [full_filename $prefix.start.*]] \
	    [glob -nocomplain [full_filename $prefix.orig.*]] \
	    [glob -nocomplain [full_filename $prefix.base.*]] \
	    [glob -nocomplain [full_filename $prefix\[0-9\].mod]] \
	    [glob -nocomplain [full_filename $prefix\[0-9\].out]] \
	    [glob -nocomplain [full_filename $prefix\[0-9\].smp]] \
	    [glob -nocomplain [full_filename $prefix\[0-9\]\[0-9\]*.mod]] \
	    [glob -nocomplain [full_filename $prefix\[0-9\]\[0-9\]*.out]] \
	    [glob -nocomplain [full_filename $prefix\[0-9\]\[0-9\]*.smp]]]

    if {!$testonly} {
	eval delete_files_forcibly [glob -nocomplain [full_filename tmp.*]] $baylist
    }
    return $baylist
}

# solar::cleanmodel -- private
#
# Purpose:  Clear parameter values [NOT YET IMPLEMENTED]
#
# Usage:    cleanmodel
#
# Notes:    Use this model to clear out obsolete parameter values but keep the
#           major model settings (trait, covariates, options, etc.)
#
#           This is particularly useful after a convergence failure has put
#           NaN values in some or all parameters.  Such a model cannot be restarted.
#
#           When a parameter has all zero values, starting points and boundaries
#           are setup automatically during maximization.
#-

proc clearmodel {} {
}

# solar::newmod --
#
# Purpose:  Start a new model
#
# Usage:    newmod [<trait>]+
#
#           <trait>  Set the trait(s) to this/these trait(s).  (The trait(s)
#                    can be specified later.  If not specified here, they
#                    become <undefined>.)
# Notes:
#
#    (1)   This combines "model new", "outdir -default", and optionally
#          trait [<trait>]+ .  This is now preferred to using the separate
#          commands, because it is shorter.  For example, the command:
#
#        newmod q1 q2
#
#           takes the place of the commands:
#
#        outdir -default
#        model new
#        trait q1 q2
#
#           Clearly the "newmod" form is superior, it preserves the
#           essential information while reducing redundant keystrokes.
#
#    (2)    Since this clears the outdir, it is adviseable to use this
#           command instead of "model new" to be sure that the outdir
#           is cleared, and not inheirited from some previous script.
#           From now on, the manual advises using "newmod" (and not
#           "model new") for this reason.  However, the behavior of
#           "model new" itself is unchanged, so that existing scripts
#           that operate correctly will continue to operate correctly.
#           When combining previously written scripts that use "model new"
#           instead of "newmod", the user must be careful to update
#           "outdir" status if required.  New scripts using "newmod" will
#           not be subject to the error of incorrectly inheiriting an
#           unwanted outdir setting.
# -

proc newmod {args} {
    outdir -default
    if {{} == $args} {
	return [model new]
    } else {
	model new
	return [eval trait $args]
    }
    return ""
}


# solar::spormod --
#
# Purpose:  Set up a sporadic model with the standard parameters
#
# Usage:    spormod
#
# Notes:    There are no arguments.  You must have previously loaded the
#           phenotypes file, selected the trait, and specified the
#           covariates.
#
#           Household effects are suspended.  If you want a 'household'
#           model, give the spormod command first, then the 'house' command.
#
#           The starting lower bound for e2 is controlled by e2lower.
#
#           Normally you do not use this command directly, but instead use 
#           the "polygenic" command to do a complete polygenic analysis,
#           which maximizes a sporadic model which was set up using this
#           command.  See the tutorial in Chapter 3.
# -

proc spormod {} {

# Suspend household effects
    house -suspend    ;# Suspend household effects, if they had been enabled

# SET UP PARAMETERS FOR UNIVARIATE...
# Since each parameter is inserted at beginning, we add them in the reverse
# from standard order.  Note that if parameter already exists, its position
# is unchanged.

    set ts [trait]
    set nts [llength $ts]
    if {$nts == 1} {
	set multi 0

# e2 and h2r

	global SOLAR_constraint_tol
	parameter -insert h2r = 0 lower -$SOLAR_constraint_tol upper 1
	parameter -insert e2 = 1 lower [e2lower] upper 1.01
	constraint h2r = 0

# Mean and SD

	parameter -insert sd
	parameter -insert mean

# SET UP PARAMETERS FOR BI/MULTIVARIATE...

    } elseif {$nts > 1} {
	set multi 1

# Set up rhoe and rhog parameters for bivariate models
# rhog must be constrained to 0 for sporadic model
    
	if {$nts == 2} {
	    parameter rhoe = 0 lower -0.9 upper 0.9
	    parameter rhog = 0 lower -0.9 upper 0.9
	    constraint rhog = 0
	} else {
            foreach prefix {rhog_ rhoe_} {
		for {set i [expr $nts - 1]} {$i > 0} {incr i -1} {
		    for {set j $nts} {$j > $i} {incr j -1} {
			parameter -insert [catenate $prefix $i $j] = 0 \
			    lower -0.9 upper 0.9
		    }
		}
	    }
	}

# Configure Mean and SD parameters

	for {set i $nts} {$i > 0} {incr i -1} {
	    set tr [lindex $ts [expr $i - 1]]

	    global SOLAR_constraint_tol
    parameter -insert h2r\($tr\) = 0 lower -$SOLAR_constraint_tol upper 1
	    parameter -insert e2\($tr\) = 1 lower [e2lower] upper 1.01
	    parameter -insert sd\($tr\)
	    parameter -insert mean\($tr\)
	    constraint <h2r\($tr\)> = 0
	}
    }

# Remove all h2q's and their constraints and matrices
#   Bivariate version does not support special constraint terms
    set starting_h2qcount [h2qcount]
    if {$starting_h2qcount > 0} {
	for {set i 1} {$i <= $starting_h2qcount} {incr i} {
	    catch {matrix delete mibd$i}
	    foreach tr $ts {
		if {$multi} {
		    set suffix ($tr)
		    constraint <e2$suffix> + <h2r$suffix> = 1
		} else {
		    set suffix ""
		    catch {constraint_remove h2q$i}
		}
		catch {parameter delete h2q$i$suffix}
	    }
	}
    }

# Update omega...
#   Multivariate versions do not support special omega terms
#     They just slap in new omega...
    if {$nts == 2} {

	omega = <sd(ti)>*<sd(tj)>* \
I*sqrt(<e2(ti)>)*sqrt(<e2(tj)>)*(tne*rhoe+teq)

    } elseif {$nts > 2} {

	omega = <sd(ti)>*<sd(tj)>*( \
I*sqrt(<e2(ti)>)*sqrt(<e2(tj)>)*(tne*rhoe_ij+teq) + \
phi2*sqrt(<h2r(ti)>)*sqrt(<h2r(tj)>)*(tne*rhog_ij+teq) )
	
    } else {
	if {"omega = Use_polygenic_to_set_standard_model_parameterization" \
		== [omega]} {
	    omega = pvar*(phi2*h2r + I*e2)
	} else {
	    omega_remove_linkage  ;# Must be done after using h2qcount
	}
    }

# Set up principal variance component constraints

    if {!$multi} {
	constraint e2 + h2r = 1
    } else {
	foreach tr $ts {
	    set suffix \($tr\)
	    constraint <e2$suffix> + <h2r$suffix> = 1
	}
    }

    return ""
}


# Apply to all variance parameters generically

proc vparameter {gname args} {
    if {[catch {trait}]} {
	error "Trait must be specified first"
    }
    set ts [trait]
    if {1 == [llength $ts]} {
	eval parameter $gname $args
    } else {
	foreach t $ts {
	    eval parameter $gname\($t\) $args
	}
    }

# Covariance Matrices Diagonal (requires rhoe start at 0)

    option CMDiagonal 1

    return ""
}

# solar::linkgsd --
# solar::linkqsd0 --
# solar::linkqsd --
#
# Purpose:  Set up linkage model with esd, gsd, qsd parameters (EXPERIMENTAL)
#
# Usage:  linkqsd <path>/<mibdfile>
#         linkqsd0 <path>/<mibdfile>
#
# Example:  model new
#           trait q4
#           covar age sex
#           polygsd
#           maximize
#           gsd2h2r
#           chromosome 9 10
#           interval 5
#           mibddir gaw10mibd
#           multipoint -link linkqsd0 -cparm {esd gsd qsd}
#
# Notes:  Polygenic parameters must already have been set up (use the
#         polygsd command).  Prefereably it should have been maximized
#         also (use the maximize command).
#
#         linkqsd modifieds the model currently in memory.  linkqsd0 assumes
#         the existance of a null0 model in the maximization output directory,
#         loads that, and then adds the linkage element.
#
#         We have not worked out suitable heuristics to force maximization
#         of difficult models, so at the present time this parameterization
#         is not as robust as our standard parameterization.
# -

# linkgsd is an alias
proc linkgsd {mibdfile args} {
    eval linkqsd $mibdfile $args
}

proc linkqsd0 {mibdfile args} {
    load model [full_filename null0]
    eval linkqsd $mibdfile $args
    return ""
}
    
proc linkqsd {mibdfile args} {
    if {![file exists $mibdfile]} {
	error "linkqsd: File $mibdfile not found."
    }
    set ts [trait]
    set ntraits [llength $ts]

    global Solar_qsd_fraction
    if {[if_global_exists Solar_qsd_fraction]} {
	set qsd_fraction $Solar_qsd_fraction
    } else {
	set qsd_fraction 0.02
    }

    if {$ntraits == 1} {
	if {![if_parameter_exists esd] || ![if_parameter_exists gsd]} {
	    error "linkqsd: Must run polygsd first to create esd and gsd"
	}
	set esd [parameter esd =]
	set gsd [parameter gsd =]
	set sd [expr sqrt ($esd*$esd + $gsd*$gsd)]

	set qsd [expr $sd * $qsd_fraction]

	if {$esd > $gsd} {
	    set esd [expr sqrt ($sd*$sd - $gsd*$gsd - $qsd*$qsd)]
	} else {
	    set gsd [expr sqrt ($sd*$sd - $esd*$esd - $qsd*$qsd)]
	}

	set upper_qsd [parameter gsd upper]
	parameter qsd1 = $qsd lower 0 upper $upper_qsd
	parameter esd = $esd
	parameter gsd = $gsd
	matrix load $mibdfile mibd1
	omega = I*esd*esd + phi2*gsd*gsd + mibd1*qsd1*qsd1
    } else {
	foreach tr $ts {
	    if {![if_parameter_exists esd($tr)] || \
		    ![if_parameter_exists gsd($tr)]} {
		error "linkqsd: Must run polygsd first to create esd and gsd"
	    }
	    set esd [parameter esd($tr) =]
	    set gsd [parameter gsd($tr) =]
	    set sd [expr sqrt ($esd*$esd + $gsd*$gsd)]

	    set qsd [expr $sd * $qsd_fraction]

	    if {$esd > $gsd} {
		set esd [expr sqrt ($sd*$sd - $gsd*$gsd - $qsd*$qsd)]
	    } else {
		set gsd [expr sqrt ($sd*$sd - $esd*$esd - $qsd*$qsd)]
	    }

	    set upper_qsd [parameter gsd($tr) upper]
	    parameter qsd1($tr) = $qsd lower 0 upper $upper_qsd
	    parameter esd($tr) = $esd
	    parameter gsd($tr) = $gsd
	}
	matrix load $mibdfile mibd1
	if {$ntraits == 2} {
	    parameter rhoq1 = 0 lower -1 upper 1
	    omega = I*<esd(ti)>*<esd(tj)>*(tne*rhoe+teq) + \
		phi2*<gsd(ti)>*<gsd(tj)>*(tne*rhog+teq) + \
		mibd1*<qsd1(ti)>*<qsd1(tj)>*(tne*rhoq1+teq)
	} else {
	    for {set i [expr $ntraits - 1]} {$i > 0} {incr i -1} {
		for {set j $ntraits} {$j > $i} {incr j -1} {
		    parameter [catenate rhoq1_ $i $j] = 0 \
			lower -1 upper 1
		}
	    }
	    omega = I*<esd(ti)>*<esd(tj)>*(tne*rhoe_ij+teq) + \
		phi2*<gsd(ti)>*<gsd(tj)>*(tne*rhog_ij+teq) + \
		mibd1*<qsd1(ti)>*<qsd1(tj)>*(tne*rhoq1_ij+teq)
	}
    }
    return ""
}


# solar::intraitclass
#
# purpose: set up trait with classwise inormalization
#
# Usage:  intraitclass <traitname> [<class>]+
#
# Example: trait averagefa 0 2 3
#
# Notes: Do model new before using this command.
#
# -

proc intraitclass {traitname args} {
    set def ""
    foreach class $args {
	if {$def == ""} {
	    set def "(class==$class)*inormalc_$class\_$traitname"
	} else {
	    set def "$def + (class==$class)*inormalc_$class\_$traitname"
	}
    }
    define i_$traitname = $def
    trait i_$traitname
}

# solar::sporclass --
# solar::polyclass -- (EXPERIMENTAL)
#
# Purpose:  Set up polygenic model with class specific parameterization
#
# Usage:    polyclass [-g] [-intrait] [-incovar] [<class-start>[-<class-end>]]+
#                     [-comb] [-maxi] [-rincovar] [-maxsnp <snp_name>]
#           sporclass [-g] [-intrait] [-incovar] [<class-start>[-<class-end>]]+
#                     [-comb] [-maxi] [-rincovar] [-maxsnp <snp_name>]
#
#           -g   Use global phenotypic values to set parameter adjustments
#                (otherwise, means are determined for each class)
#
#           -intrait  inormalize trait values on a per-class basis
#           -resmax inormalize residual values in place of traits
#           -incovar  (NOT WORKING IN version 7.1.2) inormalize covar values
#                     on a per-class basis (only used for simple linear
#                     covariates, no interactions or exponents)
#           -comb     all classes combined model
#           -max      after building the model, maximize it
#
#           -maxsnp <snp_name>  Maximize and include  snp_name as covariate
#                        in the model and determine statistics for it: beta,
#                        beta se, chi, p, and variance explained (varexp).  
#                        H2r's are reported for the models with and
#                        without the snp.
#           -append   Append results to existing output file(s) if any
#
# Short Example:
#
#            trait q4
#            covariate age sex
#            polyclass 1-3 9
#            maximize -q
#
# Notes: One phenotypes file must have a field named "class" which defines
#        the class value for each person in the sample.
#
#        Class specific parameters are given names with _c<class> appended.
#
#        User covariates are transformed into class-specific mu addends.
#        All individuals in sample must have all variables specified as
#        covariates.
#
#        After choosing trait and covariates, do either sporclass or
#        polyclass.  You cannot do a second polyclass on a sporclassed model
#        to make it polygenic.
#
#        Unbalanced covariates for multivariate traits are not supported.
#        This is different from ordinary covariate behavior for multivariate
#        traits--which permits covariates to be missing in the sample if they
#        are specific to a missing trait.
#
#        A defined pseudo-covariate named "blank_classes()" restricts the
#        sample to the union of all classes specified.
#
#        The maximized model is asved in the output directory as
#        polyclassmax.mod with output file polyclassmax.out.  Note that if
#        -intrait option is selected, trait name and default output
#        directory will have leading i_ prefix (for the inormalization).
#        If the -resmax option is selected, the trait will be named
#        "residual" or "i_residual" if -intrait is also selected.
#
#-

proc sporclass {args} {
    return [eval polysporclass -s $args]
}

proc polyclass {args} {
    return [eval polysporclass -p $args]
}

proc polysporclass {args} {

    set grand 0
    set sporadic 0
    set intrait 0
    set incovar 0
    set resmax 0
    set comb 0
    set maxi 0
    set nextintrait 0
    set maxsnp ""
    set allterms ""
    set snpappend 0

    if {[lindex $args 0] == "-p"} {
	set args [lrange $args 1 end]
    }
    if {[lindex $args 0] == "-s"} {
	set sporadic 1
	set args [lrange $args 1 end]
    }

    set classargs [read_arglist $args -g {set grand 1} -intrait {set intrait 1}\
		       -maxsnp maxsnp -append {set snpappend 1} \
       -incovar {set incovar 1} -comb {set comb 1} -maxi {set maxi 1} \
		       -max {set maxi 1} -resmax {set resmax 1; set maxi 1}
]

    if {$maxsnp != ""} {
	set maxi 1
	set prefix [string tolower [string range $maxsnp 0 3]]
	if {$prefix != "snp_"} {
	    error "snp names must begin with snp_ prefix"
	}
	set phens [lrange [phenotypes] 1 end]
	if {-1 == [lsearch -exact $phens $maxsnp]} {
	    error "$maxsnp not in phenotypes file (case significant now)"
	}
    }

    if {$resmax} {
	if {$intrait} {
	    set intrait 0
	    set nextintrait 1
	}
    }

    set classlist [expand_ranges $classargs]

    foreach pclass $classlist {
	if {![is_integer $pclass]} {
	    error "$pclass is not a valid phenotype class"
	}
    }
    if {0 == [llength $classlist]} {
	error "polyclass: Classes not specified!"
    }
    puts "full class list is $classlist"

    if {$maxsnp != ""} {
	covariate $maxsnp
    }

    set ts [trait]
    set nt [llength $ts]
    set covs [covariate]
    set oldcovs [covariate]
    set ncov [llength $covs]
    set ntraits [llength $ts]

    set savename [full_filename solar.polyclass]
    save model $savename.start

# if grand mean model, initializations are not class specific so do here    

    if {$grand} {
	model new
	eval trait $ts
	eval covariate $covs
	covariate class()
	polymod
	maximize -initpar
	save model $savename.allc
	set pars [parameter -return]
    }

    set newbetas ""
    set oldbetas ""
    set newmu ""
    set residualmup "\[lindex \$line 1\]"
    set residualmu ""
    set residualmustarted 0
    set newomega ""
    set first 1
    set k -1
    set newtsinit ""
    set newts ""
    set has_sex 0

    foreach ic $classlist {

# if non-grand mean model, we must do class-specific initializations here
# also, accumulate grand definition of all class inormalizations

	if {!$grand} {
	    model new
	    if {!$intrait} {
		set newtsinit $ts

	    } else { ;# FOR -intrait
		set newtsinit ""
		foreach tr $ts {
		    set newt i_$tr
		    lappend newtsinit $newt
		    define $newt = inormalc_$ic\_$tr
		}
	    }
	    if {!$incovar} {
		set newcs $covs
	    } else {
		set newcs ""
		foreach cov $covs {
		    if {-1 != [string first * $cov] || -1 != [string first ^ $cov] \
			    || -1 != [string first \( $cov] || $cov == "sex"} {
			lappend newcs $cov
		    } else {
			set newcov c$ic\_$cov
			define $newcov = inormalc_$ic\_$cov
			lappend newcs $newcov
		    }
		}
	    }

	    eval trait $newtsinit
	    eval covariate $newcs
	    define blank_classes = blank*(class!=$ic)
	    covariate blank_classes()
	    if {$sporadic} {
		spormod
	    } else {
		polymod
	    }
	    set maxstatus [catch {maximize -initpar} errmes]
	    if {$maxstatus != 0} {
		puts $errmes
		error "Error (above) prevents using class $ic"
	    }
	    save model $savename.$ic
	    set pars [parameter -return]
	}
#
# For the first class, grand or not, start the whole model
# This is where we do the real model new and start
#
# We reload the previous model on each pass after the first
#
	if {$first} {
	    model new
	    if {!$intrait} {
		set newts $ts
	    } else {
		set newts ""
		foreach tr $ts {
		    set newt i_$tr
		    lappend newts $newt
		    define $newt = inormalc_$ic\_$tr  ;# temp definition
# we accumulate the string required for the final definition of this trait
		    set intraitdef($newt) "0"
		}
	    }
#	    puts "new traits are $newts"
	    eval trait $newts
	    set defstring "blank"
	    foreach icc $classlist {
		set defstring "$defstring*(class!=$icc)"
	    }
	    eval define blank_classes = $defstring
	    covariate blank_classes()
	    set first 0
	} elseif {!$grand} {
	    load model $savename.temp
	}
	if {$intrait} {
	    set nindex 0
	    foreach newt $newts {
		set oldtr [lindex $ts $nindex]
		incr nindex
		set intraitdef($newt) \
"$intraitdef($newt) + (class==$ic)*inormalc_$ic\_$oldtr"
	    }
	}

# create and initialize parameters for this class

	foreach par $pars {
	    if {$nt == 1 || ([string range $par 0 2] == "rho")} {
		set newpname [lindex $par 0]_c$ic
		set oldpname [lindex $par 0]
	    } else {
		set newp [lindex $par 0]
		set ppos [string first \( $newp]
		set pre [string range $newp 0 [expr $ppos - 1]]
		set post [string range $newp [expr $ppos + 1] end]
		set pp ""
		if {$intrait && $grand} {
		    set pp i_
		}
		set newpname "$pre\_c$ic\($pp$post"
		set oldpname [lindex $par 0]
	    }
	    set newpstring "parameter $newpname = [lrange $par 2 end]"
#	    puts "Evaluating $newpstring"
	    eval $newpstring
	    if {[string index $newpname 0] == "b"} {
		lappend newbetas $newpname
		lappend oldbetas $oldpname
	    }
	}

# create constraints for this class

	if {$sporadic} {
	    if {$nt==1} {
		constraint h2r_c$ic = 0
	    } else {
		foreach tr $newts {
		    constraint <h2r_c$ic\($tr\)> = 0
		}
	    }
	}

	if {$nt==1} {
	    constraint e2_c$ic + h2r_c$ic = 1
	} else {
	    foreach tr $newts {
		constraint <e2_c$ic\($tr\)> + <h2r_c$ic\($tr\)> = 1
	    }
	}



# create omega terms for this class

	set newomega "$newomega + (class_i==$ic)"
	if {$nt==1} {
	    set newomega "$newomega*sd_c$ic*sd_c$ic*(phi2*h2r_c$ic + I*e2_c$ic)"
	} elseif {$nt==2} {
	    set newomega "$newomega*<sd_c$ic\(ti\)>*<sd_c$ic\(tj\)>*( \
I*sqrt(<e2_c$ic\(ti\)>)*sqrt(<e2_c$ic\(tj\)>)*(tne*rhoe_c$ic+teq) + \
phi2*sqrt(<h2r_c$ic\(ti\)>)*sqrt(<h2r_c$ic\(tj\)>)*(tne*rhog_c$ic+teq))"

	} else {
	    set newomega "$newomega*<sd_c$ic\(ti\)>*<sd_c$ic\(tj\)>*( \
I*sqrt(<e2_c$ic\(ti\))*sqrt(<e2_c$ic\(tj\))*(tne*rhoe_c$ic\_ij+teq) + \
phi2*sqrt(<h2r_c$ic\(ti\)>)*sqrt(<h2r_c$ic\(tj\)>)*(tne*rhog_c$ic\_ij+teq))"
	}
#	puts "newomega is $newomega"

# create mu terms for this class and trait

	set tno 0
	foreach trai $newts {
	    incr tno

	    if {$nt==1} {
		set newmu "$newmu + (class==$ic)*(mean_c$ic"
		if {$residualmustarted} {
		    if {$comb} {
			set residualmu "$residualmu - (\[parameter mean = \]"
		    } else {
			set residualmu "$residualmu - (\[lindex \$line 2\]==$ic)*(\[parameter mean_c$ic = \]"
		    }
		} else {
		    if {$comb} {
			set residualmu "$residualmup - (\[parameter mean = \]"
		    } else {
			set residualmu "$residualmup - (\[lindex \$line 2\]==$ic)*(\[parameter mean_c$ic = \]"
		    }
		    set residualmustarted 1
		}
	    } else {
		set newmu "$newmu + t$tno*(class==$ic)*(<mean_c$ic\($trai\)>"
	    }
	    
	    set nbetas [llength $newbetas]

#	    puts "COVS are $covs"
#	    puts "BETAS are $newbetas"

	    for {set i 0} {$i < $ncov} {incr i} {
		set cov [lindex $covs [expr $i]]
		if {[string range $cov end-1 end] == "()"} {
		    covariate $cov
		    continue
		}
		if {$incovar} {
		    if {-1 != [string first * $cov] || -1 != [string first ^ $cov] \
			    || -1 != [string first \( $cov] || $cov == "sex"} {
		    } else {
			set newcov c$ic\_$cov
			set cov $newcov
		    }
		}
		incr k
		set newbeta [lindex $newbetas $k]
		set terms [split $cov *]
		set nexps [llength [split $cov ^]]
		set nterms [llength $terms]

		if {$nt==1&&$nterms==1&&$nexps==1} {
		    set newmu "$newmu + $newbeta"
		} else {
		    set newmu "$newmu + <$newbeta>"
		}
		if {$comb} {
		    set oldbeta [lindex $oldbetas $k]
		    set residualmu "$residualmu + \[parameter $oldbeta =\]"
		} else {
		    set residualmu "$residualmu + \[parameter $newbeta =\]"
		}
		foreach term $terms {
		    set undervex [split $term ^]
		    set exp 1
# check for exponent and split off actual term		    
		    if {[llength $undervex] > 1} {
			set term [lindex $undervex 0]
			set exp [lindex $undervex 1]
		    }
# check if this is sex
		    if {[string tolower $term] == "sex"} {
			set avalue 0
			set term female
			set has_sex 1
		    } else {
# if not sex, add this to terms we need from phen file
# and determine adjustment value
			set allterms [setappend allterms $term]
			if {$term == $maxsnp} {
			    set avalue 1
			} elseif {[getvar -d solar.out $term]} {
			    set avalue [getvar -min solar.out $term]
			} else {
			    set avalue [getvar -mean solar.out $term]
			}
		    }
# set up newmu and residualmu for each term (multiplying for exp's)
		    for {set j 0} {$j < $exp} {incr j} {
			set newmu "$newmu*($term - $avalue)"
			if {$term == "female"} {
			    set residualmu "$residualmu*(\$Female(\[lindex \$line 0\]))"
			} elseif {$term==$maxsnp} {
			    set residualmu "$residualmu*0"
			} else {
			    set residualmu "$residualmu*(\[lindex \$line \
\$Cpos($term)\] - $avalue)"
			}
		    }
		}
	    }
	    set newmu "$newmu\)"
	    set residualmu "$residualmu\)"
	    if {!$grand} {
		save model $savename.temp
	    }
	}
    }

    set newomega [lrange $newomega 1 end]
    set newmu [lrange $newmu 1 end]
    puts "eval> omega = $newomega"
    eval omega = $newomega
    puts "eval> mu = $newmu"
    eval mu = $newmu
    if {$maxsnp != ""} {
	puts "ResidualMu = $residualmu"
    }
#
# make final definition for each trait
#
    if {$intrait} {
	foreach newt $newts {
	    define $newt = $intraitdef($newt)
	}
    }
#
# Now if this is combined-model, start all over with new traits and covars
# and run polymod
#
    if {$comb} {
	model new
	eval trait $newts
	eval covar $covs

	if {$sporadic} {
	    spormod
	} else {
	    polymod
	}
    }

    if {!$grand} {
	catch {file delete [full_filename polclass.tmp.mod]}
    }

    option polyclasses [join $classlist ,]

    if {$maxi} {

	if {$maxsnp != "" && !$comb} {
	    foreach ic $classlist {
		parameter b$maxsnp\_c$ic = 0
		constraint b$maxsnp\_c$ic = 0
	    }
	}
	maximize -o polyclassmax.out
	save model [full_filename polyclassmax.mod]
	if {$maxsnp != ""} {
	    if {$comb} {
		set sd [parameter sd =]
	    } else {
		foreach ic $classlist {
		    set sd$ic [parameter sd_c$ic =]
		}
	    }
#
# compute residuals for maxsnp
#
	        set outresidname [full_filename polyclass.residuals.out]
		set outresid [open $outresidname w]

		if {$has_sex} {
		    set pedfile [tablefile open pedindex.out]
		    tablefile $pedfile start_setup
		    tablefile $pedfile setup id
		    tablefile $pedfile setup sex
		    while {{} != [set line [tablefile $pedfile get]]} {
			set Female([lindex $line 0]) [expr [lindex $line 1]-1]
		    }
		    tablefile $pedfile close
		}

	        covariate class()
		maximize -sampledata
	        set sfilename [full_filename sampledata.out]
		load model [full_filename polyclassmax.mod]

		set sfile [tablefile open $sfilename]
		tablefile $sfile start_setup
		tablefile $sfile setup id
		tablefile $sfile setup trait1
	        puts "class being setup"
		tablefile $sfile setup class
		set residheader id,residual,[trait],class
		set cpos 2
		foreach sterm $allterms {
		    set Cpos($sterm) [incr cpos]
#		    puts "set up position $cpos for $sterm"
		    tablefile $sfile setup $sterm
#		    puts "$sterm set up"
		    set residheader $residheader,$sterm
		}
		if {$has_sex} {
		    set residheader $residheader,female
		}
		puts $outresid $residheader

		while {{} != [set line [tablefile $sfile get]]} {
		    set residual [eval expr $residualmu]
		    set outline [lindex $line 0],$residual,[lindex $line 1],[lindex $line 2]
		    set ucpos 2
		    foreach sterm $allterms {
			set outline $outline,[lindex $line [incr ucpos]]
		    }
		    if {$has_sex} {
			set outline $outline,$Female([lindex $line 0])
		    }
		    puts $outresid $outline
		}
		close $outresid
		tablefile $sfile close

	}
	if {$resmax} {
	    set routfile [full_filename polyclassres.out]
	    if {!$comb} {
		residual polyclassmax.out -out $routfile -class
	    } else {
		residual polyclassmax.out -out $routfile
	    }
	    load phenotypes $routfile
	    model new
	    trait residual
	    set newargs [remove_from_list $args -resmax]
	    puts "executing polyclass $newargs -maxi on residuals"
	    return [eval polyclass $newargs -maxi]
	}
	puts "raw likelihood is [loglike]"
	set all_like [loglike]
	puts ""
	set retline ""
	set first 1
	if {[llength $newts] == 1} {
	    if {$comb} {
		set h2rf [fformat %-12.6y [parameter h2r =]]
		set sef [fformat %-12.6y [parameter h2r se]]
		set retline "h2r = $h2rf se $sef"
		if {$maxsnp != ""} {
		    set snpf [fformat %-12.6y [parameter b$maxsnp =]]
		    set snpse [fformat %-12.6y [parameter b$maxsnp se]]
		    set retline "$retline b$maxsnp = $snpf se $snpse"
		    global SOLAR_constraint_tol
		    parameter b$maxsnp = 0 lower -$SOLAR_constraint_tol upper $SOLAR_constraint_tol
		    constraint b$maxsnp = 0
		    maximize
		    set clike [loglike]
		    set chisq [expr 2 * ($all_like - $clike)]
		    set fchisq [fformat %-12.6y $chisq]
		    set retline "$retline chi $fchisq"
		    if {$chisq < 0} {set chisq 0}
		    set pval [chi -number $chisq 1]
		    set fpval [fformat %-12.6y $pval]
		    set retline "$retline   p $fpval"
		    set csdic [parameter sd =]
		    set varexp [highest 0 [expr 1 - pow($sd/$csdic,2)]]
		    set fvarexp [fformat %-12.6y $varexp]
		    set retline "$retline varexp $fvarexp"
		    set nsh2rf [fformat %-12.6y [parameter h2r =]]
		    set nsse [fformat %-12.6y [parameter h2r se]]
		    set retline "$retline (nosnp)h2r = $nsh2rf"
		    set retline "$retline se $nsse"
		} else {
		    global SOLAR_constraint_tol
		    parameter h2r = 0 lower -$SOLAR_constraint_tol
		    constraint h2r = 0
		    parameter e2 = 1 upper 1.01
		    maximize
		    set clike [loglike]
		    set chisq [expr 2 * ($all_like - $clike)]
		    if {$chisq < 0} {set chisq 0}
		    set pval [chi -number $chisq 1]
		    set apval [expr $pval / 2.0]
		    set fpval [fformat %-12.6y $apval]
		    set retline "$retline p $fpval"
		}
		load model [full_filename polyclassmax]

	    } else { ;# not -comb
		set firstclass 1
		foreach ic $classlist {
		    if {$maxsnp == ""} {
			set se [fformat %-12.6y [parameter h2r\_c$ic se]]
			set h2rf [fformat %-12.6y [parameter h2r\_c$ic =]]
			if {$first} {
			    set retline "h2r_c$ic = $h2rf se $se"
			    set first 0
			} else {
			    set retline "$retline\nh2r_c$ic = $h2rf se $se"
			}
		    } else {
#
# To test non-comb snps:
#   Use the residual created earlier as trait
#   blank unused classes 
#   run mga
#   accumulate mga results
#   restore original phenotypes and model

			set originalmodelname [full_filename polyclassmax]
			set originalphens [phenotype -files]

			load phen $outresidname
			model new
			trait residual
			blank -o class!=$ic
			if {[catch {mga -q}]} {
			    puts "mga for class $ic failed"
			} else {
			    puts " "
			    global SOLAR_mga_last_out
			    global SOLAR_mga_header
			    if {$firstclass} {
				set firstclass 0
				if {$snpappend} {
				    set retline ""
				} else {
				    puts "doing header"
				    set retline "class,$SOLAR_mga_header\n"
				}
			    }
			    puts "adding $ic to results..."
			    set retline "$retline$ic,$SOLAR_mga_last_out\n"
			}
			eval load phen $originalphens
			model new
			load model $originalmodelname
		    }
		}
	    }
	} else {
	    foreach newt $newts {
		if {$comb} {
		    set se [fformat %-12.6y [parameter h2r($newt) se]]
		    set h2rf [fformat %-12.6y [parameter h2r($newt) =]]
		    if {$first} {
			set retline "h2r($newt) = $h2rf se $se"
			set first 0
		    } else {
			set retline "$retline\nh2r($newt) = $h2rf se $se"
		    }
		} else {
		  foreach ic $classlist {
		    set se [fformat %-12.6y [parameter h2r_c$ic\($newt\) se]]
	            set h2rf [fformat %-12.6y [parameter h2r_c$ic\($newt\) =]]
		    if {$first} {
			set retline "h2r_c$ic\($newt\) = $h2rf se $se"
			set first 0
		    } else {
		       set retline "$retline\nh2r_c$ic\($newt\) = $h2rf se $se"
		    }
		  }
		}
            }
        }
	if {$snpappend} {
	    set outfile [open [full_filename polyclass.out] a+]
	} else {
	    set outfile [open [full_filename polyclass.out] w]
	}
	if {!$comb && $maxsnp != ""} {
	    puts $outfile [string range $retline 0 end-1]
	} else {
	    puts $outfile $retline
	}
	close $outfile

	if {$maxsnp != ""} {
	    if {$comb} {
		set combfix .comb
	    } else {
		set combfix ""
	    }
	    set snpname [string range $maxsnp 4 end]
	    set snpfilename [full_filename polyclass$combfix.$snpname.out]
	    if {$snpappend} {
		set snpout [open $snpfilename a+]
	    } else {
		set snpout [open $snpfilename w]
	    }
	    if {$comb} {
		puts $snpout $retline
	    } else {
		puts $snpout [string range $retline 0 end-1]
	    }
	    close $snpout
	}
	return $retline
    }
    return ""
}



# solar::polygsd --
#
# Purpose:  Set up polygenic model esd and gsd parameters (EXPERIMENTAL)
#
# Usage:    polygsd
#
# Note:     "model new" and "trait" commands should be given first.
#           After polygsd, you should use "maximize" command.
#
#           Use the gsd2h2r command to convert resulting esd,gsd parameters
#           to h2r value.
#
#           Use the linkqsd command to add in linkage element afterwards.
#
# Example:  model new
#           trait q4
#           covar age sex
#           polygsd
#           maximize
#           linkqsd gaw10mibd/mibd.9.18.gz  ;# could maximize after this
#           chromosome 9 10
#           interval 5
#           mibddir gaw10mibd
#           multipoint -link linkqsd0 -cparm {esd gsd qsd}
#
#-

proc polygsd {} {
    set ts [trait]
    set ntraits [llength $ts]
    if {$ntraits == 1} {
	omega = I*esd*esd + phi2*gsd*gsd
    } elseif {$ntraits == 2} {
	parameter -insert rhog = 0 lower -1 upper 1
	parameter -insert rhoe = 0 lower -1 upper 1
	omega = I*<esd(ti)>*<esd(tj)>*(tne*rhoe+teq) + \
	    phi2*<gsd(ti)>*<gsd(tj)>*(tne*rhog+teq)
    } else {
	foreach prefix {rhog_ rhoe_} {
	    for {set i [expr $ntraits - 1]} {$i > 0} {incr i -1} {
		for {set j $ntraits} {$j > $i} {incr j -1} {
		    parameter -insert [catenate $prefix $i $j] = 0 \
			lower -1 upper 1
		}
	    }
	}
	omega = I*<esd(ti)>*<esd(tj)>*(tne*rhoe_ij+teq) + \
	    phi2*<gsd(ti)>*<gsd(tj)>*(tne*rhog_ij+teq)
    }

# Multivariate needs phi2.gz

    if {$ntraits > 1 && (-1==[lsearch -exact [matrix] phi2])} {
	load matrix phi2.gz phi2
    }

    set savename [full_filename solar.polygsd.temp]
    for {set i 0} {$i < $ntraits} {incr i} {
	set tr [lindex $ts [expr $ntraits - ($i + 1)]]
#
# Don't use raw stats anymore because defined traits and restricted sample
#	set stats [stats $tr -q -return]

	save model $savename
	set covs [covariate]
	model new
	trait $tr
	eval covariate $covs
	polymod
	maximize -sampledata -q
	if {[catch {set stats [stats trait1 -q -return -file \
				   [full_filename sampledata.out]]}]} {
	    puts "Warning: Unable to initialize parameters for trait $tr"
	    set mean 0
	    set sd 0
	    set mean_upper 0
	    set mean_lower 0
	    
	    set esd_start 0
	    set esd_upper 0
	    set gsd_start 0
	    set gsd_upper 0
	} else {
	    set mean [stats_get $stats mean]
	    set sd [stats_get $stats sd]
	    if {$mean >= 0} {
		set mean_lower [expr (0 - $mean) - 0.5]
		set mean_upper [expr $mean * 3 + 0.5]
	    } else {
		set mean_lower [expr ($mean * 3) - 0.5]
		set mean_upper [expr (0 - $mean) + 0.5]
	    }
	    set esd_upper [expr $sd * 2 + 0.1]
	    set gsd_upper [expr $sd * 2 + 0.1]

	    global Solar_gsd_fraction
	    if {[if_global_exists Solar_gsd_fraction]} {
		set gsd_fraction $Solar_gsd_fraction
	    } else {
		set gsd_fraction 0.2
	    }

	    set gsd_start [expr $sd * $gsd_fraction]
#	    set gsd_start 1
	    set esd_start [expr $sd * (sqrt (1 - $gsd_fraction*$gsd_fraction))]
	}
	load model $savename
	catch {file delete $savename}
	set suffix ""
	if {$ntraits > 1} {
	    set suffix ($tr)
	}
	parameter -insert gsd$suffix = $gsd_start lower 0 upper $gsd_upper
	parameter -insert esd$suffix = $esd_start lower 0 upper $esd_upper
	parameter -insert mean$suffix = $mean lower $mean_lower \
	    upper $mean_upper
    }
    return ""
}

proc polyqsd {mibdfile args} {
    eval polygsd $mibdfile $args
}


# solar::gsd2h2r --
# solar::gsd2sd  --
# solar::gsd2h2q --
#
# Purpose:  Convert esd,gsd,[qsd1] parameters to standard parameters
#
# Usage:                         ;# trait only required for multivariate model
#           gsd2h2r [<trait>]    ;# compute h2r from esd,gsd,[qsd1]
#           gsd2sd  [<trait>]    ;# compute SD from esd,gsd,[qsd1]
#           gsd2h2q [<trait>]    ;# compute h2q1 from esd,gsd,[qsd1]
#
#
# Note:     Use polygsd command to set up model, and maximize to maximize it
#           first, followed by linkgsd for linkage models.
#
#           See the documentation for the polygsd, linkgsd.
# -

proc gsdsuffix {args} {
    set suffix ""
    if {1 < [llength [trait]]} {
	if {"" == $args} {
	    error "gsd2: Must specify trait for multivariate"
	}
	if {-1 == [lsearch [trait] $args]} {
	    error "gsd2: Invalid trait specified"
	}
	set suffix "($args)"
    }
    return $suffix
}

proc gsd2h2r {args} {
    set suffix [gsdsuffix $args]
    set esd [parameter esd$suffix =]
    set gsd [parameter gsd$suffix =]
    set sum [expr $esd*$esd + $gsd*$gsd]
    if {[if_parameter_exists qsd1$suffix]} {
	set qsd [parameter qsd1$suffix =]
	set sum [expr $sum + $qsd*$qsd]
    }
    return [expr ($gsd*$gsd)/$sum]
}

proc gsd2sd {args} {
    set suffix [gsdsuffix $args]
    set esd [parameter esd$suffix =]
    set gsd [parameter gsd$suffix =]
    set sum [expr $esd*$esd + $gsd*$gsd]
    if {[if_parameter_exists qsd1$suffix]} {
	set qsd [parameter qsd1$suffix =]
	set sum [expr $sum + $qsd*$qsd]
    }
    return [expr sqrt ($sum)]
}

proc gsd2h2q {args} {
    set suffix [gsdsuffix $args]
    if {![if_parameter_exists qsd1$suffix]} {
	error "gsd2h2q: No qsd1 pmarameter"
    }
    set esd [parameter esd$suffix =]
    set gsd [parameter gsd$suffix =]
    set sum [expr $esd*$esd + $gsd*$gsd]
    set qsd [parameter qsd1$suffix =]
    set sum [expr $sum + $qsd*$qsd]
    return [expr ($qsd*$qsd)/$sum]
}

proc qsd2h2q {args} {
    return [gsd2h2q $args]
}

proc qsd2h2r {args} {
    return [gsd2h2r $args]
}

proc qsd2sd {args} {
    return [gsd2sd $args]
}


# solar::blank --
#
# Purpose:  Blank individuals according to variable data conditions
#
# Usage:    blank [-o] [-q] [-n] [<conditional expression>]
#
# <conditional expression> can be any solar variable expression (as allowed
# by the define command for covariates) that adds up to zero or non-zero.
# If it adds to non-zero for a person, that person is removed from the sample.
#
#           [-q]  Go about blanking quietly.
#
#           [-o]  Force overwrite of existing definition having same name
#                 (see below for example of definition naming).
#
#           [-n]  Make new definition name if this would otherwise
#                 conflict with existing definition name
#
#           With no arguments, blank shows blanking definitions currently in
#           effect.  To see all the definitions available, use define command.
#
# Examples:
#
#           blank class!=1           ;# include only class=1 in sample
#
#           blank age<<55 + sex==1   ;# blank all but old guys
#
#           blank age>=55 * sex==2   ;# blank only old guys
#
# Notes:
# 
# 1.  blank creates a definition and a null covariate to achieve the
#     desired blanking.  It shows you what it does, and then suggests
#     how this blanking may be deleted:
#
#     solar> blank age<<55 + sex==1
#
#     define blank_age = blank * (0!= (age<<55 + sex==1)
#     covariate blank_age()
#     To delete: covariate delete blank_age()
#
#     solar>
#
# 2.  blanking is cumulative through the effect of all blanking covariates
#     that remain in effect.  If you choose a condition which would create
#     the same name as used by a previous condition  (see example above) it
#     will raise an error.  You can force overwrite with -o.
#
# 3.  To restrict sample based on available of some variable, use a regular
#     null covariate for that variable, as documented for the covariate
#     command, for example:
#
#     covariate age()
#
#     null covariates (having following empty parentheses) are not included
#     in likelihood estimation, but are used to delimit the available sample,
#     just as blanking covariates are.
#
# 4.  You may also create covariate definitions just like blank does.  But
#     be careful because it is easy to do it wrong.
# -

proc blank {args} {

    set quiet 0
    set overwrite 0
    set new 0

    if {$args == ""} {
	set covs [covariates]
	set blanks ""
	foreach cov $covs {
	    if {[string range $cov 0 5] == "blank_"} {
		catch {set blanks "$blanks\n[define [string range $cov 0 end-2]]"}
	    }
	}
	return [string range $blanks 1 end]
    }

    set trying 1
    while {$trying} {
	set first [lindex $args 0]
	if {$first == "-q"} {
	    set quiet 1
	    set args [lrange $args 1 end]

	} elseif {$first == "-o"} {
	    set overwrite 1
	    set args [lrange $args 1 end]

	} elseif {$first == "-n"} {
	    set new 1
	    set args [lrange $args 1 end]
	    
	} else {
	    set trying 0
	}
    }

    set elength [string length $args]
    set estart -1
    set namestring ""

    for {set i 0} {$i < $elength} {incr i} {
	set ch [string index $args $i]
#	puts "testing $ch"
	if {[is_alpha $ch]} {
	    set estart $i
	    break
	}
    }
    if {$estart>=0} {
	for {set i [expr $estart+1]} {$i < $elength} {incr i} {
	    set ch [string index $args $i]
#	    puts "testing $ch"
	    if {![is_alnum $ch]} {
		set namestring [string range $args $estart [expr $i - 1]]
		break
	    }
	}
    }
    if {$namestring == ""} {
	error "conditions contain no variable name"
    }
    set extend ""
#   puts "checking this string $args"
    if {-1 != [string first << $args]} {
	set extend _lt
    } elseif {-1 != [string first >> $args]} {
	set extend _gt
    } elseif {-1 != [string first == $args]} {
	set extend _eq
    } elseif {-1 != [string first >= $args]} {
	set extend _ge
    } elseif {-1 != [string first <= $args]} {
	set extend _le
    }
    set namestring blank_$namestring$extend

    if {!$overwrite && -1!=[lsearch [define names] $namestring]} {
	if {$new} {
	    set counter 2
	    set newnamestring $namestring\_$counter
	    while {-1 != [lsearch [define names] $newnamestring]} {
		incr counter
		set newnamestring $namestring\_$counter
	    }
	    set namestring $newnamestring
	} else {
	    error "existing $namestring: use -o to overwrite or -n for newname"
	}
    }

# do the work

    set definition "define $namestring = blank * (0 != \($args\))"
    set covariate "covariate $namestring\(\)"
    eval $definition
    eval $covariate

# output description of work

    if {!$quiet} {
	puts ""
	puts "$definition"
	puts "$covariate"
	puts "To delete: covariate delete $covariate"
	puts ""
    }

    return ""
}

# solar::polymod --
#
# Purpose:  Set up polygenic model with the standard parameters
#
# Usage:    polymod [-d]
#
# IMPORTANT::  Phenotypes, trait, and covariate commands must be
#              given beforehand.
#
#           -d  Check for discrete trait(s) and make necessary changes.
#               In most cases, this option is not necessary because
#               "maximize" later checks for discrete traits and can also
#               make these changes: constraining SD to 1 and making
#               sure phi2 matrix is loaded, for each discrete trait.
#               However, use of -d option can make the constraint or matrix
#               order inside complex models easier to deal with.
#
# Notes:    The starting lower bound for e2 is controlled by e2lower.
#
#           Normally you do not use this command directly, but instead use 
#           the "polygenic" command to do a complete polygenic analysis,
#           which maximizes a polygenic model which was set up using this
#           command.  See the tutorial in Chapter 3.
#
#           polymod will modify an existing sporadic or linkage model
#           to change it to polygenic.  Use spormod to set up a
#           sporadic model, and linkmod to set up a linkage model.
#           None of these commands maximize models, they just set up
#           or modify the parameters and omega as required.
#           
#           This command removes a house parameter (if present) from
#           the omega, since a "polygenic" model is distinct from a
#           "household polygenic" model.  If you want the latter, call
#           polymod first, then house.  Or call house, THEN polygenic,
#           since the polygenic command will check for and handle household
#           effect properly.
# -

proc polymod {args} {

    set check_discrete 0
    set norhoij 0

# Suspend household effects
    house -suspend    ;# Suspend household effects, if they had been enabled

    if {[llength $args]} {
	if {$args == "-d"} {
	    set check_discrete 1
	} elseif {$args == "-norhoij"} {
	    set norhoij 1
	} else {
	    error "invalid polymod argument $args"
	}
    }


    set multi 0        ;# bivariate and above
    set trivariate 0

    set ts [trait]
    set ntraits [llength $ts]
    if {$ntraits > 1} {
	set multi 1
	if {$ntraits > 2} {
	    set trivariate 1
	}
    }

# Since each parameter is inserted at beginning, we add them in the reverse
# from standard order.  Note that if parameter already exists, its position
# is unchanged.

# Rho's (for multivariate models)

    foreach ty {g e} {
	if {$ntraits == 2 || ($ntraits > 2 && $norhoij)} {
	    set rholow -0.9
	    set rholowers lower
	    set rhoup 0.9
	    set rhouppers upper
	    if {[if_parameter_exists rho$ty]} {
		if {"" != [parameter rho$ty fixlower]} {
		    set rholow [parameter rho$ty fixlower]
		    set rholowers fixlower
		}
		if {"" != [parameter rho$ty fixupper]} {
		    set rhoup [parameter rho$ty fixupper]
		    set rhouppers fixupper
		}
	    }
	    parameter -insert rho$ty = 0 $rholowers $rholow $rhouppers $rhoup

	} elseif {$ntraits > 2} {

	    set trivariate 1
	    set prefix rho$ty\_

	    for {set i [expr $ntraits - 1]} {$i > 0} {incr i -1} {
		for {set j $ntraits} {$j > $i} {incr j -1} {
		    set pname [catenate $prefix $i $j]
		    set rholow -0.9
		    set rholowers lower
		    set rhoup 0.9
		    set rhouppers upper
		    if {[if_parameter_exists $pname]} {
			if {"" != [parameter $pname fixlower]} {
			    set rholow [parameter $pname fixlower]
			    set rholowers fixlower
			}
			if {"" != [parameter $pname fixupper]} {
			    set rhoup [parameter $pname fixupper]
			    set rhouppers fixupper
			}
		    }
		    parameter -insert $pname = 0 $rholowers $rholow \
			$rhouppers $rhoup
		    set qindex 1
		    while {[if_parameter_exists rhoq$qindex\_$i$j]} {
			parameter delete rhoq$qindex\_$i$j
			incr qindex
		    }
		}
	    }
	}
    }

    for {set it [expr $ntraits - 1]} {$it > -1} {incr it -1} {

	set tr [lindex $ts $it]

	set suf ""
	if {$ntraits > 1} {
	    set suf \($tr\)
	}

# e2 and h2r

	set h2rlow 0
	set h2rlowers lower

	set h2rup 1
	set h2ruppers upper

	set e2low [e2lower]
	set e2lowers lower

	set e2up 1
	set e2uppers upper

	if {[if_parameter_exists h2r$suf]} {
	    if {"" != [parameter h2r$suf fixlower]} {
		set h2rlow [parameter h2r$suf fixlower]
		set h2rlowers fixlower
	    }
	    if {"" != [parameter h2r$suf fixupper]} {
		set h2rup [parameter h2r$suf fixupper]
		set h2ruppers fixupper
	    }
	}
	if {[if_parameter_exists e2$suf]} {
	    if {"" != [parameter e2$suf fixlower]} {
		set e2low [parameter e2$suf fixlower]
		set e2lowers fixlower
	    }
	    if {"" != [parameter e2$suf fixupper]} {
		set e2up [parameter e2$suf fixupper]
		set e2uppers fixupper
	    }
	}

	parameter -insert h2r$suf $h2rlowers $h2rlow $h2ruppers $h2rup
	parameter -insert e2$suf $e2lowers $e2low $e2uppers $e2up
	catch {constraint delete h2r$suf}

# Mean and SD (initialized now in c++)

	parameter -insert sd$suf
	parameter -insert mean$suf

# Remove all h2q's and their constraints and matrices
	set starting_h2qcount [h2qcount]
	if {$starting_h2qcount > 0} {
	    for {set i 1} {$i <= $starting_h2qcount} {incr i} {
		catch {matrix delete mibd$i}
		catch {constraint_remove h2q$i$suf}
		catch {parameter delete h2q$i$suf}
	    }
	}

# set starting values for e2 and h2r
# Prior to 7.1.0, this was always forced to 0.9/0.1 to start
#   Now, under certain circumstances, existing values are left alone.
#   These conditions must be satisfied:
#     a) e2 and h2r add up to 1.0
#     b) e2 is not 1.0
#   Otherwise, if 0.9/0.1 fit into boundaries, they are used
#     Otherwise error

	set sum [expr [parameter e2$suf =] + [parameter h2r$suf =]]
	set eps 1e-6
	if {([parameter e2$suf =] == 1) || \
		($sum + $eps) < 1.0 || ($sum - $eps) > 1.0} {

	    if {[parameter e2$suf upper] >= 0.9} {
		parameter e2$suf = 0.9
	    } else {
		error "h2r$suf and e2$suf do not sum to 1.0"
	    }
	    if {[parameter h2r$suf lower] <= 0.1} {
		parameter h2r$suf = 0.1
	    } else {
		error "h2r$suf and e2$suf do not sum to 1.0"
	    }
	}

    } ;# end setting parameters for each trait

# Bivariate needs phi2.gz

    if {$multi && (-1==[lsearch -exact [matrix] phi2])} {
	load matrix phi2.gz phi2
    }


# Update omega...
#   Bi/Multivariate version does not support special omega terms

    if {$trivariate && !$norhoij} {

# Multivariate (> 2)

	omega = <sd(ti)>*<sd(tj)>*( \
I*sqrt(<e2(ti)>)*sqrt(<e2(tj)>)*(tne*rhoe_ij+teq) + \
phi2*sqrt(<h2r(ti)>)*sqrt(<h2r(tj)>)*(tne*rhog_ij+teq) )

    } elseif {$multi} {

# Bivariate

	omega = <sd(ti)>*<sd(tj)>*( \
I*sqrt(<e2(ti)>)*sqrt(<e2(tj)>)*(tne*rhoe+teq) + \
phi2*sqrt(<h2r(ti)>)*sqrt(<h2r(tj)>)*(tne*rhog+teq) )

    } else {
	if {"omega = Use_polygenic_to_set_standard_model_parameterization" \
		== [omega]} {
	    omega = pvar*(phi2*h2r + I*e2)
	} else {
	    omega_remove_linkage  ;# Must be done after using h2qcount
	}
    }

# Set up constraints

    if {!$multi} {
	constraint e2 + h2r = 1
    } else {
	if {[if_parameter_exists rhog]} {
	    	catch {constraint delete rhog}
	}
	foreach tr $ts {
	    set suffix \($tr\)
	    constraint <e2$suffix> + <h2r$suffix> = 1
	}
	set qindex 1
	while {[if_parameter_exists rhoq$qindex]} {
	    parameter delete rhoq$qindex
	    incr qindex
	}
    }

# Covariance matrices aren't diagonal now
    option cmdiagonal 0

# Check for discrete if requested

    if {$check_discrete} {
	setup_discrete
    }

    return ""
}

#
# Discrete setup...
# check if trait is discrete, and, if so, constrain SD to 1 and ensure
# phi2 is loaded
#
proc setup_discrete {} {
    set traits [trait]
    set ntraits [llength $traits]
    set found 0
    foreach tr $traits {
	set rstats [stats $tr -q -return]
	set discrete [stats_get $rstats discrete]
	if {$discrete} {
	    set found 1
	    if {$ntraits == 1} {
		constraint sd = 1
		parameter sd = 1 lower 0 upper 2
	    } else {
		constraint sd($tr) = 1
		parameter sd($tr) = 1 lower 0 upper 2
	    }
	}
    }
    if {$found && 0 > [lsearch [string tolower [matrix]] phi2]} {
	matrix load phi2.gz phi2
    }
}

    

	    
	



#
# Constraint editing procedures
#   SOLAR procedures should use these procedures to "edit" constraints as
#   needed rather than just deleting and replacing all prior constraints
#
# constraint_include ensures vcomp is included in a constraint which
#   sums to a nonzero value (this would be the e2 + ... = 1 constraint)
#   the base term defaults to "e2" but also may be caller specified to
#   allow multivariate terms such as e2(trait)

proc constraint_include {vcomp {basevc e2}} {

# If component names contain (), must wrap with <>

    if {-1 != [string first \( $vcomp]} {
	set vcomp <$vcomp>
	set basevc <$basevc>
    }

# convert to lower case for comparisons

    set lvcomp [string tolower $vcomp]
    set lbasevc [string tolower $basevc]

#   puts "names are $lvcomp and $lbasevc"

    set cons [constraint command]

# If currently constrained to zero, delete those constraints

    while {0 != [set zero_constraint [is_constrained_to_zero $lvcomp]]} {
#	puts "deleting constraint"
	constraint delete $zero_constraint
    }

# See if currently constrained to non-zero value
#   If so, we're finished

    if {0 != [is_constrained_to_nonzero $lvcomp]} {
	return ""
    }

# Find constraint(s) with basevc, add vcomp to end

    set cons [constraint command]
    set cnum 0
    set done 0
    foreach con $cons {
	set con [string tolower $con]
	incr cnum
	if {-1 != [string first $lbasevc $con]} {
	    regsub = $con "+ $vcomp =" new_const
#	    puts "deleting constraint $cnum"
	    constraint delete $cnum
#	    puts "making constraint $new_const"
	    eval $new_const
	    set done 1
	    break
#	} else {
#	    puts "constraint did not match: $con"
	}
    }

# No constraint with e2, so make one

    if {!$done} {
	if {0==[string compare $basevc $vcomp]} {
	    constraint $basevc = 1
	} else {
#	    puts "making new constraint $basevc + $vcomp = 1"
	    constraint $basevc + $vcomp = 1
	}
    }
    return ""
}

# solar::fix
#
# Purpose:  Constrain a parameter to its current value
#
# Usage:    fix <name>    ; Name is the name of the parameter you want to fix
#
# Example:  fix h2r

proc fix {pname} {
constraint $pname = [parameter $pname =]
}


# This is unnecessary and depricated now
# The constraint command itself now does this, and more...
proc constrain_to_zero {param} {
    if {![is_constrained_to_zero $param]} {
	constraint $param = 0
    }
    return ""
}

# find_simple_constraint returns the value a parameter is constrained to
# in a "simple" constraint (containing no other terms, e.g. h2r = 1)
# If there is no such constraint, an error is raised, so this is normally
# included in a "catch" statement.

proc find_simple_constraint {param} {
    set cons [constraint command]
    foreach con $cons {
	set con [remove_whitespace $con]
	set con [string range $con [string length constraint] end]
	set eindex [string first = $con]
	set lexp [string range $con 0 [expr $eindex - 1]]
# Remove <> if found in constraint specification but not parameter spec
	if {[string_imatch [string index $lexp 0] < ] && \
		[string_imatch [string range $lexp end end] > ] && \
		![string_imatch [string index $param 0] < ]} {
	    set lexp [string range $lexp 1 [expr [string length $lexp] - 2]]
	}
	if {[string_imatch $param $lexp]} {
	    return [string range $con [expr $eindex + 1] end]
	}
    }
    error "Parameter $param has no simple constraint"
}

#
# is_constrained procs return 0 if not constrained to zero/nonzero, or
#   constraint number (1..N) of the first constraint to zero/nonzero
#     Note: both could be true (either constraint could have other terms)
#
# WARNING!  These scripts, based on fairly simple text comparison, have
# some pretty terrible bugs.  For example, if <bage*sex> is constrained,
# it will appear that bage is constrained also.  It is recommended that
# these be phased out; current usage with standard variance components is
# probably OK.
#

proc is_constrained_to_zero {param} {
    return [is_constrained $param 0]
}

proc is_constrained_to_nonzero {param} {
    return [is_constrained $param 1]
}

proc is_constrained {param nonzero} {
# Case insensitivity
    set param [string tolower $param]
    set cons [string tolower [constraint command]]

    set found_constraint 0
    set cnum 0
    foreach con $cons {
	incr cnum
	if {-1<[string first $param $con]} {
	    set eindex [string first = $con]
	    if {$eindex>-1} {
		set rexp [string range $con [expr $eindex + 1] end]
		if {1==[scan $rexp %f rval]} {
		    if {$nonzero} {
			if {$rval != 0.0} {
			    set found_constraint $cnum
			    break
			}
		    } else {
			if {$rval == 0.0} {
			    set found_constraint $cnum
			    break
			}
		    }
		}
	    }
	}
    }
    return $found_constraint
}

# Remove constraint references to a particular parameter
#   Constraints with JUST that parameter are deleted
#   Constraints with multiple parameters are edited

proc constraint_remove {vcomp} {

#   puts "ENTERING constraint_remove"

    set hit 1
# Each time we delete a constraint, we must restart since they are reordered
    while {$hit} {
	set hit 0
	set cons [constraint command]
	set cnum 0
	foreach con $cons {
	    incr cnum
# Remove leading word "constraint"
	    set con [lrange $con 1 end]
# See if our vcomp is used
	    if {-1<[string first $vcomp $con]} {
		set iplus [string first + $con]
		if {$iplus==-1} {
# This constraint has only one term--our vcomp.  Delete it.
		    constraint delete $cnum
		    set hit 1
		    break
		} else {
# Our vcomp is being added to something else.  Excise it.

# Check if this vc contains parens, if so, adapt for pattern usage
# because parens are a metacharacter

		    catch {
		    if {-1 != [string first \( $vcomp]} {
			regsub "\\(" $vcomp "\\\\(" vcomp2
			regsub "\\)" $vcomp2 "\\\\)" vcomp3
#			puts "vcomp2 is $vcomp2"
#			puts "vcomp3 is $vcomp3"
			set vcomp $vcomp3
		    }
		    }

#		    set target "( )*\\+( )*$vcomp"
#		    puts "target is $target"
		    if {0<[regsub -nocase "( )*\\+( )*$vcomp" $con "" \
			    newcon]} {
			constraint delete $cnum
			eval constraint $newcon
			set hit 1
			break
# If our vcomp comes first, a slightly different regexp is required
		    } elseif {0<[regsub -nocase "( )*$vcomp\( )*\\+( )*" $con \
			    "" newcon]} {
			constraint delete $cnum
			eval constraint $newcon
			set hit 1
			break
		    } else {
			error "Can't cope with constraint $con"
		    }
		}
	    }
	}
    }
    return ""
}


#
# suspend2constrain changes suspended covariates to constrained betas
#   covariate suspension is an optimization, but users prefer to
#   see constrained betas.  Eventually, constraining will automatically
#   become suspension internally, explicit suspension will be phased out,
#   so this shouldn't be needed.
#
proc suspend2constrain {} {
    set covarlist [covariates -applicable]
    set betalist [covariates -betanames]
    for {set i 0} {$i < [llength $covarlist]} {incr i} {
	set tcovar [lindex $covarlist $i]
	set tbeta [lindex $betalist $i]
	if {-1 != [string first "Suspended\[" $tcovar]} {
	    set covarname [string range $tcovar 10 \
		    [expr [string length $tcovar] - 2]]
	    covariate restore $covarname
	    parameter $tbeta = 0
	    if {-1 != [string first * $tbeta] || \
		    -1 != [string first ^ $tbeta] || \
		    -1 != [string first - $tbeta] || \
		    -1 != [string first "(" $tbeta]} {
		constrain <$tbeta> = 0
	    } else {
		constrain $tbeta = 0
	    }
	    global SOLAR_constraint_tol
	    if {[parameter $tbeta lower] > 0} {
		parameter $tbeta lower -$SOLAR_constraint_tol
	    }
	    if {[parameter $tbeta upper] < 0} {
		parameter $tbeta upper $SOLAR_constraint_tol
	    }
	}
    }
}


#
# Omega editing functions
#   SOLAR procedures should use these procedures to "edit" the omega
#   equation rather than just setting it to some new expression.
#


proc omega_remove_linkage {} {
    return [omega_remove h2q H2q h2Q H2Q]
}

proc omega_add {vcterm} {
    set current_omega [omega]
    if {")" == [string range $current_omega end end]} {
	set last [expr [string length $current_omega] - 2]
	set new_omega [string range $current_omega 0 $last]
	set new_omega "$new_omega + $vcterm)"
    } else {
	set new_omega "$current_omega + $vcterm"
    }
    eval $new_omega
    return $new_omega
}

proc omega_remove {args} {
    set omega_list [split [omega] ""]
    set new_omega ""
    set operator " "
    set term ""
    set after_first_paren 0
    set after_first_term 0
    foreach char $omega_list {

# Copy everything until the first left paren
	if {!$after_first_paren} {
	    set new_omega "$new_omega$char"
	    if {"(" == $char} {
		set after_first_paren 1
	    }
	    continue
	}

# variance components are separated by + (no other +'s allowed)

	if {$char == "+"} {
	    set remove_term 0
	    foreach arg $args {
		if {-1 != [string first $arg $term]} {
		    set remove_term 1
		    break
		}
	    }
	    if {!$remove_term} {
		set new_omega "$new_omega$term"
		set after_first_term 1
	    }
	    if {$after_first_term} {
		set term "+"
	    } else {
		set term ""
	    }
	} else {
	    set term "$term$char"
	}
    }

# decide what to do with last term

    if {"" != $term} {
	set remove_term 0
	foreach arg $args {
	    if {-1 != [string first $arg $term]} {
		set remove_term 1
		break
	    }
	}
	if {!$remove_term} {
	    set new_omega "$new_omega$term"
	}
    }

# Add final parentheses

    set lparens [char_count $new_omega "("]
    set rparens [char_count $new_omega ")"]
    if {$lparens > $rparens} {
	if {" " == [string range $new_omega end end]} {
	    set last [expr [string length $new_omega] - 2]
	    set new_omega [string range $new_omega 0 $last]
	}
	while {$lparens > $rparens} {
	    set new_omega "$new_omega)"
	    incr rparens
	}
    }
    eval $new_omega
    return $new_omega
}

# solar::epistasis
#
# Purpose:  Use command:  multipoint -epistasis <N>
#
# Usage:    multipoint -epistasis <N>  (<N> is the mibd index of interest)
#
# The 'epistasis' command itself is reserved for future use.
# -

proc epistasis {} {
    return [helpscript epistasis]
}


#
# Purpose: Set up analysis of epistasis
#
# Usage:  epistasis          ;# Set up epistasis for first 2 currently loaded
#                            ;#   matrices
#         epistasis -delete  ;# Delete epistasis parameters
#
# Note:   The matrices may loaded using the 'linkmod' or 'multipoint' commands.
#-


proc epistasis-future {args} {
    error "This is not yet supported"
    if {2 > [h2qcount]} {
	error "Two ibd/mibd matrices must already have been loaded"
    }
    if {[if_parameter_exists]} {
    }
}


# solar::h2power --
#
# Purpose:  Perform heritability power calculations
#
# Usage:    h2power [-prev] [-grid {<from> <to> <incr>}] [-data <fieldname>]
#                   [-nreps <nreps>] [-seed <seed>] [-overwrite] [-plot]
#                   [-nosmooth]
#
#           h2power -restart [-grid {<from> <to> <incr>}] [-nreps <nreps>]
#                   [-plot] [-nosmooth]
#
#
#           This command performs a power calculation for the currently
#           loaded pedigree, with the following default assumptions:
#
#               (1) the trait to be studied is either quantitative or
#                   dichotomous (e.g. affected/unaffected)
#
#               (2) the trait to be studied is influenced by additive
#                   genetics
#
#               (3) all pedigree members will be phenotyped for the trait
#                   to be studied (unless the -data option is used to
#                   exclude those individuals who will not have phenotypic
#                   data; see the description of this option below)
#
#           Simulation is used to estimate the frequency with which one
#           would expect to obtain a significantly non-zero estimate of
#           heritability given that a specified fraction of the phenotypic
#           variance is due to additive genetics.  Twice the difference in
#           the loglikelihoods of the polygenic and sporadic models is
#           asymptotically distributed as a 1/2:1/2 mixture of a chi-square
#           random variate with one degree of freedom and a point mass at 0.
#           A result is considered significant if the probability of
#           obtaining the observed chi-square value, in the absence of a
#           genetic effect, is less than or equal to .05.
#
#           The default is to perform 10 replicates of the simulation for
#           each heritability in the range .01, .02, .03, ..., .99.  For
#           each replicate, a polygenic model is fitted to the simulated
#           data, and the resulting heritability estimate and chi-square
#           statistic are recorded.  The observed chi-squares are converted
#           to power, i.e. the power to detect the corresponding observed
#           heritability at a significance level of .05.
#
#           The following options give the user some control over the power
#           calculation procedure:
#
#               -prev     If the trait to be studied is dichotomous, SOLAR
#                         will assume the existence of an unobserved liability
#                         distribution. Individuals with liabilities above
#                         some threshold value will be "affected", i.e. they
#                         will have the larger of the two trait values (for
#                         example, a 1 for a 0/1 trait.) The -prev option
#                         is used to specify the "disease" prevalence, or
#                         fraction of individuals who are "affected", which
#                         in turn determines the liability threshold.
#
#               -grid     Specify the set of heritabilities for which power
#                         will be computed. At each grid point, trait data
#                         having that expected heritability are simulated,
#                         sporadic and polygenic models are fitted to the
#                         data, and the loglikelihoods of the models are
#                         compared. The observed chi-square test statistics
#                         are averaged to obtain the expected chi-square
#                         value for that heritability. The grid is given by
#                         a set of three numbers enclosed in curly braces:
#
#                             {<from> <to> <incr>}
#
#                         where <from> is the starting heritability, <to>
#                         is the last heritability considered, and <incr>
#                         is the interval between grid points.  If the
#                         desired grid consists of a single effect size,
#                         the three-number list can be replaced by that
#                         single number and curly braces are not required.
#
#               -data     Exclude individuals from the power calculation
#                         who are missing data for phenotype <fieldname>.
#
#               -nreps    Perform <nreps> simulations at each grid point.
#                         The default number of replicates is 100.
#
#               -seed     Set the random number generator seed.  The default
#                         is to set the seed based on the date and time.
#
#               -plot     At the end of the power calculations, display a
#                         plot of power versus QTL heritability.  To display
#                         this plot for a previously completed calculation,
#                         use the command "plot -h2power".
#
#               -nosmooth By default, the power curve is smoothed by fitting
#                         a line through the observed chi-square values as
#                         a function of the heritability squared prior to
#                         converting the chi-square values to power.  This
#                         option turns the smoothing off.
#
#               -overwrite (or -ov)  Overwrite the results of a previous
#                                    power calculation.
#
#               -restart (or -r)     Restart a power calculation.
#
#
# Notes:    It is possible to change the grid of heritabilities and the number
#           of replicates when restarting a calculation.  The calculation
#           will not be restarted if a grid is chosen that does not include
#           all the points in the previously specified grid unless the
#           -overwrite option is included, in which case the simulation
#           replicates for any extra grid points are discarded.  Similarly,
#           the -overwrite option is necessary if fewer replicates are
#           requested than were done previously, in which case any extra
#           replicates are discarded.
#
#           The following files are created:
#
#               h2power.out   A space-delimited file containing a line for
#                             each grid point in the format X Y, which is
#                             suitable for input to plotting packages such
#                             as xmgr.  The first (or X) column contains the
#                             heritability.  The second (or Y) column contains
#                             the power.
#
#               h2power.info  Stores the various options selected along with
#                             the chi-square statistic, averaged over the
#                             replicates, at each grid point.
#
#               h2power.chis  Stores the results of the simulation replicates
#                             run at each grid point.  This file, along with
#                             h2power.info, is used to restart an interrupted
#                             power calculation.
#
#            During a power calculation, various files named "simqtl.*" are
#            created along with a trait directory named "simqt". These will
#            be removed at the end of the run.
#

proc h2power {args} {

    set eps 1e-7

    set data ""
    set grid ""
    set h2_from 0
    set h2_to 0
    set h2_incr 0.01
    set prev ""
    set nreps ""
    set seed ""
    set overwrite 0
    set restart 0
    set plot 0
    set nosmooth 0
    set noavg 0

    set badargs [read_arglist $args -grid grid -data data -nreps nreps \
                   -seed seed -restart {set restart 1} -r {set restart 1} \
                   -overwrite {set overwrite 1} -ov {set overwrite 1} \
                   -plot {set plot 1 } -nosmooth {set nosmooth 1} -prev prev]

    if {$badargs != ""} {
        error "Invalid h2power command: bad arguments: $badargs"
    }

    if {$restart} {
        if {[catch {set rstf [open h2power.info r]}]} {
            error \
            "Can't restart power calculations: file h2power.info not found"
        }

        if {$data != ""} {
            puts "The -data option is ignored on a restart."
            set data ""
        }

        set record [gets $rstf]
        if {[lindex $record 0] == "data"} {
            set data [lindex $record 2]
            set phenfile_name [lindex $record 8]
            set vnum [verbosity -number]
            verbosity min
            phenotypes load $phenfile_name
            verbosity $vnum
            set record [gets $rstf]
        }

        if {[lindex $record 0] == "nreps"} {
            set onreps [lindex $record 2]
            if {$nreps == ""} {
                set nreps $onreps
            }
        } else {
            error \
            "Can't restart power calculations: file h2power.info is corrupted"
        }

        set record [gets $rstf]
        if {[lindex $record 0] == "grid"} {
            set oh2_from [lindex $record 2]
            set oh2_to [lindex $record 3]
            set oh2_incr [lindex $record 4]
            if {$oh2_from == 0 && $oh2_to == 0} {
                set oh2_from .01
                set oh2_to .99
                set oh2_incr .01
                set noavg 1
            }
            if {$grid == ""} {
                set grid "$oh2_from $oh2_to $oh2_incr"
            }
            if {[llength $record] == 8 && [lindex $record 5] == "prev"} {
                set prev [lindex $record 7]
            }
        } else {
            error \
            "Can't restart power calculations: file h2power.info is corrupted"
        }
        close $rstf

    } else {
        if {[file exists h2power.info] && !$overwrite} {
            error \
            "Power calculations have already been run. Use -overwrite option."
        }
    }

    if {$grid != ""} {
        if {[llength $grid] != 1 && [llength $grid] != 3} {
            error \
    "You must specify the h2 grid with a list {<from> <to> <incr>} or a single constant value."
        }

        set h2_from [lindex $grid 0]
        if {[scan $h2_from "%f" tmp] != 1 || $h2_from < 0 || $h2_from > 1} {
            error "Starting h2 in grid must be between 0 and 1."
        }

        if {[llength $grid] == 1} {
            set h2_to $h2_from
            set h2_incr 1

        } else {
            set h2_to [lindex $grid 1]
            if {[scan $h2_to "%f" tmp] != 1 || $h2_to < 0 || $h2_to > 1} {
                error "Ending h2 in grid must be between 0 and 1."
            }
            if {$h2_to < $h2_from} {
                error "Ending h2 in grid cannot be less than the starting h2."
            }

            set h2_incr [lindex $grid 2]
            if {[scan $h2_incr "%f" tmp] != 1 || $h2_incr <= 0} {
                error "Increment in h2 grid must be greater than 0."
            }
        }

        if {$restart && !$overwrite} {
            set h2 $h2_from
            set oh2 $oh2_from
            set done 0
            while {!$done} {
                if {$oh2 < [expr $h2 - $eps]} {
                    error \
"The requested grid does not include all of the existing grid \{$oh2_from $oh2_to $oh2_incr\}.
Add -overwrite if you really want to discard the extra grid points."
                } elseif {$oh2 > [expr $h2_to + $eps]} {
                    error \
"The requested grid does not include all of the existing grid \{$oh2_from $oh2_to $oh2_incr\}.
Add -overwrite if you really want to discard the extra grid points."
                } else {
                    set done 1
                    if {$oh2 > [expr $h2 - $eps] && $oh2 < [expr $h2 + $eps]} {
                        set oh2 [expr $oh2 + $oh2_incr]
                        set done 0
                    }
                    if {$h2 < [expr $h2_to - $eps]} {
                        set h2 [expr $h2 + $h2_incr]
                        set done 0
                    }
                    if {$oh2 > [expr $oh2_to + $eps]} {
                        set done 1
                    }
                }
            }
        }
    }

    if {$h2_from == 0 && $h2_to == 0} {
        set h2_from .01
        set h2_to .99
        set h2_incr .01
        set noavg 1
        if {$nreps == ""} {
            set nreps 10
        }
    }

    if {$prev != ""} {
        if {[scan $prev "%f" tmp] != 1 || $prev <= 0 || $prev >= 1} {
            error "Prevalence must be greater than 0 and less than 1."
        }
    }

    if {$nreps == ""} {
        set nreps 100
    }

    if {[scan $nreps "%d" tmp] != 1 || $nreps <= 0} {
        error "Number of replicates must be a positive integer."
    }

    if {$seed == ""} {
        set seed 0
    }

    if {$restart && $nreps < $onreps && !$overwrite} {
        error \
"The requested number of replicates is less than the existing number ($onreps).
Add -overwrite if you really want to discard the extra replicates."
    }

    if {[scan $seed "%d" tmp] != 1 || $seed < 0} {
        error "Random number seed must be a non-negative integer."
    }

    set phenfile_name ""
    if {[catch {set infof [open phenotypes.info r]}]} {
        if {$data != ""} {
            error "A phenotypes file has not been loaded."
        }

    } else {
        gets $infof phenfile_name
        close $infof
        if {$data != ""} {
            if {$phenfile_name == "simqtl.phn"} {
                error \
"The currently loaded phenotypes file is named \"simqtl.phn\", which is a
filename reserved for use by the power command. You will have to rename
the phenotypes file and then reload it."
            }
            if {[catch {set phenfile [tablefile open $phenfile_name]}]} {
                error "Can't find phenotypes file $phenfile_name"
            }
            if {![tablefile $phenfile test_name $data]} {
                tablefile $phenfile close
                error "The phenotypes file does not contain a field named $data."
            }
            tablefile $phenfile start_setup
            tablefile $phenfile setup $data
            set navail 0
            while {"" != [set record [tablefile $phenfile get]]} {
                if {$record != "{}"} {
                    incr navail
                }
            }
            tablefile $phenfile close
        }
    }

    set outf [open h2power.info w]
    if {$data != ""} {
        puts $outf "data = $data  navail = $navail  phenf = $phenfile_name"
    }
    puts $outf "nreps = $nreps  seed = $seed"
    if {$noavg} {
        puts -nonewline $outf "grid = 0 0 0.01"
    } else {
        puts -nonewline $outf "grid = $h2_from $h2_to $h2_incr"
    }
    if {$prev != ""} {
        puts -nonewline $outf "  prev = $prev"
    }
    puts $outf ""
    flush $outf

    drand $seed

    if {$restart} {
        exec mv h2power.chis h2power.chis.tmp
        set ochisf [open h2power.chis.tmp r]
        set oh2 $oh2_from
    }

    set chisf [open h2power.chis w]
    set h2 $h2_from
    set nh2 0
    set schi 0
    set sh2 0
    set rep1 0

    while {$h2 <= [expr $h2_to + $eps]} {

        if {$restart && $oh2 < [expr $h2 - $eps]} {
            while {$rep1 < $onreps && {} != [set chirec [gets $ochisf]]} {
                incr rep1
            }
            set rep1 0
            set oh2 [expr $oh2 + $oh2_incr]
            continue

        } elseif {$restart && $oh2 > [expr $h2 - $eps] && \
                  $oh2 < [expr $h2 + $eps]} {
            while {$rep1 < $onreps && {} != [set chirec [gets $ochisf]]} {
                if {$rep1 < $nreps} {
                    set chi [lindex $chirec 1]
                    if {$noavg} {
                        set chis($nh2) [lindex $chirec 1]
                        set o_h2($nh2) [lindex $chirec 2]
                        puts $chisf [format "%5d%12.6f%12.6f" [expr $nh2 + 1] \
                                     [lindex $chirec 1] [lindex $chirec 2]]
                        incr nh2
                    } else {
                        set schi [expr $schi + $chi]
                        set sh2 [expr $sh2 + [lindex $chirec 2]]
                        puts $chisf [format "%5d%12.6f%12.6f" [expr $rep1 + 1] \
                                     [lindex $chirec 1] [lindex $chirec 2]]
                    }
                }
                incr rep1
            }
            set th2 $h2
            set nexth2 [expr $h2 + $h2_incr]
            set nextoh2 [expr $oh2 + $oh2_incr]

        } else {
            set th2 $h2
            set nexth2 [expr $h2 + $h2_incr]
            if {$restart} {
                set nextoh2 $oh2
            }
        }

        if {$data != ""} {
            simqtl -freq .5 -mean {100 100 100} -sdev 10 -h2r $th2 -cov $data
        } else {
            simqtl -freq .5 -mean {100 100 100} -sdev 10 -h2r $th2
        }

        for {set i $rep1} {$i < $nreps} {incr i} {
            if {$phenfile_name != ""} {
                phenotypes load $phenfile_name
            }
            simqtl
            if {$prev != ""} {
                mkdisc $prev
            }
            phenotypes load simqtl.phn
            model new
            trait simqt

            spormod
            option standerr 0
            set errmsg [maximize_quietly last]
            if {$errmsg != ""} {
                if {[string compare [verbosity] "verbosity min"] != 0} {
                    puts "    *** Error maximizing, rep $i"
                }
                set i [expr $i - 1]
                continue
            }
            set slike [loglike]

            polymod
            option standerr 0
            set errmsg [maximize_quietly last]
            if {$errmsg != ""} {
                if {[string compare [verbosity] "verbosity min"] != 0} {
                    puts "    *** Error maximizing, rep $i"
                }
                set i [expr $i - 1]
                continue
            }

            set chi [expr 2*([loglike] - $slike)]
            if {[string compare [verbosity] "verbosity min"] != 0} {
                if {$noavg} {
                    puts [format "%5d%12.6f%12.6f" [expr $nh2 + 1] $chi \
                          [parameter h2r =]]
                } else {
                    puts [format "%5d%12.6f%12.6f" [expr $i + 1] $chi \
                          [parameter h2r =]]
                }
            }
            if {$noavg} {
                puts $chisf [format "%5d%12.6f%12.6f" [expr $nh2 + 1] $chi \
                             [parameter h2r =]]
            } else {
                puts $chisf [format "%5d%12.6f%12.6f" [expr $i + 1] $chi \
                             [parameter h2r =]]
            }
            flush $chisf

            if {$noavg} {
                set chis($nh2) $chi
                set o_h2($nh2) [parameter h2r =]
                incr nh2
            } else {
                set schi [expr $schi + $chi]
                set sh2 [expr $sh2 + [parameter h2r =]]
            }
        }

        if {!$noavg} {
            set chis($nh2) [expr $schi/$nreps]
            puts [format "h2 = %g  EChi2 = %.6g" $th2 $chis($nh2)]
            puts $outf [format "h2 = %g  EChi2 = %.6g" $th2 $chis($nh2)]
            flush $outf
            incr nh2
            set schi 0
            set sh2 0
        }

        if {$restart} {
            set oh2 $nextoh2
        }
        set h2 $nexth2
        set rep1 0
    }

    if {$restart} {
        close $ochisf
        file delete h2power.chis.tmp
    }

    close $chisf
    close $outf

    if {$nosmooth} {
        set outf [open "|[usort] -nu > h2power.out" w]
    } else {
        set xx 0
        set xy 0
    }

    set h2 $h2_from
    set nh2 0
    set last_h2 [expr $h2_to + $eps]
    while {$h2 <= $last_h2} {
        if {$noavg} {
            for {set i 0} {$i < $nreps} {incr i} {
                set chi $chis($nh2)
                if {$chi < 0} {
                    set chi 0
                }
                if {$nosmooth} {
                    puts $outf "[format %.4g $o_h2($nh2)] \
                                 [expr 1 - [chinc 2.705542 1 $chi]]"
                } elseif {$chi < 13.6796} {
                    set xx [expr $xx + pow($o_h2($nh2),4)]
                    set xy [expr $xy + pow($o_h2($nh2),2)*$chi]
                }
                incr nh2
            }
        } else {
            set chi $chis($nh2)
            if {$chi < 0} {
                set chi 0
            }
            if {$nosmooth} {
                puts $outf "[format %.4g $h2] \
                             [expr 1 - [chinc 2.705542 1 $chi]]"
            } elseif {$chi < 13.6796} {
                set xx [expr $xx + pow($h2,4)]
                set xy [expr $xy + pow($h2,2)*$chi]
            }
            incr nh2
        }
        set h2 [expr $h2 + $h2_incr]
    }

    if {!$nosmooth} {
        set outf [open h2power.out w]
        set last [expr int(100*$h2_to)]
        for {set i 0} {$i < $last} {incr i} {
            puts $outf [format "%.4g %.6g" [expr $i*.01] \
                [expr 1 - [chinc 2.705542 1 [expr pow($i*.01,2)*$xy/$xx]]]]
        }
    }
    close $outf

    if {$plot} {
        plot_h2power
    }

    exec rm -rf simqt
    delete_files_forcibly simqtl.qtl simqtl.par simqtl.dat

    if {$phenfile_name != ""} {
        delete_files_forcibly simqtl.phn
        set vnum [verbosity -number]
        verbosity min
        catch {phenotypes load $phenfile_name}
        verbosity $vnum
    }
}


proc mkdisc {prev} {
    set ifp [tablefile open simqtl.phn]
    tablefile $ifp start_setup
    set qtfld [tablefile $ifp setup simqt]
    set n 0
    set qts {}
    while {{} != [set rec [tablefile $ifp get]]} {
        if {[lindex $rec 0] != ""} {
            lappend qts [lindex $rec 0]
            incr n
        }
    }
    tablefile $ifp close
    set qts [lsort -real -decreasing $qts]
    set thresh [lindex $qts [expr int($prev*$n)]]

    set ofp [open simqtl.tmp w]
    set ifp [tablefile open simqtl.phn]
    tablefile $ifp start_setup
    set flds [tablefile $ifp names]
    tablefile $ifp setup [lindex $flds 0]
    puts -nonewline $ofp [lindex $flds 0]
    for {set i 1} {$i < [llength $flds]} {incr i} {
        tablefile $ifp setup [lindex $flds $i]
        puts -nonewline $ofp ",[lindex $flds $i]"
    }
    puts $ofp ""

    while {{} != [set rec [tablefile $ifp get]]} {
        if {[lindex $rec $qtfld] == ""} {
            continue
        }
        puts -nonewline $ofp [lindex $rec 0]
        for {set i 1} {$i < [llength $rec]} {incr i} {
            if {$i == $qtfld} {
                if {[lindex $rec $i] > $thresh} {
                    puts -nonewline $ofp ",1"
                } else {
                    puts -nonewline $ofp ",0"
                }
            } else {
                puts -nonewline $ofp ",[lindex $rec $i]"
            }
        }
        puts $ofp ""
    }
    tablefile $ifp close
    close $ofp
    exec mv simqtl.tmp simqtl.phn
}


# solar::plot_h2power -- private
#
# Purpose:  Implements "plot -h2power"
#
# Usage:    plot_h2power [-title <plot_title>]
#
# -

proc plot_h2power {args} {
    set title "Power"
    read_arglist $args -title title -h2power {set ignore_this 0}

    if {[catch {tclgr open} errmsg]} {
	if {[string compare $errmsg \
		"tclgr session already opened from this solar session"]} {
	    error $errmsg
	}
    }

    if {[catch {set inf [open h2power.out r]}]} {
        error "Cannot open h2power.out"
    }

    set x {}
    set y {}
    while {-1 != [gets $inf line]} {
        lappend x [lindex $line 0]
        lappend y [lindex $line 1]
    }
    close $inf

    set npts [llength $x]
    if {$npts == 0} {
	error "No data in h2power.out ... can't plot"
    }

    tclgr send kill graphs
    tclgr send clear line
    tclgr send clear string
    tclgr send focus g0

    tclgr send title \"$title\"

    tclgr send world xmin 0
    set max_x [lindex $x [expr [llength $x] - 1]]
    set max_x [expr ceil(10*$max_x)/10]
    tclgr send world xmax $max_x
    tclgr send xaxis tick major 0.1
    tclgr send xaxis tick minor 0.05

    tclgr send world ymin 0
    tclgr send world ymax 1
    tclgr send yaxis tick major 0.1
    tclgr send yaxis tick minor 0.05

    global env
    if {[file exists $env(SOLAR_LIB)/h2power.gr]} {
	set mpathname [glob $env(SOLAR_LIB)/h2power.gr]
	tclgr send read \"$mpathname\"
    }
    if {[file exists ~/lib/h2power.gr]} {
	set mpathname [glob ~/lib/h2power.gr]
	tclgr send read \"$mpathname\"
    } 
    if {[file exists h2power.gr]} {
	tclgr send read \"h2power.gr\"
    } 

    for {set i 0} {$i < $npts} {incr i} {
	tclgr send s0 color 2
	tclgr send g0.s0 point "[lindex $x $i],[lindex $y $i]"
    }

    tclgr send redraw
}


# solar::house --
#
# Purpose:  Enable analysis of household effects
#
# Usage:    house            ; enable analysis of household effects
#           house -suspend   ; temporarily suspend household effects
#                            ; (the c2 parameter is constrained to zero)
#           house -delete    ; delete household effects (c2 is deleted)
#
# Examples:
#
#  ** new model ** Note you must give house command after trait or model new
#
#         solar> model new
#         solar> trait weight
#         solar> covar age^1,2#sex
#         solar> house                       ; activates h. effects
#         solar> polygenic -screen           ; polygenic analysis w. screening
#
# ** old model ** 
#
#         solar> load model poly
#         solar> house
#         solar> polygenic
#
# Notes:  This may be used for any common environmental effects (not
#         necessarily "household" effects).  The house command changes the
#         current model as follows:
#
#             1) A parameter c2 (c stands for "common") is created
#             2) The house matrix is loaded
#             3) A c2*house term is added to the omega
#             4) The c2 parameter is added to the e2 + ... = 1 constraint
#             5) The starting value of c2 is carved away from the value of
#                e2 so that the constraint remains satisfied
#
#         The pedigree file must contain a HHID field.  If so, the
#         'load pedigree' command produces a matrix named house.gz.
#         That matrix will be used.  If house.gz is not present, this
#         command will fail, although you can map HHID to any particular
#         field in your pedigree file using the "field" command.
#
#         WARNING!  If you load a pedigree without a HHID field (or a field
#         mapped to it with the field command) a pre-existing house.gz, now
#         assumed to be obsolete, will be deleted.  This is to prevent you
#         from making the mistake of using an obsolete house.gz.
#
#         HHID can be a number or an alphanumeric name (with no internal
#         spaces or tabs) but the number 0 (zero) has a special meaning.
#         Zero indicates singleton households--each individual with HHID
#         of zero is a separate household, not associated with other
#         individuals having HHID of zero.  Blank or null has the same
#         effect as zero.
#
#         The 'house' command should be specified after such commands as
#         automodel, trait, polymod, or spormod (which initialize polygenic
#         or sporadic models) and/or just before commands which maximize
#         models such as 'polygenic,' 'maximize,' or 'multipoint.'  This
#         is because "polygenic" or "sporadic" models, by definition,
#         do not have household effects.  But the polygenic command will
#         do the household "analysis" if it detects the presence of a c2
#         parameter which is not constrained to zero.
#
#         We define the following model classes:
#
#         sporadic  (covariates only...and e2)
#           household (covariates and c2)
#           polygenic (covariates and h2r)
#             household polygenic (covariates, c2, and h2r)
#
#         To create a pure "household" model with no active genetic component,
#         give the commands "spormod" and "house" in that order after setting
#         up the trait(s) and covariate(s).
#
#         By default, if a household element is in the model, pedigrees will
#         be merged whenever individuals in separate pedigrees share the same
#         household.  The resulting groups are called "pedigree-household"
#         groups.  This may significantly increase memory requirements.
#         Pedigree merging is controlled by two SOLAR options (see the
#         option command).  The default of 1 for MergeHousePeds means that
#         pedigree merging, as just described, will be done.  This feature
#         may be changed by setting MergeHousePeds to zero prior to the
#         polygenic or maximize commands:
#
#         solar> option mergehousepeds 0
#         solar> polygenic -screen
#
#         The MergeAllPeds option combines all the pedigrees into one large
#         group if set to 1.  This is an alternative simpler method of
#         merging, but it may increase memory requirements even more.
#         
#-

proc house {args} {

# Handle arguments

    if {"" != $args} {
	if {"-suspend" == $args} {
	    return [outhouse]
	} elseif {"-delete" == $args} {
	    return [nohouse]
	}
	error "house: invalid argument"
    }

# Multiple traits?

    set ts [trait]
    set ntraits [llength $ts]

# If no model yet, make this sporadic (sporadic+house = household model)
# (Note: spormod calls "house -suspend", so house actually gets called back)

    if {$ntraits > 1} {
	set firste2 e2\([lindex $ts 0]\)
	if {![if_parameter_exists $firste2]} {
	    spormod
	}
    } else {
	if {![if_parameter_exists e2]} {
	    spormod
	}
    }

# Add house matrix (make sure only once)

    if {"" == [housematrix]} {
	catch {matrix delete house}
	matrix load house.gz house
    }

# Setup omega
#   If bivariate, also setup rhoc (required for omega)


    if {$ntraits > 2} {
	for {set i 1} {$i < $ntraits} {incr i} {
	    for {set j [expr $i + 1]} {$j <= $ntraits} {incr j} {
		set rname [catenate rhoc_ $i $j]
		parameter $rname = 0 lower -1 upper 1
		catch {constraint_remove $rname}
	    }
	}
	if {-1 == [string first \
		       house*sqrt(<c2(ti)>)*sqrt(<c2(tj)>)*(tne*rhoc_ij+teq) \
		       [omega]]} {
	    omega_add house*sqrt(<c2(ti)>)*sqrt(<c2(tj)>)*(tne*rhoc_ij+teq)
	}
    } elseif {$ntraits == 2} {
	parameter rhoc = 0 lower -1 upper 1
	catch {constraint_remove rhoc}
	if {-1 == [string first \
		house*sqrt(<c2(ti)>)*sqrt(<c2(tj)>)*(tne*rhoc+teq) \
		[omega]]} {
	    omega_add house*sqrt(<c2(ti)>)*sqrt(<c2(tj)>)*(tne*rhoc+teq)
	}
    } else {
	if {-1 == [string first c2*house [omega]]} {
	    omega_add c2*house
	}
    }
	
# For each trait, create c2 parameter, add to vc constraint, carve new value

    foreach tr $ts {

# Add c2 parameter with starting value of 0.01
# Put in variance components constraint and omega

	set suffix ""
	if {$ntraits > 1} {
	    set suffix ($tr)
	}

	parameter c2$suffix lower 0 upper 1

	constraint_include c2$suffix e2$suffix  ;# This also purges zero constraints
	if {0.0==[parameter c2$suffix =]} {
	    carve_new_value c2$suffix 0.01
	}
	if {1<[parameter e2$suffix upper]} {
	    parameter e2$suffix upper 1
	}
    }

    option cmdiagonal 0
    return ""
}


# Remove household effect from model
# Remove completely (even if partially removed before)
# Do not crash if household effect is partially or not at all present

proc nohouse {} {
    catch {matrix delete house}

    set ts [trait]
    set ntraits [llength $ts]

# Delete Rho's and remove house term from omega

    if {$ntraits > 2} {
	for {set i 1} {$i < $ntraits} {incr i} {
	    for {set j [expr $i + 1]} {$j <= $ntraits} {incr j} {
		set rname [catenate rhoc_ $i $j]
		catch {parameter $rname delete}
	    }
	}
	catch {
	    set newomega [stringsub [omega] \
		"+ house*sqrt(<c2(ti)>)*sqrt(<c2(tj)>)*(tne*rhoc_ij+teq)" ""]
	    eval $newomega
	}
    } elseif {$ntraits == 2} {
	catch {parameter rhoc delete}
	catch {
	    set newomega [stringsub [omega] \
		    "+ house*sqrt(<c2(ti)>)*sqrt(<c2(tj)>)*(tne*rhoc+teq)" ""]
	    eval $newomega
	}
    } else {
	catch {omega_remove c2}
    }

# Move variance back to e2's and delete c2's
# Edit c2 from constraints

    foreach tr $ts {
	set suffix ""
	set cname c2
	if {$ntraits > 1} {
	    set suffix \($tr\)
	    set cname <c2$suffix>
	}
	if {[if_parameter_exists c2$suffix]} {
	    parameter e2$suffix = [expr [parameter e2$suffix =] + \
		    [parameter c2$suffix =]]
	    catch {constraint_remove $cname}
	    parameter c2$suffix delete
	    make_bounds_ok_for_value e2$suffix 0 1
	}
    }
    return ""
}


# house -suspend
# IMPORTANT: DO NOT ASSUME HOUSE IS ACTIVE!
# This is called w/o seeing if model has house in the first place
proc outhouse {} {
    set ts [trait]
    set ntraits [llength $ts]

# Constrain c2's

    foreach tr $ts {
	set suffix ""
	set cname c2
	if {$ntraits > 1} {
	    set suffix \($tr\)
	    set cname <c2$suffix>
	}
	if {[if_parameter_exists c2$suffix]} {
	    parameter e2$suffix = [expr [parameter e2$suffix =] + \
		    [parameter c2$suffix =]]
	    make_bounds_ok_for_value e2$suffix 0 1
	    catch {constraint_remove $cname}
	    constraint $cname = 0
	    parameter c2$suffix = 0
	}
    }

# Edit omega and constrain rho's (if applicable)

    if {$ntraits > 2} {
	for {set i 1} {$i < $ntraits} {incr i} {
	    for {set j [expr $i + 1]} {$j <= $ntraits} {incr j} {
		set rname [catenate rhoc_ $i $j]
		if {[if_parameter_exists $rname]} {
		    parameter $rname = 0
		    constraint $rname = 0
		}
	    }
	}
	set newomega [stringsub [omega] \
	  "+ house*sqrt(<c2(ti)>)*sqrt(<c2(tj)>)*(tne*rhoc_ij+teq)" ""]
	eval $newomega

    } elseif {$ntraits == 2} {
	if {[if_parameter_exists rhoc]} {
	    parameter rhoc = 0
	    constraint rhoc = 0
	}
	set newomega [stringsub [omega] \
          "+ house*sqrt(<c2(ti)>)*sqrt(<c2(tj)>)*(tne*rhoc+teq)" ""]
	eval $newomega
    } else {
	catch {omega_remove c2}
    }

    return ""
}

#
# check_house and check_epistasis
#   return 1 if C2 parameter is active, 0 otherwise
#
proc check_house {} {
    set ts [trait]
    set ntraits [llength $ts]

    if {-1 < [string first c2 [omega]]} {
	foreach tr $ts {
	    set test c2
	    set testc c2
	    if {$ntraits > 1} {
		set test [catenate c2 ( $tr ) ]
		set testc <$test>
	    }
	    if {[if_parameter_exists $test]} {
		if {![is_constrained_to_zero $testc]} {
		    return 1
		}
	    }
	}
    }
    return 0
}


proc check_epistasis {} {
    if {[if_parameter_exists h2qe1]} {
	if {-1<[string first h2qe1 [omega]]} {
	    if {![is_constrained_to_zero h2qe1]} {
		return 1
	    }
	}
    }
    return 0
}

#
# Procedures parameter_roll_into and carve_new_value make it easier
# to add or adjust variance components which need to add up to 1.0
#
proc parameter_roll_into {old new} {
    if {0.0 != [parameter $old =]} {
	parameter $new = [expr [parameter $new =] + [parameter $old =]]
	if {[parameter $new upper] < [parameter $new =]} {
	    parameter $new upper [expr [parameter $new =] + 0.01]
	}
	parameter $old = 0.0
	parameter $old lower -0.01
    }
}

# solar::carve_new_value -- private
#
# Purpose: carve starting value for new variance component (see notes).
#
# Usage:   carve_new_value vcomp newvalue [tryfirst]
#
# Notes:
#
# carve_new_value "carves" a starting value for a new variance
# component from other (standard) variance components.  Custom
# variance components are not supported, and carve_new_value should
# not be called for custom variance components.  However, it is NOT
# assumed that all variance components are "known" or that known
# components add up to 1.0, only that when a new one is added, others
# (known ones, what else?) must be decreased by a corresponding amount.  
# (That is what "carving" means.)
#
# carve_new_value is now extended to handle bivariate:
#  target vcomp should be specified with required trait suffix
#  HOWEVER, tryfirst should be specified generically WITH NO SUFFIX
#    (suffix will be copied from vcomp)
#  
# The current value is vcomp is NOT assumed to be zero.  However it
# probably works better if it IS zero, or only a very tiny change is
# being made, or if the "tryfirst" option is likely to succeed.
#
# The entire difference is taken from the first variance component
# having sufficient distance from the natural boundaries of 0 and 1
# and which is not already constrained.  This simple algorithm is
# subject to change in the future.  perturb and linkmod use somewhat
# more sophisticated algorithms which may distribute the required
# deltas and may subtract the differences from existing parameters
# in a different order.
#
# Boundaries of changed variance components are made compatible if
# necessary, except for the new variance component.  It is assumed
# that the caller has plans to set boundaries for the new variance
# component, or leave that to solarmain, which simply defaults them
# to 0 and 1.
#
# Actual setting of the value for the new variance component must
# be done here, since allowances for a previous value are made.
# (Strictly speaking, the "new" variance component need not actually
# be new.)
# -
 
proc carve_new_value {vcomp newvalue {tryfirst e2}} {

#   puts "Entering carve_new_value"

    set s ""
    set multi 0
    set firstp [string first \( $vcomp]
    if {-1 != $firstp} {
	set lastp [string first \) $vcomp]
	set s [string range $vcomp $firstp $lastp]
#	puts "suffix is $s"
	set multi 1
    }

    set amount [expr $newvalue - [parameter $vcomp =]]

    set clist [list $tryfirst$s e2$s h2r$s h2q1$s c2$s h2qe1$s]
    for {set i 2} {$i <= [h2qcount]} {incr i} {
	lappend clist h2q$i$s
    }
    set done 0

    foreach comp $clist {
	if {$comp == $vcomp} {
	    continue
	}
#	puts "testing parameter $comp"
#
# Cannot carve from parameter that has "simple" constraint, or is
# within 0.03 of ultimate boundaries
#
	if {[if_parameter_exists $comp]} {
#	    puts "Parameter $comp exists"
	    if {[catch {find_simple_constraint $comp}]} {
#		puts "No constraint found for $comp"
		set current [parameter $comp =]
		set maxval 0.97
		if {"" != [parameter $comp fixupper]} {
		    set maxval [parameter $comp fixupper]
		}
		if {($amount > 0 && $current - $amount > 0.03) || \
		    ($amount < 0 && $current - $amount < $maxval)} {
		    set carve_value [expr $current - $amount]
#
# OK, carve value from this parameter
#
		    parameter $comp = $carve_value
		    make_bounds_ok_for_value $comp 0 1
		    set done 1
		    break
		}
	    }
	}
    }
    if {!$done} {
	error "Couldn't carve value for $vcomp from other variance components"
    }
#
# Now set target parameter
#
    parameter $vcomp = $newvalue
#
# Caller assumes responsibility for bounds.
#   Not suitable for generic solution.
#   (usually this is done automatically by SOLARMAIN)
#   parameter $vcomp lower 0 upper 1
#   make_bounds_ok_for_value $vcomp 0 1

    return ""
}

# solar::make_bounds_ok_for_value -- private
#
# Purpose: Make upper and lower bounds compatible with current value
#
# Usage:   make_bounds_ok_for_value par <lowest-possible> <highest-possible>
#
# Notes:   Boundaries >= 0.01 away from current value are unchanged.
#
#          If boundary is moved, it is moved 0.01 away from current value,
#          if possible.
#
#          Upper boundary is moved no higher than <highest-possible>.
#          Lower boundary is moved no lower than <lowest-possible>.
# -

proc make_bounds_ok_for_value {comp nlower nupper} {
    set olower [parameter $comp lower]
    set oupper [parameter $comp upper]
    set value [parameter  $comp =]
    if {$value < $olower + 0.01} {
	if {"" == [parameter $comp FixLower]} {
	    parameter $comp lower [highest $nlower [expr $value - 0.01]]
	} else {
	    puts "Warning.  Unable to lower fixed lower boundary for $comp"
	}
    }
    if {$value > $oupper - 0.01} {
	if {"" == [parameter $comp FixUpper]} {
	    parameter $comp upper [lowest $nupper [expr $value + 0.01]]
	} else {
	    puts "Warning.  Unable to raise fixed upper boundary for $comp"
	}
    }
    return ""
}


# solar::linkmod --
#
# Purpose:  Set up parameters and constraints for multipoint linkage model
#
# Usage:    linkmod [-add [-epistasis]] [-zerostart] [-2p] <ibdfile>
#                   [-cparm] [-se]
#
#                    -add means add this to previous linkage elements
#                       (otherwise, it supercedes the previous one)
#                    -zerostart is used for score analysis (see below)
#                    -2p  twopoint (ibd not mibd)
#                    -epistasis sets up epistasis parameter between new
#                         element and previous linkage element.  Use with
#                         "-add".  Not supported for bivariate.
#                    -se     Turn on standard error calculation option.
#                            The default is to turn it off.
#                    -cparm  "Custom Parameterization"  Simply replace old
#                            matrix with new matrix.  Parameters, constraints,
#                            and omega are unchanged.  A "prototype" model
#                            with suitable matrix, parameters, omega, and
#                            constraints must already be loaded.  See
#                            final note below for more information.
#                            Note: if -cparm is specified, standard errors
#                            are NOT turned off, but left in whatever state
#                            they were in when linkmod was called.
#
# Notes:    Use the -2p option for twopoint models.  linkmod2p is now
#           obsolescent (linkmod -2p is invoked when you give the linkmod2p
#           command).
#
#           A polygenic or linkage model should already have been
#           created and maximized first.  Boundaries are set around existing
#           values assuming this has been done.
#
#           Multiple linkage terms will be included if Solar_Fixed_Loci
#           is defined.  The script multipoint does this.
#
#           By default, standard error is turned off.  You may turn it on
#           again by giving the command 'option standerr 1' after running
#           linkage and before running maximize.
#
#           The -zerostart option starts the new linkage component at 0.
#           (The linkage component MUST NOT HAVE BEEN ALREADY CREATED!)
#           This is used for score test analysis.
#
#           The -cparm option requires that a prototype linkage model
#           with all required matrices, parameters, omega terms, and
#           constraints be already loaded.  Other than that, however, it
#           ATTEMPTS to be as general as possible.  However, it is
#           necessary to make one assumption regarding the name of
#           the first matrix.  If the -2p option is specified, the
#           relevant matrix that will be modified must be named
#           ibd or ibd1, ibd2, etc.  Otherwise, the relevant matrix
#           must be named mibd or mibd1, mibd2, etc.  It is the
#           ibd or mibd matrix with the highest number, if any,
#           which will be replaced.  If a second matrix column such 
#           as d7 or delta7 is included, it will be assumed to be
#           included in the replacement matrix as well.  This option
#           is used by "multipoint -cparm" and "twopoint -cparm".
# -

proc linkmod {args} {

    set add 0
    set zs 0
    set previous_h2q_value none
    set epistasis 0
    set twopoint 0
    set noparm 0
    set secalc 0

    set ibdfilename [read_arglist $args -zerostart {set zs 1} \
	    -epistasis epistasis \
	    -2p {set twopoint 1} \
	    -cparm {set noparm 1} \
	    -noparm {set noparm 1} \
      	    -se {set secalc 1} \
	    -lasth2q previous_h2q_value -add {set add 1}]
    if {1!=[llength $ibdfilename]} {
	error "Invalid or missing arguments to linkmod"
    }

# Edit options

    option cmdiagonal 0

    if {$secalc} {
	option standerr 1
    } else {
	option standerr 0
    }

# Noparm option, quick and easy

    if {$noparm} {
	set matrices [string tolower [matrix]]
	set second_matrix ""
	set index ""
	set prefix mibd
	if {$twopoint} {
	    set prefix ibd
	}
	for {set tindex 1} {1} {incr tindex} {
	    if {-1 != [lsearch $matrices $prefix$tindex]} {
		set index $tindex
	    } else {
		break
	    }
	}
	set position [lsearch $matrices $prefix$index]
	if {-1 == $position} {
	    error "$prefix$index matrix must already be loaded in prototype model"
	}
	set second [lindex $matrices [expr $position + 1]]
	if {"matrix" == $second} {
	    set second ""
	}
	eval matrix load $ibdfilename $prefix$index $second
	return ""
    }

# Normal parameter tracking and such (this gets complicated!)

    if {$add} {
	set new_h2q_index [expr 1 + [h2qcount]]
    } else {
	set new_h2q_index [expr 1 + [use_global_if_defined Solar_Fixed_Loci 0]]
    }

    set ts [trait]
    if {[llength $ts] == 1} {
	set multi 0
	set suffix ""
    } else {
	set multi [llength $ts]

# Adjust boundaries for rhoe and rhog

	if {$multi == 2} {
	    set rhog [parameter rhog =]
	    parameter rhog lower [highest -1 [expr $rhog - 0.1]]
	    parameter rhog upper [lowest 1 [expr $rhog + 0.1]]

	    set rhoe [parameter rhoe =]
	    parameter rhoe lower [highest -1 [expr $rhoe - 0.05]]
	    parameter rhoe upper [lowest 1 [expr $rhoe + 0.05]]
	} else {
	    foreach par [parameter -names] {
		if {0 == [string compare "rho" [string range $par 0 2]]} {
		    set tolerance 0.1
		    if {0 == [string compare "e" [string index $par 3]]} {
			set tolerance 0.05
		    }
		    set rho [parameter $par =]
		    parameter $par lower [highest -1 [expr $rho - $tolerance]]
		    parameter $par upper [lowest 1 [expr $rho + $tolerance]]
		}
	    }
	}
    }

# Track household parameters similarly to h2q's

    if {[check_house] && ![boundarywide]} {
	foreach tr $ts {
	    if {!$multi} {
		set pname c2
	    } else {
		set pname c2($tr)
	    }
	    if {[if_parameter_exists $pname]} {
		set c2upper [parameter $pname upper]
		set c2upper [lowest 1 [expr [parameter $pname =] + \
						  [h2q_float]]]
		parameter $pname upper $c2upper
	    }
	}
    }
	    
# Create new rhoq's If they do not already exist
    
    set RHO rhoq$new_h2q_index

    if {$multi} {
	if {$multi == 2} {
	    if {![if_parameter_exists $RHO]} {
		parameter $RHO = 0 lower -1 upper 1
	    }
	} else {
	    if {![if_parameter_exists [catenate $RHO _12]]} {
		set nts [llength $ts]
		for {set i 1} {$i < $nts} {incr i} {
		    for {set j [expr $i + 1]} {$j <= $nts} {incr j} {
			set rhoname [catenate $RHO _$i$j]
			if {![if_parameter_exists $rhoname]} {
			    parameter $rhoname = 0 \
				lower -1 upper 1
			}
		    }
		}
	    }
	}
    }

    foreach tr $ts {
	if {$multi} {
	    set suffix ($tr)
	}

# Get starting variance parameter values; create h2q1 if it doesn't exist


	if {[catch {set e2start [parameter e2$suffix =]}]} {
	    error \
	    "Parameter e2$suffix does not exist.  Start with polygenic model."
	}
	if {[catch {set h2rstart [parameter h2r$suffix =]}]} {
	    error \
            "Parameter h2r$suffix does not exist.  Start with polygenic model."
	}
	parameter h2q1$suffix
	set h2q1start [parameter h2q1$suffix =]


# I could check for summing to 1 here, but it's pretty complicated because
# the latest $h2q may or may not exist.  Errors will be caught anyway.

# Set e2 boundaries
# This is done conservatively (adjusting as little as necessary)
# But using boundary heuristics as needed

	set e2low [parameter e2$suffix lower]
	set e2upp [parameter e2$suffix upper]
	if {$e2upp == 0 || $e2low > $e2start || $e2upp < $e2start} {
	    error \
	 "Parameter e2$suffix improperly bounded.  Start with polygenic model."
	} elseif {$e2start < [e2lower]} {

# Assume it was maximized to zero or nearly so (could be bad user setup too)
# Let "perturb" (later) take care of "starting on boundary condition", if
#   applicable, by adjusting starting value.

	} elseif {$e2start == 1} {

# Previous model was sporadic or had zero heredity
# Set lower bound heuristically no lower than current setting

	    soft_lower_bound e2$suffix [highest $e2low [e2lower]]
	} else {

# E2 somewhere in between [e2lower] and 1
# Set lower bound using e2squeeze heuristic, e2lower, and "soft" bounding

	    soft_lower_bound e2$suffix \
		[highest [e2lower] [expr $e2start - [e2squeeze]]]
	}

# Set E2 upper bound no higher than 1, current bound, or bracketed range

	parameter \
	    e2$suffix  upper [lowest 1 $e2upp [expr $e2start + [e2squeeze]]]

# setup h2r bounds and starting point

	parameter h2r$suffix lower 0
	set h2rupper 1
	set maxh2rupper [expr 1 - [parameter e2$suffix lower]]
	if {[h2rf] > 0 && ![catch {set null0h2r [nulln 0 h2r$suffix]}]} {
	    set h2rupper [expr [h2rf] * $null0h2r]
	}
	if {$h2rupper > $maxh2rupper} {
	    set h2rupper $maxh2rupper
	}
	parameter h2r$suffix upper $h2rupper
	constraint_include h2r$suffix e2$suffix

	set H2Q h2q$new_h2q_index
	set K2 K2_$new_h2q_index
	if {$twopoint} {
	    set MIBD ibd$new_h2q_index
	} else {
	    set MIBD mibd$new_h2q_index
	}

	set constrain_new_to_zero 0
	if {$epistasis} {

# If new chromosome and locus are the same as epistasis base, we
#   contrain the new h2q to zero

	    set matlist [matrix]
	    set mfile ""
	
	    for {set i 0} {"" != [set line [line_index $matlist $i]]} \
		{incr i} {
		    set matname [lindex $line 3]
		    if {![string compare mibd$epistasis $matname]} {
			set mfile [lindex $line 2]
			break
		    }
		}
	    if {"" == $mfile} {
		error "Can't find matrix for epistasis"
	    }
	    set echr [get_chromosome $mfile]
	    set eloc [get_locus $mfile]
	    set tchr [get_chromosome $ibdfilename]
	    set tloc [get_locus $ibdfilename]
	    if {$echr == $tchr && $eloc == $tloc} {
		set constrain_new_to_zero 1
	    }

# Setup epistasis interaction term

	    if {![if_parameter_exists h2qe1]} {
		parameter h2qe1 lower 0.0 upper 1.0
		carve_new_value h2qe1 0.01
	    } else {
		parameter h2qe1 lower 0.0
	    }
	    constraint_include h2qe1
	    omega_remove h2qe1
	    omega_add mibd$epistasis*$MIBD*h2qe1
	}

# Find good bounds for h2q unless -zerostart

	if {$zs} {
	    parameter $H2Q$suffix start 0.0 lower -0.00001 upper 1.0
	} else {

# If previous $H2Q value specified or found, base h2q upper bound on that

	    if {"none" != $previous_h2q_value} {
		set h2qupper [lowest 1.0 [expr [h2q_float] + $previous_h2q_value]]
# If no linkage analysis has been done, upper bound will be 1.0 and value
# will be close to zero (less than 0.01) if not exactly zero
	    } elseif {![catch {set h2q_upper [parameter $H2Q$suffix upper]}] && \
			  ($h2q_upper != 1.0 || [parameter $H2Q$suffix =] > 0.01)} {
		set h2qupper [lowest 1.0 \
				  [expr [h2q_float] + [parameter $H2Q$suffix start]]]
	    } else {

# Use h2qsupper ("boundary start upper") to set upper bound for new h2q*

		set h2qupper [h2qsupper -get $new_h2q_index]
	    }


# if $H2Q doesn't exist, create it
# or, if it does exist, but its value is 0, set to 0.01

	    if {[catch {set h2q_start [parameter $H2Q$suffix start]}] || \
		    $h2q_start <= 0} {
		
		parameter $H2Q$suffix start 0.01 lower 0 upper $h2qupper

# Find a place to carve new $H2Q from:
#   Subtract 0.01 from h2r for initial h2q, or
#   subtract 0.01 from e2 for initial h2q, or
#   if new_h2q_index==1, start e2 and h2r at 0.9 and 0.01, or
#   subtract 0.01 from first H2qN larger than 0.02

	    set min_e2_test [expr .02 + [parameter e2$suffix lower]]
	    
	    if {0.02 < [parameter h2r$suffix =]} {
		parameter h2r$suffix = [expr -0.01 + [parameter h2r$suffix =]]
	    } elseif {$min_e2_test < [parameter e2$suffix =]} {
		parameter e2$suffix = [expr -0.01 + [parameter e2$suffix =]]
	    } elseif {$new_h2q_index == 1} {
		parameter e2$suffix = 0.9
		parameter h2r$suffix = 0.09
	    } else {
		set done 0
		puts "    *** Attempting to carve $H2Q$suffix from other H2Qs"
		for {set i 1} {$i < $new_h2q_index} {incr i} {
		    if {0.02 < [parameter h2q$i$suffix =]} {
			parameter h2q$i$suffix = \
				[expr -0.01 + [parameter h2q$i$suffix =]]
			set done 1
			break
		    }
		}
		if {!$done} {
	       error "Unable to carve $H2Q$suffix starting value from anywhere"
		}
	    }
	    } else {
		parameter $H2Q$suffix lower 0.0 upper $h2qupper
	    }
	    perturb
	}

	if {$twopoint} {
	    if {-1 != [string first "D7" [omega]] || \
		    -1 != [string first "d7" [omega]]} {
		matrix load $ibdfilename $MIBD d7
	    } else {
		matrix load $ibdfilename $MIBD
	    }
	} else {
	    if {[ifneedk2]} {
		eval [use_global_if_defined Solar_Linkage_Matrix \
			  {matrix load $ibdfilename $MIBD $K2}]
	    } else {
		eval [use_global_if_defined Solar_Linkage_Matrix \
			  {matrix load $ibdfilename $MIBD}]
	    }
	}

# Edit constraints

	constraint_include $H2Q$suffix e2$suffix

# If this must be constrained to zero for epistasis, do so now

	if {$constrain_new_to_zero} {
	    parameter_roll_into $H2Q$suffix h2r$suffix
	    constrain_to_zero $H2Q$suffix
	    parameter $H2Q$suffix lower -0.01
	} else {

# Otherwise, make sure lower bound is 0, unless zerostart

	    if {0 > [parameter $H2Q$suffix lower] && !$zs} {
		parameter $H2Q$suffix lower 0.0
	    }
	}
    }

# Edit omega

    if {$multi==2} {
	if {-1 == [string first $H2Q\(ti\) [omega]]} { 
	   omega_add sqrt(<$H2Q\(ti\)>)*sqrt(<$H2Q\(tj\)>)*$MIBD*(tne*$RHO+teq)
	}
    } elseif {$multi > 2} {
	if {-1 == [string first $H2Q\(ti\) [omega]]} { 
	    omega_add sqrt(<$H2Q\(ti\)>)*sqrt(<$H2Q\(tj\)>)*$MIBD*(tne*[catenate $RHO _ij]+teq)
	}
    } else {
	omega_remove $H2Q
	omega_add $MIBD*$H2Q
    }

# Now, apply special boundary heuristics
#  they must be applied after setting up omega since h2qcount is based
#  on omega.

    foreach tr $ts {
	if {"none" == $previous_h2q_value} {
	    boundarywide -apply   ;# Apply current boundarywide status
	}
	boundarynull -apply
    }

    return ""
}

# solar::linkmod2p -- private
#
# Purpose:  Use linkmod -2p instead.  linkmod2p is now obsolescent.
#-

# Set up parameters and constraints for twopoint linkage 
#           models
#
# Usage:    linkmod2p <ibdfile>
#
# Notes:    A polygenic or linkage model should already have been
#           created and maximized.  Boundaries are set around existing
#           values.
#
#           By default, standard error is turned off.  You may turn it on
#           again by giving the command 'option standerr 1' after running
#           linkmod and before running maximize.
#
#           To save memory, the d7 (dominance) values from the ibd matrix
#           will not be loaded unless d7 is found in the omega.
# -

proc linkmod2p {args} {
    return [eval linkmod -2p $args]
}

proc linkmod2p-obsolete {ibdfilename} {

    set new_h2q_index 1

# Catch missing basic parameters...instruct the dummies on what to do

    if {[catch {set e2start [parameter e2 start]}]} {
	error "Parameter e2 does not exist.  Start with polygenic model."
    }
    if {[catch {set h2rstart [parameter h2r start]}]} {
	error "Parameter h2r does not exist.  Start with polygenic model."
    }
    if {[catch {set h2q1start [parameter h2q1 start]}]} {
	error "Parameter h2q1 does not exist.  Start with polygenic model."
    }

# I could check for summing to 1 here, but it's pretty complicated because
# the latest $h2q may or may not exist.  Errors will be caught anyway.

# Set e2 boundaries
# This is done conservatively (adjusting as little as necessary)
# But using boundary heuristics as needed

    set e2low [parameter e2 lower]
    set e2upp [parameter e2 upper]
    if {$e2upp == 0 || $e2low > $e2start || $e2upp < $e2start} {
	error "Parameter e2 improperly bounded.  Start with polygenic model."
    } elseif {$e2start < [e2lower]} {

# Assume it was maximized to zero or nearly so (could be bad user setup too)
# Let "perturb" (later) take care of "starting on boundary condition", if
#   applicable, by adjusting starting value.

    } elseif {$e2start == 1} {

# Previous model was sporadic or had zero heredity
# Set lower bound heuristically no lower than current setting
	soft_lower_bound e2 [highest $e2low [e2lower]]
    } else {

# E2 somewhere in between [e2lower] and 1
# Set lower bound using e2squeeze heuristic, e2lower, and "soft" bounding
	soft_lower_bound e2 [highest [e2lower] [expr $e2start - [e2squeeze]]]
    }
# Set E2 upper bound no higher than 1, current bound, or bracketed range
    parameter e2 upper [lowest 1 $e2upp [expr $e2start + [e2squeeze]]]

# setup h2r bounds and starting point

    parameter h2r lower 0
    set h2rupper 1
    set maxh2rupper [expr 1 - [parameter e2 lower]]
    if {[h2rf] > 0 && ![catch {set null0h2r [nulln 0 h2r]}]} {
	set h2rupper [expr [h2rf] * $null0h2r]
    }
    if {$h2rupper > $maxh2rupper} {
	set h2rupper $maxh2rupper
    }
    parameter h2r upper $h2rupper
    constraint_include h2r

# Find good bounds for h2q
	
    set h2qupper [h2qsupper -get $new_h2q_index]

    set H2Q h2q1
    set IBD ibd1
    set D7 d7

# if $H2Q doesn't exist, create it
# or, if it does exist, but its value is 0, set to 0.01

    if {[catch {set h2q_start [parameter $H2Q start]}] || \
	$h2q_start <= 0} {
		
	parameter $H2Q start 0.01 lower 0 upper $h2qupper

# Find a place to carve new $H2Q from:
#   Subtract 0.01 from h2r for initial h2q, or
#   subtract 0.01 from e2 for initial h2q, or
#   if new_h2q_index==1, start e2 and h2r at 0.9 and 0.01, or
#   subtract 0.01 from first H2qN larger than 0.02

	set min_e2_test [expr .02 + [parameter e2 lower]]

	if {0.02 < [parameter h2r start]} {
	    parameter h2r start [expr -0.01 + [parameter h2r start]]
	} elseif {$min_e2_test < [parameter e2 start]} {
	    parameter e2 start [expr -0.01 + [parameter e2 start]]
	} elseif {$new_h2q_index == 1} {
	    parameter e2 start 0.9
	    parameter h2r start 0.09
	} else {
	    error "Unable to carve $H2Q starting value from anywhere"
	}
    } else {
	set h2qupper [expr [h2q_float] + $h2q_start]
	parameter $H2Q lower 0.0 upper $h2qupper
    }

    perturb

# include d7 only if found in omega
    if {-1 != [string first "D7" [omega]] || \
	    -1 != [string first "d7" [omega]]} {
	matrix load $ibdfilename $IBD $D7
    } else {
	matrix load $ibdfilename $IBD
    }

# old
#    eval [use_global_if_defined Solar_2P_Linkage_Matrix \
#	    {matrix load $ibdfilename $IBD $D7}]

# Edit constraints and omega equation

    constraint_include $H2Q
    omega_remove $H2Q
    omega_add $IBD*$H2Q

    option cmdiagonal 0
    option standerr 0

    boundarywide -apply   ;# Apply current boundarywide status
    boundarynull -apply
    return ""
}

# solar::tdist
#
# Purpose:  Set up t option for robust estimation of mean and variance
#
# Usage:    tdist        set up t option
#           tdist -off   turn off t option
#
# Notes:    tdist creates a parameter t_param and sets tdist option
#           tdist -off deletes t_param and unsets tdist option
#
#-
proc tdist {args} {
    if {"" == $args || "-on" == $args || "on" == $args} {
	set on 1
    } elseif {"-off" == $args || "off" == $args} {
	set on 0
    } else {
	error "Invalid argument to tdist"
    }

    if {$on} {
	if {![if_parameter_exists t_param]} {
	    parameter t_param start 10 lower 0.5 upper 500
	}
	option tdist 1
    } else {
	parameter delete t_param
	option tdist 0
    }
}

#
# check_artificial_boundaries and adjust_boundaries go together
# check_artificial_boundaries will not include fixed boundaries in the list
#   of problems, so adjust_boundries does not have to check the fix status
#
proc adjust_boundaries {} {
    set boundary_problems [check_artificial_boundaries]
    if {{} == $boundary_problems} {
	error "No variance component boundary conditions detected"
    }
    set adjustment [boundary_change]
    foreach problem $boundary_problems {
	set component [lindex $problem 0]
	set direction [lindex $problem 1]
	set value [parameter $component =]
	set delta [expr [parameter $component upper] - \
		       [parameter $component lower]]
	if {"lower" == $direction} {
	    set min_lower 0
	    if {"rho" == [string tolower [string range $component 0 2]]} {
		set min_lower -1
	    }
	    set newlower [expr $value - (0.998 * $delta)]
	    set newupper [expr $value + (0.001 * $delta)]
	    if {$newlower < $min_lower} {set newlower $min_lower}
	    if {"e2" == [string tolower [string range $component 0 1]]} {
		if {[trace_boundaries]} {
		   puts "Using soft bounds for $component"
		}
		soft_lower_bound $component -push $newlower
		parameter $component upper $newupper
	    } else {
		parameter $component lower $newlower upper $newupper
	    }
	} else {
	    set newlower [expr $value - (0.001 * $delta)]
	    set newupper [expr $value + (0.998 * $delta)]
	    if {$newupper > 1} {set newupper 1}
	    parameter $component lower $newlower upper $newupper
	}
    }
    return $boundary_problems
}


# do_boundries crunches boundaries after convergence error
# Fix boundaries if they need fixing (done by c++ now)
# Otherwise crunch boundaries

proc do_boundaries {args} {

# The minimum boundary delta is established in preopt.f
    set rmin 1e-4

    set crunch [bcrunch]
    if {"" != $args} {
	set crunch $args
	ensure_float $args
    }
    set checklist [get_vc_list]

# Check for invalid boundaries

    set invalid_boundaries 0
    foreach check $checklist {
	set value [parameter $check =]
	set lower [parameter $check lower]
	set upper [parameter $check upper]
	set delta [expr $upper - $lower]
	if {$delta < $rmin || $upper < $value || $value < $lower} {
	    set invalid_boundaries 1
	}
    }	

# Either crunch all boundaries, or fix invalid ones

    set crunched {}
    foreach check $checklist {
	set value [parameter $check =]
	set lower [parameter $check lower]
	set upper [parameter $check upper]
	set delta [expr $upper - $lower]

	if {[trace_boundaries]} {
	    set rdelta [format %.4g $delta]
	    puts "Old boundary range for $check is $rdelta"
	}	

	set crunchthis $crunch
	if {"rhoe" == $check} {
	    set crunchthis [expr 0.3 * $crunch]
	}


	if {$crunchthis < $rmin} {
	    set crunchthis $rmin
	}

# If there are invalid boundaries, fix them, but leave others alone
# Otherwise, if valid boundaries are tighter than new setting, leave them alone

	if {$delta > $rmin && $upper - $value > $rmin && $value - $lower > $rmin} {
	    if {$invalid_boundaries} {
#
# Invalid boundaries are now fixed in C++
# so we just re-run model
#
		continue
	    }

# Don't crunch rhoqX until second level

	    if {"rhoq" == [string tolower [string range $check 0 3]]} {
		if {$crunchthis >= 0.09} {
		    continue
		} else {
		    set crunchthis [expr 5 * $crunchthis]
		}
	    }
	    if {$delta < $crunchthis*2} {
		if {[trace_boundaries]} {
		    puts "Boundaries for $check not changed"
		}
		continue
	    }
	}

	set bottom 0
	if {"e2" == [string tolower [string range $check 0 1]] || \
		"h2r" == [string tolower [string range $check 0 2]]} {
	    set bottom 0.01
	    if {$value <= $bottom + 0.001} {
		set bottom 0
	    }
	} elseif {"rho" == [string tolower [string range $check 0 2]]} {
	    set bottom -1
	}

	if {"" == [parameter $check fixlower]} {
	    parameter $check lower [highest $bottom \
					[expr $value - $crunchthis]]
	}
	if {"" == [parameter $check fixupper]} {
	    parameter $check upper  [lowest 1.0 [expr $value + $crunchthis]]
	}

	set newdelta [expr [parameter $check upper] - [parameter $check lower]]
	if {[trace_boundaries]} {
	    set rdelta [format %.4g $newdelta]
	    puts "New boundary range for $check is $rdelta"
	}
	lappend crunched $check
    }

    if {$invalid_boundaries} {
	error "   *** Fixed bad boundaries: $crunched"
    }
    return $crunched
}


# get_vc_list
# Get a list of active variance components and rho's

proc get_vc_list {} {
    set checklist {}
    set h2qi [h2qcount]

# For multiple traits
    set nt [llength [trait]]
    if {1 < $nt} {
	foreach tname [trait] {
	    lappend checklist e2\($tname\)
	    if {[if_parameter_exists h2r\($tname\)]} {
		lappend checklist h2r\($tname\)
	    }
	    for {set i 1} {$i <= $h2qi} {incr i} {
		lappend checklist h2q$i\($tname\)
	    }
	}
	foreach par [parameter -names] {
	    if {0 == [string compare "rho" [string range $par 0 2]]} {
		lappend checklist $par
	    }
	}
	if {[check_house]} {
	    foreach tname [trait] {
		if {[if_parameter_exists c2($tname)]} {
		    lappend checklist c2($tname)
		}
	    }
	}

# For single trait
    } else {
	lappend checklist e2
	if {[if_parameter_exists h2r]} {
	    lappend checklist h2r
	}
	for {set i 1} {$i <= $h2qi} {incr i} {
	    lappend checklist h2q$i
	}
	for {set i 1} {$i <= $h2qi} {incr i} {
	    lappend checklist h2q$i
	}
	if {[check_house]} {
		lappend checklist c2
	}

# Epistasis boundary control only supported for univariate

	if {[check_epistasis]} {
	    lappend checklist h2qe1
	}
    }

    if {[solardebug]} {
	puts "solardebug: Recognized variance components are: $checklist"
    }
    return $checklist
}    


proc check_artificial_boundaries {} {
    set blist {}
    set checklist [get_vc_list]
    foreach check $checklist {
	set value [parameter $check =]
	set upper [parameter $check upper]
	set lower [parameter $check lower]
	set fixupper [parameter $check fixupper]
	set fixlower [parameter $check fixlower]
	if {"rho" == [string tolower [string range $check 0 2]]} {
	    set min_lower -1
	    set max_upper 1
	} else {
	    set min_lower 0
	    set max_upper 1
	}
	if {$value <= $lower && $lower > $min_lower && $fixlower == ""} {
	    lappend blist [list $check lower]
	}
	if {$value >= $upper && $upper < $max_upper && $fixupper == ""} {
	    lappend blist [list $check upper]
	}
    }
    return $blist
}

proc check_real_upper_boundaries {} {
    set blist {}
    set checklist [get_vc_list]
    foreach check $checklist {
# RhoqX hitting 1 is normal
	if {"rhoq" == [string tolower [string range $check 0 3]]} {
	    continue
	}
#
# check that we didn't just hit upper bound because of constraint
#
	if {[is_constrained_to_nonzero $check]} {
	    continue
	}
	if {0 != [string first e2 $check]} {
	    set value [parameter $check =]
	    if {$value >= 1.0} {
		lappend blist [list $check upper]
	    }
	}
    }
    return $blist
}



# solar::perturb
#
# Purpose:  Perturb starting values for E2, H2r, and H2q's at bounds
#
# Usage:    perturb
#
# Notes:   perturb is specially tailored to the standard parameterization
#          of e2, h2r, h2q1, etc.  perturb does nothing silently if
#          parameters e2 and h2r are not present.
#
#          It is no longer necessary or possible to specify h2qindex as an
#          unqualified argument as in earlier versions.  If an unqualified
#          argument is specified, it is ignored.
#
#          This is used automatically by the 'linkmod' script, and therefore
#          also by multipoint and twopoint.
#
#          perdelta is the quantity used in the adjustment (this may be set
#          with the perdelta command).  It defaults to 0.001
#
#          Accumulated deltas are distributed only to other parameters whose
#          values are 3*perdelta away from the relevant bound, and then only
#          in the perdelta quantity.
#
#          This does not handle conditions where parameters exceed boundaries
#          by more than a very small amount.  (Of course, they shouldn't
#          exceed the boundaries at all, but sometimes they do by very small
#          amounts.  Recent changes to the maximization routines ought to
#          eliminate that.)
# -

proc perturb {args} {

    set give_warning 0
    ifverbplus set give_warning 1
    ifdebug set give_warning 1
    set give_first_warning $give_warning

    set adj [perdelta]
    set adj2 [expr $adj * 2]
    set adj3 [expr $adj * 3]
#
# We deal with the variance components for each trait in a separate pass
#
    set tnames [trait]
    foreach tname $tnames {
	if {1 == [llength $tnames]} {
	    set parvector [get_variance_components]
	} else {
	    set parvector [get_variance_components $tname]
	}

	if {{} == $parvector} continue

# Adjust parameters on boundaries

	set accum 0
	set parvals {}
	set unchanged $parvector
	set count [llength $parvector]
	set constraints_found 0
	foreach par $parvector {
	    set val [parameter $par start]
	    lappend parvals $val
	    if {![catch {find_simple_constraint $par}]} {
		incr constraints_found
		continue
	    }
	    if {$val <= [parameter $par lower] && $val + $adj3 < 1} {
		if {$give_first_warning} {puts "    *** Perturbing parameters"}
		parameter $par start [expr $val + $adj]
		set accum [expr $accum - 1]
		set give_first_warning 0
		set unchanged [remove_from_list $unchanged $par]
		ifdebug puts "    *** Raising $par from lower bound"
		if {[parameter $par =] >= [parameter $par upper]} {
		    parameter $par upper [expr [parameter $par =] + $adj]
		    ifdebug puts "perturb:  raising $par upper bound"
		}
	    } elseif {$val >= [parameter $par upper] && $val - $adj3 > 0} {
		if {$give_first_warning} {puts "    *** Perturbing parameters"}
		parameter $par start [expr $val - $adj]
		set accum [expr $accum + 1]
		set give_first_warning 0
		set unchanged [remove_from_list $unchanged $par]
		ifdebug puts "    *** Lowering $par from upper bound"
		if {[parameter $par =] <= [parameter $par lower]} {
		    parameter $par lower [expr [parameter $par =] - $adj]
		    ifdebug puts "perturb:  lowering $par lower bound"
		}
	    }
	}

# Actually, if there are N-1 constraints for N parameters, we can't
# perturb, so we skip to the restore part

	if {$count - 1 > $constraints_found} {

# Now, find places to "dump" the deltas we've accumulated
# Allow 3*NPAR passes, if we can't do it by then, give up

	if {{} == $unchanged} {
	    set unchanged $parvector
	}

	set limit [expr 3 * [llength $parvector]]
	for {set i 0} {$i < $limit} {incr i} {
	    if {$accum == 0} break
	    foreach par $unchanged {
		if {![catch {find_simple_constraint $par}]} continue
		if {$accum == 0} break
		set val [parameter $par start]
		if {$accum > 0} {
		    if {[parameter $par upper] - $val > $adj3} {
			parameter $par start [expr $val + $adj]
			set accum [expr $accum - 1]
			ifdebug puts "perturb:  raising $par to compensate"
		    }
		} else {
		    if {$val - [parameter $par lower] > $adj3} {
			parameter $par start [expr $val - $adj]
			set accum [expr $accum + 1]
			ifdebug puts "perturb:  lowering $par to compensate"
		    }
		}
	    }
	}
#
# Didn't find place to dump deltas.  That could be because of narrow artificial
# boundaries.  Try again, but this time permit the moving of artificial
# boundaries to within $adj of 0 or 1 and allowing $adj between boundary and
# value.
#
	if {$accum != 0} {
	    ifdebug puts "    *** Second phase of perturb required"
	    for {set i 0} {$i < $limit} {incr i} {
		if {$accum == 0} break
		foreach par $unchanged {
		    if {![catch {find_simple_constraint $par}]} continue
		    if {$accum == 0} break
		    set val [parameter $par =]
		    if {$accum > 0} {
			if {[parameter $par =] + $adj3 <= 1} {
			    parameter $par = [expr $val + $adj]
			    parameter $par upper [expr $val + $adj2]
			    set accum [expr $accum - 1]
			    ifdebug puts "perturb:  raising $par to compensate"
			}
		    } else {
			if {[parameter $par =] -  $adj3 >= 0} {
			    parameter $par = [expr $val - $adj]
			    parameter $par lower [expr $val - $adj2]
			    set accum [expr $accum + 1]
			    ifdebug puts "perturb:  lowering $par to compensate"
			}
		    }
		}
	    }
	}
#
# If failed, restore original parameters for this trait and quit
#
        }
	if {$accum != 0} {
	    if {$count - 1 > $constraints_found} {
		puts "    *** Warning: Unable to perturb variance parameters\n"
	    } elseif {$give_warning} {
		puts "    *** Can't perturb because $constraints_found constraints for $count parameters"
	    }
	    for {set i 0} {$i < $count} {incr i} {
		parameter [lindex $parvector $i] = [lindex $parvals $i]
		if {$give_warning} {
		    puts "    *** Restoring parameter [lindex $parvector $i] to [lindex $parvals $i]"
		}
	    }
	}
    }
    return ""
}


# solar::allcovar --
#
# Purpose:  Set up all non-trait variables as covariates
#
# Usage:    allcovar
#
# Notes:    Phenotypes and trait commands must already have been given.
#           If there is a variable named "age," it will be set up
#           as "age^1,2#sex."  If undesired variables are made into
#           covariates, they should be removed with the covariate
#           delete command.
#
#           allcovar will not include names mapped to any of the standard
#           field variables (see 'help field').  Be sure to set up field
#           mappings (if required) first to ensure you don't get extra
#           covariates for the likes of ID, FAMID, etc.
#
#           allcovar will also not include any names added to the 'exclude'
#           list.  Use the 'exclude' command to add names to the exclude
#           list, or to display the exclude list.  By default, the exclude
#           list includes some standard PEDSYS mnemonics
# -

proc allcovar {} {
    if {0 == [string length [trait]]} {error "Trait not specified"}
    set ltrait [string tolower [trait]]
#
# Get phenotypes list and clean it from multiple phenotype names
#
    set rawlist [lrange [phenotypes] 1 end]
    set covarlist {}

    foreach cov $rawlist {
	if {[string compare ":" [string range $cov end end]]} {
	    lappend covarlist $cov
	}
    }

    set covarlist [string tolower $covarlist]
    set covarlist [remove_from_list $covarlist $ltrait]
    set covarlist [remove_from_list $covarlist ibdid]
#
# Setup covariates
#
    covariate sex
    foreach cov $covarlist {
	if {![string compare age $cov]} {
	    covariate age^1,2#sex
	} elseif {![string compare [string tolower [field ID]] $cov]} {
	} elseif {![string compare [string tolower [field FA]] $cov]} {
	} elseif {![string compare [string tolower [field MO]] $cov]} {
	} elseif {![string compare [string tolower [field SEX]] $cov]} {
	} elseif {![string compare [string tolower EGO] $cov]} {
	} elseif {![string compare [string tolower SIRE] $cov]} {
	} elseif {![string compare [string tolower DAM] $cov]} {
	} elseif {![string compare [string tolower GENDER] $cov]} {
	} elseif {![string compare [string tolower [field PROBND]] $cov]} {
	} elseif {![string compare [string tolower [field MZTWIN]] $cov]} {
	} elseif {![string compare [string tolower [field FAMID]] $cov]} {
	} elseif {![string compare [string tolower [field SEX]] $cov]} {
	} elseif {-1 < [lsearch [exclude] $cov]} {
	} else {
	    covariate $cov
	}
    }
}

# solar::pedlod --
#
# Purpose:  Calculate pedigree-specific LOD scores
#
# Usage:    pedlod [<test-model> [<null-model>]]
#
# Notes:    If no model is specified, the model currently in memory
#           is used as the test-model (useful if you have just run
#           multipoint or twopoint), and its null-model (having
#           one less linkage element) is used as the null model.
#
#           If only one model is specified, the null model is taken
#           from the outdir after the specified model is loaded.
#
#           The pedigree numbers used are based on SOLAR's pedigree
#           index "pedindex.out" created by the "load pedigree"
#           command.  These do not necessarily correspond to user
#           pedigree numbers (and there is not necessarily even
#           a one-to-one correspondence).  Refer to pedindex.out
#           to associate your ID's with the pedigrees associated
#           by SOLAR.  (Note: pedindex.out has a code file pedindex.cde
#           and is best read using PEDSYS, but may also be read fairly
#           well as a text file if PEDSYS is not available.)
#
#           Note that the LOD score calculation may be affected by the
#           number trait(s), and the lodp options.  See the documentation
#           for the "lodp" command for further details.  When applicable,
#           SOLAR converts 2df bivariate LODs to "1df effective" LODs.
# -

proc pedlod {args} {

    full_filename foo   ;# test trait/outdir

    delete_files_forcibly [full_filename pedlod.out]

# Check inputs

    set test_model ""
    set null_model ""

    if {2 < [llength $args]} {
	error "Only two models may be specified"
    } elseif {2 == [llength $args]} {
	set test_model [lindex $args 0]
	set null_model [lindex $args 1]
    } elseif {1 == [llength $args]} {
	set test_model [lindex $args 0]
    }

# Save starting model

    set start_model ""
    if {![catch {trait}]} {
	set start_model [full_filename pedlod.start]
	save model $start_model
    }

# Check for existence of models
# null_model and test_model are assigned if not already
# test_model is loaded if not already

    if {"" != $test_model} {
	if {![file exists $test_model] && ![file exists $test_model.mod]} {
	    error "Can't find $test_model; did you specify full path?"
	}
	load model $test_model  ;# Needed for h2qcount below
    } else {
	set test_model $start_model
    }

    if {"" == $null_model} {
	set null_index [expr [h2qcount] - 1]
	if {$null_index < 0} {
	    error "Test model has no linkage elements"
	}
	set null_model [full_filename null$null_index.mod]
	if {![file exists $null_model]} {
	    error \
            "Can't find implied null model [full_filename null$null_index.mod]"
	}
    } else {
	if {![file exists $null_model] && ![file exists $null_model.mod]} {
	    error "Can't find $null_model; did you specify full path?"
	}
    }

# Do pedlike on test_model; save as pedlod.test.out

    puts "\nComputing per-pedigree loglikehoods for test model...\n"

    pedlike $test_model
    file rename -force [full_filename pedlike.out] [full_filename pedlod.test.out]

# Do pedlike on null_model; save as pedlod.null.out

    puts "\nComputing per-pedigree loglikehoods for null model...\n"

    pedlike $null_model
    file rename -force [full_filename pedlike.out] [full_filename pedlod.null.out]

# Read both pedlike files and write LOD scores to output file

    set outfile [open [full_filename pedlod.out] w]
    set testfile [tablefile open [full_filename pedlod.test.out]]
    set nullfile [tablefile open [full_filename pedlod.null.out]]

    tablefile $testfile start_setup
    tablefile $nullfile start_setup

    set need_famid 0
    tablefile $testfile setup Loglikelihood
    tablefile $testfile setup PEDNO

    tablefile $nullfile setup Loglikelihood
    tablefile $nullfile setup PEDNO

    puts "\nComputing per-pedigree LOD scores...\n"
    putstee $outfile "PEDNO, LOD"

    set after_sum 0.0
    while {1} {
	set testrec [tablefile $testfile get]
	set nullrec [tablefile $nullfile get]

# Check for EOF

	if {{} == $testrec && {} == $nullrec} {
	    break
	} elseif {{} == $testrec} {
	    tablefile $testfile close
	    tablefile $nullfile close
	    close $outfile
	    error "Premature end of testfile"
	} elseif {{} == $nullrec} {
	    tablefile $testfile close
	    tablefile $nullfile close
	    close $outfile
	    error "Premature end of nullfile"
	}
	    
# Compare PEDNO and FirstID for consistency (and FAMID, if present)

	if {[string compare [lindex $testrec 1] [lindex $nullrec 1]]} {
	    tablefile $testfile close
	    tablefile $nullfile close
	    close $outfile
	    error "PEDNO out of sequence"
	}

# Get loglikelihoods and compute LOD
# If either file is missing likelihood, use error message instead

	set testlike [lindex $testrec 0]
	set nulllike [lindex $nullrec 0]
	if {![is_float $testlike]} {
	    set lodscore $testlike
	} elseif {![is_float $nulllike]} {
	    set lodscore $nulllike
	} else {
	    set lodscore [lod $testlike $nulllike]
	    set lodscore [format %.5f $lodscore]
	    set after_sum [expr $after_sum + $lodscore]
	}

# Write record to output file and screen

	putstee $outfile [format "%5s,  %s" \
		    [lindex $testrec 1] $lodscore]
    }

# Close files

    tablefile $testfile close
    tablefile $nullfile close
    close $outfile

# Reload test_model (or original model)

    if {"" != $start_model} {
	load model $start_model
    }

    set pout [open [full_filename pedlod.info] w]
    close $pout
    puts " "
    set pinfo [open [full_filename pedlike.info]]
    while {-1 != [gets $pinfo record]} {
	if {-1 != [lsearch $record pedigrees]} {
	    putsout pedlod.info "$record"
	}
    }
    close $pinfo
    putsout pedlod.info "LOD sum (of rounded scores) is [format %.4f $after_sum]"
    putsout pedlod.info "Results have also been written to [full_filename pedlod.out] and [full_filename pedlod.info]"
}


# solar::pedlike --
#
# Purpose:  Calculate pedigree-specific loglikelihoods
#
# Usage:    pedlike [-q] [<model>]
#
#           -q    (quiet) Supress output to terminal
#
# Notes:    Default model will be current model, if current model has
#           been maximized.  If changes have been made to current model
#           since the last maximization, results may not be predictable.
#
#           If current model has not been maximized, default model is
#           the null0 model in current outdir.
#
#           Results are written to "pedlike.out" in the outdir
#           and also shown on terminal with some additional summary info.
#
#           The pedigree numbers used are based on SOLAR's pedigree
#           index "pedindex.out" created by the "load pedigree"
#           command.  These do not necessarily correspond to user
#           pedigree numbers (and there is not necessarily even
#           a one-to-one correspondence).  Refer to pedindex.out
#           to associate your ID's with the pedigrees associated
#           by SOLAR.  (Note: pedindex.out has a code file pedindex.cde
#           and is best read using PEDSYS.)
#
#     
# -

proc pedlike {args} {

    file delete [full_filename pedlike.info]
    file delete [full_filename pedlike.out]

    set output 1
    set q ""
    if {"" != $args && "-q" == [lindex $args 0]} {
	set output 0
	set q "-q"
	set args [lrange $args 1 end]
    }

# Determine what model to use, save current model if necessary

    set start_model ""
    if {![catch {trait}]} {
	set start_model [full_filename pedlike.start]
	save model $start_model
    }

    set model_message ""
    set modelname $args
    if {[llength $args] > 1} {
	error "specific: Invalid arguments"
    } elseif {[llength $args] == 0} {
	if {![catch {loglike}]} {
	    set model_message "\nResults for model currently loaded"
	    set modelname ""
	} else {
	    set modelname [full_filename null0]
	}
    }
    if {"" != $modelname} {
	set model_message "\nResults for model $modelname"
	load model $modelname
    }

    if {$output} {puts "Scanning pedigrees..."}

# Read pedindex.out to get highest pedno
    set last_pedno 0
    set pin [tablefile open pedindex.out]
    tablefile $pin start_setup
    tablefile $pin setup PEDNO
    while {{} != [set record [tablefile $pin get]]} {
	set last_pedno [lindex $record 0]
    }
    tablefile $pin close

# Get and write loglikelihoods for each pedigree

    if {$output} {puts "Estimating likelihoods for each pedigree..."}
    option maxiter 1
    option standerr 0
    option pedlike 1
    if {[catch {set errmsg [maximize_quietly pedlikemod.out]} cmessage]} {
	set errmsg "Unknown error"
    }
    if {![string compare [string range $errmsg 0 0] "\n"]} {
	set errmsg [string range $errmsg 1 end]
    }
    if {"" != $errmsg} {
	error "pedlike maximization failed: $errmsg"
    }

# Read pedexclude.dat for excluded pedigrees
    set pex {}
    if {[file exists [full_filename pedexclude.dat]]} {
	set pxf [open [full_filename pedexclude.dat]]
	gets $pxf  ;# skip header
	while {-1 != [gets $pxf record]} {
	    lappend pex $record
	}
	close $pxf
    }

# Read pedlike.dat and write pedlike.out

    set pex_save $pex
    set pin [open [full_filename pedlike.dat]]
    gets $pin ;# skip over header
    set oname [full_filename pedlike.out]
    set pout [open $oname w]
    close $pout
    eval putsout $q pedlike.out \"  PEDNO,    Loglikelihood\"
    set pedno 0
    set pinc 0
    set pexc 0
    set plast 0
    set lsum 0.0
    while {-1 != [gets $pin loglike]} {
#
# Skip over deleted pedigrees
#
	incr pedno
	while {[llength $pex] && [lindex $pex 0] == $pedno} {
	    incr pedno
	    incr pexc
	    set pex [lrange $pex 1 end]
	}
	eval putsout $q pedlike.out \"[format %7d $pedno], [format %14.6f $loglike]\"
	set lsum [expr $lsum + $loglike]
	set plast $pedno
	incr pinc
    }
    close $pin
    ifdebug puts "pexc: $pexc  last_pedno: $last_pedno  plast: $plast"
    set pexc [expr $pexc + ($last_pedno - $plast)]
#
# Print summary information
#
    eval putsout $q pedlike.info $model_message
    eval putsout $q pedlike.info  \"Results written to [full_filename pedlike.out] and [full_filename pedlike.info]\"
    eval putsout $q pedlike.info \"$pinc pedigrees included, $pexc pedigrees excluded\"
    eval putsout $q pedlike.info \"Loglikelihood sum is [format %14.6f $lsum]\"
    if {$output} {puts ""}
#
# Restore original model status
#
    if {"" != $start_model} {
	load model $start_model
	file delete $start_model
    } else {
	model new
    }
    return ""
}


# solar::hlod
#
# Purpose:  Heterogeneity test for linkage
#
# Usage:    hlod [-step <stepsize>]
#
#           <stepsize> is size of increment in h2q1 and h2r.  Default is
#             0.1, 0.05, 0.02, or 0.01 chosen to be about 1/10 of total
#             heritability in starting model (with a minimum of
#             8 test points plus maximized h2q1 and total heretability).
#
# Notes:    Linkage test model must be loaded at beginning.  Corresponding
#           null model will be found in current output directory.
#
#           Complete results are written to hlod.out in maximization output
#           directory.
#
#           Linkage model may only have 1 linkage element; null model
#           is "polygenic" (null0) model in current output directory.
#
#           H0 vs. H1 test is only considered significant if p < 0.0001
#           (LOD 3).  If this is not significant, there will be a warning
#           that the H1 vs H2 test makes no sense.
#
#           hlod uses "homo" program written by Harald Goring for which
#           documentation is provided in Appendix 5 of the documentation
#           (use "doc" command to browse, click on Table of Contents).
#
#-

proc hlod {args} {

# Delete old output and intermediate files

    file delete [full_filename hlod.out]
    file delete [full_filename homo.in]
    
# Check for linkage model

    if {"" == [matrix]} {
	error "hlod: You must load linkage model first"
    }
    if {[h2qcount] != 1} {
	error "hlod: Only works with 1 QTL linkage model"
    }

# Examine and save current model

    save model [full_filename hlod.start]
    set h2t [expr [parameter h2q1 =] + [parameter h2r =]]
    set h2q1_max [parameter h2q1 =]
    puts "    *** Maximized h2q1 is $h2q1_max"
    if {$h2q1_max==0} {
	putsout hlod.out "    *** Warning: current model has h2q1 = 0\n"
    }
    putsout hlod.out "    *** Total heritability is [format %.10g $h2t]"

# Process arguments

    set step ""

    set badargs [read_arglist $args "-step" step]

    if {{} != $badargs} {
	error "Invalid arguments to hlod: $badargs"
    }

# Set and report step size

    if {"" == $step} {
	if {$h2t > 0.8} {
	    set step 0.1
	} elseif {$h2t > 0.4} {
	    set step 0.05
	} elseif {$h2t > 0.16} {
	    set step 0.02
	} else {
	    set step 0.01
	}
	putsout hlod.out "    *** Step size of $step will be used"
    } else {
	ensure_float $step
	putsout hlod.out "    *** Step size of $step chosen"
    }

# Save previous pedlike results

    save_pedlike hlod

# Load best polygenic model

    puts "    *** Getting per-pedigree likelihoods for polygenic model"
    pedlike -q [full_filename null0.mod]
    set h2r_0 [hlod_storepedlike]
    
# Setup vector and header for intermediate file

    set header "pedno,H0(0.0)"
    set vector ""
    set h2q1_max_output 0
    for {set h2q1 0.0} {$h2q1 < $h2t} \
	{set h2q1 [expr $h2q1 + $step]} {
	if {$h2q1 > $h2q1_max && $h2q1 != $h2q1_max && !$h2q1_max_output} {
	    set h2q1f [format %.10g $h2q1_max]
	    set header "$header,$h2q1f"
	    set vector "$vector$h2q1f "  ;# space required at end
            set h2q1_max_output 1
	}
	set h2q1f [format %.10g $h2q1]
	set header "$header,$h2q1f"
	set vector "$vector$h2q1f "    ;# space required at end
    }
    set h2q1f [format %.10g $h2t]
    set header "$header,$h2q1f"
    set vector "$vector$h2q1f"

# Step through possible values for h2q1

    set i -1
    foreach h2q1 $vector {

	incr i
	set last_i $i

        if {$h2q1 >= $h2t} {
	    set h2r 0
        } else {
	    set h2r [expr $h2t - $h2q1]
	}

	load model [full_filename hlod.start]

# Setup parameters and contraints

	parameter h2q1 = $h2q1
	constraint h2q1 = $h2q1
	if {$h2q1 == 0.0} {
	    parameter h2q1 lower -0.01
	} else {
	    parameter h2q1 lower 0.0
	}
	if {$h2q1 == 1.0} {
	    parameter h2q1 upper 1.01
	} else {
	    parameter h2q1 upper 1.0
	}

	set h2r $h2r
	constraint h2r = $h2r
	if {$h2r == 0.0} {
	    parameter h2r lower -0.01
	} else {
	    parameter h2r lower 0.0
	}
	if {$h2r == 1.0} {
	    parameter h2r upper 1.01
	} else {
	    parameter h2r upper 1.0
	}

	puts "    *** Getting per-pedigree likelihoods for null1.mod at h2q1=$h2q1"

	option MaxIter 1
	maximize -q -o hlod.test.out

	save model [full_filename hlod.test]
	pedlike -q [full_filename hlod.test]
	set h2q1_$i [hlod_storepedlike]
    }
    set npeds [llength $h2q1_1]

# Now, write results to intermediate file

    set ofile [open [full_filename homo.in] w]
    puts $ofile $header
    for {set pedno 1} {$pedno <= $npeds} {incr pedno} {
# Buildup output line
	set record "$pedno,[lindex $h2r_0 [expr $pedno - 1]]"
	for {set i 0} {$i <= $last_i} {incr i} {
	    set record "$record,[lindex [eval set foo \$h2q1_$i] \
                                [expr $pedno - 1]]"
	}
	puts $ofile $record
    }
    close $ofile

# Run the homo program

    load model [full_filename hlod.start]
    set pwd [pwd]
    cd [full_filename ""]
    if {[catch {exec homo -c} errmsg]} {
	cd $pwd
	hlod_cleanup
	error "Program homo returned error: $errmsg"
    }
    cd $pwd

# Read through homo.out, copying and echoing as desired

    set hf [open [full_filename homo.out]]
    set terminal 0
    set file 0
    while {-1 != [gets $hf record]} {

# Look for guideposts

	if {0 == [string compare "description of hypotheses:" $record]} {
	    set file 1
	    putsout -q hlod.out ""
	} elseif {0 == [string compare "test results:" $record]} {
	    set terminal -1
	} elseif {-1 != [string first "pedigree-specific" $record]} {
	    set terminal 0
	}

# copy and/or display as required

	if {$terminal == 1} {
	    putsout hlod.out $record
	} else {
	    if {$terminal == -1} {
		set terminal 1
	    }
	    if {$file} {
		putsout -q hlod.out $record
	    }
	}
    }
    close $hf

# Cleanup files and restore user pedlike files, if applicable

    hlod_cleanup
    restore_pedlike hlod

    return "Complete results have been written to [full_filename hlod.out] (use more to read)"
}

proc save_pedlike {prefix} {
    set saved ""
    if {[file exists [full_filename pedlike.info]]} {
	file rename -force [full_filename pedlike.info] [full_filename $prefix.pedlike.info]
	lappend saved pedlike.info
    }
    if {[file exists [full_filename pedlike.out]]} {
	file rename -force [full_filename pedlike.out] [full_filename $prefix.pedlike.out]
	lappend saved pedlike.out
    }
    if {[file exists [full_filename pedlike.dat]]} {
	file rename -force [full_filename pedlike.dat] [full_filename $prefix.pedlike.dat]
	lappend saved pedlike.dat
    }
    return $saved
}

proc restore_pedlike {prefix} {
    set files [glob -nocomplain [full_filename $prefix.pedlike.*]]
#   puts "Restoring files $files"
    foreach file $files {
	if {".info" == [file extension $file]} {
	    file rename -force $file [full_filename pedlike.info]
	} elseif {".out" == [file extension $file]} {
	    file rename -force $file [full_filename pedlike.out]
	} elseif {".dat" == [file extension $file]} {
	    file rename -force $file [full_filename pedlike.dat]
	} else {
#	    puts "file $file didn't match any patterns"
	}
    }
}

	

proc hlod_cleanup {} {
    eval file delete [full_filename pedlike.info pedlike.out pedlike.dat pedlikemod.out homo.out pedlike.start.mod]
}

proc hlod_storepedlike {} {
    set ulist ""
    set pfile [tablefile open [full_filename pedlike.out]]
    tablefile $pfile start_setup
    tablefile $pfile setup Loglikelihood
    while {"" != [set record [tablefile $pfile get]]} {
	lappend ulist [lindex $record 0]
    }
    tablefile $pfile close
    return $ulist
}


# solar::lodadj --
#
# Purpose:  Use or calculate an empirical LOD adjustment
# 
# Usage:    lodadj [-calc] [-off] [-null <N>] [-nreps <#replicates>] [-restart]
#                  [-restorenull0] [-query] [-summary]
#
#              lodadj   If no arguments are given, this turns ON the
#                       previously calculated empirical LOD adjustment
#                       for the current trait/outdir.  This value is
#                       stored in a file named lodadj.info if currently
#                       ON or lodadj.off if currently OFF.
#                       It is an error if the null0 model has a later
#                       timestamp than the lodadj.info file.  (You can
#                       update the timestamp of the lodadj.info file with
#                       the Unix "touch" command if you are sure it is OK.)
#     
#             -off      Turn OFF empirical LOD adjustment.
#
#             -query    Return the LOD adjustment currently in effect
#                       (1.0 if none).
#
#             -calc     Calculate and use a new empirical LOD adjustment.
#                       (This requires an existing null0.mod file from the
#                       polygenic command.)  The adjustment is turned ON.
#
#             -null     Use an existing nullN model instead of null0.
#
#             -nreps    Number of replicates.  In each replicate, a
#                       fully-informative marker, unlinked to the trait,
#                       is simulated, IBDs are calculated for this marker,
#                       and a LOD is computed for linkage of the trait
#                       to this marker.  The default number is 10000.
#
#             -restart  (or -r) Perform additional replicates, adding the
#                       LODs to the set of previously computed LODS, until
#                       the total number of replicates (old and new) reaches
#                       the number specified by the -nreps argument.  The
#                       same null model is used as in the previous replicates;
#                       the -null argument is ignored if present.
#
#             -cutoff   Specify the fraction of the highest observed LODs
#                       that will not be used to compute the empirical LOD
#                       adjustment.  For example, if the cutoff is .01, then
#                       the largest 1% of the observed LODs will be ignored
#                       when the LOD adjustment is calculated.  The default
#                       cutoff is .05.
#
#             -overwrite       (or -ov) Recalculate the empirical LOD
#                              adjustment.  Existing LOD adjustment output
#                              files in the trait/outdir will be overwritten.
#
#             -restorenull0    Restore the null0 model in effect at the time
#                              the last empirical LOD adjustment was
#                              calculated.  This will overwrite a later
#                              null0 model.
#
#             -summary  Display a summary of the LOD adjustment calculation.
#                       The summary shows the distribution of the original
#                       and adjusted LOD scores, the number of replicates
#                       performed, and the name of the null model.
#
# Notes:     The -calc option produces output files in the trait/outdir:
#            lodadj.out, lodadj.lods, and lodadj.info.  lodadj.out contains
#            summary information, lodadj.lods contains the raw actual vs.
#            theoretical LODs, and lodadj.info contains state information
#            including the null model currently in effect.
#
#            The lodadj value and state (on or off) is saved in each
#            trait/outdir (by the lodadj.info or lodadj.off file).  This
#            is now preserved when restarting SOLAR.
#
#            lodadj is now supported for bivariate lods.  Since the
#            correction is always computed with one additional degree of
#            freedom, the lodadj adjustment is applied AFTER the
#            lod correction to one degree of freedom, and the user is
#            advised not to disable the one degree of freedom correction
#            with the lodp command.
#-

# Additional private arguments:
#
#             -inform  <outfile>  Write message to <outfile>

proc lodadj {args} {

    full_filename test  ;# ensure trait/outdir specified

    global Solar_LOD_Adjustment

    set null 0
    set nreps -1
    set calc 0
    set restart 0
    set overwrite 0
    set cutoff -1
    set off 0
    set restorenull0 0
    set query 0
    set informfile 0
    set summary 0

    set badargs [read_arglist $args \
	    -null null \
	    -calc {set calc 1} \
	    -nreps nreps \
	    -restart {set restart 1} -r {set restart 1} \
	    -overwrite {set overwrite 1} -ov {set overwrite 1} \
	    -cutoff cutoff \
	    -off {set off 1} \
	    -restorenull0 {set restorenull0 1} \
	    -verify {set verify 1} \
	    -query {set query 1} \
	    -inform informfile \
	    -summary {set summary 1} \
	    ]

    set use_default_cutoff 0
    if {$cutoff < 0} {
        if {$cutoff == -1} {
            set cutoff .05
            set use_default_cutoff 1
        } else {
            error "Cutoff must be between 0 and 1"
        }
    } elseif {$cutoff > 1} {
        error "Cutoff must be between 0 and 1"
    }

    if {$summary} {
        if {$use_default_cutoff} {
	    lodadj -restart -nreps 0
        } else {
	    lodadj -restart -nreps 0 -cutoff $cutoff
        }
	return ""
    }

    if {$restart} {
        set calc 1
    }

    if {![is_integer $nreps]} {
	error "Invalid nreps value; must be integer"
    }
    if {$nreps != -1} {
	set calc 1
    } else {
	set nreps 10000
    }

    if {![is_integer $null] || $null < 0} {
	error "Invalid null value; must be a non-negative integer"
    }

    if {$off} {
	set Solar_LOD_Adjustment -1.0
	catch {
	    file rename -force [full_filename lodadj.info] [full_filename lodadj.off]
	}
	return ""
    }

    if {$restorenull0 == 1} {
	if {![file exists [full_filename lodadj.info]]} {
	    error "Can't find lodadj.info from previous lodadj -calc"
	}
	set ifile [open [full_filename lodadj.info] r]
	gets $ifile irecord
	set null 0
	if {[llength $irecord] == 6 || [llength $irecord] == 8}  {
	    set null [lindex $irecord 5]
	}
	set ofile [open [full_filename null$null.mod] w]
	while {-1 != [gets $ifile irecord]} {
	    puts $ofile $irecord
	}
	close $ifile
	close $ofile
	load model [full_filename null$null.mod]
	exec touch [full_filename lodadj.info]
	return [lodadj]
    }


#  *** QUERY ***

    if {$query} {

# If lodadj.info doesn't exist in current trait/outdir, lodadj is -off

	if {![file exists [full_filename lodadj.info]]} {
	    set Solar_LOD_Adjustment -1.0
	    set lodadj_value 1.0
	} else {

# Otherwise, read lodadj from lodadj.info

	    set lodadj_value [lodadj]

	}

# Output message to "informfile" (e.g. multipoint.out) if required

	if {$lodadj_value != 1.0} {
	    if {[string compare $informfile 0]} {
		if {"stderr" == $informfile || "stdout" == $informfile} {
		    set ifile $informfile
		    set mustclose 0
		} else {
		    set ifile [open [full_filename $informfile] a]
		    set mustclose 1
		}
		puts $ifile \
		 "\n    *** Using LOD Adjustment:  [format %.5f $lodadj_value]"
		if {$mustclose} {close $ifile}
	    }
	}

# Return lodadj value and QUERY is done

	return $lodadj_value
    }

# Now if not calculating, must be turning LOD adjustment on
#   (Or, knowing it's on, reading its value)

    if {!$calc} {
	if {![file exists [full_filename lodadj.info]]} {
	    if {[file exists [full_filename lodadj.off]]} {
		file rename -force [full_filename lodadj.off] [full_filename lodadj.info]
	    } else {
		error \
"No stored LOD adjustment found for this trait (lodadj.info|off not found)"
            }
        }
	set lodadj_time [file mtime [full_filename lodadj.info]]
	set ifile [open [full_filename lodadj.info] r]
	set lastring [gets $ifile]
        set ntraits 0
	close $ifile
	set null 0
	if {[llength $lastring] == 6 || [llength $lastring] == 8}  {
	    set null [lindex $lastring 5]
	    if {![is_integer $null]} {
	        purge_global Solar_LOD_Adjustment
	        error "Invalid lodadj.info file"
	    }
	}
	set null_time [file mtime [full_filename null$null.mod]]
	if {$lodadj_time < $null_time} {
	    error "The null$null model was changed after last lodadj -calc"
	}
	set Solar_LOD_Adjustment [lindex $lastring 1]
	if {![is_float $Solar_LOD_Adjustment]} {
	    purge_global Solar_LOD_Adjustment
	    error "Invalid lodadj.info file"
	}
	return $Solar_LOD_Adjustment
    }

# Calculate LOD adjustment...
    
    set outfile [full_filename lodadj.lods]

    if {![file exists [full_filename null$null.mod]]} {
	if {!$null} {
	    error "Model [full_filename null$null] not found.\
\nThis can be created with polygenic command."
	} else {
	    error "Model [full_filename null$null] not found."
	}
    }

    set soutf [full_filename siminf.out]
    set sibdf [full_filename ibd.siminf]

    set quiet 1
    if {[verbosity -number] > 0} {
        set quiet 0
    }

    set rep1 1
    set llods {}

    if {[file exists $outfile]} {
        if {$restart} {
            if {[file exists [full_filename lodadj.off]]} {

# If LOD adjustment is currently off, turn it on to restore lodadj.info

                lodadj
            }
            if {[file exists [full_filename lodadj.info]]} {

# If LOD adjustment is now on, get null model index from lodadj.info;
#   otherwise, lodadj.info was not created in previous run, so null
#   model index will come from -null argument or default to 0

                set ifile [open [full_filename lodadj.info] r]
                set lastring [gets $ifile]
                close $ifile
                set null 0
                if {[llength $lastring] == 6 || [llength $lastring] == 8}  {
                    set null [lindex $lastring 5]
                }
                if {$use_default_cutoff} {
                    if {[llength $lastring] == 8}  {
                        set cutoff [lindex $lastring 7]
                    }
                }
            }
            set outfd [open $outfile r]
            while {-1 < [gets $outfd line]} {
                if {1 == [scan $line "%f" tlod]} {
                    lappend llods $tlod
                    incr rep1
                } else {
                    error "Error reading [full_filename lodadj.lods]"
                }
            }
            close $outfd
            if {$nreps < $rep1} {
                set nreps [expr $rep1 - 1]
            }
            set outfd [open $outfile a]
        } else {
            if {!$overwrite} {
	        error "LOD adjustment output files exist. Use -overwrite option."
            }
            set outfd [open $outfile w]
        }
    } else {
        set outfd [open $outfile w]
    }

# Remove lodadj state file (if one exists) so that any stored LOD
#   adjustment information is completely gone before computing new
#   or additional lodadj reps

    delete_files_forcibly [full_filename lodadj.info]
    delete_files_forcibly [full_filename lodadj.off]

#    catch {
#	load model [full_filename null$null]
#        set ntraits [llength [trait]]
#        if {1 < $ntraits} {	
#	    puts "\n    Warning!  lodadj is not supported for Bivariate models!"
#	    puts "    You can calculate adjustment, but it may be inaccurate"
#	    puts "    and it will not be used.\n"
#	}
#    }

    for {set rep $rep1} {$rep <= $nreps} {incr rep} {
        if {!$quiet} {
            puts -nonewline "Computing LOD adjustment, replicate $rep "
            flush stdout
        }
        load model [full_filename null$null.mod]
        siminf -out $soutf -ibd $sibdf
        linkmod -add $sibdf
        set errmsg [maximize_quietly last]
        if {$errmsg != ""} {
            puts "    *** Error maximizing, rep $rep"
            set rep [expr $rep - 1]
            continue
        }

        lappend llods [lodn $null]
        puts $outfd [lodn $null]
        flush $outfd

        if {!$quiet} {
            puts -nonewline "\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b"
            puts -nonewline "\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b"
            flush stdout
        }
    }

    close $outfd
    set outfd [open $outfile w]

    set sllods [lsort -real $llods]
    set sxx 0
    set sxy 0

    for {set rep 0} {$rep < $nreps} {incr rep} {
        set x [lindex $sllods $rep]
        set t [expr double($rep) / double($nreps) ]
        if { $t <= 0.5} {
            set y 0
        } else {
            set y [chi -inverse [expr 2 * (1 - $t)] 1]
            set y [expr $y / 4.60517]
        }
        puts $outfd "$x $y"
        if {$rep < [expr (1 - $cutoff) * $nreps]} {
            set sxx [expr $sxx + $x * $x]
            set sxy [expr $sxy + $x * $y]
        }
    }

    close $outfd

    global Solar_LOD_Adjustment
    purge_global Solar_LOD_Adjustment
    set Solar_LOD_Adjustment 1
    if {$sxx} {
        set Solar_LOD_Adjustment [expr $sxy / $sxx]
    }

    set o5 0
    set o9 0
    set o95 0
    set o975 0
    set o99 0
    set o999 0
    set o9999 0
    set a5 0
    set a9 0
    set a95 0
    set a975 0
    set a99 0
    set a999 0
    set a9999 0

    for {set rep 1} {$rep <= $nreps} {incr rep} {
        set y [lindex $sllods [expr $rep - 1]]
        if {$y < 3.003382} {
            set o9999 $rep
            if {$y < 2.073655} {
                set o999 $rep
                if {$y < 1.175178} {
                    set o99 $rep
                    if {$y < 0.834162} {
                        set o975 $rep
                        if {$y < 0.587501} {
                            set o95 $rep
                            if {$y < 0.356637} {
                                set o9 $rep
                                if {$y < 0.0000001} {
                                    set o5 $rep
                                }
                            }
                        }
                    }
                }
            }
        }

        set ay [expr $y * $Solar_LOD_Adjustment]
        if {$ay < 3.003382} {
            set a9999 $rep
            if {$ay < 2.073655} {
                set a999 $rep
                if {$ay < 1.175178} {
                    set a99 $rep
                    if {$ay < 0.834162} {
                        set a975 $rep
                        if {$ay < 0.587501} {
                            set a95 $rep
                            if {$ay < 0.356636} {
                                set a9 $rep
                                if {$ay < 0.0000001} {
                                    set a5 $rep
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    set outfile [full_filename lodadj.out]
    set outfd [open $outfile w]

    puts "                                                     "
    puts $outfd "                                                     "
    puts "    LOD Correction Constant = [format %10.6f $Solar_LOD_Adjustment]"
    puts $outfd "    LOD Correction Constant = [format %10.6f $Solar_LOD_Adjustment]"
    puts ""
    puts $outfd ""
    puts "    LOD          % Original    % Adjusted    % Normal"
    puts $outfd "    LOD          % Original    % Adjusted    % Normal"
    puts "    ---------    ----------    ----------    --------"
    puts $outfd "    ---------    ----------    ----------    --------"

    set opct [expr double($o5) / $nreps]
    set apct [expr double($a5) / $nreps]
    puts \
"    =  0          [format %8.6f $opct]      [format %8.6f $apct]      0.5000"
    puts $outfd \
"    =  0          [format %8.6f $opct]      [format %8.6f $apct]      0.5000"

    set opct [expr 1 - (double($o9) / $nreps)]
    set apct [expr 1 - (double($a9) / $nreps)]
    puts \
"    >= 0.357      [format %8.6f $opct]      [format %8.6f $apct]      0.1000"
    puts $outfd \
"    >= 0.357      [format %8.6f $opct]      [format %8.6f $apct]      0.1000"

    set opct [expr 1 - (double($o95) / $nreps)]
    set apct [expr 1 - (double($a95) / $nreps)]
    puts \
"    >= 0.588      [format %8.6f $opct]      [format %8.6f $apct]      0.0500"
    puts $outfd \
"    >= 0.588      [format %8.6f $opct]      [format %8.6f $apct]      0.0500"

    set opct [expr 1 - (double($o975) / $nreps)]
    set apct [expr 1 - (double($a975) / $nreps)]
    puts \
"    >= 0.834      [format %8.6f $opct]      [format %8.6f $apct]      0.0250"
    puts $outfd \
"    >= 0.834      [format %8.6f $opct]      [format %8.6f $apct]      0.0250"

    set opct [expr 1 - (double($o99) / $nreps)]
    set apct [expr 1 - (double($a99) / $nreps)]
    puts \
"    >= 1.175      [format %8.6f $opct]      [format %8.6f $apct]      0.0100"
    puts $outfd \
"    >= 1.175      [format %8.6f $opct]      [format %8.6f $apct]      0.0100"

    set opct [expr 1 - (double($o999) / $nreps)]
    set apct [expr 1 - (double($a999) / $nreps)]
    puts \
"    >= 2.074      [format %8.6f $opct]      [format %8.6f $apct]      0.0010"
    puts $outfd \
"    >= 2.074      [format %8.6f $opct]      [format %8.6f $apct]      0.0010"

    set opct [expr 1 - (double($o9999) / $nreps)]
    set apct [expr 1 - (double($a9999) / $nreps)]
    puts \
"    >= 3          [format %8.6f $opct]      [format %8.6f $apct]      0.0001"
    puts $outfd \
"    >= 3          [format %8.6f $opct]      [format %8.6f $apct]      0.0001"
    puts ""
    puts $outfd ""

    puts "    #Reps: $nreps    Cutoff: $cutoff    Null Model: [full_filename null$null.mod]"
    puts $outfd "    #Reps: $nreps    Cutoff: $cutoff    Null Model: [full_filename null$null.mod]"
    puts ""
    puts $outfd ""

    close $outfd

    delete_files_forcibly $soutf $sibdf.gz
    eval exec echo "#lodadj: $Solar_LOD_Adjustment  nreps: $nreps  null: $null  cutoff: $cutoff" \
         >[full_filename lodadj.info]
    eval exec cat [full_filename null$null.mod] >>[full_filename lodadj.info]
}


# solar::empp --
#
# Purpose:  Calculate an empirical p-value from lodadj results
# 
# Usage:    empp <lod>
#-

proc empp {args} {

    if {[catch {trait}]} {
	error "Trait must be specified first"
    }

    set lod [read_arglist $args ]
    if {$lod == ""} {
        error "A LOD must be specified"
    }
    if {[scan $lod "%f" tmp] != 1 || $lod <= 0} {
        error "A LOD greater than 0 must be specified"
    }

    if {![file exists [full_filename lodadj.lods]]} {
        error "File [full_filename lodadj.lods] not found. Have you run lodadj?"
    }

    set lodfile [open [full_filename lodadj.lods] r]
    set n 0
    set p 0
    while {-1 != [gets $lodfile record]} {
        incr n
        if {[scan $record "%f" olod] != 1} {
            error "Error reading [full_filename lodadj.lods], line $n"
        }
        if {$olod < $lod} {
            incr p
        }
    }

    if {$p < $n} {
        puts "p = [format "%.7g" [expr 1 - $p/double($n)]]"
    } else {
        puts "p < [format "%.7g" [expr 1/double($n)]]"
    }

    if {$n < 10000} {
        puts \
"Warning: Only $n observations in [full_filename lodadj.lods]; p-value may not be reliable."
    }
}

# solar::loadkin --
#
# Purpose:  Load a matrix named phi2.gz containing phi2 and delta7
#
# Usage:    loadkin
#
# Notes:    If the file phi2.gz does not exist, this command will be
#           silently ignored.  This command is mainly for scripts.  You
#           can perform the same action with a known matrix file with:
#
#           matrix load phi2.gz phi2 delta7
# -

proc loadkin {} {
    if {[file exists "phi2.gz"]} {
	matrix load phi2.gz phi2 delta7
    }
}

#
# Write to file AND terminal
#
proc putstee {args} {
    if {0==[string compare [lindex $args 0] -nonewline]} {
	puts -nonewline [lindex $args 2]
	puts -nonewline [lindex $args 1] [lindex $args 2]
    } else {
	puts [lindex $args 1]
	puts [lindex $args 0] [lindex $args 1]
    }
}

#
# Write (append) to filename AND terminal
#
proc putsteer {args} {
    if {0==[string compare [lindex $args 0] -nonewline]} {
	puts -nonewline [lindex $args 2]
	set ofile [open [lindex $args 1] a]
	puts -nonewline $ofile [lindex $args 2]
	close $ofile
    } else {
	puts [lindex $args 1]
	set ofile [open [lindex $args 0] a]
	puts $ofile [lindex $args 1]
	close $ofile
    }
}


# solar::putsout
#
# Purpose:  Write message to terminal and/or file [obsolescent]
#
# NOTE: This is inefficient and no longer recommended.  New code should
#       use putsa to append to a file and putsat to append to file
#       and write to terminal.  See help putso.
#
# Usage:    putsout [-q] [-d.] [-nonewline] <filename> <message>
#
#           -q            No output to terminal
#           -d.           Write file to current directory
#           -nonewline    As with puts command (note: may delay output)
#           <filename>    *name* of file in current output directory (outdir)
#           <message>     string
#
# Simple Example:  putsout mine.out "The result was $result"
#
#
# Advanced Example: (Beginners ignore!)
#
#            set q ""
#            ifverbmax set q "-q"
#            eval putsout $q \"Iteration: $i   Value: $value\"
#
# Note: If using a variable for -q which could be "", be sure to use
#       eval, otherwise "" would be considered filename argument, then
#       remember to \ the quotes or they disappear during the eval.
# -


proc inout {args} {
    return $args
}

proc putsout {args} {
    set terminal 1
    if {"-q" == [lindex $args 0]} {
	set terminal 0
	set args [lrange $args 1 end]
    }
    upvar putsout_quiet putsout_quiet
    if {![catch {set foo $putsout_quiet}] && \
	    $putsout_quiet} {
	set terminal 0
    }

    set outfunc full_filename
    if {"-d." == [lindex $args 0]} {
	set outfunc inout
	set args [lrange $args 1 end]
    }
    if {0==[string compare [lindex $args 0] -nonewline]} {
	if {$terminal} {puts -nonewline [lindex $args 2]}
	if {![catch {set ofile [open [$outfunc [lindex $args 1]] a]}]} {
	    puts -nonewline $ofile [lindex $args 2]
	    close $ofile
	}
    } else {
	if {$terminal} {puts [lindex $args 1]}
	if {![catch {set ofile [open [$outfunc [lindex $args 0]] a]}]} {
	    puts $ofile [lindex $args 1]
	    close $ofile
	}
    }
}

# solar::putsnew --
# solar::putsat --
# solar::putsa --
#
# Purpose: Write to file and/or terminal without having to open and close
#
# Usage: putsnew <filename>            ;# create new file or truncate old file
#        putsa <filename> <string>     ;# append to file
#        putsat <filename> <string>    ;# append to file and write to terminal
#
# Notes:
#
# 1.  putsnew creates the file if not yet created, or truncates existing
#     file to zero length.  This is generally needed before starting to
#     append to file using putsa unless you know the file already exists.
#
# 2.  These procedures are NOT as efficient as Tcl's built-in operations
#     for writing to a file (open, puts, close).  If efficiency is
#     paramount, and you have a tight writing loop not subject to
#     exceptions or with fully handled exceptions, it is preferable to
#     use the Tcl operations.  However these procedures are believed to
#     be as efficient as shell append piping (>>).  These procedures are
#     considerably more efficient than the now obsolescent "putsout".
#
# 3.  putsa writes (appends) to the existing file
#
# 4.  putsat writes (appends) to the existing file, and to the terminal
#
# 5.  If an absolute filename is not specified, the file path is relative to
#     the current working directory.  If you need to write to a file in the
#     maximization output directory, use the procedure full_filename first
#     to produce the required pathname.  Beware that the pathname produced
#     by full_filename is also relative to the current working directory, so
#     that if you change the current working directory, or the trait, or the
#     outdir, you should run full_filename again, if you want to continue
#     writing to a file in the current maximization directory (though it
#     will then be a different directory and a different file).
#
#  Example:
#
#       # writing to file in maximization output directory
#       set myfullname [full_filename myname.out]
#       putsnew $myfullname
#       putsat $myfullname "Writing to $myfullname"
#       putsa $myfullname "writing more, but not to terminal this time"
#       trait newtrait
#       set myfullname [full_filename myname.out]
#       putsnew $myfullname
#       putsat $myfullname "Now writing to $myfullname"
# -

proc putsnew {filename} {
    set outfile [open $filename w]
#    puts -nonewline $outfile ""
    close $outfile
}

proc putsa {filename string} {
    set outfile [open $filename a]
    puts $outfile $string
    close $outfile
}

proc putsat {filename string} {
    set outfile [open $filename a]
    puts $outfile $string
    close $outfile
    puts $string
}

# solar::full_filename --
#
# Purpose:  Prepend the maximization output directory name to filename(s)
#
# Usage:    full_filename [<filename>]+
#
# Note:     See "help outdir".  full_filename is intended for scripts.
# -

proc full_filename {args} {
    if {[if_global_exists Solar_Out_Dir]} {
	global Solar_Out_Dir
	set pathname $Solar_Out_Dir
    } else {
	if {[catch {set pathname [join [trait] .]}]} {
	    error "Trait, outdir, or model must be specified first"
	}
    }
    if {"Must_give_command_model_new" == $pathname} {
	error "Invalid trait change; must give command 'model new' first"
    }
    if {![file exists $pathname]} {file mkdir $pathname}
    set outputnames {}
    foreach filename $args {
	lappend outputnames "$pathname/$filename"
    }
    return $outputnames
}
 
#
# Procedures to get information from model files
#

# solar::lod_or_slod_n -- private
#
# Purpose:  Calculate current lod or slod relative to nullN
#
# Usage: current_lod_or_slod_n ifslod n  ;# ifslod is 0 or 1
# -
proc lod_or_slod_n {ifslod n} {
    if {$ifslod==1} {
	set newn [expr $n + 1]
	set rlod [slod h2q$newn]
    } else {
	set rlod [lodn $n]
    }
    return $rlod
}

# solar::oldlod -- private
#
# Purpose:  Calculate LOD score for current model (relative to nullX.mod)
#
# Usage:    lod
#
# Notes:    The current model must have been maximized, either through the
#           "twopoint" or "multipoint" command, or directly with the
#           "maximize" command.
#
#           The null model should be saved as nullX.mod (for example, null0.mod
#           or null1.mod) where X is the number of active linkage elements,
#           which is assumed to be one less than the current model for this
#           command.  Linkage parameters must be specified as h2q1, h2q2,
#           etc.  The null model should be saved in the maximization output
#           directory.  Then, the loglikelihood of the current model is
#           compared with the loglikelihood of the model in the applicable
#           null model to get the LOD.
#
#           SOLAR LODs are intended to be easy to interpret in the usual
#           cases.  By default, SOLAR applies empirical LOD adjustment, if
#           currently in effect (see lodadj command), and also converts the
#           LOD to be equivalent to a 1dF univariate LOD.  If you need to
#           change these defaults, use the lodp command, which sets LOD
#           calculation preferences.  LODs are actually calculated by
#           the clod command, which lets you specify the likelihood values
#           and options directly.  The "lod" command is merely a simplified
#           interface to clod.
#
#           If you are attempting to do some special statistical test,
#           for example, involving the constraint of rho parameters, it
#           may be preferable to use the raw loglikelihood values, which are
#           reported in models and multipoint1.out files, and accessible
#           for the current model with the loglike command.  That way any
#           default conversions built in to the LOD calculation are bypassed.
#
#           See also: lodp, clod, lodn, lodadj, loglike, read_model
# -

# solar::lodn --
#
# Purpose:  Calculate LOD score for current model relative to nullX
#
# Usage:    lodn X <options>
#
#                X   Number indicating index of relevant null model (for
#                    example, 0 for null0, the model having no linkage
#                    elements).
#                <options>  See "help lod".
#
# Notes:    In many cases you can more easily use the "lod" command, which
#           determines the applicable null model automatically, or, you can
#           specify the loglikelihoods.  "lodn" may be useful if you
#           are not sure whether the current model contains h2q parameters,
#           for example, if it includes a custom parameterization.
#
#           The current model must have been maximized, either through the
#           "twopoint" or "multipoint" command, or directly with the
#           "maximize" command.
#
#           The null model should be saved as nullX.mod (for example, null0.mod
#           or null1.mod) where X is the number of active linkage elements.
#
#           There are many special options for LOD calculation.  See
#           "help lodp" for more information.  The primary LOD calculating
#           procedure in SOLAR is "lod" which lets you specify the
#           loglikelihood values and option(s) directly.
#
#           See also lod, lodp.
# -

proc lodn {N args} {return [eval lod [loglike] [nulln $N loglike] $args]}

# solar::lodp
#
# Purpose:  Change LOD preferences (such as conversion to 1df)
#
# Usage:    lodp [-auto | -off | [-traits <N>] [-rhoq <N>]]
#                [-1t|-2t|-3t|-4t|-t1|-t2|-t3|-t4]
#
#           (If no argument is given, current preferences are displayed.)
#
#           -auto    Convert LODs to 1 degree of freedom (1dF) effective LODs 
#                    automatically based on traits and rhoq constraints
#                    in current model (default).
#
#           -off     Do not perform LOD conversion to 1 df equivalence
#
#           -traits <N>      Convert assuming there are/were <N> traits
#           -1t or -t1 etc.  Shortcuts for 1 trait, 2 traits, etc. up to 4
#           -rhoq <N>        Convert assuming <N> rhoq's are constrained
#
# Notes:    If -traits is specified without -rhoq, -rhoq is assumed to be 0.
#           If -rhoq is specified without -traits, trait count is determined
#           automatically (and might be determined to be 1, in which case
#           rhoq specification is irrelevant).  If you need to set both
#           -traits and -rhoq, you must give both in the same lodp command.
#
#           This should not be confused with lodadj (see).  The lodp command
#           sets global preferences for "degrees of freedom" issues which
#           arise with multivariate models.  The default "-auto" will
#           apply conversion based on the number of traits in the current
#           model and the number of relevant rhoq's (defined below) which
#           are constrained.  LODs will be converted to 1 degree of freedom
#           effective LODs (for which the traditional cutoff in statistical
#           genetics for a genome-wide linkage scan is 3).
#
#           Relevant rhoq's are parameters prefixed rhoq which correspond
#           to the highest numbered linkage element.  For example, in a
#           bivariate linkage model with one linkage element, the relevant
#           rhoq whould be "rhoq1", but with two linkage elements, it would
#           be "rhoq2".  For a trivariate model with one linkage element,
#           the relevant rhoq's would be: rhoq1_12, rhoq1_13, rhoq1_23.
#
#           The preferences set by this command will apply to all LOD scores
#           subsequently calculated by SOLAR, including those reported by
#           the twopoint and multipoint commands, and the lod and lodn
#           commands.  The lod command, which is what ultimately
#           calculates all LOD scores, has options which are similar to
#           lodp.
#
#           Changes to lodp preferences apply only within the current
#           SOLAR session, so the command must be re-entered each time
#           SOLAR is started or at the beginning of SOLAR scripts when
#           you need to change the defaults.
#
#           For more discussion of the how the conversion is performed,
#           which rhoq constraints are relevant, etc., see help for
#           the lod command.
#           
#           See also lod, lodn, and lodadj.
#-

proc loddf {args} {
    error "loddf has been replaced by the lodp command; see help lodp"
}

proc lodp {args} {

# If no arguments, return current state from globals
# If all globals are blank, return default state -auto

    global SOLAR_LODP
    global SOLAR_LODP_RHOQ
    if {{} == $args} {
	set lodp ""
	set rho ""
	if {[if_global_exists SOLAR_LODP]} {
	    set lodp $SOLAR_LODP
	    if {"Error!" == $lodp} {
		error "Error in last lodp specification!"
	    }
	}
	if {[if_global_exists SOLAR_LODP_RHOQ]} {
	    set rho $SOLAR_LODP_RHOQ
	}
	set state -auto
	if {"" != $lodp} {
	    set state $lodp
	    if {"" != $rho} {
		set state "$lodp $rho"
	    }
	} elseif {"" != $rho} {
	    set state $rho
	}
	return $state
    }

# If attempt to change preferences fails, we fall into an error state

    set SOLAR_LODP Error!
    set SOLAR_LODP_RHOQ ""

# Handle -off

    if {-1 != [lsearch -exact $args "-off"]} {
	if {"-off" != $args} {
	    error "lodp: -off is incompatible with other arguments"
	}
	set SOLAR_LODP -off
	set SOLAR_LODP_RHOQ ""
	return ""
    }

# Handle -auto

    if {-1 != [lsearch -exact $args "-auto"]} {
	if {"-auto" != $args} {
	    error "lodp: -auto is incompatible with other arguments"
	}
	set SOLAR_LODP -auto
	set SOLAR_LODP_RHOQ ""
	return ""
    }

# Handle -trait and -rhoq

    set traits ""
    set rhoq ""

    set badargs [read_arglist $args \
		     -traits traits \
		     -rhoq rhoq \
		     -1t {set traits 1} \
		     -2t {set traits 2} \
		     -3t {set traits 3} \
		     -4t {set traits 4} \
		     -t1 {set traits 1} \
		     -t2 {set traits 2} \
		     -t3 {set traits 3} \
		     -t4 {set traits 4} \
		    ]

    if {"" != $badargs} {
	error "lodp: Invalid arguments: $badargs"
    }

    if {"" != $traits} {
	ensure_integer $traits
    }

    if {"" != $rhoq} {
	ensure_integer $rhoq
    }


# At this point, no more errors are possible

    set SOLAR_LODP ""
    set SOLAR_LODP_RHOQ ""

    if {"" != $traits} {
	set SOLAR_LODP "-traits $traits"
    }

    if {"" != $rhoq} {
	set SOLAR_LODP_RHOQ "-rhoq $rhoq"
    }
    
    return ""
}


# solar::clod --
# solar::lod --
#
# Purpose:  Calculate LOD score
#
# Usage:    lod [<test-loglike> <null-loglike>] [<options>]
#           options := [-auto|-off|-raw] [-trait <N>] [-rhoq <N>] [-v]
#                      [-1t|-2t|-3t|-4t|-t1|-t2|-t3|-t4] [-nolodadj]
#
#           If no likelihoods are specified, the likelihoods of the
#           "current" model and the applicable "null" model are used.
#
#           -auto        Convert multivariate LOD to 1df effective LODs based
#                          on number of traits in current model and constraint
#                          of relevant rhoq's (default)
#           -off         Do not convert LODs to 1df effective
#           -raw         Do not perform LOD conversion or lodadj
#           -traits <N>  Convert multivariate LOD to 1dF assuming <N> traits
#           -1t or -t1   Assume 1 trait (same as "-traits 1")
#           -2t or -t2   Assume 2 traits (same as "-traits 2")
#           -3t or -t3   Assume 3 traits (same as "-traits 3")
#           -4t or -t4   Assume 4 traits (same as "-traits 4")
#           -rhoq <N>    Convert multivariate LOD to 1df assuming <N>
#                          constraints of relevant rhoq's
#           -nolodadj    Do not perform lod adjustment (lodadj)
#           -v           verbose: Show adjustment and conversion steps
#
# Examples: outdir test1
#           load model test1/null1
#           lod
#           lod -v
#           lod -2000.51 -2030.87
#           lod -trait 3 -rhoq 1 -v -2000 -2030
#           lod -raw -2000 -2030
#
# Notes:    If no likelihoods are specified, the current model must have
#           been maximized through a command such as "maximize," "twopoint",
#           or "multipoint", and the applicable null model should be saved as
#           nullX.mod (e.g. null0.mod, null1.mod) where X is the number
#           of active linkage elements, which is assumed to be one less
#           linkage element than in the current model.  Linkage elements are
#           parameters named h2q1, h2q2, etc.  The null model must have
#           been saved in the maximization output directory, either named
#           after the trait or set by the outdir command.
#
#           By default, SOLAR provides easily interpreted "1 df effective" LODs
#           which are equivalent to those in univariate models.
#           However, you can also have complete control over the LOD
#           conversion performed either using arguments here or
#           preferences set globally with the lodp command.  Options
#           specified here override the defaults and lodp preferences.
#
#           The correction of 2 trait LODs to 1dF effective LODs is based
#           on this formula: the LOD is converted to chi square with
#           1/2 1df, 1/4 3df, and 1/4 point mass at zero.  If rhoq is
#           constrained, the formula is 1/2 1df, 1/4 2df, and 1/4
#           point mass at zero.  This is then converted to a 1/2 1df
#           chi square of equivalent p-value, which is divided by
#           2ln10 to get the 1df effective lod score.
#
#           The correction of 3 trait LODs to 1dF effective LODs is based
#           on the formula: the LOD is converted to chi square with
#           3/8 1df, 3/8 3df, 1/8 6df, and 1/8 point mass at zero.
#           For each rhoq constrained, the 6df is changed downward
#           by 1df.
#
#           The conversion of higher multivariate LODs follows a similar
#           expanding sum.  If you wish to see the weights used, use the
#           lod command with the -v option.
#
#           Empirical LOD adjustment, if any, is automatically applied (see
#           the lodadj command) unless the -raw option is used.  Unless you
#           specify -raw, SOLAR will need to search the output directory for
#           a lodadj.info file, which means that a trait or outdir must
#           have been selected.
#
#           Empirical LOD adjustment is not yet supported for bivariate
#           models.  The lodadj value is ignored when bivariate LODs are
#           computed, and, in the cases where the lodadj value would be
#           shown (such as in the multipoint.out file, or if lod is called
#           from the command prompt) a warning message is shown instead.
#
#           In SOLAR version 3.0.2, the "clod" and "lod" commands were
#           combined into a new "lod" command.  The options allowed
#           have changed compared with the earlier "clod" ; the original
#           "lod" command did not allow any arguments.
#
#           Use the "lodn" command if you the current model may not use
#           the "h2q1" linkage parameter and you are not specifying
#           loglikelihoods explicitly.
#           
#           See also lodn, lodp, lodadj.
# -

proc clod {args} {
    return [eval lod $args]
}

proc lod {args} {

# Read and check arguments...note they changed in 3.0.2

    set auto_arg ""
    set traits_arg ""
    set rhoq_arg ""
    set cverb 0
    set nolodadj 0

    set nonoptions [read_arglist $args \
			-auto {set auto_arg -auto} \
			-off {set auto_arg -off} \
			-raw {set auto_arg -raw} \
			-traits traits_arg \
			-1t {set traits_arg 1} \
			-2t {set traits_arg 2} \
			-3t {set traits_arg 3} \
			-4t {set traits_arg 4} \
			-t1 {set traits_arg 1} \
			-t2 {set traits_arg 2} \
			-t3 {set traits_arg 3} \
			-t4 {set traits_arg 4} \
			-rhoq rhoq_arg \
			-nolodadj {set nolodadj 1} \
			-v {set cverb 1} \
			-* returned
		       ]

# If no likelihoods specified, get then from current and null models

    if {[llength $nonoptions] == 0} {
	set x [expr [h2qcount] - 1]
	if {$x < 0} {
	    error \
		    "lod: Unable to determine required null model\n\
Current model does not have a h2q1 parameter\n\
Note: Specify loglikelihoods or use lodn command for custom parameterizations"
	}
	return [eval lodn $x $args]
    }
    if {[llength $nonoptions] == 1} {
	error "lod: Invalid option or only one loglikelihood specified"
    }

    set invalid_options [lrange $nonoptions 2 end]

    set logl [lindex $nonoptions 0]
    set null_logl [lindex $nonoptions 1]

    if {![is_float $logl]} {
	lappend invalid_options $logl
    }
    if {![is_float $null_logl]} {
	lappend invalid_options $null_log
    }
    if {{} != $invalid_options} {
	error "lod: Invalid options:  $invalid_options\n\
Note: Options have changed!  See help for lod."
    }

    if {"" != $traits_arg} {
	ensure_integer $traits_arg
    }

    if {"" != $rhoq_arg} {
	ensure_integer $rhoq_arg
    }

# Initialize default preferences

    set auto_mode -auto
    set assume_traits ""
    set assume_rhoq ""

# Get preferences from lodp

    set lodp [lodp]
    if {$cverb} {
	puts "Preferences from lodp: $lodp"
    }
    if {[string_imatch $lodp -auto]} {
	set auto_mode -auto
    } elseif {[string_imatch $lodp -off]} {
	set auto_mode -off
    } else {
	read_arglist $lodp \
	    -traits assume_traits \
	    -rhoq assume_rhoq
	set auto_mode ""
    }

# Let arguments supercede preferences

    if {"" != $auto_arg} {
	set auto_mode $auto_arg
	set assume_traits ""
	set assume_rhoq ""
    }

    if {"" != $traits_arg} {
	set assume_traits $traits_arg
	set assume_rhoq ""
	set auto_mode ""
    }

    if {"" != $rhoq_arg} {
	set assume_rhoq $rhoq_arg
	set auto_mode ""
    }

# Assume 0 constraints if traits specified but contraints not specified

    if {"" != $assume_traits && "" == $assume_rhoq} {
	set assume_rhoq 0
    }

# assume we need to convert lod if mode defaulted or specified auto
#   and 1 trait NOT assumed...then reset if we find only 1 trait

    set convert_lod 0

    if {"-auto" == $auto_mode || \
	    "" == $auto_mode && ![string_imatch 1 $assume_traits]} {
	set convert_lod 1
    }

# Now, get information from current model IF we need it

    set use_traits $assume_traits
    set c_rhoqs $assume_rhoq

# See how many traits...if 1, then no conversion anyway

    if {$convert_lod} {

	if {"" == $assume_traits} {
	    if {[catch {set use_traits [llength [trait]]}]} {
		error "lod: No current model...need to know how many traits"
	    } elseif {$cverb} {
		puts "Found $use_traits traits"
	    }
	}
	if {$use_traits == "1"} {
	    set convert_lod 0
	}
    }

# Search for constrained rhoqs at the current h2q level

    if {$convert_lod && "" == $assume_rhoq} {

	set finished 0
	catch {
	    set c_rhoqs 0
	    set index [h2qcount]
	    set vclist [get_vc_list]
	    set vcend [expr 3 + [string length $index]]
	    foreach vc $vclist {
		if {[string_imatch rhoq$index [string range $vc 0 $vcend]]} {
		    set lcindex [expr $vcend + 1]
		    set endchar [string range $vc $lcindex $lcindex]
		    if {![string compare "" $endchar] || \
			    ![string compare _ $endchar]} {
			if {![catch {find_simple_constraint $vc}]} {
			    incr c_rhoqs
			}
		    }
		}
	    }
	    set finished 1
	}
	if {!$finished} {
	    error "Unable to determine if any rhoq parameters are constrained."
	}
	if {$cverb} {
	    puts "Found $c_rhoqs constrained rhoq's"
	}
    }

    if {"" == $c_rhoqs} {
	set c_rhoqs 0
    }

# Compute LOD, return here if raw mode

    set rawlod [expr ($logl - $null_logl) / log (10)]
    if {$cverb} {
	puts "Raw LOD = ($logl - $null_logl) / log(10) = $rawlod"
    }
    if {"-raw" == $auto_mode} {
	return $rawlod
    }

# See if lod correction to 1df is required

    if {!$convert_lod} {
	if {$cverb} {
	    puts \
"Only one trait so correction to 1df is not required"
	}
	set lod1 $rawlod
    } else {

# Convert bivariate or trivariate LOD to 1dF

    if {$cverb} {
        puts \
           "Correcting LOD to 1df from $use_traits traits and $c_rhoqs constrained rhoq's"
    }

    set chisq [expr 2*log(10)*$rawlod]

    set negate 0
    if {$chisq < 0} {
	set chisq [expr 0.0 - $chisq]
	set negate 1
    }

    if {$cverb} {puts "Chisq = 2*log(10)*abs(LOD) =  $chisq"}

    set p 0.0
    set n $use_traits
    for {set i 1} {$i <= $n} {incr i} {
	set weight [expr pow(0.5,$n) * [factorial $n] / \
			([factorial $i]*[factorial [expr $n - $i]])]
	set cdf [expr $i*($i+1)/2]
	
	if {$i == $n && $c_rhoqs} {
	    set cdf [expr $cdf - $c_rhoqs]
	}
	
	set thisp [expr $weight*[chi -number $chisq $cdf]]
	
	if {$cverb} {
	    puts "Weight: $weight  df: $cdf  w*p: $thisp"
	}
	set p [expr $p + $thisp]
    }

    set usep [expr 2.0*$p]
    if {$cverb} {
	puts "2*sum(w*p's) = $usep"
    }
    if {$usep < 1.0} {
	set newchisq [chi -inverse $usep 1]
	if {$cverb} {
	    puts "Corrected chisq = \[chi -inverse $usep 1\] = $newchisq"
	}
	set lodc1 [expr $newchisq / (2.0*log (10))]
	if {$cverb} {
	    puts "abs(LOD) = $newchisq / (2*log(10)) = $lodc1"
	}
	if {$negate} {
	    set lodc1 [expr 0.0 - $lodc1]
	    if {$cverb} {
		puts "Negated."
	    }
	}
    } else {
	set lodc1 0.0
    }
    set lod1 $lodc1
    }

# Apply lodadj, if applicable

    if {!$nolodadj} {
	if {[catch {set loda [lodadj -query]}]} {
	    error "lod: trait or outdir not specified to find lodadj.info"
	}
	if {$cverb || 1 == [info level]} {
	    lodadj -query -inform stdout
	}
	set lod1 [expr $loda * $lod1]
    }

   return $lod1
}

# solar::factorial --
#
# Purpose:  Compute factorial
#
# Usage:    factorial N
#
# Example:  set big [factorial 10]
#
# Notes:    A double precision value is returned, since double precision can
#           represent larger numbers exactly than integers.
#
#           Non-integral N is rounded to the nearest integer first, then
#             the factorial is computed for that integer.
#
#           For large enough N, the value returned might not be exact.
#              (Currently this happens for N > 18.)
#
#           Negative N (after rounding) raises a range error
#
#           This may be, but need not be, used in an "expr".
# -

proc factorial {n} {
    if {[catch {set n [expr round($n)]}]} {
	error "factorial: Syntax error: $n is not a number"
    }

    if {$n < 0} {
	error "factorial: $n is out of range"
    } elseif {$n < 2} {
	set fact 1.0
    } else {
	set fact 1.0
	for {set i 2} {$i <= $n} {incr i} {
	    set fact [expr $fact * $i]
	}
    }
    return $fact
}
	

# solar::slod --
#
# Purpose:  Calculate slod (score-based LOD equivalent) on current model
#
# Usage:    slod newparam
#
# Example:  slod h2q1
#-

proc slod {np} {
    set pscore [parameter $np score]
    if {$pscore <= 0.0} {
	return 0.0
    }
    set se [parameter $np se]
    if {0.0 == $se} {
	error "Unable to evaluate stest for parameter $np: missing S.E."
    }
    set stest [expr $pscore*$pscore*$se*$se]
    set slod [expr $stest/(2.0*log(10))]
    return $slod
}

# solar::null --
#
# Purpose:  Return an optimized parameter from null.mod
#
# Usage:   null loglike
#          null h2q
#
# Notes:   see also nulln
# -

proc null {parameter_name} {
    return [oldmodel "null" $parameter_name]
}

# solar::nulln --
#
# Purpose:  Return an optimized parameter from nullX.mod
#            (e.g. null0.mod,  null1.mod, etc.).
#
# Usage:   nulln 0 loglike
#          nulln 1 loglike
#          nulln 2 h2q1
# -

proc nulln {N parameter_name} {
    return [oldmodel "null$N" $parameter_name]
}

# solar::Solar_Model_Startup -- private
#
# Purpose: To run model initialization file .solar_model_new
#
# Note: Since this is invoked by "model new", it must not itself run model new
#       or any other command, such as "load model" which does.  Hence the only
#       allowed commands are: 
#
#       option trait parameter covariate constraint omega define
#
# 
# -

proc Solar_Model_Startup {} {
    if {[file exists .solar_model_new]} {
	set allowed {option trait parameter covariate constraint omega define}
	set commands [listfile .solar_model_new]
	foreach command $commands {
	    if {[llength $command] && [string index $command 0] != "\#"} {
		set commandname [lindex $command 0]
		if {-1 == [lsearch $allowed $commandname]} {
		    error "\nForbidden command \"$commandname\" in .solar_model_new\nPermitted commands: $allowed\n"
		}
		if {[catch {eval $command}]} {
		    error "\nError executing command in .solar_model_new\n  $command\n"
		}
	    }
	}
	if {[info level] == 1} {
	    puts "executed commands in .solar_model_new"
	}
    }
    return ""
}


# solar::read_model --
#
# Purpose:  Read a parameter value or likelihood from any saved model
#
# Usage:    read_model <model-name> loglike             ; returns loglikelihood
#           read_model <model-name> <parameter>         ; returns mle value
#           read_model <model-name> <parameter> -se     ; standard error
#           read_model <model-name> <parameter> -lower  ; lower bound
#           read_model <model-name> <parameter> -upper  ; upper bound
#           read_model <model-name> <parameter> -score  ; score
#
# Model is read from current maximization output directory (see
# help outdir).
#
# Example:
#
#            trait q4
#            read_model null0 h2r
# -

proc read_model {model_name parameter_name {getfield -value}} {
    return [oldmodel $model_name $parameter_name $getfield]
}

proc oldmodel {model_name parameter_name {getfield -value}} {

# loglike handled by proc modloglike

    if {0 == [string compare $parameter_name loglike]} {
	return [modloglike $model_name]
    }

# open file and scan for parameter data

    set f [open [full_filename [append_mod $model_name]] r]
    set parameter_seen 0
    while {-1 < [gets $f line]} {

# See if this is a matching parameter statement

	if {1 == [scan $line "parameter %s" pname] && \
		[string_imatch $parameter_name $pname]} {
	    set parameter_seen 1

# See if info requested is found

	    for {set i 2} {$i < [llength $line]-1} {incr i 2} {
		set token [lindex $line $i]
		set value [lindex $line [expr $i + 1]]
		if {"-value" == $getfield} {
		    if {[string_imatch = $token] || \
			    [string_imatch start $token]} {
			close $f
			return $value
		    }
		} elseif {"-lower" == $getfield} {
		    if {[string_imatch lower $token]} {
			close $f
			return $value
		    }
		} elseif {"-upper" == $getfield} {
		    if {[string_imatch upper $token]} {
			close $f
			return $value
		    }
		} elseif {"-se" == $getfield} {
		    if {[string_imatch se $token]} {
			close $f
			return $value
		    }
		} elseif {"-score" == $getfield} {
		    if {[string_imatch score $token]} {
			close $f
			return $value
		    }
		} else {
		    close $f
		    error "oldmodel: Invalid field requested: $getfield"
		}
	    }
	}
    }
    close $f

# Try old parameter name convention (used : instead of * in bage*sex)

    if {-1 != [string first * $parameter_name]} {
	set new_parameter_name [substar $parameter_name :]
#	puts "retrying parameter $parameter_name as $new_parameter_name"
	if {0 == [catch {oldmodel $model_name $new_parameter_name $getfield} \
		rv]} {
	    return $rv
	}
    }

# If standard error or score weren't found, but parameter was, return 0

    if {"-se" == $getfield || "-score" == $getfield} {
	if {$parameter_seen} {
	    return 0
	}
    }

# Error: data not found

    set field_name [string range $getfield 1 end]
    error "$field_name for parameter $parameter_name not found in $model_name"
    return ""
}

	
# solar::find_string -- private
#
# Purpose:  Return the first line containing any string from a file.
#           If the string is not found, the empty string is returned.
#           (Note: comparison is case insensitive)
#
# Example: find_string poly.out "The analysis sample size is"
# -

proc find_string {filename keystring} {
    set f [open $filename r]
    set ikeystring [string tolower $keystring]
    while {-1 < [gets $f line]} {
	set iline [string tolower $line]
	if {0 <= [string first $ikeystring $iline]} {
	    close $f
	    return $line
	}
    }
    close $f
    return ""
}

proc outfile_sample_size {outfile_name} {
    set sstring [find_string [full_filename $outfile_name] \
	    "analysis sample size is"]
    if {"" != $sstring} {
	set ll [llength $sstring]
	set last_elem [lindex $sstring [expr $ll - 1]]
	set size [expr round($last_elem)]
	return $size
    }
    error "Can't find analysis sample size in output file $outfile_name"
}

proc oldsporadic {model_name} {
    set sstring [find_string [full_filename [append_mod $model_name]] \
	    "constraint h2r = 0"]
    if {"" == $sstring} {
	return 0
    }
    return 1
}

	
proc modloglike {model_name} {
    set f [open [full_filename [append_mod $model_name]] r]
    while {-1 < [gets $f line]} {
	if {1 == [scan $line "loglike set %s" number]} {
	    close $f
	    return $number
	}
    }
    close $f
    error "Loglikelihood not saved"
}


# solar::usage
#
# Purpose:  Print short "usage" message about a command
#
# Usage:    usage <command>
#
# Example:  usage multipoint    ;# shows usage of multipoint command
#
# Notes:  Since this is printed directly to terminal, it will stay visible for
#         next command.
#
#         If help message contains no "Usage" section, the first 15 lines will
#         be printed.
#
# -

proc usage {command} {
    help-on-command -usage $command
    return ""
}

# solar::help-list-of-commands -- private
#
# Purpose:  Get a list of documented commands
#
# Usage:    help-list-of-commands [-describe <doclist>] [-user]
#         
#           <doclist> becomes an array of one-line descriptions for each
#            command
#
#           -user   specifies search only for user commands (all *.tcl
#                   files in . and ~/lib execpt solar.tcl)
#
#                   (otherwise, search active solar.tcl and all commands
#                    included in $SOLAR_LIB are included)
#-

proc help-list-of-commands {args} {
    global env
    set descriptions ""
    set user 0
    set carray(0) 0
    read_arglist $args -describe descriptions -user {set user 1}
    if {"" != $descriptions} {
	upvar $descriptions command_array
    }

    set command_list {}
    if {$user} {
	set homelib $env(HOME)/lib
	set user_paths ". $homelib"
	set sourcefilenames ""
	foreach d $user_paths {
	    set candidates [glob -nocomplain $d/*.tcl]
	    foreach candidate $candidates {
		if {"solar.tcl" != [file tail $candidate]} {
		    lappend sourcefilenames $candidate
		}
	    }
	}
    } else {
	set script_path [find_script_path solar.tcl]
	set sourcefilenames "$script_path/solar.tcl"
	set libfilenames [glob -nocomplain $env(SOLAR_LIB)/*.tcl]
	foreach libfilename $libfilenames {
	    if {"solar.tcl" != [file tail $libfilename]} {
		lappend sourcefilenames $libfilename
	    }
	}
    }
    ifdebug puts "searching files $sourcefilenames"
    foreach sourcefilename $sourcefilenames {
	set sourcefile [open $sourcefilename r]
	set packagename [file tail $sourcefilename]
	set endchar [expr [string first . $packagename] - 1]
	set packagename [string range $packagename 0 $endchar]
	set targetname "\# [catenate $packagename ::]"
	set coffset [expr [string length $targetname] - 0]
#	ifdebug puts "packagename: $packagename  coffset: $coffset"
    while {-1 != [gets $sourcefile source_line]} {
	set subs [string range $source_line 0 [expr $coffset - 1]]
	if {0 == [string compare $subs $targetname]} {
	    set sbase [string range $source_line $coffset end]
	    if {1 == [scan $sbase "%s" command_name]} {
		if {-1 != [lsearch -exact $source_line private]} {
		    continue
		}
		if {$user && \
		    ![catch {set aaa $carray($command_name)}]} {
		    continue
		} elseif {$user} {
		    set carray($command_name) 1
		}
		lappend command_list $command_name
#
# Get 'Purpose' statement and put into associative array
#
		if {"" == $descriptions} {continue}
		set last_pos [tell $sourcefile]
		set got_purpose 0
		while {-1 != [gets $sourcefile source_line]} {
		    if {0 == [string compare "Purpose" \
			    [string range $source_line 2 8]]} {
			set purpose [string range $source_line 11 end]
			set command_array($command_name) $purpose
			set got_purpose 1
			break
		    }
		    set piece [string range $source_line 0 8]
		    if {0 == [string compare $piece "# solar::"]} {
#			break
		    }
		}
		if {!$got_purpose} {
		    set command_array($command_name) " "
		}
		seek $sourcefile $last_pos
	    }
	}
    }
    close $sourcefile
    }
    set command_list [lsort $command_list]
    return $command_list
}

# solar::updatedoc -- private
#
# Purpose:  Generate auto-generated documentation
#
# Usage:    updatedoc
#
# Notes: index.html, 91.appendix_1_text.html 94.appendix_4.html and
#        README.news are created
#
# Part of the work is actually done by updatechanges.
#
# -

proc updatedoc {args} {

    set indexfile_name "index.html"
    set outfile_name "91.appendix_1_text.html"

    set command_list [help-list-of-commands -describe command_descriptions]

# Write the index file

    puts "Writing index file $indexfile_name"
    set ifile [open $indexfile_name w]

    puts $ifile <html>
    puts $ifile <head>
    puts $ifile "<title>SOLAR Manual Appendix 1</title>"
    puts $ifile </head>
    puts $ifile ""
    puts $ifile "<body bgcolor=ffffff text=#000000>"
    puts $ifile ""
    puts $ifile "<font size=+1><a href=00.contents.html>Go to full SOLAR Manual (Table of Contents)</a></font>"
    puts $ifile "<br><a href=03.chapter.html>Go to SOLAR tutorial</a>"
    puts $ifile "<br><a href=http://www.tcl.tk/man/tcl8.0/TclCmd/contents.htm>Manual for Tcl Commands (External Link)</a>"
    puts $ifile ""
    puts $ifile <h1><center>
    puts $ifile "Appendix 1"
    puts $ifile </center></h1>
    puts $ifile ""
    puts $ifile <h1><center>
    puts $ifile "SOLAR Command Descriptions"
    puts $ifile </center></h1>
    puts $ifile ""
    puts $ifile ""

# Write command names in a quick-access table

    puts $ifile <p><table>
    set number_of_commands [llength $command_list]
    set command_lines_required [expr ($number_of_commands / 6) + \
	    (($number_of_commands%6)?1:0)]
    for {set i 0} {$i < $command_lines_required} {incr i} {
	set command1 [lindex $command_list $i]
	set command2 [lindex $command_list [expr $command_lines_required+$i]]
	set command3 [lindex $command_list [expr 2*$command_lines_required+$i]]
	set command4 [lindex $command_list [expr 3*$command_lines_required+$i]]
	set command5 [lindex $command_list [expr 4*$command_lines_required+$i]]
	set command6 [lindex $command_list [expr 5*$command_lines_required+$i]]

	puts $ifile <tr>
	puts $ifile "<td><a href=91.appendix_1_text.html#$command1>"
	puts $ifile $command1
	puts $ifile </a></td>

	if {"" != $command2} {
	    puts $ifile "<td><a href=91.appendix_1_text.html#$command2>"
	    puts $ifile $command2
	    puts $ifile </a></td>
	}
	if {"" != $command3} {
	    puts $ifile "<td><a href=91.appendix_1_text.html#$command3>"
	    puts $ifile $command3
	    puts $ifile </a></td>
	}
	if {"" != $command4} {
	    puts $ifile "<td><a href=91.appendix_1_text.html#$command4>"
	    puts $ifile $command4
	    puts $ifile </a></td>
	}
	if {"" != $command5} {
	    puts $ifile "<td><a href=91.appendix_1_text.html#$command5>"
	    puts $ifile $command5
	    puts $ifile </a></td>
	}
	if {"" != $command6} {
	    puts $ifile "<td><a href=91.appendix_1_text.html#$command6>"
	    puts $ifile $command6
	    puts $ifile </a></td>
	}
	puts $ifile </tr>
    }
    puts $ifile </table></p>
    puts $ifile ""

# Now write 1-line command descriptions in a second table

    puts $ifile <p><table>
    foreach command $command_list {
	puts $ifile <tr>
	set description $command_descriptions($command)
	puts $ifile "<td><a href=91.appendix_1_text.html#$command>"
	puts $ifile $command
	puts $ifile </a></td>
	puts $ifile <td>
	puts $ifile $description
	puts $ifile </td>
	puts $ifile </tr>
    }
    puts $ifile </table></p>
    puts $ifile ""

    puts $ifile </body>
    puts $ifile </html>
    close $ifile

# Write text file

    puts "Writing text file $outfile_name"

    set outfile [open $outfile_name w]
    puts $outfile "<html>"
    puts $outfile "<head>"
    puts $outfile "<title>SOLAR Manual Appendix 1</title>"
    puts $outfile "</head>"
    puts $outfile ""
    puts $outfile "<body bgcolor=ffffff text=#000000>"
    puts $outfile ""
    puts $outfile "<h1><center>"
    puts $outfile "Appendix 1"
    puts $outfile "</center></h1>"
    puts $outfile ""
    puts $outfile "<h1><center>"
    puts $outfile "SOLAR Command Descriptions"
    puts $outfile "</center></h1>"
    puts $outfile ""
    puts $outfile ""
    puts $outfile "<p>"
    puts $outfile " \
NOTICE: <b>SOLAR</b> is an evolving work of software and is subject to change.\n\
Commands, arguments, features, performance, and availability are\n\
all subject to change.  There is no committment to support scripts\n\
written using the current commands in future releases.\n"
    puts $outfile "</p>"
    puts $outfile "<a href=index.html>Return to command index</a>"
    close $outfile

    set i 0
    foreach command $command_list {
	incr i
	puts "Documenting $command..."

	set outfile [open $outfile_name a]
	puts $outfile "<h2>"
	puts $outfile "<a name=$command>"
	puts $outfile "A1.$i  $command"
	puts $outfile "</a>"
	puts $outfile "</h2>"
	puts $outfile ""
	puts $outfile "<p>"
	puts $outfile "<pre><font size=-1>"

	set command_helpfile [help-on-command $command]
	set hf [open $command_helpfile]
	while {-1 != [gets $hf line]} {
	    puts $outfile [html-fix $line]
	}
	close $hf
	delete_files_forcibly $command_helpfile

	puts $outfile "</font></pre>"
	puts $outfile "</p>"
	puts $outfile "<a href=index.html>Return to command index</a>"
	puts $outfile ""
	puts $outfile ""
	close $outfile
    }
    set outfile [open $outfile_name a]
    puts $outfile "</body>"
    puts $outfile "</html>"
    close $outfile
    updatechanges
    puts "new files: index.html 91.appendix_1_text.html 94.appendix_4.html README.news"
}    

proc html-fix {text} {
    regsub -all < $text {\&lt;} text1
    regsub -all > $text1 {\&gt;} text2
    return $text2
}


# solar::help-all-commands -- private
#
# Purpose:  Display help message listing all commands
#
# Usage:  help-all-commands
#- 

proc help-all-commands {args} {
    set user 0

    if {"-user" == $args} {
	set user 1
	set command_list [help-list-of-commands -describe command_array -user]
	if {{} == $command_list} {
	    error "No user commands are documented"
	}
    } else {
	set command_list [help-list-of-commands -describe command_array]
    }
    set ofilename /tmp/commands.[pid]
    set ofile [open $ofilename w]
    if {$user} {
	puts $ofile \
"The following user-defined commands are available:\n"
    } else {
	puts $ofile \
      "The following solar commands are available (and may be abbreviated):\n"
    }
    set number_of_commands [llength $command_list]
    set command_lines_required [expr ($number_of_commands / 4) + \
	    (($number_of_commands%4)?1:0)]
    for {set i 0} {$i < $command_lines_required} {incr i} {
	puts -nonewline $ofile "[format %-20s [lindex $command_list $i]] "
	puts -nonewline $ofile \
      "[format %-20s [lindex $command_list [expr $command_lines_required+$i]]]"
	puts -nonewline $ofile \
    "[format %-20s [lindex $command_list [expr 2*$command_lines_required+$i]]]"
	puts $ofile \
    "[format %-19s [lindex $command_list [expr 3*$command_lines_required+$i]]]"
    }
    puts $ofile "\n"
    if {!$user} {
    puts $ofile \
{You can also use any TCL or Unix command (but wildcards require [glob]).}
    puts $ofile \
"For more information about a particular command, use 'help <command>'"
    }
    puts $ofile \
"Brief description of each command follows:"
    puts $ofile ""
    foreach command $command_list {
	puts $ofile \
  "[format %-19s $command] [string range $command_array($command) 0 57]"
    }
    close $ofile
    return $ofilename
}


proc helpscript {command} {
    set outfile [help-on-command $command]
    exec more $outfile >&@stdout
    delete_files_forcibly $outfile
}


# Help for a particular command

proc help-on-command {args} {

    if {"-user" == $args} {
	return [help-all-commands -user]
    }	

    set use_more 1
    set usage_only 0
    set name [read_arglist $args -nomore {set use_more 0} \
	    -usage {set usage_only 1; set use_more 0}]
    set nlen [string length $name]

# First, find if one or more script name matches given name
#   Look first at solar.tcl currently in use.  Then look at all other
#   *.tcl files in tcl search paths: . ~/lib $SOLAR_LIB
#
#   Header line for help message is "# <packagename>::<commandname>" where
#   packagename is the first dotted segment of filename.  For example,
#   if filename is solar.john.tcl, packagename is "solar" but if filename
#   is john.solar.tcl, packagename is "john".
#
# Remove duplicate paths, auto_path sometimes has duplicates
# Current solar.tcl is put at front of list...it has "priority"
#
    set script_path [find_script_path solar.tcl]
    ifdebug puts "script path is $script_path"
    global auto_path
    set sourcefilenames ""
    set checked_path ""
    foreach d $auto_path {
	if {-1==[lsearch $checked_path $d]} {
	    lappend checked_path $d
	    catch {set sourcefilenames "$sourcefilenames [glob $d/*.tcl]"}
	}
    }
    set sourcefilenames \
	[remove_from_list $sourcefilenames $script_path/solar.tcl]
    set sourcefilenames "$script_path/solar.tcl $sourcefilenames"
    ifdebug puts "searching files $sourcefilenames"
    set matched 0
    set exact_matched 0
    set mfilename ""
    set mtargetname ""
    set mcoffset 0
    set matchlist ""
    while {$matched == 0 && 0 != [llength $sourcefilenames]} {
	set sourcefilename [lindex $sourcefilenames 0]
	set sourcefilenames [lrange $sourcefilenames 1 end]
	set sourcefile [open $sourcefilename r]
	set packagename [file tail $sourcefilename]
	set endchar [expr [string first . $packagename] - 1]
	set packagename [string range $packagename 0 $endchar]
	set targetname "\# [catenate $packagename ::]"
	set coffset [expr [string length $targetname] - 0]
	ifdebug puts "searching $sourcefilename for $targetname"
	while {-1 != [gets $sourcefile source_line]} {
	    set subs [string range $source_line 0 [expr $coffset - 1]]
	    if {0 == [string compare $subs $targetname]} {
		set sbase [string range $source_line $coffset end]
		if {1 == [scan $sbase "%s" sall]} {
		    set sshort [string range $sall 0 [expr $nlen - 1]]
#		    puts "sall is $sall, name is $name"
		    if {0 == [string compare $sshort $name]} {
			set mfilename $sourcefilename
			set mtargetname $targetname
			set mcoffset $coffset
			set matched [expr $matched + 1]
			lappend matchlist $sall
			set target $sall
			if {0 == [string compare $sall $name]} {
			    set exact_matched 1
			    break
			}
		    }
		}
	    }
	}
	close $sourcefile
    }	
#
# Check for ambiguity or non-presence
#
    if {$matched > 1 && $exact_matched == 0} {
	error "Ambiguous command name $name:\n  $matchlist"
    }
    if {$matched == 0} {
	error "Help not found for $name"
    }
#
# Output help file to /tmp
#
    set ofilename /tmp/$name.[pid]
    if {$use_more} {
	set ofile [open $ofilename w]
    }
    set sourcefile [open $mfilename r]
    while {-1 != [gets $sourcefile source_line]} {
	set subs [string range $source_line 0 [expr $mcoffset - 1]]
	if {0 == [string compare $subs $mtargetname]} {
	    set sbase [string range $source_line $mcoffset end]
	    if {1 == [scan $sbase "%s" sall] && \
		    0 == [string compare $sall $target]} {
		set just_starting 1
		set line_count 0
		set found_usage 0
		while {-1 != [gets $sourcefile source_line]} {
		    incr line_count
		    if {$just_starting && 4>[string length $source_line]} {
			continue
		    }
		    set just_starting 0
		    if {0 == [string compare $source_line "# -"]} break
		    if {[string range $source_line 0 0] != "#"} break
		    if {$use_more} {
			puts $ofile [string range $source_line 2 end]
		    } else {
			puts [string range $source_line 2 end]
		    }
		    if {$usage_only} {
			if {-1 != [string first "Usage:" $source_line]} {
			    set found_usage 1
			}
			if {$found_usage && 5 > [string length $source_line]} {
			    break
			}
			if {$line_count >= 15} break
		    }
		}
		break
	    }
	}
    }
    if {![catch {shortcut $sall}]} {
	if {$use_more} {
	    puts $ofile "\n[shortcut $sall]\n"
	    flush $ofile
	} else {
	    puts "\n[shortcut $sall]\n"
	}
    }
    close $sourcefile
    if {$use_more} {
	close $ofile
	return $ofilename
    }
    return ""
}

proc moreout {name} {
    eval exec more $name
}

proc catout {name} {
    eval exec cat $name
}

proc find_script_path {scriptname} {
    global auto_path
    foreach d $auto_path {
	if {[file exists "$d/$scriptname"]} {
	    return $d
	}
    }
    puts "Directory not found"
    error "Directory not found"
}

# Column ruler:
#12345678|234567890123456789012345678901234567890123456789012345678901234567890

# solar::ifverbplus -- private
#
# Purpose:  Execute a tcl command if verbosity level is "plus" or higher
#
# Usage:    ifverbplus <command>
#-

proc ifverbplus {args} {
    set vlevel [verbosity -number]
    if {$vlevel >= 0x3ff} {
	return [uplevel $args]
    }
    return ""
}

proc ifverbmax {args} {
    set vlevel [verbosity]
    if {"verbosity max" == $vlevel} {
	return [uplevel $args]
    }
    return ""
}

# solar::evdout --
# solar::evdmat --
# solar::evdinx --
# solar::evdiny --
# solar::evdinz --
# solar::evdinev --
#
# Purpose: I/O for EVD data
#
# Usage:   trait ...
#          covar ...
#
#          evdout [<-evectors>[<-all>]] ;# write evddata.out,evectors if asked
#
#          evdinx  [<evdfile>] ;# return X* matrix from evddata.out file
#          evdiny  [<evdfile>] ;# return Y* matrix from evddata.out file
#          evdinz  [<evdfile>] ;# return Z* matrix from evddata.out file
#
#          evdinev [<-all>]  ;# load eigenvector matrix(es) from file(s) saved
#                            ;# by evdout -evectors
#
# Note: You must select trait, covariates, and anything else that would
#       restrict the sample size before invoking evdout.  You
#       do not need a maximized model, just trait and covariates.
#
# evdout writes out evd transformed variables and the eigenvalues
#       (which are called lambda) to a file named evddata.out in the
#       maximization output directory but without actually doing a
#       model maximization.  The trait value, which is not demeaned,
#       is written to variable <traitname>_evd.  The covariate values,
#       which are demeaned or scaled appropriately, are written to
#       variables named evd2_<varname>_evd.  Ignore other fields.
#
#       Additionally, you can write out the eigenvectors, either in
#       normal per-pedigree mode (one matrix for each family) or in
#       entire-sample mode if you select the -all option.  Filenames are
#       evectors.family<n>.mat.csv for per-family matrices and
#       evectors.mat.csv for entire sample (-all) matrix.  Matrix files
#       are written to the outdir.
#
#       Current methods rely on the EVD transformation of variables
#       and do not require the Eigenvectors to be output, so generally
#       speaking you should not use the -evectors option unless you
#       know you need it.
#
# evdinx returns the X* matrix corresponding to current evddata.out.
#       The X* matrix has from left to right: (1) a column of 1's, (2+) one or
#       more columns of EVD transformed covariates which have been scaled to
#       mean of zero, in the order in which covariates occur in the model.
#       evdinx has a -method2 option needed for the -method2 option of fphi.
#
# evdiny returns the Y* matrix, which is a one vertical column matrix
#       (aka vector) of EVD transformed trait values.
#
# evdinz returns the Z matrix, which has has a column of 1's and a column
#        of lambda's.
#
# evdinev returns a list of eigenvector matrices, one for each family,
#       or if the -all option is specified, just one matrix for the entire
#       sample.  In order to use the -all option, you must have previously
#       done evdout with the -evectors and -all options.
#
# Example:
#
#       foreach covar $testvars {
#           model new
#           trait q4
#           covar $covar
#           evdout
#           set X [evdinx]
#           set Y [evdiny]
#           solve $X $Y
#       }
#-

# Junk documentation for features not supported
#          evdmatx [<-last>] ;# return X* matrix from current model or last EVD
#          evdmaty [<-last>] ;# return Y* matrix from current model or last EVD
#          evdmatxy    ;# return X* and Y* matrix from current model as list
#          evdmatev [<-all>] ;# return Eigenvector matrix or matrices
# -



proc evdinev {args} {
    if {$args == "-all"} {
	return [load matrix [full_filename evectors.mat.csv]]
    } elseif {$args != ""} {
	error "Invalid argument $args to evdin"
    }
    set ms [glob [full_filename evectors.family*.mat.csv]]
    set allms ""
    foreach mms $ms {
	lappend allms [load matrix $mms]
    }
    return $allms
}

proc evdoldjunk {} {
# figure out cols from current covariates
    set covs [covariates]
    foreach cov $covs {
	set newcov ""
	for {set i 0} {$i < [string length $cov]} {incr i} {
	    set ch [string index $cov $i]
	    if {$ch == "*"} {
		set ch X
	    } elseif {$ch == "^"} {
		set ch "up"
	    }
	    set newcov "$newcov$ch"
	}
	lappend covcols evd2_$newcov\_evd
    }
}

proc evdinx {{evdfile ""} args} {
    set method 1
    if {$args == "-method2"} {
	set method 2
    } elseif {$args != ""} {
	error "Invalid argument $args to evdinx"
    }

    if {$method==2} {
	set covcols {{1}}
    } else {
	set covcols ""
    }

    if {$evdfile == ""} {
	set evdfile [full_filename evddata.out]
    }
    set inevd [open $evdfile]
    gets $inevd line
    close $inevd

    set linelist [split $line ,]
    set length [llength $linelist]
    for {set i 8} {$i < $length} {incr i} {
	set test [lindex $linelist $i]
	if {[string range $test 0 4] == "evd2_"} {
	    if {$method!=2} {
		set covcols [concat $covcols $test]
	    }
	} else {
	    break
	}
    }
    if {$method==2} {
	for {} {$i < $length} {incr i} {
	    set test [lindex $linelist $i]
	    set covcols [concat $covcols $test]
	}
    }

    if {[info level]==0} {
	puts "evdinx columns: $covcols"
    }
    set X [load matrix -cols $covcols [full_filename evddata.out]]
    return $X
}

proc evdiny {{evdfile ""}} {
    if {$evdfile == ""} {
	set evdfile [full_filename evddata.out]
    }
    set Y [load matrix -cols [trait]_evd $evdfile]
    return $Y
}

proc evdinz {{evdfile ""}} {
    if {$evdfile == ""} {
	set evdfile [full_filename evddata.out]
    }
    set zcols {{1} lambda}
    set Z [load matrix -cols $zcols [full_filename evddata.out]]
    return $Z
}


proc evdoutev {args} {
    set evdmode 1
    if {$args == "-all"} {
	set evdmode 2
    } elseif {$args != ""} {
	error "invalid argument: $args"
    }
    return [evdmat $evdmode]
}

# Internally used options
# option eigenvectors: 0=do not output eigenvectors
#                      1=write eigenvector(s) for each ped (or all)
#                      2=make eigenvector matrix for each ped (or all)
#                      (each or all determined by evdmat)
#
# option evdmat: (for X Y matrixes)
# 4 means make matrix for all peds
# 3 means make matrix for each ped (default for evdmat)
# 2 means write file for all peds
# 1 means write files for each ped (default for evdout)
#
# option evdcovars: count of covariates in evd model
#
# evdout is the main procedure which does evdout or evdmat operations
# evdmatx,y are front ends
#

proc evdmat {args} {
    error "evdmat obsolete.  Use evdout then evdinx, evdiny, and/or evdinev"
}

proc evdmatxNotWorking {args} {
    global SOLAR_evdmatx
    global SOLAR_evdmaty

    if {$args == "-last"} {
	if {[if_global_exists SOLAR_evdmatx]} {
	    return $SOLAR_evdmatx
	} else {
	    error "No previous evdmatx or evdmaty"
	}
    } elseif {$args != ""} {
	error "invalid argument to evdmatx"
    }
    evdout -mat
    set SOLAR_evdmaty [mathmatrix lastid]
    set pentid [string range $SOLAR_evdmaty 4 end]
    set SOLAR_evdmatx .mm.[expr $pentid - 1]
    return $SOLAR_evdmatx
}

proc evdmatyNotWorking {args} {
    global SOLAR_evdmatx
    global SOLAR_evdmaty

    if {$args == "-last"} {
	if {[if_global_exists SOLAR_evdmatx]} {
	    return $SOLAR_evdmaty
	} else {
	    error "No previous evdmatx or evdmaty"
	}
    } elseif {$args != ""} {
	error "invalid argument to evdmaty"
    }
    evdout -mat
    set SOLAR_evdmaty [mathmatrix lastid]
    set pentid [string range $SOLAR_evdmaty 4 end]
    set SOLAR_evdmatx .mm.$pentid

    return $SOLAR_evdmaty
}

proc evdmatxyNotWorking {} {
    global SOLAR_evdmatx
    global SOLAR_evdmaty

    evdout -mat 
    set SOLAR_evdmaty [mathmatrix lastid]
    set pentid [string range $SOLAR_evdmaty 4 end]
    set SOLAR_evdmatx .mm.[expr $pentid - 1]
    return {$SOLAR_evdmatx $SOLAR_evdmaty}
}

proc evdmatevNotWorking {args} {
    set all ""
    set lastid [mathmatrix lastid]
    set outlist {}
    if {$args != ""} {
	if {$args == "-all"} {
	    set all "-all"
	} else {
	    error "Invalid argument $args to evdmatev"
	}
    }
    eval evdout $all -evectors -mat
    set startid [string range $lastid 4 end]
    set endid [string range [mathmatrix lastid] 4 end]
    for {set i $startid+1} {$i <= $endid} {incr i} {
	lappend outlist .mm.$i
    }
    return $outlist
}

proc evdout {args} {
    set eigenvectors 0
    set all 0
    set matonly 0
    set bad [read_arglist $args \
		 -evectors {set eigenvectors 1} \
		 -all {set all 1} \
	         -mat {set matonly 1}]

    if {$bad != ""} {
	error "evdout: invalid argument(s) $bad"
    }

    set savedname [full_filename evdout.startmodel]
    save model $savedname
    if {-1 != [string first "to_set_standard_model" [omega]]} {
	polymod
    }

    option modeltype evd2
    option evdphase 1
    option eigenvectors $eigenvectors
    option evdcovs [llength [covariates]]
    if {$matonly} {
	if {$all} {
	    option evdmat 4
	} else {
	    option evdmat 3
	}
    } else {
	if {$all} {
	    option evdmat 2
	} else {
	    option evdmat 1
	}
    }
    if {$all} {
	option mergeallpeds 1
    }
    set code [catch {maximize -q} rets]
    load model $savedname
    if {$code} {
	return -code $code $rets
    }
    set retval ""
}


proc evdmatOLD {args} {
    set evdmode 3 ;# default to one matrix
    if {$args == "-all"} {
	set evdmode 4
    } elseif {$args == 1 || $args == 2} {
	set evdmode $args
    } elseif {$args != ""} {
	error "Invalid argument $args to evdmat or evdout"
    }

    set startid 0
    if {$evdmode == 3} {
	if {![catch {mathmatrix lastid}]} {
	    set startid [string range [mathmatrix lastid] 4 end]
	}
    }

    set savedname [full_filename evdout.startmodel]
    save model $savedname
    option modeltype evd2
    option eigenvectors $evdmode
    if {1 - ($evdmode % 2)} {
	option mergeallpeds 1        ;# one big matrix produced
    }
    option evdphase 1
    if {-1 != [string first "to_set_standard_model" [omega]]} {
	polymod
    }
    if {$evdmode & 0x8} {
	error "maximize would be next"
    }
    set code [catch {maximize -q} rets]
    load model $savedname
    if {$code} {
	return -code $code $rets
    }
    set retval ""
    if {$evdmode == 3} {
	set endid [string range [mathmatrix lastid] 4 end]
	for {set id [expr $startid + 1]} {$id < $endid} {incr id} {
	    lappend retval .mm.$id
	}
    } elseif {$evdmode == 4} {
	set retval [mathmatrix lastid]
    }
    return $retval
}




# solar::maximize --
#
# Purpose:  Find the maximum loglikelihood of a model by adjusting
#           parameter values within specified constraints.
# 
# Usage:    maximize [-quiet] [-out <filename>]
#
#               -quiet  (or -q) Use minimum verbosity while maximizing
#               -out (or -o)    Write results to this filename.  The default
#                               is 'solar.out' in a subdirectory named after
#                               the current trait, or the current 'outdir.'
#                               If a filename is specified without any /
#                               characters, it will also be written in the
#                               default location.  If the filename contains
#                               / characters, it is used as a full pathname.
#
#               -noquad         Do not test quadratic
#
#               -who            Do not maximize, but list who would be
#                               included in analysis.  File "who.out" is
#                               written to trait/outdir containing list
#                               of pedindex.out sequential ID's.  This
#                               option is used by the "relatives" command
#
#                -runwho        Maximize, AND produce who.out file as above.
#
#                -sampledata    Do not maximize, but write out the data that
#                               would be included in the analysis to a file
#                               named "sampledata.out" in the maximization
#                               output directory.  [WARNING!  The fields in
#                               this file are preliminary and subject to
#                               change!]  The trait data (fields trait1,
#                               trait2, etc.) might be from a phenotype or
#                               an expression created by the "define" command.
#
# Notes:    This is the key command of solar.  It is used by polygenic,
#           twopoint, multipoint, bayesavg, and other scripts to find
#           the model parameter values which produce the maximum
#           loglikelihood.
#
#           The final values are not saved to a model file.  To do that,
#           issue a 'save model' command after maximization.
#
#           Multiple solar processes should not be writing to the same
#           directory.  Use the outdir command to specify different output
#           directories.
#
#           Advanced SOLAR users sometimes use the raw "cmaximize" command
#           which bypasses many of the retry mechanisms (and their implicit
#           assumptions) now built-in to SOLAR.  This is not recommended for
#           most users.
#-


# This was a layer around the old Tcl maximize, now called "tmaximize"
# However, currently it does nothing extra since the reduction of conv is
# not currently deemed necessary, so maximize and tmaximize are identical.
#
# Some scripts (which could do their own conv reduction) call tmaximize
# directly.

proc maximize {args} {
    return [eval tmaximize $args]
}

#
# This is the additional code in the pre-6.4.0 version of maximize
# long since made obsolete, which did conv reduction, now completely
# removed for better efficiency.
proc pre640_maximize {args} {
    if {0} {
    set rets ""
    set mfile [full_filename last_first]
    save model $mfile
    set tries 0
    set max_tries 2

    set quiet 0
    catch {read_arglist $args -q {set quiet 1} -quiet {set quiet 1}}
    ifdebug set quiet 0

    global errorInfo errorCode
    set code [catch {eval tmaximize $args} rets]
    if {!$code} {
	return $rets
    } else {
	if {-1<[string first "CONVERGENCE FAILURE" $rets]} {
	    if {!$quiet} {puts "\n    *** Retrying with reduced conv"}
	    load model $mfile
	    option conv 1e-4
# default for discrete anyway
#	    option conv(discrete) 1e-4   
	    set code [catch {eval tmaximize $args} rets]
	    if {!$code} {
		return $rets
	    }
	}
    }
    return -code $code -errorinfo $errorInfo -errorcode $errorCode $rets
    }
}
	    
# tmaximize:
# This is the main maximize procedure which checks for convergence and
# normalized quadratic unity and does retries after boundary modification
# which is intended to be applicable to all models with standard
# parameterization.  It calls cmaximize, which handles the trap system
# introduced in version 6.

proc tmaximize {args} {

    ifdebug puts "Entering tmaximize"

# Process arguments and options

    set quiet 0
    set crunch_count 0
    set fix_count 0
    set outfile solar.out
    set noquad 0
    set opts {}
    set badargs [read_arglist $args \
	    -output outfile -out outfile -o outfile \
	    -noquad {set noquad 1} \
	    -who {lappend opts -who} \
	    -runwho {lappend opts -runwho} \
	    -sampledata {lappend opts -sampledata} \
	    -initpar {lappend opts -initpar} \
	    -quiet {set quiet 1} -q {set quiet 1}]

    if {"" != $badargs} {
	error "Unexpected maximize argument(s): $badargs"
    }

    if {-1 == [string first "/" $outfile]} {
	set outfile [full_filename $outfile]
    }

# Check for multivariate

    set ts [trait]
    if {[llength $ts] == 1} {
	set multi 0
    } else {
	set multi 1
    }

# Check for arbitrary parameterization

    set aparama 0
    if {!$multi} {
	if {![if_parameter_exists e2] || ![if_parameter_exists h2r]} {
	    ifdebug puts "This model has arbitary parameterization...cannot check boundaries"
	    set aparama 1
	}
    } else {
	foreach t $ts {
	    if {![if_parameter_exists e2\($t\)] || \
		    ![if_parameter_exists h2r\($t\)]} {
		ifdebug puts "This model has arbitary parameterization...cannot check boundaries"
		set aparama 1
	    }
	}
    }
    if {$aparama == 0} {
	ifdebug puts "This model has checkable boundaries"
    }


# Initialize retry variables

    set tried_perturb_last 0
    set no_previous_messages 1
    set quadratic_retries 0
    set smaller_ll_retries 0
    set code 0
    set first_time 1

# We loop over retries required to get it right
#   exiting on success, failure, or exhaustion

    while {1} {

# Perturb before all retries

	if {!$aparama && !$first_time} {
	    perturb
	}
	set first_time 0

# Boundary tracing

	global errorInfo errorCode
	set savelevel [verbosity -number]
	if {$quiet!=0} {verbosity min}
	if {!$aparama && 1==[llength [trait]] && \
	  ([trace_boundaries] ||[string_imatch "verbosity max" [verbosity]])} {
	    set h2qc [h2qcount]
	    puts -nonewline "    *** Parameters    e2   h2r"
	    for {set i 1} {$i <= $h2qc} {incr i} {
		puts -nonewline "   h2q$i"
	    }
	    puts ""

	    puts -nonewline "    ***       Values  [parameter e2 start]"
	    catch {puts -nonewline " [parameter h2r start]"}
	    for {set i 1} {$i <= $h2qc} {incr i} {
		puts -nonewline " [parameter h2q$i start]"
	    }
	    puts ""

	    puts -nonewline "    *** Upper Bounds  [parameter e2 upper]"
	    catch {puts -nonewline " [parameter h2r upper]"}
	    for {set i 1} {$i <= $h2qc} {incr i} {
		puts -nonewline " [parameter h2q$i upper]"
	    }
	    puts ""

	    puts -nonewline "    *** Lower Bounds  [parameter e2 lower]"
	    catch {puts -nonewline " [parameter h2r lower]"}
	    for {set i 1} {$i <= $h2qc} {incr i} {
		puts -nonewline " [parameter h2q$i lower]"
	    }
	    puts ""
	}

# Count number of maximizations...see also "countmax"

	set showmax 0
	if {[if_global_exists SOLAR_Maximizes]} {
	    global SOLAR_Maximizes
	    incr SOLAR_Maximizes
	    puts "    *** Beginning maximization number $SOLAR_Maximizes"
	    set showmax 1
	}
	ifdebug set showmax 1

	set rets ""
	set code [catch {eval cmaximize $outfile $opts} rets]
# Note: EVD Phase 2 is trapped and handled by cmaximize.  We simply exit here.
	if {-1 != [string first "Trap EVD Phase 2" $rets]} {
	    return $rets
	}
	verbosity $savelevel

# break out for evdout
	if {[option evdmat] > 0} {
	    return ""
	}


# -who and -sampledata options don't actually maximize, so no error check

	if {-1<[string first "-who" $opts]} {
	    return -code $code $rets
	}
	if {-1<[string first "-sampledata" $opts]} {
	    return -code $code $rets
	}
	if {-1<[string first "-initpar" $opts]} {
	    return -code $code $rets
	}

# If option maxiter 1, convergence and quadratic are irrelevant
# So exit now

	if {[option maxiter] == 1} {
	    return -code $code $rets
	}

# Report error message if applicable

# If hard convergence failure, try crunching

	if {!$aparama && -1<[string first "Convergence failure" $rets]} {
	    if {"" != $rets && (!$quiet || [trace_boundaries] || $showmax)} {
		puts "$rets\n"
	    }
	    set ready_to_retry 0
	    set said_useable 0
	    while {$crunch_count < [maxcrunch] && $fix_count < 2} {
		set quadratic_retries 0
		if {![restart-if-can] || \
			-1==[string first "estartable" $rets]} {
		    if {!$said_useable && \
			    !$quiet || [trace_boundaries] || $showmax} {
			puts "    *** Unable to use current model: Reloading last"
			set said_useable 1
		    }
		    load model [full_filename last.mod]
		} else {
		    if {!$said_useable && \
			    !$quiet || [trace_boundaries] || $showmax} {
			puts "    *** Current model useable"
			set said_useable 1
		    }
		}
		set margin [expr (1.0/pow(5.0,$crunch_count))*[bcrunch]]
		if {[catch {set progress [do_boundaries $margin]} errorstring]} {
# Fixed boundaries, didn't crunch them
		    if {!$quiet || [trace_boundaries] || $showmax} {
			set rmargin [format %.5g $margin]
			puts $errorstring
		    }
		    set ready_to_retry 1
		    incr fix_count
		    break
		}
		incr crunch_count
		if {{} != $progress} {
		    if {!$quiet || [trace_boundaries] || $showmax} {
			set rmargin [format %.5g $margin]
			puts "    *** Crunched boundaries to $rmargin: $progress\n"
		    }
		    set ready_to_retry 1
		    break
		}
	    }
	    if {!$ready_to_retry} {
		error "CONVERGENCE FAILURE"
	    }
	    continue
	}


	if {$crunch_count} {
	    save model [full_filename converged.mod]
	}
	set crunch_count 0
	set fix_count 0

# If last likelihood inferior, try retries

	if {-1<[string first "Last likelihood significantly smaller" $rets]} {
	    if {"" != $rets && (!$quiet || [trace_boundaries] || $showmax)} {
		puts "$rets\n"
	    }
	    if {$smaller_ll_retries <= 5} {
		incr smaller_ll_retries
		continue
	    } else {
		error "\nConvergence failure (restartable): $rets"
	    }
		
	    
	}
	set smaller_ll_retries 0

# Return with error here if unfixable error

	if {$code == 1} {
	    return -code $code -errorinfo $errorInfo -errorcode $errorCode \
		    $rets
	}

# If arbitrary parameterization, exit here (don't check boundaries or quad)

	if {$aparama} {
	    ifdebug puts "Unusual parameterization; unable to check boundaries"
	    break
	}

# Check for artificial boundary problems (if found, expand and retry)

	ifdebug puts "Checking for boundary conditions"
	set real_boundary_conditions [check_real_upper_boundaries]
	set artificial_boundary_conditions [check_artificial_boundaries]
	if {{} != $artificial_boundary_conditions} {
	    set tried_perturb_last 0
	    set quadratic_retries 0
	    if {!$quiet || [trace_boundaries] || $showmax} {
		if {$no_previous_messages} {
		    set no_previous_messages 0
		    puts " "
		}
		puts -nonewline "    *** Retry with moved boundary for:"
		foreach boundary $artificial_boundary_conditions {
		    puts -nonewline " [lindex $boundary 0]"
		}
		puts "\n"
	    }
	    adjust_boundaries

# If we hit real boundaries (and no artificial ones) perturb and try again

	} elseif {{} != $real_boundary_conditions && !$tried_perturb_last} {
	    set tried_perturb_last 1
	    set quadratic_retries 0
	    if {!$quiet || [trace_boundaries] || $showmax} {
		if {$no_previous_messages} {
		    set no_previous_messages 0
		    puts " "
		}
		puts -nonewline \
		       "    *** Will perturb because hit real upper bound for:"
		foreach boundary $real_boundary_conditions {
		    puts -nonewline " [lindex $boundary 0]"
		}
		puts ""
	    }	    
	    perturb

# Check quadratic

	} elseif {[option evdphase] == 3} {
	    break
	} elseif {!$noquad && 0==[option tdist] && \
		[catch {find_simple_constraint SD}] && \
		[option maxiter] != 1 && \
		([quadratic] > [expr 1.0 + [boundary_quadratic]] || \
		 [quadratic] < [expr 1.0 - [boundary_quadratic]])} {
	    if {$multi} {
#		putsout $outfile "Warning: Quadratic is [quadratic]"
		break
	    } elseif {$quadratic_retries == 0} {
		set last_ll [loglike]
	    } else {
		if {$last_ll >= [loglike]} {
		    load model [full_filename last.mod]
		    error "\nConvergence failure (restartable)"
		}
	    }
	    incr quadratic_retries
	    if {$quadratic_retries > 10} {
		error \
			"\nConvergence failure (restartable) after 10 retries"
	    }
	    continue

# No problems found, so break out of loop

	} else {
	    break
	}
    }
    return -code $code $rets
}

# New in version 4.4.0, cmaximize is Tcl, ccmaximize is C++
# cmaximize handles trap for zscore

proc cmaximize {args} {
    set debug0 0

    if {[option modeltype] == "evd2"} {
	set EVD2_dirname [full_filename ""]
	set outfilename [full_filename evd2]
	set userout [lindex $args 0]
	set descout [lindex $args 0].desc
	set args [concat $descout [lrange $args 1 end]]
    }
    set badargs [read_arglist $args -output outfile -*]
#
# If evd2 model, save original model here and
# change covariates to definitions which allows the required scaling
# for each internal term.  Variables must be pre-scaled because EVD is
# computed immediately after sample delineation, and normally scaling is
# done only later during actual maximization interations.
#
    if {[option modeltype] == "evd2"} {
	save model $outfilename.evdphase1
	option modeltype Default
	option evdphase 0
	maximize -who -q -o $outfilename.getmeans.out
	load model $outfilename.evdphase1
	option evdphase 1

	set covars [covariates]
	set traits [trait]
	set oldomega [omega]
	set matrices [matrix -return]
	set parameters [parameter -return]
	set ntraits [llength $traits]

	foreach cov $covars {
	    ifdebug0 puts "processing $cov"

# suspended covariates used w/o change

	    if {"Suspended\[" == [string range $cov 0 9] ||
		"\(\)" == [string range $cov end-1 end]} {


	    } else {
#
# This is an active covariate. Make a definition name, a definition,
#   and a covariate which simply uses that definition
#     first, definition (and covariate) name, free of special cov characters:
#
		set dname "evd2_"
		for {set i 0} {$i < [string length $cov]} {incr i} {
		    set ch [string index $cov $i]
		    if {$ch == "*"} {
			set ch X
		    } elseif {$ch == "^"} {
			set ch "up"
		    }
		    set dname "$dname$ch"
		}
#
# now make definition
#   remove parenthesized trait specifier if present from name and expr

		set tsub ""
		set dname_nos $dname
		if {-1 != [set pos [string first \( $dname]]} {
		    set dname_nos [string range $dname 0 [expr $pos - 1]]
		    set tsub [string range $dname $pos end]
		    set tssub [string range $dname $pos+1 end-1]
		}
		set expr_nos [string range $cov 0 end]
		if {-1 != [set pos [string first \( $expr_nos]]} {
		    set expr_nos [string range $expr_nos 0 $pos-1]
		}

# Now we need to scale every term properly
#
		set slist [split $expr_nos *]
		ifdebug0 puts "slist is $slist"
		set newexp ""
		set define_names [define names]
		set noscale 0
		set vsums 0
		foreach sterm $slist {
		    set adjval 0
		    set newterm $sterm
		    set sulist [split $sterm ^]
		    set sel [lindex $sulist 0]
		    set s2 [lindex $sulist 1]
		    if {[string tolower $sel] == "sex"} {
			set newterm \(sex-1\)
		    } else {
#
# accumulate sum of squares for each term range
#
			set termexp 1
			if {[llength $sulist] > 1} {
			    set termexp $s2
			}
			set tmin [getvar -min $EVD2_dirname/evd2.getmeans.out \
				      $sel]
			set tmax [getvar -max $EVD2_dirname/evd2.getmeans.out \
				      $sel]
			set trange [expr $tmax - $tmin]
			ifdebug0 puts "range for $sel is $trange"
			set vterm $trange
			if {$trange != 0} {
			    for {set iexp 2} {$iexp <= $termexp} {incr iexp} {
				set vterm [expr $vterm * $trange]
			    }
			    set vsums [expr $vsums + $vterm * $vterm]
			}
# check for definition
			if {-1 != [lsearch $define_names $sel]} {

# this term is a definition...
# no current capability to have definitions within definitions
# if this is an interaction covariate, give an error
# if this is a scalar covariate, give a warning and abort scaling

			    if {-1 != [string first * $expr_nos] || -1 != \
				    [string first ^ $expr_nos]} {
				error \
"EVD2 does not yet support defined terms in interaction covariates\n\
Make a definition for $cov including required interactions and scaling"
			    } else {
				puts \
"Warning!  EVD2 cannot yet scale defined covariates.\n\
  You must scale them within their own definitions."
			    }
			    set noscale 1
			    break
			}
			set isd [getvar -d evd2.getmeans.out $sel]
			if {$isd} {
			    set adjval [getvar -min evd2.getmeans.out $sel]
			} else {
			    set adjval [getvar -mean evd2.getmeans.out $sel]
			}
			set newterm \($sel-$adjval\)
		    }
		    if {$s2 != ""} {
			set newterm "$newterm^$s2"
		    }
		    if {$newexp == ""} {
			set newexp $newterm
		    } else {
			set newexp "$newexp*$newterm"
		    }
		}

		if {!$noscale} {
		    ifdebug0 puts "define $dname_nos = $newexp"
		    eval define $dname_nos = $newexp
#
# Get original start, lower, and upper values for beta parameter(s)
#   there is one beta parameter for each trait if not qualified
#   then delete covariate, add new covariate, and transfer values
#
# Much easier for univariate
#
		    if {$ntraits==1 || $tsub != ""} {
			set start [parameter b$cov =]
			set upper [parameter b$cov upper]
			set lower [parameter b$cov lower]
			set fixupper [parameter b$cov fixupper]
			set fixlower [parameter b$cov fixlower]
			covariate delete $cov
			covariate $dname
		      parameter b$dname$tsub = $start lower $lower upper $upper
			set p1names(b$dname$tsub) b$cov
			if {"" != $fixupper} {
			    parameter b$dname$tsub fixupper $fixupper
			}
			if {"" != $fixlower} {
			    parameter b$dname$tsub fixlower $fixlower
			}
#
# If not already set,
# compute beta bounds following same algorithm as in covariate.cc
#
			ifdebug0 puts "sum of squares for $cov is $vsums"
			set divisor [expr sqrt ($vsums)]
			if {$divisor == 0} {set divisor 1}
			ifdebug0 puts "divisor is $divisor"
			set thistraitmax [getvar -max $EVD2_dirname/evd2.getmeans.out $traits]
			set thistraitmin [getvar -min $EVD2_dirname/evd2.getmeans.out $traits]
			set deltaT [expr $thistraitmax - $thistraitmin]
			set spreadfactor [option autocovarbound]
			set mulT [expr $deltaT * $spreadfactor]
			ifdebug0 puts "mulT is $mulT"
			set maxbound [expr $deltaT * $spreadfactor / $divisor]
			if {$maxbound < 2.0e-4} {set maxbound 2.0e-4}
			if {$lower == 0 && \
			    "" == [parameter b$dname$tsub fixlower] } {
			    parameter b$dname$tsub lower -$maxbound
			    ifdebug0 puts "setting lower bound to -$maxbound"
			}
			if {$upper == 0 && \
				"" == [parameter b$dname$tsub fixupper]} {
			    parameter b$dname$tsub upper $maxbound
			    ifdebug0 puts "setting upper bound to $maxbound"
			}
		    } else {
			foreach tr $traits {
			    set start($tr) [parameter b$cov\($tr\) =]
			    set upper($tr) [parameter b$cov\($tr\) upper]
			    set lower($tr) [parameter b$cov\($tr\) lower]
			    set fixupper($tr) [parameter b$cov\($tr\) fixupper]
			    set fixlower($tr) [parameter b$cov\($tr\) fixlower]
			}
			covariate delete $cov
			covariate $dname
			foreach tr $traits {
			    set pbname b$dname\($tr\)
			    parameter $pbname = $start($tr) upper $upper($tr)\
				lower $lower($tr)
			    set p1names($pbname) b$cov\($tr\)
			    if {"" != $fixupper($tr)} {
				parameter $pbname fixupper $fixupper($tr)
			    }
			    if {"" != $fixlower($tr)} {
				parameter $pbname fixlower $fixlower($tr)
			    }
			}
#
# If not already set,
# compute beta bounds following same algorithm as in covariate.cc
#
			ifdebug0 puts "sum of squares for $cov is $vsums"
			set divisor [expr sqrt ($vsums)]
			if {$divisor == 0} {set divisor 1}
			ifdebug0 puts "divisor is $divisor"
			foreach tr $traits {
			    set tsub \($tr\)
			    set thistraitmax [getvar -max $EVD2_dirname/evd2.getmeans.out $tr]
			    set thistraitmin [getvar -min $EVD2_dirname/evd2.getmeans.out $tr]
			    set deltaT [expr $thistraitmax - $thistraitmin]
			    set spreadfactor [option autocovarbound]
			    set mulT [expr $deltaT * $spreadfactor]
			    ifdebug0 puts "mulT is $mulT"
			    set maxbound [expr $deltaT * $spreadfactor / $divisor]
			    if {$maxbound < 2.0e-4} {set maxbound 2.0e-4}
			    if {$lower($tr) == 0 && \
			         "" == [parameter b$dname$tsub fixlower]} {
				parameter b$dname$tsub lower -$maxbound
				ifdebug0 puts "setting lower bound to -$maxbound"
			    }
			    if {$upper($tr) == 0 && \
			         "" == [parameter b$dname$tsub fixupper]} {
				parameter b$dname$tsub upper $maxbound
				ifdebug0 puts "setting upper bound to $maxbound"
			    }
			}
		    }
		}
	    }
	} ;# end foreach covariate
    } ;# we have converted evd2 covariates to phase 1

    zscorexp -reset
    for {set i 0} {$i < 2} {incr i} {
	set rets ""

#*******************************************************************
#   Actual maximize is done here with errors caught
#     If this is EVD2, we simply run this to get evd's calculated then
#     real maximize is done later with evd transformed phenotypic data
#*******************************************************************

	set code [catch {eval ccmaximize $args} rets]
	ifdebug0 puts "covariates converted"

	ifdebug0 puts "\nString maximize returned is $rets; code is $code\n"

	if {-1 != [string first "Trap EVD Phase 2" $rets]} {

	    if {[option evdmat] > 0} {
#		puts "Eigenvectors Made, returning blank"
		return ""
	    }
	    option evdphase 2
	    set maxibd [lindex $rets end]
	    ifdebug0 puts "Max IBDID is $maxibd"
#
# Use info file rather than globals because info file
# remains available in future session should user forget
#
	    set EVD_phenname [phenotypes -files]
	    set outfile [open .evd2_info w]
	    puts $outfile "EVD2_dirname=$EVD2_dirname"
	    puts $outfile "EVD2_phenname=$EVD_phenname"
	    close $outfile

	    exec mv pedindex.out [full_filename evd2.pedindex.out]
	    exec mv pedindex.cde [full_filename evd2.pedindex.cde]
	    exec mv phi2.gz [full_filename evd2.phi2.gz]
	    makefakepedindex $maxibd
	    makefakephi2 $maxibd
#
# translate model to EVD2 Phase 2 parameterization
#
	    set ntraits [llength [traits]]

	    load phenotypes [full_filename evddata.out]
	    set covars1 [covariates]
	    set parameters [parameter -return]

# ************************* Phase Two Model ****************************

	    model new
	    ifdebug0 puts "STARTING NEW EVD2 phase 2 model"
	    set newtraits {}
	    set newcovars {}
#
# Traits each have _evd appended
#
	    foreach trait $traits {
		lappend newtraits $trait\_evd
	    }
	    eval trait $newtraits
#
# Covariates have _evd appended at the end of each name
#   use new cov names as in phase 1 model
#
	    foreach cov $covars1 {
		ifdebug0 puts "analyzing cov $cov"
#
# transformed data is already sample restricted, so we may ignore suspended
# covariates in the phase 2 model, and we may also ignore null-trait covariates
#
		if {"Suspended\[" != [string range $cov 0 9] &&
		    "\(\)" != [string range $cov end-1 end]} {

# No need for special handling interactions cause they were all converted
#   prior to phase one (see above).
# However, all covariates need evd appended prior to ()
		    if {-1 == [set pos [string first \( $cov]]} {
			set cov $cov\_evd
		    } else {
			set cov "[string range $cov 0 $pos-1]_evd[string range $cov $pos end]"
		    }
		    eval covariate $cov
		    noscale $cov
		}
	    }

#	    global pnames
	    covariate tmean
	    noscale tmean

# Replay matrix load commands

	    foreach mat $matrices {
		ifdebug0 puts "evaulating $mat"
		if {[lindex $mat 2] != "phi2.gz"} {
		    eval $mat
		}
	    }

# Replay old parameter values, adjusting as needed
# first do mean, btmean, and sd parameters

	    ifdebug0 puts "REPLAYING PARAMETER VALUES"

	    set vparameters {} ;# do variance components last seems to help
	    foreach par $parameters {
		set line "parameter $par"
		ifdebug0 puts "line is $line"
		set pname [lindex $line 1]
		set newpname $pname
		set pname3 [string range $pname 0 2]
		set pname5 [string range $pname 0 4]
#
# parameter mean gets played into btmean
# actual parameter "mean" is a dummy constrained to 0
# setup btmean with correct start, upper, and lower bounds for true mean
# setup pnames array with back translations
#
		if {($ntraits==1 && $pname == "mean") || \
			($ntraits>1 && $pname5 == "mean\(")} {
		    if {$ntraits==1} {
# univariate
			parameter mean = 0 lower 0 upper 1

			set pname btmean
			set pnames(btmean) mean
			set pnames(mean) *
			set ttmean 0
			if {[lindex $line 2] == "="} {
			    set ttmean [lindex $line 3]
			}
			if {$ttmean != 0} {
			    ifdebug0 puts \
				"setting btmean to $ttmean from user"
			    parameter btmean = $ttmean
			} else {
			    set ttmean [getvar -mean \
				    $EVD2_dirname/evd2.getmeans.out $traits]
			    ifdebug0 puts \
				"setting btmean to $ttmean from stats"
			    parameter btmean = $ttmean
			}
			set ttlower 0
			if {[lindex $line 4] == "lower"} {
			    set ttlower [lindex $line 5]
			}
			if {$ttlower != 0} {
			    parameter btmean lower $ttlower
			    ifdebug0 puts \
				"setting btmean lower to $ttlower from user"
			} else {
			    set ttlower [getvar -min \
				       $EVD2_dirname/evd2.getmeans.out $traits]
			    parameter btmean lower $ttlower
			    ifdebug0 puts \
				"setting btmean lower to $ttlower from stats"
			}
			set ttupper 0
			if {[lindex $line 6] == "upper"} {
			    set ttupper [lindex $line 7]
			}
			if {$ttupper != 0} {
			    parameter btmean upper $ttupper
			    ifdebug0 puts \
				"setting btmean upper to $ttupper from user"
			} else {
			    set ttupper [getvar -max \
				       $EVD2_dirname/evd2.getmeans.out $traits]
			    parameter btmean upper $ttupper
			    ifdebug0 puts \
				"setting btmean upper to $ttupper from stats"
			}
		    } else {
# multivariate mean
			ifdebug0 puts "translate multivariate mean parameter"

			set meanname "[string range $pname 0 end-1]_evd\)"
			set pnames(bt$meanname) $pname
			set pnames($meanname) *

			parameter $meanname = 0 lower 0 ;# dummy "mean" param

			set newline "parameter bt$meanname [lrange $line 2 end]"
			ifdebug0 puts "eval $newline"
			eval $newline
			set firstc [string first \( $pname]
			set newtname [string range $pname [expr $firstc+1] end-1]
			if {[parameter bt$meanname =] == 0} {
			    set newtmean [getvar -mean \
					      $EVD2_dirname/evd2.getmeans.out \
					      $newtname]
			    ifdebug0 puts "setting bt$meanname to $newtmean"
			    parameter bt$meanname = $newtmean
			}
			if {[parameter bt$meanname lower] == 0} {
			    set newtmin [getvar -min \
			       $EVD2_dirname/evd2.getmeans.out $newtname]
			  ifdebug0 puts "setting bt$meanname lower to $newtmin"
			    parameter bt$meanname lower $newtmin
			}
			if {[parameter bt$meanname upper] == 0} {
			    set newtmax [getvar -max \
				     $EVD2_dirname/evd2.getmeans.out $newtname]
			  ifdebug0 puts "setting bt$meanname upper to $newtmax"
			    parameter bt$meanname upper $newtmax
			}
		    }
		} elseif {($ntraits==1 && $pname == "sd") || \
			      ($ntraits>1 && $pname3 == "sd\(")} {

# unlike mean, for sd there is no ignored base parameter and bt parameter

		    if {$ntraits==1} {
			set pnames($pname) $pname
			ifdebug0 puts "evaluating $line"
			eval $line
			if {[parameter sd =] == 0} {
			    set tsd [getvar -std \
				    $EVD2_dirname/evd2.getmeans.out $traits]
			    ifdebug0 puts "setting sd to $tsd using stats"
			    parameter sd = $tsd
			}
			if {[parameter sd upper] == 0} {
			    parameter sd upper [expr [parameter sd =] * 5]
			}
# lower defaults to zero unless already set otherwise
		    } else {
			set newpname "[string range $pname 0 end-1]_evd\)"
			set pnames($newpname) $pname
			set line "parameter $newpname [lrange $line 2 end]"
			ifdebug0 puts "evaluating $line"
			eval $line
			if {[parameter $newpname  =] == 0} {
			    set firstc [string first \( $pname]
			    set newtname [string range $pname \
					      [expr $firstc+1] end-1]
			    set tsd [getvar -std \
                                $EVD2_dirname/evd2.getmeans.out $newtname]
			    parameter $newpname = $tsd
			    ifdebug0 puts "setting $newpname to $tsd"
			}
			if {[parameter $newpname upper] == 0} {
			  parameter $newpname upper [expr [parameter $newpname \
							       = ] * 5]
			}
# sd lower would simply be set to zero if not already zero                 
		    }
		} else {
		    ifdebug0 puts "  appending to vparameters"
		    lappend vparameters $line
		}
	    }


#  *** for non-mean non-sd parameters ***
	    ifdebug0 puts "VPARAMTERS!"
	    set last_pname ""
	    foreach line $vparameters {
		ifdebug0 puts "line is $line"
		set pname [lindex $line 1]
		if {$pname == $last_pname} {
		    set domeans 0
		} else {
		    set domeans 1
		}
		set last_pname $pname

		if {[string index $pname 0] == "b"} {
#
# If this is a beta parameter, evd is appended prior to (trait)...if any
#   and each variable included must have _evd appended also
#
# Boundary setting was done prior to EVD-calculating "maximize" so is not
# needed here, we just copy the existing boundaries.
#
		    set oldindex 0
		    set arrow 0  ;# set after arrow since exponent no _evd
		    set newpname ""
		    set thistrait $traits
		    set thisterm ""
		    set vsums 0
		    while {"" != [set nc [string index $pname $oldindex]]} {
			if {$nc == "*" || $nc == "^" || $nc == "("} {
			    if {!$arrow} {
				set newpname "$newpname\_evd$nc"
			    } else {
				set newpname "$newpname$nc"
			    }
			    if {$nc == "^"} {
				set arrow 1
			    } else {
				set arrow 0
			    }
			    if {$nc == "("} {
			       set thistrait [string range $pname \
						  [expr $oldindex+1] \
						  end-1]
			    }
			    set thisterm ""
			} else {
			    set newpname "$newpname$nc"
			    if {$oldindex > 0 && !$arrow} {
				set thisterm "$thisterm$nc"
			    }
			}
			incr oldindex
		    }
		    if {$ntraits==1 && !$arrow} {
			set newpname "$newpname\_evd"
		    }
		    
		} else {

# for non-beta parameters, change nothing now

		    set newpname $pname
		}

# Now, for each multivariate parameter having (tr), including betas,
#  suffix the tr with _evd

		if {$ntraits > 1} {
		    if {"()" != [string range $newpname end-1 end]} {
			if {-1 != [set cpos [string first \( $newpname]]} {
			    set newpname \
				"[string range $newpname 0 end-1]_evd\)"
			}
		    }
		}

# Now we have created new parameter name, associate it with original

		set pnames($newpname) $pname

# Substitute back into line and do the command

		set line "parameter $newpname [lrange $line 2 end]"
		ifdebug0 puts "evaluating: $line"
		eval $line
	    }

# Replay old constrants
#   and options

	    set inmodel [open $outfilename.evdphase1.mod]
	    gets $inmodel line
	    while {"" != $line} {
		if {[lindex $line 0] == "constraint"} {
		    if {$ntraits == 1} {
			ifdebug0 puts "evaluating $line"
			eval $line
		    } else {
			set newline ""
			while {-1 != [set ppos [string first ")" $line]]} {
			    set newline "$newline[string range $line 0 [expr $ppos-1]]_evd\)"
			    set line [string range $line [expr $ppos+1] end]
			}
			set newline $newline$line
			ifdebug0 puts "evaluating $newline"
			eval $newline
		    }
		} elseif {[lindex $line 0] == "option"} {
		    set optionname [lindex $line 1]
		    set ifevd [string range [lindex $line 2] 0 2]
		    ifdebug0 puts "ifevd is $ifevd"
		    if {$ifevd != "evd" && $optionname != "MergeAllPeds"} {
			ifdebug0 puts "evaluating $line"
			eval $line
		    }
		}
		set linelen [gets $inmodel line]
		ifdebug0 puts "got line $line with len $linelen"
		if {$linelen <= 0} {
		    ifdebug0 puts "breaking"
		    break
		}
	    }
	    ifdebug0 puts "closing model file"
	    close $inmodel

# special new constraints for mean

	    foreach tr $newtraits {
		if {$ntraits > 1} {
		    ifdebug0 puts "doing constraint mean($tr) = 0"
		    constraint mean($tr) = 0
		} else {
		    constraint mean = 0
		}
	    }

# do user omega if provided

	    if {[if_global_exists SOLAR_EVD2_omega]} {
		global SOLAR_EVD2_omega
		if {[catch {eval $SOLAR_EVD2_omega} errmes]} {
		    evd2_restore_phen
		    error "user omega returned $errmes"
		}
	    } else {		


	    if {$ntraits  == 1} {
		omega = pvar*(h2r*lambda_i + e2)
	    } elseif {$ntraits == 2} {
                omega = <sd(ti)>*<sd(tj)>*(teq*(<h2r(ti)>*lambda_i + <e2(ti)>) + tne*(sqrt(abs(<h2r(ti)>))*sqrt(abs(<h2r(tj)>))*lambda_i*rhog + sqrt(abs(<e2(ti)>))*sqrt(abs(<e2(tj)>))*rhoe))
	    } else {
		ifdebug0 puts "doing trivariate omega"
                omega = <sd(ti)>*<sd(tj)>*(teq*(<h2r(ti)>*lambda_i + <e2(ti)>) + tne*(sqrt(<h2r(ti)>)*sqrt(<h2r(tj)>)*lambda_i*rhog_ij + sqrt(<e2(ti)>)*sqrt(<e2(tj)>)*rhoe_ij))
	    }

	}
	    option evdoptimizations 1
	    save model $outfilename.evdphase2

	    ifdebug0 puts "begging phase 2 maximization"
	    set code2 [catch {maximize -o $userout} rets2]
	    ifdebug0 puts "Phase 2 maximize returned $code2 $rets2"

	    set loglike [loglike]
	    set maxpars [parameter -return]

	    save model $outfilename.evdphase2m
	    set thisout $userout

	    if {[string range $userout end-3 end] != ".out"} {
		set thisout $userout.out
	    }
	    exec cat $descout.out $thisout >$userout.tmp
	    exec mv $userout.tmp $thisout
#
# Restore original pedigree and phenotypes
#
	    evd2_restore_phen
#
# (since we just saved this model, it is mostly assumed to be "correct")
#
# Restore original model, then load phenotype values from phase 2 model into it
#
	    set imean 0
	    load model $outfilename.evdphase1

# Read parameter values from phase2 model

	    foreach mp $maxpars {
		ifdebug0 puts "Processing: $mp"
		set pname2 [lindex $mp 0]
		if {[catch {set pname $pnames($pname2)}]} {
		    puts "these are available: [array names pnames]"
		    error "Don't recognize parameter $pname2"
		}
		ifdebug0 puts "Parameter name is now $pname"

		if {[catch {set pname $p1names($pname)}]} {}

		if {$pname == "*"} {continue} ;# skip mean parameter
		set line "parameter $pname [lrange $mp 1 end]"
		ifdebug0 puts "   Evaluating:  $line"
		eval $line
	    }

	    loglike set $loglike
	    option modeltype evd2
	    option evdphase 3
	    return -code $code2 $rets2 ;# don't return evd2 trap
	    }

	if {[lindex $rets 0] != "Trap"} {

# TRAP is used when previous maximize sets up global env so next maximize
# can run.  It is now used for zscore in expressions, but could be extended
# to inormal, etc.
	    break
	}
    }
    return -code $code $rets
}

proc copyparameter {name values} {
    parameter $name = [parameter $values =]
    parameter $name se [parameter $values se]
    parameter $name lower [parameter $values lower]
    parameter $name upper [parameter $values upper]
    parameter $name score 0
    return ""
}

# solar::evd2 -- private
# solar::evd -- private
#
# EVD1 and EVD2
#
# EVD1 is our first EVD method, only works for univariate quantitative traits.
# EVD2 is our second EVD method, intended for all standard models, including
#   multivariate.
#
# To nudge a model to be an EVD1 model you give the command:
#
# option modeltype evd
#
# and to nudge a model to be an EVD2 model, you give the command:
#
# option modeltype evd2
#
# Then you set the trait, covariates, etc., just as for a regular model.
# Then you can run "maximize", "polygenic", or other model maximizing
# command as with original models.
#
# EVD2 maximization is faster because it eliminates the need for matrix
# inversions, and it is no more complicated for the user than normal
# maximization.  However, the maximization process is fairly complicated.
# During a phase 1 maximization the sample is determined as usual and
# transformed data files are output.  Covariates are converted into definitions
# so that each term can be scaled correctly.  A special model is created based
# on the user model that uses these transformed data, and maximized.
# Then the model is translated back into the usual form and the original
# phenotypes and pedigree files are enabled without having to be reloaded.
#
# Because of this complexity, it is possible that EVD2 maximization will
# fail and leave the state of pedigree and phenotypes loaded as invalid..  If
# this happens, it is recommended to reload the pedigree file, reload the
# phenotypes file, and give the command "model new" before creating
# additional models.  If the failure has just happened, you may also be
# able to use the shortcut command:
#
#   evd2_restore_phen
#
# to restore the pedigree and phenotypes to a valid state.  Note that the
# transformed phenotypes and special model files are stored in the current
# maximization output directory.  Note that EVD maximization of both types
# suppresses the usual iteration output for greater efficiency.
#
#-


# SOLAR::evd2_restore_phen --
#
# Purpose: restore the pedigree and phenotypes files used prior to EVD2 phase 2
#
# Usage: evd2_restore_phen
#
# Note: saves fake pedigree files in phase two output directory
# -

proc evd2_restore_phen {} {

    if {![file exists .evd2_info]} {
	return "Did not run EVD2 phase 1 to completion, or phen already restored"
    }
    set infile [open .evd2_info]
    gets $infile line
    set EVD2_dirname [lindex [split $line =] 1]
    gets $infile line
    set EVD2_phenname [lindex [split $line =] 1]
    close $infile

    exec mv pedindex.out [full_filename evd2.pedindex.out].phase2
    exec mv pedindex.cde [full_filename evd2.pedindex.cde].phase2
    exec mv phi2.gz [full_filename evd2.phi2.gz].phase2

    exec mv $EVD2_dirname\evd2.pedindex.out pedindex.out
    exec mv $EVD2_dirname\evd2.pedindex.cde pedindex.cde
    exec mv $EVD2_dirname\evd2.phi2.gz phi2.gz
    eval load phen $EVD2_phenname
    file delete .evd2_info
    return OK
}

#
# SOLAR::makefakepedindex -- private
#
# Purpose: generate an all-founders pedindex for EVD2
#
# Usage: makefakepedindex maxibdid
#-

proc makefakepedindex {maxibdid} {
#
# copy new pedindex.cde from example
#
    set ofile [open pedindex.out w]
    set idlen [string length $maxibdid]
    for {set i 1} {$i <= $maxibdid} {incr i} {
	set id [format %5s $i]
	set idl [format %$idlen\s $i]
	puts $ofile \
	    "$id     0     0 1   0 $id     0 $idl"
    }
    close $ofile
    makefakepedindexcde $idlen
}

proc makefakepedindexcde {idlen} {
    set ofile [open pedindex.cde w]
    puts $ofile "pedindex.out                                          "
    puts $ofile " 5 IBDID                 IBDID                       I"
    puts $ofile " 1 BLANK                 BLANK                       C"
    puts $ofile " 5 FATHER'S IBDID        FIBDID                      I"
    puts $ofile " 1 BLANK                 BLANK                       C"
    puts $ofile " 5 MOTHER'S IBDID        MIBDID                      I"
    puts $ofile " 1 BLANK                 BLANK                       C"
    puts $ofile " 1 SEX                   SEX                         I"
    puts $ofile " 1 BLANK                 BLANK                       C"
    puts $ofile " 3 MZTWIN                MZTWIN                      I"
    puts $ofile " 1 BLANK                 BLANK                       C"
    puts $ofile " 5 PEDIGREE NUMBER       PEDNO                       I"
    puts $ofile " 1 BLANK                 BLANK                       C"
    puts $ofile " 5 GENERATION NUMBER     GEN                         I"
    puts $ofile " 1 BLANK                 BLANK                       C"
    puts $ofile " $idlen ID                    ID                          C"
    close $ofile
}

proc makefakephi2 {maxibdid} {
    set ofile [open phi2 w]
    for {set i 1} {$i < $maxibdid} {incr i} {
	set id [format %5s $i]
	puts $ofile "$id $id  1.0000000  1.0000000"
    }
    close $ofile
    exec gzip phi2
}


# SOLAR::restart-if-can -- private
#
# Purpose: Block SOLAR crunching retries (default is not blocked)
#
# Usage:  restart-if-can 1    ; allow retries
#         restart-if-can 0    ; block retries
#         restart-if-can      ; show current status
#
# Note:   this is used inside the maximize procedure
#-

proc restart-if-can {args} {
    global Solar_Restart_If_Can
    if {[llength $args]} {
	set Solar_Restart_If_Can $args
	return
    }
    return [use_global_if_defined Solar_Restart_If_Can 1]
}


# solar::inormal -- 
#
# Purpose:  Save inverse normal transformation to a file (see also define)
#
# IMPORTANT: To create a model using an inverse normal transformation,
#            it is more convenient to use the "define" command, and NOT
#            the inormal command.  The "inormal" command itself is for
#            those rare situations where you need to save the inverse
#            normal transformation to a file for some other purpose.
#
# Usage:    define <defname> = inormal_<phenotype>
#           trait <defname>
#
#           inormal -trait <trait> [-file <filename>] -out <filename>
#                   [-phenfile] [-nofamid] [-class <class>]
#
#           -class <class> only include ID's when their class variable equals this value
#
#           (See notes below for obscure forms of the inormal command not
#            recommended for most users.)
#
# Notes:    For the "define" command, the <defname> can be any name you
#           can make up.  The inormal_ prefix may be abbreviated down to inor_ .
#           The <phenotype> is any phenotypic variable in the currently
#           loaded phenotypes file.
#
#           For the "inormal" command itself, you must use one of the
#           arguments "-phenfile" or "-file <filename>".  The "-phenfile"
#           argument is a shorthand way of specifying the currently loaded
#           phenotypes file.  The "-file <filename>" argument is used to
#           specify any file.  In either case, the file must be in the form
#           of a phenotypes file, with fields for <trait> and ID (and FAMID
#           if required to make the ID's unique).  BE SURE TO SPECIFY THE
#           "-out" ARGUMENT FOR THE OUTPUT FILE.
#
#           The inverse normal transformation of a dataset is performed
#           by the following procedure:
#
#             The trait values are sorted, and for any value V found
#             at position I in the sorted list, a quantile is computed
#             for it by the formula I/(N+1).  The inverse normal
#             cumulative density function (see "normal") is computed for
#             each quantile and stored in an array keyed by ID, and 
#             FAMID if applicable.  When the value V occurs multiple times,
#             the inverse normal is computed for each applicable quantile,
#             averaged, then the average is what is stored for each ID.
#             These values are accessed when the ID is provided.  The
#             array for each trait is deleted by the -free option.
#           
#           See also the "normal" command, which computes normal distribution
#           functions.  inormal uses a "normal -inverse" command.
#
#           OBSCURE FORMS OF THE INORMAL COMMAND
#
#           Largely for internal purposes, such as the implementation of
#           the define command, there are additional obscure forms of the
#           inormal command which save the inverse normal results in
#           a tcl variable for access on an individual ID basis:
#
#           inormal -trait <trait> [-file <filename>] -tclvar
#                   [-phenfile] [-nofamid] 
#           inormal -trait <trait> -id <id> [-famid <famid>]
#           inormal -free <trait>
#           inormal -reset
#
#           The first form above is like the standard form, except that the
#           -out argument is replaced with a -tclvar argument, signifying
#           that the inverse normal results are to be saved to a Tcl variable
#           associated with the trait name.  In the second form, a result is
#           obtained from previously stored data for each ID.  In the third
#           form, stored data for a particular trait is freed.  In the
#           fourth form, all stored data is freed.
#
#           The -out and -tclvar arguments cannot be used at the same time.
#           If the -out argument is used, inverse normals are simply written
#           to a file and nothing is stored, so the second form cannot be
#           used.
#
#           FAMID should only be specified if required.
#           The rules regarding FAMID are almost identical with
#           those used during maximization, so that in general you don't
#           have to think about them.  If FAMID field is found in both
#           pedigree and phenotypes files, or if pedigree file isn't loaded
#           (which wouldn't be allowed during maximization) and FAMID is
#           found (only) in phenotypes file,  FAMID is automatically required,
#           unless the -nofamid argument is used.  If FAMID is found in
#           only one of the two files (and both are loaded), a test for
#           ID uniqueness is performed, then if ID's are unique without FAMID,
#           it is not required, otherwise FAMID is required and if not present,
#           it is an error.  FAMID can be mapped to any other field name using
#           the field command.
#
#           When using these obscure forms of the inormal command, it is
#           recommended to load the data and then use it in short order,
#           even though the inormal command doesn't intrinsically require
#           this.  Internal "inormal" data is not saved from one SOLAR
#           session to the next.
#
#           BEWARE that "maximize" or any SOLAR command that performs
#           maximization, such as "polygenic" or "multipoint", may clear
#           out inverse normal data stored using -tclvar.  Also, if
#           different layers of procedures get inormals on traits
#           with the same name from different files, and their inormal
#           operations overlap, there
#           could be problems.
#
#           When the -class option is used, the traitname
#           is qualified with a suffix like .SOLARclass.1 (where 1 is the class
#           number).  To free such a classed trait, the fully suffixed name must
#           be used.  For example, for trait q4 and class 1, the command would be:
#
#               inormal -free q4.SOLARclass.1
# -

# Globals:
# SOLAR_Inormal_Traits:  List of trait names currently stored
# SOLAR_Inormal_Results_<i>(<id>)
#   Arrays of results for each <i> where <i> is trait index
#   and <id> is formatted ID.  ID is formatted like this:
#     <id>.famid.<famid>
#   <famid> can be null if famid not present or -nofamid arg given
#   Why this ugly name to combine id and famid?  Because it's unambiguous
#   and ugly enough that no one would be likely to preformat their ID's
#   this way.
#
#   Examples:
#       299.famid.       ID=299  no famid
#       299.famid.10     ID=299  FAMID=10
#-


proc inormal {args} {

    global SOLAR_Inormal_Traits   ;# Stored traits, indexes into other lists

    set LOCAL_DEBUG 0

    set phen_filename ""
    set traitname ""
    set free_traitname ""
    set get_id ""
    set famid ""
    set nofamid 0
    set phenfile 0
    set reset 0
    set outfilename ""
    set tclvar 0
    set class ""
    set classpos 2

    set badargs [read_arglist $args \
		     -file phen_filename \
                     -out outfilename \
		     -free free_traitname \
		     -trait traitname \
		     -id get_id \
		     -nofamid {set nofamid 1} \
		     -famid famid \
		     -debug {set LOCAL_DEBUG 1} \
		     -phenfile {set phenfile 1} \
		     -reset {set reset 1} \
		     -tclvar {set tclvar 1} \
		     -class class
		     ]
    if {"" != $badargs} {
	error "inormal: Invalid arguments: $badargs"
    }
#
# Check for valid options
#
    if {$reset && 1 != [llength $args]} {
	error "inormal: -reset must not be used with other arguments"
    }
    if {"" != $outfilename && 0 == $phenfile && "" == $phen_filename} {
	error "inormal: -out option requires -file or -phenfile option"
    }
#
# If class option, add qualifier to trait name
#
    set traitclassname $traitname
    if {$class != ""} {
	set traitclassname $traitname.SOLARclass.$class
    }
#
# Load new trait from new datafile *******************************************
#
    if {"" != $phen_filename || $phenfile} {
	if {"" == $traitname} {
	    error "inormal: -trait required with -file"
	}
	if {"" != $phen_filename && $phenfile} {
	    error "inormal: Either specify -phenfile or -file <filename>, not both"
	}
	if {"" != $outfilename} {
	    catch {file delete $outfilename}
	    if {[file exists $outfilename]} {
		error "File $outfilename is protected from deletion"
	    }
	}

#
# Check for previous data
#
	if {[if_global_exists SOLAR_Inormal_Traits] && \
		-1 < [lsearch $SOLAR_Inormal_Traits $traitclassname]} {
	    error "inormal: Must free previous data:  inormal -free $traitclassname"
	}
#
# Get Ranking Identifier (rid) to identify this ranking information
#
	if {![if_global_exists SOLAR_Inormal_Traits]} {
	    set SOLAR_Inormal_Traits [list $traitclassname]
	} else {
	    lappend SOLAR_Inormal_Traits $traitclassname
	}
	set rid [expr [llength $SOLAR_Inormal_Traits] - 1]
	if {$LOCAL_DEBUG} {puts "Using RID $rid"}
#
# Find trait in phenotypes file(s)
#
	if {$phenfile} {
	    set files [phenotypes -files]
	    foreach file $files {
		if {[file exists $file]} {
		    set sid [solarfile open $file]
		    if {[solarfile $sid test_name $traitname]} {
			solarfile $sid close
			set phen_filename $file
			break
		    }
		    solarfile $sid close
		}
	    }
	    if {"" == $phen_filename} {
		error "inormal: trait not found in phenotypes file(s)"
	    }
	}

#
# Open datafile
#
	if {![file exists $phen_filename]} {
	    catch {inormal -free $traitclassname}
	    error "inormal: Data file $phen_filename not found"
	}
	set datafile [solarfile open $phen_filename]

# set up id, trait, [famid]

	set inormal_data {}

	solarfile $datafile start_setup
	if {[catch {solarfile $datafile setup id}]} {
	    solarfile $datafile close
	    catch {inormal -free $traitclassname}
	    error "inormal: ID field not found in $phen_filename"
	}
	if {[catch {solarfile $datafile setup $traitname}]} {
	    solarfile $datafile close
	    catch {inormal -free $traitclassname}
	    error "inormal: Variable $traitname not found in $phen_filename"
	}
	set need_famid 0
	global SOLAR_Inormal_Famid
	if {!$nofamid} {
	    if {[catch {set need_famid [check_phenotypes $phen_filename]} errmes]} {
		if {-1 == [string first uplicate $errmes]} {
		    error "inormal: error reported in phenotypes file:\n$errmes"
		}
		set need_famid 1
	    }
	}
	if {$need_famid} {
	    lappend SOLAR_Inormal_Famid $traitclassname
	} else {
	    if {[if_global_exists SOLAR_Inormal_Famid]} {
		set SOLAR_Inormal_Famid [remove_from_list $SOLAR_Inormal_Famid $traitclassname]
	    }
	}
	if {$need_famid} {
	    incr classpos
	    solarfile $datafile setup famid
	}
	if {$class != ""} {
	    solarfile $datafile setup class
	}
	if {"" != $outfilename} {
	    if {$need_famid} {
		putsout -q -d. $outfilename "id,famid,inormal_$traitname"
	    } else {
		putsout -q -d. $outfilename "id,inormal_$traitname"
	    }
	}
#
# Read data, inserting data and key values in list
# to be sorted later
#
	set Count 0
	set check_keys {}
	while {{} != [set line [solarfile $datafile get]]} {
	    set id [lindex $line 0]
	    set data [lindex $line 1]
	    if {{} == $data} {continue}  ;# Missing data?
	    if {$class != ""} {
		set classfound [lindex $line $classpos]
		if {$classfound != $class} {continue}
	    }
	    set famid ""
	    if {$need_famid} {
		set famid [lindex $line 2]
	    }
	    set key "$id.famid.$famid"
#	    puts "key is $key"
#	    puts "check_keys is $check_keys"
	    if {-1 != [lsearch $check_keys $key]} {
		solarfile $datafile close
		catch {inormal -free $traitclassname}
		error "inormal: ID $id repeated in $phen_filename; FAMID required"
	    }
	    lappend check_keys $key
#
# Append new data to list
#
	    lappend inormal_data [list $key $data]
	    incr Count
	}
	solarfile $datafile close
#
# Now, sort list and create array with results
#
	set inormal_data [lsort -index end -real $inormal_data]
	global SOLAR_Inormal_Results_$rid

	set last_data {}
	set last_position -1

	set data_length [llength $inormal_data]
	set current_position 0
	set force_last 0
	foreach element $inormal_data {
	    incr current_position
	    set key [lindex $element 0]
	    set data [lindex $element 1]

 	    set pass 1
	    while {$pass || $force_last} {
		set pass 0
		if {$LOCAL_DEBUG} {puts "current_position: $current_position"}

		if {{} == $last_data} {
		    set last_data $data
		    set last_position $current_position
		} else {
		    if {$data > $last_data || $force_last} {
#
# New different data value
# So process previous same data values
#
			set matches 0
			set sum 0.0
			for {set i $last_position} {$i < $current_position} {incr i} {
			    set pct [expr double($i) / double($Count + 1.0)]
			    set z [normal -inverse $pct]
			    set sum [expr $sum + double($z)]
			    incr matches
			}
			set zavg [expr $sum / $matches]
			for {set i $last_position} {$i < $current_position} {incr i} {
			    set ele [lindex $inormal_data [expr $i - 1]]
			    set key [lindex $ele 0]
			    if {$LOCAL_DEBUG} {puts "Assigning $zavg to $key"}
			    eval set [catenate SOLAR_Inormal_Results_$rid (\$key)] \$zavg
			    if {"" != $outfilename} {
				set endpos [string first .famid. $key]
				set id [string range $key 0 [expr $endpos - 1]]
				set famid [string range $key [expr $endpos + 7] end]
				if {$need_famid} {
				    putsout -q -d. $outfilename "$id,$famid,$zavg"
				} else {
				    putsout -q -d. $outfilename "$id,$zavg"
				}
			    }
			}
			
#
# Now store new data value
#
			set last_data $data
			set last_position $current_position
			if {$force_last} {
			    break
			}
		    }
#
# Force one last iteration to store last element(s)
#
		    if {$current_position == $data_length} {
			set force_last 1
			incr current_position
		    }
		}
	    }
	}
	set save_count $Count

	if {"" != $outfilename} {
	    inormal -free $traitclassname
	}

	return $save_count
    }
#
# Get Percentile *************************************************************
#
    if {"" != $get_id} {
	if {"" == $traitname} {
	    error "inormal: -id option requires -trait"
	}
	if {![if_global_exists SOLAR_Inormal_Traits]} {
	    error "inormal: no traits have been loaded for ranking"
	}

# build key, adding famid only if required, but requiring it if it is required

	global SOLAR_Inormal_Famid
	if {"" == $famid} {
	    if {[if_global_exists SOLAR_Inormal_Famid]} {
		if {-1 != [lsearch $SOLAR_Inormal_Famid $traitclassname]} {
		    error "inormal: -famid required for trait $traitname"
		}
	    }
	    set get_id [catenate $get_id .famid.]
	} else {
	    if {![if_global_exists SOLAR_Inormal_Famid] || \
		    -1 == [lsearch $SOLAR_Inormal_Famid $traitclassname]} {
		set get_id [catenate $get_id .famid.]
	    } else {
		set get_id [catenate $get_id .famid. $famid]
	    }
	}

	set rid [lsearch $SOLAR_Inormal_Traits $traitclassname]
	if {-1 == $rid} {
	    if {$class == ""} {
		error "inormal: trait $traitname is not loaded"
	    } else {
		error "inormal: trait $traitname with class $class not loaded"
	    }
	}

	global SOLAR_Inormal_Results_$rid

	set inormal {}
	catch {
	   eval set inormal [catenate \$SOLAR_Inormal_Results_ $rid (\$get_id)]
	}
#
# Return data found or null
#
	return $inormal
    }
#
# "-free" option
#
    if {"" != $free_traitname} {
	if {"" != $class} {
	    set free_traitname $free_traitname.SOLARclass.$class
	}
	if {[if_global_exists SOLAR_Inormal_Traits]} {
	    if {-1 != [set rid [lsearch $SOLAR_Inormal_Traits $free_traitname]]} {
		global SOLAR_Inormal_Results_$rid
		catch {unset SOLAR_Inormal_Results_$rid}
		set SOLAR_Inormal_Traits [remove_from_list $SOLAR_Inormal_Traits $free_traitname]
		if {[if_global_exists SOLAR_Inormal_Famid]} {
		    global SOLAR_Inormal_Famid
		    set SOLAR_Inormal_Famid [remove_from_list $SOLAR_Inormal_Famid $free_traitname]
		}
		return ""
	    }
	    error "inormal:  No data saved for trait $free_traitname"
	}
	error "inormal:  No data saved"
    }
#
# -reset option
#
    if {"" != $reset} {
	if {[if_global_exists SOLAR_Inormal_Traits]} {
	    global SOLAR_Inormal_Traits
	    set rid -1
	    foreach trait $SOLAR_Inormal_Traits {
		incr rid
		global SOLAR_Inormal_Results_$rid
		catch {unset SOLAR_Inormal_Results_$rid}
	    }
	    unset SOLAR_Inormal_Traits
	}
	if {[if_global_exists SOLAR_Inormal_Famid]} {
	    global SOLAR_Inormal_Famid
	    unset SOLAR_Inormal_Famid
	}
	return ""
    }
	
    error "inormal: invalid command form (combination of arguments)"
}


# solar::normal --
#
# Purpose:  Normal distribution functions
#
# Usage:    normal -i[nverse] <p>
#
# Notes:    Currently, the only supported function is the "inverse normal
#           cumulative density function", which maps the open range
#           0,1 to the whole real line.  (The values for 0 and 1 are
#           out of range because they would be negative and positive
#           infinity.)
#
#           This normal function is used by the inormal procedure to
#           perform an inverse normal transformation on a dataset.
#           For further information, see the help for "inormal".
#           In turn, the inormal procedure is part of the mechanism
#           behind the "inormal_" prefix which may be applied to
#           phenotypes in the define command.
#
#           We will add additional normal distribution functions here as
#           we need them.
#
#           Our implementation is indirectly based on:
#
#             Cody, W.D. (1993). "ALGORITHM 715: SPECFUN - A Portabel FORTRAN
#             Package of Special Function Routines and Test Drivers"
#             ACM Transactions on Mathematical Software. 19, 22-32.
# -


# solar::stats --
#
# Purpose:  Get and/or show statistics for any variable in a file
#
# Usage:    stats [<variable>+ | -all [-file <filename>]] [-q] [-return]
#                 [-out <outfile>] -sample
#
#           -all     show stats for all variables in phenotypes file
#           -return  do not write output file, return list of stats;
#                      use stats_get to parse return list
#           -q       do not display to terminal
#           -out     specify alternate output filename; also returns list
#                    of stats
#           -sample  Use only the sample of the current model (there must be
#                    a current model, or at least a defined trait, and 
#                    statistics can only be computed for the trait(s) and
#                    covariate(s) in that model)
#
#           The default variable is the current trait, and the default
#           filename is the currently loaded phenotypes file.  You may also
#           specify one or more variables.
#
#           Results are written to stats.out in the CURRENT WORKING DIRECTORY.
#           (Not the usual output directory, since the trait need not be set.)
#           They are also displayed on the terminal.
#
#           The statistics computed are mean, minimum, maximum, standard
#           deviation, skewness, and kurtosis.  (Note: We define kurtosis
#           as 0 for a standard normal distribution; 3 has already been
#           subtracted from the normalized 4th central moment.)
#
#           See also the zscore command, which uses these stats to
#           zscore the current trait during maximization.  The zscore
#           procedure uses stats with the -out option.
#
#           If there are multiple phenotypes files, joinfiles will be
#           used to create a joined file in the working directory named
#           joinfiles.stats.[pid].[uname -n].out.  Non-unique fieldnames
#           will be renamed following the rules of joinfiles.  Under most
#           circumstances, this joined file will be deleted before
#           stats returns.  To run through the entire contents (and names)
#           in the joined file, use the "stats -all" command.
#
#           Variables created by a "define" command will work with stats
#           beginning with SOLAR version 4.2.2.  Note that such variables
#           are not evaluated by the command "stats -all".
# -

proc stats {args} {

    set variable ""
    set pf ""
    set outfile ""
    set outr ""
    set allvars 0
    set quiet 0
    set return 0
    set joinedfile 0
    set deletefile ""
    set fortrand 0
    set use_sample 0
    set add_vars ""

    if {{} != $args} {
	set variable [read_arglist $args -file pf -out outfile \
			  -all {set allvars 1} -q {set quiet 1} \
			  -sample {set use_sample 1} \
			  -return {set return 1}]
    }

    if {$use_sample} {
	if {[catch {trait}]} {
	    error "stats: trait must be defined for -sample"
	}
    }

    if {"" == $outfile} {
	set outfile stats.out
    } else {
	set return 1
    }

    if {$quiet} {
	set putsout_quiet 1
    } else {
	set putsout_quiet 0
    }
    
    set pfs $pf
    if {"" == $pf} {
	set pf [set pfs [phenotypes -files]]
	if {[llength $pfs] > 1} {
	    set deletefile joinfiles.stats.[pid].[exec uname -n].out
	    set pf $deletefile
	    eval joinfiles $pfs -o $pf
	    set joinedfile 1
	}
    }

    if {$allvars} {
	if {$use_sample} {
	    set testvariable [concat [trait] [covar]]
	    set variable ""
	    foreach var $testvariable {
		set subvar [split $var "*^"]
		set firstvar [lindex $subvar 0]
		if {"sex" != $firstvar} {
		    setappend variable $firstvar
		}
	    }
	} else {
	set phen [tablefile open $pf]
	set variable [tablefile $phen names]
	tablefile $phen close
	set variable [remove_from_list_if_found $variable id]
	set variable [remove_from_list_if_found $variable fa]
	set variable [remove_from_list_if_found $variable mo]
	set variable [remove_from_list_if_found $variable ego]
	set variable [remove_from_list_if_found $variable sire]
	set variable [remove_from_list_if_found $variable dam]
	}

    } elseif {"" == $variable} {
	if {[catch {set variable [trait]} errmsg]} {
	    if {$joinedfile} {
		file delete $deletefile
	    }
	    error $errmsg
	}
    } 

    notquiet puts ""
    putsout -d. $outfile "Phenotypes File: $pfs"
    set trno 0
    set dnames [define names]
    set got_sample 0
    foreach tr $variable {
	incr trno
	if {$use_sample} {
	    if {!$got_sample} {
		set savemodelname stats.presave.[pid]
		save model $savemodelname
		if {-1 != [string first Use_polygenic_to [omega]]} {
		    spormod
		}
		maximize -q -sampledata
		load model $savemodelname
		file delete $savemodelname.mod
		set got_sample 1
	    }
	    set phen [tablefile open [full_filename sampledata.out]]
	    if {-1 != [lsearch [trait] $tr]} {
		set tindex [expr 1 + [lsearch [trait] $tr]]
		set tabname trait$tindex
	    } else {
		set tabname $tr
	    }
	} elseif {-1 != [lsearch $dnames $tr]} {
	    set savemodelname stats.presave.[pid]
	    save model $savemodelname
	    model new
	    trait $tr
	    spormod
	    notquiet puts "\nEvaluating Definition: [define $tr]"
	    maximize -q -sampledata
	    set phen [tablefile open [full_filename sampledata.out]]
	    set tabname trait1
	    load model $savemodelname
	    file delete $savemodelname.mod
	} else {
	    set phen [tablefile open $pf]
	    set tabname $tr
	}
	tablefile $phen start_setup
	if {[catch {tablefile $phen setup $tabname} errmsg]} {
	    tablefile $phen close
	    if {1 == [llength $pfs]} {
		if {$use_sample} {
		    error "Variable $tabname not included in model"
		} else {
		    error $errmsg
		}
	    } else {
		if {$joinedfile} {
		    file delete $deletefile
		}
		if {$use_sample} {
		    error "Variable $tabname not included in model"
		} else {
		    error "$errmsg\n  $tr is not found or unique in files specified"
		}
	    }
	}
#
# Read records to compute mean and check if discrete
#
	set mean 0
	set min 0
	set max 0
	set sd NaN
	set skew NaN
	set kurt NaN
	set discrete 1
	set alpha 0

	set count 0
	set sum 0.0
	set missing 0

	while {{} != [set record [tablefile $phen get]]} {
	    set value [lindex $record 0]
	    if {{} == $value} {
		incr missing
	    } else {
		incr count
		if {$count == 1} {
		    set min $value
		}
		if {!$alpha} {
		    if {![is_float $value]} {
			set trialfloat [regsub D $value E]
			if {[is_float $trialfloat]} {
			    set value $trialfloat
			    set fortrand 1
			} else {
			    set alpha 1
			    set max $value
			    set discrete 0
			}
		    }
		}
		if {!$alpha} {
		    set sum [expr $sum + $value]
		    if {$count == 1} {
			set max $value
			set value1 $value
			set values_seen 1
		    } else {
			if {$min > $value} {set min $value}
			if {$max < $value} {set max $value}
			if {$values_seen == 1} {
			    if {$value != $value1} {
				set value2 $value
				set values_seen 2
			    }
			} elseif {$values_seen == 2} {
			    if {$value != $value1 && $value != $value2} {
				set values_seen 3
				set discrete 0
			    }
			}
		    }
		}
	    }
	}
	
	if {!$alpha && $count > 0} {
	    set mean [expr double ($sum) / double ($count)]
	}
	set dc [expr double ($count)]
#
# Reread records to compute SD
#
	if {!$alpha} {
	    if {$count > 1} {
		tablefile $phen rewind
		set sumsd 0.0
		while {{} != [set record [tablefile $phen get]]} {
		    set value [lindex $record 0]
		    if {{} != $value} {
			if {$fortrand} {
			    set value [regsub D $value E]
			}
			set dev [expr double($value) - $mean]
			set sdev [expr $dev * $dev]
			set sumsd [expr $sumsd + $sdev]
		    }
		}
		set var [expr $sumsd / double ($count - 1)]
		set sd [expr sqrt ($var)]
	    }
	
#
# Reread records to compute skewness and kurtosis
#
	    if {$count > 2 && $sd != 0} {
		tablefile $phen rewind
		set d3 0.0
		set d4 0.0
		while {{} != [set record [tablefile $phen get]]} {
		    set value [lindex $record 0]
		    if {{} != $value} {
			if {$fortrand} {
			    set value [regsub D $value E]
			}
			set dev [expr double($value) - $mean]
			set devd [expr $dev / $sd]
			set d3 [expr $d3 + ($devd * $devd * $devd)]
			set d4 [expr $d4 + ($devd * $devd * $devd * $devd)]
		    }
		}
		set skew [expr $d3 * $dc / (($dc - 1.0) * ($dc - 2.0))]
		catch {
		    set kurt [expr ($d4 * ($dc*($dc+1.0)) / \
					(($dc-1.0)*($dc-2.0)*($dc-3.0))) - \
				  ((3.0*($dc-1.0)*($dc-1.0)) / \
				       (($dc-2.0)*($dc-3.0)))]
		}
	    }
	}
#
# Report and save statistics
#
	tablefile $phen close
	if {$use_sample} {
	    putsout -d. $outfile "\nVariable:  $tr  Sample Size: $count"
	} else {
	    putsout -d. $outfile "\nVariable:  $tr  Sample Size: $count  Missing: $missing"
	}
	if {$alpha} {
	    putsout -d. $outfile   "First:                 $min"
	    putsout -d. $outfile   "Last:                  $max"
	    putsout -d. $outfile   "This variable contains alphanumeric entries and cannot be used as a phenotype."
	} else {
	    putsout -d. $outfile "\nMean:                  [format %.9g $mean]"
	    putsout -d. $outfile   "Minimum:               [format %.9g $min]"
	    putsout -d. $outfile   "Maximum:               [format %.9g $max]"
	    putsout -d. $outfile   "Standard Deviation:    [format %.9g $sd]"
	    putsout -d. $outfile   "Skewness:              [format %.7g $skew]"
	    putsout -d. $outfile   "Kurtosis:              [format %.7g $kurt]"
	}
#
# Check for discrete
#
	if {!$alpha && $discrete} {
	    if {($min==$max) || $max-$min>0.99 && $max-$min<1.01} {
		putsout -d. $outfile "This variable is discrete."
	    } else {
		putsout -d. $outfile "This variable seems discrete, but is not coded properly;\nSee \"help discrete-notes\""
		set discrete -1
	    }
	}
#
# Add to return list for -out option
#
	set outr "$outr variable: $tr"
	set outr "$outr count: $count"
	set outr "$outr missing: $missing"
	set outr "$outr mean: $mean"
	set outr "$outr min: $min"
	set outr "$outr max: $max"
	set outr "$outr sd: $sd"
	set outr "$outr skewness: $skew"
	set outr "$outr kurtosis: $kurt"
	set outr "$outr discrete: $discrete"
	set outr "$outr alpha: $alpha"
#
# Repeat for all variables
#
    }
    notquiet puts ""
    if {$return} {
	return $outr
    }
    if {$joinedfile} {
	file delete $deletefile
    }
    return ""
}

# solar::stats_get --
#
# Purpose:  Retrieve statistics from list returned by stats
#
# Usage:    stats_get <stats> <statistic> [<variable>]
#
#           <stats>      list returned by stats procedure
#           <statistic>  name of statistic desired (see below for list)
#           <variable>   select this variable (default: <first>)
#
# Example:  set stat [stats -q -return q1]
#           set kurt [stats_get $stat kurtosis]
#           set skew [stats_get $stat skewness]
#
# Notes:    The following statistics are available:
#
#           variable     name of variable
#           count        number of individuals having this variable (sample)
#           missing      number of individuals missing this variable
#           mean         mean
#           min          minimum value
#           max          maximum value
#           sd           standard deviation
#           skewness     skewness
#           kurtosis     kurtosis
#           discrete     0 if quantitative, 1 if discrete, -1 if not coded
#                          properly
#           alpha        0 if valid numbers; 1 if alphanumeric
#
#           Of course, if a variable is selected, that variable must have
#           been included in the stats list.  When running the stats command
#           you may select any number of variables or use the -all option.
#           See the stats command for further information.
#-

proc stats_get {stats field {variable ""}} {

    if {"" != $variable} {
	set target "variable: $variable"
	set index [string first $target $stats]
	if {-1 == $index} {
	    error "stats_get: variable $variable not found in stats list"
	}
	set index [expr [string length $target] + 1 + $index]
	set stats [string range $stats $index end]
    }

    set index [lsearch $stats [catenate $field :]]
    if {-1 == $index} {
	error "stats_get: field $field not found"
    }
    set value [lindex $stats [expr 1 + $index]]

    return $value
}


# solar::zscore --
#
# Purpose:  Zscore current trait(s) or covariate(s)
#
# Usage:    define defname = zscore_phenotype 
#           trait defname
#           OR
#           covariate defname
#
#           (defname is any user defined name, phenotype is any phenotype name)
#
# Notes:    zscore_ is a prefix that may be used in the define command,
#           similar to the inormal_ prefix.  Once a definition has been
#           created, it may be used in either the trait or covariate commands.
#           For further information, see "help define".
#
#           The Mean and SD are obtained from the current maximization sample,
#           not the entire phenotypes file.
#
#           In versions of SOLAR prior to 4.4.0, zscore was a command that
#           could be only used to zscore the current trait.  That command
#           is still available as before, but was considered obsolescent.
#           It was difficult and problemantical.  For information about that
#           command, for understanding previous uses, see "help old_zscore".
#
#-

# solar::old_zscore --
# solar::zs
#
# Purpose:  The old zscore command to zscore current trait
#          
# Old Usage:    zscore [-off] [-q]
#               zs     [-off]       ;# Perform zscore quietly
#
#               -off                Turn off zscore
#               -q                  Perform zscore quietly
#
# Notes:    The "Mean" and "SD" values used by zscore are computed only
#           once, at the time the zscore command is given.  Thus they do
#           not reflect later changes to the phenotypes file, or to the
#           sample, which might be restricted due to individuals missing
#           covariates added later.  Generally, for this reason the
#           zscore command should be given after the covariates command
#           and immediately before a model maximizing command such as
#           polygenic.
#
#           Starting with SOLAR Version 4.0.9, the trait mean and SD
#           are computed from the actual sample that would be included
#           in an analysis (at the time the zscore command is given).
#
#           As described in the notes below, you can adjust the Mean
#           and SD by using "option zmean1" and "option zsd1" to set
#           the values actually used.  These values are applied to
#           the trait values during maximization.
#
#           If the trait is changed without giving the "model new"
#           command, the new trait will be zscored automatically.
#           This feature is obsolescent.  In a future update, zscore
#           will be turned off when the trait is changed.
#
#           An alternative to zscore is to define the trait as the
#           inverse normal transformation of a variable.  See
#           "help inormal" and "help define" for further details.
#           
#           zscore will also calculate a number of statistics
#           for the trait: mean, minimum, maximum, standard deviation,
#           skewness, and kurtosis.  These will be written to the file
#           zscore.out in the current output directory.  As of version
#           4.0.9, these statistics are no longer written to the terminal.
#           Instead, a single line is displayed with the trait name,
#           mean, and SD.  Even that line is not shown if zscore is
#           invoked from a script or the zs abbreviation of the command
#           is used.
#
#           To calculate these statistics for any phenotypic variable without
#           zscoring and without necessarily making it the trait, use the
#           "stats" command instead.
#
#           A trait must already have been selected with the trait command
#           or loaded model.  Also the phenotypes file must have been loaded.
#
#           When a maximization is performed, trait values are replaced with
#           their zscored values.  The formula is:
#
#           zscored = (value - Mean) / SD
#
#           zscore is a model dependent option controlled by "option zscore".
#           It remains in effect until another model is loaded or the
#           "model new" command is given.  When models maximized with zscore
#           are reloaded, zscore is again activated.
#
#           "option zscore" is set to 1 ("on") by this command, and the
#           related options zmean1 and zsd1 (mean and standard deviation
#           for the first trait) and zmean2 and zsd2 (mean and standard
#           deviation for the second trait) are set as required.  You can
#           adjust these options directly to fine tune the mean and standard
#           deviation values used, but be sure that zscore is not set to 1
#           until the mean and (non-zero !) standard deviation values are
#           set for all traits in the model.
#
#           In a multivariate model, zscore will only be applied to the
#           first two traits.
#
#           Whenever zscore is activated or deactivated, parameters mean
#           and SD are reset to zero to force setting new boundaries and
#           starting point during the next maximization.
#
#           If a new phenotypes file is loaded, the zscore command should be
#           repeated to reflect the new file.
#-

proc zscorexp {args} {
    set outlist ""
    global SOLAR_Zscore_Phenotypes
    if {![if_global_exists SOLAR_Zscore_Phenotypes]} {
	set SOLAR_Zscore_Phenotypes ""
    }
    if {$args == "-reset"} {
	set SOLAR_Zscore_Phenotypes ""
	return ""
    }
    set opname [lindex $args 0]
    set pname [lindex $args 1]
    if {$opname == "set"} {
	global SOLAR_Zmean_$pname
	global SOLAR_ZSD_$pname
	set SOLAR_Zmean_$pname [lindex $args 2]
	set SOLAR_ZSD_$pname [lindex $args 3]
	setappend SOLAR_Zscore_Phenotypes $pname
	return ""
    } elseif {$opname == "get"} {
	if {-1 != [lsearch $SOLAR_Zscore_Phenotypes $pname]} {
	    set type [lindex $args 2]
	    if {$type == "mean"} {
		if {[if_global_exists SOLAR_Zmean_$pname]} {
		    global SOLAR_Zmean_$pname
		    eval return \$SOLAR_Zmean_$pname
		}
	    } elseif {$type == "sd"} {
		if {[if_global_exists SOLAR_ZSD_$pname]} {
		    global SOLAR_ZSD_$pname
		    eval return \$SOLAR_ZSD_$pname
		}
	    }
	}
    }
    return "not found"
}

proc zs {args} {
    return [eval zscore -q $args]
}

proc zscore {args} {

    set zscored 0
    set zfile [full_filename zscore.out]
    set ztemp [full_filename zscore.temp.out]
    set samplefile [full_filename sampledata.out]
    set savename [full_filename solar.zscore.orig.mod]
    file delete $zfile

    set off 0
    set quiet 0
    set badargs [read_arglist $args -off {set off 1} -q {set quiet 1}]
    if {{} != $badargs} {
	error "zscore: Invalid argument: $badargs"
    }

    set ts [trait]
    if {[llength $ts] == 1} {
	set multi 0
    } else {
	set multi 1
    }
#
# zscore always turned off now to prevent recursive zscore
#
    option zscore 0

    if {$off} {
	option zmean1 0.0
	option zsd1 0.0
	option zmean2 0.0
	option zsd2 0.0
	if {!$multi} {
	    if {[if_parameter_exists mean]} {
		parameter mean = 0 lower 0 upper 0
	    }
	    if {[if_parameter_exists sd]} {
		parameter sd = 0 lower 0 upper 0
	    }
	} else {
	    foreach tr $ts {
		if {[if_parameter_exists mean($tr)]} {
		    parameter mean($tr) = 0 lower 0 upper 0
		}
		if {[if_parameter_exists sd($tr)]} {
		    parameter sd($tr) = 0 lower 0 upper 0
		}
	    }
	}
	return ""
    }
#
# Save initial model
# Add default omega if not already defined
#
    set definitions [string tolower [define names]]
    save model $savename
    if {"omega = Use_polygenic_to_set_standard_model_parameterization" \
	    == [omega]} {
	polymod
    }
#
# Adjust parameters and set options
#
    maximize -sampledata -q
    set tindex 0
    set suffix ""
    foreach tr $ts {
	incr tindex
	if {$tindex > 2} {
	    puts "Warning.  Only first 2 traits are zscored."
	    break
	}
	if {-1 != [lsearch -exact $definitions [string tolower $tr]]} {
	    load model $savename
	    zscore -off
	    error \
		"Error!  Trait $ts is a definition.  Zscore has been deactivated."
	}
	set meansd [stats trait$tindex -file $samplefile -q \
			-out $ztemp]
	exec cat $ztemp >>$zfile
	file delete $ztemp
	set mean [stats_get $meansd mean]
	set sd [stats_get $meansd sd]
	set discrete [stats_get $meansd discrete]
	if {$discrete && [option enablediscrete]} {
	    load model $savename
	    zscore -off
	    error \
		"Error!  Trait $ts is discrete.  Zscore has been deactivated.\n"
	}
	if {[llength $ts] > 1} {
	    set suffix "($tr)"
	}
	if {[if_parameter_exists mean$suffix]} {
	    parameter mean$suffix = 0 lower 0 upper 0
	}
	if {[if_parameter_exists sd$suffix]} {
	    parameter sd$suffix = 0 lower 0 upper 0
	}
	load model $savename
	option zmean$tindex $mean
	option zsd$tindex $sd
	set zscored 1
	save model $savename
	if {!$quiet && [info level] < 2} {
	    puts "Trait: $tr   Mean: [format %.8g $mean]   SD: [format %.8g $sd]"
	}
    }

#    if {![if_global_exists SOLAR_zscore_warning] && [info level] < 2 && !$quiet} {
#	global SOLAR_zscore_warning
#	set SOLAR_zscore_warning 1
#	puts "Warning.  mean and sd set only when zscore command is given."
#   }

    if {$zscored} {
	option zscore 1
    }
    file delete $samplefile
    file delete $savename
    file delete [full_filename solar.out]
    file delete [full_filename last.mod]
    return ""
}

# solar::screencov --
#
# Purpose:  Perform polygenic analysis with covariate screening
#             Same as 'polygenic -screen'
#
# solar::sporadic --
# solar::polygenic --
#
# Purpose:  Perform polygenic, sporadic, and/or household analysis
#             Calculate H2r, significance of H2r, and proportion of variance
#               contributed by covariates.
#             Optionally performs covariate screening (determine significance
#               level of each covariate).
#
# Usage:   polygenic [-screen] [-all] [-p | -prob <p>] [-fix <covar>]
#                    [-testcovar <covar>] [-testrhoe] [-testrhog] [-testrhoc]
#                    [-sporadic] [-keephouse] [-testrhop] [-rhopse]
#
#          (screencov is an alias for 'polygenic -screen')
#          (sporadic is an alias for 'polygenic -sporadic')
#
#          Typically before giving this command, you will give trait,
#          covariate, and house (if applicable) commands.  You will also load
#          pedigree and phenotypes files if they have not already been loaded.
#
#              solar> load pedigree ped
#              solar> load phenotypes phen
#              solar> trait hbp
#              solar> covariate age sex age*sex smoke
#              solar> polygenic -screen
#
#          Alternatively, you may use the "automodel" command first to
#          include all available phenotypes as covariates.  See note 2
#          below and "help automodel".
#
#          -screen   (or -s)  Perform covariate screening:
#                    Calculate significance level for each covariate, and run
#                    only the significant covariates in the final analysis.
#                    An inclusive significance threshold of 0.1 is used,
#                    but may be changed with the -prob option.  Covariates
#                    may be locked in regardless of significance with the
#                    -fix or -all options.
#
#          (An alternative method of covariate analysis using bayesian
#           model averaging is available with the command:
#               bayesavg -covariates)
#
#          -p        (or -prob)  p is the probability level for keeping
#                    covariates as "significant."  The default is 0.1.
#                    It is set to be generous so that covariates are not
#                    removed unnecessarily.  (The probability levels for
#                    H2r and C2 are fixed at 0.05, however, H2r is never
#                    removed from the final model even if it judged to
#                    be not significant, and C2 is only removed from the
#                    model if it is zero in the final model and therefore
#                    has no effect at all.)
#
#          -fix      (or -f) "fix" (lock in) this particular covariate
#                    regardless of significance level.  NOTE: a -fix or -f
#                    qualifier is required for each covariate to be fixed,
#                    for example:  -f age -f sex
#
#          -all      (or -a) Keep all covariates in final anaysis regardless
#                    of significance level.
#
#          -testcovar <covar>  Test the probability of this covariate only.
#                     All other covariates are fixed and unscreened.  This
#                     argument is incompatible with -screen (screen all
#                     covariates).  The tested covariate is not removed from
#                     final model regardless of probability.  For -testcovar,
#                     the default probability level for declared
#                     "significance" is 0.05 (which can be changed with -p
#                     option).  Also, the reported proportion of variance
#                     is for the tested covariate only.
#
#          -testrhoe  (Bivariate only)  Test significance of rhoe difference
#                     from 0 by running model where rhoe is constrained to 0.
#                     The p value is shown in the same line as the RhoE value.
#
#          -testrhog  (Bivariate only)  Test significance of rhog differences
#                     from zero and from 1 (if positive) or -1 (if negative).
#                     Because there may be two p values, they are shown
#                     in line(s) below the RhoG result and standard error.
#
#          -testrhoc  (Bivariate Household only) Test significance of rhoc
#                     differences from zero and 1 (if positive) and -1 (if
#                     negative).  Because there may be two p values, they are
#                     shown in line(s) below the RhoC result and std. error.
#
#          -testrhop  (Bivariate polygenic only) Test significance of derived
#                     estimate of phenotypic correlation differences
#                     (difference from 0).
#
#          -rhopse     (-testrhop must be specified also) Get standard error
#                      of rhop, saved in model file rhop.mod and variable
#                      SOLAR_RhoP_SE
#
#          -sporadic  Only evaluate sporadic models, not polygenic.
#
#          -keephouse Keep "household effect" C2 parameter in final model
#                     even if it maximizes to zero in the best polygenic
#                     (or sporadic) model.
#
#          -residinor After maximizing final sporadic model (after covariate
#                     testing, if that is done), residualize the final model
#                     and inormalize the residual trait.  (Warning!  The
#                     phenotypes file loaded at the end of analysis will be
#                     the residual phenotypes file.)
#
# Notes:    (1) Output is written to directory selected by 'outdir' command,
#           or, if none is selected, to a directory named by the trait.  This
#           is called the "maximization output directory."  Polygenic results
#           are in file named polygenic.out.  Important loglikelihoods and
#           statistical computations are recorded in polygenic.out.logs.  If
#           the -sporadic option is selected, the files are sporadic.out and
#           sporadic.out.logs.  For univariate models, the residuals are
#           computed and written to a file named polygenic.residuals (or
#           sporadic.residuals), then the statistics of those residuals
#           are written to a file named polygenic.residuals.stats (or
#           sporadic.residuals.stats).  If the residual kurtosis is
#           above 0.8, you get a special warning (see note 5 below).  You
#           also get a special warning if the trait standard deviation is
#           below 0.5, which is undesireable for numerical reasons.
#
#           (2) Prior to running polygenic, you should set up the trait and
#           covariates.  You may use the trait and covariate commands, or
#           use the "automodel" command. "automodel" selects all variables
#           otherwise unaccounted for in the phenotypes file as candidate
#           covariates, and also sex and the standard interactions with
#           sex and age.  (If you are unfamiliar with "automodel" it would
#           be a good idea to examine the covariates afterwards with the
#           covariates command...)
#
#           (3) If household effect (see "house") is in effect when the
#           polygenic command is given, it will be included in the analysis.
#           If the household parameter C2 is 0 in the household polygenic
#           model, it will be removed from the final model regardless of
#           whether "covariate screening" is performed, unless -keephouse
#           is specified.  The p value for C2 will be computed (if C2 is
#           nonzero), but the p value will not cause C2 to be removed from
#           the final model. The p value of the C2 parameters is not
#           computed for bivariate models.
#
#           (4) If any covariates have been constrained by the user,
#           certain tests are not allowed: the determination of total
#           variance due to covariates, or the Leibler-Kullback R
#           squared (done for discrete traits).  Also, such covariates
#           are not included in the "screening" if the screening option
#           is selected.
#
#           (5) If you get the message about Residual Kurtosis being too high
#           because it is above 0.8, there is danger of LOD scores  being
#           estimated too high in a subsequent linkage analysis.  You should
#           start over using either tdist or lodadj or inormal (see 
#           documentation) to protect against this.  If you are already
#           using tdist or lodadj, you may ignore this warning, but it would
#           be fair to report both the Residual Kurtosis and the method
#           you are using to deal with it.  We most strongly recommend
#           inormal, which in conjunction with the define command creates
#           an inverse normalized transformation of your trait(s).
#
#           If there are no covariates, the Kurtosis is computed from the
#           trait itself, and no "residuals" are computed.  The same warning
#           threshold applies.  We define Kurtosis as 0 for a standard
#           normal distribution; 3 has already been subtracted from the
#           normalized 4th central moment.
#
#           (6) The polygenic command only supports our "standard"
#           parameterizations.  If you would like to use the esd,gsd,qsd
#           parameterization, use the polygsd command (see "help polygsd"
#           for more information) instead.
#
#           (7) For bivariate polygenic models only, a derived estimate of
#           RhoP, the phenotypic correlation, is displayed on terminal
#           and written to polygenic.out.  This estimate is computed from the
#           h2r's, rhog, and rhoe according to the following formula:
#
#               sqrt(h2r(ti))*sqrt(h2r(tj))*rhog + 
#                   sqrt(1-h2r(ti))*sqrt(1-h2r(tj))*rhoe
#
#           To determine the significance of RhoP by comparing models with
#           a rhop parameter and a rhop parameter constrained to zero, use
#           the -testrhop option.  Additional models rhop.mod and rhop0.mod
#           are written to the output directory.
#
#           (8) The polygenic command creates global variables which may
#           be accessed later (which is often useful in scripts).  The
#           variables are:
#
#               SOLAR_Individuals  number of individuals included in sample
#               SOLAR_H2r_P        p value for h2r
#               SOLAR_Kurtosis     residual trait kurtosis
#               SOLAR_Covlist_P    list of p values for covariates
#               SOLAR_Covlist_Chi  list of chi values for covariates
#               SOLAR_RhoP         derived estimate of phenotypic correlation
#                                    for bivariate polygenic models, {} if
#                                    not calculated
#               SOLAR_RhoP_P       -testrhop sets this to p value of rhop
#                                    being nonzero
#               SOLAR_RhoP_SE      -rhopse sets this to se value of rhop
#               SOLAR_RhoP_OK      -testrhop sets this if likelihood of rhop
#                                    parameterized model matches polygenic,
#                                    as it should
#
#           The covariate lists are created only if the -screen option
#           is used.  All screened variables are included, regardless of
#           whether they were retained in the final model.  Before you
#           can access any of these variables in a script, you must
#           use a "global" command.  For example:
#
#               global SOLAR_Kurtosis
#               if {$SOLAR_Kurtosis > 4} {puts "Very bad kurtosis!"}
#
#           (9) The default is for the standard error option to be turned
#           on (and temporarily off, when desireable for certain tests).
#           However, if you turn the standard error option off before
#           starting polygenic, it will remain off.
#           
# -

proc screencov args {
    return [eval polygenic -s $args]
}

proc sporadic args {
    return [eval polygenic -sporadic $args]
}

proc polygenic args {

    global SOLAR_H2r_P
    global SOLAR_Kurtosis
    global SOLAR_Covlist_P
    global SOLAR_Covlist_Chi
    global SOLAR_Individuals
    global SOLAR_RhoP
    global SOLAR_RhoP_P
    global SOLAR_RhoP_OK
    global SOLAR_RhoP_SE
    set SOLAR_Kurtosis 0	
    set SOLAR_Covlist_P {}
    set SOLAR_Covlist_Chi {}
    set SOLAR_Individuals ""
    set SOLAR_RhoP {}
    set SOLAR_RhoP_P {}
    set SOLAR_RhoP_OK 0
    set SOLAR_RhoP_SE 0
    set covbase 20  ;# index to covar info in image output

    global SOLAR_old_phenotypes_files
    set SOLAR_old_phenotypes_files [phenotypes -files]

    set qu -q
    ifverbplus set qu ""

    set imoutvalid [imout -valid]
    if {$imoutvalid} {imout -puts -1 -vol 0}

# Remember current standerr status

    set prestanderr [option standerr]

# Examine current covariates...

    set some_covariates_removed 0
    set found_constrained_covariates 0
    set active_covar_list [covariates -active]
    set covar_list $active_covar_list

    if {{} == $active_covar_list} {
	set any_covariates_exist 0
    } else {
	set any_covariates_exist 1

# Check for covariates that are constrained to some value
# Remove them from covar_list, we don't mess with them

	set betalist [covariate -betanames]
	set const_covar_list ""
	foreach beta $betalist {
	    if {![catch {find_simple_constraint $beta}] || \
		    ![catch {find_simple_constraint <$beta>}]} {
		if {!$found_constrained_covariates} {
		    puts " "
		}
		puts "    *** Note: $beta is constrained\n"
		set found_constrained_covariates 1
		lappend const_covar_list [string range $beta 1 end]
	    }
	}
	foreach const $const_covar_list {
	    catch {set covar_list [remove_from_list $covar_list $const]}
	}
    }
	    
    set final_covar_list $covar_list
    set number_of_covariates [llength $covar_list]
    if {0<$number_of_covariates} {
	set free_covariates_exist 1
    } else {
	set free_covariates_exist 0
    }

# Set defaults and read arguments

    set covscreen 0
    set testcovar ""
    set fixall 0
    set test_zero_c2 0
    set testrhoe 0
    set testrhog 0
    set testrhoc 0
    set sporadic 0
    set keephouse 0
    set do_kullback 0
    set got_kullback 0
    set do_var_due2cov 0
    set fix_list {}
    set probability_level 0.1
    set user_probability_level -1
    set vc_probability_level 0.05
    set testrhop 0
    set rhopse 0
    set residinor 0

    set extra_args [read_arglist $args \
	    -screen {set covscreen 1} -s {set covscreen 1} \
	    -prob user_probability_level -p user_probability_level \
	    -fix {lappend fix_list VALUE} \
	    -f {lappend fix_list VALUE} \
	    -residinor {set residinor 1} \
            -keephouse {set keephouse 1} \
	    -testrhoe {set testrhoe 1} \
	    -testrhog {set testrhog 1} \
	    -testrhoc {set testrhoc 1} \
	    -testrhop {set testrhop 1} \
	    -sporadic {set sporadic 1} \
	    -rhopse {set rhopse 1} \
	    -testcovar testcovar \
	    -all {set fixall 1}]

# check arguments for consistency

    if {$covscreen!=0 && $testcovar!=""} {
	error "polygenic -screen and -testcovar are incompatible"
    }
# OK, now that we've determined user isn't using those inconsistently,
# we use testcovar as a special case of covscreen
    if {$testcovar != ""} {
	set covscreen 1
	set fixall 1
	set probability_level 0.05
	if {-1 == [lsearch $active_covar_list $testcovar]} {
# covariate not present try adding it
	    covariate $testcovar
	} elseif {-1 != [lsearch $const_covar_list $testcovar]} {
	    error "covariate $testcovar must be unconstrained first"
	}
	set covar_list $testcovar
	set final_covar_list $testcovar
	set number_of_covariates 1
	set free_covariates_exist 1
    }

    if {$user_probability_level != -1} {
	set probability_level $user_probability_level
    }

    ensure_float $probability_level
    if {$probability_level > 1 || $probability_level < 0} {
	error "Invalid probability level $probability_level"
    }
    if {[llength $extra_args]} {
	error "Invalid argument(s): $extra_args"
    }
    set ihouse [check_house]
    set hmatrix ""
    if {$ihouse} {
	set hmatrix [housematrix]
    }

    set ts [trait]
    set nts [llength $ts]
    set multi 0
    if {$nts > 1} {set multi 1}
    if {$nts != 2} {
	if {$testrhog || $testrhoe || $testrhoc} {
      error "polygenic -testrhoe -testrhog -testrhoc not possible\nexcept in bivariate (2 trait) models"
	}
    }
    if {$nts > 1} {
	if {$covscreen} {
	    error "Covariate screening or testing not supported for multivariate models\nScreen univariate models first"
	}
	if {$sporadic && $testrhog} {
	    error "rhog test not available for household (sporadic) models"
	}
    }
#
# Set up summary output files in output directory
#
    if {$sporadic} {
	set basename sporadic
	purge_sporadic_output_directory
    } else {
	set basename polygenic
	purge_polygenic_output_directory
    }
    set results_filename [full_filename $basename.out]
    set init_results [open $results_filename w]
    puts $init_results "The last run of $basename did not run to completion."
    puts $init_results "Check logs file, or individual fisher output files."
    close $init_results

    set logs_file [full_filename $basename.logs.out]
    putsnew $logs_file

    if {$free_covariates_exist && $covscreen} {


puts "**********************************************************************"
puts "*  (Screening)  Get starting beta values using sporadic type model   *"
puts "*  with diagonal covariance matrices (default for sporadic models)   *"
puts "**********************************************************************"

# Check that all fixed covariates are listed

    foreach fixed $fix_list {
	if {-1 == [lsearch $active_covar_list $fixed]} {
	    error "Fixed covar $fixed not present in model"
	}
    }
    spormod
    option standerr 0
    eval maximize $qu -o s0
    model save [full_filename s0]
    set bll [set sll [loglike]]
    putsa $logs_file \
"    *** Loglikelihood of sporadic model with all covars is $sll\n"

    if {$sporadic} {
	set basemodel s0
    } else {
        set basemodel p0

puts ""
puts "**********************************************************************"
puts "*  (Screening)  Maximize polygenic model with all covariates         *"
puts "**********************************************************************"

        polymod
        eval maximize $qu -o p0
        model save [full_filename p0]
        set bll [set pll [loglike]]
        putsa $logs_file \
"    *** Loglikelihood of polygenic model with all covars is $pll\n"

    }
    if {$ihouse} {
        set basemodel h0
puts " "
puts "**********************************************************************"
puts "*  (Screening)  Maximize household model with all covariates         *"
puts "**********************************************************************"

        load model [full_filename s0]
        house
        if {$hmatrix!=""} {eval $hmatrix}
        eval maximize $qu -o h0
        model save [full_filename h0]
        set bll [set pll [loglike]]
        putsa $logs_file \
"    *** Loglikelihood of household model with all covars is $pll"
        putsa $logs_file " "

        if {!$sporadic} {
	    set basemodel hp0
puts " "
puts "**********************************************************************"
puts "*  (Screening)  Maximize household polygenic model                   *"
puts "*  with all covariates                                               *"
puts "**********************************************************************"

            load model [full_filename p0]
            house
            if {$hmatrix!=""} {eval $hmatrix}
            eval maximize $qu -o hp0
            model save [full_filename hp0]
            set bll [set pll [loglike]]
            putsa $logs_file \
"    *** Loglikelihood of household polygenic model with all covars is:\n\t\t\t\t $pll"
            putsa $logs_file " "
        }
    }

puts " "
puts "**********************************************************************"
    if {$ihouse && !$sporadic} {
puts "*  (Screening)  Maximize household polygenic models                  *"
    } elseif {$ihouse} {
puts "*  (Screening)  Maximize household models                            *"
    } elseif {$sporadic} {
puts "*  (Screening)  Maximize sporadic models                             *"
    } else {
puts "*  (Screening)  Maximize polygenic models                            *"
    }
puts "*  one with each covariate deactivated                               *"
puts "**********************************************************************"

    set final_covar_list {}
    set report_list {}
    set remove_covar_list {}

    set covindex -1
    foreach covar $covar_list {
        incr covindex
        puts " "
	puts "    *** Testing covariate $covar by suspending it ***"
	model load [full_filename $basemodel]
	covariate suspend $covar
        option standerr 0
	eval maximize $qu -o no$covar
	model save [full_filename no$covar]
	set chill [expr 2.0 * ($bll - [loglike])]
	set deg 1
	set testchi [catch {set pstring [chi $chill $deg]}]
	if {$testchi != 0} {set pstring "p = 1.0"}
	lappend SOLAR_Covlist_P [lindex $pstring 2]
	set keep 1    
	set comment "(Significant)"
	if {[lindex $pstring 2] >= $probability_level} {
	    set comment "(Not Significant)"
	    if {-1 != [lsearch $fix_list $covar] || $fixall} {
		set comment "(Not Sig., but fixed)"
	    } else {
		set keep 0
	    }
	}
	if {$keep == 1} {
	    lappend final_covar_list $covar
	} else {
	    lappend remove_covar_list $covar
	}

	set ll [loglike]
        catch {[set ll [format "%.6f" $ll]]}
	putsat $logs_file \
"\n    *** Loglikelihood w/o covar $covar is $ll"

        catch {[set chill [format %.4f $chill]]}
        putsat $logs_file "    *** chi = $chill, deg = $deg"
	lappend SOLAR_Covlist_Chi $chill
        if {0==[string compare [verbosity] "verbosity max"]} {
	    putsat $logs_file "    *** Total process memory is [memory]"
	}

	set report "$pstring  $comment"
	lappend report_list $report
        putsat $logs_file "    *** $report"

        if {$imoutvalid} {
           imout -puts [lindex $pstring 2] -vol [expr $covbase+2+($covindex*4)]
           imout -puts $chill -vol [expr $covbase+3+($covindex * 4)]
        }
    }

puts " "
    set nfinal [llength $final_covar_list]
    model load [full_filename s0]
    if {!$fixall} {
	foreach covar $remove_covar_list {
	    set some_covariates_removed 1
	    covariate delete $covar
	}
    }
}
# end if $covscreen && $free_covariates_exist

# The following is done regardless of whether we are doing covariate screening

puts "**********************************************************************"
    if {$covscreen && $free_covariates_exist} {
puts "*  Covariate screening completed                                     *"
puts "*  Now using models with only significant or fixed covariates        *"
    }
puts "*  Maximize sporadic model                                           *"
puts "**********************************************************************"
putsat $logs_file ""

    option standerr $prestanderr
    spormod
    eval maximize $qu -o spor
    if {[option modeltype] == "evd2"} {
	set tstats [stats -return -q]
	set discrete_trait [stats_get $tstats discrete]
    } else {	
	if {[trait -is-trait-discrete?]} {
	    set discrete_trait 1
	} else {
	    set discrete_trait 0
	}
    }
    set as_quantitative 0
if {{} != [find_string [full_filename spor.out] "quantitative!"]} {
	set as_quantitative 1
    }
    model save [full_filename spor]
    set spor_loglike [loglike]
    putsat $logs_file \
"    *** Loglikelihood of sporadic model is $spor_loglike"
    foreach t $ts {
	if {!$multi} {
	    set suffix ""
	} else {
	    set suffix \($t\)
	}
	set sd_covar [parameter SD$suffix =]
	set sd_noh_covar $sd_covar
    }
    set best_nh_loglike [loglike]
    set best_loglike [loglike]
    set finalmodels spor
    set bestmodel spor
    set best_nh_model spor
    set bestdesc sporadic
    set best_nh_desc sporadic

    if {$residinor} {
putsat $logs_file ""
puts "**********************************************************************"
puts "*  Residualize and Inormalize trait                                  *"
puts "**********************************************************************"
puts ""
	set residinorname [full_filename $basename.residinor]
	residual spor.out -out $residinorname
	load phenotypes $residinorname
	model new
	trait residual
    }	
    if {!$sporadic} {    
putsat $logs_file ""
puts "**********************************************************************"
puts "*  Maximize polygenic model                                          *"
puts "**********************************************************************"
puts ""

        polymod
        eval maximize $qu -o poly
        model save [full_filename poly]
        set poly_loglike [loglike]
        putsat $logs_file \
"    *** Loglikelihood of polygenic model is $poly_loglike"


        foreach t $ts {
	    if {!$multi} {
		set suffix ""
	    } else {
		set suffix \($t\)
	    }
	    set h2r [parameter h2r$suffix =]
	    catch {[set h2r [format %.7f $h2r]]}
	    putsat $logs_file \
"    *** H2r$suffix in polygenic model is $h2r"
            set sd_covar [parameter SD$suffix =]
            set sd_noh_covar $sd_covar
        }
	set best_nh_loglike [loglike]
	set best_loglike [loglike]
	set finalmodels "poly, $finalmodels"
	set bestmodel poly
        set best_nh_model poly
	set bestdesc polygenic
	set best_nh_desc polygenic
    }

    set needhouse 0
    if {$ihouse} {
putsat $logs_file ""
puts "**********************************************************************"
puts "*  Maximize household model                                          *"
puts "**********************************************************************"
puts ""
        load model [full_filename spor]
        house
        if {$hmatrix!=""} {eval $hmatrix}
        eval maximize $qu -o house
        set house_loglike [loglike]
        putsat $logs_file \
"    *** Loglikelihood of household model is $house_loglike"
        model save [full_filename house]
        set c1 0
        set c2 0
        set tno 0
        foreach tr $ts {
	    incr tno
	    set suf ""
	    if {$multi} {
		set suf \($tr\)
	    }
	    set house_c2 [parameter c2$suf =]
	    catch {[set house_c2 [format %.7f $house_c2]]}
	    putsat $logs_file \
"    *** C2$suf in household model is $house_c2"
            set c$tno $house_c2
            set sd_covar [parameter SD$suf =]
        }
	set best_loglike [loglike]
	set bestmodel house
        set bestdesc household
        set finalmodels "house, $finalmodels"

        if {!$sporadic} {
putsat $logs_file ""
puts "**********************************************************************"
puts "*  Maximize household polygenic model                                *"
puts "**********************************************************************"
puts ""
            load model [full_filename poly]
            house
            if {$hmatrix!=""} {eval $hmatrix}
            eval maximize $qu -o housepoly
            model save [full_filename housepoly]
            set housepoly_loglike [loglike]
            putsat $logs_file \
"    *** Loglikelihood of household polygenic model is $housepoly_loglike"
            set c1 0
            set c2 0
            set tno 0
            foreach tr $ts {
	        set suf ""
	        if {$multi} {
		    set suf \($tr\)
	        }
	        set h2r [parameter h2r$suf =]
	        catch {[set h2r [format %.7f $h2r]]}
	        putsat $logs_file "    *** H2r$suf in household polygenic model is $h2r"
		set sd_covar [parameter SD$suf =]
	    }
	    putsat $logs_file " "
	    foreach tr $ts {
		incr tno
		set suf ""
		if {$multi} {
		    set suf \($tr\)
		}
		set housepoly_c2 [parameter c2$suf =]
		catch {[set housepoly_c2_form [format %.7f $housepoly_c2]]}
		putsat $logs_file \
"    *** C2$suf in household polygenic model is $housepoly_c2_form"
                set c$tno $housepoly_c2
            }
	    set best_loglike [loglike]
	    set bestmodel housepoly
	    set bestdesc "household polygenic"
	    set finalmodels "housepoly, $finalmodels"
	}

# The following is for testing purposes...replicating a hard to
# reproduce condition for bivariate household models...bivariate
# models with this troublesome condition usually don't even converge

	if {[if_global_exists SOLAR_test_zero_c2]} {
	    puts "c1 is $c1, c2 is $c2, now setting to 0"
	    set c1 [set c2 0]
	}

# determine retention and significance of C2

	if {$c1 == 0 && $c2 == 0 && !$keephouse} {
	    putsat $logs_file \
"\n    *** Removing C2 because it is 0.0 in $bestdesc model"
            set bestmodel $best_nh_model
            set bestdesc $best_nh_desc
            set sd_covar $sd_noh_covar

# Note: this simply removes the constrained-to-zero C2 from the previously
# maximized polygenic or sporadic model.  It should not need re-maximization.

            load model [full_filename $bestmodel]
            set best_loglike [loglike]
            house -delete 
            save model [full_filename $bestmodel]

        } elseif {$c1 != 0 || $c2 != 0} {
	    if {!$multi} {
		putsat $logs_file "\n    *** Determining significance of C2"
		putsat $logs_file \
"    *** Comparing $bestdesc and $best_nh_desc models"
                set chill [expr 2.0 * ($best_loglike - $best_nh_loglike)]
                set testchi [catch {set chiip [chi -number $chill 1]}]
                if {$testchi != 0} {set chiip 1.0}
		set hpspr [expr $chiip / 2.0]
# Must reformat because we operated on it
                set hpspsign =
                set hpsp [format %.7f $hpspr]
		if {[format %.5f $hpsp] == 0} {
		    set hpsp [format %.8g $hpspr]
		}
		catch {[set chill [format %.4f $chill]]}
		putsat $logs_file \
"    *** chi = $chill, deg = 1, p $hpspsign $hpsp"
            } else {
		putsat $logs_file \
"\n    *** SOLAR does not compute the significance of C2 for multivariate models"
            }
            set needhouse 1
	}
    }
#
# Figure chi (loglikelihood) for either poly/spor or housepoly/house pair
# to calculate significance of h2r (if neither sporadic nor bivariate)
#
    if {!$sporadic && !$multi} {
	if {$needhouse} {
	    set chill [expr 2.0 * ($housepoly_loglike - $house_loglike)]
	    set bestpair "household polygenic and household"
	} else {
	    set chill [expr 2.0 * ($poly_loglike - $spor_loglike)]
	    set bestpair "polygenic and sporadic"
	}
#
# Since in a sporadic model H2 is fixed to a boundary
# (analogous to single ended), we must divide p for H2 by 2
#
	set testchi [catch {set chiip [chi -number $chill 1]}]
	if {$testchi != 0} {set chiip 1.0}
	set SOLAR_H2r_P [expr $chiip / 2.0]
# Must reformat because we operated on it
	set pspsign =
	set pspf [format "%.7f" $SOLAR_H2r_P]
	if {$pspf == 0} {
	    set pspf [format %.8g $SOLAR_H2r_P]
	}
	set SOLAR_H2r_P $pspf

	catch {[set chill [format %.4f $chill]]}
	putsat $logs_file "\n    *** Determining significance of H2r"
	putsat $logs_file "    *** Comparing $bestpair models"
	putsat $logs_file "    *** chi = $chill, deg = 1, p $pspsign $SOLAR_H2r_P"
    }

    if {!$multi && !$found_constrained_covariates} {
	if {$free_covariates_exist && 0!=[llength $final_covar_list]} {
            if {$discrete_trait && !$as_quantitative} {
		set do_kullback 1
	    } elseif {[catch {find_simple_constraint sd}]} {
		set do_var_due2cov 1   ;# Only if SD is *NOT* constrained
	    }
        }
    }

    if {!$residinor && ($do_kullback || $do_var_due2cov)} {
puts ""
puts "**********************************************************************"
puts "*  Maximize $bestdesc model with NO covariates"
	if {$do_var_due2cov} {
puts "*  to determine proportion of variance due to covariates"
        } else {
puts "*  to compute Kullback-Leibler R-squared"
        }
puts "**********************************************************************"

        set finalmodels "$finalmodels, nocovar"
	eval covariate suspend $final_covar_list
	eval maximize $qu -o nocovar
	model save [full_filename nocovar]
        if {$do_var_due2cov} {
	    set sd_nocovar [parameter SD start]
	    set prop_var [expr 1.0 - (($sd_covar * $sd_covar) / \
		  ($sd_nocovar * $sd_nocovar))]
	    catch {[set prop_var [format %.7f $prop_var]]}
	    catch {[set sd_covar [format %.7f $sd_covar]]}
	    catch {[set sd_nocovar [format %.7f $sd_nocovar]]}
	    putsat $logs_file \
"\n    *** Trait SD in model with covariates is $sd_covar"
            putsat $logs_file \
"    *** Trait SD in model without covariates is $sd_nocovar"
            putsat $logs_file \
"    *** Proportion of variance explained by covariates is $prop_var"
            putsa $logs_file " "
        } else {
	    putsat $logs_file \
"\n    *** Loglikelihood of model with no covariates is [loglike]"
	    if {![catch {set kullback [expr 1.0 - \
		  ($best_loglike/[loglike])]}]} {
	        set got_kullback 1
	        if {$kullback < 0.0} {
		    set kullback 0.0
		}
		putsat $logs_file \
	      "    *** Kullback-Leibler R-squared is [format %.7f $kullback]"
		putsa $logs_file " "
	    }

	}
	model load [full_filename $bestmodel]
    }
#
# Compute derived estimate of RhoP (Phenotypic correlation)
#
    if {!$sporadic && $nts == 2 && !$ihouse} {
	set rph2r(1) [parameter h2r\([lindex $ts 0]\) = ]
        set rph2r(2) [parameter h2r\([lindex $ts 1]\) = ]
        set rpre [parameter rhoe =]
        set rprg [parameter rhog =]

        set SOLAR_RhoP [expr sqrt($rph2r(1))*sqrt($rph2r(2))*$rprg + \
    sqrt(1-$rph2r(1))*sqrt(1-$rph2r(2))*$rpre]
#
# Test significance of RhoP if requested
#
        if {$testrhop} {
puts ""
puts "**********************************************************************"
puts "*  Maximize models with estimated rhop and rhop constrained to zero  *"
puts "*  to determine significance of rhop being different from zero       *"
puts "**********************************************************************"
puts ""
            set plike [loglike]
            parameter delete rhoe
	    parameter rhop = $SOLAR_RhoP lower -0.9 upper 0.9
            omega = <sd(ti)>*<sd(tj)>*( phi2*sqrt(<h2r(ti)>)*sqrt(<h2r(tj)>)*(tne*rhog + teq) + I*sqrt(<e2(ti)>)*sqrt(<e2(tj)>) * (teq + tne* ((rhop - sqrt(<h2r(ti)>)*sqrt(<h2r(tj)>)*rhog) / (sqrt(<e2(ti)>)*sqrt(<e2(tj)>)) )))
            option standerr 0
	    if {$rhopse} {
		option standerr 1
	    }
	    eval maximize $qu -o rhop
            save model [full_filename rhop]
	    set rp [parameter rhop =]
	    set rhop_ll [loglike]
            putsat $logs_file \
"    *** Loglikelihood of model rhop with estimated rhop is $rhop_ll"
            puts "\n    *** Warning.  Next model might take awhile."
            puts "    *** Run using 'verbosity max' to see maximization details.\n"
	    if {$rhopse} {
		set SOLAR_RhoP_SE [parameter rhop se]
		option standerr 0
	    }
	    parameter rhop = 0
	    constraint rhop = 0
	    eval maximize $qu -o rhop0
            save model [full_filename rhop0]
	    set rhop0_ll [loglike]
            putsat $logs_file \
"    *** Loglikelihood of model rhop0 with rhop constrained to 0 is $rhop0_ll"
            set rhop_test [expr 2*($rhop_ll - $rhop0_ll)]
            if {$rhop_test < 0} {
	        set rhop_test 0
	    }   
 	    set SOLAR_RhoP_P [chi -number $rhop_test 1]
            putsat $logs_file \
"    *** chi = $rhop_test, deg = 1, p = $SOLAR_RhoP_P"
#rhop-parameterized model should have same likelihood as polygenic...
	    if {$plike == $rhop_ll} {
	        set SOLAR_RhoP_OK 1
	    } else {
		putsat $logs_file \
"    *** Warning.  Loglikehood of RhoP parameterized model doesn't match polygenic"
	    }
            model load [full_filename $bestmodel]
        }
    }
    
#
# Determine significance of rhoe, rhog, and/or rhoc when asked
#
    set p_rhoe0 ""
    set p_rhog0 ""
    set p_rhog1 ""
    set p_rhogn1 ""
    set p_rhoc0 ""
    set p_rhoc1 ""
    set p_rhocn1 ""

    if {$testrhoe && [parameter rhoe =] != 0} {
puts ""
puts "**********************************************************************"
puts "*  Maximize model with rhoe constrained to zero                      *"
puts "*  to determine significance of rhoe being different from zero       *"
puts "**********************************************************************"
puts ""
	global SOLAR_constraint_tol
	parameter rhoe = 0 lower -$SOLAR_constraint_tol upper $SOLAR_constraint_tol
	constraint rhoe = 0
	eval maximize $qu -o rhoe0
	set no_rhoe_ll [loglike]
	putsat $logs_file \
"    *** Loglikelihood of model with rhoe constrained to 0 is $no_rhoe_ll"
	set chill [expr 2.0 * ($best_loglike - $no_rhoe_ll)]
	if {$chill < 0} {
	    set p_rhoe0 "p = 1"
	} else {
	    catch {set p_rhoe0 [chi $chill 1]}
        }
	putsat $logs_file \
"    *** chi = $chill, deg = 1, $p_rhoe0"
	load model [full_filename $bestmodel]
    }

    if {$testrhog} {
	set rhog [parameter rhog =]
	if {$rhog != 0} {
puts ""
puts "**********************************************************************"
puts "*  Maximize model with rhog constrained to zero                      *"
puts "*  to determine significance of rhog being different from zero       *"
puts "**********************************************************************"
puts ""
	    parameter rhog = 0 lower -0.01 upper 0.01
	    constraint rhog = 0
	    eval maximize $qu -o p_rhog0
	    set no_rhog_ll [loglike]
	    putsat $logs_file \
"    *** Loglikelihood of model with rhog constrained to 0 is $no_rhog_ll"
	    set chill [expr 2.0 * ($best_loglike - $no_rhog_ll)]
            if {$chill < 0} {
	        set p_rhog0 "p = 1"
	    } else {
	        catch {set p_rhog0 [chi $chill 1]}
	    }
	    putsat $logs_file \
"    *** chi = $chill, deg = 1, $p_rhog0"
	    load model [full_filename $bestmodel]
	}
	if {$rhog >= 0 && $rhog != 1} {
puts ""
puts "**********************************************************************"
puts "*  Maximize model with rhog constrained to 1.0                       *"
puts "*  to determine significance of rhog being different from 1.0        *"
puts "**********************************************************************"
puts ""
	    parameter rhog = 1 lower 0.99 upper 1.01
	    constraint rhog = 1
	    eval maximize $qu -o p_rhog1
	    set rhog_1_ll [loglike]
	    putsat $logs_file \
"    *** Loglikelihood of model with rhog constrained to 1.0 is $rhog_1_ll"
	    set chill [expr 2.0*($best_loglike - $rhog_1_ll)]
	    if {$chill < 0} {
	        set p_rhog1 "p = 1"
	    } else {
	        catch {
		    set p_rhog1 [chi -number $chill 1]
	            set p_rhog1 [expr $p_rhog1 / 2.0]
	            set f [format %.7f $p_rhog1]
	            if {$f == 0} {
		        set f [format %.8g $p_rhog1]
	            }
		    set p_rhog1 "p = $f"
	        }
	    }
	    putsat $logs_file \
"    *** chi = $chill, deg = 1, $p_rhog1"
	    load model [full_filename $bestmodel]
	}
	if {$rhog <= 0 && $rhog != -1} {
puts ""
puts "**********************************************************************"
puts "*  Maximize model with rhog constrained to -1.0                      *"
puts "*  to determine significance of rhog being different from -1.0       *"
puts "**********************************************************************"
puts ""
	    parameter rhog = -1 lower -1.01 upper -0.99
	    constraint rhog = -1
	    eval maximize $qu -o rhog-1
	    set rhog_n1_ll [loglike]
	    putsat $logs_file \
"    *** Loglikelihood of model with rhog constrained to -1.0 is $rhog_n1_ll"
	    set chill [expr 2.0*($best_loglike - $rhog_n1_ll)]
	    if {$chill < 0} {
	        set p_rhogn1 "p = 1"
	    } else {
	        catch {
		    set p_rhogn1 [chi -number $chill 1]
		    set p_rhogn1 [expr $p_rhogn1 / 2.0]
		    set f [format %.7f $p_rhogn1]
		    if {$f == 0} {
		        set f [format %.8g $p_rhogn1]
		    }
		    set p_rhogn1 "p = $f"
	        }
	    }
	    putsat $logs_file \
"    *** chi = $chill, deg = 1, $p_rhogn1"
	    load model [full_filename $bestmodel]
	}
    }	        
    if {$testrhoc && $needhouse} {
	set rhoc [parameter rhoc =]
	if {$rhoc != 0} {
puts ""
puts "**********************************************************************"
puts "*  Maximize model with rhoc constrained to zero                      *"
puts "*  to determine significance of rhoc being different from zero       *"
puts "**********************************************************************"
puts ""
	    parameter rhoc = 0 lower -0.01 upper 0.01
	    constraint rhoc = 0
	    eval maximize $qu -o p_rhoc0
	    set no_rhoc_ll [loglike]
	    putsat $logs_file \
"    *** Loglikelihood of model with rhoc constrained to 0 is $no_rhoc_ll"
	    set chill [expr 2.0 * ($best_loglike - $no_rhoc_ll)]
            if {$chill < 0} {
	        set p_rhoc0 "p = 1"
	    } else {
	        catch {set p_rhoc0 [chi $chill 1]}
	    }
	    putsat $logs_file \
"    *** chi = $chill, deg = 1, $p_rhoc0"
	    load model [full_filename $bestmodel]
	}
	if {$rhoc >= 0 && $rhoc != 1} {
puts ""
puts "**********************************************************************"
puts "*  Maximize model with rhoc constrained to 1.0                       *"
puts "*  to determine significance of rhoc being different from 1.0        *"
puts "**********************************************************************"
puts ""
	    parameter rhoc = 1 lower 0.99 upper 1.01
	    constraint rhoc = 1
	    eval maximize $qu -o p_rhoc1
	    set rhoc_1_ll [loglike]
	    putsat $logs_file \
"    *** Loglikelihood of model with rhoc constrained to 1.0 is $rhoc_1_ll"
	    set chill [expr 2.0*($best_loglike - $rhoc_1_ll)]
	    if {$chill < 0} {
	        set p_rhoc1 "p = 1"
	    } else {
	        catch {
		    set p_rhoc1 [chi -number $chill 1]
	            set p_rhoc1 [expr $p_rhoc1 / 2.0]
	            set f [format %.7f $p_rhoc1]
	            if {$f == 0} {
		        set f [format %.8g $p_rhoc1]
	            }
		    set p_rhoc1 "p = $f"
	        }
	    }
	    putsat $logs_file \
"    *** chi = $chill, deg = 1, $p_rhoc1"
	    load model [full_filename $bestmodel]
	}
	if {$rhoc <= 0 && $rhoc != -1} {
puts ""
puts "**********************************************************************"
puts "*  Maximize model with rhoc constrained to -1.0                      *"
puts "*  to determine significance of rhoc being different from -1.0       *"
puts "**********************************************************************"
puts ""
	    parameter rhoc = -1 lower -1.01 upper -0.99
	    constraint rhoc = -1
	    eval maximize $qu -o rhoc-1
	    set rhoc_n1_ll [loglike]
	    putsat $logs_file \
"    *** Loglikelihood of model with rhoc constrained to -1.0 is $rhoc_n1_ll"
	    set chill [expr 2.0*($best_loglike - $rhoc_n1_ll)]
	    if {$chill < 0} {
	        set p_rhocn1 "p = 1"
	    } else {
	        catch {
		    set p_rhocn1 [chi -number $chill 1]
		    set p_rhocn1 [expr $p_rhocn1 / 2.0]
		    set f [format %.7f $p_rhocn1]
		    if {$f == 0} {
		        set f [format %.8g $p_rhocn1]
		    }
		    set p_rhocn1 "p = $f"
	        }
	    }
	    putsat $logs_file \
"    *** chi = $chill, deg = 1, $p_rhocn1"
	    load model [full_filename $bestmodel]
	}
    }
puts ""
puts "******************************************************************************"
puts "*                          Summary of Results                                *"
puts "******************************************************************************"
puts ""
    if {!$sporadic} {
	model save [full_filename null0]
    }
    set results [open $results_filename w]
    set outputdir [full_filename {}]
#
# Report data files used
#
    putstee $results "\tPedigree:    [topline pedigree.info]"
    set phenfilelist [phenotypes -files]
    putstee $results "\tPhenotypes:  $phenfilelist"
    if {[llength $phenfilelist] > 1} {
        putstee $results " "
    }
#
# Find # of individuals
#
    set ind_string [find_string [full_filename $bestmodel.out] \
	    "The sample size including probands used is"]
    if {1 != [scan $ind_string " The sample size including probands used is %d" SOLAR_Individuals]} {
	set SOLAR_Individuals ""
    }
    if {$imoutvalid} {
        imout -puts $SOLAR_Individuals -vol 11
    }

    putstee $results \
"\tTrait:       [format %-20s [trait]]  Individuals:  $SOLAR_Individuals\n"



catch {
    set constrained_to_1 0
    if {![catch {set cval [find_simple_constraint sd]}] && $cval == 1} {
	set constrained_to_1 1
    }
    if {(!$discrete_trait || $as_quantitative) && $constrained_to_1} {
	    putstee $results \
"\tWARNING!  YOU HAVE PARAMETER SD CONSTRAINED TO 1.0 !"
	    putstee $results \
"\tThis is probably not what you intended."
	    putstee $results \
"\tYou need to use command \"model new\" before changing from discrete to"
	    putstee $results \
"\tquantitative trait analysis.\n"
    }
    if {$as_quantitative} {
        putstee $results \
"\tWarning!  You are analyzing a discrete trait as quantitative!"
	putstee $results \
"\tSee \"help discrete-notes\" for discussion.\n"
    }
}
#
# Warn if SD below 0.5
#
    if {$multi} {
	foreach tr $ts {
	    if {0.5 > [parameter sd($tr) =]} {
		putstee $results \
"\tWARNING!  Estimated Standard Deviation for $tr is [format %.5f [parameter sd($tr) =]]"
		putstee $results "\tWhen Trait SD is below 0.5, results are sometimes incorrect"
		if {0 < [parameter sd($tr) =]} {
	            set sd_factor [format %.1f [expr 1.0 / [parameter sd($tr) =]]]
	            putstee $results "\tMultiplying trait by a factor such as $sd_factor is recommended.\n"
		}
	    }
	}
    } else {
	if {0.5 > [parameter sd =]} {
       	    putstee $results "\tWARNING!  Estimated Trait Standard Deviation is [format %.5f [parameter sd =]]"
	    putstee $results "\tWhen Trait SD is below 0.5, results are sometimes incorrect"
	    if {0 < [parameter sd =]} {
	        set sd_factor [format %.1f [expr 1.0 / [parameter sd =]]]
	        putstee $results "\tMultiplying trait by a factor such as $sd_factor is recommended"
	    }
	    putstee $results ""
	}
        if {$imoutvalid} {
	    imout -puts [parameter sd =] -vol 7
            imout -puts [parameter sd se] -vol 8
        }
    }
#
# Results for h2r, rhoe, and rhog
#
    set high_h2r 0
    if {$multi} {
	if {!$sporadic} {
	set comment ""
	foreach tr $ts {
	    set h2r [parameter h2r($tr) =]
            if {$h2r > 0.9} {
                set high_h2r 1
            }
            catch {[set h2r [format %.7f $h2r]]}
            putstee $results "\t\t\t H2r($tr) is $h2r  $comment"
            putsa $logs_file "    H2r is $h2r  $comment"

            set h2rse [parameter h2r($tr) se]
            if {0 != $h2rse} {
	        catch {set h2rse [format %.7f $h2rse]}
	        putstee $results "\t       H2r($tr) Std. Error:  $h2rse\n"
	        putsa $logs_file "\t H2r($tr) Std. Error:  $h2rse"
            } else {
	        putstee $results ""
            }
	}
        if {$high_h2r} {
    	    putstee $results "\tWarning.  Unexpectedly high heritabilities might result from"
	    putstee $results "\tnumerical problems, especially if mztwins are present."
	    putstee $results ""
	}
    }
#
# RhoE and RhoG for bivariate models
#
        set pars [parameter -names]
        foreach par $pars {
	    if {"rhoe" == [string range $par 0 3]} {

		set rhoe [parameter $par =]
                catch {set rhoe [format %.7f $rhoe]}
                set rhoese [parameter $par se]
                catch {set rhoese [format %.7f $rhoese]}
                if {0 == $rhoese} {
		    set rhoese "Not Computable"
		}    
	        putstee $results \
	            "\t\t\t [rhocap $par] is $rhoe  $p_rhoe0"
	        putsa $logs_file \
	            "    [rhocap $par] is $rhoe  $p_rhoe0"

                if {$prestanderr} {
		    putstee $results \
	            "\t       [rhocap $par] Std. Error:  $rhoese\n"
		    putsa $logs_file \
	            "    [rhocap $par] Std. Error:  $rhoese\n"
		}
	    }
        }
	if {!$sporadic} {
	foreach par $pars {
	    if {"rhog" == [string range $par 0 3]} {	

                set rhog [parameter $par =]
                catch {set rhog [format %.7f $rhog]}
                set rhogse [parameter $par se]
                catch {set rhogse [format %.7f $rhogse]}
                if {0 == $rhogse} {set rhogse "Not Computable"}
    
	        putstee $results \
	            "\t\t\t [rhocap $par] is $rhog"
	        putsa $logs_file \
	            "    [rhocap $par] is $rhog"

                if {$prestanderr} {
		    putstee $results \
	            "\t       [rhocap $par] Std. Error:  $rhogse\n"
		    putsa $logs_file \
	            "    [rhocap $par] Std. Error: $rhogse\n"
		}
            }
        }

        if {"" != [catenate $p_rhog0 $p_rhog1 $p_rhogn1]} {
            putstee $results ""
            putsa $logs_file ""
        }

	if {"" != $p_rhog0} {
	    putstee $results \
		"\t       RhoG different from zero  $p_rhog0"
	    putsa $logs_file \
		"    RhoG different from zero  $p_rhog0"
	}
	if {"" != $p_rhog1} {
	    putstee $results \
		"\t       RhoG different from 1.0   $p_rhog1"
	    putsa $logs_file \
		"    RhoG different from 1.0   $p_rhog1"
	}
	if {"" != $p_rhogn1} {
	    putstee $results \
		"\t       RhoG different from -1.0  $p_rhogn1"
	    putsa $logs_file \
		"    RhoG different from -1.0  $p_rhogn1"
	}
 #
# Phenotypic correlation
#
        if {[llength $SOLAR_RhoP]} {
	    putstee $results \
"\t       Derived Estimate of RhoP is [format %.7f $SOLAR_RhoP]"
            if {[llength $SOLAR_RhoP_P]} {
		putstee $results \
"\t       RhoP different from zero  p = $SOLAR_RhoP_P"
		if {!$SOLAR_RhoP_OK} {
		    putstee $results \
"\t        Warning.  Loglikelihood of RhoP parameterized model didn't match polygenic"
		}
	    
            }
        }
        putstee $results ""
        putsa $logs_file ""
        }
    } elseif {!$sporadic} {

# Univariate Polygenic

	if {$SOLAR_H2r_P <= $vc_probability_level} {
            set comment "(Significant)"
        } else {
            set comment "(Not Significant)"
        }
        if {$SOLAR_H2r_P != 1.0} {set comment "p $pspsign $SOLAR_H2r_P  $comment"}
	set h2r [parameter h2r =]
        catch {[set h2r [format %.7f $h2r]]}
        putstee $results "\t\t\t H2r is $h2r  $comment"
        putsa $logs_file "    H2r is $h2r  $comment"

        set h2rse [parameter h2r se]
        if {$imoutvalid} {
	    imout -puts $h2r -vol 1
            imout -puts $h2rse -vol 2
	    imout -puts $SOLAR_H2r_P -vol 3
	    imout -puts [loglike] -vol 4
            if {[if_parameter_exists mean]} {
                imout -puts [parameter mean =] -vol 9
                imout -puts [parameter mean se] -vol 10
            }
	}
        if {0 != $h2rse} {
	    catch {set h2rse [format %.7f $h2rse]}
	    putstee $results "\t       H2r Std. Error:  $h2rse\n"
	    putsa $logs_file "\t H2r Std. Error:  $h2rse"
        } else {
	    putstee $results ""
        }
	if {$h2r > 0.9} {
	    putstee $results "\tWarning.  Unexpectedly high heritabilities might result from"
	    putstee $results "\tnumerical problems, especially if mztwins are present."
	    putstee $results ""
	}
    }
#
# Results for household effects
#
    if {$needhouse} {
	if {$multi} {
	    foreach tr $ts {
		set pname C2\($tr\)
		set c2 [parameter $pname =]
		catch {[set c2 [format %.7f $c2]]}
		putstee $results "\t\t\t  $pname is $c2"
		putsa $logs_file "     $pname is $c2"
		set c2se [parameter $pname se]
		if {0 != $c2se} {
		    catch {set c2se [format %.7f $c2se]}
		    putstee $results "\t        $pname Std. Error:  $c2se\n"
		    putsa $logs_file "     $pname Std. Error:  $c2se"
		}
	    }
            set pars [parameter -names]
            foreach par $pars {
                if {"rhoc" == [string range $par 0 3]} {
	            set rhoc [parameter $par =]
	            catch {set rhoc [format %.7f $rhoc]}
	            putstee $results "\t\t\t  [rhocap $par] is $rhoc"
		    putsa $logs_file "     [rhocap $par] is $rhoc"
		    set rhocse [parameter $par se]
		    if {0 != $rhocse} {
			catch {set rhocse [format %.7f $rhocse]}
			putstee $results "\t        [rhocap $par] Std. Error:  $rhocse\n"
			putsa $logs_file "     [rhocap $par] Std. Error:  $rhocse"
		    }
		}
            }
            if {"" != $p_rhoc0} {
                putstee $results \
                "\t       RhoC different from zero  $p_rhoc0"
                putsa $logs_file \
		"    RhoC different from zero  $p_rhoc0"
            }
	    if {"" != $p_rhoc1} {
	        putstee $results \
                "\t       RhoC different from 1.0   $p_rhoc1"
		putsa $logs_file \
		"    RhoC different from 1.0   $p_rhoc1"
	    }
	    if {"" != $p_rhocn1} {
	        putstee $results \
		"\t       RhoC different from -1.0  $p_rhocn1"
		putsa $logs_file \
		"    RhoC different from -1.0  $p_rhocn1"
	    }
	} else {
	    if {$hpsp < $vc_probability_level} {
		set comment "(Significant)"
		set further_comment ""
	    } else {
		set comment "(Not Significant)"
		set further_comment "     (C2 retained because nonzero)"
	    }
	    if {$hpsp != 1.0} {set comment "p $hpspsign $hpsp  $comment"}
	    set c2 [parameter c2 =]
	    catch {[set c2 [format %.7f $c2]]}
	    putstee $results "\t\t\t  C2 is $c2  $comment"
	    putsa $logs_file "     C2 is $c2  $comment"
	    set c2se [parameter c2 se]
	    if {0 != $c2se} {
		catch {set c2se [format %.7f $c2se]}
		putstee $results "\t        C2 Std. Error:  $c2se $further_comment\n"
		putsa $logs_file "\t  C2 Std. Error:  $c2se $further_comment"
	    } else {
		putstee $results ""
	    }
	}
    } elseif {$ihouse} {
	foreach tr $ts {
	    set pname C2
	    if {$multi} {
		set pname C2\($tr\)
	    }
	    putstee $results "\t\t\t  $pname is 0.0000000"
	    putsa $logs_file "    $pname is 0.0000000"
	}
	if {!$keephouse} {
	    if {$multi} {
		putstee $results "\n\tSince they were zero, all C2 parameters have been deleted."
	    } else {
		putstee $results "\n\tSince it was zero, the C2 parameter has been deleted."
	    }
	    putstee $results "\tTo keep C2 parameters even when they are all zero,"
	    putstee $results "\tuse the -keephouse option."
        } else {
	    putstee $results "\n\tC2 parameter(s) retained because of -keephouse option."
	}
    }
#
# Results for each covariate
#
    if {$covscreen} {
	for {set i 0} {$i < $number_of_covariates} {incr i} {
	    set covar [format "%40s" [lindex $covar_list $i]]
	    set report [lindex $report_list $i]
	    putstee $results " $covar  $report"
	}
    }
    if {$imoutvalid} {
	set i 0
        foreach covar $covar_list {
	    if {[if_parameter_exists b$covar]} {
               imout -puts [parameter b$covar =] -vol [expr $covbase+0+($i*4)]
               imout -puts [parameter b$covar se] -vol [expr $covbase+1+($i*4)]
            } else {
                imout -puts 0 -vol [expr $covbase + 0 + ($i * 4)]
                imout -puts 0 -vol [expr $covbase + 1 + ($i * 4)]
            }
            incr i
        }
    }


#
# Pedigree-household group merging
#
    if {$needhouse} {
	putstee $results ""
	if {"" != [set merge_string [find_string \
		[full_filename $bestmodel.out] "pedigrees merged into" ]]} {
	    set merge_string [trim_left $merge_string]
	    putstee $results "\t$merge_string"
	} elseif {"" != [set merge_string [find_string \
		[full_filename $bestmodel.out] "pedigrees shared house" ]]} {
	    set merge_string [trim_left $merge_string]
	    putstee $results "\t$merge_string"
	} elseif {1==[option MergeAllPeds]} {
	    putstee $results "\tAll pedigrees merged into one pedigree-household group"
	} elseif {0==[option MergeHousePeds]} {
	    putstee $results "\tPedigrees not merged into household groups (option MergeHousePeds 0)"
	}
    }
#
# Proportion of variance
#

    if {$some_covariates_removed} {
	putstee $results "\n\tThe following covariates were removed from final models:"
	putstee $results "\t$remove_covar_list"
	putsa $logs_file "\n\tThe following covariates were removed from final models:"
	putsa $logs_file "\t$remove_covar_list"

    }
    if {$any_covariates_exist && !$multi} {
	if {$found_constrained_covariates} {
	    putstee $results \
"\n\tSome covariates are constrained.  This prevents computation of"
            putstee $results \
"\tvariance due to all covariates or Kullback-Leibler R-squared."
        } elseif {0==[llength $final_covar_list]} {
	    putstee $results "\n\tNo covariates were included in the final model"
        } elseif {$residinor} {
	    putstee $results "\n\tProportion of Variance due to Covariates not computed due to -residinor"
        } elseif {$do_var_due2cov} {
	    if {$testcovar == ""} {
		putstee $results \
		    "\n\tProportion of Variance Due to All Final Covariates Is"
	    } else {
		putstee $results \
		    "\n\tProportion of Variance Due to Tested Covariate Is"
	    }
	    if {$prop_var < 0} {
		putstee $results "\t\t\[Not Estimable due to Instability\]"
		putstee $results \
		"\t\t\[Some covariate effects may be relatively small\]"
		putstee $results \
		"\t\t\[Consider running fewer covariates or 'bayesavg -cov'\]"
	    } else {
		putstee $results [format "\t\t\t\t  %s" $prop_var]
                if {$imoutvalid} {
		    imout -puts $prop_var -vol 5
	        }
            }
	} elseif {$got_kullback} {
	    putstee $results \
		    "\n\tKullback-Leibler R-squared is [format %.7f $kullback]"
	} elseif {!$do_kullback && ![catch {find_simple_constraint sd}]} {
	    putstee $results \
		    "\n\tCannot determine variance due to covariates because SD is constrained"
	}
    }
#
# Notes
#
    set outputdir [full_filename {}]
    puts "\n\tOutput files and models are in directory $outputdir"
    putsa $logs_file "\n"
    putsat $logs_file "\tSummary results are in $results_filename"
    puts $results ""
    set rlogsfile $logs_file
    if {[string length [full_filename ""]] > 18} {
        set rlogsfile "\n\t  $rlogsfile"
    }
    putstee $results "\tLoglikelihoods and chi's are in $rlogsfile"
    set and_null0 ""
    if {!$sporadic} {
	set and_null0 " and null0"
	catch {file copy -force $outputdir$bestmodel.out [catenate $outputdir null0.out]}
    }
    puts $results "\tBest model is named $bestmodel$and_null0"
    puts "\tBest model is named $bestmodel$and_null0 (currently loaded)"
    putstee $results "\tFinal models are named $finalmodels"
    if {$covscreen && $free_covariates_exist} {
	if {$sporadic} {
	  puts $results "\tInitial sporadic model is s0"
	} else {
	  puts $results "\tInitial sporadic and polygenic models are s0 and p0"
	}
	if {$ihouse} {
	    if {$sporadic} {
		puts $results "\tInitial household model is h0"
	    } else {
		puts $results "\tInitial household and household polygenic models are h0 and hp0"
	    }
	}
	puts "\tConstrained covariate models are named no<covariate name>"
	puts $results "\tConstrained covariate models are named no<covariate name>"
    }
    close $results

# For univariate, do special residual test.

if {!$multi && !$discrete_trait && !$residinor} {
    set did_residual 0
    catch {
	if {"" == $final_covar_list} {
	    if {-1 != [lsearch -exact [string tolower [define names]] \
		       [string tolower [trait]]]} {
		maximize -q -sampledata
		set st [stats -q trait1 -file [full_filename sampledata.out] -return -out [full_filename $basename.residuals.stats]]
	    } else {
		set st [stats -q -return -out [full_filename $basename.residuals.stats]]
	    }
	} else {
	    residual $bestmodel.out -q -out [full_filename $basename.residuals] -fewest
	    set st [stats residual -q -return -file [full_filename $basename.residuals] -out [full_filename $basename.residuals.stats]]
	}
    set kurt [stats_get $st kurtosis]
    catch {set kurt [format %.4f $kurt]}
    set SOLAR_Kurtosis $kurt
    if {$kurt > 0.8} {
	putsat $results_filename "\n\tWarning!  Residual Kurtosis is $kurt which is too high."
	set kurt_bad 1
	if {[if_parameter_exists t_param]} {
	    putsat $results_filename "\tSince you are using tdist, it is probably OK."
	    set kurt_bad 0
	}
	if {$kurt_bad} {
	    putsat $results_filename "\tSee note 5 in \"help polygenic\"."
	}
    } else {
	putsat $results_filename "\n\tResidual Kurtosis is $kurt, within normal range"
    }
    if {$imoutvalid} {
	imout -puts $kurt -vol 6
    }
    set did_residual 1
    }
    if {!$did_residual} {
	putsat $results_filename "\n\tWarning:  An error occurred while computing residual kurtosis"
	putsat $results_filename "\tTry running residual command to see what happened"
    }
}
if {$residinor} {
    puts "\n\tWarning: Residual phenotypes file loaded."
    puts "\tTo restore original: load phenotypes $SOLAR_old_phenotypes_files"
    puts "\tOr use command restore_phen\n"
}
    if {$imoutvalid} {imout -puts 1 -vol 0}
    return ""
}

proc restore_phen {} {
    if {[if_global_exists SOLAR_old_phenotypes_files]} {
	global SOLAR_old_phenotypes_files
	eval load phen $SOLAR_old_phenotypes_files
    } else {
	error "restore_phen only works after using phenotypes -residinor"
    }
    return ""
}


proc rhocap {word} {
    return [catenate [string toupper [string index $word 0]] \
		[string range $word 1 2] \
		[string toupper [string index $word 3]] \
		[string range $word 4 end]]
}

# solar::relpairs --
# solar::relatives --
#
# Purpose:  Show relationships of relative pairs included in analysis
#             (having all required variables)
#
# Usage:    relatives [-meanf]
#                      -meanf causes Mean f to be reported
#           relpairs             ;# alias for "relatives -meanf"
#
#
# Notes:    output is returned (displayed) and also written to file named
#           relatives.out in current trait/outdir.
#
#           Uses previously stored null0 model in current trait/outdir.
#           Polygenic command should have been run previously to create
#           null0 model.
# -

proc status_message {message} {
    puts -nonewline [format "%-72s\r" $message]
    flush stdout
}

proc relpairs {args} {
    return [relatives -meanf -full $args]
}

proc relatives {args} {
    return [eval pedigree classes -model $args]
}

# solar::residual --
#
# Purpose:  Compute residuals for a maximized model and phenotypes file
#
# Usage:    residual [solarfile] [-out <residualfile>] [-phen <phenfile>] 
#                    [-fewest] [-needped]
#
#           solarfile       solar maximization output file which is expected
#                             in the current outdir.  The default is null0.out
#                             which is created by the polygenic command.
#                             The default directory is the current outdir,
#                             but you may specify a relative pathname to
#                             the current directory.
#
#                           EVD2 models must have actual model currently
#                           in memory (such as is the case immediately after
#                           running polygenic).
#
#                            If the "define" command is used to define the
#                            names used for trait(s) or covariate(s), there
#                             must be a model with the same rootname in
#                             the same directory as the output file.  The
#                             default is null0.mod.
#
#                            Handling of the "scale" (and "noscale") commands
#                             also requires the presence of the model with
#                             the same rootname in the same directory as the
#                             output file.  If this model is not present,
#                             residual will finish but print a warning if
#                             not inside a script.
#
#           residualfile    new phenotypes file with trait 'residual' (the
#                             default is 'residual.out' written to the working
#                             directory).
#           phenfile        the base phenotypes file; the default is to use
#                             the currently loaded phenotypes file(s).
#           -fewest         Copy fewest fields to residualfile (this would be
#                             ID, FAMID (if required), trait, and residual.
#                             The default is to include the above along with
#                             all other (unused) variables from the phenfile.
#           -needped        Only include individuals in the pedigree file.
#                             (By default, all individuals in the phenotypes
#                              file would be included, unless there is a
#                              covariate including sex which require the
#                              pedigree file.)
#
# Example:
#           solar> automodel pheno atrait
#           solar> polygenic -s
#           solar> residual
#
# MOST IMPORTANT!
#
#           This procedure requires a maximization output file.
#           Unless otherwise specified, the default is assumed to be null0.out
#           which is produced by the "polygenic" command.  If this is not
#           what you want, you need to specify the maximization output file.
#           You cannot specify a model file, that is insufficient.
#
# Additional Notes:
#
#           Univariate quantitative trait only.
#           The trait or outdir must be specified first.
#           Must be run in the directory in which pedigree was loaded.
#           FAMID is included if present in both phenotypes and pedigree files.
#           residualfile will be written in comma delimited format.
#           This procedure does not handle hand-written 'mu' equations, only
#             standard covariates.
#           Not applicable to discrete traits.
#-

proc residual {args} {

# Parse arguments

    set pearson 0
    set fewest 0
    set needped 0
    set quiet 0
    set class 0
    set phenfile ""
    set residualfile residual.out
    set outputfile [set default_outputfile [full_filename null0.out]]
    set orargs [read_arglist $args \
	    -out residualfile \
	    -phen phenfile \
	    -phenotypes phenfile \
            -needped {set needped 1} \
	    -q {set quiet 1} \
	    -class {set class 1} \
	    -fewest {set fewest 1}]
    if {1 == [llength $orargs]} {
	set outputfile $orargs
    } elseif {1 < [llength $orargs]} {
	error \
" Invalid argument(s)\n\
Usage: residual \[solarfile\] \[-out residualfile\] \[-phen phenfile\] \[-fewest\]"
    }

# See if user specified path in filename; if not, add 

    if {-1 == [string first / $outputfile]} {
	set outputfile [full_filename $outputfile]
    }

# See if output file exists

    if {![file exists $outputfile]} {
	if {$default_outputfile == $outputfile} {
	    error "$outputfile not found.\nMaybe you need to specify maximization output file."
	} else {
	    error "residual: File $outputfile not found."
	}
    }

    notquiet puts "Using maximization output file: $outputfile"

# Ensure that this was not a discrete maximization
#
#    if {"" != [find_string $outputfile \
#		   "Using SOLAR Discrete and Mixed Trait Modeling"]} {
#	puts "** Discrete trait, results will be scaled to Pearson Residuals"
#	set pearson 1
#    }

# Open "SOLAR output" file

    global solar_residual_ofile
    set ofile [open $outputfile r]
    set solar_residual_ofile $ofile
    set ofiledir [file dirname $outputfile]

# Scan "output" file to find phenotypes file

    if {"" == $phenfile} {
	set phenfile [phenotypes -files]
	if {[llength $phenfile] > 1} {
	    eval joinfiles -o $ofiledir/join.residual.out $phenfile
	    set phenfile $ofiledir/join.residual.out
	}
	if {"" == $phenfile} {
	    close $solar_residual_ofile
	    error \
              "Can't find Phenotypes filename in SOLAR output file $outputfile"
	}
    }

# Open the other files

    if {[catch {set phenfile [solarfile open $phenfile]}]} {
	close $solar_residual_ofile
	error "Can't find phenotypes file $phenfile"
    }

    if {[catch {set rfile [open $residualfile w]}]} {
	close $solar_residual_ofile
	solarfile $phenfile close
	error "Can't open $residualfile for writing"
    }

    if {[catch {set pedfile [tablefile open pedindex.out]}]} {
	close $ofile
	solarfile $phenfile close
	close $rfile
	error "residual must run from directory in which pedigree was loaded"
    }
	
# The actual body is a subroutine so that we can control
# File opening and closing here

# However, in case trait is defined trait, the name of the "maximization
# output" file is included for possible very constrained use.

    if {[catch {res_body $ofile $phenfile $rfile $pedfile $fewest $needped $outputfile $quiet $pearson $class} \
	    errormsg]} {
	close $solar_residual_ofile
	solarfile $phenfile close
	close $rfile
	tablefile $pedfile close
	error $errormsg
    }
    close $solar_residual_ofile
    solarfile $phenfile close
    close $rfile
    tablefile $pedfile close
    return ""
}

proc res_body {ofile phenfile rfile pedfile fewest needped ofilename \
		   quiet pearson ifclass} {

    set allclasses ""
    global solar_residual_ofile

# Get all user definitions

    set definitions [string tolower [define names]]

# Scan "output" file to find all parameters

    set found_parameter_start 0
    set found_parameter_mean 0

    ifdebug puts "scanning output file"
    while {-1 != [gets $ofile line]} {
	if {-1 != [string first "Final Val" $line]} {
	    gets $ofile
	    set found_parameter_start 1
	    break
	}
    }
    if {!$found_parameter_start} {
	error "Output file does not include parameters"
    }
    set pnames {}
    set pvalues {}
    set pcount 0
    set bcount 0

# For evd2 covariates, we must get actual covariate names from the loaded
# model (not the output file as usual).  Once we have actual covariate
# names, we can handle them as usual

    set rocovars [covariates]
    set ocovars {}
    foreach cov $rocovars {
	if {"Suspended\[" != [string range $cov 0 9] &&
	    "\(\)" != [string range $cov end-1 end]} {
	    lappend ocovars $cov
	}
    }

    while {-1 != [gets $ofile line]} {
	if {2 > [llength $line]} {break}
	incr pcount
	set thispname  [lindex $line 0]
	ifdebug puts "thispname is $thispname"
	if {[string range $thispname 0 5] == "bevd2_"} {
	    set thiscname "[lindex $ocovars $bcount]"
#
# If the beta name has bevd2_ prefix, and if it doesn't simply
# match the covariate name, it must be translated name, so we start
# from original covariate name and add "b" prefix.
# Verify this would be the correct name in reverse
#
	    if {"b$thiscname" != $thispname} {
		set thispname2 "bevd2_"
		for {set i 0} {$i < [string length $thiscname]} {incr i} {
		    set ch [string index $thiscname $i]
		    if {$ch == "*"} {
			set ch X
		    } elseif {$ch == "^"} {
			set ch "up"
		    }
		    set thispname2 "$thispname2$ch"
		}
		if {$thispname != $thispname2} {
		    puts "error translating $thispname to $thispname2"
		    error "EVD2 models must run polygenic before residual"
		}
		set thispname b$thiscname
	    }
	    incr bcount
	}
	lappend pnames  $thispname

	lappend pvalues [lindex $line 1]
	gets $ofile
	gets $ofile
	set testpname [string tolower [lindex $line 0]]
	if {!$ifclass} {
	    if {0==[string compare $testpname mean]} {
		set parameter_mean [lindex $line 1]
		set found_parameter_mean 1
	    }
	} else {
	    if {0==[string compare [string range $testpname 0 5] mean_c]} {
		set classindex [string range $testpname 6 end]
		set parameter_mean($classindex) [lindex $line 1]
		set found_parameter_mean 1
	    }
	}
    }
    if {!$found_parameter_mean} {
	error "residual works only for univariate models with mean parameter"
    }


# Get covariate beta's from list of parameters

    set betacount 0
    set betanames {}
    set betavalues {}
    for {set i 0} {$i < $pcount} {incr i} {
	if {"b" == [string range [lindex $pnames $i] 0 0]} {
	    lappend betanames [lindex $pnames $i]
	    lappend betavalues [lindex $pvalues $i]
	    incr betacount
	}
    }
    ifdebug puts "betanames are $betanames"
    if {$betacount < 1} {
	error "Can't find any covariates"
    }
#
# Parse each covariate name and derive Mu expression from all of them
#   Meanwhile collect list of covariate variables (covariables)

    set expcovar 0
    set checkname ""
    set checknames ""
    if {!$ifclass} {
	set expression ""
    }
    set covariables {}
    for {set i 0} {$i < $betacount} {incr i} {
	set name [string range [lindex $betanames $i] 1 end]
	if {$ifclass} {
	    set upos [string last _c $name]
	    set thisclass [string range $name [expr $upos+2] end]
	    set name [string range $name 0 [expr $upos - 1]]
	}

	set checkname $name

	if {-1 != [string first "(" $name]} {
	    set name [string range $name 0 [expr [string first "(" $name] - 1]]
	    set checkname $name
	}
	set term [lindex $betavalues $i]
#
# Parse multipliers (up to final multiplicand)
#
	while {-1 != [string first "*" $name]} {
	    set name1 [string range $name 0 [expr [string first * $name] - 1]]
	    set checkname $name1
	    set name [string range $name [expr [string first * $name] + 1] end]
#
# Parse exponenential expressions (only name^number is possible)
#
	    if {-1 != [string first ^ $name1]} {
		set number [string range $name1 [expr [string first ^ \
			$name1] + 1] end]
		set name1 [string range $name1 0 [expr [string first ^ \
			$name1] - 1]]
		set checkname $name1
		set aname1 "\${v_$name1}-\${x_$name1}"
		set adj_name1 "pow(${aname1},$number)"
	    } else {
#
# No exponent simple case
#
		set adj_name1 "(\${v_$name1}-\${x_$name1})"
	    }
#
# Append multiplier to term
#
	    set term "$term*$adj_name1"
	    setappend covariables $name1
	    setappend checknames $checkname
	}
#
# Now we're at final multiplier in term...
# Parse exponential expression in final multiplier
#
	if {-1 != [string first ^ $name]} {
	    set number [string range $name [expr [string first ^ \
		    $name] + 1] end]
	    set name [string range $name 0 [expr [string first ^ \
		    $name] - 1]]
	    set aname "\${v_$name}-\${x_$name}"
	    set adj_name "pow($aname,$number)"
	} else {
#
# Non-exponent simple case
#
	    set adj_name "(\${v_$name}-\${x_$name})"
	}

# Append multiplier to term

	setappend checknames $name
	set term "$term*$adj_name"
	setappend covariables $name

# Append term to expression

	if {!$ifclass} {
	    set expression "$expression + $term"
	} else {
	    if {[catch {llength $expression($thisclass)}]} {
		set expression($thisclass) ""
	    }
	    set expression($thisclass)  "$expression($thisclass) + $term"
	    setappend allclasses $thisclass
	}
    }

    if {!$ifclass} {
	set expression [string tolower [string range $expression 3 end]]
    } else {
	foreach class $allclasses {
	    set expression($class) [string tolower [string range $expression($class) 3 end]]
	}
    }
    set covariables [string tolower $covariables]
    notquiet puts "Covariate variables: $covariables"

# Find variable means in output file

    set found_variables {}
    set found_variable_means 0

    if {[option modeltype] == "evd2"} {
	close $ofile
	set ofile [open $ofilename r]
	set solar_residual_ofile $ofile
    }
    while {-1 != [gets $ofile line]} {
	if {-1 != [string first "Descriptive Statistics for the Variables" \
		$line]} {
	    gets $ofile
	    gets $ofile line
	    set found_variable_means 1
	    break
	}
    }
    if {!$found_variable_means} {
	error "Couldn't find variable means in output file"
    }

# Determine which covariates are actually definitions

    set checknames [string tolower $checknames]
    set defcovars ""
     foreach checkname $checknames {
	if {-1 != [lsearch -exact $definitions $checkname]} {
	    setappend defcovars $checkname
	}
    }

    set expcovar 0
    if {0 < [llength $defcovars]} {
	notquiet puts "Defined covariates are $defcovars"
	set expcovar 1
    }

# Read variable means from output file

    set traitname ""
    while {-1 != [gets $ofile line]} {
	set discretevar 0
	if {6 != [llength $line]} {break}
	set varname [string tolower [lindex $line 0]]
	set varname_prefix [string range [string tolower $varname] \
				0 3]

	if {"*" == [string range $varname end end]} {
	    set varname [string range $varname 0 [expr \
			[string length $varname] - 2]]
	    set discretevar 1
	}
	
	if {$varname_prefix == "snp_" || \
		$varname_prefix == "hap_" } {
	    eval global x_$varname
	    eval set x_$varname 0
	} elseif {$discretevar} {
	    eval global x_$varname
	    eval set x_$varname \"[lindex $line 3]\"
	} else {
	    eval global x_$varname
	    eval set x_$varname \"[lindex $line 1]\"
	}
	if {"" == $traitname} {
	    set traitname [string tolower $varname]
	    if {$discretevar} {
		puts "** Discrete trait, results will be scaled to Pearson Residuals"
		set pearson 1
	    }
	}
	lappend found_variables [string tolower $varname]
    }
    

# Fix EVD variable name

    if {-1 < [string first _evd $traitname] && [option modeltype] == "evd2"} {
	regsub -all _evd $traitname "" newpname
	set traitname $newpname
    }

# Report trait name

    notquiet puts "Trait variable:      $traitname"
# If FAMID is present in both pedigree and phenotypes files, we assume it
# is needed

    set need_famid 0
    if {[tablefile $pedfile test_name FAMID]} {
	if {[solarfile $phenfile test_name famid]} {
#	    puts "FAMID is present in both pedigree and phenotypes files"
	    set need_famid 1
	}
    }

    set exptrait 0
    if {-1 == [lsearch -exact $definitions $traitname]} {
	if {!$ifclass} {
	    set expression "\${v_$traitname} - (\$parameter_mean + $expression)"
	} else {
	    foreach class $allclasses {
		set expression($class) "\${v_$traitname} - (\$parameter_mean($class) + $expression($class))"
	    }
	}
    } else {
	set exptrait 1
    }

# If defined trait or covars, read in "sampledata.out" to get actual data
# store in array expdata for retrieval during output loop below

    set modelname [file rootname $ofilename]
    if {$exptrait || $expcovar} {

	save model residual.save
	catch {
	    ifdebug puts "Expression trait...maximizing $modelname"
	    load model $modelname
	    set quietmax -q
	    ifdebug set quietmax ""
	    option modeltype Default
	    eval maximize $quietmax -sampledata
	} errmes
	load model residual.save
	if {"" != $errmes} {
	    error "Unable to get definition data for model $modelname"
	}
	set ofiledir [file dirname $ofilename]
	set wfile [tablefile open [file join $ofiledir sampledata.out]]
	set wdcount 0
	catch {
	    tablefile $wfile start_setup
	    tablefile $wfile setup id
	    set tindex 1
	    if {$need_famid} {
		set tindex 2
		tablefile $wfile setup famid
	    }
	    if {$exptrait} {
		tablefile $wfile setup trait1
	    }
	    foreach dcovar $defcovars {
		tablefile $wfile setup $dcovar
	    }
	    while {{} != [set record [tablefile $wfile get]]} {
		if {$exptrait} {
		set tvalue [format %.12g [lindex $record $tindex]]
		if {[is_float $tvalue] && $tvalue != -.1e21} {
		    if {$need_famid} {
			set expdata([lindex $record 0].[lindex $record 1]) $tvalue
		    } else {
			set expdata([lindex $record 0]) $tvalue
		    }
		}
		}
		set bindex [expr $tindex + $exptrait]
		foreach dvar $defcovars {
		    set vvalue [lindex $record $bindex]
		    set fvalue [format %.12g $vvalue]
		    if {[is_float $fvalue] && $fvalue != -.1e21} {
			if {$need_famid} {
			set cexpdata($dvar.[lindex $record 0].[lindex $record 1]) $fvalue
		    } else {
			set cexpdata($dvar.[lindex $record 0]) $fvalue
		    }
		    incr bindex
		}
		}

		incr wdcount
	    }
	} errmes
	tablefile $wfile close
	ifdebug puts "Read $wdcount records from sampledata.out"
	if {"" != $errmes} {
	    error $errmes
	}
	if {$exptrait} {
	    if {!$ifclass} {
		set expression "\$trait1_expression - (\$parameter_mean + $expression)"
	    } else {
		foreach class $allclasses {
		    set expression($class) "\$trait1_expression - (\$parameter_mean($class) + $expression($class))"
		}
	    }
	}
    }

# Report expression

    if {$pearson} {
	if {!$ifclass} {
	    notquiet puts "RawResidual = $expression"
	} else {
	    foreach class $allclasses {
		notquiet puts "RawResidual\($class\) = $expression($class)"
	    }
	}
    } else {
	if {!$ifclass} {
	    notquiet puts "Residual = $expression"
	} else {
	    foreach class $allclasses {
		notquiet puts "Residual\($class\) = $expression($class)"
	    }
	}	    
    }

# If sex is a covariable, we must get sex from pedigree file

    set need_sex 0
    if {$needped || (-1 != [lsearch $covariables sex])} {
	set sex_id_table {}
	set sex_v_table {}
	set need_sex 1
	set x_sex 1
	tablefile $pedfile start_setup
	tablefile $pedfile setup ID
	if {$need_famid} {
	    tablefile $pedfile setup FAMID
	}
	tablefile $pedfile setup SEX
	while {"" != [set record [tablefile $pedfile get]]} {
	    set sex_id_rec [lrange $record 0 $need_famid]

	    set sexcode [lindex $record end]
	    set sexvalue 0
	    if {$sexcode == "1" || "m" == $sexcode || "M" == $sexcode} {
		set sexvalue 1
	    } elseif {$sexcode == "2" || "f" == $sexcode || "F" == $sexcode} {
		set sexvalue 2
	    }
	    if {$sexvalue != 0} {
		lappend sex_id_table $sex_id_rec
		lappend sex_v_table $sexvalue
	    }
	}
	lappend found_variables sex
    }

# Check modelfile for "scale" commands

    set foundmodel 0
    set scaledvars {}
    set scalevals {}
    catch {
	if {[file exists $modelname.mod]} {
	    set modfile [open $modelname.mod]
	    set foundmodel 1
	    while {-1 != [gets $modfile modline]} {
		if {"scale" == [lindex $modline 0]} {
		    lappend scaledvars [string tolower [lindex $modline 1]]
		    lappend scaledvals [lindex $modline 2]
		}
	    }
	    close $modfile
	}
    }
    if {!$foundmodel} {
	notquiet puts "\nWarning.  Can't find model $modelname.mod"
	notquiet puts "Note: Without model file, cannot adjust for scale commands, if any.\n"
    }
		    

# Check to see we got all variable means we need
# Adjust for scale commands
# Print final values

#    puts "found variables $found_variables"
    foreach variable $covariables {
	set variable [string tolower $variable]
	if {-1 == [lsearch $found_variables $variable]} {
	    error "Didn't find variable $variable in output file"
	}
	set scaleindex [lsearch $scaledvars $variable]
	if {$scaleindex != -1} {
	    set scaleval [lindex $scaledvals $scaleindex]
	    eval set x_$variable $scaleval
	}
	notquiet puts -nonewline "x_$variable = "
	notquiet eval puts \${x_$variable}
    }	

# Write first record of residual file
# and setup output format

    set extra_vars {}
    set id_name [string tolower [solarfile $phenfile establish_name id]]

    if {[catch {set famid_name [string tolower [solarfile $phenfile establish_name famid]]}]} {

# The above should not have failed if famid properly mapped
# So, just force it to "famid" in case improperly mapped, and
# this will exclude it from output file anyway

	set famid_name famid
    }

    if {!$exptrait} {
	set formatend "\${v_$traitname},\$residual"
    } else {
	set formatend "\$trait1_expression,\$residual"
    }
    if {$need_famid} {
	puts -nonewline $rfile "$id_name,$famid_name,$traitname,residual"
	set outformat "\$id,\$famid,$formatend"
	set checklist "$id_name $famid_name $traitname"
    } else {
	puts -nonewline $rfile "$id_name,$traitname,residual"
	set outformat "\$id,$formatend"
	set checklist "$id_name $famid_name $traitname"
    }
    set checklist [string tolower $checklist]
    if {!$fewest} {
	set names [string tolower [solarfile $phenfile names]]
	foreach name $names {
	    if {-1 == [lsearch $checklist $name]} {
		if {-1 == [lsearch $covariables $name]} {
		    puts -nonewline $rfile ",$name"
		    set outformat "$outformat,\${v_$name}"
		    lappend extra_vars $name
		}
	    }
	}
    }
    puts $rfile ""
    ifdebug puts "outformat: $outformat"

# Set up phenfile for required variables
# setup list of variable names

    set variablenames {}
    set num_vars_needed 0
    solarfile $phenfile start_setup
    solarfile $phenfile setup [field id]
    lappend variablenames id
    incr num_vars_needed
    if {$need_famid} {
	solarfile $phenfile setup [field famid]
	lappend variablenames famid
	incr num_vars_needed
    }
    if {$ifclass} {
	solarfile $phenfile setup class
	lappend variablenames class
	incr num_vars_needed
    }
    if {!$exptrait} {
	solarfile $phenfile setup $traitname
	incr num_vars_needed
	lappend variablenames v_$traitname
    }
    foreach covar $covariables {
	if {"sex" == $covar} {continue}
	if {-1 != [lsearch -exact $defcovars $covar]} {continue}
	solarfile $phenfile setup $covar
	incr num_vars_needed
	lappend variablenames v_$covar
    }
    foreach extra_var $extra_vars {
	solarfile $phenfile setup $extra_var
	lappend variablenames v_$extra_var
    }
    set vcount [llength $variablenames]

# Main loop:
# Read phenotypes file, calculate, and output residuals

    set indcount 0
    while {"" != [set record [solarfile $phenfile get]]} {
	set incomplete 0
	for {set i 0} {$i < $vcount} {incr i} {
	    set value [lindex $record $i]
	    if {{} == $value && $i < $num_vars_needed} {
		set incomplete 1
		break
	    }
	    eval set [lindex $variablenames $i] \$value
	}
	if {$incomplete} {continue}
	if {$needped || $need_sex} {
	    set sex_found 0
	    set target [list $id]
	    if {$need_famid} {set target [list $id $famid]}
	    set pindex [lsearch -exact $sex_id_table $target]
	    if {$pindex == -1} {
		set incomplete 1
	    }
	    set v_sex [lindex $sex_v_table $pindex]
	}
	if {$exptrait && $need_famid} {
	    if {[catch {set trait1_expression $expdata($id.$famid)}]} {
		set incomplete 1
	    }
	} elseif {$exptrait} {
	    if {[catch {set trait1_expression $expdata($id)}]} {
		set incomplete 1
	    }
	}
	if {$expcovar} {
	    foreach dvar $defcovars {
		set dok 0
		catch {
		    if {!$need_famid} {
			set v_$dvar $cexpdata($dvar.$id)
		    } else {
			set v_$dvar $cexpdata($dvar.$id.$famid)
		    }
		    set dok 1
		}
		if {!$dok} {
		    set incomplete 1
		}
	    }
	}
	if {!$incomplete} {
	    ifdebug puts "Eval for id $id"
	    if {!$ifclass} {
		set use_exp $expression
	    } else {
		set use_exp $expression($class)
	    }
	    ifdebug puts "expression is $use_exp"
	    set residual [eval expr $use_exp]

	    if {$pearson} {
		set predicted $residual
		set phi [alnorm $predicted t]
		set traitval [eval expr [lindex $use_exp 0]]
		ifdebug puts "pearson traitvalue is $traitval"
		set residual [expr (double($traitval)-$phi)/(sqrt($phi)*sqrt(1-$phi))]
	    }
	    set residual [format %.9f $residual]
	    incr indcount
	    if {[catch {eval puts $rfile \"$outformat\"}]} {
		error "\nIllegal character in phenotype name"
	    }
	}
    }
    notquiet puts "$indcount records written"
}
    

# solar::gridh2r
#
# purpose: grid around the h2r value in polygenic model
#
# usage:   gridh2r [-lower <lower>] [-upper <upper>] [-step <step>]
#
#          -lower <lower>  Lowest h2r; default is current value less 0.1
#          -upper <upper>  Highest h2r; default is current value plus 0.1
#          -step <step>    step; default is 0.01
#
# Notes:   polygenic should be run first.  Only univariate models with
#          only e2,h2r parameters are supported.
#
#          Out is written to a file named gridh2r.out in the maximization
#          output directory.  The starting model is listed first regardless
#          of whether it is in the range specified.
#      
#          After completion, the model with the best loglikelihood will
#          be loaded, but with the h2r constraint (if any) deleted.  This
#          might be the starting model even if it isn't in the specified
#          range.
#
#          Each point requires a maximization, so they might come out
#          slowly.  For full maximization detail, give the "verbosity plus"
#          or "verbosity max" command beforehand.
# -

proc gridh2r {args} {

    file delete [full_filename gridh2r.out]

    set lower [highest 0 [expr [parameter h2r =] - 0.1]]
    set upper [lowest 1 [expr [parameter h2r =] + 0.1]]
    set step 0.01

    set badarg [read_arglist $args -lower lower -upper upper -step step]
    if {"" != $badarg} {
	error "gridh2r: Invalid argument: $badarg"
    }

    puts " "
    putsout gridh2r.out "h2r,loglike"
    putsout gridh2r.out "[parameter h2r =],[loglike]"
    set best_loglike [loglike]
    set best_h2r [parameter h2r =]
    save model [full_filename gridh2r.best]

    for {set test $lower} {$test < $upper} {set test [expr $test + $step]} {

# constraint handles parameter and bounds during maximization

	set h2r [format %.10g $test]
	constraint h2r = $h2r
	parameter e2 = [set e2 [expr 1.0 - $h2r]]
	if {[expr $e2 - 0.01] < [parameter e2 lower]} {
	    parameter e2 lower [expr $e2 - 0.01]
	}
	if {[expr $e2 + 0.01] > [parameter e2 upper]} {
	    parameter e2 upper [expr $e2 + 0.01]
	}

	maximize_quietly solar
	putsout gridh2r.out "$h2r,[loglike]"

	if {![is_nan [loglike]] && [loglike] > $best_loglike} {
	    set best_loglike [format %.10g [loglike]]
	    set best_h2r [parameter h2r =]
	    save model [full_filename gridh2r.best]
	}
    }

    load model [full_filename gridh2r.best]
    catch {constraint delete h2r}
    puts "\nBest loglikelihood is $best_loglike at h2r = $best_h2r (loaded)"
    return ""
}


# solar::grid --
#
# Purpose:  Find the highest likelihood in the vicinity of marker(s)
#
# Usage:    grid <marker1> [<marker2> ...]
#
# Example:  grid APOa D6S2436
#
# Notes:    outdir (or trait) and ibddir must previously be specified.
#           ibd matrices for each marker must already have been computed.
#
#           A model named "null0.mod" is expected in the output directory.
#           That can be created with the polygenic command.
#
#           Summary output is displayed on your terminal and written to a file
#           named grid.out.  An output and model file are saved for each
#           marker with the name <markername>_best_theta.
#
#           The twopoint command also has a "-grid" option which will grid
#           around every marker evaluated.
#
#           A special "-bsearch" option sets point at which a "golden section"
#           search begins.  By default, grid single-steps by 0.01 from 0.00 
#           to 0.05 and then begins a golden section search.  (This is on the
#           assumption that the peak will be reached before 0.05 in the vast
#           majority of cases.)  If you have a significant number of cases
#           above 0.05, you might want to change this, for example:
#
#                grid mrk -bsearch 0.01
#
#           would start the golden section search after 0.01 (which will be
#           faster if the value is expected to be greater than 0.05, but
#           slower if the value is going to be less than 0.05).  Note: 0.01 is
#           the smallest value at which the search can begin.  On the other
#           hand if you wanted to single-step all the way to 0.10, you would
#           give the command:
#
#                grid mrk -bsearch .1
#-

proc grid {args} {

# -twopoint is a private option used by "twopoint -grid"

    set begin_search 0.05
    set search_begun 0
    set golden_mean 0.38197

    set twopoint 0
    set markers [read_arglist $args \
	    -twopoint {set twopoint 1} -bsearch begin_search]

# Check things

    ibddir
    full_filename foo
    if {![file exists [full_filename null0.mod]]} {
	error "Model [full_filename null0] not found.\
\nThis can be created with polygenic command."
    }
    foreach marker $markers {
	if {![file exists [ibddir]/ibd.$marker.gz]} {
	    error "IBD matrix not found for marker $marker"
	}
    }
    if {1 > [llength $markers]} {
	error "No markers specified"
    }

    set h2q_index 1
    set boundary_error_flag 0

    if {$twopoint} {
	set outfilename twopoint.out
	if {1 != [llength $markers]} {
	    error "Only one marker allowed with -twopoint option"
	}
    } else {
	set outfilename grid.out
	outheader grid.out 1 LOD 0
	load model [full_filename null0.mod]
    }

    set best_results {}
    set best_result {}
    set best_theta 0

    proc save_best_result {marker theta result} {
	upvar twopoint twopoint
	upvar best_result best_result
	upvar best_ll best_ll
	upvar best_theta best_theta

	set best_result $result
	set best_ll [loglike]
	if {!$twopoint} {
	    set modelname \
		    [full_filename [catenate $marker _best_theta]]
	    save model $modelname
	    file copy -force [full_filename last.out] $modelname.out
	}
    }

    proc evaluate_theta {marker theta} {
	upvar outfilename outfilename
	upvar h2q_index h2q_index
	upvar boundary_error_flag boundary_error_flag

	mibdt $marker $theta
	linkmod2p [full_filename mibd.out.gz]
	set errmsg [maximize_quietly last]
	if {$errmsg != ""} {
	    puts \
	"    *** Error maximizing marker $marker with Theta = $theta"
	    break
	}
	delete_files_forcibly [full_filename mibd.out.gz]

	set result [outresults $outfilename \
		$marker\([string range $theta 1 end]\) [lodn 0] \
		[loglike] $h2q_index $boundary_error_flag]
	return $result
    }


    foreach marker $markers {

# Must remove mibdchr0.loc before first evaluate_theta for each marker

        delete_files_forcibly [full_filename mibdchr0.loc]

	if {$twopoint} {
	    save_best_result $marker 0.00 {}
	} else {
	    set result [evaluate_theta $marker 0.00]
	    save_best_result $marker 0.00 $result
	}

# Use "golden section" method (heavily weighted toward starting position)
# Using begin_theta, left_theta, right_theta, and end_theta

	set begin_theta 0.00
	set begin_ll [loglike]

	if {$search_begun} {
	    set middle_theta 0.19
	} else {
	    set middle_theta 0.01
	}	    

	set end_theta 0.50
	set end_ll [expr [loglike] - 1e20]  ;# can't be >= 0.50 anyway

# Get the ball rolling by doing middle theta

	set result [evaluate_theta $marker $middle_theta]
	set middle_ll [loglike]


	if {$search_begun || ([loglike] > $best_ll)} {
	    save_best_result $marker $middle_theta $result

	    if {$search_begun} {
		set test_theta 0.12
	    } elseif {$begin_search < .02} {
		set test_theta 0.20
		set search_begun 1
	    } else {
		set test_theta 0.02
	    }

	    while {1} {

# Do test evaluation and save results if best so far

		set result [evaluate_theta $marker $test_theta]
		set test_ll [loglike]
		if {[loglike] > $best_ll} {
		    save_best_result $marker $middle_theta $result
		}

# Decide what to do next...

		if {($middle_theta < $test_theta) == ($test_ll > $middle_ll)} {

# Shift to rightmost triad

		    if {$middle_theta < $test_theta} {
			set begin_theta $middle_theta
			set begin_ll $middle_ll
			set middle_theta $test_theta
			set middle_ll $test_ll
			# end unchanged
		    } else {
			set begin_theta $test_theta
			set begin_ll $test_ll
                        # middle unchanged
			# end unchanged
		    }

		} else {

# Shift to leftmost triad

		    if {$middle_theta < $test_theta} {
			# beginning unchanged
			# middle unchanged
			set end_theta $test_theta
			set end_ll $test_ll
		    } else {
			# beginning unchanged
			set end_theta $middle_theta
			set end_ll $middle_ll
			set middle_theta $test_theta
			set middle_ll $test_ll
		    }
		}

# Pick bigger side for next test point using golden mean
# (If there isn't any room, we are done...)

		set left_size [format %.2f [expr $middle_theta - $begin_theta]]
		set right_size [format %.2f [expr $end_theta - $middle_theta]]

		if {$left_size <= 0.01 && $right_size <= 0.01} {
		    break ;# done
		} elseif {!$search_begun && ($middle_theta < $begin_search)} {
		    set test_theta [format %.2f [expr .01 + $middle_theta]]
		} elseif {$left_size >= $right_size} {
		    set search_begun 1
		    set test_theta [format %.2f [expr $middle_theta - \
			    ($golden_mean * ($middle_theta - $begin_theta))]]
		} else {
		    set search_begun 1
		    set test_theta [format %.2f [expr $middle_theta + \
			    ($golden_mean * ($end_theta - $middle_theta))]]
		}
	    }
	}
	lappend best_results $best_result
    }

    delete_files_forcibly [full_filename mibdchr0.loc] [full_filename mibdchr0.mean] \
               [full_filename mibdchr0.mrg.gz]

    if {!$twopoint} {
	if {1==[llength $markers]} {
	    puts \
"\n                                  Best Result\n"
        } else {
	    puts \
"\n                                  Best Results\n"
        }
	foreach result $best_results {
	    puts $result
	}
    } elseif {$best_theta != 0.00} {
	return $best_result
    }
    return ""
}

#dummy version of mibdt
#proc mibdt {markerfile theta} {
#    set fchar [string first . $markerfile]
#    set nstring [string range $markerfile [expr $fchar + 1] end]
#    set number 0
#    scan $nstring %d number
#    set newnum [expr $number + [format %.0f [expr 100*$theta]]]
#    file copy -force [mibddir]/mibd.2.$newnum.gz [full_filename \
#	    mibdchr0.mrg.gz]
#}


proc mibdt {marker theta} {

# If mibdchr0.loc doesn't exist in trait/outdir, this is the first
# evaluate_theta for this marker, so create the files needed by multipnt

    if {![file exists [full_filename mibdchr0.loc]]} {
        set f [open [full_filename mibdchr0.map] w]
        puts $f 0
        puts $f $marker
        close $f
        exec mrgibd [full_filename mibdchr0.map] [ibddir] [full_filename "."]
        file delete [full_filename mibdchr0.map]
        exec getmeans [full_filename mibdchr0.mrg.gz] \
                      [full_filename mibdchr0.mean] 1
        set f [open [full_filename mibdchr0.loc] w]
        puts $f "NLOCI = 1"
        puts $f "LOCATIONS IN CENTIMORGANS:"
        puts $f [format "%-11s 0   0.0" $marker]
        close $f
    }

# Operate in trait/outdir.  Catch errors to restore original directory.
    
    set startdir [pwd]
    set odir [full_filename "."]
    cd $odir
    set rcode [catch {

# Convert theta to map distance in cM using Haldane mapping function

    set qtloc [expr -50 * log(1 - 2*$theta)]
    set cqtloc [format %.0f $qtloc]
    exec multipnt mibdchr0.loc mibdchr0.mrg.gz mibdchr0.mean $qtloc $cqtloc n
    exec gzip -f mibd.out
    matcrc mibd.out.gz

# Be sure to restore original directory.  Re-raise error if error occurred.

    } errmsg]
    cd $startdir
    if {$rcode == 1} {
	global errorInfo errorCode
	return -code $rcode -errorinfo $errorInfo -errorcode $errorCode $errmsg
    }
    return ""
}


# solar::siminf --
#
# Purpose:  Simulate a fully-informative marker and compute its IBDs
#
# Usage:    siminf -out <markerfile> -ibd <ibdfile>
#
#               -out  Write simulated marker genotypes to this filename.
#                     The default is 'siminf.out' in the current working
#                     directory.
#
#               -ibd  Write marker-specific IBDs for the simulated marker
#                     to this filename.  The default is 'ibd.siminf' in the
#                     current working directory.  The file will be gzipped.
# -

proc siminf {args} {
    set outname siminf.out
    set ibdname ibd.siminf
    set badargs [read_arglist $args -out outname -ibd ibdname]
    if {[llength $badargs]} {
	error "Invalid arguments $badargs"
    }

# If IBD filename already has a .gz extension, get rid of it
    if {![string compare [string range $ibdname \
                          [expr [string length $ibdname] - 3] \
                          [expr [string length $ibdname] - 1]] .gz]} {
        set ibdname [string range $ibdname 0 \
                     [expr [string length $ibdname] - 4]]
    }

    if {[catch {set pedfile [tablefile open pedindex.out]}]} {
	error "Pedigree data have not been loaded."
    }

    set outfile [open $outname w]

    tablefile $pedfile start_setup
    tablefile $pedfile setup IBDID
    tablefile $pedfile setup FIBDID
    tablefile $pedfile setup MIBDID

    while {"" != [set record [tablefile $pedfile get]]} {
        set id [lindex $record 0]
        set fa [lindex $record 1]
        if {$fa == 0} {
            set a1($id) [expr $id * 2 - 1]
            set a2($id) [expr $id * 2]
        } else {
            set mo [lindex $record 2]
            if { [drand] < .5} {
                set a1($id) $a1($fa)
            } else {
                set a1($id) $a2($fa)
            }
            if { [drand] < .5} {
                set a2($id) $a1($mo)
            } else {
                set a2($id) $a2($mo)
            }
        }
        puts $outfile [format "%d %d/%d" $id $a1($id) $a2($id)]
    }

    close $outfile
    tablefile $pedfile close

    ibd -inform $outname $ibdname
}

# solar::twopoint --
#
# Purpose:  Perform "Twopoint" analysis on directory of ibd files
#
# Usage:    twopoint [-append] [-overwrite] [-grid] [-cparm {[<parameter>]*}]
#                    -saveall
#
#           -overwrite  (or -ov) Overwrite existing twopoint.out file.
#
#           -append     (or -a)  Append to existing twopoint.out file.
#
#           -cparm {}     Custom parameters.  Scanning will consist of
#                         replacing one matrix with another matrix, everything
#                         else is unchanged.  The starting model MUST be
#                         a "prototype" linkage model with the desired
#                         parameters, omega, and constraints.  Starting
#                         points and boundaries for the parameters must be
#                         explicitly specified.  Following the -cparm tag,
#                         there must be a list of parameters in curly braces
#                         that you want printed out for each model.  The
#                         list can be empty as is indicated with a pair of
#                         curly braces {}.  There must be a model named null0
#                         in the maximization output directory for LOD
#                         computation purposes.  The matrix to be replaced
#                         must have name ibd or ibd1, ibd2, etc.  The highest
#                         such ibd will be replaced.  If the matrix is loaded
#                         with two "columns," such as "d7," each succeeding
#                         matrix will be loaded with two columns also.
#                         See section 9.4 for an example involving dominance.
#                        
#           -grid  Enables the "grid" option, which estimates recombination
#                  fractions in the range theta=0 to 0.45, finding the
#                  optimal value to the nearest 0.01.  (Note: this option is
#                  not important for most twopoint users.  It also
#                  increases evaluation time considerably.  Consider using
#                  the separate "grid" command with only the markers of
#                  greatest interest.)
#
#           -saveall  Save all twopoint models in the maximization output
#                     directory.  The models are named "ibd.<marker>".
# Notes:
#          The trait or outdir must be specified before running twopoint.
#
#          There must be a null0.mod model in the trait or outdir
#          directory.  This can be created with the polygenic command
#          prior to running multipoint.  (This model may include
#          household and covariate effects.  See the help for the
#          polygenic command for more information.)
#
#          An output file named twopoint.out will be created in the trait
#          or outdir directory.  If that file already exists, the user must
#          choose the -append or -overwrite option.
#
#          The best twopoint model is saved as two.mod in the trait or outdir
#          directory.  It is also loaded in memory at the completion of the
#          twopoint command.
#
#          IBDDIR should be set with the ibddir command prior to running
#          twopoint.
#
#          If models have two traits, the 2df LOD scores will be
#          converted to 1df effective LOD scores, with the assumption
#          that parameter RhoQ1 is not intentionally constrained.
#          To override this, use the lodp command (see).  This feature
#          was first included with beta version 2.0.1.
#              
# -

proc twopoint {args} {

    set gridding 0
    set append 0
    set overwrite 0
    set saveall 0
    set plist \'

    set badargs [read_arglist $args \
	    -append {set append 1} -a {set append 1} \
	    -overwrite {set overwrite 1} -ov {set overwrite 1} \
	    -grid {set gridding 1} \
	    -saveall {set saveall 1} \
	    -cparm plist \
	]

    if {{} != $badargs} {
	error "Invalid argument(s) to twopoint: $badargs"
    }

    if {"\'" != $plist} {
	set noparama "-cparm"
	set aparama 1
    } else {
	set noparama ""
	set aparama 0
	set plist ""
    }
    
    set qu -q
    ifverbplus set qu ""

    full_filename test  ;# ensure trait/outdir specified

    set useibddir [ibddir]

    if {![file exists [full_filename null0.mod]]} {
	error "Model [full_filename null0] not found.\
\nThis can be created with polygenic command."
    }

    set highest_old_lod -10000
    set highest_old_record ""
    set done_list {}

    set twopoint_exists 0
    if {[file exists [full_filename twopoint.out]]} {
	set twopoint_exists 1
	if {!$overwrite && !$append} {
	    error \
         "twopoint.out file already exists.  Use -append or -overwrite option."
	}
    }

    if {$overwrite} {
	delete_files_forcibly [full_filename two.mod]
	delete_files_forcibly [full_filename two.out]
    }

    set appending 0
    set open_option -create
    if {$twopoint_exists && $append} {
	set appending 1
	set open_option -append
	set f [open [full_filename twopoint.out] "r"]
	gets $f line
	gets $f line
	while {-1 < [gets $f line]} {
	    set name [lindex $line 0]
	    lappend done_list $useibddir/ibd.$name.gz
	    set lod [lindex $line 1]
	    if {$lod > $highest_old_lod} {
		set highest_old_lod $lod
		set highest_old_record $line
	    }
	}
	close $f
    }

    set wildcard [format %s/ibd.*.gz $useibddir]
    set file_list [glob -nocomplain $wildcard]
    set file_list_len [llength $file_list]

    if {$file_list_len==0} {
	error "No ibd files found in ibddir:\n ($useibddir/ibd.*.gz)"
    }

    set file_list [lsort -dictionary $file_list]

    set do_file_list {}
    for {set i 0} {$i < $file_list_len} {incr i} {
	set test_file [lindex $file_list $i]
	if {-1 == [lsearch -exact $done_list $test_file]} {
	    lappend do_file_list $test_file
	}
    }

    set do_file_list_len [llength $do_file_list]
    if {0 == $do_file_list_len} {
	error "All ibddir ibd files already processed...see [full_filename twopoint.out]"
    }

    lodadj -query -inform stdout

# Setup resultfile

    set headings "Model LOD Loglike"
    set formats "%19s %11s %12.3f"
    set expressions "\$name \$flod \[loglike\]"
    if {$aparama} {
	foreach par $plist {
	    lappend formats %9.6f
	    lappend headings $par
	    lappend expressions "\[parameter $par =\]"

	}
    } elseif {1 < [llength [trait]]} {
	set ts [trait]
	foreach tr $ts {
	    lappend headings H2r($tr)
	    lappend formats %9.6f
	    lappend expressions "\[parameter H2r($tr) =\]"
	    lappend headings H2q1($tr)
	    lappend formats %9.6f
	    lappend expressions "\[parameter H2q1($tr) =\]"
	}
    } else {
	lappend headings "H2r"
	lappend formats "%9.6f"
	lappend  expressions "\[parameter h2r =\]"
	lappend headings "H2q1"
	lappend formats "%9.6f"
	lappend  expressions "\[parameter h2q1 =\]"
    }
    set resultf [resultfile $open_option [full_filename twopoint.out] \
		     -headings $headings -expressions $expressions \
		     -formats $formats -display]
    if {$appending} {
	puts "[centerline "***  New Twopoint Results  ***" 72]\n"
	resultfile $resultf -header -displayonly
    } else {
	resultfile $resultf -header
    }

    set highest_new_lod -10000
    set highest_new_record ""
    set newrs {}
    global Solar_Fixed_Loci
    set Solar_Fixed_Loci 0
    set h2q_index 1
    for {set i 0} {$i < $do_file_list_len} {incr i} {
	set do_file [lindex $do_file_list $i]

	ifverbplus puts "\n    *** Analyzing new ibd $do_file\n"

        if {!$aparama} {
	    load model [full_filename null0.mod]
	}
	eval linkmod $noparama -2p $do_file
	set highest 0
#
# maximize but catch errors
#  unfortunately, maxtry doesn't catch all errors
#
	if {0 != [catch {set max_status [maxtry 1 $h2q_index $do_file [full_filename temp] 1]}]} {
	    set max_status "Unknown retry error"
	} elseif {$saveall} {
	    save model [full_filename [file tail [file rootname $do_file]]]
	}
	set boundary_error_flag 0
	if {$max_status == "" && ![catch {loglike}]} {
	    set loglik [loglike]
	    set flod [lodn 0]
	    if {$flod > $highest_new_lod} {
		save model [full_filename t]
		file copy -force [full_filename temp.out] [full_filename t.out]
		set highest_new_lod $flod
		set highest 1
	    }
	    set flod [format %.4f $flod]
	} else {
	    puts "error maximizing $do_file: $max_status"
	    if {$max_status != "Convergence Error"} {
		set boundary_error_flag 1
	    }
	    set loglik NaN
	    set flod " "
	}
	file delete [full_filename temp.out]
	set first_char [expr 4 + [string first "ibd." $do_file]]
	set last_char [expr [string last ".gz" $do_file] - 1]
	set name [string range $do_file $first_char $last_char]
	set result [resultfile $resultf -write]
	lappend newrs $result
	set gridresult ""
	if {$gridding} {
	    set gridresult [grid -twopoint $name]
	}
	if {"" != $gridresult} {
	    set result $gridresult
	}
	if {$highest} {
	    set highest_new_record $result
	}
    }
    if {$highest_old_lod > -10000} {
	puts "\n[centerline "Highest Old Result" 72]\n"
        puts $highest_old_record
    }

    if {$highest_new_lod > -10000} {
	puts "\n[centerline "Highest New Result" 72]\n"
        puts $highest_new_record
    }
    puts \
      "\n    *** Results have been written to [full_filename twopoint.out]"

# Load best model

    load model [full_filename null0.mod]
    if {$highest_new_lod > $highest_old_lod} {
	file rename -force [full_filename t.mod] [full_filename two.mod]
	file rename -force [full_filename t.out] [full_filename two.out]
	load model [full_filename two]
	puts "\n    *** Best model saved as [full_filename two.mod] which is now loaded."
    } else {
	if {$append} {
	    if {[file exists [full_filename two.mod]]} {
		load model [full_filename two.mod]
		puts "\n    *** Reloaded best old model [full_filename two.mod]"
	    } else {
		puts \
                    "\n    *** Best model from previous twopoint run not found"
	    }
	} else {
	    puts "\n    *** No satisfactory twopoint model found"
	}	
    }
    return ""
}


# solar::e2squeeze -- private
# 
# Purpose:  Set bounds around e2 based on previous value
#
# Usage:    e2squeeze <fraction>  ; 0.05 might be a good choice
#
# Notes:  e2squeeze is applied by multipoint and twopoint.  The default value
#         0.1, so if e2 were maximized to be 0.3 its bounds are set to 0.2
#         nd 0.4.
#
#         The highest lower bound found by applying e2lower and e2squeeze
#         is applied.  e2squeeze is considered a better tool than e2lower.
#
#         Setting e2squeeze to 1 disables the feature.  0 is invalid.
# -

proc e2squeeze {args} {
    if {$args == {}} {
	if {0 == [llength [info globals Solar_E2_Squeeze]]} {
	    return 0.1
	}
	global Solar_E2_Squeeze
	return $Solar_E2_Squeeze
    }
    global Solar_E2_Squeeze
    ensure_float $args
    if {$args <= 0 || $args > 1} {
	error "Invalid number for e2squeeze, must be >0 and <=1"
    }
    set Solar_E2_Squeeze $args
    return ""
}

# soft_lower_bound:
#   You have to "-push" to set bound below 0.01
#   Then you can only go down to 0.01 if above 0.01
#   or down to 0.005 if above 0.005
#   

proc soft_lower_bound {param args} {
    set push 0
    set newbound [read_arglist $args -push {set push 1}]
    ensure_float $newbound
    set oldbound [parameter $param lower]
    if {$newbound >= 0.01} {
	parameter $param lower $newbound
    } elseif {$push} {
	if {$oldbound <= 0.005} {
	    parameter $param lower $newbound
	} elseif {$oldbound <= 0.01} {
	    parameter $param lower [highest 0.0049 $newbound]
	} else {
	    parameter $param lower 0.01
	}
    }
}

# solar::define --
#
# Purpose:  Define an expression to be used in place of a trait or covariate
#
# Usage:    define <name> = <expression>  ; create a definition
#           trait <name> [,<name>]+       ; use definition as trait(s)
#
#           define                        ; show all defininitions
#           define <name>                 ; show definition for <name>
#           define delete <name>          ; delete define for name
#           define new                    ; delete all expressons
#           define delete_all             ; delete all expressons
#           define rename <name1> <name2> ; rename define name
#           define names                  ; return list of all names
#
#           <name> can be any alphanumeric string with underscore, do not
#           use these reserved words as names:
#
#               delete delete_all rename names new
#
#           <expression> is formatted algebraically using standard
#           math operators + - * / and ^ (power) and () parentheses, and
#           also all math functions defined by the C Programming Language
#           which includes "log" for natural logarithm, trig functions,
#           and hyperbolic functions, among others.  Here is a list:
#           erfc, erf, lgamma, gamma, j1, j0, y1, y0, rint, floor, ceil, 
#           tanh, cosh, sinh, atan, acos, asin, tan, cos, sin, expm1, exp,
#           logb, log1p, log10, log, cbrt, sqrt, and abs.  In addition,
#           the inverse normal transformation (see help for "inormal") may
#           be applied using the "inormal_" prefix (for example,
#           inormal_q4 for trait q4).  "inormal_" may be abbreviated
#           down to "inor_".  
#
#           If a phenotype name within the expression contains special
#           characters (anything other than letters, numbers, and underscore)
#           it should be enclosed in angle brackets <>, and the angle brackets
#           must also include any special operator prefix such as "inorm_".
#           For example, given a trait named q.4 (with a dot), you could
#           have a define command like this:
#
#               define i4 = <inorm_q.4>
#
#           Note: similar rules apply within the constraint and omega commands
#           because those commands also allow expressions that could contain
#           decimal constant terms and math operators.
#
#           A debugging function named "print" is also available which
#           prints and return the value of the expression it encloses.
#           After printing, it pauses until the RETURN key is pressed.
#           RETURN may be held down to pass through a lot of prints.
#           Examples of the print command are given in the documentation
#           for the "omega" command.
#
#           The following relational operators may also be used between
#           any two terms.  If the relation is true, 1 is returned,
#           otherwise 0 is returned.  This enables you to construct
#           compound conditional expressions having the same effect as
#           could have been done with "if" statements.  The C operators
#           < and > have been replaced with << and >> so as not to be
#           confused with the <> quotation of variable names in SOLAR.
#
#           C Format    Fortran Format    Test
#           --------    --------------    ----
#
#           ==          .eq.              if equal
#           !=          .ne.              if not equal
#           >=          .ge.              if greather than or equal
#           <=          .le.              if less than or equal
#           >>          .gt.              if greater than
#           <<          .lt.              if less than
#
#
#           An expression is understood to be quantitative unless the
#           top level operator is a relational operator, in which case
#           it is understood to be discrete.
#
#           Names used must not match the names of any phenotype.  When
#           there is an unintended match, the definition can not be used for
#           trait names since it would be ambiguous.
#
#           Once a valid definition has been created, it can be used in
#           the trait command.  Any or all of the traits can be definitions.
#           All definitions will be saved in the model file, and loaded back
#           in when that model is reloaded.  Definitions in a model file
#           will override current definitions.  It is possible to save a
#           model with nothing but definitions if desired.  The only
#           way to delete definitions is with the "new" "delete" or
#           "delete_all" options, or by restarting SOLAR.  The "model new"
#           command has no effect on definitions.
#
#           Expression names are not case sensitive.  Assigning a new
#           expression to a name replaces the expression previously
#           assigned to that name, even if it differs in letter case.
#           Renaming a definition to a name differing only in letter
#           case is possible.
#
#           For covariates only, it is possible to include in definition a
#           constant called "blank".  If an evaluation of the expression
#           returns blank, that individual is counted as missing from the
#           sample.  The best way to use this constant is with one or
#           more conditionals like this:
#
#           define sample = blank*(age<<22)*(sex==2)
#           covariate sample()
#
#           This blanks any male (sex==2) having age less than 22.
#           blank is the number -1e-20, so any numerical operation may
#           change it to a non-blank small number.  It should only be
#           multiplied by 0 or 1.  The empty parentheses after sample() mean
#           that it is not a maximized parameter, it is a null covariate
#           only used to delimit the sample.
#
# Examples:
#
#           define loga = log(a)
#           define eq1 = (q1 - 3.1)^2
#           define dq4 = q4 .gt. 12
# -

# solar::exclude --
#
# Purpose:  Excude phenotypes from use as covariates by automodel
#           and allcovar commands.
# 
# Usage:    exclude <var> <var> ... ; Add variable(s) to exclude
#           exclude                 ; List all excluded variables
#           exclude -reset          ; Reset to default exclude list
#           exclude -clear          ; Remove all variables from list
#
# Notes: You may add to the exclude list with one or more exclude commands.
#
#        By default, all variables named and/or mapped by the FIELD command
#        will be excluded (except for SEX).  The exclude command lets you
#        exclude additional variables.  (The FIELD command variables are
#        pedigree variables such as ID which would never be wanted as
#        covariates.)
#
#        The default exclude list will include the following standard PEDSYS
#        pedigree mnemonics: 
#
#          seq fseq mseq sseq dseq ego id fa mo sire dam pedno famno twin
#          mztwin ibdid fibdid mibdid blank kid1 psib msib fsib birth exit
#
#        If you are excluding more variables that you are keeping, you might
#        consider simply specifying the covariates you want explicitly
#        rather than using the allcovar or automodel commands, or creating
#        a new phenotypes file with fewer fields.
#
#        The variable name you enter will be converted to lower case.  Solar
#        is intended to handle phenotypic and pedigree variables in a case
#        insensitive manner.
# -

proc exclude {args} {
    global Solar_Exclude_List
    if {$args == {}} {
	if {![if_global_exists Solar_Exclude_List]} {
	    return "seq fseq mseq sseq dseq ego id fa mo sire dam pedno famno twin mztwin ibdid fibdid mibdid blank kid1 psib msib fsib birth exit"
	}
	return "$Solar_Exclude_List"
    }
    if {$args == "-clear"} {
	set Solar_Exclude_List ""
	return ""
    }
    if {$args == "-reset"} {
	if {[if_global_exists Solar_Exclude_List]} {
	    unset Solar_Exclude_List
	}
 	return ""
    }
    set Solar_Exclude_List [exclude]
    foreach arg $args {
	set arg [string tolower $arg]
	if {-1 == [lsearch $Solar_Exclude_List $arg]} {
	    if {[string length $Solar_Exclude_List]} {
		set Solar_Exclude_List "[exclude] $arg"
	    } else {
		set Solar_Exclude_List $arg
	    }
	}
    }
    return ""
}

# solar::perdelta --
#
# Purpose:  Set delta used by perturb
# 
# Usage: perdelta <number>
#
# Notes: Defaults to 0.001
# -

proc perdelta {args} {
    if {$args == {}} {
	if {0 == [llength [info globals Solar_Perturb_Delta]]} {
	    return 0.001
	}
	global Solar_Perturb_Delta
	return "$Solar_Perturb_Delta"
    }
    ensure_float $args
    global Solar_Perturb_Delta
    set Solar_Perturb_Delta $args
    return ""
}


# solar::maximize_quietly -- private
#
# Usage: maximize_quietly <outputfilename>
#
# This is the maximize used by most scripts for convenience
#   Verbosity is minimum unless "plus" verbosity was set by user 
# Convergence failure message is returned, all other errors are raised
# (The boundary check is now done by maximize itself.)
#-

proc maximize_quietly {outfilename {p maximize}} {
    set qu -q
    ifverbplus set qu ""

    if {![catch {eval $p $qu -o $outfilename} omessage]} {
	return ""
    }
    if {-1<[string first "convergence failure" [string tolower $omessage]]} {
	return $omessage
    }
    error $omessage
}

# solar::maximize_goodlod -- private
#
# Usage: maximize_goodlod <outfilename> [<procedure> [<minlike>]]
#
# maximize_goodlod is used by maxtry used by multipoint and twopoint.
# It invokes maximize_quietly to do the maximization.  However, if the
# maximization is successful but the resulting likelihood is significantly
# less than <minlike> (the likelihood of null model), it uses a special retry
# strategy to try to get the likelihood better, usually means to get LOD of 0
# or better.  The retries are ended when the loglikelihood is better than
# <minlod>, so only as many retries as needed are actually done.
#
# The retries are as follows:
#
# 1. simply maximize again, this works in 50% of cases.
# 2. start from null model, using standard starting point for h2q = 0.01
# 3. start from previous locus (or null if applicable) starting h2q at 0 
#     (this almost always fixes negative LOD problems)
# 4. start from null model, starting h2q at 0
#
# The intent  is to attempt if at all possible to avoid starting h2q1 at zero
# because it's possible that will miss actual positive heritability.  However,
# if other options fail, starting at zero almost always works to fix negative
# LOD cases,so that is the last resort, tried in two different ways, if
# necessary.  If all these attempts fail, a negative LOD could still result.
#
# Note: twopoint doesn't use these retries, minlike from there defaults to "".
# twopoint would have a problem with retry 2, since the linkmod would 
# then require the -2p option just for the twopoint cases.
#
# Lack of convergence retries are not generally done here as they are done by
# the caller maxtry.  Convergence error in first try is returned immediately,
# but later retries simply continue.
#
# This is intended to be called only from maxtry.  In particular, retries 2 
# and 4 assume that last_maxtry_model.mod exists in the output directory.
# That is done so that this procedure does not have to save the starting
# model a second time for every single maximize (troubled or not).
#
# The actual retries used can be controlled by a global variable called
# SOLAR_maximize_goodlod.  It is set to a sum of the following codes of
# retry types desired (see above list of retry types):
#
# 1.  Code 1
# 2.  Code 2
# 3.  Code 4
# 4.  Code 8
#
# The default is 15, which includes all types. For example, to enable only
# retry types 3 and 4, you would set the code to 12, like this in a script:
#
# global SOLAR_maximize_goodlod
# set SOLAR_maximize_goodlod 12
#
# From the SOLAR prompt, all variables are global, so the global command would
# not be needed.
# -

proc maximize_goodlod {outfilename minlike h2q_index ibdfile} {
    set errmsg [maximize_quietly $outfilename tmaximize]
    if {$errmsg != ""} {return $errmsg}
    if {$minlike == ""}	 {return ""}
    if {$minlike <= [loglike]} {return ""}
# 
# 1. simply maximize again, this works in 50% of cases.
#
    set trymask [use_global_if_defined SOLAR_maximize_goodlod 15]
    ifdebug puts "trymask is $trymask"
    if {$trymask & 1} {
	ifdebug puts "***** retrying to get non-negative lod"
	save model [full_filename last_maximize_goodlod]
	set lastlike [loglike]
	set errmsg [maximize_quietly $outfilename tmaximize]
	if {$errmsg == ""} {
	    if {$minlike <= [loglike]} {return ""}
	}
    }
#
# 2. start from null model, using standard starting point for h2q = 0.01
#
    
    set fixed_loci [expr $h2q_index - 1]
    if {$trymask & 2} {
	ifdebug puts "***** restarting from null to get non-negative lod"
	load model [full_filename null$fixed_loci]
	linkmod $ibdfile ;# this would need to change for twopoint
	set errmsg [maximize_quietly $outfilename tmaximize]
	if {$errmsg == ""} {
	    if {$minlike <= [loglike]} {return ""}
	}
    }
#
# 3. start from previous locus (or null if applicable) with h2q = 0 (usually works)
#
    if {$trymask & 4} {
	ifdebug puts "***** Trying restart from h2q$h2q_index = 0"
	load model  [full_filename last_maxtry_model]
	set ts [trait]
	set nt [llength $ts]
	set tsuf ""
	foreach tr $ts {
	    if {$nt > 1} {
		set tsuf \($tr\)
	    }
	    parameter h2r$tsuf = [expr [parameter h2r$tsuf =] + [parameter h2q$h2q_index$tsuf =]]
	    if {[parameter h2r$tsuf upper] < [parameter h2r$tsuf =]} {
		parameter h2r$tsuf upper [lowest [expr [parameter h2r$tsuf =] + 0.05] 1]
	    }
	    parameter h2q$h2q_index$tsuf = 0 lower 0
	}
	set errmsg [maximize_quietly $outfilename tmaximize]
	if {$errmsg == ""} {
	    if {$minlike <= [loglike]} {return ""}
	}
    }
#
# 4. start from null model, with h2q = 0
#
    if {$trymask & 8} {
	ifdebug puts "***** restarting from null with h2q$h2q_index = 0"
	load model [full_filename null$fixed_loci]
	linkmod -zerostart $ibdfile ;# this would need to change for twopoint
	set errmsg [maximize_quietly $outfilename tmaximize]
	if {$errmsg == ""} {
	    if {$minlike <= [loglike]} {return ""}
	}
    }
#
# Could not get non-negative LOD
#   Just take first successful result.
#
    load model [full_filename last_maximize_goodlod]
    return ""
}


# solar::maxtry -- private
#
# Implements retry strategy for multipoint and twopoint
#
# Retry strategy for non-convergence:
#   1.  Try same model twice again (maximization uses random numbers)
#   2.  If didn't begin with nullX model, reload nullX model (applying
#         new linkage parameters) and retry
#   3.  Set equal H* parameters and retry
#
# If parameter h2q1 doesn't exist, 2 and 3 are not done.
#
# maxtry itself uses maximize_goodlod, which forces re-maximizations of the
# ongoing model if the new likelihood is significantly smaller than minlike,
# the null model loglikelihood, but otherwise convergence occurred.
# -

proc maxtry {try_switches h2q_index ibdfile outfilename verbose {minlike ""}} {

    if {[solardebug]} {
	puts "Entering maxtry"
	set verbose 1
    }

    model save [full_filename last_maxtry_model]
    set errmsg [maximize_goodlod $outfilename $minlike $h2q_index $ibdfile]
    if {$errmsg == ""} {
	return ""
    }
#
#   1. Just try same model all over again 2x
#   This fixes most convergence errors because random numbers are used
#   and sometimes they just don't work out.  maximization is a
#   non-deterministic process, though results most often seem deterministic
#   with problematic likelihood space, the non-deterministic nature is
#   exposed.
#
    load model [full_filename last_maxtry_model]
    set errmsg [maximize_goodlod $outfilename $minlike $h2q_index $ibdfile]
    if {$errmsg == ""} {
	return ""
    }

    load model [full_filename last_maxtry_model]
    set errmsg [maximize_goodlod $outfilename $minlike $h2q_index $ibdfile]
    if {$errmsg == ""} {
	return ""
    }
#
#   2.  If didn't begin with nullX model, reload nullX model (applying
#
    if {$try_switches & 2} {
	set fixed_loci [expr $h2q_index - 1]
	if {$verbose} {
	    puts \
	"\n    *** Retry with null$fixed_loci model starting parameters"
	}
	model load [full_filename null$fixed_loci]
	linkmod $ibdfile
	set errmsg [maximize_goodlod $outfilename $minlike $h2q_index $ibdfile]
	if {$errmsg == ""} {
	    return ""
	}
    }
#
#   3.  Set equal H* parameters and retry
#       Note: this is only done for univariate models, because multi would be
#       much more complicated, and I don't actually think this does much
#       good anyway, but I was asked to include this strategy long ago.
#
    if {1 == [llength [trait]]} {

	if {![if_parameter_exists h2q1] || ![if_parameter_exists h2r] || \
		![if_parameter_exists e2]} {
	    set try_switches 0
	}

	if {$try_switches & 1} {
	    model load [full_filename last_maxtry_model]
	    set hstart [expr (1.0-double([parameter e2 start]))/ \
			    double ($h2q_index + 1.0)]
	    if {$verbose} {
		puts "\n    *** Retry with equal H* parameters: $hstart"
	    }
	    for {set i 0} {$i <= $h2q_index} {incr i} {
		if {$i == 0} {
		    set hname h2r
		} else {
		    set hname h2q$i
		}
		parameter $hname start $hstart
		set hupper [parameter $hname upper]
		set hlower [parameter $hname lower]
		if {$hstart >= $hupper} {
		    parameter $hname upper [expr $hstart + 0.001]
		}
		if {$hstart <= $hlower} {
		    parameter $hname lower [expr $hstart - 0.001]
		}
	    }
	}
	set errmsg [maximize_goodlod $outfilename $minlike $h2q_index $ibdfile]
	if {$errmsg == ""} {
	    return ""
	}
    }

#   pre640_maxtry_sections was included here

# If we get here, all attempts to converge have failed

    if {$verbose} {
	puts \
	    "\n    *** Failure to converge after programmed retries"
    }
    return $errmsg
}

# The following used to be included in maxtry but has been long
# obsolete

proc pre640_maxtry_sections {} {

# If this is a bivariate linkage model, try constraining rhoq1
# This section is now obsoleted.

    if {0} {
    if {$try_switches != 0 && $multi && [if_parameter_exists rhoq1]} {
	if {$verbose} {
	    puts "    *** Constraining RhoQ1 to 1"
	}
	load model [full_filename last_maxtry_model]
	constraint rhoq1 = 1
	set llrhoq11 -1e20
	set errmsg1 [maximize_quietly $outfilename tmaximize]
	if {"" == $errmsg} {
	    set llrhoq11 [loglike]
	    save model [full_filename maxtry_rhoq1]
	    if {$verbose} {
		puts "    *** Model converged"
	    }
	}
	if {$verbose} {
	    puts "    *** Constraining RhoQ1 to -1"
	}
	load model [full_filename last_maxtry_model]
	constraint rhoq1 = -1
	set llrhoq1_1 -1e20
	set errmsg2 [maximize_quietly $outfilename tmaximize]
	if {"" == $errmsg2} {
	    set llrhoq1_1 [loglike]
	    if {$verbose} {
		puts "    *** Model converged"
	    }
	}
	if {$llrhoq11 != -1e20 || $llrhoq1_1 != -1e20} {
	    if {$llrhoq11 > $llrhoq1_1} {
		load model [full_filename maxtry_rhoq1]
	    }
	    return "ConsRhoq"
	}
    }
    }

# OK, as a last resort, try modifiying conv
# This section is now obsoleted.

    if {0} {
    if {$verbose} {
	puts "    *** Reducing CONV to 1e-5"
    }
    load model [full_filename last_maxtry_model]
    option conv 1e-5
    set errmsg [maximize_quietly $outfilename tmaximize]
    option conv 1e-6
    if {"" == $errmsg} {
	puts "    *** Warning.  Parameter accuracy reduced.  See 'help accuracy'."
	return ""
    }

    if {$verbose} {
	puts "    *** Reducing CONV to 1e-4"
    }
    load model [full_filename last_maxtry_model]
    option conv 1e-4
    set errmsg [maximize_quietly $outfilename tmaximize]
    option conv 1e-6
    if {"" == $errmsg} {
	puts "    *** Warning.  Parameter accuracy reduced..  See 'help accuracy'."
	return ""
    }
    }
}


# fels_maxtry is now obsolete and removed from the maxtry calling sequence
# prior to version 6.4.0, this procedure used to be called "maxtry" and the
# now again "maxtry" was "omaxtry"
#
proc fels_maxtry {try_switches h2q_index ibdfile outfilename verbose {minlike ""}} {
    set status [maxtry $try_switches $h2q_index $ibdfile $outfilename \
		    $verbose $minlike]
#
# The following code is switched off, related to "fels problem"
#   to turn on again, change test below to ""
#
    if {"zero_lod_expected" == $status} {

# Check for fels problem (zero lod expected)

	set hi [h2qcount]
	set nu [expr $hi - 1]
	if {$hi >= 1} {
	    if {[if_parameter_exists h2q$hi]} {
		if {0.0==[parameter h2q$hi =]} {
		    if {[file exists [full_filename null$nu.mod]]} {
			if {abs([oldmodel null$nu loglike] - [loglike]) \
				> 0.05} {
			    if {[if_global_exists SOLAR_FELS]} {
				set oldmatrix [matrix]
				save model [full_filename zero_lod_expected]
				load model [full_filename null$nu]
				parameter h2q$hi = 0  upper 0.2
				carve_new_value h2q$hi 0.01
				set cstring "constraint e2 + h2r"
				for {set i 1} {$i <= $hi} {incr i} {
				    set cstring "$cstring + h2q$i"
				}
				set cstring "$cstring = 1"
				eval $cstring
				eval $oldmatrix
				set status [maxtry $try_switches $h2q_index $ibdfile $outfilename $verbose]
				if {"" != $status} {
				   load model [full_filename zero_lod_expected]
				    set status "ZeroLodX"
				} else {
				    if {$hi >= 1} {
					if {[if_parameter_exists h2q$hi]} {
					    if {0.0==[parameter h2q$hi =]} {
						if {[file exists [full_filename null$nu.mod]]} {
						    if {abs([oldmodel null$nu loglike] - [loglike]) \
							    > 0.05} {
							set status "ZeroLodX"
						    }
						}
					    }
					}
				    }
				}
			    } else {
				set status "ZeroLodX"
			    }
			}
		    }
		}
	    }
	}
    }
    return $status
}

# Now, why is h2qcount based on omega and not parameters ?

proc h2qcount {} {
    set count 0
    set omega_equation [omega]
    set suffix [lindex [trait] 1]
    if {{} != $suffix} {set suffix ($suffix)}

    for {set i 1} {1} {incr i} {
	if {-1==[string first h2q$i $omega_equation] && \
		-1==[string first H2q$i $omega_equation] && \
		-1==[string first h2Q$i $omega_equation] && \
		-1==[string first H2Q$i $omega_equation]} break

# Make sure this is an actual parameter	
	
	if {[if_parameter_exists h2q$i$suffix]} {
	    incr count
	}
    }
    return $count
}

proc read_bayes {modelname outfilename what} {
    if {![file exists [full_filename $outfilename]]} {
	error "Output file $outfilename does not exist"
    }
    set soutfile [open [full_filename $outfilename]]
    gets $soutfile  ;# skip over header
    gets $soutfile  ;# skip over ------
    while {-1 != [gets $soutfile line]} {
	set mname [lindex $line 0]
	if {$modelname == $mname} {
	    close $soutfile
	    if {"bic" == $what} {
		return [lindex $line 1]
	    } elseif {"loglike" == $what} {
		return [lindex $line 2]
	    } else {
		error "read_bayes does not read $what"
	    }
	}
    }
    close $soutfile
    error "Record for model $modelname not found"
}
	

proc get_prev_lod_and_h2q {h2q_index chrom loc} {
    if {![file exists [full_filename multipoint$h2q_index.out]]} {
	error "Output file multipoint$h2q_index.out does not exist"
    }
    set soutfile [open [full_filename multipoint$h2q_index.out]]
    while {-1 != [gets $soutfile line]} {
	if {5 != [scan $line "%s %s %s %d %f" cid chromnum lid locnum olod]} {
	    continue
	}
	if {[string compare $cid "chrom"] || \
		[string compare $lid "loc"]} {
	    continue
	}
	if {![string compare "0" [string index $chromnum 0]]} {
	    set chromnum [string range $chromnum 1 end]
	}
	if {$chromnum != $chrom || $locnum != $loc} {
	    continue
	}
	set h2q [lindex $line end]
	if {![is_float $h2q]} {
	    close $soutfile
	    error "Error in record for chrom $chrom loc $loc: $h2q"
	}
	close $soutfile
	return [list $olod $h2q]
    }
    close $soutfile
    error "Record for chrom $chrom loc $loc not found"
    return ""
}

# solar::fatal_error_checks -- private
#
# Does fatal_error_checks.  Returns error that will stop solar
# Called during solar startup before check_os.
# -

proc fatal_error_checks {} {
    catch {ranch_check}
    if {[llength [info procs ranch_check]]} {
	set result [ranch_check]
	if {"" != $result} {
	    error $result
	}
    }
    set result [check_version_compatibility]
    if {"" != $result} {
	error $result
    }
    return ""
}

proc solarversion {} {
    return "SOLAR Eclipse version [solar_tcl_version], last updated on [solar_up_date], [solar_up_year]\nSOLAR main binary was compiled on [solar_compiled_date] at [solar_compiled_time]"
}

proc solar_tcl_startup {} {
 global Solar_Batch
 if {!$Solar_Batch} {
     catch {
     puts "\nSOLAR Eclipse version [solar_tcl_version], last updated on [solar_up_date], [solar_up_year]"
     puts "Copyright (c) 1995-[solar_up_year] Texas Biomedical Research Institute"
     }
  puts "Enter help for help, exit to exit, doc to browse documentation.\n"
  global Solar_Gotcl
  if {[llength $Solar_Gotcl]} {
      puts "Using solar.tcl in $Solar_Gotcl"
  }
 }
}

proc check_version_compatibility {} {
    if {![string compare 8.1.1 \
	      [solar_binary_version]]} {
	return ""
    } else {
	puts " "
	puts stderr "solar.tcl is version [solar_tcl_version]"
	puts stderr "solarmain binary is version [solar_binary_version]"
	global Solar_Gotcl
	if {[llength $Solar_Gotcl]} {
	    set errmsg "\nMismatched version of solar.tcl in $Solar_Gotcl"
	} else {
	    global env
	    set errmsg "\nMismatched version of solar.tcl in $env(SOLAR_LIB)"
	}
    }
    error $errmsg
}
    


# solar::no_check_os -- private
#
# Purpose:  Turn off OS and library advisory checking
#
# Usage:    no_check_os
#
# Notes:   When starting, solar checks for bugs which may result when the
#          version of the OS you are running is lower than the one used
#          when solar was built.  (Forward compatibility is usually
#          available, but backward compatibility is not.)  One of these
#          relates to the proper handling of IEEE NaNs (not a number).
#          Normally, NaN's arise only when convergence fails.
#
#          If the test fails, you are given a warning but must press
#          return to continue.
#
#          If you would like to turn off the checking (at your own
#          risk), you may give the no_check_os command in your .solar
#          startup script.  This will disable OS testing and will enable
#          you to run shell scripts which call solar even with a release
#          that is too early.
# -

proc no_check_os {} {
    eval {proc check_os {} {return "OS checking has been turned off"} }
    return ""
}

proc check_os {} {}

proc check_nan {} {
    set foo NaN
    if {[catch {format %9.6f NaN}] || \
	    $foo > 1} {
	puts \
	"NaN bug: Solar was linked under a later OS than you are running"
	puts -nonewline \
  "This might cause problems.  Press return to continue> "
	flush stdout
	gets stdin junk
	if {$junk == "q"} {error "Script terminated at user request"}
	return ""
    }
    return "NaN handling OK"
}

# solar::usort --
#
# Purpose:  Define unix sort program name 
#           (used for multipoint*.out files)
#
# Usage: usort <program>         ; use program
#        usort ""                ; disables sort feature}
#        usort                   ; show current program
#
# Notes: The default is /usr/bin/sort, which should work on most system.
#        It is necessary to include a path for users which have PEDSYS,
#        which has its own program named "sort."  The program must be
#        compatible with unix sort and have -n -o and -k arguments.
#
# Example:  usort /usr/local/bin/sort
#-

proc usort {args} {
    global Solar_Unix_Sort
    if {$args == {}} {
	if {[if_global_exists Solar_Unix_Sort]} {
	    return $Solar_Unix_Sort
	} else {
	    if {[file exists /bin/sort]} {
		return /bin/sort
	    } elseif {[file exists /usr/bin/sort]} {
		return /usr/bin/sort
	    } else {
	    error "Unix sort not found.  Use usort command to set full path."
	    }
	}
    }
    set Solar_Unix_Sort $args
}


proc topline {filename} {
    if {[catch {set filep [open $filename r]}]} {
	return ""
    }
    set count [gets $filep line]
    close $filep
    if {$count > 0} {
	return $line
    }
    return ""
}


# solar::newtcl --
#
# Purpose:  Recognize new or changed Tcl procedures in Tcl scripts
#
# Usage:    newtcl
#
# Notes:    At the time a SOLAR session is started, all Tcl scripts
#           (files ending with ".tcl") are scanned.  The newtcl
#           command forces another such scan in order to recognize
#           new Tcl procedures (created AFTER the start of the SOLAR
#           session), or to recognize changes to Tcl procedures since
#           the first time those procedures were used (see explanation
#           below).  You could also accomplish this by exiting from
#           and restarting SOLAR, but that is often inconvenient
#           because it causes the loss of session state.
#
#           The following directories are scanned by SOLAR for user scripts:
#
#                .      (the current working directory)
#                ~/lib  (the lib subdirectory of your home directory, if it exists)
#
#           A procedure found in "." will supercede one found in "~/lib" having
#           the same name.  Also beware that if the same procedure name is used
#           in more than one script file, the first one encountered will be
#           the one actually used.  If the same procedure name is found in two
#           files in the same directory, the precedence is not predictable.
#
#           The scanning process simply looks through each script file for
#           "proc" (procedure) statements.  An index of all the procedures
#           is then written to a file named tclIndex in the working directory.
#           This file will only be created if user-defined Tcl scripts are found.
#
#           Tcl procedures are only loaded into SOLAR the first time they
#           used.  Once loaded, they stay loaded, and may no longer reflect
#           the Tcl files in the scan path if those Tcl files are changed.
#           The newtcl command flushes all currently loaded procedures, so
#           the next time any procedure is invoked, it will be reloaded from
#           the file.
#
#           The main Tcl file used by SOLAR is named solar.tcl and is located in
#           the lib subdirectory of the SOLAR installation.  This defines all
#           the fundamental procedures used by SOLAR.  User-defined procedures
#           having the same name as built-in procedures will supercede them.
# -

# newtcl is actually created by the binary solarmain since it must exist in order
# for solar.tcl itself to be scanned.



# solar::needk2 --
#
# Purpose:  Keep K2 (phi2) terms from MIBD matrices
#
# Usage:    needk2
#           needk2 off
#
# Notes:    This command is now obsolescent and should not be used.
#
#           The K2 in MIBD files is obsolescent.  We now maintain
#           a separate phi2.gz file for discrete trait analyses, and
#           for quantitative trait analyses, the K2 (phi2) values are
#           computed as needed.
#
# Old Notes:
#
#           If you need to use any of the K2_* matrix values, issue the needk2
#           command before loading the matrix (or running 'multipoint.')
#
#           Normally the K2 values from matrix files are not used because
#           they are identical to the K2 values computed by SOLAR as needed.
#
#           The default (of not saving K2) cuts matrix memory usage in half.
# -

proc needk2 {args} {
    global Solar_Save_K2
    if {$args == "off"} {
	set Solar_Save_K2 0
    } elseif {$args != ""} {
	error "Invalid needk2 argument"
    } else {
	set Solar_Save_K2 1
    }
    return ""
}

proc ifneedk2 {} {
    global Solar_Save_K2
    if {[if_global_exists Solar_Save_K2]} {
	return $Solar_Save_K2
    } else {
	return 0
    }
}

# solar::memory
#
# Purpose:  Show total memory used by this SOLAR process
#
# Usage:    memory
#
# Notes:    This is intended primarily for internal debugging purposes.
#           Now works on all supported systems.
# -



proc procmem {} {return [memory]}

proc memory {} {
    set memstring 0
    set osname [string tolower [exec uname]]
    if {$osname == "sunos"} {
	catch {set memstring [eval exec du -ks /proc/[pid]/as]}
	if {[llength $memstring] != 2  || $memstring == 0} {
	    set memstring ""
	}
    } else {
	if {$osname == "linux"} {
	    catch {set memstring [eval exec ps -H v [pid]]}
	} else {
	    catch {set memstring [eval exec ps -l [pid]]}
	}
	set first [lsearch $memstring PID]
	if {$first == -1} {
	    puts $memstring
	    error "unable to get memory PID"
	}
	set rsspos [lsearch $memstring RSS]
	if {$rsspos == -1} {
	    puts $memstring
	    error "unable to get memory RSS"
	}
	set lfpos [lsearch $memstring [pid]]
	if {$lfpos == -1} {
	    return $memstring
	    error "unable to get memory data"
	}
	set pos [expr $lfpos + ($rsspos - $first)]
	set memstring [lindex $memstring $pos]
    }
    if {$memstring != ""} {
	    return "[lindex $memstring 0] K Bytes in use by SOLAR"
    }
    return "0 (unable to get mem size)"
}


# solar::benice --
#
# Purpose:  Lower priority of SOLAR to allow more CPU for other jobs
#           or lower priority of one SOLAR run relative to another
#
# Usage:    benice              ; Set "nice" level to 15
#           benice <LEVEL>      ; LEVEL is between 1 and 20
#                               ; 20 is "most nice"
#
# Notes:    This is intended for use on Unix systems which support the
#           "renice" command, including Solaris 2.5 and above
#
#           Once you have set a nice level, you cannot go back to a 
#           higher priority on this process.  You must exit and restart.
#
#           The default unix scheduling allows some time even for
#           very "nice" jobs.  However, they get somewhat less CPU than
#           other jobs.
#
#           On the SFBR Ranch, scheduling is absolute, so that "nice"
#           jobs will be suspended until all other jobs are done (or
#           waiting for a system resource such as disk access).  Nice
#           jobs have minimal (<1%) impact on other jobs, unless they
#           hog huge gobs of memory.
# -

proc benice {args} {
    if {$args == {}} {set args 15}
    eval exec renice $args -p [pid]
}

# solar::qtnm --
# 
# Purpose:  Marginal tests for bayesavg -qtn
#
# Usage:    [allsnp]
#           bayesavg -qtn -max 1
#           [load map snp.map]
#           qtnm [-noplot] [-nomap]
#
#           -noplot  Do not plot results
#           -nomap   Do not use map file; SNP locations are encoded in names
#
# Notes:    You must do bayesavg -qtn [-max 1] first, then qtnm.  qtnm
#           writes a datafile qtnm.out in the outdir, then invokes
#           plotqtn to plot it.  (The -max 1 is optional, however,
#           if you want to do this quickly, you had best include it.)
#
#           To include all snps as covariates in the starting model, use
#           the separate command "allsnp".
#
#           SNP covariate names (after the snp_ or hap_ prefix) will be
#           mapped to locations using the currently loaded map file,
#           which must be loaded prior to running qtnm.  Map files stay
#           loaded from one solar session to the next (in the same working
#           directory) so once you have loaded it, you do not need to load
#           it again.
#
#           Beginning with version 3.0.3, snp names will always be mapped
#           to locations using a loaded map file.  However, you can revert
#           to the previous method, in which the locations are encoded into
#           the snp "names" using the -nomap option.
#
#           Beginning with SOLAR version 3.0.2, the qtnm.out file
#           has the following 5 columns:
#
#           SNP Name (or location if numeric)
#           SNP location
#           Chi Squared
#           p
#           log(p)
#
#          Previously there was no "SNP Name" column because it was
#          assumed to be the location.  Note that plotqtn accepts
#          qtnm.out files with either 4 or 5 columns.
# -

proc qtnm {args} {
    set plot 1
    set nomap 0

    set badargs [read_arglist $args -noplot {set plot 0} -nomap {set nomap 1}]
    if {{} != $badargs} {
	error "qtnm: Bad arguments: $badargs"
    }
#
# Get list of snp covariates from starting model
#
    save model [full_filename bayesavg_ma.orig]
    load model [full_filename cov.orig.mod]
    set covariates [covariates -applicable]

    set poslist {}
    set namelist {}
    set n 0
    foreach cov $covariates {
	set target [string tolower [string range $cov 0 3]]
	if {"snp_" == $target || "hap_" == $target} {
	    set number [set name [string range $cov 4 end]]
	    if {!$nomap} {
		set number [map locn $name]

	    } elseif {![is_float $number]} {
		error "qtnm: Using -nomap and $cov is invalid"
	    }
	    lappend poslist $number
	    lappend namelist $name
	    incr n
	}
    }
    load model [full_filename bayesavg_ma.orig]
    puts "There appear to be $n snps..."
#
# Make list of model names we want
#
    set wanted {}
    for {set i 0} {$i <= $n} {incr i} {
	lappend wanted cov$i
    }
#
# Scan through output file picking up all data we want
#
    set found 0
    if {[catch {set ifile [open [full_filename bayesavg_cov.out]]}]} {
	set ifile [open [full_filename bayesavg_cov.est]]
    }
    gets $ifile
    gets $ifile
    while {$n+1 > $found} {
	if {-1 == [gets $ifile line]} {
	    close $ifile
	    error "Incomplete results: only $found out of $n+1"
	}
	set modelname [lindex $line 0]
	if {-1 != [lsearch $wanted $modelname]} {
	    set loglikes($modelname) [lindex $line 2]
	    incr found
	}
    }
    close $ifile
#
# Capture output to lists and variables also
#
    set all_points {}
    set marker_labels {}
    set outfile [open [full_filename qtnm.out] w]
    set log0 $loglikes(cov0)
    for {set i 1} {$i <= $n} {incr i} {
	set modname cov$i
	set logn $loglikes($modname)
	set chisq [expr 2*($logn - $log0)]
	set p [chi -number $chisq 1]
	set name [lindex $namelist [expr $i - 1]]
	set lp [expr log10($p)]
	set pos [lindex $poslist [expr $i - 1]]
#	puts $outfile [format "%8d  %1g %10.7g %9.6g" $name $chisq $p $lp]
	puts $outfile \
	    [format "%9s %8d  %1g %10.7g %9.6g" $name $pos $chisq $p $lp]
    }
    close $outfile
#
# Do plotting if desired
#
    if {$plot} {
	plotqtn
    }
    return ""
}

# solar::plotqtn
#
# Purpose:  Plot qtn marginal tests (qtnm.out)
#
# Usage:    plotqtn [-nolabels] [-nomarkers] [-file filename] [-local]
#
#           -nolabels     do not include "marker" labels (ticks only)
#           -nomarkers    do not include marker ticks or labels
#           -file         Use named file instead of qtnm.out in outdir
#           -local        Ignore default plot parameters; use only local file
#
# Notes:    You must select the trait or outdir first.  See qtnm for
#           more information.  It must be possible to find the qtnm.out
#           file in the outdir.
#
#           The plot parameter file (in SOLAR_LIB) is qtn.gr.  You
#           may override it with a copy in ~/lib or your working
#           directory.  Your version need only include the parameters
#           you would like to change.  This should work in most cases.
#           If you specify -local, however, the qtn.gr in SOLAR_LIB
#           is completely ignored, and your qtn.gr must be complete,
#           which might get around some very obscure conflict between
#           the two plot parameter files.
#
#           plotqtn accepts either the original 4 or the new 5 column qtnm.out
#           files.  The 5 column files begin with the snp name that is not
#           necessarily the location.
# -

proc plotqtn {args} {


# Important constants
    set x_expansion_factor 1.12  ;# Expansion of X beyond input range

    set debug [solardebug]

    set marker_margin 4.5
    set nolabels 0
    set noticks 0
    set filename ""
    set local_gr 0

    set badargs [read_arglist $args \
		     -nolabels {set nolabels 1} \
		     -nomarkers {set nolabels 1; set noticks 1} \
		     -marker_margin marker_margin \
		     -file filename \
		     -local {set local_gr 1} \
		]
    if {"" != $badargs} {
	error "plotqtn: Invalid arguments: $badargs"
    }
    if {"" == $filename} {
	set filename [full_filename qtnm.out]
    }

    if {$local_gr && $debug} {puts "Warning!  Ignoring qtn.gr IN SOLAR_BIN"}


# Read qtnm.out

    set marker_labels {}
    set all_points {}
    set max_mrk_name_len 0
    set first 1
    set qfile [open $filename]
    set lines 0
    while {-1 != [gets $qfile line]} {
	if {0 == [llength $line]} {
	    continue
	}
	incr lines

# Determine if this is old or new style output file

	if {[llength $line] == 4} {
	    set pos [lindex $line 0]
	    set name $pos
	    set lp [expr 0.0 - [lindex $line 3]]
	} elseif {[llength $line] == 5} {
	    set name [lindex $line 0]
	    set pos [lindex $line 1]
	    set lp [expr 0.0 - [lindex $line 4]]
	} else {
	    error "qtnmplot: Incorrect record length in qtnm.out: $line"
	}
	if {$first} {
	    set min_pos $pos
	    set max_pos $pos
	    set min_lp $lp
	    set max_lp $lp
	    set first 0
	}
	lappend all_points [list $pos $lp]
	lappend marker_labels [list $pos $name $pos]
	if {[string length $name] > $max_mrk_name_len} {
	    set max_mrk_name_len [string length $pos]
	}
	if {$pos < $min_pos} {
	    set min_pos $pos
	}
	if {$pos > $max_pos} {
	    set max_pos $pos
	}
	if {$lp < $min_lp} {
	    set min_lp $lp
	}
	if {$lp > $max_lp} {
	    set max_lp $lp
	}
    }
    close $qfile
    if {$lines < 2} {
	error "File $filename contains only $lines lines"
    }

    ifdebug puts "min_pos $min_pos max_pos $max_pos"
    ifdebug puts "min_lp $min_lp max_lp $max_lp"

# Sort points if required (yes)

    proc sort_by_first_num {a b} {
	set a0 [lindex $a 0]
	set b0 [lindex $b 0]
	if {$b0 > $a0} {
	    return -1
	} elseif {$b0 < $a0} {
	    return 1
	}
	return 0
    }

    set all_points [lsort -command sort_by_first_num $all_points]


# Sort marker ticks and labels too

    set marker_ticks $marker_labels
    set marker_ticks [lsort $marker_ticks]
    set marker_labels [lsort -command sort_by_first_num $marker_labels]

# Begin plotting...

    set setnum 1
    set graphnum 1

# Open new or existing tclgr session

    if {[catch {tclgr open} errmsg]} {
	if {[string compare $errmsg  "tclgr session already opened from this solar session"]} {
	    error $errmsg
	}
    }

# Kill previous set, and graph if not overlaying

    tclgr send kill graphs
    tclgr send clear line
    tclgr send clear string
    tclgr send focus g$graphnum

# Calculate scaling and ticks for x

    set max_x_unscaled $max_pos
    set min_x_unscaled $min_pos
    set delta_x_unscaled [expr $max_x_unscaled - $min_x_unscaled]
    set center_x [expr ($min_pos + $max_pos) / 2]
    ifdebug puts "delta_x_unscaled: $delta_x_unscaled  center_x:  $center_x"

    set mmlist [majorminor $delta_x_unscaled $min_x_unscaled $max_x_unscaled]
    set major_x [lindex $mmlist 0]
    set minor_x [lindex $mmlist 1]
    ifdebug puts "major_x: $major_x  minor_x: $minor_x"

    set delta_x [expr $delta_x_unscaled * $x_expansion_factor]
    set max_x [expr $center_x + ($delta_x / 2.0)]
    set min_x [expr $center_x - ($delta_x / 2.0)]
    set use_x [expr ($min_x + $min_x_unscaled) / 2.0]
    set firstlabel_x [expr $major_x * ceil ($use_x / $major_x)]
    ifdebug puts "xmin: $min_x   xmax: $max_x   firstlabel: $firstlabel_x"

# Setup scaling and ticks for x

    tclgr send world xmin $min_x
    tclgr send world xmax $max_x
    tclgr send xaxis tick op bottom
    tclgr send xaxis tick major $major_x
    tclgr send xaxis tick minor $minor_x

    tclgr send xaxis ticklabel op bottom
    tclgr send xaxis ticklabel start type spec
    tclgr send xaxis ticklabel start $firstlabel_x

    if {abs($min_x) <= 2.0e9 && abs($max_x) <= 2.0e9} {
	tclgr send xaxis ticklabel format decimal
	tclgr send xaxis ticklabel prec 0
    }

# Set up y tick marks (major and minor)

#    ************* Set up Y Axis *****************


# This fiendishly complicated method to set up y axis is borrowed from
# proc plot.  It sets up nice major and minor ticks consistent with xmgr
# regardless of size, following a 1/2/5 rule.

# The design maximum Log10(P) is 2000
#   (highest tick range is therefore 750-1500)
#   (I don't like limits, but this is way beyond anything I can imagine)
# maxy must be 4*10^N or 1*10^N
# the other values are related

    set maxy  1000
    set major 200
    set major_digit 2
    set minor 100

    set maxdiv 2.5
    set majdiv 2
    set mindiv 2

# Since we mess with global variable tcl_precision
# We must catch errors to be sure it gets restored

    global tcl_precision
    set save_tcl_precision $tcl_precision
    set tcl_precision 6
    set not_ok [catch {
    while {1} {
	if {$max_lp > $maxy} {
	    ifdebug puts "Y Axis Major: $major Minor: $minor"
	    tclgr send yaxis tick major $major
	    tclgr send yaxis tick minor $minor
	    break
	}
	set maxy [expr double($maxy) / $maxdiv]
	set major [expr double($major) / $majdiv]
	set minor [expr double($minor) / $mindiv]

	if {$major_digit == 2} {
	    set major_digit 1
	    set maxdiv 2
	    set majdiv 2
	    set mindiv 2
	} elseif {$major_digit == 1}  {
	    set major_digit 5
	    set maxdiv 2
	    set majdiv 2.5
	    set mindiv 2.5
	} else {
	    set major_digit 2
	    set maxdiv 2.5
	    set majdiv 2
	    set mindiv 2
	}
    }
    } caught_error]
    set tcl_precision $save_tcl_precision
    if {$not_ok} {
	error $caught_error
    }

# Create y expansion factor to permit showing markers labels at top

    if {$nolabels || $noticks} {
	set max_mrk_name_len 0
    }
    set marker_margin 2.0
    set chars_that_fit_vertically 55.0  ;# using default font size (was 58)
    set prop_screen_for_markers [expr ($max_mrk_name_len + $marker_margin)  / $chars_that_fit_vertically]
    set prop_screen_for_mticks 0.04
    set prop_screen_for_mleaders 0.04
    set prop_screen_for_ttext 0.015
    set prop_screen_for_plot [expr 1.0 - ($prop_screen_for_mticks +  $prop_screen_for_mleaders +  $prop_screen_for_ttext  +  $prop_screen_for_markers )]
    set expand_y [expr 1.0 / $prop_screen_for_plot]
    ifdebug puts "expand_y is $expand_y"

# Set up y axis and ticks

    set max_y_unscaled $max_lp
    set min_y_unscaled $min_lp

    set max_y_ticklabel [expr $major*ceil(double($max_lp)/$major)]
    ifdebug puts "max_y_ticklabel: $max_y_ticklabel"
    set max_y [expr $max_y_ticklabel * $expand_y]
    set min_y [expr -0.06 * $max_y]
    tclgr send world ymin $min_y
    tclgr send world ymax $max_y
    tclgr send yaxis ticklabel start type spec
    tclgr send yaxis ticklabel stop type spec
    tclgr send yaxis ticklabel start 0
    tclgr send yaxis ticklabel stop $max_y_ticklabel
#
# xmgr spaces Y axis label too close to single digits
#
    if {$max_y_ticklabel > 4 && $max_y_ticklabel < 10} {
	tclgr send yaxis label place spec
	tclgr send yaxis label place 0.043,0
    }
#
# Get user overrides from ma.gr
#
    global env
    if {!$local_gr} {
	if {[file exists $env(SOLAR_LIB)/qtn.gr]} {
	    set mpathname [glob $env(SOLAR_LIB)/qtn.gr]
	    tclgr send read \"$mpathname\"
	}
	if {[file exists ~/lib/qtn.gr]} {
	    set mpathname [glob ~/lib/qtn.gr]
	    tclgr send read \"$mpathname\"
	}
    }
    if {[file exists qtn.gr]} {
	tclgr send read \"qtn.gr\"
    } 
#
#           ***   PLOT POINTS   ***
#
    foreach point $all_points {
	set x [lindex $point 0]
	set y [lindex $point 1]

	tclgr send g$graphnum.s$setnum point $x,$y
	ifdebug ifverbmax puts "graphing g$graphnum.s$setnum point $x,$y"
    }
#
#           ***     PLOT MARKERS    ***
#
    set max_markers 42
    set max_ticks 100
    set max_ticks_modified_xmgr 1000
    if {!$nolabels || !$noticks} {

# Determine if modified xmgr is available, allows 1000 ticks

	set modified_xmgr 0
	set u_name [string tolower [exec uname]]
	if {![string compare $u_name sunos] ||  ![string compare $u_name osf1] ||  [string match *linux* $u_name] ||  [string match *irix* $u_name]} {
	    set modified_xmgr 1
	    set max_ticks $max_ticks_modified_xmgr
	}
#
# See if we can handle this number of markers in any way
#
	set num_markers [llength $marker_labels]
	ifdebug puts "There appear to be $num_markers markers"
	if {$num_markers> $max_ticks} {
	    if {!$modifed_xmgr} {
		ifdebug puts "Modified version of XMGR not available on this system."
	    }
	    set nolabels 1
	    set noticks 1
	    ifdebug puts "Too many many marker ticks to display..."
        }
    }
    if {!$nolabels || !$noticks} {
#
# Remove marker labels down to maximum we can handle
#
	if {!$nolabels && $num_markers > $max_markers} {
	    ifdebug puts "Removing marker labels down to $max_markers"
	    while {$num_markers > $max_markers} {
		set marker [lindex $marker_labels 0]
		set lowest_value [lindex $marker 0]
		set lowest_index 0
		for {set i 0} {$i < $num_markers} {incr i} {
		    set marker [lindex $marker_labels $i]
		    set value [lindex $marker 0]
		    if {$value < $lowest_value} {
			set lowest_value $value
			set lowest_index $i
		    }
		}
		set marker_labels [lreplace $marker_labels  $lowest_index $lowest_index]
		set num_markers [expr $num_markers - 1]
	    }
	}
	if {!$nolabels} {
#
# Move remaining marker labels around to fit
# This should only take 1.5 passes with current algorithm
#   (old algorithm repeated until no more moves)

	    set total_x $delta_x_unscaled
	    set starting_x $min_x_unscaled
	    set min_distance [format %.9f [expr 0.0241 * $total_x]]
	    set markers_moved_left 0
	    set markers_moved_right 0
	    set pass_count 0
	    set max_pass_count 1.5
	    while {$pass_count<1 ||  $markers_moved_left>0|| $markers_moved_right>0} {
		incr pass_count
		set markers_moved_left 0
		set markers_moved_right 0

# Scan left to right, moving "next_marker" right if necessary

		for {set i -1} {$i < $num_markers - 1} {incr i} {
		    if {$i == -1} {
			set nloc [format %.10f [expr $starting_x - $min_distance]]
		    } else {
			set nloc [lindex [lindex $marker_labels $i] 0]
		    }
		    set next_marker [lindex $marker_labels [expr $i + 1]]
		    set next_nloc [lindex $next_marker 0]
		    set distance [format %.10f [expr $next_nloc - $nloc]]
		    if {$distance < $min_distance} {

# Move next marker by 1/2 distance required if this is the first pass
#  (makes offsets more symmetrical)
# Move next marker by entire distance required if this is subsequent pass
# Must maintain rounding to 0.001 precision

			set move_distance [expr $min_distance - $distance]
			if {$pass_count == 1} {
			    set move_distance [expr $move_distance / 2]
			}
			set move_distance [format %.10f $move_distance]
			set new_position [expr $next_nloc + $move_distance]
			set new_position [format %.10f $new_position]
			set j [expr $i + 1]
			set next_marker  [lreplace $next_marker 0 0 $new_position]
			set marker_labels  [lreplace $marker_labels $j $j $next_marker]
			incr markers_moved_right
		    }
		}
#
# This has been proven to take no more than 1.5 passes
#
		if {$pass_count > $max_pass_count} {
		    break
		}
#
# Scan right to left, moving next_marker right if necessary
#
		for {set i $num_markers} {$i > 0} {set i [expr $i - 1]} {
		    if {$i == $num_markers} {
			set nloc [format %.10f [expr $max_x_unscaled +  $min_distance]]
		    } else {
			set nloc [lindex [lindex $marker_labels $i] 0]
		    }
		    set next_marker [lindex $marker_labels [expr $i - 1]]
		    set next_nloc [lindex $next_marker 0]
		    set distance [format %.10f [expr $nloc - $next_nloc]]
		    if {$distance < $min_distance} {
			set move_distance [expr $min_distance - $distance]
			set move_distance [format %.10f $move_distance]
			set new_position [expr $next_nloc - $move_distance]
			set new_position [format %.10f $new_position]
			set j [expr $i - 1]
			set next_marker  [lreplace $next_marker 0 0 $new_position]
			set marker_labels [lreplace $marker_labels $j $j  $next_marker]
			incr markers_moved_left
		    }
		}
	    }
	}

	ifdebug puts "Beginning to draw"

# Draw marker ticks

	proc view_cx {worldc} {
	    if {1} {
		return $worldc
	    } else {
#		upvar min_x min_x
#		upvar max_x max_x		set 
		set min_x -1100
		set max_x 1100
		return [expr 0.15 + (0.7 * ($worldc-double($min_x)) /  ($max_x-double($min_x)))]
	    }
	}

	set vertical_ticks 0

# Vertical ticks only available for Solaris and Alpha versions
#   requires a recompile of xmgr for linux...not done yet

	set tickb [expr $max_y * (1.0 - $prop_screen_for_mticks)]
	set llb [expr $tickb - ($max_y * $prop_screen_for_mleaders)]
	set textt [expr $llb - ($max_y * $prop_screen_for_ttext)]
	ifdebug puts "max_y: $max_y  tickb: $tickb  llb: $llb   textt: $textt"
	ifdebug puts "Drawing marker ticks"
	if {$modified_xmgr || $nolabels} {
	    set vertical_ticks 1
	    foreach mloc $marker_ticks {
		set mloc [lindex $mloc 0]
		tclgr send with line
		tclgr send line loctype world
		tclgr send line [view_cx $mloc], $max_y, [view_cx $mloc],$tickb
		tclgr send line def
	    }
        }

# Write marker labels (when they fit)

	if {!$nolabels} {
	    set last_nloc -100
	    set num_markers [llength $marker_labels]
	    for {set i 0} {$i < $num_markers} {incr i} {
		set marker_label [lindex $marker_labels $i]
		set mname [lindex $marker_label 1]
		set mloc [lindex $marker_label 2]
		set nloc [lindex $marker_label 0]

# setup leader line for label 

		tclgr send with line
		tclgr send line loctype world
		if {$vertical_ticks} {
		    tclgr send line [view_cx $mloc],$tickb,[view_cx $nloc],$llb
		} else {
		    tclgr send line [view_cx $mloc],$max_y,[view_cx $nloc],$llb
		}
		tclgr send line def
		
		# setup marker label
		
		tclgr send with string
		tclgr send string on
		tclgr send string loctype world
		tclgr send string g$graphnum
		tclgr send string [view_cx $nloc], $textt
		tclgr send string rot 270
		tclgr send string char size 0.60
		tclgr send string def \"$mname\"
	    }
	}
    }

# Done with markers

# DRAW !!!

    tclgr send redraw

    return ""

}

# solar::plotqtld
#
# Purpose:  Plot qtld (qtld.out)
#
# Usage:    plotqtld <type> [-nolabels] [-nomarkers] [-file filename] [-local]
#
#           <type> is one of: strat, mgeno, qtdt, qtld
#
#           -nolabels     do not include "marker" labels (ticks only)
#           -nomarkers    do not include marker ticks or labels
#           -file         Use named file instead of qtldm.out in outdir
#           -local        Ignore default plot parameters; use only local file
#
# Notes:    You must select the trait or outdir first.
#
#           The plot parameter file (in SOLAR_LIB) is qtld.gr.  You
#           may override it with a copy in ~/lib or your working
#           directory.  Your version need only include the parameters
#           you would like to change.  This should work in most cases.
#           If you specify -local, however, the qtld.gr in SOLAR_LIB
#           is completely ignored, and your qtld.gr must be complete,
#           which might get around some very obscure conflict between
#           the two plot parameter files.
#
# -

proc plotqtld {type args} {


# Important constants
    set x_expansion_factor 1.12  ;# Expansion of X beyond input range

    set debug [solardebug]

    set marker_margin 4.5
    set nolabels 0
    set noticks 0
    set filename ""
    set local_gr 0

    set badargs [read_arglist $args \
		     -nolabels {set nolabels 1} \
		     -nomarkers {set nolabels 1; set noticks 1} \
		     -marker_margin marker_margin \
		     -file filename \
		     -local {set local_gr 1} \
		]
    if {"" != $badargs} {
	error "plotqtn: Invalid arguments: $badargs"
    }
    if {"" == $filename} {
	set filename [full_filename qtld.out]
    }

    if {$local_gr && $debug} {puts "Warning!  Ignoring qtld.gr IN SOLAR_BIN"}

    if {$type == "strat"} {
	set tindex 2
    } elseif {$type == "geno" || $type == "mgeno"} {
	set tindex 3
    } elseif {$type == "qtdt"} {
	set tindex 4
    } elseif {$type == "qtld"} {
	set tindex 5
    } else {
	error "type must be strat, mgeno, qtdt, or qtld"
    }


# Read qtld.out

    set marker_labels {}
    set all_points {}
    set max_mrk_name_len 0
    set first 1
    set qfile [open $filename]
    set lines 0
    while {-1 != [gets $qfile line]} {
	if {0 == [llength $line]} {
	    continue
	}
	incr lines

	set name [lindex $line 1]
	set pos [map locn $name]
	set lp [lindex $line $tindex]
	set lp [expr 0 - log10($lp)]

	if {$first} {
	    set min_pos $pos
	    set max_pos $pos
	    set min_lp $lp
	    set max_lp $lp
	    set first 0
	}
	lappend all_points [list $pos $lp]
	lappend marker_labels [list $pos $name $pos]
	if {[string length $name] > $max_mrk_name_len} {
	    set max_mrk_name_len [string length $pos]
	}
	if {$pos < $min_pos} {
	    set min_pos $pos
	}
	if {$pos > $max_pos} {
	    set max_pos $pos
	}
	if {$lp < $min_lp} {
	    set min_lp $lp
	}
	if {$lp > $max_lp} {
	    set max_lp $lp
	}
    }
    close $qfile
    if {$lines < 2} {
	error "File $filename contains only $lines lines"
    }

    ifdebug puts "min_pos $min_pos max_pos $max_pos"
    ifdebug puts "min_lp $min_lp max_lp $max_lp"

# Sort points if required (yes)

    proc sort_by_first_num {a b} {
	set a0 [lindex $a 0]
	set b0 [lindex $b 0]
	if {$b0 > $a0} {
	    return -1
	} elseif {$b0 < $a0} {
	    return 1
	}
	return 0
    }

    set all_points [lsort -command sort_by_first_num $all_points]


# Sort marker ticks and labels too

    set marker_ticks $marker_labels
    set marker_ticks [lsort $marker_ticks]
    set marker_labels [lsort -command sort_by_first_num $marker_labels]

# Begin plotting...

    set setnum 1
    set graphnum 1

# Open new or existing tclgr session

    if {[catch {tclgr open} errmsg]} {
	if {[string compare $errmsg  "tclgr session already opened from this solar session"]} {
	    error $errmsg
	}
    }

# Kill previous set, and graph if not overlaying

    tclgr send kill graphs
    tclgr send clear line
    tclgr send clear string
    tclgr send focus g$graphnum

# Calculate scaling and ticks for x

    set max_x_unscaled $max_pos
    set min_x_unscaled $min_pos
    set delta_x_unscaled [expr $max_x_unscaled - $min_x_unscaled]
    set center_x [expr ($min_pos + $max_pos) / 2]
    ifdebug puts "delta_x_unscaled: $delta_x_unscaled  center_x:  $center_x"

    set mmlist [majorminor $delta_x_unscaled $min_x_unscaled $max_x_unscaled]
    set major_x [lindex $mmlist 0]
    set minor_x [lindex $mmlist 1]
    ifdebug puts "major_x: $major_x  minor_x: $minor_x"

    set delta_x [expr $delta_x_unscaled * $x_expansion_factor]
    set max_x [expr $center_x + ($delta_x / 2.0)]
    set min_x [expr $center_x - ($delta_x / 2.0)]
    set use_x [expr ($min_x + $min_x_unscaled) / 2.0]
    set firstlabel_x [expr $major_x * ceil ($use_x / $major_x)]
    ifdebug puts "xmin: $min_x   xmax: $max_x   firstlabel: $firstlabel_x"

# Setup scaling and ticks for x

    tclgr send world xmin $min_x
    tclgr send world xmax $max_x
    tclgr send xaxis tick op bottom
    tclgr send xaxis tick major $major_x
    tclgr send xaxis tick minor $minor_x

    tclgr send xaxis ticklabel op bottom
    tclgr send xaxis ticklabel start type spec
    tclgr send xaxis ticklabel start $firstlabel_x

    if {abs($min_x) <= 2.0e9 && abs($max_x) <= 2.0e9} {
	tclgr send xaxis ticklabel format decimal
	tclgr send xaxis ticklabel prec 0
    }

# Set up y tick marks (major and minor)

#    ************* Set up Y Axis *****************


# This fiendishly complicated method to set up y axis is borrowed from
# proc plot.  It sets up nice major and minor ticks consistent with xmgr
# regardless of size, following a 1/2/5 rule.

# The design maximum Log10(P) is 2000
#   (highest tick range is therefore 750-1500)
#   (I don't like limits, but this is way beyond anything I can imagine)
# maxy must be 4*10^N or 1*10^N
# the other values are related

    set maxy  1000
    set major 200
    set major_digit 2
    set minor 100

    set maxdiv 2.5
    set majdiv 2
    set mindiv 2

# Since we mess with global variable tcl_precision
# We must catch errors to be sure it gets restored

    global tcl_precision
    set save_tcl_precision $tcl_precision
    set tcl_precision 6
    set not_ok [catch {
    while {1} {
	if {$max_lp > $maxy} {
	    ifdebug puts "Y Axis Major: $major Minor: $minor"
	    tclgr send yaxis tick major $major
	    tclgr send yaxis tick minor $minor
	    break
	}
	set maxy [expr double($maxy) / $maxdiv]
	set major [expr double($major) / $majdiv]
	set minor [expr double($minor) / $mindiv]

	if {$major_digit == 2} {
	    set major_digit 1
	    set maxdiv 2
	    set majdiv 2
	    set mindiv 2
	} elseif {$major_digit == 1}  {
	    set major_digit 5
	    set maxdiv 2
	    set majdiv 2.5
	    set mindiv 2.5
	} else {
	    set major_digit 2
	    set maxdiv 2.5
	    set majdiv 2
	    set mindiv 2
	}
    }
    } caught_error]
    set tcl_precision $save_tcl_precision
    if {$not_ok} {
	error $caught_error
    }

# Create y expansion factor to permit showing markers labels at top

    if {$nolabels || $noticks} {
	set max_mrk_name_len 0
    }
    set marker_margin 2.0
    set chars_that_fit_vertically 55.0  ;# using default font size (was 58)
    set prop_screen_for_markers [expr ($max_mrk_name_len + $marker_margin)  / $chars_that_fit_vertically]
    set prop_screen_for_mticks 0.04
    set prop_screen_for_mleaders 0.04
    set prop_screen_for_ttext 0.015
    set prop_screen_for_plot [expr 1.0 - ($prop_screen_for_mticks +  $prop_screen_for_mleaders +  $prop_screen_for_ttext  +  $prop_screen_for_markers )]
    set expand_y [expr 1.0 / $prop_screen_for_plot]
    ifdebug puts "expand_y is $expand_y"

# Set up y axis and ticks

    set max_y_unscaled $max_lp
    set min_y_unscaled $min_lp

    set max_y_ticklabel [expr $major*ceil(double($max_lp)/$major)]
    ifdebug puts "max_y_ticklabel: $max_y_ticklabel"
    set max_y [expr $max_y_ticklabel * $expand_y]
    set min_y [expr -0.06 * $max_y]
    tclgr send world ymin $min_y
    tclgr send world ymax $max_y
    tclgr send yaxis ticklabel start type spec
    tclgr send yaxis ticklabel stop type spec
    tclgr send yaxis ticklabel start 0
    tclgr send yaxis ticklabel stop $max_y_ticklabel
#
# xmgr spaces Y axis label too close to single digits
#
    if {$max_y_ticklabel > 4 && $max_y_ticklabel < 10} {
	tclgr send yaxis label place spec
	tclgr send yaxis label place 0.043,0
    }
#
# Get user overrides from ma.gr
#
    global env
    if {!$local_gr} {
	if {[file exists $env(SOLAR_LIB)/qtld.gr]} {
	    set mpathname [glob $env(SOLAR_LIB)/qtld.gr]
	    tclgr send read \"$mpathname\"
	}
	if {[file exists ~/lib/qtld.gr]} {
	    set mpathname [glob ~/lib/qtld.gr]
	    tclgr send read \"$mpathname\"
	}
    }
    if {[file exists qtld.gr]} {
	tclgr send read \"qtld.gr\"
    } 
#
#           ***   PLOT POINTS   ***
#
    foreach point $all_points {
	set x [lindex $point 0]
	set y [lindex $point 1]

	tclgr send g$graphnum.s$setnum point $x,$y
	ifdebug ifverbmax puts "graphing g$graphnum.s$setnum point $x,$y"
    }
#
#           ***     PLOT MARKERS    ***
#
    set max_markers 42
    set max_ticks 100
    set max_ticks_modified_xmgr 1000
    if {!$nolabels || !$noticks} {

# Determine if modified xmgr is available, allows 1000 ticks

	set modified_xmgr 0
	set u_name [string tolower [exec uname]]
	if {![string compare $u_name sunos] ||  ![string compare $u_name osf1] ||  [string match *linux* $u_name] ||  [string match *irix* $u_name]} {
	    set modified_xmgr 1
	    set max_ticks $max_ticks_modified_xmgr
	}
#
# See if we can handle this number of markers in any way
#
	set num_markers [llength $marker_labels]
	ifdebug puts "There appear to be $num_markers markers"
	if {$num_markers> $max_ticks} {
	    if {!$modifed_xmgr} {
		ifdebug puts "Modified version of XMGR not available on this system."
	    }
	    set nolabels 1
	    set noticks 1
	    ifdebug puts "Too many many marker ticks to display..."
        }
    }
    if {!$nolabels || !$noticks} {
#
# Remove marker labels down to maximum we can handle
#
	if {!$nolabels && $num_markers > $max_markers} {
	    ifdebug puts "Removing marker labels down to $max_markers"
	    while {$num_markers > $max_markers} {
		set marker [lindex $marker_labels 0]
		set lowest_value [lindex $marker 0]
		set lowest_index 0
		for {set i 0} {$i < $num_markers} {incr i} {
		    set marker [lindex $marker_labels $i]
		    set value [lindex $marker 0]
		    if {$value < $lowest_value} {
			set lowest_value $value
			set lowest_index $i
		    }
		}
		set marker_labels [lreplace $marker_labels  $lowest_index $lowest_index]
		set num_markers [expr $num_markers - 1]
	    }
	}
	if {!$nolabels} {
#
# Move remaining marker labels around to fit
# This should only take 1.5 passes with current algorithm
#   (old algorithm repeated until no more moves)

	    set total_x $delta_x_unscaled
	    set starting_x $min_x_unscaled
	    set min_distance [format %.9f [expr 0.0241 * $total_x]]
	    set markers_moved_left 0
	    set markers_moved_right 0
	    set pass_count 0
	    set max_pass_count 1.5
	    while {$pass_count<1 ||  $markers_moved_left>0|| $markers_moved_right>0} {
		incr pass_count
		set markers_moved_left 0
		set markers_moved_right 0

# Scan left to right, moving "next_marker" right if necessary

		for {set i -1} {$i < $num_markers - 1} {incr i} {
		    if {$i == -1} {
			set nloc [format %.10f [expr $starting_x - $min_distance]]
		    } else {
			set nloc [lindex [lindex $marker_labels $i] 0]
		    }
		    set next_marker [lindex $marker_labels [expr $i + 1]]
		    set next_nloc [lindex $next_marker 0]
		    set distance [format %.10f [expr $next_nloc - $nloc]]
		    if {$distance < $min_distance} {

# Move next marker by 1/2 distance required if this is the first pass
#  (makes offsets more symmetrical)
# Move next marker by entire distance required if this is subsequent pass
# Must maintain rounding to 0.001 precision

			set move_distance [expr $min_distance - $distance]
			if {$pass_count == 1} {
			    set move_distance [expr $move_distance / 2]
			}
			set move_distance [format %.10f $move_distance]
			set new_position [expr $next_nloc + $move_distance]
			set new_position [format %.10f $new_position]
			set j [expr $i + 1]
			set next_marker  [lreplace $next_marker 0 0 $new_position]
			set marker_labels  [lreplace $marker_labels $j $j $next_marker]
			incr markers_moved_right
		    }
		}
#
# This has been proven to take no more than 1.5 passes
#
		if {$pass_count > $max_pass_count} {
		    break
		}
#
# Scan right to left, moving next_marker right if necessary
#
		for {set i $num_markers} {$i > 0} {set i [expr $i - 1]} {
		    if {$i == $num_markers} {
			set nloc [format %.10f [expr $max_x_unscaled +  $min_distance]]
		    } else {
			set nloc [lindex [lindex $marker_labels $i] 0]
		    }
		    set next_marker [lindex $marker_labels [expr $i - 1]]
		    set next_nloc [lindex $next_marker 0]
		    set distance [format %.10f [expr $nloc - $next_nloc]]
		    if {$distance < $min_distance} {
			set move_distance [expr $min_distance - $distance]
			set move_distance [format %.10f $move_distance]
			set new_position [expr $next_nloc - $move_distance]
			set new_position [format %.10f $new_position]
			set j [expr $i - 1]
			set next_marker  [lreplace $next_marker 0 0 $new_position]
			set marker_labels [lreplace $marker_labels $j $j  $next_marker]
			incr markers_moved_left
		    }
		}
	    }
	}

	ifdebug puts "Beginning to draw"

# Draw marker ticks

	proc view_cx {worldc} {
	    if {1} {
		return $worldc
	    } else {
#		upvar min_x min_x
#		upvar max_x max_x		set 
		set min_x -1100
		set max_x 1100
		return [expr 0.15 + (0.7 * ($worldc-double($min_x)) /  ($max_x-double($min_x)))]
	    }
	}

	set vertical_ticks 0

# Vertical ticks only available for Solaris and Alpha versions
#   requires a recompile of xmgr for linux...not done yet

	set tickb [expr $max_y * (1.0 - $prop_screen_for_mticks)]
	set llb [expr $tickb - ($max_y * $prop_screen_for_mleaders)]
	set textt [expr $llb - ($max_y * $prop_screen_for_ttext)]
	ifdebug puts "max_y: $max_y  tickb: $tickb  llb: $llb   textt: $textt"
	ifdebug puts "Drawing marker ticks"
	if {$modified_xmgr || $nolabels} {
	    set vertical_ticks 1
	    foreach mloc $marker_ticks {
		set mloc [lindex $mloc 0]
		tclgr send with line
		tclgr send line loctype world
		tclgr send line [view_cx $mloc], $max_y, [view_cx $mloc],$tickb
		tclgr send line def
	    }
        }

# Write marker labels (when they fit)

	if {!$nolabels} {
	    set last_nloc -100
	    set num_markers [llength $marker_labels]
	    for {set i 0} {$i < $num_markers} {incr i} {
		set marker_label [lindex $marker_labels $i]
		set mname [lindex $marker_label 1]
		set mloc [lindex $marker_label 2]
		set nloc [lindex $marker_label 0]

# setup leader line for label 

		tclgr send with line
		tclgr send line loctype world
		if {$vertical_ticks} {
		    tclgr send line [view_cx $mloc],$tickb,[view_cx $nloc],$llb
		} else {
		    tclgr send line [view_cx $mloc],$max_y,[view_cx $nloc],$llb
		}
		tclgr send line def
		
		# setup marker label
		
		tclgr send with string
		tclgr send string on
		tclgr send string loctype world
		tclgr send string g$graphnum
		tclgr send string [view_cx $nloc], $textt
		tclgr send string rot 270
		tclgr send string char size 0.60
		tclgr send string def \"$mname\"
	    }
	}
    }

# Done with markers

# DRAW !!!

    tclgr send redraw

    return ""

}


# solar::majorminor -- private
#
# Purpose:  Determine best major and minor ticks given a range
#           Range must be positive, but can be < 1 or exponential
#
# Usage:    majorminor <range> [<xmin>] [<xmax>]
#
#           <range> is range (e.g. xmin - xmax)
#           <xmin> is min X value, specify if X only
#           <xmax> is max X value, specify if X only
#
# Returns:  list of major and minor tick distances
#
# Note:  Of course this embodies a particular style.  In this case,
#        major ticks begin with either 1, 2, or 5, and they are divided
#        by 2 or 5 minor ticks.  It is intended that there be from 3-7
#        major ticks, which is good for an X axis, not necessarily so
#        good for the Y axis where more ticks might be desired.  (In
#        actual practice, if range is expanded, there might be up to
#        8 major ticks.)
#
#        xmin,xmax is used for xaxis only.
#        This is used to allow extra spacing for very big numbers.
#        Starting when any abx(xmin,xmax) > 10^6, fudge factors
#        are multiplied into the range to provide more spacing:
#
#           abs(xmin,xmax) < 10^6             1.0 (no fudging)
#           abs(xmin,xmax) >= 10^6            1.0 + 0.1 for each digit past 5
# -

proc majorminor {range {xmin 1} {xmax 1}} {

    ifdebug puts "Entering majorminor"
    set debug [solardebug]

    set xmax [expr abs($xmax)]
    set xmin [expr abs($xmin)]
    if {$xmin > $xmax} {
	set xmax $xmin
    }

    if {$xmax >= 1.0e6} {
	set fudge [expr 1.0 + ((log10(1.0*$xmax) - 5.0) * 0.1)]
	if {$debug} {puts "fudging range by $fudge"}
	set range [expr $range * $fudge]
    }

# Get first sig digit
# and, if possible, second sig digit

    set rlen [string length $range]
    set first_digit ""
    for {set i 0} {$i < $rlen} {incr i} {
	set s [string index $range $i]
	if {"." != $s} {
	    if {0 != $s} {
		set first_digit $s
		set second_digit [string index $range [expr $i + 1]]
		if {"e" == $second_digit} {
		    set second_digit ""
		} elseif {"." == $second_digit} {
		    set second_digit [string index $range [expr $i + 2]]
		}
		break
	    }
	}
    }
    if {"" == $first_digit} {
	error "Can't plot when X range is 0"
    }

    if {$debug} {puts "first,second digits are $first_digit,$second_digit"}

    set factor [expr pow (10, floor (log10 ($range)))]
    if {$factor < 1} {
	set factor [format %.1g $factor]
    }
	
    if {$debug} {puts "factor is $factor"}

    while {1} {
	set major .2
	set minor .1
	if {$first_digit == 1} {
	    if {"" != $second_digit} {
		if {$second_digit >= 5} {
		    set major .5
		}
		if {$second_digit >= 8} {
		    set minor .1
		}
	    }
	    break
	}
	if {$first_digit == 2} {
	    set major .5
	    set minor .1
	    break
	}
	if {$first_digit < 7} {
	    set major 1
	    set minor .5
	    break
	}
	set major 2
	set minor 1
	break
    }

    set major [expr $major * $factor]
    set minor [expr $minor * $factor]
    if {$major >= 1} {
	set major [expr round ($major)]
    } else {
	set major [format %.1g $major]
    }
    if {$minor > 3 || $minor == 1} {
	set minor [expr round ($minor)]
    } else {
	set minor [format %.2g $minor]
    }
    if {$debug} {puts "major: $major  minor: $minor"}
    return [list $major $minor]
}


# solar::stepfor --
#
# Purpose: Foward stepwise covariate screening
#
# Usage:   stepfor [-list listfile] [-list list] [-verbose] [-v]
#                  [-fix listfile]  [-fix fixlist] [-max maxdim]
#                  [-p pvalue] [-test othertest] [-par] [-parclean]
#
#          stepclean     ;# Remove fully_typed covariate and unload file
#
#          By default, stepfor will test all covariates in the current
#          model, testing them all and then fixing the best one, and then
#          repeating the process until the best one does not meet the
#          default pvalue of 0.05, or user specified p-value or test (see
#          below).  The final model will contain all the covariates which met
#          the screening test.  A file named stepfor.out is written to the
#          output directory with all the loglikelihoods, and a file named
#          stepfor.history is written with other information.  All of the
#          best models for each number of covariates are saved as
#          stepfor.null<i> where <i> is the number of tested covariates.
#
#          To ensure that all models use the same sample, a new file named
#          fully_typed.out is created in the output directory which
#          defines a variable named "fully_typed" for each fully typed
#          individual.  This file is added to the list of open phenotypes
#          files, and the variable "fully_typed" is added to the
#          model as a "null" covariate which has no effect on the model
#          other than restricting the sample to fully typed individuals.
#
#          To remove the fully_typed covariate and unload the fully_typed.out
#          phenotypes file, give the command "stepclean" after stepfor has
#          completed.
#
#          -list listfile    listfile is a file containing a list of all
#                            covariates to be tested, one on each line.
#                            The filename cannot contain spaces.  These
#                            covariates may or may not be in the model when
#                            the command is given.  If the -list option is
#                            specified, all other covariates in the starting
#                            model are automatically fixed.
#          -list list        Alternatively, a Tcl list of covariates to
#                            be tested can be specified.  Tcl lists are
#                            space delimited and enclosed in quotes or curly
#                            braces.
#
#          -fix list         list is a Tcl list of covariates to be
#                            included in every model and not tested.  Their
#                            values will be estimated by maximum likelihood
#                            for every model, unless you constrain them.
#                            These covariates may or may not in the model
#                            when the command is given.  For -fix, a list
#                            could be simply one phenotype, and that
#                            supercedes a file with the same name.
#          -fix listfile     Alternatively, a file containing a list of all
#                            covariates to be included in every model may
#                            be specified.  The filename cannot contain
#                            spaces.  The list of covariates to be fixed
#                            will supercede the list of covariates to be
#                            tested if the same covariate occurs on both
#                            lists, however a warning will be given.
#
#           -p pvalue        pvalue is the highest p value allowed for
#                            a covariate to be included.  The default is 0.05.
#
#           -max maxdim      maxdim is the maximum number of test covariates
#                            to be included in a model (the maximum dimension).
#
#           -verbose         Show maximization output during maximizations.
#           -v               Same as -verbose
#
#          -par              New and EXPERIMENTAL!  This option turns on Parallel
#                            processing on the SFBR GCC Compute Ranch.
#                            WARNING!  Do not run more than one instance of
#                            stepfor -par from the same working directory.
#                            Parallel stepfor will use many (but not all!) ranch
#                            machines, and access for other users and jobs may
#                            be delayed due to gridware thrashing.  The usual
#                            output is not printed to the terminal to save time
#                            but numerous parallel status messages are printed
#                            to help the developers make this program better.
#                            The parallel operation is automatic and the
#                            parallel status messages may be ignored by most
#                            users most of the time unless there is no output
#                            for more than fifteen minutes.  Note: If model
#                            includes linkage element matrices loaded from
#                            some mibddir, those matrices should be relocated
#                            to the working directory, or specified with an
#                            absolute pathname in the model file.  This is
#                            because in parallel operation the model is loaded
#                            not in the current working directory but in a
#                            subdirectory of /tmp.
#
#          -parclean         Normally, parallel stepfor cleans up after itself.
#                            However, if it is necessary to force a shutdown
#                            of a parallel stepfor, normal cleanup is not
#                            done.  "stepfor -parclean" cleans up all the
#                            junk stepfor files in /tmp directories on all
#                            ranch machines.  This must be run on medusa.  Do
#                            not run if you have any other running parallel
#                            jobs (parallel stepfor, parallel bayesavg, or any
#                            parallel job using "launch" or "doscript") as
#                            their files may be deleted too.
#                            See also "doranch" for other ranch cleanup
#                            procedures.  Cleanup history is written to a file
#                            named cleantmp.out.
#
#           -test othertest  othertest is a user defined Tcl proc that judges
#                            whether or not a covariate should be included.
#                            The test model with the best covariate is loaded
#                            at the time this procedure is called.  This
#                            procedure takes two mandatory arguments (whether
#                            they are needed by the procedure or not).
#
#                            loglike0 nullmodelname
#
#                            loglike0 is the loglikelihood of the null model
#                            which does not contain the current test covariate.
#                            nullmodelname is the pathname to the null model
#                            itself.  The procedure may obtain the
#                            loglikelihood of the current model with the
#                            loglike command.  The default procedure looks
#                            like this:
#
#                        proc stepfortest {loglike0 nullmodel} {
#                            set chisq [expr 2.0 * ([loglike] - $loglike0)]
#                            if {$chisq >= 0} {
#                              	 set pvalue [chi -number $chisq 1]
#                            } else {
#	                         set pvalue 1
#                            }
#                            set pvalue [chi -number $chisq 1]
#                            putsout stepfor.history "\n    *** p = $pvalue"
#                            global SOLAR_stepfor_pvalue
#                            if {$pvalue <= $SOLAR_stepfor_pvalue} {
#                                return 1
#                            }
#                            return 0
#                        }
#
#                            Note that the default procedure does not use
#                            the nullmodel argument, but it does use a
#                            global variable that you will not have to use.
#                            The global supports the -p argument.  The
#                            procedure may load the nullmodel without
#                            restoring the current model; that is handled
#                            by the stepfor procedure itself.
#          
# -

proc stepfor {args} {
#
# Do parallel cleanup if requested
#
    if {$args == "-parclean"} {
	set uname [lindex [exec who -m] 0]
	return [doranch cleantmp $uname.]
    }
#
# Save initial model and delete old files
#
    save model [full_filename stepfor.orig]
    file delete [full_filename stepfor.history]
    file delete [full_filename stepfor.out]
    set modfiles [glob -nocomplain [full_filename stepfor.null*.mod]]
    foreach modfile $modfiles {
	file delete $modfile
    }
#
# read arguments
#
    set stepverbosity -q
    set pvalue 0.05
    set maxdim 0
    set listfile {}
    set fixlist {}
    set ucovlist {}
    set stepfortest stepfortest    ;# the default test, not a typo
    set parallel 0
    set badargs [read_arglist $args \
		     -list listfile \
		     -fix fixlist \
		     -p pvalue \
		     -test stepfortest \
		     -verbose {set stepverbosity ""} \
		     -v {set stepverbosity ""} \
		     -max maxdim \
		     -par {set parallel 1} \
		     ]
    if {{} != $badargs} {
	error "stepfor: Invalid argument $badargs"
    }
    global SOLAR_stepfor_pvalue
    set SOLAR_stepfor_pvalue $pvalue
#
# Get list of covariates to test
#
    if {{} != $listfile} {
	if {-1 == [string first " " $listfile]} {
	    set covlist [listfile $listfile]
	} else {
	    set covlist $listfile
	}
	set ucovlist $covlist
    } else {
	set covlist [covar]
    }
#
# Because a one element fixlist is possible
# If one element is specified, first check if it is a phenotype
#	
    if {{} != $fixlist} {
	if {-1 == [string first " " $fixlist]} {
	    if {-1 == [lsearch -exact [concat sex [lrange [phenotypes] \
						       1 end]] $fixlist]} {
		if {![file exists $fixlist]} {
		    error "stepfor: No phenotype or file named $fixlist"
		}
		set fixlist [listfile $fixlist]
	    }
	}
    }
#
# Be sure fully_typed() is incorporated into fixlist at least once
# and only once
#
    if {-1 == [lsearch -exact $fixlist fully_typed()]} {
	lappend fixlist fully_typed()
    }
#
# Now remove fixed covariates from list of covariates to test
#
    foreach fix $fixlist {
	if {-1 != [set foundpos [lsearch -exact $covlist $fix]]} {
	    set covlist [lreplace $covlist $foundpos $foundpos]
	    if {{} != $ucovlist} {
		if {-1 == [lsearch -exact $ucovlist $fix]} {
		    putsout stepfor.history "Warning.  Removing $fix from test list because fixed"
		}
	    }
	}
    }
#
# Show list of covariates
    set n [llength $covlist]
    set printlist $covlist
    set maxline 78
    while {{} != $printlist} {
	set line "    [lindex $printlist 0]"
	set printlist [lrange $printlist 1 end]
	while {{} != $printlist} {
	    set next [lindex $printlist 0]
	    if {$maxline < [expr [string length $line]+2+[string length $next]\
			       ]} {
		break
	    }
	    set line "$line $next"
	    set printlist [lrange $printlist 1 end]
	}
	putsout stepfor.history $line
    }
    putsout stepfor.history "\n    *** Testing $n covariates"
#
# Add fixed covariates (if not already added) except for fully_typed
#   (fully_typed gets added later)
#
    foreach cov $fixlist {
	if {$cov != "fully_typed()"} {
	    covariate $cov
	}
    }
#
# Add listed covariates (if not already added) and then delete (later).
# This ensures covariates CAN be added, and that they are now deleted.
#
    foreach cov $covlist {
	covariate $cov
    }
#
# If no omega, make it polygenic
#
    if {[omega] == \
	    "omega = Use_polygenic_to_set_standard_model_parameterization"} {
	putsout "No predefined omega; defaulting to polygenic"
	polymod
    }
#
# Now that all covariates have been added to model
# create fully_typed.out file
#
    eval maximize $stepverbosity -sampledata
    set insample [solarfile open [full_filename sampledata.out]]
    solarfile $insample start_setup
    solarfile $insample setup id
    set outsample [open [full_filename fully_typed.out] w]
    puts $outsample "id,fully_typed"
    while {{} != [set record [solarfile $insample get]]} {
	set id [lindex $record 0]
	puts $outsample "$id,1"
    }
    solarfile $insample close
    close $outsample
#
# Removed fully_typed.out from old output directories
# and add in new fully_typed.out from current output directory
#
    set oldphenotypes [phenotype -files]
    set newphenotypes {}
    foreach oldphenotype $oldphenotypes {
	set phentail [file tail $oldphenotype]
	if {[string compare $phentail fully_typed.out]} {
	    lappend newphenotypes $oldphenotype
	}
    }
    lappend newphenotypes [full_filename fully_typed.out]
    eval load phenotypes $newphenotypes

    set oldcovariates [covariate]
    if {-1 == [lsearch -exact $oldcovariates fully_typed()]} {
	covar fully_typed()
    }
#
# OK, now delete the variable covariates
#
    foreach cov $covlist {
	covariate delete $cov
    }
#
# Maximize "null" model
#
    puts "    *** Maximizing null model\n"
    eval maximize $stepverbosity
    save model [full_filename stepfor.null0]
    set null_like0 [loglike]
    putsout stepfor.out [format "%24s   %s   %s               %s      %s" \
			     Model Loglike Covariate Chi^2 p]
    set basemodelname cov
    putsout stepfor.out [format "%24s  %.3f" $basemodelname [loglike]]
#
# Main loop
#
    set loop 1
    set nullindex -1
    set nextindex 0
    set fixcovs {}
    set early_exit 0
    set addedlist {}
    while {[llength $covlist]} {
	if {$maxdim!=0 && $nextindex>=$maxdim} {
	    putsout stepfor.history "    *** Exiting because -max $maxdim"
	    putsout stepfor.history "    *** Final covariates are: $fixcovs"
	    load model [full_filename stepfor.null$nextindex]
	    set early_exit 1
	    break
	}
	incr nullindex
	incr nextindex
	set bestlike ""
	set bestcov ""
	set covindex 0
	set testedcov 0
	if {$parallel} {
	    set bestlikeshort ""
	    set bestcovs ""
	    set number_tests [expr [llength $covlist] - $nullindex]
	    puts "Running $number_tests tests in parallel"

	    puts [exec date]
	    set trynum 1
	    set cfilename [full_filename stepfor.covlist]
	    set cfileout [open $cfilename w]
	    foreach cov $covlist {
		puts $cfileout $cov
	    }
	    close $cfileout
	    set poutfilename [full_filename stepfor.out.d$nextindex]
	    file delete -force $poutfilename
	    file delete -force $poutfilename.tmp

	    set tmodfilename [full_filename stepfor.testmods.1]
	    set tmodfile [open $tmodfilename w]
	    set covindex 0
	    foreach cov $covlist {
		incr covindex
		if {-1 != [lsearch -exact $addedlist $cov]} {
		    continue
		}
		puts $tmodfile $covindex
	    }
	    close $tmodfile

	    global STEP_recordsize
	    set STEP_recordsize 5
	    puts "calling launch"
	    step_parallel_launch stepfor $number_tests [expr $nullindex + 1] \
		$basemodelname
#
# returns only when all results written to stepfor.out.d$nextindex
#
	    set par_infile [open [full_filename stepfor.out.d$nextindex]]
	    set par_outfile [open [full_filename stepfor.out] a]
	    set covindex 0
	    foreach cov $covlist {
		incr covindex
		if {-1 != [lsearch -exact $addedlist $cov]} {
		    continue
		}
		gets $par_infile current_result
		puts $par_outfile $current_result
		set logl [lindex $current_result 1]
		set testedcov 1
		if {"" == $bestlikeshort || $logl > $bestlikeshort} {
		    set bestlikeshort $logl
		    lappend bestcovs $cov
		    set bestindex $covindex
		}
	    }
	    close $par_infile
	    close $par_outfile
	} else {

# Non-parallel

	foreach cov $covlist {
	    incr covindex
	    if {-1 != [lsearch -exact $addedlist $cov]} {
		continue
	    }
	    set testedcov 1
	    load model [full_filename stepfor.null$nullindex]
	    covariate $cov
	    eval maximize $stepverbosity
	    set newmodelname [stepname $basemodelname $covindex]

	    eval set chisq [expr 2.0 * ([loglike] - \$null_like$nullindex)]
	    if {$chisq >= 0} {
		set pval [chi -number $chisq 1]
	    } else {
		set pval 1
	    }
	    set fchisq [format %.4f $chisq]

	    if {-1 != [set epos [string first e $pval]]} {
		set mstring [string range $pval 0 5]
		set estring [string range $pval $epos end]
		set pval "$mstring$estring"
	    }

	    putsout stepfor.out \
		[fformat "%24s  %.3f  %-19s %9s  %s" \
		     $newmodelname [loglike] $cov $fchisq $pval]

	    if {"" == $bestlike} {
		set bestlike [loglike]
		set bestcov $cov
		save model [full_filename stepfor.null$nextindex]
		set bestindex $covindex
	    } elseif {[loglike] > $bestlike} {
		set bestlike [loglike]
		set bestcov $cov
		save model [full_filename stepfor.null$nextindex]
		set bestindex $covindex
	    }
	}
        }

#
# See if criterion is satisfied
#
	if {!$testedcov} {
	    break
	}
	if {!$parallel} {
	    load model [full_filename stepfor.null$nextindex]
	} else {
	    set testbestlike ""
	    foreach testbestcov $bestcovs {
		load model [full_filename stepfor.null$nullindex]
		covariate $testbestcov
		eval maximize $stepverbosity
		if {$testbestlike == "" || [loglike] > $testbestlike} {
		    set bestcov $testbestcov
		    save model [full_filename stepfor.null$nextindex]
		    set bestlike [loglike]
		}
	    }
	}	    
	set null_like$nextindex $bestlike
  	if {![eval $stepfortest \$null_like$nullindex stepfor.$nullindex]} {
	    putsout stepfor.history "    *** Best covariate $bestcov lod did not meet criterion"
	    putsout stepfor.history "    *** Final covariates are: $fixcovs"
	    load model [full_filename stepfor.null$nullindex]
	    set early_exit 1
	    break
	}
	lappend fixcovs $bestcov
	lappend addedlist $bestcov
        putsout stepfor.history "    *** Best covariate $bestcov added to model stepfor.null$nextindex\n"
	set basemodelname [stepname $basemodelname $bestindex]
	set bestmodelname stepfor.null$nextindex
    }
    if {!$early_exit} {
	putsout stepfor.history "\n    *** All covariates were added to final model (stepfor.best):\n$fixcovs"
    }

    save model [full_filename stepfor.best]
    return ""
#
# No longer bother with deleting fully_typed()...
#
#    covariate delete fully_typed()
#    save model [full_filename stepfor.best]
#    puts "\nRe-maximizing stepfor.best with largest possible sample...\n"
#    eval maximize $stepverbosity
#    save model [full_filename stepfor.best]
#
# Alternatively, we could clean up all stepfor models
#
#    for {set i 0} {$i < $bestindex} {incr i} {
#	puts "fixing model stepfor.null$i"
#	load model [full_filename stepfor.null$i]
#	covariate delete fully_typed()
#	save model [full_filename stepfor.null$i]
#    }
#    eval load phen $oldphenotypes
}

proc stepclean {} {
    catch {covariate delete fully_typed()}
    set phenfiles [phenotypes -files]
    set newphens {}
    foreach phenfile $phenfiles {
	if {0!=[string compare [file tail $phenfile] fully_typed.out]} {
	    lappend newphens $phenfile
	}
    }
    if {{} == $newphens} {
	catch {load phenotypes ""}
    } else {
	eval load phenotypes $newphens
    }
    return ""
}


proc stepfortest {loglike0 nullmodel} {
    set chisq [expr 2.0 * ([loglike] - $loglike0)]
    putsout stepfor.history "\n    *** chisq = [format %.8g $chisq]"
    if {$chisq >= 0} {
	set pvalue [chi -number $chisq 1]
    } else {
	set pvalue 1
    }
    putsout stepfor.history "    *** p = $pvalue"
    global SOLAR_stepfor_pvalue
    if {$pvalue <= $SOLAR_stepfor_pvalue} {
	return 1
    }
    return 0
}

proc stepname {basename covindex} {
    set namelist [split $basename .]
    if {[set namelen [llength $namelist]] < 2} {
	return $basename.$covindex
    }
    set newnamelist {}
    for {set i 1} {$i < $namelen} {incr i} {
	if {[lindex $namelist $i] > $covindex} {
	    set newnamelist [concat [lrange $namelist 0 [expr $i - 1]] \
				 $covindex [lrange $namelist $i end]]
	    break
	}
    }
    if {{} == $newnamelist} {
	set newnamelist [concat $namelist $covindex]
    }
    return [join $newnamelist .]
}

# Parallel support for stepfor
#

proc step_parallel_launch {stepname number_tests dimension basemodelname} {

# Clean out old files
#
    set oldfiles [glob -nocomplain [full_filename step.*.*.*.*]]
    foreach oldfile $oldfiles {
	catch {exec rm -rf $oldfile}
    }

#
    set trynum 1
    set nullindex [expr $dimension - 1]
#
#       Make file with list of all files required
#         start with stuff generally needed by SOLAR models...
#
    set lfilename $stepname.files.list
    set outprefix ""
    if {"/" == [string index $lfilename 0]} {
	error "For parallel $stepname, outdir must be relative to working directory"
    }
    set lfile [open $lfilename w]
#
# Determine if matrices
#
    set all_matrixes [matrix]
    set length_all_matrixes [llength $all_matrixes]
    set matrix_names {}
    for {set imatrix 0} {$imatrix < $length_all_matrixes} {incr imatrix} {
	set iname [lindex $all_matrixes $imatrix]
	if {$iname == "matrix"} {
	    incr imatrix
	    set iname [lindex $all_matrixes $imatrix]
	    if {$iname == "load"} {
		incr imatrix
		set iname [lindex $all_matrixes $imatrix]
		if {$iname != ""} {
		    lappend matrix_names $iname
		}
	    }
	}
    }
#
# Add matrices and check for non-local matrices
#
    set matrix_names_needed {}
    foreach matrix_name $matrix_names {
	if {"/" != [string index $matrix_name 0]} {
	    if {-1 != [string first / $matrix_name]} {
		close $lfile
		puts "Model contains matrix $matrix_name"
		error "For parallel $stepname, model matrices should be in working directory\nor specified with absolute pathname in the model file."
	    }
	    puts $lfile $matrix_name
	}
    }
    puts $lfile "pedindex.out"
    puts $lfile "pedindex.cde"
#
# One or more phenotypes files, with or without .cde file
#
    set pifile [open phenotypes.info]
    set pofile [open [full_filename phenotypes.info] w]
    while {-1 != [gets $pifile phenname]} {
	puts $lfile $phenname
	if {-1 == [string first / $phenname]} {
	    puts $pofile $phenname
	} else {
#
# In our phenotypes.info, reference non-wd phenfiles in wd
#   because doscript copies them there
#
	    puts $pofile [file tail $phenname]
	}
	if {[file exists [file rootname $phenname].cde]} {
	    puts $lfile  [file rootname $phenname].cde
	}
	if {[file exists [file rootname $phenname].CDE]} {
	    puts $lfile [file rootname $phenname].CDE
	}
	if {[file exists $phenname.cde]} {
	    puts $lfile $phenname.cde
	}
	if {[file exists $phenname.CDE]} {
	    puts $lfile $phenname.CDE
	}
    }
    close $pifile
    close $pofile
    puts $lfile [full_filename phenotypes.info]
    puts "writing other variables to file"
#
# Write other variables to a file
#
    set vfilename [full_filename $stepname.vars]
    set vfile [open $vfilename w]
#    puts $vfile $null0_like
#    puts $vfile $log_n
    puts $vfile $nullindex
    puts $vfile $basemodelname
    close $vfile
    puts $lfile $vfilename
#
# Required files in outdir
#
    puts $lfile [full_filename $stepname.null$nullindex.mod]
    puts $lfile [full_filename $stepname.testmods.$trynum]
    puts $lfile [full_filename $stepname.covlist]
    close $lfile
#
# Note: all files in lfile will be copied to /tmp working directory
#   in parallel tasks
#
# Continue launch in recursive routine which also handles relaunches
#
     step_launch $stepname $dimension $number_tests $trynum $lfilename \
		$basemodelname
#
# Rename or rewrite temp output file to final one
#
    if {$stepname == "stepfor"} {
	puts "doing one last sort"
# one last sort required now
	set intfilename [full_filename $stepname.out.d$dimension.tmp]
	set sintfilename $intfilename.sort
	exec [usort] -k 1,1n $intfilename >$sintfilename
	puts "reformatting new results"
	set intfile [open $sintfilename]
	set outdfile [open [full_filename $stepname.out.d$dimension] w]
	while {-1 != [gets $intfile line]} {
	    set modelname [stepname $basemodelname [lindex $line 0]]
	    set thislike [lindex $line 1]
	    set use_covar [lindex $line 2]
	    set fchisq [lindex $line 3]
	    set pval [lindex $line 4]
	    set newline [fformat "%24s  %.3f  %-19s %9s  %s" \
			     $modelname $thislike $use_covar $fchisq $pval]
	    puts $outdfile $newline
	}
	close $intfile
	close $outdfile
	exec rm -f $intfilename
	exec rm -f $sintfilename
    } else {
	file rename [full_filename $stepname.out.d$dimension.tmp] \
	    [full_filename $stepname.out.d$dimension]
    }
#
# Done
#
    return ""
}


proc step_launch {stepname dimension number_tests trynum lfilename \
		basemodelname} {
    puts "There are $number_tests tasks"
    puts "Determining number of jobs to launch..."
#
# Determine available machines
#
    set cpus_useable 0
    while {1} {
	set cpus_useable [cpus_available_soft_margin]
	if {$cpus_useable > 100 || $cpus_useable*20 > $number_tests} break
	puts "Too few machines now available, waiting for more..."
	waitdots 60
    }
#
# Now, determine how many we are actually going to use
#
#   (2 below was parallel minjobsize, but that seems to have 2 meanings)
#
#   set maximum_cpus [expr 1 + ($number_tests-1) / 2]
    set maximum_cpus $number_tests
#
# Avoid "one task" situation because it can lead to 100% noncompletion
#   which wastes a lot of time
#
    if {$maximum_cpus <= [parallel minjobsize]} {
	set maximum_cpus $number_tests
    }
    if {$maximum_cpus <= $cpus_useable} {
	set reserved_cpus $maximum_cpus
    } else {
	set reserved_cpus $cpus_useable
    }
#
# Clear launch working directory, if any
#
    set tan [take_a_number]
    set launchdir [full_filename $stepname.[pid].$tan.$dimension.$trynum]
    exec rm -rf $launchdir
    exec mkdir $launchdir
#
# Get current program name
#
    global env
    set programpath $env(SOLAR_PROGRAM_NAME)
    set programname [file tail $programpath]
#
# Launch scripts
#
    set starting_task 1
    set solarname solar
    set jobsize [expr int (ceil ($number_tests / double($reserved_cpus)))]
    puts "precomputed maximum cpus for this dimension is $reserved_cpus"
    puts "precomputed jobsize is $jobsize"
    set launchstat [launch -proc $stepname\_step -extra_args \
			"$trynum" -filelist $lfilename \
			-n $number_tests -jobsize $jobsize \
			-outputdir $launchdir]
    puts "launch status: $launchstat"
#
# Note: launch uses the minimum number of jobs consistent with getting all
# tasks completed with a given jobsize.  Therefore, it might not use all the
# machines we said it could use.  We can get the actual number of machines
# used, which is very important, from the launch.info file.  This is somewhat
# counterintuitively called "maxjobs" it should be "actualjobs".
#
    getinfo launch.info maxjobs
    puts "Launched $maxjobs jobs"

    return [step_shepherd $stepname $dimension $maxjobs $jobsize $trynum \
		$lfilename $tan $launchstat $number_tests $basemodelname]
}


#
# solar::step_shepherd -- private
#
# Purpose:  "Shepherd" parallel tasks, making sure they ultimately complete
#           successfully.  Uses "relaunch" to get all tasks launched OK.
#           Uses "finish" to finish, combined with modified cleanrecords
#           (modified to deal with lack of comma delimiting) to compile
#           all results for a single df.  Results are ultimately written
#           to the output file stepfor.out.
#
#           If this procedure fails to get everything completed, it recurses
#           back through step_launch to launch the remaining stragglers.
#
#-

proc step_shepherd {stepname dimension njobs jobsize trynum lfilename \
			  tan launchstat ntests basemodelname} {
    set start_time [clock seconds]
    set started 0
    set fraction_started 0
    set newtrynum [expr $trynum + 1]
#
# Wait until all jobs have started, or would be expected to be started
#
    puts "Waiting for most jobs to start..."

    set failure_time [expr 60 * 15]
    set initial_wait 10

    set ldirname [full_filename $stepname.[pid].$tan.$dimension.$trynum]
    set expected_total_time 0
    while {1} {
    	after [expr $initial_wait*1000]
	puts "Looking at $ldirname"
	set started [lindex [exec ls $ldirname | wc] 0]
	if {$started == $njobs} {
	    puts "All tasks started, breaking wait"
	    break
	}
	set current_time [clock seconds]
	if {$current_time - $start_time > $failure_time} {
#
# On complete failure, assume launch failed and recurse to do it
#
	    if {$started == 0} {
		puts "Launch failed completely, recursing to redo"
		set oldjobarray [lindex $launchstat 2]
		set oldjobnumber [lindex [split $oldjobarray .] 0]
		puts "Deleting old job array $oldjobnumber"
		catch {exec qdel -f $oldjobnumber}
		set oldfilename [full_filename $stepname.testmods.$trynum]
		set newname [full_filename $stepname.testmods.$newtrynum]
		file copy -force $oldfilename $newname
		set lfile [open $stepname.files.list a]
		puts $lfile $newname
		close $lfile
		return [step_launch $stepname $dimension $ntests \
			    $newtrynum $lfilename $basemodelname]
	    }
#
# On partial failure, this is probably a "slow start" caused by someone
# else taking some of the machines we thought we were going to get before we
# got them.
#
# But what does this means in terms of how long to set the relaunch wait?
# Initial guess: average of this long wait and zero wait
#
	    puts "$failure_time seconds exceeded, breaking wait"
	    set expected_total_time [expr $failure_time / 2]
	    break
	}
	set fraction_started [expr double($started) / $njobs]
	puts "[format %.4g [expr 100.0 * $fraction_started]]% of jobs now started"
	if {$expected_total_time == 0} {
	    if {$fraction_started >= 0.5} {
		puts "Entering second phase of wait"
		set overcompletion_factor 1.4
		set fraction_time [expr [clock seconds] - $start_time]
		set reciprocal_started [expr 1.0 / $fraction_started]
		set expected_total_time [expr round($fraction_time * \
							$overcompletion_factor * \
							$reciprocal_started)]
	    }
	} elseif {$current_time - $start_time > $expected_total_time} {
	    puts "Jobs were expected to be started by now, breaking wait"
	    break
	}
    }
    set relaunch_time $expected_total_time
    if {$relaunch_time == 0} {
	puts "Setting expected to actual time"
	set relaunch_time [expr [clock seconds] - $start_time]
    }
    set relaunch_time [expr $relaunch_time / 2]
    if {$relaunch_time < $initial_wait} {
	puts "Setting expected to initial time"
	set relaunch_time $initial_wait
    }
    puts "Wait time is $relaunch_time"
#
# Now, keep relaunching until everything is started, at total time intervals
# But only launch as allowed by margin
#
    puts "Looking for $ldirname"
    set started [lindex [exec ls $ldirname | wc] 0]
    puts "$started out of $njobs have started"
    if {$started < $njobs} {
	set all_started 0
	set large_count 0
	while {1} {
	    set max_relaunch [cpus_available_soft_margin]
	    puts "Maximum available cpus for relaunching is $max_relaunch"
	    set status [relaunch -y -max $max_relaunch]
	    set third_status [lindex $status 2]
	    set status [lindex $status 0]
	    puts "STATUS is $status"
	    if {$status == "All"} {break}
	    if {$third_status == "relaunched" && \
		    [is_integer $status] && $status > 500} {
		if {$large_count > 2} {
		    puts "Many large relaunches.  Something isn't good.  Wait 30 minutes."
		    set relaunch_time 1800
		} else {
		    puts "Extremely large relaunch.  Wait 5 minutes."
		    set relaunch_time 300
		}
		incr large_count
	    } else {
		puts "Waiting $relaunch_time seconds for all jobs to start"
	    }
	    waitdots $relaunch_time
	}
    }
#
# Gather results ("finish")
# If jobsize==1, we're done
# Otherwise, recheck at relaunch_time intervals
#   (turns out to be better than a big long predicted wait that is usually
#    way too long because start time is far longer than the time of one
#    iteration, there is really no good handle on incremental time)
#    Upon 97% of jobs fully completed, we quit.
#       (A fully completed job has all its tasks completed.)
#    When number of tasks completed in
#      last iteration drops below 5% of the average tasks completed
#      per iteration, we quit.  (Average does not count "first interation"
#      because that included many jobs with more than one complete.)

    set last_jobs_completed ""
    set initial_jobs_completed 0
    set initial_time 0
    set penultimate 0
    while {1} {
	puts [exec date]
	puts "Gathering results..."
	set remaining_time ""
	set gathering_begin [clock seconds]
	set status [finish -any -n]
	set gathering_time [expr [clock seconds] - $gathering_begin]
	puts "Result gathering time: $gathering_time seconds"
	set percent_fully_complete [lindex $status 6]
	set total_jobs [lindex $status 3]
	set status [lindex $status 0]
	puts "$status tasks done"
	if {$jobsize==1} {break}
	if {$status == "All"} {
	    puts "Finish reports all jobs done, breaking wait"
	    break
	}
	if {$percent_fully_complete > 98} {
	    puts "More than 98% of jobs completed fully, breaking wait"
	    break
	}
	if {![is_integer $status]} {continue}
	set percent_complete [expr 100*double($status)/$total_jobs]
	set remaining_time ""
	if {$last_jobs_completed == {}} {
	    set first_tcpi $status
	    set last_jobs_completed $status
	    set initial_jobs_completed $status
	    set initial_time [clock seconds]
	    set timestamps $initial_time
	    set countstamps $status
	    set pent_index -1    ;# pent_index is end-1	    
	} else {
	    set current_time [clock seconds]
	    lappend timestamps $current_time
	    lappend countstamps $status
	    incr pent_index
	    set latest_complete [expr $status - [lindex $countstamps $pent_index]]
	    puts "$latest_complete tasks completed in this iteration"
	    set average_tcps [expr (double($status) - $first_tcpi) / \
				  ($current_time - $initial_time)]
	    puts "About [format %.4g $percent_complete]% tasks completed"
	    puts "About [format %.4g $percent_fully_complete]% of jobs fully complete"
	    puts "Note: this counts some tasks which may be redundant"
	    puts "Average tasks per second: [format %.4g $average_tcps]"
#
# Compute tasks/sec for the most recent minute or more
#
	    set breakout 0
	    for {set i $pent_index} {$i > 0} {incr i -1} {
		set old_time [lindex $timestamps $i]
		if {$current_time - $old_time > 60} {
		    set old_count [lindex $countstamps $i]
		    set tasks_completed [expr $status - $old_count]
		    set time_elapsed [expr $current_time - $old_time]
		    set current_tcps [expr double($tasks_completed) / \
					  $time_elapsed]
		    puts "Current tasks per second: [format %.4g $current_tcps]"
		    if {$current_tcps < 0.02 * $average_tcps} {
			if {$penultimate > 0} {
			    puts "Rate below %2 of average on second attempt"
			    set breakout 1
			} else {
			    if {$average_tcps > 1.0/30} {
				puts "Rate below 2% of average, with high average"
				set breakout 1
			    } else {
				puts "Rate below 2% of average"
				puts "Trying once more after 60 seconds..."
				set penultimate 1
				waitdots 60
			    }
			}
		    } else {
			set penultimate 0
		    }
		    break
		}
	    }
	    if {$breakout} {
		puts "Breaking wait for results"
		break
	    }
	    set last_jobs_completed $status
	    set total_time [expr round($total_jobs * \
				       (double($current_time - $initial_time) / \
					($status - $initial_jobs_completed)))]
	    puts "Projected total time is $total_time seconds"
	    set remaining_time [expr round($total_time - \
					       ($current_time - $initial_time))]
	    puts "Projected remaining time is $remaining_time seconds"
	}
#
# If this is not a special case of being nearly done
# Determine time to wait before collecting results again
#
	if {$penultimate < 1} {
#
# Start with "relaunch_time" which is derived from the time it took to launch
# all the jobs and do 1 iteration and so therefore reflects grid load
#
# (This could all be replaced in a future revision simply based on predicted
#  completion time...)
#
	    set finish_wait $relaunch_time
#
# limit wait time to three minutes, the maximum ordinary start, to filter
# out slow starts that occur if machines are grabbed by others before we get
# them.
#
	    set max_task_wait_time 180
	    if {$finish_wait > $max_task_wait_time} {
		set finish_wait $max_task_wait_time
	    }
#
# But then, allow at least 2x as much time as it takes to gather results
#   so that most of the time, result gathering is not happening, because
#   gathering lowers efficiency of result production
#
	    if {$finish_wait < 2 * $gathering_time} {
		set finish_wait [expr 2 * $gathering_time]
	    }
#
# But do not allow more wait time than computed remaining time + 15 seconds
# OR, 20 seconds on the first go (maximum we are prepared to lose, since this
# might be nearly done already)
#
	    if {{} != $remaining_time} {
		set allow_time [expr $remaining_time + 15]
	    } else {
		set allow_time 20
	    }
	    if {$finish_wait > $allow_time} {
		set finish_wait $allow_time
	    }
	    puts "Waiting $finish_wait seconds for all jobs completed"
	    waitdots $finish_wait
	}
    }
#
# Remove duplicate and incomplete records
#   and append good records to temp copy of ultimate output file
#   accumulate new testmods file of models not yet done
#
    puts [exec date]
    puts "Testing results for completion..."
    exec [usort] -k 1,1n $ldirname.all >$ldirname.sort

    set infile [open $ldirname.sort]
    set nfilename [full_filename $stepname.testmods.$trynum]
    set nfile [open $nfilename]


    set newfilename [full_filename $stepname.testmods.$newtrynum]
    set newfile [open $newfilename w]  ;# this gets filled with undone models

    set outfile [open [full_filename $stepname.out.d$dimension.tmp] a]

    set lastnumber ""
    set testnumber ""
    set missing_count 0
    global STEP_recordsize
    while {-1 != [gets $infile line]} {
	if {"" == $line} {continue}
	if {[llength $line] != $STEP_recordsize} {continue}
	set number [lindex $line 0]
	if {0 == [string compare $number $lastnumber]} {
	    puts "removing duplicate record for $number"
	    continue
	}
	set lastnumber $number
	while {1} {
	    if {-1 == [gets $nfile testnumber]} {
		puts "ERROR getting model number to test"
		break
	    }
	    if {$testnumber == ""} {
		puts "ERROR testnumber is blank"
		continue
	    }
	    if {0==[string compare $number $testnumber]} {
		break
	    } else {
		puts $newfile $testnumber
		puts "Missing model $testnumber"
		incr missing_count
	    }
	}
	puts $outfile $line
    }
    while {-1 != [gets $nfile testnumber]} {
	if {$testnumber == ""} {continue}
	puts "Missing additional model $testnumber"
	puts $newfile $testnumber
	incr missing_count
    }

    close $infile
    close $outfile
    close $nfile
    close $newfile
#
# If nothing is missing, we're done
#
    if {$missing_count == 0} {
	puts "No missing names"
	puts [exec date]
	return OK
    }
#
# Recurse to handle new missing records
#
    set lfile [open $stepname.files.list a]
    puts $lfile [full_filename $stepname.testmods.$newtrynum]
    close $lfile
    puts "Now recursing to handle missing records"
    return [step_launch $stepname $dimension $missing_count $newtrynum $lfilename $basemodelname]
}

proc stepfor_step {directory tasksize taskno trynum} {

    newtcl
    set outputfile $directory/$taskno
    file delete $outputfile
    set covlist [listfile stepfor.covlist]
    set othervars [listfile stepfor.vars]
    set nullindex [lindex $othervars 0]
    set basemodelname [lindex $othervars 1]
    set tmodfile [open stepfor.testmods.$trynum]

# Note: taskno starts with 1, but first_task_num is an index that starts with 0
    set first_task_num [expr ($taskno-1)*$tasksize]
#    set last_task_num [expr $first_task_num + $tasksize - 1]

    for {set i 0} {$i < $first_task_num} {incr i} {
	gets $tmodfile line
    }
    set testmods {}
    for {set i 0} {$i < $tasksize} {incr i} {
	gets $tmodfile line
	lappend testmods $line
    }
    close $tmodfile
#
# Maximize each model and write results
#
    foreach covnumber $testmods {
	puts "covnumber is $covnumber"
	set use_covar [lindex $covlist [expr $covnumber - 1]]
	set modelname "$covnumber"
	set bestlike ""
	catch {file delete stepfor.step.best.mod}
	after 500
	for {set maxcount 0} {$maxcount < 3} {incr maxcount} {
	    set did_maximize 0
	    set did_max_count 0
	    while {$did_maximize == 0 && $did_max_count < 10} {
		incr did_max_count
		catch {
		    puts "Evaluating model $modelname number $maxcount"
		    load model stepfor.null$nullindex
		    set nullike [loglike]
		    covariate $use_covar
		    maximize
		    set newlike [loglike]
		    if {$bestlike == "" || $newlike > $bestlike} {
			puts "[loglike] is best so far"
			save model stepfor.step.best
			set bestlike $newlike
		    }
		    set did_maximize 1
		}
	    }
	}
	puts "loading best model"
	load model stepfor.step.best

	set chisq [expr 2.0 * ([loglike] - $nullike)]

	if {$chisq >= 0} {
	    set pval [chi -number $chisq 1]
	} else {
	    set pval 1
	}
	set fchisq [format %.4f $chisq]

	if {-1 != [set epos [string first e $pval]]} {
	    set mstring [string range $pval 0 5]
	    set estring [string range $pval $epos end]
	    set pval "$mstring$estring"
	}
	puts "Writing to $outputfile"
	set ofile [open $outputfile a]
	puts "formattion oline"
	set oline [fformat "%24s  %.3f  %-19s %9s  %s" \
			 $modelname [loglike] $use_covar $fchisq $pval]
	puts "got oline"
	puts $ofile $oline
	close $ofile
    }
}


# solar::stepup --
#
# Purpose: Covariate screening by Step Up algorithm, useful for QTN analysis
#
# Usage:   stepup [-list listfile] [-list list] [-verbose]
#                 [-fix listfile]  [-fix fixlist]
#                 [-cutoff cutoff] [-logn logn] [-finishlogn]
#                 [-symmetric] [-cornerdf df] [-par]
#                 [-parclean]
#
#          stepup is an fast version of bayesavg and may be used in
#          QTN analysis.
#
#          By default, stepup will test all covariates in the current
#          model one at a time, then add all the new covariate models
#          within the BIC cutoff to the window.  Then the window models are
#          subjected to another round of testing against all covariates,
#          and the process repeats until no more models are added to the
#          window.  Unlike bayesavg, this algorithm doesn't test all
#          possible models, just those that are derived from those in
#          the window.  When completed, it writes files named stepup.win
#          and stepup.avg to the output directgory containing posterior
#          probabilities for the window models and components.
#
#          To ensure that all models use the same sample, a new file named
#          fully_typed.out is created in the output directory which
#          defines a variable named "fully_typed" for each fully typed
#          individual.  This file is added to the list of open phenotypes
#          files, and the variable "fully_typed" is added to the
#          model as a "null" covariate which has no effect on the model
#          other than restricting the sample to fully typed individuals.
#          This covariate is removed from the final best model stepup.best,
#          so you may get a different likelihood in subsequent maximization.
#
#          Up to dimension 3, all models with BIC better than the null model
#          are retained.  (This feature may be controlled with the -cornerdf
#          option.)  Also, the default "strict" rule is only applied to
#          remove apparently redundant higher dimensional models at the
#          very end after all important dimensions have been scanned.
#
#          -list listfile    listfile is a file containing a list of all
#                            covariates to be tested, one on each line.
#                            The filename cannot contain spaces.  These
#                            covariates may or may not be in the model when
#                            the command is given.  If the -list option is
#                            specified, all other covariates in the starting
#                            model are automatically fixed.
#          -list list        Alternatively, a Tcl list of covariates to
#                            be tested can be specified.  Tcl lists are
#                            space delimited and enclosed in quotes or curly
#                            braces.
#
#          -fix list         list is a Tcl list of covariates to be
#                            included in every model and not tested.  Their
#                            values will be estimated by maximum likelihood
#                            for every model, unless you constrain them.
#                            These covariates may or may not in the model
#                            when the command is given.  For -fix, a list
#                            could be simply one phenotype, and that
#                            supercedes a file with the same name.
#          -fix listfile     Alternatively, a file containing a list of all
#                            covariates to be included in every model may
#                            be specified.  The filename cannot contain
#                            spaces.  The list of covariates to be fixed
#                            will supercede the list of covariates to be
#                            tested if the same covariate occurs on both
#                            lists, however a warning will be given.
#
#          -cutoff cutoff    Set the final BIC cutoff.  The default is 6.
#
#          -logn logn        Use this fixed value for log(N) from the
#                            beginning.
#
#          -finishlogn logn  Recompute results of previous analysis with this
#                            log(N) value.  Sometimes stepup fails at the end
#                            because the standard error of the SE parameter
#                            of the best BIC model cannot be computed, and
#                            that is needed to compute the final log(N).
#                            This option allows you to finish such a run that
#                            nearly completed previously.  Be sure that
#                            starting conditions (such as loaded pedigree,
#                            phenotypes, model, outdir) and options are
#                            exactly the same as before.  The original startup
#                            (stepup.orig.mod) and null models from the output
#                            directory will be loaded.  Note that the temporary
#                            log(N) used by stepup by default is simply the
#                            log of the sample size, and this is reported
#                            to the stepup.history file.  You may choose to
#                            use that or some other estimate.  A special file
#                            required is stepup.winmods.prelim, which was
#                            produced by the previous incompleted run of
#                            stepup.
#
#          -verbose          Show maximization output during maximizations.
#          -v                Same as -verbose
#
#          -cornerdf df      EXPERIMENTAL.  This sets the last degree of
#                            freedom that uses a loose test to include models
#                            in the window.  Models need only have a better
#                            BIC than the null model up to and including
#                            this df.  The default is 3.
#
#          -symmetric        Apply symmetric rule rather than strict.  This
#                            results in a larger window.
#
#          -par              This option turns on Parallel
#                            processing on the SFBR GCC Compute Ranch.
#                            WARNING!  Do not run more than one instance of
#                            stepup -par from the same working directory.
#                            Parallel stepup will use many (but not all!) ranch
#                            machines, and access for other users and jobs may
#                            be delayed due to gridware thrashing.  The usual
#                            output is not printed to the terminal to save time
#                            but numerous parallel status messages are printed
#                            to help the developers make this program better.
#                            The parallel operation is automatic and the
#                            parallel status messages may be ignored by most
#                            users most of the time unless there is no output
#                            for more than fifteen minutes.  Note: If model
#                            includes linkage element matrices loaded from
#                            some mibddir, those matrices should be relocated
#                            to the working directory, or specified with an
#                            absolute pathname in the model file.  This is
#                            because in parallel operation the model is loaded
#                            not in the current working directory but in a
#                            subdirectory of /tmp.
#
#          -parclean         Normally, parallel stepup cleans up after itself.
#                            However, if it is necessary to force a shutdown
#                            of a parallel stepup, normal cleanup is not
#                            done.  "stepup -parclean" cleans up all the
#                            junk stepup files in /tmp directories on all
#                            ranch machines.  This must be run on medusa.  Do
#                            not run if you have any other running parallel
#                            jobs (parallel stepup, parallel bayesavg, or any
#                            parallel job using "launch" or "doscript") as
#                            their files may be deleted too.
#                            See also "doranch" for other ranch cleanup
#                            procedures.  Cleanup history is written to a file
#                            named cleantmp.out.
#                            
# -

# Object Design:
#   Domain  {<phenotype names>+}
#   ModelName  {(cov.i.j.k...n) toSet setTo}
#   Model    {<ModelName> loglike BIC}
#   Df       {<Model>+}
#   Window   {<Df>+ New Add Purge makeTests}

#
# Parallel wish list of additional suggested improvements:
#   Additional job collection run in case of 97% of jobs completed fully; one
#     more run might push it to 100% in some cases.
#   Do one-to-three job recursions with inline stepup_step.
#   Don't print missing models to screen.

proc stepup {args} {
#
# Do parallel cleanup if requested
#
    if {$args == "-parclean"} {
	set uname [lindex [exec who -m] 0]
	return [doranch cleantmp $uname.]
    }
    file delete [full_filename stepup.win]
    file delete [full_filename stepup.avg]
#
# read arguments
#
    set parallel 0
    set cornerdf 3
    set stepverbosity -q
    set excluded 0
    set symmetric 0
    set cutoff 6.0
    set p_cutoff 0.1
    set user_log_n 0
    set verbose 1
    set listfile {}
    set fixlist {}
    set ucovlist {}
    set finishlogn 0
    set stepuptest stepuptest    ;# the default test, not a typo
    set badargs [read_arglist $args \
		     -list listfile \
		     -fix fixlist \
		     -cutoff cutoff \
		     -logn user_log_n \
		     -verbose {set verbose 1; set stepverbosity ""} \
		     -v {set verbose 1; set stepverbosity ""} \
		     -par {set parallel 1} \
		     -symmetric {set symmetric 1} \
		     -cornerdf cornerdf \
		     -finishlogn finishlogn \
		     ]
    if {{} != $badargs} {
	error "stepup: Invalid argument $badargs"
    }
#
# Save initial model and delete old files
#
    if {$finishlogn} {
	set user_log_n $finishlogn
	load model [full_filename stepup.orig]
    } else {
	save model [full_filename stepup.orig]
	file delete [full_filename stepup.history]
	file delete [full_filename stepup.out]
	set modfiles [glob -nocomplain [full_filename stepup.null*.mod]]
	foreach modfile $modfiles {
	    file delete $modfile
	}
    }
#
# Get list of covariates to test
#
    if {{} != $listfile} {
	if {-1 == [string first " " $listfile]} {
	    set covlist [listfile $listfile]
	} else {
	    set covlist $listfile
	}
	set ucovlist $covlist
    } else {
	set covlist [covar]
    }
#
# Because a one element fixlist is possible
# If one element is specified, first check if it is a phenotype
#	
    if {{} != $fixlist} {
	if {-1 == [string first " " $fixlist]} {
	    if {-1 == [lsearch -exact [concat sex [lrange [phenotypes] \
						       1 end]] $fixlist]} {
		if {![file exists $fixlist]} {
		    error "stepup: No phenotype or file named $fixlist"
		}
		set fixlist [listfile $fixlist]
	    }
	}
    }
#
# Be sure fully_typed() is incorporated into fixlist at least once
# and only once
#
    if {-1 == [lsearch -exact $fixlist fully_typed()]} {
	lappend fixlist fully_typed()
    }
#
# Now remove fixed covariates from list of covariates to test
#
    foreach fix $fixlist {
	if {-1 != [set foundpos [lsearch -exact $covlist $fix]]} {
	    set covlist [lreplace $covlist $foundpos $foundpos]
	    if {{} != $ucovlist} {
		if {-1 == [lsearch -exact $ucovlist $fix]} {
		    putsout stepup.history "Warning.  Removing $fix from test list because fixed"
		}
	    }
	}
    }
#
# Show list of covariates
    set n [llength $covlist]
    set printlist $covlist
    set maxline 78
    while {{} != $printlist} {
	set line "    [lindex $printlist 0]"
	set printlist [lrange $printlist 1 end]
	while {{} != $printlist} {
	    set next [lindex $printlist 0]
	    if {$maxline < [expr [string length $line]+2+[string length $next]\
			       ]} {
		break
	    }
	    set line "$line $next"
	    set printlist [lrange $printlist 1 end]
	}
	putsout stepup.history $line
    }
    putsout stepup.history "\n    *** Testing $n covariates"
#
# Add fixed covariates (if not already added) except for fully_typed
#   (fully_typed gets added later)
#
    foreach cov $fixlist {
	if {$cov != "fully_typed()"} {
	    covariate $cov
	}
    }
#
# Add listed covariates (if not already added) and then delete (later).
# This ensures covariates CAN be added, and that they are now deleted.
#
    foreach cov $covlist {
	covariate $cov
    }
#
# If no omega, make it polygenic
#
    if {[omega] == \
	    "omega = Use_polygenic_to_set_standard_model_parameterization"} {
	putsout "No predefined omega; defaulting to polygenic"
	polymod
    }
#
# Now that all covariates have been added to model
# create fully_typed.out file
#
    if {[catch {eval maximize $stepverbosity -sampledata} errmess]} {
	error "Unable to maximize model with covariates:\n$errmess"
    }

    if {![file exists [full_filename sampledata.out]]} {
	puts "\nError testing starting model, suspect misspelled snp names"
	catch maximize errmess
	error $errmess
    }
	
    set insample [solarfile open [full_filename sampledata.out]]
    solarfile $insample start_setup
    solarfile $insample setup id
    set outsample [open [full_filename fully_typed.out] w]
    puts $outsample "id,fully_typed"
    while {{} != [set record [solarfile $insample get]]} {
	set id [lindex $record 0]
	puts $outsample "$id,1"
    }
    solarfile $insample close
    close $outsample
#
# Removed fully_typed.out from old output directories
# and add in new fully_typed.out from current output directory
#
    set oldphenotypes [phenotype -files]
    set newphenotypes {}
    foreach oldphenotype $oldphenotypes {
	set phentail [file tail $oldphenotype]
	if {[string compare $phentail fully_typed.out]} {
	    lappend newphenotypes $oldphenotype
	}
    }
    lappend newphenotypes [full_filename fully_typed.out]
    eval load phenotypes $newphenotypes

    set oldcovariates [covariate]
    if {-1 == [lsearch -exact $oldcovariates fully_typed()]} {
	covar fully_typed()
    }
#
# OK, now delete the variable covariates
#
    foreach cov $covlist {
	covariate delete $cov
    }
#
# Maximize "null" model
#
    if {!$finishlogn} {
	puts "    *** Maximizing null model\n"
	eval maximize $stepverbosity -o cov.out
	save model [full_filename stepup.null0]
    } else {
	load model [full_filename stepup.null0]
    }
    set null0_like [loglike]
    set best_bic 0
    set samplesize [outfile_sample_size cov.out]
    putsout stepup.history "    *** Samplesize is $samplesize"
    set log_n [format %.7g [expr log($samplesize)]]
    if {$user_log_n} {
	set log_n $user_log_n
	putsout stepup.history "    *** User-specified log(N) is $log_n\n"
    }
    if {!$finishlogn} {
	putsout stepup.history "    *** Estimated log(N) is $log_n\n"
	putsout stepup.out [format "%30s   %s      %s" Model Loglikelihood BIC]
	set basemodelname cov
	putsout stepup.out [format "%30s     %.3f      %.4f" \
				$basemodelname [loglike] $best_bic]
    }
    
#
# Main loop
#
    global window
    set window [Window_New $null0_like]
    set parallel_infile ""
    set parallel_stepup_out ""
    if {!$finishlogn} {
    for {set testdf 1} {$testdf <= $n} {incr testdf} {
	if {$parallel_infile != ""} {
	    close $parallel_infile
	    set parallel_infile ""
	}
	if {$parallel_stepup_out != ""} {
	    close $parallel_stepup_out
	    set parallel_stepup_out ""
	}
	if {$parallel} {
	    puts [exec date]
	    puts "Making new tests..."
	    set trynum 1
	    set tfilename [full_filename stepup.testmods.$trynum]
	    set number_tests [Window_MakeTestFile $window $n $tfilename]
	    puts "Done at [exec date] with $number_tests tests"
	} else {
	    set newtests [Window_MakeTests $window $n]
	    set number_tests [llength $newtests]
	}
	set newdfwindow {}
	set starting_best_bic $best_bic

	set parallel_launch_done 0

	for {set testnumber 0} {$testnumber < $number_tests} {incr testnumber} {
#
# Parallel operation
#
	    if {$parallel} {
		if {!$parallel_launch_done} {
		    puts "Setting up for parallel launch"
		    set covfilename [full_filename stepup.covlist]
		    set covfile [open $covfilename w]
		    foreach covline $covlist {
			puts $covfile $covline
		    }
		    close $covfile
		    set poutfilename [full_filename stepup.out.d$testdf]
		    file delete -force $poutfilename
		    file delete -force $poutfilename.tmp
		    stepup_begin_launch $testdf $number_tests \
			$null0_like $log_n
#
# returns only when all results written to stepup.out.d$dimension
#
		    set parallel_launch_done 1
		    set parallel_infile [open $poutfilename]
		    set parallel_stepup_out [open [full_filename stepup.out] a]
		    puts "Writing dimension $testdf to stepup.out..."
		}
		gets $parallel_infile current_result
		set test [lindex $current_result 0]
		set loglike [lindex $current_result 1]
		set bic [lindex $current_result 2]

	    } else {
#
# If non-parallel operation
#   Get next test from list of tests
#   Make testmodel
#   Maximize
#   Set up modelname, loglike, and bic from first result
#
		set test [lindex $newtests $testnumber]
		load model [full_filename stepup.null0]
		set testcovs [name2set $test]
		foreach covnumber $testcovs {
		    covariate [lindex $covlist [expr $covnumber - 1]]
		}
		eval maximize $stepverbosity
		set loglike [format %.3f [loglike]]
		set bic [format %8.4f [bic_calculate [loglike] $null0_like \
				      [llength $testcovs] $log_n]]
	    }
#
# Process one result in either parallel or non-parallel mode
#
	    if {$testdf <= $cornerdf} {
		if {$bic <= 0} {
		    ifdebug puts "added to window"
		    lappend newdfwindow [list [lindex $test 0] $loglike $bic]
		}
	    } else {
		if {$bic <= $best_bic + $cutoff} {
		    lappend newdfwindow [list [lindex $test 0] $loglike $bic]
		}
	    }
	    if {$bic < $best_bic} {
		set best_bic $bic
	    }
	    if {!$parallel} {
		putsout stepup.out [format "%30s     %s    %s" \
					$test $loglike $bic]
	    } else {
		puts $parallel_stepup_out [format "%30s     %s    %s" \
					       $test $loglike $bic]
	    }
	}
	if {$parallel_infile != ""} {
	    close $parallel_infile
	    set parallel_infile ""
	}
	if {$parallel_stepup_out != ""} {
	    close $parallel_stepup_out
	    set parallel_stepup_out ""
	}
#
# Since best bic is a moving target, we must go through window again
# to apply last best bic cutoff
#
	if {$testdf > $cornerdf} {
	    set retested_window {}
	    set tested_models {}
	    foreach newmodel $newdfwindow {
		set bic [lindex $newmodel 2]
		if {$bic < $best_bic + $cutoff} {
		    lappend retested_window $newmodel
		    lappend tested_models [lindex $newmodel 0]
		}
	    }
	    set newdfwindow $retested_window
	}

	if {1 > [llength $newdfwindow]} {
	    putsout stepup.history "    *** No models of df $testdf added to window"
	    set early_exit 1
	    break
	}
#
# Add new models to window and purge
#
	set window [Window_Add $window $newdfwindow]
	if {$testdf > $cornerdf} {
	    if {$symmetric || $testdf <= $n} {
		set dosymmetric 1
	    } else {
		set dosymmetric 0
	    }
	    set window [Window_Purge $window [expr $best_bic+$cutoff] \
			    $dosymmetric]
	}
#
# Print models added to window
#
	set lastdf [lindex $window end]
	if {{} != $lastdf} {
	    if {$verbose} {
		putsout stepup.history "\n    *** Models added to window:"
		set modindex 0
		set lastindex [llength $lastdf]
		while {$modindex < $lastindex} {
		    set line "       "
		    set position 7
		    set oldline $line
		    set oldposition $position
		    while {$position < 78 && $modindex < $lastindex} {
			set oldline $line
			set oldposition $position
			set line "$line [lindex [lindex $lastdf $modindex] 0]"
			incr modindex
			set position [string length $line]
		    }
		    if {$position > 78 && [llength $line] > 1} {
			set line $oldline
			set position $oldposition
			incr modindex -1
		    }
		    putsout stepup.history $line
		}
		putsout stepup.history ""
	    }
	}
    }
#
# Save models for possible later -finishlogn option
#
	set winout [open [full_filename stepup.winmods.prelim] w]
	foreach win $window {
	    puts $winout $win
	}
	close $winout
    } else {
#
# Load previous models for -finishlogn option
#
	set window ""
	if {![file exists [full_filename stepup.winmods.prelim]]} {
	    error "Can't find file [full_filename stepup.winmods.prelim] for -finishlogn"
	}
	set winout [open [full_filename stepup.winmods.prelim]]
	while {-1 != [gets $winout line]} {
	    lappend window $line
	}
	close $winout
	set user_log_n $finishlogn
	putsout stepup.history "\nContinuing with window models from previous run..."
    }
#
# Now get sorted list of models in window
#
    set newbic 0
    global winmods
    set winmods [Window_Models $window]
#
# Recompute log_n based on model with best BIC
#
    set best_mod [lindex $winmods 0]
    set best_mod_name [lindex $best_mod 0]
    load model [full_filename stepup.null0]
    set testcovs [name2set $best_mod_name]
    foreach covnumber $testcovs {
	covariate [lindex $covlist [expr $covnumber - 1]]
    }
    option standerr 1
    eval maximize $stepverbosity
    save model [full_filename $best_mod_name]
    if {!$user_log_n} {
	if {[parameter sd se] == 0} {
	    putsout stepup.history "Couldn't get standard error of SE to compute log(n)"
	    putsout -q stepup.history \
		"Re-run stepup using -finishlogn $log_n or other estimate of log(n)"
	    load model [full_filename stepup.orig]
	    error "Re-run stepup using -finishlogn $log_n or other estimate of log(n)"
	}
	set sd [parameter sd =]
	set sdse [parameter sd se]
	set log_n [format %.9g [expr log ($sd*$sd/(2.0*$sdse*$sdse))]]
	putsout stepup.history "\n    *** log(n) calculated from $best_mod_name is $log_n"
#
# Recompute best BIC
#
	set newbic 1
    }
    if {$newbic || $finishlogn} {
	set best_bic [bic_calculate [loglike] $null0_like \
			  [llength $testcovs] $log_n]
	set newbic 1
    }
#
# Write results of window
#  (re-purge now forced to ensure final strict rule, if applicable
#   but most often there will be a new BIC now anyway)
#
    set window [Window_Purge \
		    $window [expr $best_bic+$cutoff] $symmetric $log_n]
    set winmods [Window_Models $window]
    if {$newbic} {
	puts "\n    *** Models in Window (stepup.win) with recomputed BIC:\n"
    } else {
	puts "\n    *** Models in Window (stepup.win):\n"
    }
    foreach mod $winmods {
	set modname [lindex $mod 0]
	set loglike [lindex $mod 1]
	set bic [lindex $mod 2]
	putsout stepup.win [format "%30s     %.3f     %8.4f" \
			      $modname $loglike $bic]
    }
    load model [full_filename $best_mod_name]
    save model [full_filename stepup.best]
    stepup_post $winmods $cutoff $covlist $stepverbosity $log_n $user_log_n
    load model [full_filename stepup.best]
    puts ""
    puts "    *** Averages written to [full_filename stepup.avg]"
    puts "    *** Window file is      [full_filename stepup.win]"
    puts "    *** Messages written to [full_filename stepup.history]"
    puts "    *** Model with best BIC loaded: $best_mod_name"
}

proc stepup_post {winmods cutoff covlist stepverbosity log_n user_log_n} {
#
#  Note: We need to use arcane i/p organization of variable names, because
#  without it we would be using parameter names, which could really
#  foul things up with special characters in them.  (Similar method
#  used in bayesavg_post).
#
#  Even with arrays, there is a similar problem with required internal
#  quotation.  Braces don't seem to work properly in that context.
#
#  This follows because SOLAR parameter names are not restricted to Tcl's
#  name rules.
#
# To fit heritabilities into set of parameters:
# index 1..n for covariates 1..n
# index 0 is h2r
# index -1 is h2q1, etc.
#
# Suffixes for pseudolists use integer or _integer: $p
#   psum$p       Parameter weighted sum
#   sesum$p      Standard Error sum
#   ppprob$p     Parameter posterior probability
#
# Other variables:
#
#   sxbics       Sum of exp(-1/2*BIC)
#   fpprob       Formatted posterior probablity (for models)
#   mpprob       Model posterior probability
#
# Determine range of indexes
#
    set lastindex [llength $covlist]
    set firstindex 1
    set h2rs ""
    if {[if_parameter_exists h2r]} {
	set firstindex 0
	set h2rs h2r
    }
    set numh2qs 0
    for {set i 0} {1} {incr i} {
	if {![if_parameter_exists h2q$i]} {break}
	set firstindex [expr 0 - $i]
	set h2rs "h2q$i $h2rs"
	incr numh2qs
    }
#
# Now initialize sums for parameters in this range
#
    for {set i $firstindex} {$i <= $lastindex} {incr i} {
	set p $i   ;# Make nice suffix
	if {$p < 0} {
	    set p [catenate _ [expr abs($i)]]
	}
	set psum$p 0.0
	set ppprob$p 0.0
	set sesum$p 0.0
    }
#
# Setup allparams...list of all parameter names
#
    set allparams $h2rs
    foreach cov $covlist {
	set allparams "$allparams b$cov"
    }
#
# Compute sxbics: sum of exp(-1/2*BIC)
#
    ifdebug puts "Summing exp(-1/2 * BIC)"
    set bestbic [lindex [lindex $winmods 0] 2]
    set sxbics 0.0
    set modcount 0
    set winnames ""
    foreach mod $winmods {
	set bic [lindex $mod 2]
	if {$bic - $bestbic <= $cutoff} {
	    set ebic [expr exp (-0.5 * $bic)]
	    set sxbics [expr $sxbics + $ebic]
	}
    }
#
# Compute PProbs for each model and write window file
#
    ifdebug puts "Computing model pprobs and writing window file"
    set wheadings "Model BIC PProb SNP_Flags"
    set wformats "%20s %11.4f %10s %[expr 2*$lastindex]s"
    set wexpressions {{$modname} {$modbic} {$fpprob} {$snpflags}}
    set wfile [resultfile -create [full_filename stepup.win] \
		   -headings $wheadings -formats $wformats \
		   -expressions $wexpressions]
    resultfile $wfile -header

    set pprobs {}  ;# why is this needed?
    set wincount 0
    set forced_cov0 0
    foreach mod $winmods {
	set modname [lindex $mod 0]
	set modbic [lindex $mod 2]
	if {$modname == "cov" && $modbic - $bestbic > $cutoff} {
	    ifdebug puts "Calculating pprob for forced null model"
	    set forced_cov0 1
	    set mpprob [expr 1.0 / ($sxbics + 1.0)]
	    set fpprob [format %10.6f $mpprob]
	    if {0.0001 >  $fpprob} {
		set fpprob [format %10.3g $mpprob]
	    }
	} else {
	    lappend winnames [lindex $mod 0]
	    incr wincount
	    ifdebug puts "Calculating mpprob the standard way"
	    set mpprob [expr exp(-0.5*$modbic) / $sxbics]
	    set fpprob [format %10.6f $mpprob]
	}
	set snpflags ""
	set snpset [name2set $modname]
	for {set j 1} {$j <= $lastindex} {incr j} {
	    if {-1 != [lsearch $snpset $j]} {
		set snpflags "$snpflags 1"
	    } else {
		set snpflags "$snpflags 0"
	    }
	}
	resultfile $wfile -write
#
# Now, re-maximize window models for standard errors and compute param sums
#
	ifdebug puts "Re-maximizing $modname"
	load model [full_filename stepup.null0]
	set testcovs [name2set $modname]
	set testparams $h2rs
	foreach cnum $testcovs {
	    set covname [lindex $covlist [expr $cnum - 1]]
	    covariate $covname
	    set testparams "$testparams b$covname"
	}
	option standerr 1
	eval maximize $stepverbosity
	save model [full_filename $modname]

	ifdebug puts "Computing parameter sums"
	if {$forced_cov0 && $modname == "cov"} continue
	for {set i $firstindex} {$i <= $lastindex} {incr i} {
	    set p $i   ;# Make nice suffix
	    if {$p < 0} {
		set p [catenate _ [expr 0 - $i]]
	    }
	    set param [lindex $allparams [expr $i + $numh2qs]]
	    ifdebug puts "  for Parameter $param"
	    if {$i < 0 || 0<=[lsearch -exact $testparams $param]} {
		ifdebug puts "    found parameter in $testparams"
		set value [parameter $param =]
		set weighted [expr $mpprob * $value]
		ifdebug puts \
		 "for $i adding value $value mpprob $mpprob weighted $weighted"
		set psum$p [eval expr \${psum$p} + $weighted]
#
# If element is "non-zero"
#   it adds to the posterior probability of that parameter
#
		if {abs($value) >= 1e-12} {
		    set ppprob$p [eval expr \${ppprob$p} + $mpprob]
		}
#
# Compute SE square sums
#
		set pse [parameter $param se]
		set variance [expr $pse * $pse]
		set sesum$p [eval expr \${sesum$p} + \
				 ($mpprob * ($variance + ($value*$value)))]
	    }
	}
    }
#
# Now compute actual SE values from sums
#
    ifdebug puts "Computing SE values"
    for {set i $firstindex} {$i <= $lastindex} {incr i} {
	set p $i
	if {$p < 0} {
	    set p [catenate _ [expr abs($i)]]
	}
	set emean2 [eval square \${psum$p}]
	ifdebug puts "for $i, emean2 is $emean2"
	ifdebug eval puts \"sesum$p is \${sesum$p}\"
	set sesum$p [expr sqrt ([eval expr \${sesum$p} - $emean2])]
    }
#
# Write stepup.avg
#

    set outfilename [full_filename stepup.avg]
    set soutfile [open $outfilename w]
    putstee $soutfile "    *** Number of Models in Window: $wincount"
    if {$user_log_n} {
	putstee $soutfile "    *** User-supplied log(n) is $user_log_n"
    } else {
	putstee $soutfile "    *** Computed log(n) is $log_n"
    }

    putstee $soutfile "    *** Window:  $winnames"
    putstee $soutfile ""
    close $soutfile

    set headings {Component Average {Std Error} Probability}
    set width_needed 32
    for {set i $firstindex} {$i < $lastindex} {incr i} {
	set component [lindex $covlist $i]
	if {$width_needed < [string length $component]} {
	    set width_needed [string length $component]
	}
    }
    set aformats "%[catenate $width_needed s] %-13.8g %-13.8g %-12.7g"

    set resultf [resultfile -append $outfilename -display \
	    -headings $headings \
	    -formats $aformats \
	    -expressions {$component $average $stderror $postprob}]
    resultfile $resultf -header

    for {set i $firstindex} {$i <= $lastindex} {incr i} {
	set p $i
	if {$i == 0} {
	    set name h2r
	} elseif {$i < 0} {
	    set p [catenate _ [expr abs($i)]]
	    set name h2q$[expr abs($i)]
	} else {
	    set name [lindex $covlist [expr $i - 1]]
	}
	if {$i == 0} {
	    set component H2r
	} elseif {$i < 0} {
	    set component H2q[expr 0 - $i]
	} else {
	    set component [lindex $covlist [expr $i-1]]
	}
	set average [eval expr \${psum$p}]
	set stderror [eval expr \${sesum$p}]
	set postprob [eval expr \${ppprob$p}]
	resultfile $resultf -write
    }
    return ""
}





proc Window_New {like0} {
    set newwindow [list [list [list cov $like0 0]]]
    return $newwindow
}

proc Window_Add {window newdfwindow} {
    return [lappend window $newdfwindow]
}

#
# Window_Purge will purge models below BIC cutoff
#   By design, null model is never purged, callers beware.
# If log_n is provided, BIC's will be recalculated
#
proc Window_Purge {window cutpoint symmetric {log_n 0}} {

    set excluded {}
    ifdebug puts "Window_Purge with cutpoint $cutpoint"
    set null0_like [lindex [lindex [lindex $window 0] 0] 1]
    set newwindow [Window_New $null0_like]
    set dfcnt [llength $window]
    for {set dfin 1} {$dfin < $dfcnt} {incr dfin} {
	set df [lindex $window $dfin]
	set newmodels {}
	foreach model $df {
	    set thisname [lindex $model 0]
	    set thislike [lindex $model 1]
	    set thiset [name2set $thisname]
	    ifdebug puts " testing window model $thisname"
	    if {$log_n} {
		set usedf [llength $thiset]
		set bic [bic_calculate $thislike $null0_like $usedf $log_n]
		ifdebug puts "  with new calculated BIC: $bic"
	    } else {
		set bic [lindex $model 2]
	    }
	    set strict 1
	    if {$bic > $cutpoint} {
		ifdebug puts "   Below cutoff...Removed"
	    } else {
		if {!$symmetric} {
		    ifdebug puts "   Strict Test..."

#
# Apply "strict" test
# Use newwindow because it will contain recalculated BICs, if any
#
		    if {$dfin > 1} {
			for {set tdfin 0} {$tdfin < $dfin} {incr tdfin} {
			    set testdf [lindex $newwindow $tdfin]
			    foreach testmod $testdf {
				ifdebug puts \
			      "    Comparing with [lindex $testmod 0]"
				set testbic [lindex $testmod 2]
				if {$testbic > $bic} {
				    ifdebug puts "     Better BIC!"
				} else {
				    set testname [lindex $testmod 0]
				    set testset [name2set $testname]
				    if {![subset $thiset $testset]} {
					ifdebug puts "     Set mismatch"
				    } else {
					lappend excluded [lindex $model 0]
					set strict 0
					break
				    }
				}
			    }
			    if {!$strict} {
				break
			    }
			}
		    }
		}
		if {$strict} {
		    lappend newmodels [list $thisname $thislike $bic]
		}
	    }
	}
	set newwindow [Window_Add $newwindow $newmodels]
    }
#
# Show list of models excluded by strict rule
#
    if {{} != $excluded} {
	set maxline 78
	putsout stepup.history "\n    *** Models excluded by strict rule:"
	set beyondindex [llength $excluded]
	set exindex 0
	while {$exindex < $beyondindex} {
	    set line "       "
	    set position 7
	    set oldline $line
	    set oldposition $position
	    while {$position < 78 && $exindex < $beyondindex} {
		set oldline $line
		set oldposition $position
		set line "$line [lindex $excluded $exindex]"
		incr exindex
		set position [string length $line]
	    }
	    if {$position > 78 && [llength $line] > 1} {
		set line $oldline
		set position $oldposition
		incr exindex -1
	    }
	    putsout stepup.history $line
	}
    }
    ifdebug puts $window
    return $newwindow
}

#
# Window_Models returns a single list of models in BIC sorted order
#   Assumption: Window_Purge was called earlier

proc Window_Models {window} {
    set mods {}
    foreach dfwindow $window {
	foreach dfmod $dfwindow {
	    lappend mods $dfmod
	}
    }
    set mods [lsort -index 2 -real $mods]
    return $mods
}

#
# Window_MakeTests returns list of "test model names"
# based on last df in window
#
proc Window_MakeTests {window ndf} {
    set tests {}
    set basedf [lindex $window end]
    foreach basemodel $basedf {
	set nullmodelname [lindex $basemodel 0]
	set basecovs [name2set $nullmodelname]
	for {set i 1} {$i <= $ndf} {incr i} {
	    if {-1 == [lsearch -exact $basecovs $i]} {
		set newcovs [concat $basecovs $i]
		set newmodelname [set2name $newcovs]
		setappend tests $newmodelname
	    }
	}
    }
    return $tests
}
proc Window_MakeTestFile {window ndf outfilename} {
    set tests {}
    set basedf [lindex $window end]
    set workfile [open $outfilename.work w]
    foreach basemodel $basedf {
	set nullmodelname [lindex $basemodel 0]
	set basecovs [name2set $nullmodelname]
	for {set i 1} {$i <= $ndf} {incr i} {
	    if {-1 == [lsearch -exact $basecovs $i]} {
		set newcovs [concat $basecovs $i]
		set newmodelname [set2name $newcovs]
		puts $workfile $newmodelname
	    }
	}
    }
    close $workfile
    if {[catch {eval exec [usort] \
		    $outfilename.work >$outfilename.sort}]} {
	puts "/var/tmp has inadequate space for sorting results!"
	global env
	puts "Trying to use $env(HOME)/tmp for sort scratchfile..."
	exec mkdir -p $env(HOME)/tmp
	eval exec [usort] -T $env(HOME)/tmp \
	    $outfilename.work >$outfilename.sort
    }
    exec uniq $outfilename.sort >$outfilename
    file delete $outfilename.work
    file delete $outfilename.sort
    set count [lindex [exec wc $outfilename] 0]
    return $count
}


proc set2name {covlist} {
    set name cov
    set covs [lsort -integer $covlist]
    foreach cov $covs {
	set name "$name.$cov"
    }
    return $name
}

proc name2set {name} {
    return [lrange [split $name .] 1 end]
}

proc bic_calculate {loglike null_loglike df log_n} {
    set lodp [lod -raw $loglike $null_loglike]
    set lambda [expr $lodp * 2 * log (10)]
    set BIC [expr ($df * $log_n) - $lambda]
    return $BIC
}

#
# Parallel support procedures:
#   stepup_begin_launch
#   stepup_launch_shepherd
#   stepup_step
#
# Files:
# outdir/stepup.covlist                    list of all test covariates
# outdir/stepup.vars                       other essential variables
# outdir/stepup.files.list                 list of all files required by p jobs
# outdir/stepup.testmods.<trynum>          all tests for this DF
# outdir/stepup.null0                      null and base model
# outdir/stepup.[pid].<tn>.$dimension.<retrynum>  launch output directory
#                                          <tn> is "take a number" never used
#                                             more than once.
# outdir/setpup.out.d$dimension            output file to be written
# outdir/stepup.launchfiles                file listing files required
#
# Files created or copied to working directory:
# stepup.files.list
#
# -

proc stepup_begin_launch {dimension number_tests null0_like log_n} {
#
# Clean out old files
#
    set oldfiles [glob -nocomplain [full_filename stepup.*.*.*.*]]
    foreach oldfile $oldfiles {
	catch {exec rm -rf $oldfile}
    }
#
    set trynum 1
    set predimension [expr $dimension - 1]
#
#       Make file with list of all files required
#         start with stuff generally needed by SOLAR models...
#
    set lfilename stepup.files.list
    set outprefix ""
    if {"/" == [string index $lfilename 0]} {
	error "For parallel stepup, outdir must be relative to working directory"
    }
    set lfile [open $lfilename w]
#
# Determine if matrices
#
    set all_matrixes [matrix]
    set length_all_matrixes [llength $all_matrixes]
    set matrix_names {}
    for {set imatrix 0} {$imatrix < $length_all_matrixes} {incr imatrix} {
	set iname [lindex $all_matrixes $imatrix]
	if {$iname == "matrix"} {
	    incr imatrix
	    set iname [lindex $all_matrixes $imatrix]
	    if {$iname == "load"} {
		incr imatrix
		set iname [lindex $all_matrixes $imatrix]
		if {$iname != ""} {
		    lappend matrix_names $iname
		}
	    }
	}
    }
#
# Add matrices and check for non-local matrices
#
    set matrix_names_needed {}
    foreach matrix_name $matrix_names {
	if {"/" != [string index $matrix_name 0]} {
	    if {-1 != [string first / $matrix_name]} {
		close $lfile
		puts "Model contains matrix $matrix_name"
		error "For parallel stepup, model matrices should be in working directory\nor specified with absolute pathname in the model file."
	    }
	    puts $lfile $matrix_name
	}
    }
    puts $lfile "pedindex.out"
    puts $lfile "pedindex.cde"
#
# One or more phenotypes files, with or without .cde file
#
    set pifile [open phenotypes.info]
    set pofile [open [full_filename phenotypes.info] w]
    while {-1 != [gets $pifile phenname]} {
	puts $lfile $phenname
	if {-1 == [string first / $phenname]} {
	    puts $pofile $phenname
	} else {
#
# In our phenotypes.info, reference non-wd phenfiles in wd
#   because doscript copies them there
#
	    puts $pofile [file tail $phenname]
	}
	if {[file exists [file rootname $phenname].cde]} {
	    puts $lfile  [file rootname $phenname].cde
	}
	if {[file exists [file rootname $phenname].CDE]} {
	    puts $lfile [file rootname $phenname].CDE
	}
	if {[file exists $phenname.cde]} {
	    puts $lfile $phenname.cde
	}
	if {[file exists $phenname.CDE]} {
	    puts $lfile $phenname.CDE
	}
    }
    close $pifile
    close $pofile
    puts $lfile [full_filename phenotypes.info]
#
# Write other variables to a file
#
    set vfilename [full_filename stepup.vars]
    set vfile [open $vfilename w]
    puts $vfile $null0_like
    puts $vfile $log_n
    close $vfile
    puts $lfile $vfilename
#
# Required files in outdir
#
    puts $lfile [full_filename stepup.null0.mod]
    puts $lfile [full_filename stepup.testmods.$trynum]
    puts $lfile [full_filename stepup.covlist]
    close $lfile
#
# Note: all files in lfile will be copied to /tmp working directory
#   in parallel tasks
#
# Continue launch in recursive routine which also handles relaunches
#
    stepup_launch $dimension $number_tests $trynum $lfilename
#
# Rename temp output file to final one
#
    file rename [full_filename stepup.out.d$dimension.tmp] \
	[full_filename stepup.out.d$dimension]
#
# Done
#
    return ""
}

proc use_outdir_if_available {args} {
    if {[catch {return [full_filename $args]}]} {
	return $args
    }
}


proc cpus_available {} {
    puts "Querying gridware..."
    while {0 != [catch { \
	     exec qstat -g c >[use_outdir_if_available stepup.launch.qstat] \
			 }]} {
	puts "Gridware swamped, trying again..."
	after 2000
    }
    set qsfile [open [use_outdir_if_available stepup.launch.qstat]]
    set availi 4

    gets $qsfile line
    gets $qsfile line
    gets $qsfile line
    set cpus_available [lindex $line $availi]
    puts "qstat -g c: $cpus_available"
    close $qsfile
    return $cpus_available
}

# solar::howmanyranch --
#
# Purpose: Show how many ranch machines a user is using
#
# Usage:  howmanyranch <userid>
#
# Notes:  See also whoranch.  Requires use of Grid Engine software.
# -

# solar::whoranch
#
# Purpose: Show how many ranch machines each ranch user is using
#
# Usage:   whoranch
#
# Notes:   See also howmanyranch.  Requires use of Grid Engine software.
# -

# solar::parallel -- private
#
# Purpose: EXPERIMENTAL parallel operation parameters
#
# Usage:   parallel [margin [<marginsize>]]
#          parallel [minjobsize [<minjobsize>]]
#          parallel [fraction [<fraction>]
#          parallel [ignore [<ignoreid> | <ignoreidlist>]]
#          parallel ignoreadd <ignoreid>
#
# When only the parameter (such as "margin") is named, the procedure
# returns the currently operational value, either from the default or
# previous user selection.  If a value is specified, that replaces the
# current value.  ignoreadd is a special argument used only to add to
# the existing ignore list.
#
# The defaults are programmed for a medium-to-low priority background task,
# however there is a first-come-first-served aspect also.
#
#           margin       the number of machines to be left over after a new
#                          parallel job is launched.  The default is 3500.
#
#           minjobsize   the minimum size of any job.  Often it is more efficient
#                          to have jobs larger than size 1.  The default is 4
#                          meaning that each job has to have at least 2 tasks
#                          unless there are fewer than 2 tasks total.
#
#           fraction     the fraction of available machines which can be used
#                           if the margin is exceeded.  The default is 0.35
#
#           ignore       list of user id's to ignore in evaluating machine
#                        availability, as these user's jobs are temporary
#                        or fallback jobs.  2nd argument can either be single
#                        userid or list of userid's enclosed in braces.
#
#           ignoreadd <newid>    Add <newid> to list of user id's to ignore.
# -

proc parallel {args} {
    global SOLAR_Parallel_Margin
    global SOLAR_Parallel_Fraction
    global SOLAR_Parallel_MinJobsize
    global SOLAR_Parallel_State
    global SOLAR_Parallel_Ignore

    if {$args == "margin"} {
	if {[if_global_exists SOLAR_Parallel_Margin]} {
	    return $SOLAR_Parallel_Margin
	}
	return 3500
    }
    if {"margin" == [lindex $args 0]} {
	set newmargin [lindex $args 1]
	if {![is_integer $newmargin]} {
	    error "margin must be an integer"
	}
	return [set SOLAR_Parallel_Margin $newmargin]
    }
    if {$args == "minjobsize"} {
	if {[if_global_exists SOLAR_Parallel_MinJobsize]} {
	    return $SOLAR_Parallel_MinJobsize
	}
	return 3
    }
    if {"minjobsize" == [lindex $args 0]} {
	set newminjobsize [lindex $args 1]
	if {![is_integer $newminjobsize]} {
	    error "minjobsize must be an integer"
	}
	return [set SOLAR_Parallel_MinJobsize $newminjobsize]
    }
    if {$args == "fraction"} {
	if {[if_global_exists SOLAR_Parallel_Fraction]} {
	    return $SOLAR_Parallel_Fraction
	}
	return 0.35
    }
    if {"fraction" == [lindex $args 0]} {
	set newfraction [lindex $args 1]
	if {![is_float $newfraction]} {
	    error "fraction must be a floating point number"
	}
	return [set SOLAR_Parallel_Fraction $newfraction]
    }
    if {$args == "ignore"} {
	if {[if_global_exists SOLAR_Parallel_Ignore]} {
	    return $SOLAR_Parallel_Ignore
	}
	return eugeneid
    }
    if {"ignore" == [lindex $args 0]} {
	set newignore [lindex $args 1]
	return [set SOLAR_Parallel_Ignore $newignore]
    }
    if {"ignoreadd" == [lindex $args 0]} {
	set newignore [lindex $args 1]
	return [set SOLAR_Parallel_Ignore [concat [parallel ignore] $newignore]]
    }
    error "Invalid argument to parallel"
}

# SOLAR::parallel-tips -- private
#
# Ways to increase machine usage from low priority defaults:
#   parallel margin 0    use all available machines
#   parallel fraction 1  use 100% of available machines
#   parallel ignore ...  ignore the machines used by these users
# -


#
# "margin" is soft
# If fewer than FRACTION (0.3) of the cpus are available after margin,
# We get FRACTION of the cpus instead
#
proc cpus_available_soft_margin {args} {
    set margin [parallel margin]
    set cpus_available [cpus_available]
    set ignore [parallel ignore]
    foreach ig $ignore {
	catch {
	    set thisig [howmanyranch $ig]
	    if {[is_integer $thisig]} {
		set cpus_available [expr $cpus_available + $thisig]
	    }
	}
    }
    set cpus_useable [expr $cpus_available - $margin]
    if {$cpus_useable < $cpus_available * [parallel fraction]} {
	set cpus_useable [expr $cpus_available * [parallel fraction]]
	if {$cpus_useable < 0} {
	    error "No cpus available for parallel operation"
	}
    }
    return [expr int($cpus_useable)]
}

proc take_a_number {} {
    global SOLAR_Take_A_Number
    if {![if_global_exists SOLAR_Take_A_Number]} {
	set SOLAR_Take_A_Number 0
    }
    incr SOLAR_Take_A_Number
    return $SOLAR_Take_A_Number
}

proc stepup_launch {dimension number_tests trynum lfilename} {
    puts "There are $number_tests tasks"
    puts "Determining number of jobs to launch..."
#
# Determine available machines
#
    set cpus_useable 0
    while {1} {
	set cpus_useable [cpus_available_soft_margin]
	if {$cpus_useable > 100 || $cpus_useable*20 > $number_tests} break
	puts "Too few machines now available, waiting for more..."
	waitdots 60
    }
#
# Now, determine how many we are actually going to use
#
    set maximum_cpus [expr 1 + ($number_tests-1) / [parallel minjobsize]]
#
# Avoid "one task" situation because it can lead to 100% noncompletion
#   which wastes a lot of time
#
    if {$maximum_cpus <= [parallel minjobsize]} {
	set maximum_cpus $number_tests
    }
    if {$maximum_cpus <= $cpus_useable} {
	set reserved_cpus $maximum_cpus
    } else {
	set reserved_cpus $cpus_useable
    }
#
# Clear launch working directory, if any
#
    set tan [take_a_number]
    set launchdir [full_filename stepup.[pid].$tan.$dimension.$trynum]
    exec rm -rf $launchdir
    exec mkdir $launchdir
#
# Get current program name
#
    global env
    set programpath $env(SOLAR_PROGRAM_NAME)
    set programname [file tail $programpath]
#
# Launch scripts
#
    set starting_task 1
    set solarname solar
    set jobsize [expr int (ceil ($number_tests / double($reserved_cpus)))]
    puts "precomputed maximum cpus for this dimension is $reserved_cpus"
    puts "precomputed jobsize is $jobsize"
    set launchstat [launch -proc stepup_step -extra_args \
			"$trynum" -filelist $lfilename \
			-n $number_tests -jobsize $jobsize \
			-outputdir $launchdir]
    puts "launch status: $launchstat"
#
# Note: launch uses the minimum number of jobs consistent with getting all
# tasks completed with a given jobsize.  Therefore, it might not use all the
# machines we said it could use.  We can get the actual number of machines
# used, which is very important, from the launch.info file.  This is somewhat
# counterintuitively called "maxjobs" it should be "actualjobs".
#
    getinfo launch.info maxjobs
    puts "Launched $maxjobs jobs"

    return [stepup_shepherd $dimension $maxjobs $jobsize $trynum \
		$lfilename $tan $launchstat $number_tests]
}

proc waitdots {waitsecs} {
    set dots [expr round($waitsecs / 10)]
    set extra [expr round ($waitsecs - 10*$dots)]
    while {$dots > 0} {
	after 10000
	puts -nonewline .
	flush stdout
	incr dots -1
    }
    if {$extra} {
	after [expr 1000 * $extra]
	puts -nonewline .
	flush stdout
    }
    puts ""
}

    


#
# solar::stepup_shepherd -- private
#
# Purpose:  "Shepherd" parallel tasks, making sure they ultimately complete
#           successfully.  Uses "relaunch" to get all tasks launched OK.
#           Uses "finish" to finish, combined with modified cleanrecords
#           (modified to deal with lack of comma delimiting) to compile
#           all results for a single df.  Results are ultimately written
#           to the output file stepup.out.
#
#           If this procedure fails to get everything completed, it recurses
#           back through step_launch to launch the remaining stragglers.
#
#-

proc stepup_shepherd {dimension njobs jobsize trynum lfilename \
			  tan launchstat ntests} {
    set start_time [clock seconds]
    set started 0
    set fraction_started 0
    set newtrynum [expr $trynum + 1]
#
# Wait until all jobs have started, or would be expected to be started
#
    puts "Waiting for most jobs to start..."

    set failure_time [expr 60 * 15]
    set initial_wait 10

    set ldirname [full_filename stepup.[pid].$tan.$dimension.$trynum]
    set expected_total_time 0
    while {1} {
    	after [expr $initial_wait*1000]
	puts "Looking at $ldirname"
	set started [lindex [exec ls $ldirname | wc] 0]
	if {$started == $njobs} {
	    puts "All tasks started, breaking wait"
	    break
	}
	set current_time [clock seconds]
	if {$current_time - $start_time > $failure_time} {
#
# On complete failure, assume launch failed and recurse to do it
#
	    if {$started == 0} {
		puts "Launch failed completely, recursing to redo"
		set oldjobarray [lindex $launchstat 2]
		set oldjobnumber [lindex [split $oldjobarray .] 0]
		puts "Deleting old job array $oldjobnumber"
		catch {exec qdel -f $oldjobnumber}
		set oldfilename [full_filename stepup.testmods.$trynum]
		set newname [full_filename stepup.testmods.$newtrynum]
		file copy -force $oldfilename $newname
		set lfile [open stepup.files.list a]
		puts $lfile $newname
		close $lfile
		return [stepup_launch $dimension $ntests \
			    $newtrynum $lfilename]
	    }
#
# On partial failure, this is probably a "slow start" caused by someone
# else taking some of the machines we thought we were going to get before we
# got them.
#
# But what does this means in terms of how long to set the relaunch wait?
# Initial guess: average of this long wait and zero wait
#
	    puts "$failure_time seconds exceeded, breaking wait"
	    set expected_total_time [expr $failure_time / 2]
	    break
	}
	set fraction_started [expr double($started) / $njobs]
	puts "[format %.4g [expr 100.0 * $fraction_started]]% of jobs now started"
	if {$expected_total_time == 0} {
	    if {$fraction_started >= 0.5} {
		puts "Entering second phase of wait"
		set overcompletion_factor 1.4
		set fraction_time [expr [clock seconds] - $start_time]
		set reciprocal_started [expr 1.0 / $fraction_started]
		set expected_total_time [expr round($fraction_time * \
							$overcompletion_factor * \
							$reciprocal_started)]
	    }
	} elseif {$current_time - $start_time > $expected_total_time} {
	    puts "Jobs were expected to be started by now, breaking wait"
	    break
	}
    }
    set relaunch_time $expected_total_time
    if {$relaunch_time == 0} {
	puts "Setting expected to actual time"
	set relaunch_time [expr [clock seconds] - $start_time]
    }
    set relaunch_time [expr $relaunch_time / 2]
    if {$relaunch_time < $initial_wait} {
	puts "Setting expected to initial time"
	set relaunch_time $initial_wait
    }
    puts "Wait time is $relaunch_time"
#
# Now, keep relaunching until everything is started, at total time intervals
# But only launch as allowed by margin
#
    puts "Looking for $ldirname"
    set started [lindex [exec ls $ldirname | wc] 0]
    puts "$started out of $njobs have started"
    if {$started < $njobs} {
	set all_started 0
	set large_count 0
	while {1} {
	    set max_relaunch [cpus_available_soft_margin]
	    puts "Maximum available cpus for relaunching is $max_relaunch"
	    set status [relaunch -y -max $max_relaunch]
	    set third_status [lindex $status 2]
	    set status [lindex $status 0]
	    puts "STATUS is $status"
	    if {$status == "All"} {break}
	    if {$third_status == "relaunched" && \
		    [is_integer $status] && $status > 500} {
		if {$large_count > 2} {
		    puts "Many large relaunches.  Something isn't good.  Wait 30 minutes."
		    set relaunch_time 1800
		} else {
		    puts "Extremely large relaunch.  Wait 5 minutes."
		    set relaunch_time 300
		}
		incr large_count
	    } else {
		puts "Waiting $relaunch_time seconds for all jobs to start"
	    }
	    waitdots $relaunch_time
	}
    }
#
# Gather results ("finish")
# If jobsize==1, we're done
# Otherwise, recheck at relaunch_time intervals
#   (turns out to be better than a big long predicted wait that is usually
#    way too long because start time is far longer than the time of one
#    iteration, there is really no good handle on incremental time)
#    Upon 97% of jobs fully completed, we quit.
#       (A fully completed job has all its tasks completed.)
#    When number of tasks completed in
#      last iteration drops below 5% of the average tasks completed
#      per iteration, we quit.  (Average does not count "first interation"
#      because that included many jobs with more than one complete.)

    set last_jobs_completed ""
    set initial_jobs_completed 0
    set initial_time 0
    set penultimate 0
    while {1} {
	puts [exec date]
	puts "Gathering results..."
	set remaining_time ""
	set gathering_begin [clock seconds]
	set status [finish -any -n]
	set gathering_time [expr [clock seconds] - $gathering_begin]
	puts "Result gathering time: $gathering_time seconds"
	set percent_fully_complete [lindex $status 6]
	set total_jobs [lindex $status 3]
	set status [lindex $status 0]
	puts "$status tasks done"
	if {$jobsize==1} {break}
	if {$status == "All"} {
	    puts "Finish reports all jobs done, breaking wait"
	    break
	}
	if {$percent_fully_complete > 98} {
	    puts "More than 98% of jobs completed fully, breaking wait"
	    break
	}
	if {![is_integer $status]} {continue}
	set percent_complete [expr 100*double($status)/$total_jobs]
	set remaining_time ""
	if {$last_jobs_completed == {}} {
	    set first_tcpi $status
	    set last_jobs_completed $status
	    set initial_jobs_completed $status
	    set initial_time [clock seconds]
	    set timestamps $initial_time
	    set countstamps $status
	    set pent_index -1    ;# pent_index is end-1	    
	} else {
	    set current_time [clock seconds]
	    lappend timestamps $current_time
	    lappend countstamps $status
	    incr pent_index
	    set latest_complete [expr $status - [lindex $countstamps $pent_index]]
	    puts "$latest_complete tasks completed in this iteration"
	    set average_tcps [expr (double($status) - $first_tcpi) / \
				  ($current_time - $initial_time)]
	    puts "About [format %.4g $percent_complete]% tasks completed"
	    puts "About [format %.4g $percent_fully_complete]% of jobs fully complete"
	    puts "Note: this counts some tasks which may be redundant"
	    puts "Average tasks per second: [format %.4g $average_tcps]"
#
# Compute tasks/sec for the most recent minute or more
#
	    set breakout 0
	    for {set i $pent_index} {$i > 0} {incr i -1} {
		set old_time [lindex $timestamps $i]
		if {$current_time - $old_time > 60} {
		    set old_count [lindex $countstamps $i]
		    set tasks_completed [expr $status - $old_count]
		    set time_elapsed [expr $current_time - $old_time]
		    set time_elapsed [highest 0.5 $time_elapsed]
		    set current_tcps [expr double($tasks_completed) / \
					  $time_elapsed]
		    puts "Current tasks per second: [format %.4g $current_tcps]"
		    if {$current_tcps < 0.02 * $average_tcps} {
			if {$penultimate > 0} {
			    puts "Rate below %2 of average on second attempt"
			    set breakout 1
			} else {
			    if {$average_tcps > 1.0/30} {
				puts "Rate below 2% of average, with high average"
				set breakout 1
			    } else {
				puts "Rate below 2% of average"
				puts "Trying once more after 60 seconds..."
				set penultimate 1
				waitdots 60
			    }
			}
		    } else {
			set penultimate 0
		    }
		    break
		}
	    }
	    if {$breakout} {
		puts "Breaking wait for results"
		break
	    }
	    set last_jobs_completed $status
	    set total_time [expr round($total_jobs * \
		(double($current_time - $initial_time) / \
	     [highest 0.5 [expr ($status - $initial_jobs_completed)]]))]
	    puts "Projected total time is $total_time seconds"
	    set remaining_time [expr round($total_time - \
					       ($current_time - $initial_time))]
	    puts "Projected remaining time is $remaining_time seconds"
	}
#
# If this is not a special case of being nearly done
# Determine time to wait before collecting results again
#
	if {$penultimate < 1} {
#
# Start with "relaunch_time" which is derived from the time it took to launch
# all the jobs and do 1 iteration and so therefore reflects grid load
#
# (This could all be replaced in a future revision simply based on predicted
#  completion time...)
#
	    set finish_wait $relaunch_time
#
# limit wait time to three minutes, the maximum ordinary start, to filter
# out slow starts that occur if machines are grabbed by others before we get
# them.
#
	    set max_task_wait_time 180
	    if {$finish_wait > $max_task_wait_time} {
		set finish_wait $max_task_wait_time
	    }
#
# But then, allow at least 2x as much time as it takes to gather results
#   so that most of the time, result gathering is not happening, because
#   gathering lowers efficiency of result production
#
	    if {$finish_wait < 2 * $gathering_time} {
		set finish_wait [expr 2 * $gathering_time]
	    }
#
# But do not allow more wait time than computed remaining time + 15 seconds
# OR, 20 seconds on the first go (maximum we are prepared to lose, since this
# might be nearly done already)
#
	    if {{} != $remaining_time} {
		set allow_time [expr $remaining_time + 15]
	    } else {
		set allow_time 20
	    }
	    if {$finish_wait > $allow_time} {
		set finish_wait $allow_time
	    }
	    puts "Waiting $finish_wait seconds for all jobs completed"
	    waitdots $finish_wait
	}
    }
#
# Remove duplicate and incomplete records
#   and append good records to temp copy of ultimate output file
#
    puts [exec date]
    puts "Testing results for completion..."
    exec [usort] -k 1,1 $ldirname.all >$ldirname.sort

    set infile [open $ldirname.sort]
    set nfile [open [full_filename stepup.testmods.$trynum]]

    set newfilename [full_filename stepup.testmods.$newtrynum]
    set newfile [open $newfilename w]

    set outfile [open [full_filename stepup.out.d$dimension.tmp] a]
    set lastname ""
    set testname ""
    set missing_count 0
    while {-1 != [gets $infile line]} {
	puts "got line $line"
	if {"" == $line} {continue}
	if {[llength $line] != 3} {continue}
	set name [lindex $line 0]
	puts "Testing name $name"
	if {0 == [string compare $name $lastname]} {
	    puts "removing duplicate record for $name"
	} else {
	    while {1} {
		if {-1 == [gets $nfile testname]} {
		    puts "ERROR getting model name to test"
		    break
		}
		puts "got testname $testname"
		set testname [lindex $testname 0]
		if {$testname == ""} {continue}
		if {0==[string compare $name $testname]} {
		    break
		} else {
		    puts $newfile $testname
		    puts "Missing model $testname"
		    incr missing_count
		    set testname ""
		}
	    }
#	    puts "  Write record for $name"
	    puts $outfile $line
	    set lastname $name
	}
    }
    while {-1 != [gets $nfile testname]} {
	if {$testname == ""} {continue}
	puts "Missing additional model $testname"
	puts $newfile $testname
    }

    close $infile
    close $outfile
    close $nfile
    close $newfile
#
# If nothing is missing, we're done
#
    if {$missing_count == 0} {
	puts "No missing names"
	puts [exec date]
	return OK
    }
#
# Recurse to handle new missing records
#
    set lfile [open stepup.files.list a]
    puts $lfile [full_filename stepup.testmods.$newtrynum]
    close $lfile
    puts "Now recursing to handle missing records"
    return [stepup_launch $dimension $missing_count $newtrynum $lfilename]
}

proc stepup_step {directory tasksize taskno trynum} {

    newtcl
    set outputfile $directory/$taskno
    file delete $outputfile
    set all_tests [listfile stepup.testmods.$trynum]
    set covlist [listfile stepup.covlist]
    set number_tests [llength $all_tests]
    set othervars [listfile stepup.vars]
    set null0_like [lindex $othervars 0]
    set log_n [lindex $othervars 1]
    
# Note: taskno starts with 1, but first_task_num is an index that starts with 0
    set first_task_num [expr ($taskno-1)*$tasksize]
    set last_task_num [expr $first_task_num + $tasksize - 1]
#
# Maximize each model and write results
#
    for {set task $first_task_num} {$task <= $last_task_num} {incr task} {
	set modelname [lindex $all_tests $task]
	puts "Evaluating model $modelname"
	load model stepup.null0
	set testcovs [name2set $modelname]
	foreach covnumber $testcovs {
	    covariate [lindex $covlist [expr $covnumber - 1]]
	}
	eval maximize
	set loglike [format %.3f [loglike]]
	set bic [format %8.4f [bic_calculate [loglike] $null0_like \
				   [llength $testcovs] $log_n]]
	puts "Writing to $outputfile"
	set ofile [open $outputfile a]
	puts $ofile "$modelname $loglike $bic"
	close $ofile
    }
}


# solar::doranch --
#
# Purpose:  execute a script on every ranch machine (usually for /tmp cleanup)
#
# DO NOT USE THIS FOR SUBMISSION OF REGULAR JOBS because it bypasses
# the Gridware queing system, which it must do for cleanup of ALL machines.
#
# MUST BE RUN ON MEDUSA (only medusa addresses all other ranch machines)
#
# See also "stepup -parclean" which uses doranch to cleanup junk created by
# forcing a "stepup -par" job to quit.
#
# Usage:    doranch <procname> <argument>
#
#           doranch cleanuser <username>   ;# delete ALL user's /tmp files on
#                                          ;# the ranch (Note: you can only
#                                          ;# delete files for which you have
#                                          ;# delete privilege, usually because
#                                          ;# of owning them.)
#
#           doranch finduser <username>    ;# find all my /tmp files on the
#                                          ;# ranch but do not delete them.
#                                          ;# Findings are written
#                                          ;# to finduser.out.  If -all is
#                                          ;# used, all users are shown.
#
#           doranch cleantmp <dirname>.    ;# same as "stepup -parclean"
#                                          ;# delete all /tmp/<dirname>.*
#                                          ;# files.  (parallel stepup dirs
#                                          ;# are prefixed with <dirname>
#                                          ;# followed by dot.
#
#           doranch findtmp <dirname>      ;# find all name* directories
#                                          ;# but do not delete them.  Findings
#                                          ;# are written to findtmp.out.
#
#           doranch cleanme now            ;# same as
#                                          ;# doranch cleantmp <username>
#
#           make_rhosts                    ;# make a new .rhosts file, or
#                                          ;# append to existing one to
#                                          ;# make it complete.  It may be
#                                          ;# useful to delete old .rhosts
#                                          ;# file first if it contains errors.
#
#           showspace                      ;# Return sorted list of /tmp
#                                          ;# storage used by all users
#                                          ;# in showspace.out.  Uses
#                                          ;# doranch finduser -all, unless
#                                          ;# existing finduser.out is found.
#
#
#           <procname> is the name of the procedure to be run on every
#                      machine.  procedures cleanuser, finduser, cleantmp,
#                      findtmp, and cleanme are provided, but user-written
#                      scripts could be used also.
#
#           <username> is the username.
#
#           cleantmp is a procedure that deletes all files and directories
#           in /tmp which match the specified prefix, after which a wildcard
#           * is assumed.  For example "cleantmp charlesp." would delete a
#           directory named "/tmp/charlesp.11019.2"
#
# Notes:  It is useful to run ranch jobs in subdirectories of the /tmp
#         directory to minimize network traffic.  Jobs should be designed to
#         cleanup after themselves in normal operation by deleting the
#         /tmp subdirectory that was used as a working directory.
#
#         However, even when jobs are designed to cleanup after themselves,
#         if the jobs do not run to completion, the cleanup code might never
#         be run.  This is especially true when a user or administrator
#         shuts down a large array job (such as "stepup -par") because of
#         a mistake or emergency.
#
#         That is when "doranch" may be useful.  The "cleanuser" procedure
#         deletes all files owned by the user in /tmp directories on
#         all ranch machines.  The "cleantmp" procedure deletes all files
#         and directories in /tmp prefixed by the cleantmp argument on all
#         ranch machines.
#
#         The doranch procedures listed above may be used in creating custom
#         cleanup options for other scripts.
#
#         Such an emergency cleanup option is already built into the stepup
#         command as option "-parclean".  That uses doranch and cleantmp
#         as shown above.  Authors of other parallel scripts for general
#         create similar script options tailored to the names of /tmp
#         subdirectories they use.
#
#         To see what the "finduser" script looks like, in order to write
#         something similar, use the solar command "showproc finduser".
#
#         All the doranch procedures write to a file named by the specified
#         procname, for example cleanuser writes to a file named cleanuser.out
#         for each file found.  Usually this has two columns, node name
#         and filename.  However, for "finduser" a middle column is added
#         which lists total diskspace used in kbytes.
#
#         Note that a valid .rhosts file is required for usage, and
#         the make_rhosts file will make one.  doranch will complain
#         if the .rhosts file is not present or incomplete.
#
#         If doranch reports failure in connecting to some hosts, it is
#         probably because the passwd and shadow files involved in userid
#         authentication have not been properly updated on those hosts.
#
#         If doranch reports failure in connecting to every ranch host, it
#         probably means that the .rhosts file is invalid, and you should then
#         delete the old .rhosts file and run make_rhosts.
#
#         If doranch hangs at a particular host, that machine is probably
#         down in some unusual way that is not known to gridware.
#         
#-


proc doranch {procname argument} {

    catch {file delete -force $procname.out}
    set hostlist [gethostlist]
    check_rhosts $hostlist
    global env
    set solarpath $env(SOLAR_PROGRAM_NAME)
    puts "Using solarpath $solarpath"
    foreach host $hostlist {
	puts "doing $host..."
	if {[catch {exec rsh $host "$solarpath $procname \
          $argument"}]} {
	    puts "    $host not accessible"
	}
    }
}


proc check_rhosts {hostlist} {
    global env
    set home $env(HOME)
    if {![file exists $home/.rhosts]} {
	error "No .rhosts file; use make_rhosts to create one"
    }
    set rhostlist [listfile $home/.rhosts]
    set nodelist {}
    foreach rhost $rhostlist {
	lappend nodelist [lindex $rhost 0]
    }
    lappend hostlist medusa
    lappend hostlist medusa-gw
    foreach host $hostlist {
	if {-1 == [lsearch -exact $nodelist $host.txbiomedgenetics.org]} {
	    error ".rhost file missing $host: use make_rhosts to update"
	}
    }
    return OK
}

proc getcondoruid {} {
    set infile [open .job.ad]
    while {-1 != [gets $infile line]} {
	if {"Owner" == [lindex $line 0]} {
	    set quotedname [lindex $line 2]
	    return $quotedname
	}
    }
    error "Condor Owner not found in .job.ad file"
}


proc make_rhosts {} {
    global env
    set home $env(HOME)
    set user [lindex [exec /usr/bin/who -m] 0]

    exec touch $home/.rhosts

# Get current .rhosts (if any)

    set oldrs [listfile $home/.rhosts]
    set oldhosts {}
    foreach oldr $oldrs {
	lappend oldhosts [list [lindex $oldr 0] [lindex $oldr 1]]
    }

# Get required high host number

    set hosts [gethostlist]
    set highhost 0
    foreach host $hosts {
	set hostnumber [string range $host 1 end]
	if {[is_integer $hostnumber]} {
	    if {$hostnumber > $highhost} {
		set highhost $hostnumber
	    }
	}
    }
    set highhost [expr $highhost + 500]

# First make sure .rhosts has medusa lines
    
    add_to_rhosts_if_needed medusa $user $oldhosts
    add_to_rhosts_if_needed medusa-gw $user $oldhosts

    for {set i 1} {$i < $highhost} {incr i} {
	add_to_rhosts_if_needed n$i $user $oldhosts
    }
    return OK
}    

proc add_to_rhosts_if_needed {nodename user oldhosts} {
    if {-1 == [lsearch -exact $oldhosts "$nodename.txbiomedgenetics.org $user"]} {
	add_to_rhosts $nodename $user
    }
}

proc add_to_rhosts {nodename user} {
    global env
    set home $env(HOME)
    exec echo "$nodename.txbiomedgenetics.org   $user" >>$home/.rhosts
}


proc cleantmp {name} {
    return [findtmp $name delete]
}


proc findtmp {name args} {
    global env
    set home $env(HOME)
    cd /tmp
    set nodename [exec uname -n]
    set alldirs [glob $name*]
    foreach dir $alldirs {
	if {"" != $args} {
	    catch {exec rm -rf /tmp/$dir}
	    set procname cleantmp
	} else {
	    set procname findtmp
	}
	exec echo "$nodename $dir" >>$home/$procname.out
    }
}


proc cleanuser {username} {
    return [finduser $username delete]
}


proc finduser {username args} {
    set procname finduser
    global env
    set home $env(HOME)
    cd /tmp
    set nodename [exec uname -n]
    set alldirs [glob *]
    foreach dir $alldirs {
	set fileuser ""
	catch {
	    set fulldir [exec ls -ld $dir]
	    set fileuser [lindex $fulldir 2]
	}
	if {$fileuser == $username} {
	    set nameandsize $dir
	    if {"delete" == $args} {
		catch {exec rm -rf /tmp/$dir}
		set procname cleanuser
	    } else {
		set nameandsize [exec du -sk $dir]
	    }
	    exec echo "$nodename $nameandsize" >>$home/$procname.out
	} elseif {$username == "-all"} {
	    set nameandsize [exec du -sk $dir]
	    exec echo "$fileuser $nodename $nameandsize" >>$home/$procname.out
	}
    }
}

# -f  run finduser
proc showspace {args} {
    exec rm -f showspace.out
    set finduser 0
    set badargs [read_arglist $args -f {set finduser 1}]
    if {$finduser || ![file exists finduser.out]} {
	exec rm -f finduser.out
	doranch finduser -all
    } else {
	puts "Using existing finduser file, use -f option to make new one"
    }
    exec [usort] finduser.out -k1,1 >sortuser.out
    after 1000
    set insort [open sortuser.out]
    set username ""
    set space 0
    set records 0
    while {-1 != [gets $insort line]} {
	if {[llength $line]} {
	    set this_username [lindex $line 0]
	    set this_space [lindex $line 2]
	    if {$this_username != $username} {
		if {""!=$username} {
		    lappend usernames [list $username $space]
		    incr records
		}
		set username $this_username
		set space $this_space
	    } else {
		set space [expr $this_space + $space]
	    }
	}
    }
    if {$space > 0} {
	lappend usernames [list $username $space]
	incr records
    }
    set sortnames [lsort -decreasing -integer -index 1 $usernames]
    foreach name $sortnames {
	putsout -d. showspace.out "[lindex $name 0]    [lindex $name 1]"
    }
    return ""
}


proc cleantest {} {
    global env
    set home $env(HOME)
    set ohm [file tail $home]
    exec rsh n10 mkdir /tmp/$ohm
    exec rsh n10 mkdir /tmp/$ohm/foobar
    exec rsh n10 touch /tmp/foobar
}


proc gethostlist {} {
    set hostlist {}
    exec qhost > qhost.data
    set hostdata [listfile qhost.data]
    set first 0
    foreach host $hostdata {
	if {$first < 2} {
	    incr first
	    continue
	}
	if {"-" != [lindex $host 3]} {
	    lappend hostlist [lindex $host 0]
	}
    }
    return $hostlist
}

proc cleanme {name args} {
    global env
    set home $env(HOME)
    cd /tmp
    set alldirs [glob *]
    foreach dir $alldirs {
	if {[file owned $dir]} {
	    catch {exec rm -rf /tmp/$dir}
	    exec echo "[exec uname -n] $dir" >>$home/cleanme.out
	}
    }
}

# solar::bayesavg --
#
# Purpose:  Perform bayesian oligogenic model averaging
#           on covariates or linkage components of the current model.
#
# Usage:    bayesavg [-cov[ariates]] [-ov[erwrite]] [-redo]
#                    [-max <max>] [-cutoff <cutoff>] [-symmetric]
#                    [-list <listfile>] [-fix [cov|param]]
#                    [-size_log_n] [-nose] [-old_log_n]
#                    [-sporadic] [-h2rf h2r_factor] [-saveall]
#                    [-qtn] [-stop] [-nostop]
#
#           bayesavg -r[estart]   ;# (see also -redo)
#
#   SPECIAL NOTE:  THE ALGORITHMS CHANGED in VERSION 1.7.3.  SEE NOTES 1-4.
#                  NUMERIC RESULTS MAY DIFFER FROM PREVIOUS VERSIONS.
#
#               -covariates (or -cov)  Perform bayesian model averaging
#                  on the covariates only.  (The default is to perform
#                  bayesian model averaging on the linkage elements.)
#
#               -overwrite (or -ov) means force overwrite of existing output
#                  files
#
#               -max  Only include this number of components, or fewer,
#                     at one time.  This reduces the number of models
#                     enormously, particularly for large N.
#
#               -list file contains a list of the elements to use.
#                     There is one line for each covariate or linkage
#                     parameter.  Remaining covariates or linkage parameters
#                     in the starting model are automatically fixed.
#                     Covariates need not be present in the starting model,
#                     but linkage parameters (and their matrices, etc.) must
#                     be included in the starting model.
#
#               -fix (or -f) fix (lock in) this covariate.  A fixed element
#                  covariate (specified by covariate name, e.g. "age") or 
#                  linkage element (by linkage parameter name, e.g. "h2q1")
#                  is carried through all models.  (Note: a -fix or -f 
#                  qualifier is required for each covariate to be fixed,
#                  for example:  -f age -f sex.)  When fixed elements are
#                  included, it is adviseable to run "polygenic" on the
#                  starting model first.
#                   
#               -cutoff (optional) sets the BIC limit for occam's window 
#                  (default: 6) 
#
#               -log_n specify the log(n) to use.  Normally this is first
#                  estimated from the samplesize of the unsaturated model,
#                  then recalculated from the standard deviation of the
#                  mean and it's standard error in the model with the best BIC.
#
#               -symmetric (or -sym) use "symmetric" Occam's window.
#                   The default is a "strict" Occam's window, which excludes
#                   superset models with higher BIC; symmetric Occam's window
#                   includes ALL models within BIC cutoff.
#
#               -stop   Stop when no models in the last group with the same
#                       size (degrees of freedom) have entered the window.
#                       (This is the default for -qtn.)
#
#               -nostop  Do not stop when no models in the last group with
#                        the same size have entered the window.  (Useful
#                        for overriding the default for -qtn.)  If -stop
#                        or -qtn is specified, however, the report if any
#                        models have entered the window is still given.
#
#               -restart (or -r) means restart previous bayesavg run that was
#                  terminated before completion.  This begins with the
#                  model after the last one in the output file.  Do not use
#                  -restart if last run completed.  When restarting, set
#                  the trait or outdir, then give the command "bayesavg
#                  -restart" with no other arguments.  The original model
#                  and other arguments are automatically recalled.
#                  Previous command arguments are read from
#                  bayesavg.command and the previous starting model is
#                  c.orig or cov.orig.  If you need to change anything, use
#                  the -redo option instead.  You will also need to use
#                  the -redo option if the first pass through all models
#                  completed, or if the bayesavg was started under
#                  a previous version of SOLAR.
#
#                -redo is a special form of restart that allows you to change
#                  some options.  Unlike -restart, -redo REQUIRES YOU TO
#                  SPECIFY ALL OPTIONS AND LOAD ORIGINAL STARTING MODEL.
#                  Only models not already found in the output file will be
#                  maximized.  
# 
#                  There are several cases where you must use -redo instead
#                  -restart:  (1) If you need to
#                  re-maximize models which had convergence problems
#                  previously (edit them out of bayesavg*.est file, change
#                  boundaries, then -redo).  (2) If previous bayesavg run
#                  completed but you want to try a different window cutoff or
#                  type.  (3) You deleted all files except the bayesavg.est
#                  file.  (4) You need to restart from a previous version of
#                  SOLAR.  Unlike -restart, with -redo you must set up the
#                  starting model and commands either as they were previously
#                  or with desired changes.  Since you must set up the
#                  original model EXACTLY, and specify options either EXACTLY
#                  as they were originall specified, or with the desired
#                  changes, you are advised to use this option carefully.
#                  It is a good idea to make a backup copy of the outdir
#                  first.
#
#               -saveall will force the saving of all models.  Normally only
#                  the models within Occam's window are saved.  (Note:
#                  models outside the window will not have standard errors.)
#
#               -size_log_n Use the log(n) estimated from sample size as the
#                           final log(n).  This bypasses the computation of
#                           log(n) from the S.E. of the SD parameter of the
#                           model with the best BIC.
#
#               -nose       Do not compute standard errors for any models
#                           (normally they are only computed for models
#                           in the window).  Unless you specify a particular
#                           -log_n, the log(n) estimated from sample size
#                           will be used (as with -size_log_n).
#
#               -old_log_n  This calculates log(n) the old fashioned way,
#                           using the saturated model for covariate analysis
#                           or the unsaturated model for linkage analysis.
#                           This option is provided for comparison with
#                           earlier releases, and may soon be removed.
#
#               -h2rf (optional) is used to set upper bound of H2r
#                  (default: 1.1)  See notes below.  Use of this option
#                  is now unnecessary because of automated boundary control.
#
#               -sporadic  This option is depricated.  Force all models
#                  to sporadic.  Valid only with -covariate.  Now you can
#                  accomplish the same thing by making the starting model
#                  sporadic.
#
#               -qtn   Quantitative Trait Nucleotide Analysis:
#                      A "covariate" analysis is done with "-stop" in effect.
#                      Covariates with name snp_* or hap_* are automatically
#                      included but other covariates are excluded.  A special
#                      "windowfile" named bayesavg_cov.win is also
#                      produced.  The -stop default can be overridden with
#                      -nostop.  To include all snps in the starting model,
#                      use the separate command "allsnp".
#
# Output:   In addition to the terminal display, the following files are
#           created (<outname> is "bayesavg" for linkage analysis or 
#           "bayesavg_cov" for covariate analysis):
#
#           <outname>.avg         Final averaged results
#           <outname>.out         Final BIC and other info for each model
#                                   (standard errors for models in window)
#           <outname>.history     History of analysis messages
#           <outname>.est         Estimated BIC for each model (pass 1)
#           <outname>.nose        Final BIC but no standard errors (pass 2)
#
#           Models are saved with "c" <prefix> for linkage analysis and "cov"
#           prefix for covariate analysis:
#
#           <prefix>0.mod         Unsaturated model, with standard errors
#           <prefix>1.mod         Model with element 1 (if saved)
#           <prefix>12.mod        Model with elements 1 and 2 (if saved)
#           <prefix>12_11.mod     Model with elements 1, 2, and 11.
#           <prefix>.orig.mod     Original user model when started
#           <prefix>.start.mod    Base model (unsaturated) before maximization
#           <prefix>.base.mod     Maximized base model
#
# Notes:    1)  bayesavg determines the number of variable (non-fixed)
#               elements and sets N automatically.  N and the number of
#               models are reported near the beginning.  A new algorithm
#               is used to determine all the element combinations; this
#               results in a more logical ordering in which the smallest
#               models are evaluated first.
#
#           2)  The first pass through all models is done with an approximate
#               log(n) computed from the sample size.  The resulting file
#               is bayesavg.est (or bayesavg_cov.est).  The final log(n) is
#               then computed from the model with the best BIC, and all
#               BIC's are recalculated with the resulting file being
#               bayesavg.nose (or bayesavg_cov.nose).  Then, standard
#               errors for only the models within Occam's window are
#               recalculated.  The resulting final output file is
#               bayesavg.out (or bayesavg_cov.out).  The output summary
#               averages are reported in bayesavg.avg (or
#               bayesavg_cov.avg).  This is a new algorithm designed to
#               save time (by only calculating standard errors from the
#               models in the window), be more robust, and give more
#               accurate results.  Results may differ somewhat from those
#               in earlier versions (prior to 1.7.3) of SOLAR.  Additional
#               history of the analysis (the messages starting with "***")
#               are saved in bayesavg.history (or bayesavg_cov.history).
#
#
#           3)  To permit special models (with household effects, epistasis,
#               etc.) to be handled, bayesavg no longer forces the starting
#               model to be sporadic first.  It merely maximizes the current
#               model, with all non-fixed elements removed, but with no
#               change(s) to the starting omega or constraints.
#               If the starting model cannot be maximized, the user is
#               advised to run "polygenic" first.  Running "polygenic"
#               first is probably a good idea in all -covariate cases,
#               particularly if there are non-fixed elements.
#
#           4)  Models are now "built-up" from the unsaturated model
#               rather than being "constrained down" from the saturated
#               model.  The unsaturated model itself is usually created
#               by "constraining down" the starting model.
#
#           5)  bayesavg may not support bivariate models.
# -

# John decided not to allow this option, so it is undocumented.
#
#               -sporadic_first  (For covariate analysis only.)  For each
#                  model, maximize sporadic first, then polygenic.  This
#                  will probably take longer on average (and is not compatible
#                  with special models having household effects or other
#                  special features), but may help handle some otherwise
#                  intractible models.
#
#
#   -excludenull now obsolescent...uninvokable in 1.7.3

proc bayesavg {args} {

# Ensure sort exists

    usort

# Check for parallel mode

    if {[if_global_exists PAR_jobs]} {
	return [eval parbayesavg $args]
    }
    if {[if_global_exists GRID_jobs]} {
	return [eval gridbayesavg $args]
    }

# Setup verbosity

    set verbose 0
    set qu -q

    ifverbplus set verbose 1
    ifverbplus set qu ""

    ifdebug set verbose 1

# Allow for old-Sun tail command

    if {"SunOS" == [exec uname]} {
	set head2 "-2"
	set tailn ""
    } else {
	set head2 "-n 2"
	set tailn "-n"
    }

# Initialize variables and default options

    set fix_list {}   ;# As specified; not included items automatically fixed
    set cutoff 6
    set h2r_factor 1.1
    set restart 0
    set overwrite 0
    set covar 0
    set covarlist {}
    set betalist {}
    set excludenull 0
    set saveall 0
    set savewindow 0
    set redo 0
    set rere 0
    set minBICmodel ""
    set last_minBICmodel ""
    set symmetric 0
    set use_sporadic 0
    set use_cov0 0
    set max_comb 0
    set list_file ""
    set minBIC 1e36
    set bad_results {}
    set max_e2 0.9999          ;# higher than this, and we use spormod
    set missing 0
    set no_se 0
    set winse 0
    set paramlist {}
    set sporadic_first 0
    set use_log_n {}
    set old_log_n 0
    set size_log_n 0
    set qtn 0
    global Solar_Bay_Qtn
    set Solar_Bay_Qtn 0
    set stop 0
    set nostop 0
    set maxcomb 2000000
    set maxcomb_increment 1000000

# Read arguments

    set extra_args [read_arglist $args -cutoff cutoff -h2rf h2r_factor \
	    -cov {set covar 1} -covariates {set covar 1} \
	    -qtn {set covar 1; set qtn 1; set stop 1; set Solar_Bay_Qtn 1} \
	    -stop {set stop 1} \
	    -sporadic {set covar 1; set use_sporadic 1} \
	    -spor {set covar 1; set use_sporadic 1} \
	    -restart {set restart 1} -r {set restart 1} \
	    -rere {set restart 1; set rere 1} \
	    -redo {set redo 1} \
	    -saveall {set saveall 1} \
	    -savewindow {set savewindow 1} \
	    -symmetric  {set symmetric 1} \
	    -sym  {set symmetric 1} \
	    -fix {lappend fix_list VALUE} \
	    -f {lappend fix_list VALUE} \
            -max max_comb \
            -list list_file \
	    -nose {set no_se 1} \
	    -winse {set winse 1} \
	    -size_log_n {set size_log_n 1} \
	    -log_n use_log_n \
	    -old_log_n {set old_log_n 1} \
	    -nostop {set nostop 1} \
	    -ov {set overwrite 1} -overwrite {set overwrite 1}]

    set fix_list [string tolower $fix_list]

# Check for invalid arguments

    if {[llength $extra_args]} {
	error "Invalid arguments $extra_args"
    }
    if {$restart && $redo} {
	error "Arguments -restart and -redo are incompatible"
    }
    if {$sporadic_first && $use_sporadic} {
	error "Arguments -sporadic_first and -sporadic (only) are incompatible"
    }

# Remember previous arguments if restarting

    if {$restart && !$rere} {
	if {[llength $args] > 1} {
	    error \
	       "Use 'bayesavg -restart' to restart; other arguments remembered"
	}
	set arguments ""
	set got_arguments 0
	if {[file exists [full_filename bayesavg.command]]} {
	    set cfile [open [full_filename bayesavg.command]]
	    if {-1 < [gets $cfile arguments]} {
		set got_arguments 1
	    }
	    close $cfile
	}
	if {!$got_arguments} {
	    error \
       "Unable to read previous arguments from [full_filename bayesavg.command]"
	}
	puts "    *** Restarting bayesavg with arguments: $arguments"
	return [eval bayesavg -rere $arguments]
    }
    if {$rere} {
	set overwrite 0
    }

# Setup names for covariate and non-covariate cases

    if {$covar} {
	set prefix cov
	set outname bayesavg_cov
	set basemodel cov.base
    } else {
	set prefix c
	set outname bayesavg
	set basemodel c.base
    }

# Now, if actually restarting or redoing, check for required files

    if {$restart || $redo} {
	if {![file exists [full_filename $outname.est]]} {
	    error "Unable to restart: $outname.est not saved"
	}
    }
    if {$restart} {
	if {![file exists [full_filename $prefix.orig.mod]]} {
	    error "Unable to restart: $prefix.orig.mod not saved"
	}
    }



# Check restart/overwrite status; Purge old files as appropriate

    if {$restart && $overwrite} {
	error "-restart and -overwrite arguments are incompatible"
    } elseif {!$overwrite && !$restart && !$redo} {
	if {0<[llength [bayesavg_purge -outname $outname -prefix $prefix \
		-testonly]]} {
	   error "Bayesavg output files exist.  Use -overwrite or -restart option."
	}
    } elseif {$overwrite} {
	bayesavg_purge -outname $outname -prefix $prefix
    }

# Save user's original model as ".orig" if not restarting
# If -restart, load previous ".orig" 

    if {$restart} {
	load model [full_filename $prefix.orig]
    } else {
	save model [full_filename $prefix.orig]
	exec echo $args >[full_filename bayesavg.command]
    }

# Turn off standard error and save model as ".start"

    option standerr 0
    save model [full_filename $prefix.start]

# Announce restart or redo

    if {$restart} {
	putsout $outname.history \
"\n    ********* Restarting bayesavg *********"
    }
    if {$redo} {
	putsout $outname.history \
"\n    ********* Redo bayesavg started *********"
    }


# ****************************************************************************
#     Setup for covariate case (covarlist, betalist, and N)
#
#        Note: fix_list is just what is specified by user.
#        It doesn't include automatically fixed items:
#        items not included in "-list" if list used, and constrained items.
# ****************************************************************************

    if {$covar} {

# If no -list,
# Setup covarlist and betalist with all but suspended covariates
# and covariates constrained to zero.
#
# Covariates constrained to non-zero values are automatically "fixed."

	if {{} == $list_file} {
	    set tcovarlist [covariates -applicable]
	    set tbetalist  [covariates -betanames]
	    for {set i 0} {$i < [llength $tcovarlist]} {incr i} {
		set tcovar [lindex $tcovarlist $i]
		set tbeta [lindex $tbetalist $i]
		set target [string tolower [string range $tcovar 0 3]]
		if {!$qtn || 0==[string compare "snp_" $target] || \
		    0==[string compare "hap_" $target]} {
		    if {-1 == [string first "Suspended\[" $tcovar]} {
			if {![is_constrained_to_zero $tbeta] && \
				![is_constrained_to_nonzero $tbeta]} {
			    lappend covarlist $tcovar
			    lappend betalist $tbeta
			}
		    }
		}
	    }

# If -list, we get covariate names from list
# Covariates are temporarily loaded into empty model to get their betanames
#   with the correct ordering.
# Then, listed covariates are actually added to model
# 
	} else {
	    set lfile [open $list_file r]
	    covariate delete_all
	    while {-1 != [gets $lfile newcov]} {
		covariate $newcov
	    }
	    close $lfile
	    set covarlist [covariates -applicable]
	    set betalist  [covariates -betanames]
	    load model [full_filename $prefix.start]
	    foreach ensure_covar $covarlist {
		covariate $ensure_covar
	    }
	    save model [full_filename $prefix.start]
	}

#  If there are fixed covariates, remove them from lists
#  Also, check to see that fixed covariates are actually in the model
#    If not, add them (ain't I nice)

	if {"" != $fix_list} {
	    set augmented_model 0
	    set cov_in_model_l [string tolower [covariates -applicable]]
	    foreach lfixed $fix_list {
#
# Remove fixed cov's from covarlist and betalist
# It's not important if they were there already or not
#
		for {set i 0} {$i < [llength $covarlist]} {incr i} {
		    set ucovar [lindex $covarlist $i]
		    set lcovar [string tolower $ucovar]
		    if {0==[string compare $lcovar $lfixed]} {
			set covarlist [lreplace $covarlist $i $i]
			set betalist [lreplace $betalist $i $i]
			break
		    }
		}
#
# Now, make sure fixed covariates are actually IN MODEL
#
		set found 0
		foreach cov_l $cov_in_model_l {
		    if {![string compare $lfixed $cov_l]} {
			set found 1
			break
		    }
		}
		if {!$found} {
		    covariate $fixed
		    putsout $outname.history \
                     "    *** Adding -fixed covariate $fixed to starting model"
		    set augmented_model 1
		}
	    }
	    if {$augmented_model} {
		save model [full_filename $prefix.start]
	    }
	}

# Report variable and fixed covariates

	putsout $outname.history "    *** Testing covariates: $covarlist"
	ifdebug puts "betalist is $betalist"
	set n [llength $covarlist]
    }

# ****************************************************************************
#     Setup for non-covariate case: paramlist and N
# ****************************************************************************

    if {!$covar} {
#
# If using -list...
#
	set paramlist {}
	if {{} != $list_file} {
	    set n 0
	    set lfile [open $list_file r]
	    while {-1 != [gets $lfile newparam]} {
		set newparam [string tolower [string trim $newparam]]
		if {-1 == [lsearch -exact $fix_list $newparam]} {
		    lappend paramlist $newparam
		    incr n
		}
	    }
	    close $lfile
#
# Not using list file, use all h2qi parameters, i=1,2,3...
#
	} else {
	    set ne [h2qcount]
	    set n 0
	    set paramlist {}
	    for {set i 1} {$i <= $ne} {incr i} {
		if {-1 == [lsearch -exact $fix_list h2q$i]} {
		    lappend paramlist H2q$i
		    incr n
		}
	    }
	}
#
# Make sure all fixed elements are present
#
	foreach lfixed $fix_list {
	    if {![if_parameter_exists $lfixed]} {
		error "Fixed parameter $lfixed not present in starting model"
	    }
	}
    }

# ****************************************************************************
#     Check and report N
# ****************************************************************************

    if {$n < 2} {
	if {$covar} {
	    error \
	      "Covariate averaging not possible with one or fewer covariates"
	} else {
	    puts "Use 'bayesavg -cov' if analyzing covariates"
	    error \
                "Linkage averaging not possible with one or fewer linkage elements"
	}
    }
    putsout $outname.history  "\n    *** N is $n"

# ****************************************************************************
#     Generate list of combinations (subsets?):   allcomb
# ****************************************************************************

    set allcomb {}
    if {$max_comb} {
	combinations $n -max $max_comb -list allcomb -stop $maxcomb
    } else {
	combinations $n -list allcomb -stop $maxcomb
    }
    set ncomb [llength $allcomb]
    if {$ncomb == $maxcomb} {
	putsout $outname.history "    *** Number of possible models is greater than $maxcomb"
	putsout $outname.history "    ***   (Additional models will be computed when and if needed)"
    } else {
	putsout $outname.history "    *** Number of models is [expr $ncomb + 1]\n"
    }

# ****************************************************************************
# Maximize base model (from which other models are built) and set samplesize
#   Base models include all fixed elements, but no variable ones
# ****************************************************************************

# Consider covariate case

    if {$covar} {
#
# See if basemodel already exists (if not restarting, it was deleted)
# Make sure it is sporadic if -sporadic option used
#
	set use_previous_basemodel 0
	if {$restart && [file exists [full_filename $basemodel.mod]] && \
		[file exists [full_filename $basemodel.out]]} {
	    if {$use_sporadic && ![oldsporadic $basemodel]} {
		error "-sporadic option not used in previous run"
	    }
	    set use_previous_basemodel 1
	}
#
# Suspend non-fixed elements from starting model, resave as cov.start
#
	if {!$use_previous_basemodel} {
	    set c_suspended 0
	    foreach current_covar $covarlist {
		covariate suspend $current_covar
		ifdebug puts \
			"Suspended covariate $current_covar for base model"
		set c_suspended 1
	    }
	    if {$c_suspended} {
		save model [full_filename cov.start]
	    }
#
# Force model to sporadic if -sporadic or -sporadic_first 
#   and not sporadic already
#
	    if {($sporadic_first || $use_sporadic) \
		    && ![oldsporadic cov.start]} {
		spormod
		save model [full_filename cov.start]
#
# If only sporadic_first, we maximize sporadic, then polygenic
#
		if {$sporadic_first} {
		    putsout $outname.history \
			    "    *** Maximizing unsaturated sporadic model"
		eval maximize $qu -o cov.spor.out
		putsout $outname.history \
		       "    *** Loglikelihood of sporadic model is [loglike]\n"
		    polymod
		}
	    }
#
# Maximize base model
#
	    putsout $outname.history \
		    "    *** Maximizing base model cov0 (unsaturated)"
	    
# If omega has not been defined, default to polygenic

	    if {-1 != [set foo [string first \
		   Use_polygenic_to_set_standard_model_parameterization \
				    [omega]]]} {
		puts "\n    *** Undefined omega in starting model, defaulting to polygenic\n"
		polymod
	    }

	    eval maximize $qu -o $basemodel.out
	    if {($max_e2 >= [parameter e2 =]) || $use_sporadic || \
		    [oldsporadic cov.start]} {
		save model [full_filename $basemodel]
		putsout $outname.history \
			"    *** Loglikelihood of cov0 is [loglike]\n"
	    } else {
		putsout $outname.history \
		"    *** H2r below threshold, maximize base model as sporadic"
		load model [full_filename cov.start]
		spormod
		eval maximize $qu -o $basemodel.out
		save model [full_filename $basemodel]
		putsout $outname.history \
			"    *** Loglikelihood of sporadic cov0 is [loglike]\n"
	    }
	}
    }
#
# Consider linkage case, create base model
#
    if {!$covar} {
	ifdebug puts "creating base model for linkage case"
#
# Constrain variable linkage elements to 0, resetting h2r
#
	foreach param $paramlist {
	    parameter h2r = [expr [parameter h2r =] + [parameter $param =]]
	    parameter $param = 0 lower -0.01
	    if {-1==[string first "-" $param]} {
		constraint $param = 0
	    } else {
		constraint <$param> = 0
	    }
	}
	parameter h2r upper [lowest 1.0 [highest [parameter h2r upper] \
		[expr $h2r_factor * [parameter h2r =]]]]
#
# See if basemodel already exists (if not restarting, it was deleted)
#
	set use_previous_basemodel 0
	if {$restart && [file exists [full_filename $basemodel.mod]] && \
		[file exists [full_filename $basemodel.out]]} {
	    putsout $outname.history "    *** Using previous $basemodel.mod"
	    set use_previous_basemodel 1
	}
#
# Maximize basemodel if not already found
#
	if {!$use_previous_basemodel} {
	    putsout $outname.history \
		    "    *** Maximizing base model c0 (unsaturated)"
	    set max_status [maximize_quietly [full_filename \
		    $basemodel.out]]
	    if {"" != $max_status} {
		error "Convergence error maximizing c0:  $max_status"
	    }
	    putsout $outname.history "    *** Loglikelihood of c0 is [loglike]\n"
	save model [full_filename $basemodel]
	}
    }

# ****************************************************************************
# If -old_log_n, set old_log_n now from fully saturated model (covariate)
#   or unsaturated model (linkage)
# ****************************************************************************

    if {$old_log_n} {
	load model [full_filename $basemodel]
	option standerr 1
	if {$covar} {
	    for {set i 0} {$i < $n} {incr i} {
		set covi [lindex $covarlist $i]
		covariate restore $covi
	    }
	    if {$sporadic_first} {
		spormod
		puts "    *** Maximizing saturated sporadic model"
		maximize -q -o $basemodel.spor.se.out
		polymod
	    }
	    puts "    *** Maximizing saturated model to calculate log(n)"
	    maximize -q -o $basemodel.se.out
#
# If linkage, we use unsaturated model just done, but re-maximize for S.E.
#
	} else {
	    puts \
           "    *** Maximizing unsaturated model with S.E. to calculate log(n)"
	    maximize -q -o $basemodel.se.out
	}
	set sd [parameter sd =]
	set sdse [parameter sd se]
	if {$sdse == 0} {
	    error "Attempt to use -old_log_n failed; can't get SE of SD"
	}
	set use_log_n [expr (log ($sd*$sd / (2*$sdse*$sdse)))]
    }

# ****************************************************************************
# Load basemodel, set samplesize and log_n_est
# ****************************************************************************

    load model [full_filename $basemodel]
    set samplesize [outfile_sample_size $basemodel.out]
    putsout $outname.history "    *** Samplesize is $samplesize"
    set log_n_est [expr log($samplesize)]
    if {{} != $use_log_n} {
	set log_n_est $use_log_n
	putsout $outname.history \
		"    *** log(n) specified as [format %.7f $log_n_est]\n"
    } elseif {$no_se || $size_log_n} {
	set use_log_n $log_n_est
	putsout $outname.history \
		"    *** log(n) computed from sample size is [format %.7f $log_n_est]\n"
    } elseif {[is_constrained_to_nonzero sd] || [is_constrained_to_zero sd]} {
	set use_log_n $log_n_est
	putsout $outname.history \
		"    *** SD is constrained so will use estimated log(n): [format %.7f $log_n_est]\n"
    } else {
	putsout $outname.history \
		"    *** Estimated log(n) is [format %.7f $log_n_est]\n"
    }
    set null_loglike [oldmodel $basemodel loglike]
#
# Save basemodel as c[ov]0 with suspended covariates now constrained instead
#
    suspend2constrain
    save model [full_filename [catenate $prefix 0]]

# ****************************************************************************
#   Setup output file using "resultfile" object
# ****************************************************************************

# Start with model, BIC, loglike

    set headings "Model BIC Loglike"
    set formats "%16s %11.4f %12.3f"
    set expressions {{$cname} {$BIC} {$c_loglike}}
#
# Add H2r and H2r SE
#
    lappend headings H2r
    lappend headings "H2r SE"
    set formats "$formats %-12.7g %-12.7g"
    lappend expressions "\[oldmodel \$cname h2r\]"
    lappend expressions "\[oldmodel \$cname h2r -se\]"
#
# If covariate analysis and h2q's are present, add them to the output
#
    if {$covar && [if_parameter_exists h2q1] && ![is_constrained_to_zero h2q1]} {
	for {set i 1} {[if_parameter_exists h2q$i]} {incr i} {
	    lappend headings H2q$i
	    lappend headings "H2q$i SE"
	    set formats "$formats %-12.7g %-12.7g"
	    lappend expressions "\[oldmodel \$cname h2q$i\]"
	    lappend expressions "\[oldmodel \$cname h2q$i -se\]"
	}
    }
#
# For covariate analysis, add all covariates
#   For linkage analysis, add all H2qi components
#
    for {set i 0} {$i < $n} {incr i} {
	if {$covar} {
	    set cov [lindex $covarlist $i]
	    set beta [lindex $betalist $i]
	    lappend headings $beta
	    lappend headings "$beta SE"
	    set formats "$formats %-13.8g %-13.8g"
	    lappend expressions "\[oldmodel \$cname $beta\]"
	    lappend expressions "\[oldmodel \$cname $beta -se\]"
	} else {
	    set vparam [lindex $paramlist $i]
	    lappend headings $vparam
	    lappend headings "$vparam SE"
	    set formats "$formats %-12.7g %-12.7g"
	    lappend expressions "\[oldmodel \$cname $vparam\]"
	    lappend expressions "\[oldmodel \$cname $vparam -se\]"
	}
    }
    set open_option -create
#
# For both restart and redo, must scan old output file first
#
    if {$restart} {
	set this_df 0
	set best_bic_in_df 0.0
	set models_in_this_df 1
	set early_exit 0
	set best_bic 0.0
	set best_model cov0
	set best_model_in_df cov0
	set combno 0
    }

    if {$restart || $redo} {
	delete_files_forcibly $outname.avg
	set endcomb [lindex $allcomb end]
	set endnum [lindex $endcomb 0]
	set endname [catenate $prefix $endnum]
	for {set cni [expr $endnum+1]} {$cni <= $n} {incr cni} {
	    set endname [catenate $endname _$cni]
	}
#	puts "Looking for $endname..."
	set icount [expr 5 + 2 * [llength $endcomb]]
	puts "    *** Each record must have $icount entries"
	if {!$redo && \
		![catch {read_bayes $endname $outname.est bic}]} {
	    error "Last model already done...you must use -redo"
	}
	set open_option -append
	set ofile [open [full_filename $outname.est] r]
	gets $ofile   ;# Get headings
	gets $ofile   ;# Get -------- under headings
	set last_model ""
	set number_models_read 0
	set first 1
	while {-1 != [gets $ofile lastline]} {
	    if {[llength $lastline] >= $icount} {
		incr number_models_read
		set last_model [lindex $lastline 0]
		set BIC [lindex $lastline 1]
		if {$BIC < $minBIC} {
		    set minBIC $BIC
		    set minBICmodel $last_model
		}
		if {$restart && !$first} {
		    incr combno
# Note: combno will be index for *next* combination
		}
		set first 0
		if {$stop && $restart} {
		    set df [llength [bayesavg_elements $last_model]]
#		    puts "Model $last_model has df $df and BIC $BIC"
		    if {$df != $this_df} {
			putsout $outname.history \
		"    *** Best BIC in degree $this_df is $best_bic_in_df for model $best_model_in_df"
			if {$models_in_this_df==0} {
			    putsout $outname.history \
                        "    *** No models with degree $this_df were in window"
			    if {!$nostop} {
				puts $outname.history \
				"    *** Exiting main loop by stop rule"
				set early_exit 1
				break
			    }
			}
			set this_df $df
			set models_in_this_df 0
			set best_bic_in_df $BIC
			set best_model_in_df $last_model
		    }
		    if {$BIC - $cutoff < $best_bic} {
			set models_in_this_df 1
		    }
		    if {$BIC < $best_bic} {
			set best_bic $BIC
			set best_model $last_model
		    }
		    if {$BIC < $best_bic_in_df} {
			set best_bic_in_df $BIC
			set best_model_in_df $last_model
		    }
		}
	    } else {
		close $ofile
		error "Incomplete ending record in output file: $outname.est"
	    }
	}
	if {$restart} {
	    if {$number_models_read >= 1+$ncomb} {
	error "Enough or more than enough models already...you must use -redo"
	    }
	    set should_be [lindex $allcomb [expr $combno - 1]]
	    set found_comb [bayesavg_elements $last_model]
	    if {[string compare $found_comb $should_be]} {
#		puts "found: $found_comb  should be: $should_be"
		error "Models missing or duplicated:  use -redo"
	    }
	}
	
	close $ofile
	if {"" == $last_model} {
	    error "Last bayesavg left empty output file: $outname.est"
	}
    }

    set resultf [resultfile $open_option [full_filename $outname.est] \
	    -headings $headings \
	    -formats $formats -expressions $expressions -display]
    if {$restart || $redo} {
	resultfile $resultf -header -displayonly
    } else {
	resultfile $resultf -header

# Output results for basemodel (currently loaded)

	set cname [catenate $prefix 0]
	set c_loglike [loglike]
	set BIC 0  ;# Yes, this one is easy !
	resultfile $resultf -write
    }

# if restarting and only unsaturated model in file, reset restart status now

    set end_restart 0
    set still_restarting 0
    if {$restart && 0 != [string compare $last_model [catenate $prefix 0]]} {
	set still_restarting 1
    }

# ********************************************************************
#       Evaluate each combination
# ********************************************************************

#
# Implement stopping rule.  combno initialized here unless restart.
#
    if {!$restart} {
	set this_df 0
	set models_in_this_df 1
	set early_exit 0
	set best_bic 0.0
	set best_bic_in_df 0.0
	set best_model $basemodel
	set best_model_in_df $basemodel
	set combno 0
    }

    set number_models_written 0
#
# Big loop begins here
#
    for {} {!$early_exit && ($combno < $ncomb)} {incr combno} {
	set output_needed 1
	set maximize_sporadic_first 0
	ifdebug puts "Doing combno $combno"
#
# If this is the last combno because of artificial limit, increase here
#
	if {$combno+1 == $maxcomb} {
	    set maxcomb [expr $maxcomb + $maxcomb_increment]
	    putsout $outname.history "    *** Recalculating combinations to $maxcomb"
	    set allcomb {}
	    if {$max_comb} {
		combinations $n -max $max_comb -list allcomb -stop $maxcomb
	    } else {
		combinations $n -list allcomb -stop $maxcomb
	    }
	    set ncomb [llength $allcomb]
	}
	    

#
# Get current combination, cname, and df
#
	set comb [lindex $allcomb $combno]
	set cname [catenate $prefix [lindex $comb 0]]
	for {set cni 1} {$cni < [llength $comb]} {incr cni} {
	    set cname [catenate $cname _[lindex $comb $cni]]
	}
	set df [llength $comb]
#
# Check for early exit
#
	if {$df != $this_df} {
	    putsout $outname.history \
		"    *** Best BIC in degree $this_df is $best_bic_in_df for model $best_model_in_df"
	    if {$stop && $models_in_this_df == 0} {
		putsout $outname.history \
		    "        *** No models with degree $this_df were in window"
		if {!$nostop} {
		    putsout $outname.history \
			    "        *** Exiting main loop by stop rule"
		    set early_exit 1
		    continue
		}
	    }
	}
#
# If redo, procede only if model not found in output file
#
	set use_existing_model 0
	if {$redo} {
	    if {![catch {set BIC [read_bayes $cname $outname.est bic]}]} {
		if {$df != $this_df} {
		    set this_df $df
		    set models_in_this_df 0
		    set best_bic_in_df $BIC
		    set best_model_in_df $cname
		}
		if {$BIC < $best_bic_in_df} {
		    set best_bic_in_df $BIC
		    set best_model_in_df $cname
		}
		if {$BIC - $cutoff < $best_bic} {
		    set models_in_this_df 1
		}
		if {$BIC < $best_bic} {
		    set best_bic $BIC
		}
		continue
	    }
	}
#
# Create model from base by restoring elements
#
	load model [full_filename $basemodel]
	if {$covar} {
	    for {set i 0} {$i < $n} {incr i} {
		set covi [lindex $covarlist $i]
		if {-1 != [lsearch $comb [expr $i + 1]]} {
		    covariate restore $covi
		}
	    }
	} else {
	    foreach ce $comb {
		set param [lindex $paramlist [expr $ce - 1]]
		if {-1==[string first "-" $param]} {
		    constraint delete $param
		} else {
		    constraint delete <$param>
		}
		carve_new_value $param 0.01 h2r
		parameter $param lower 0
	    }
	}
#
# Maximize and save results
#
#  covariate
#
	if {$covar} {
	    if {$sporadic_first} {
		spormod
		set error_msg [maximize_quietly last.out]
		if {"" == $error_msg} {
		    polymod
		    set error_msg [maximize_quietly last.out]
		}
	    } else {
		set error_msg [maximize_quietly last.out]
	    }
#
# linkage
#
	} else {  ;# linkage model
	    set error_msg [max_bayesavg last.out $comb]
	}
	if {0 < [llength $error_msg]} {
	    lappend bad_results $error_msg
	    if {[llength $bad_results] > 5} {
		bayesavg_bad_results $outname.est $bad_results 0
	    }
	    continue
	}
	suspend2constrain    ;# convert suspended cov's to constrained
	model save [full_filename $cname]
#
# For all models
#   Calculate BIC
#
	set save_this_one 0
	set c_loglike [oldmodel $cname loglike]
	set lodp [lod $c_loglike $null_loglike]
	set lambda [expr $lodp * 2 * log (10)]
	set BIC [expr ($df * $log_n_est) - $lambda]
	if {$BIC < $minBIC} {
	    set save_this_one 1
	    set minBIC $BIC
	    set last_minBICmodel $minBICmodel
	    set minBICmodel $cname
	}
	if {$output_needed} {
	    resultfile $resultf -write
	    incr number_models_written
	}
#
#  Delete this model if no longer needed or desired
#
        if {!$save_this_one && !$saveall} {
	    delete_files_forcibly [full_filename $cname.mod]
	}
#
# Update stuff for early stop rule
#
	if {$df != $this_df} {
	    set this_df $df
	    set models_in_this_df 0
	    set best_bic_in_df $BIC
	    set best_model_in_df $cname
	}
	if {$BIC < $best_bic_in_df} {
	    set best_bic_in_df $BIC
	    set best_model_in_df $cname
	}
	if {$BIC - $cutoff < $best_bic} {
	    set models_in_this_df 1
	}
	if {$BIC < $best_bic} {
	    set best_bic $BIC
	}

#
# Delete previous best if this is a new best
#
	if {!$saveall && $save_this_one == 1} {
	    if {"" != $last_minBICmodel && \
		    [catenate $prefix 0] != $last_minBICmodel} {
		delete_files_forcibly [full_filename $last_minBICmodel.mod]
	    }
	}
    }
    load model [full_filename $minBICmodel]
#
# Sort output file; lowest BIC first
#
    putsout $outname.history "\n    *** Sorting output file"
    if {0 != [string length [usort]]} {
	set fullname [full_filename $outname.est]
	if {0==[catch {eval exec head $head2 $fullname >$fullname.tmp}]} {
	    if {0==[catch {eval exec tail $tailn +3 $fullname | [usort] -n -k 2 \
		    >>$fullname.tmp}]} {
		if {0==[catch {eval file rename -force $fullname.tmp $fullname}]} {
		    if {[llength $bad_results] > 0} {
			bayesavg_bad_results $outname.est $bad_results 1
		    }
		    if {{} == $use_log_n} {
			return [bayesavg_rewrite $cutoff $n $outname \
				$covarlist $excludenull $symmetric $no_se \
				$null_loglike $log_n_est $basemodel \
				$paramlist $formats $headings $betalist \
				0]
		    } else {
#
# We had "exact" log_n provided...don't need to recalculate BIC's
#
			file copy -force [full_filename $outname.est] \
				[full_filename $outname.nose]
			return [bayesavg_post $cutoff $n $outname \
				$covarlist $excludenull $symmetric $no_se \
				$basemodel $paramlist $formats $headings \
				$betalist $minBICmodel]
		    }
		}
	    }
	}
    }
#
# Sorting error
#
    if {[llength $bad_results] > 0} {
	puts stderr "   *** Sort of $outname.est failed"
	bayesavg_bad_results $outname.est $bad_results 2
    }
    error "    *** Sort of $outname.est failed"
}


proc hack_mibd_index {modfilename old new} {
    exec sed s/mibd$old/mibd$new/g $modfilename > $modfilename.1
    exec sed s/h2q$old/h2q$new/g $modfilename.1 > $modfilename.2
    file rename -force $modfilename.2 $modfilename
    file delete $modfilename.1
    return ""
}

#
# bayesavg postprocessing part one...rewrite with recalculated BIC's
# using log_n from model with best BIC
#
# First, we need to rewrite bayesavg.est (or .tmp if depth > 0) to bayesavg.out
# using real log_n value from best BIC model

proc bayesavg_rewrite {cutoff maxindex outname covarlist excludenull \
	symmetric no_se null_loglike log_n_est basemodel paramlist formats \
        headings betalist depth} {
#
# Set input extension according to depth
#
    if {$depth == 0} {
	set ext est
    } else {
	set ext nose.tmp
    }
#
# Allow for old-Sun tail command
#
    if {"SunOS" == [exec uname]} {
	set head2 "-2"
	set tailn ""
    } else {
	set head2 "-n 2"
	set tailn "-n"
    }
#
# Build up resultfile for output
#
    set columns [llength $formats]
    set expressions {}
    for {set i 0} {$i < $columns} {incr i} {
	if {$i == 1} {
	    lappend expressions \$BIC
	} else {
	    lappend expressions "\[lindex \$inline $i\]"
	}
    }
    set outfile [resultfile -create [full_filename $outname.nose] \
	    -headings $headings -formats $formats -expressions $expressions]
    resultfile $outfile -header
#
# Start reading $ext file
#
    set null_loglike [format %12.3f $null_loglike]
    set infile [open [full_filename $outname.$ext] r]
    gets $infile inline
    gets $infile inline
    if {"----------" != [string range $inline 1 10]} {
	close $infile
	error "Error(s) in $outname.$ext"
    }
#
# The first post-header record is the best-bic model
#
    gets $infile inline
    set bestmodel [lindex $inline 0]
#
# Maximize best model with standard errors turned on
#
    putsout $outname.history "    *** Maximizing $bestmodel for standard errors"
    set saveiter [option maxiter]
    if {[file exists [full_filename $bestmodel.mod]]} {
	load model [full_filename $bestmodel]
	option maxiter 1
    } else {
#
# If model wasn't saved, we have to rebuild it
#
	puts "Rebuilding $bestmodel from $basemodel..."
	load model [full_filename $basemodel]
	set cfilename [file rootname [file tail $bestmodel]]
	set comb [bayesavg_elements $cfilename]
	if {{} != $covarlist} {
	    for {set i 0} {$i < $maxindex} {incr i} {
		if {-1 != [lsearch $comb [expr $i + 1]]} {
		    set covi [lindex $covarlist $i]
		    covariate restore $covi
		}
	    }
	} else {
	    foreach ce $comb {
		set param [lindex $paramlist [expr $ce - 1]]
		constraint delete $param
		carve_new_value $param 0.01 h2r
		parameter $param lower 0
	    }
	}
    }
    option standerr 1
    if {[catch {maximize -q -o $bestmodel}] || \
	[parameter sd se] == 0} {
	close $infile
        if {{} == $covarlist} {
	    load model [full_filename c.orig.mod]
	} else {
	    load model [full_filename cov.orig.mod]
	}
	error "Couldn't get standard error of SD; suggest -redo using -log_n [format %.7f $log_n_est]"
    } else {
	set sd [parameter sd =]
	set sdse [parameter sd se]
	set log_n [expr log ($sd*$sd / (2*$sdse*$sdse))]
	putsout $outname.history "    *** log(n) calculated from $bestmodel is [format %.7f $log_n]"
	suspend2constrain
	option maxiter $saveiter
	save model [full_filename $bestmodel]
#
# Copy input records to output records, but correcting BIC
#
	while {{} != $inline} {
	    set cfilename [lindex $inline 0]
	    set inbic [lindex $inline 1]
	    set inloglike [lindex $inline 2]
#
# Calculate correct BIC
#
	    if {![is_float $inloglike]} {
		set BIC [format %11s NaN]
	    } else {
		ifdebug puts "likes: $inloglike  $null_loglike"
		set lodp [lod $inloglike $null_loglike]
		set lambda [expr $lodp * 2 * log (10)]
		set df [llength [bayesavg_elements $cfilename]]
		ifdebug puts "$cfilename:  df: $df  lambda: $lambda"
		set BIC [expr ($df * $log_n) - $lambda]
		catch {set BIC [format %11.4f $BIC]}
	    }
	    ifdebug puts "Model: $cfilename   estBIC: $inbic   BIC: $BIC"
#
# splice BIC into result line
#
#	    set firstcut [expr [string first $inbic $inline] - 1]
#	    set resumec [expr $firstcut + 1 + [string length $inbic]]
#	    ifdebug puts "prefix: [string range $inline 0 $firstcut]"
#	    set outline [string range $inline 0 $firstcut]$BIC[string range $inline $resumec end]
	    set outline [replace_using_format $inline $formats 1 $BIC]
	    resultfile $outfile -write
	    gets $infile inline
	}
    }
    close $infile
#
# Resort new output file (unlikely, but might need it)
#
    putsout $outname.history \
	    "    *** Re-sorting output file with changed BIC's"
    if {0 != [string length [usort]]} {
	ifverbplus puts "\n    *** Re-sorting $outname.nose"
	set fullname [full_filename $outname.nose]
	if {0==[catch {eval exec head $head2 $fullname >$fullname.tmp}]} {
	    if {0==[catch {eval exec tail $tailn +3 $fullname | [usort] -n -k 2 \
		    >>$fullname.tmp}]} {
		set infile [open $fullname.tmp]
		gets $infile
		gets $infile
		set inline ""
		gets $infile inline
		close $infile
		if {0!=[string compare $bestmodel [lindex $inline 0]]} {
		    if {$depth < 10} {
			putsout $outname.history \
            "\n    *** Now a different model has best BIC: [lindex $inline 0]"
			return [bayesavg_rewrite $cutoff $maxindex \
				    $outname $covarlist $excludenull \
				    $symmetric $no_se $null_loglike \
				    $log_n_est $basemodel $paramlist \
				    $formats $headings $betalist \
				    [expr $depth + 1]]
		    } else {
			putsout $outname.history \
 "\n    *** Warning!  Changes in log(n) keep causing best BIC model to change"
			putsout $outname.history \
   "    *** Locking in best value after 10 retries\n"
		    }
		}
		if {0==[catch {eval file rename -force $fullname.tmp \
				   $fullname}]} {
		    return [bayesavg_post $cutoff $maxindex $outname \
			    $covarlist $excludenull $symmetric $no_se \
			    $basemodel $paramlist $formats $headings \
			    $betalist $bestmodel]
		}
	    }
	}
    }
#
# Sorting error
#
    error "    *** Sort of $outname.est failed"
}


proc bayesavg_post {cutoff maxindex outname covarlist excludenull symmetric \
    no_se basemodel paramlist formats headings betalist bestmodel} {
#
    ifdebug puts "Starting bayesavg_post"
#
#
# IMPORTANT to allow link/cov/qtn types and
# h2r in all types and
# h2q1 in cov/qtn types (when present):
#
# firstindex==0 is normal case
#   in that case, i==0 is h2r, represented by h2q0
# firstindex==-1 means h2q1 included in cov model
#   in that case, i==-1 is h2r, represented by h2q_1
#   and i==0 is h2q1, represented by h2q0
#
    set firstbelement 3
    set firstindex 0
    if {{} == $covarlist} {
	set n [llength $paramlist]
    } else {
	set n [llength $covarlist]
	if {[if_parameter_exists h2q1] && ![is_constrained_to_zero h2q1]} {
	    set firstindex -1
	    set firstbelement 5
	}
    }
#
# setup result variables
# "h2q0" is actually h2r (or h2q1, see above)
#
# OR, for covariates, 
# h2qX is really the covariate numbered X
#
#
    for {set i $firstindex} {$i <= $maxindex} {incr i} {
	set p $i
	if {$p < 0} {
	    set p [catenate _ [expr abs($i)]]
	}
	set h2q$p 0
	set h2qse$p 0
	set h2qp$p 0
    }
#
# Open file and skip over header
#
    set infile [open [full_filename $outname.nose] r]
    gets $infile inline
    gets $infile inline
    if {"----------" != [string range $inline 1 10]} {
	close $infile
	error "Error(s) in bayesavg.out"
    }
    set data_offset [tell $infile]
#
# Make windowfile for reporting about models in window
#
    global Solar_Bay_Qtn
    if {[if_global_exists Solar_Bay_Qtn] && $Solar_Bay_Qtn==1} {
	set wheadings "Model BIC PProb SNP_Flags"
	set wformats "%20s %11.4f %10s %[expr 2*$maxindex]s"
	set wexpressions {{$wfilename} {$wbic} {$wpprob} {$snp_flags}}
	set wfile [resultfile -create [full_filename $outname.win] \
		       -headings $wheadings -formats $wformats \
		       -expressions $wexpressions]
	resultfile $wfile -header
    }
#
# Pass through file summing exp(-1/2 * BIC)
#
    ifdebug puts "Summing exp(-1/2 * BIC)"
    set eBICsum 0
    set count 0
    set window {}
    set window_of_filenames {}
    set force_cov0 0
    set forced_cov0 0
    if {[if_global_exists Solar_Bay_Qtn] && $Solar_Bay_Qtn==1} {
	set force_cov0 1
    }

    while {0 < [gets $infile inline]} {
	if {2 != [scan $inline "%s %f" filename BIC]} {
	    close $infile
	    error "Error detected in $outname.out"
	}
	if {$excludenull && ("c0" == $filename || "cov0" == $filename)} continue
	if {$count == 0} {
	    set max_bic [expr $BIC + $cutoff]
	} else {
	    if {$BIC > $max_bic} {
		if {!$force_cov0} {
		    break
		} else {
		    if {-1 != [lsearch $window_of_filenames cov0]} {
			set force_cov0 0   ;# No need to force it
			break
		    }
		    set forced_cov0 1
		    set filename cov0
		    set BIC 0
		    set model_elements {}
		}
	    }
	}
	incr count
#
# For STRICT Occam's window
# See if we have encountered any models (having better BICs and)
# which are subsets of this model
#
	if {!$forced_cov0} {
	set model_elements [bayesavg_elements $filename]
	if {0==$symmetric} {
	    set found_subset 0
	    foreach comb $window {
		if {1==[subset $model_elements $comb]} {
		    set found_subset 1
		    break
		}
	    }
	    if {$found_subset} {
		continue
	    }
	}
	set eBIC [expr exp(-0.5 * $BIC)]
	set eBICsum [expr $eBICsum + $eBIC]
	}
	lappend window $model_elements
	lappend window_of_filenames $filename
	lappend window_of_bic $BIC
    }
#
# Write window file
#
    if {[if_global_exists Solar_Bay_Qtn] && $Solar_Bay_Qtn==1} {
	set window_size [llength $window]
	for {set i 0} {$i < $window_size} {incr i} {
	    set wfilename [lindex $window_of_filenames $i]
	    set wbic [lindex $window_of_bic $i]
	    if {"cov0" == $wfilename && $forced_cov0} {
		set cpprob [expr 1.0 / ($eBICsum + 1.0)]
		ifdebug puts "calculating pprob for forced cov0: $wpprob"
		set wpprob [format %10.6f $cpprob]
                if {0.0001 >  $wpprob} {
                    set wpprob [format %10.3g $cpprob]
		}
	    } else {
		ifdebug puts "calculating pprob old way"
		set wpprob [expr exp(-0.5*$wbic) / $eBICsum]
		set wpprob [format %10.6f $wpprob]
	    }
	    set snp_flags ""
	    set wset [bayesavg_elements $wfilename]
	    for {set j 1} {$j <= $maxindex} {incr j} {
		if {-1 != [lsearch $wset $j]} {
		    set snp_flags "$snp_flags 1"
		} else {
		    set snp_flags "$snp_flags 0"
		}
	    }
	    resultfile $wfile -write
	}
    }
#
# If we "forced" window to include cov0, remove it now
#   Note: not necessary to remove from $window because it's remade anyway
#         not necessary to remove from $window_of_bic since it's not reused
#
    if {$forced_cov0} {
	set last_index [expr [llength $window] - 2]
	set window_of_filenames [lrange $window_of_filenames 0 $last_index]
    }
#
# Pass through file again computing PProbs and h2q$p sums
# 
    ifdebug puts "computing h2qi sums"
    seek $infile $data_offset
    set window {}
    while {0 < [gets $infile inline]} {
	if {2 != [scan $inline "%s %f" filename BIC]} {
	    close $infile
	    error "Error detected in $outname.out"
	}
	if {$excludenull && ("c0" == $filename || "cov0" == $filename)} continue
	set BIC [lindex $inline 1]
	if {$BIC > $max_bic} break
	set model_elements [bayesavg_elements $filename]
	if {0==$symmetric} {
	    set found_subset 0
	    foreach comb $window {
		if {1==[subset $model_elements $comb]} {
		    set found_subset 1
		    break
		}
	    }
	    if {$found_subset} {
		continue
	    }
	}
	lappend window $model_elements

	set PProb [expr exp(-0.5 * $BIC) / $eBICsum]

	for {set i $firstindex} {$i <= $maxindex} {incr i} {
	    set h2q [lindex $inline [expr $firstbelement + ($i * 2)]]
	    set p $i
	    if {$p < 0} {
		set p [catenate _ [expr abs($i)]]
	    }
	    if {[is_nan $h2q]} {
		close $infile
		error "NaN(s) found in bayesavg.nose"
	    }
	    set weighted [expr $h2q * $PProb]
	    ifdebug puts "for $i adding value $h2q mpprob $PProb weighted $weighted"
	    set h2q$p [eval expr \${h2q$p} + $weighted]
#
# If this model includes element i, and if it is "non-zero"
#   it adds to the posterior probability of that element
#
	    if {($i<=0) || (0<=[lsearch -exact $model_elements $i])} {
		if {abs($h2q) >= 1e-12} {
		    set h2qp$p [eval expr \${h2qp$p} + $PProb]
		}
	    }
	}
    }
    if {$no_se} {
#
# Rename output file since s.e.'s not needed
#
	file rename [full_filename $outname.nose] [full_filename $outname.out]

    } else {
#
# Open resultfile for output of file with std errors
#
    set columns [llength $formats]
    set expressions {}
    for {set i 0} {$i < $columns} {incr i} {
	lappend expressions "\[lindex \$inline $i\]"
    }
    set outfile [resultfile -create [full_filename $outname.out] \
	    -headings $headings -formats $formats -expressions $expressions]
    resultfile $outfile -header
#
# Pass through file again computing h2qse$p sums
#
    ifdebug puts "computing h2qse sums"
    seek $infile $data_offset
    set window {}
    set finished 1
    while {0 < [gets $infile inline]} {
	set filename [lindex $inline 0]
	if {! ($excludenull && ("c0" == $filename || "cov0" == $filename))} {
	    set BIC [lindex $inline 1]
	    if {$BIC > $max_bic} {
		set finished 0
		break
	    }
	    set model_elements [bayesavg_elements $filename]
	    set found_subset 0
	    if {0==$symmetric} {
		foreach comb $window {
		    if {1==[subset $model_elements $comb]} {
			set found_subset 1
			break
		    }
		}
	    }
	    if {!$found_subset} {
		lappend window $model_elements

		if {$filename == $bestmodel  && \
			0 != [oldmodel $filename mean -se]} {
		    load model [full_filename $bestmodel]
		} else {
		    putsout $outname.history \
			    "    *** Maximizing $filename for Standard Errors"
#
# Create model from base by restoring elements
#
		    load model [full_filename $basemodel]
		    set cfilename [file root [file tail $filename]]
		    set comb [bayesavg_elements $cfilename]
		    if {{} != $covarlist} {
			for {set i 0} {$i < $n} {incr i} {
			    if {-1 != [lsearch $comb [expr $i + 1]]} {
				set covi [lindex $covarlist $i]
				covariate restore $covi
			    }
			}
		    } else {
			foreach ce $comb {
			    set param [lindex $paramlist [expr $ce - 1]]
			    if {-1==[string first "-" $param]} {
				constraint delete $param
			    } else {
				constraint delete <$param>
			    }
			    carve_new_value $param 0.01 h2r
			    parameter $param lower 0
			}
		    }
#
# Maximize model with standard errors turned on, save w/o covariates
#
		    option standerr 1
		    if {[catch {maximize -q}]} {
			close $infile
			error "Error maximizing model for Standard Errors"
		    }
		    suspend2constrain
		    save model [full_filename $filename]
		}
#
# Accumulate statistics for each variable parameter (including h2r)
#
		set PProb [expr exp(-0.5 * $BIC) / $eBICsum]
		for {set i $firstindex} {$i <= $maxindex} {incr i} {
		    set parvalueindex [expr $firstbelement + ($i * 2)]
		    set parvalue [lindex $inline $parvalueindex]
		    if {$parvalue != 0.0} {
			if {$i == $firstindex} {
			    set ename h2r
			} elseif {$i == 0} {
			    set ename h2q1
			} else {
			    if {"" != $covarlist} {
				set ename [lindex $betalist [expr $i - 1]]
			    } else {
				set ename  [lindex $paramlist [expr $i - 1]]
			    }
			}
			set se [oldmodel $filename $ename -se]
			set parcheck [oldmodel $filename $ename]
			set changeparvalue [expr abs($parvalue - $parcheck)]
			if {$changeparvalue > 0.1} {
			    puts "$ename was $parvalue now $parcheck"
			    error "Re-maximized model $filename is different"
			}
			ifdebug puts "$ename was $parvalue now $parcheck"
			if {[is_nan $se]} {
			    close $infile
			    error \
		        "NaN(s) for standard error in model [lindex $inline 0]"
			}

# Update standard error for final output file
			set p $i
			if {$p < 0} {
			    set p [catenate _ [expr abs($i)]]
			}

			set se_index [expr $firstbelement + 1 + ($i * 2)]
			set inline [lreplace $inline $se_index $se_index $se]
			set variance [expr $se * $se]
			set h2qse$p [eval expr \${h2qse$p} + \
			      ( $PProb * ( $variance + ($parvalue*$parvalue)))]
		    }
		}
	    }
	}
	resultfile $outfile -write
    }
#
# Copy rest of file (outside of window, no se's added)
#
    if {!$finished} {
	resultfile $outfile -write   ;# Write line just read
	while {-1 != [gets $infile inline]} {
	    resultfile $outfile -write
	}
    }
    close $infile
#
# Compute actual h2qse$p values
#
    for {set i $firstindex} {$i <= $maxindex} {incr i} {
	set p $i
	if {$p < 0} {
	    set p [catenate _ [expr abs($i)]]
	}
	set emean2 [eval square \${h2q$p}]
 	set sevalue [expr sqrt ([eval expr \${h2qse$p} - $emean2])]
 	set h2qse$p $sevalue
    }
    }  ;# End of if not -no_se

# *****************************************************************************
# Write bayesavg.avg with bayesian model averages
# *****************************************************************************

    set outfilename [full_filename $outname.avg]
    set soutfile [open $outfilename w]
    putstee $soutfile "    *** Number of Models in Window: [llength $window]"
    putstee $soutfile "    *** Window:  $window_of_filenames"
    putstee $soutfile ""
    close $soutfile


    set headings {Component Average {Std Error} Probability}
    if {{} != $covarlist} {
	set width_needed 32
	for {set i $firstindex} {$i < $maxindex} {incr i} {
	    set component [lindex $covarlist $i]
	    if {$width_needed < [string length $component]} {
		set width_needed [string length $component]
	    }
	}
	set aformats "%[catenate $width_needed s] %-13.8g %-13.8g %-12.7g"
    } else {
	set aformats "%25s %-12.7g %-13.8g %-12.7g"
    }

    set resultf [resultfile -append $outfilename -display \
	    -headings $headings \
	    -formats $aformats \
	    -expressions {$component $average $stderror $postprob}]
    resultfile $resultf -header

    for {set i $firstindex} {$i <= $maxindex} {incr i} {

	set p $i
	if {$p < 0} {
	    set p [catenate _ [expr abs($i)]]
	}

	if {$i == $firstindex} {
	    set component H2r
	} elseif {$i == 0} {
	    set component H2q1
	} else {
	    if {{} != $covarlist} {
		set component [lindex $covarlist [expr $i-1]]
	    } else {
		set component H2q$p
	    }
	}
	set average [eval expr \${h2q$p}]
	set stderror [eval expr \${h2qse$p}]
	set postprob [eval expr \${h2qp$p}]
	resultfile $resultf -write
    }
#
# Delete .nose file if it's the same length
#
    catch {
	if {[file exists [full_filename $outname.nose]] && \
		[file exists [full_filename $outname.out]] && \
		[file exists [full_filename $outname.est]] && \
		[file size [full_filename $outname.nose]] == \
		[file size [full_filename $outname.out]]} {
	    file delete [full_filename $outname.nose]
	}
    }
#
# Helpful messages
#
    putsout $outname.history "\n    *** Averages written to [full_filename $outname.avg]"
    putsout $outname.history "    *** Model results written to [full_filename $outname.out]"
    puts "    *** Messages written to [full_filename $outname.history]"
#
# Load best model
#
    load model [full_filename $bestmodel]
    putsout $outname.history "    *** Model with best BIC loaded: $bestmodel"
    return ""
}

proc subset {listall listsub} {
    set mismatch 0
    foreach el $listsub {
	if {-1==[lsearch $listall $el]} {
	    set mismatch 1
	    break
	}
    }
    if {0==$mismatch} {
	return 1
    }
    return 0
}

#
# Return a list of the numbered elements (1..N) included in this model
#
# _ is used between all numbers
#
proc bayesavg_elements {cfilename} {
    set elemstr [string range $cfilename 1 end]
    if {"ov" == [string range $elemstr 0 1]} {
	set elemstr [string range $elemstr 2 end]
    }
    set elements [split $elemstr _ ]
    if {![string compare $elements 0]} {
	set elements ""
    }
    return $elements
}

	
# solar::bayesmod -- private
#
# Purpose: To build a model containing only the numbered elements:
#          linkage components or elements
#
# Usage:   bayesmod [-cov] [-h2rf] [-nomax] [-new] i j k ...
#          i,j,k are the element numbers, starting with 1, that are to be
#          included.  Model is constructed from the previously saved
#          saturated model.
#
#           -cov            ;# Covariate elements (not linkage)
#           -h2rf           ;# h2r factor...see help bayesavg...default is 1.1
#           -max            ;# maximize model
#           -new            ;# build model from model in memory
#                           ;#   NOT from previous bayesavg saturated model
#
# Example:  trait bmi       ;# Set trait so output directory can be found
#           bayesmod 1 3    
#           bayesmod -cov 1 3 7 8
#
# Notes:    (1) Unlike the bayesavg procedure,
#               excluded covariates are constrained to zero rather than 
#               being "suspended."  This is equivalent, but constrained
#               covariates are easier for users to deal with.
# -

proc bayesmod {args} {

    set covar 0
    set h2r_factor 1.1
    set max 0
    set new 0
    set retval ""
    set last_verbosity [verbosity]

    set numlist [read_arglist $args -cov {set covar 1} -h2rf h2r_factor \
	    -max {set max 1} -new {set new 1}]

    foreach num $numlist {
	ensure_integer $num
    }
#
# Covariate model
#
    if {$covar} {
	if {!$new} {
	    load model [full_filename cov.sat.mod]
	}
	set betalist [covariates -betanames]
	set numcov [llength $betalist]
	foreach num $numlist {
	    if {$num > $numcov} {
		error "There are only $numcov covariates"
	    }
	}
	for {set i 1} {$i <= $numcov} {incr i} {
	    if {-1==[lsearch $numlist $i]} {
		set betaname [lindex $betalist [expr $i - 1]]
		parameter $betaname = 0
		if {-1==[string first "-" $param]} {
		    constraint $betaname = 0
		} else {
		    constraint <$betaname> = 0
		}
	    }
	}
	if {$max} {
	    puts "Maximizing..."
	    verbosity plus
	    set retval [maximize_quietly bayesmod.out]
	}
#
# Linkage model
#
    } else {
	if {!$new} {
	    load model [full_filename c.sat.mod]
	}
	set h2qc [h2qcount]
	foreach num $numlist {
	    if {$num > $h2qc} {
		error "There are only $h2qc linkage elements"
	    }
	}
	for {set i 1} {$i <= $h2qc} {incr i} {
	    if {-1==[lsearch $numlist $i]} {
		parameter h2r start [expr [parameter h2r start] + \
			[parameter h2q$p start]]
		parameter h2q$p start 0
		parameter h2q$p lower 0.01
		constraint h2q$p = 0
	    }
	}
	parameter h2r upper [lowest \
		[expr [parameter h2r start] * $h2r_factor] 1]
	if {$max} {
	    puts "Maximizing..."
	    verbosity plus
	    set retval [max_bayesavg [full_filename bayesmod.out] $numlist]
	}
    }
    eval $last_verbosity
    return $retval
}    

proc square {value} {return [expr $value * $value]}

proc bayesavg_bad_results {cfilename bad_results sorted} {
    set tempname [full_filename $cfilename.tmp]
    set permname [full_filename $cfilename]
    set tmpoutput [open $tempname w]
    foreach result $bad_results {
	if {[llength $result] > 0} {
	    puts $tmpoutput $result
	    puts stderr $result
	}
    }
    if {$sorted == 0} {
	puts $tmpoutput \
		"Convergence error limit exceeded; terminating prematurely"
	puts $tmpoutput \
		"Results are incomplete and not sorted"
    } elseif {$sorted == 1} {
	puts $tmpoutput "Results are incomplete but sorted"
    } elseif {$sorted == 2} {
	puts $tmpoutput \
		"Convergence errors occurred"
	puts $tmpoutput \
		"Results are incomplete and not sorted due to sort problem"
    }
    puts $tmpoutput ""
    close $tmpoutput
    eval exec cat $permname >>$tempname
    eval file rename -force $tempname $permname
    error "Terminating with convergence errors"
}
#
# Maximize with retries suitable for bayesavg linkage models
#
proc max_bayesavg {outfilename comb} {
    set errmsg [maximize_quietly $outfilename]
    if {$errmsg == ""} {
	return ""
    }
    if {"verbosity min" != [verbosity]} {
	puts "\n    *** Retry with equal active H* parameters"
    }
    model load [full_filename last.mod]
    set hstart [expr (1-[parameter e2 start])/ \
	    ([llength $comb] + 1)]
    set elements [linsert $comb 0 0]
    foreach i $elements {
	if {$i == 0} {
	    set hname h2r
	} else {
	    set hname h2q$i
	}
	parameter $hname start $hstart
	set hupper [parameter $hname upper]
	set hlower [parameter $hname lower]
	if {$hstart >= $hupper} {
	    parameter $hname upper [expr $hstart + 0.001]
	}
	if {$hstart <= $hlower} {
	    parameter $hname lower [expr $hstart - 0.001]
	}
    }
    set errmsg [maximize_quietly $outfilename]
    if {$errmsg == ""} {
	return ""
    }
    return "Convergence problem with combination: $comb"
}

# solar::combinations --
#
# Purpose:  Make a list or count combinations of integers 1..N of size K
#
# Usage:    combinations <N> [<K>] [-max <maxsize>] [-list list] [-force]
#                        [-count] [-counts] [-start <number>] [-stop <number>]
#
#           N        defines the range of integers 1..N of interest.  If no
#                    other arguments are specified, and N <= 10, the set of
#                    all combinations of this range of integers is returned.
#                    To get a list of combinations where possibly N > 10, 
#                    add either the -list or -force option, with -list being
#                    the preferred method.
#
#           K        only include combinations of exactly this size (as
#                    in traditional "combinations").  If this argument is
#                    not specified, the default is to include combinations
#                    of all sizes, starting from the smallest size.
#
#           -count   Only return the NUMBER of combinations, not a list
#                    of the actual combinations.
#
#           -counts   Return a list containing the number of combinations for
#                     each "size" (i.e. "K").
#
#           -max     include all combinations up to and including this size
#                    (the default is to include combinations of all sizes).
#                    The K and -max arguments may not be used at the
#                    same time.
#
#           -list    APPEND combinations to this list rather than returning
#                    them.  Specify the list variable by name, as with the
#                    Tcl lappend command (see example below).  If the variable
#                    is not already set, a new variable is created.  When this
#                    argument is used, nothing is returned.  For example:
#
#                        set comblist {}
#                        combinations 20 -max 10 -list comblist
#
#                    Be sure to empty the list first (as shown above) if you
#                    do not want to append to the previous contents, if the
#                    variable was used previously in the same procedure.  This
#                    option may save memory (as compared with -force) for
#                    very large N since only one copy of the list is ever
#                    created.
#                  
#           -force   return list ("by value") even if N > 10.  This
#                    is required for N > 10 unless the -list, -count, -counts,
#                    -start, or -stop arguments are given.  Only use this
#                    option if you are sure this is what you want to do.
#                    Read all the following paragraphs to be sure.  Generally,
#                    you would only use it inside a script, where the
#                    returned combinations are going to be immediately saved
#                    to a variable, such as:
#
#                        catch {set comblist [combinations $N -force]}
#
#                    The reason to require a -force option is that if a
#                    large N is given in an interactive session, the
#                    terminal window could be locked up for hours displaying
#                    all the combinations, with no way to break out until
#                    the terminal output buffer is empty.  If that were to
#                    happen, you would probably want to kill the whole
#                    terminal session from another terminal window.  For
#                    some users, that would probably require calling the
#                    system administrator.
#
#                    The -force option may require more memory than the -list
#                    option because a copy of the list is created in the
#                    process of "returning" it to the caller; that's just
#                    the way Tcl works, and it becomes important when creating
#                    lists with huge numbers of elements.
#
#                    If you are using this form of the command in a script,
#                    be careful that it is not the last command in the
#                    script, which Tcl automatically returns.  Then, if
#                    the user runs the script from the terminal, the
#                    terminal window would be locked up.  If you must
#                    use it as the last command in a script, you should
#                    use a "catch" command around it, as in the example
#                    above.  The catch command only returns 0 (for success)
#                    or 1 (for error).
#
#           The following options are useful when dividing up the set of
#           combinations into jobs of an equal size.  Otherwise, they may
#           seem a bit obscure.
#
#           -start   Start with combination number <number>
#
#           -stop    Stop with combination number <number>
#
# Notes:
#
# CAUTION!  The list can get VERY BIG!  Be careful if n > 15 because
# memory requirements double for each [incr n], unless you are setting k
# or -max.  ("BIG" means 100's of megabytes, gigabytes, etc.  I am not
# kidding. On Solaris systems, you can use the SOLAR "memory" command to see
# just how much memory SOLAR has consumed.)
#
# -

proc combinations {n args} {

    if {$n < 1} {
	error "N must be greater than zero"
    }

    set minsize 1
    set maxsize $n
    set max ""
    set size 0
    set listname ""
    set force 0
    set start 0
    set stop 0
    set count 0
    set do_count_only 0
    set do_count_list 0
    set count_list {}

    set ksize [read_arglist $args \
	    -max max \
	    -list listname \
	    -force {set force 1} \
	    -start start \
	    -stop stop \
	    -count {set do_count_only 1} \
	    -counts {set do_count_only 1; set do_count_list 1} \
	]

    if {"" != $ksize} {
	if {![is_integer $ksize]} {
	    error "Invalid argument(s): $ksize"
	}
	if {"" != $max} {
	    error "-max and -size arguments are incompatible"
	}
	if {$ksize < 1 || $ksize > $n} {
	    error "size $ksize is invalid"
	}
	set minsize $ksize
	set maxsize $ksize
    }

    if {"" != $max} {
	if {![is_integer $max]} {
	    error "-max requires integer value"
	}
	if {$max < 1} {
	    error "Positive -max required"
	}
	if {$max > $n} {
	    error "-max cannot be greater than N"
	}
	set maxsize $max
    }

    if {!$do_count_only && !($start && $stop)} {
	if {$n > 10 && 0 == [llength $listname] && !$force} {
	    error "N > 10 not permitted without -list or -force"
	}
    }

    if {0 == [llength $listname]} {
	set comb {}
    } else {
	upvar $listname comb
    }

    for {set i $minsize} {$i <= $maxsize} {incr i} {
	combinations_ref $n $i comb $start $stop
	if {$do_count_list} {
	    lappend count_list $count
	    set count 0
	}
	if {$stop && ($count >= $stop)} {
	    break
	}
    }

    if {$do_count_list} {
	return $count_list
    }

    if {$do_count_only} {
	if {$start} {
	    set count [expr $count + 1 - $start]
	}
	return $count
    }

    if {0 == [llength $listname]} {
	return $comb
    }
    return ""
	
}

#
# Compute traditional combinations(n,k) where k is fixed size
#
# This is NOT a user interface.  proc combinations (above) is the
# user interface.
#
# Caller's list passed by name (like call by reference) and augmented
#
# Iterative "quirky odometer" method
#   Each digit is a element
#   Start from "floor"
#   Twirl last digit and carry
#   On carry, following digits are successive integers (e.g. 5 6 7)
#   Each digit can only go to final "ceiling" value
#
proc combinations_ref {n k combname start stop} {
#
# Test for caller error
#
    if {$n < $k} {
	error "N cannot be less than K"
    }
#
# Link to caller's array
#
    upvar $combname combinations_r
#
# Link to caller's count and do_count_only
#
    upvar count count
    upvar do_count_only do_count_only
#
# Setup first ("floor") combination
#
    set floor {}
    for {set i 1} {$i <= $k} {incr i} {
	lappend floor $i
    }
    incr count
    if {!$do_count_only} {
	if {!$start || ($count >= $start)} {
	    lappend combinations_r $floor
	}
    }
    if {$stop && ($count >= $stop)} {
	return ""
    }
#
# Setup ceiling combination (for comparison use)
#
    set delta [expr $n - $k]
    set ceiling {}
    for {set i [expr 1 + $delta]} {$i <= $n} {incr i} {
	lappend ceiling $i
    }
#
# Now generate remaining combinations
#
    set done 0
    set last [expr $k - 1]
    set base $floor
    while {1} {
#
# Twirl last digit
#
	set startnum [lindex $base $last]
	set endnum [lindex $ceiling $last]
	while {$startnum < $endnum} {
	    incr startnum
	    incr count
	    set base [lreplace $base $last $last $startnum]
	    if {!$do_count_only} {
		if {!$start || ($count >= $start)} {
		    lappend combinations_r $base
		}
	    }
	    if {$stop && ($count >= $stop)} {
		return ""
	    }
	}
#
# Carry and start over (or finish)
#
	set previous [expr $last - 1]
	if {$previous < 0} {
	    break
	}
	while {[lindex $base $previous] == [lindex $ceiling $previous]} {
	    incr previous -1
	    if {$previous < 0} {
		return ""
	    }
	}
	set newvalue [expr [lindex $base $previous] + 1]
	for {set pntr $previous} {$pntr < $k} {incr pntr} {
	    set base [lreplace $base $pntr $pntr $newvalue]
	    incr newvalue
	}
	incr count
	if {!$do_count_only} {
	    if {!$start || ($count >= $start)} {
		lappend combinations_r $base
	    }
	}
	if {$stop && ($count >= $stop)} {
	    return ""
	}
    }
    return ""
}


#
# Old recursive method (now obsolete but maintained for testing purposes)
#
proc oldcombinations {n {max -1}} {
    if {$n == 1} {return 1}
    set minus1 [oldcombinations [expr $n - 1] $max]
    foreach e $minus1 {
	if {$max != -1 && $max <= [llength $e]} {continue}
	lappend comb [concat $e $n]
    }
    foreach e $minus1 {
	lappend comb $e
    }
    return [lappend comb $n]
}

proc combfile {n {max -1}} {
    if {$n == 1} {
	set ofile [open combfile.dat w]
	puts $ofile 1
	close $ofile
	return ""
    }
    combfile [expr $n - 1] $max
    set ofile [open combfile.dat a+]
    puts $ofile $n
    set end_file [tell $ofile]
    seek $ofile 0
    while {$end_file > [tell $ofile]} {
	gets $ofile cline
	set read_position [tell $ofile]
        if {$max != -1 && [llength $cline] >= $max} {continue}
	seek $ofile 2 end
	puts $ofile "$cline $n"
        flush $ofile
	seek $ofile $read_position
    }
    close $ofile
    return ""
}

# solar::resultfile -- private
#
# Purpose: output results to file and/or screen
#
# Usage:  resultfile -create <filename> -headings <headings> -formats <formats>
#             -expressions <expressions> [<options>]
#
#         (-append may be substituted for -create if appending to an existing
#           file)
#
#         This returns a resultfile object which is used in subsequent
#         commands. The resultfile object will NOT be returned if the
#         -write or -header option is used, instead, the string written
#         to file and/or terminal is returned.  [NOTE: This changed
#         with revision 2.0.2.]
#
#         resultfile <resultfile> -header
#
#         resultfile <resultfile> -write
#
#         options are:  -display (tee to standard output)
#                       -displayonly (display ONLY on standard output)
#                       -default <default result>  {0 is default default}
#
# Note:  This is intended as a replacement for outheader/outresults which
#        was becoming increasingly difficult to use.
#
#        -create, -append, -header, and -write may not be used in combination.
#
#        Closing is not required.  The file is re-opened and closed each
#        time results are written.
# -

proc resultfile {args} {

    set terminator 0
    set cfilename ""
    set afilename ""
    set header 0
    set write 0
    set formats {}
    set expressions {}
    set display 0
    set displayonly 0
    set default_result 0

    set oldresultf [read_arglist $args -create cfilename \
	    -append afilename \
	    -header {set header 1} \
	    -write {set write 1}\
            -default default_result \
	    -display {set display 1} \
	    -displayonly {set display 1; set displayonly 1} \
	    -returnonly {set displayonly 1} \
	    -headings headings -formats formats \
	    -terminator {set terminator 1} \
	    -expressions expressions]

    if {{} == $oldresultf && "" == $cfilename && "" == $afilename} {
	error "Resultfile requires -create, -append, or previous object"
    }
#
# Create a new 'resultfile'
#
    if {"" != $cfilename || $afilename != ""} {
	if {{} == $headings || {} == $formats || {} == $expressions} {
           error "Resultfile creation missing headings, formats or expressions"
	}
	if {"" != $cfilename && "" != $afilename} {
	    error "Resultfile create and append options are incompatible"
	}
	if {!$displayonly} {
	    if {"" != $cfilename} {
		set filename $cfilename
		set outfile [open $cfilename w]
		close $outfile
	    } else {
		set filename $afilename
	    }
	}
#
# Create header (regardless of whether we write it now)
#
	set headerstring ""
	set understring ""
	set widths {}
	for {set i 0} {$i < [llength $headings]} {incr i} {
	    set heading [lindex $headings $i]
	    set format [lindex $formats $i]
	    if {"" == $heading} continue   ;# Allows for optional postfix
#
# Width of field is width of heading, or width of format, whichever is
# longer
#
	    set header_length [string length $heading]
	    set format_length [format_width $format]
	    if {$format_length < 0} {
		set format_length [expr 1 - $format_length]
	    }
	    if {$format_length > $header_length} {
		set width $format_length
	    } else {
		set width $header_length
	    }
	    lappend widths $width
#
# Center the title in a field of appropriate width
#   If centering requires an uneven number of spaces, put the odd
#   space in front
#
	    set blanks [expr $width - [string length $heading]]
	    set front [expr ceil($blanks/2.0)]
	    set back [expr floor($blanks/2.0)]
	    for {set j 0} {$j < $front} {incr j} {
		set headerstring "$headerstring "
	    }
	    set headerstring "$headerstring$heading"
	    for {set j 0} {$j < $back} {incr j} {
		set headerstring "$headerstring "
	    }
#
# Create the underline string:
# One hyphen under each letter of the header, expanding out to the
#   width of the field, but centered abound the title
#
	    if {$front > $back} {
		set understring "$understring "
	    }
	    for {set j 0} {$j < $back} {incr j} {
		set understring [catenate $understring -]
	    }
	    for {set j 0} {$j < $header_length} {incr j} {
		set understring [catenate $understring -]
	    }
	    for {set j 0} {$j < $back} {incr j} {
		set understring [catenate $understring -]
	    }
#
# If there is going to be another field, add a space to each
#
	    if {$i < [llength $headings] - 1} {
		set headerstring "$headerstring "
		set understring "$understring "
	    }
	}
	return "resultfile $args -widths \{$widths\} \
	    -headerstring \{$headerstring\} -understring \{*$understring\}"
    }
#
# Load previously created object and write something
#
    set widths {}
    set formats {}
    set expressions {}
    set headings {}
    set filename ""
    set headerstring ""
    set understring ""
    set default_result 0
    set terminator 0

    set resultflen [string length $oldresultf]
    set resultf [string range $oldresultf 1 [expr $resultflen-2]]
    set foo [read_arglist $resultf -widths widths -formats formats \
		 -expressions expressions -create filename -append filename \
		 -headings headings -header {set foo 1} -write {set foo 1} \
                 -default default_result \
		 -displayonly {set displayonly 1; set display 1} \
		 -returnonly {set displayonly 1} \
		 -display {set display 1} -headerstring headerstring \
		 -terminator {set terminator 1} \
		 -understring understring]
    set understring [string range $understring 1 end]
#
# Write header if requested
#
    if {$header} {
	if {$display} {
	    puts $headerstring
	    puts $understring
	}
	if {!$displayonly} {
	    set outfile [open $filename a]
	    puts $outfile $headerstring
	    puts $outfile $understring
	    close $outfile
	}
	return $headerstring
    }
#
# Write results if requested
#
    if {$write} {
	if {{} == $widths || {} == $formats || {} == $expressions} {
	    error "resultfile requires -formats and -expressions"
	}

	set resultstring ""
	for {set i 0} {$i < [llength $expressions]} {incr i} {
	    set expression [lindex $expressions $i]
	    set formatstr [lindex $formats $i]
	    set width [lindex $widths $i]
	    set express "set resultfile_FoObAr03 $expression"
	    set result $default_result
	    catch {set result [uplevel $express]}  ;# evauation done here
#
# If format fails (maybe numeric format with alpha value)
# just procede
#
	    if {[catch {set result [format $formatstr $result]}]} {
	    }
#
# Whether format failed or not, adjust to specified width
#
	    set result [format [catenate "%$width" s] $result]
	    set resultstring "$resultstring$result"
	    if {$i < [llength $expressions]} {
		set resultstring "$resultstring "
	    }
	}
	if {$display} {
	    puts $resultstring
	}
	if {!$displayonly} {
	    set outfile [open $filename a]
	    if {!$terminator} {
		puts $outfile $resultstring
	    } else {
		puts $outfile "$resultstring *"
	    }
	    close $outfile
	}
	return $resultstring
    }

# Should not get here unless no operative option
    return ""
}


proc format_width {format} {
    set format_width_string [string range $format 1 end]
    set count [scan $format_width_string %d width]
    if {1 != $count} {
	error "Invalid format $format"
    }
    return $width
}

# solar::outheader -- private
#
# Purpose: Output header for analysis scripts
#
# Usage:   outheader filename h2qindex col_2 show_se
#
# Notes:   This is for use in combination with outresults script.
# -

proc outheader {filename h2qindex col_2 show_se {epi 0}} {
    if {"/dev/null" != $filename} {
	set soutfile [open [full_filename $filename] w]
	fioutheader $soutfile $h2qindex $col_2 $show_se $epi
	close $soutfile
    }
    fioutheader stdout $h2qindex $col_2 $show_se $epi
    return ""
}

proc fioutheader {soutfile h2qindex col_2 show_se {epi 0}} {

    puts -nonewline $soutfile "           Model   "
    set shadow                "-------------------"
    if {[llength $col_2]} {
# expand col_2 to 9 spaces centered
	set blanks [expr 9 - [string length $col_2]]
	set prefix [format [catenate % [expr ceil($blanks/2.0)] s] ""]
	set suffix [format [catenate % [expr floor($blanks/2.0)] s] ""]
	set col_2 $prefix$col_2$suffix
	puts -nonewline $soutfile "   $col_2"
	set shadow         "$shadow   ---------"
    }
    puts -nonewline $soutfile "    Loglike  "
    set shadow         "$shadow  -----------"
    puts -nonewline $soutfile "     H2r  " 
    set shadow         "$shadow  --------"
    if {$show_se} {
	puts -nonewline $soutfile [format "%10s" "H2r SE "]
	set shadow "$shadow  --------"
    }

    for {set i 1} {$i <= $h2qindex} {incr i} {
	puts -nonewline $soutfile [format "%8s  " H2q$i]
	set shadow "$shadow  --------"
	if {$show_se} {
	    puts -nonewline $soutfile [format "%10s" "H2q$i SE"]
	    set shadow "$shadow  --------"
	}
    }
    if {$epi} {
	puts -nonewline $soutfile [format "%8s  " H2qE1]
	set shadow "$shadow  --------"
    }
    puts $soutfile "\n$shadow"
}

#  solar::outresults -- private
#
# Purpose: Output a line showing LOD and other stats for current model
#
# Usage:   outresults filename modelname lod loglike h2qindex err sefile
#
# Notes:   Model should be maximized before using this script.
# -

proc outresults {filename modelid lod loglik h2qindex {err 0} {sefile none} \
	{modelfile -none} {score none} {lcse none}} {
    set outstring [formatresults $modelid $lod $loglik $h2qindex $err $sefile \
	    $modelfile $score $lcse]
    puts stdout $outstring

    set soutfile [open [full_filename $filename] a]
    puts $soutfile $outstring
    close $soutfile

    return $outstring
}


proc formatresults {modelid lod loglik h2qindex {err 0} {sefile none} \
	{modelfile -none} {score none} {lcse none}} {

    set outstring [format %19s $modelid]

    if {"NA" != $lod} {
	if {[llength $lod]} {
	    set outstring "$outstring [format %11.4f $lod]"
	} else {
	    set outstring "$outstring [format %11s ""]"
	}
    }

    if {0 != [string compare [string toupper $loglik] NAN]} {
	set outstring "$outstring [format %12.3f $loglik]"
    } else {
	set outstring "$outstring [format %12s NaN]"
    }

    if {"-none" == $modelfile} {
	set par_h2rs [parameter h2r start]
    } else {
	set par_h2rs [oldmodel $modelfile h2r]
    }

    if {[string compare [string toupper $par_h2rs] NAN]} {
	set outstring "$outstring [format %9.6f $par_h2rs]"
    } else {
	set outstring "$outstring [format %9s NaN]"
    }

    for {set i 1} {$i <= $h2qindex} {incr i} {

	if {"-none" == $modelfile} {
	    set par_h2q [parameter h2q$i start]
	} else {
	    set par_h2q [oldmodel $modelfile h2q$i]
	}

	if {[string compare [string toupper $par_h2q] NAN]} {
	    set outstring "$outstring [format %9.6f $par_h2q]"
	} else {
	    set outstring "$outstring [format %9s NaN]"
	}
    }
    if {[check_epistasis]} {
	if {"-none" == $modelfile} {
	    set par_h2q [parameter h2qe1 start]
	} else {
	    set par_h2q [oldmodel $modelfile h2qe1]
	}

	if {[string compare [string toupper $par_h2q] NAN]} {
	    set outstring "$outstring [format %9.6f $par_h2q]"
	} else {
	    set outstring "$outstring [format %9s NaN]"
	}
    }
    if {"none" != $score} {
	if {[is_nan $score]} {
	    set outstring "$outstring [format %9s NaN]"
	} else {
	    set outstring "$outstring [format %9.6f $score]"
	}
    }
    if {"none" != $lcse} {
	if {[is_nan $lcse] || $lcse == 0} {
	    set outstring "$outstring [format %9s NaN]"
	} else {
	    set outstring "$outstring [format %9.6f $lcse]"
	}
    }

    if {$err == 1} {
	set outstring "$outstring BoundrErr"
    }
    if {$err == 2} {
	set outstring "$outstring ConvrgErr"
    }
    if {$err == 3} {
	set outstring "$outstring NoStdErr"
    }
    return $outstring
}

# solar::solartcl --
# Purpose:  Check SOLAR version compatibility of tcl file
#
# solar::solarmodel --
#
# Purpose:  Check SOLAR version compatibility of model
#
# solarmodel appears a the top of all new model files and identifies the
# model version.  If the version is incompatible with the current
# version, an error message is displayed.
#
# solartcl appears at the top of all upgraded script files.  SOLAR
# programmers are encoured to use solartcl as well.
#
# To upgrade solar models, use the "upgrade" command.
# -

proc solarmodel {version {beta ""}} {
# Currently there are no earlier versions except unidentified ones
    return "";
}

proc solartcl {version {beta ""}} {
# Currently there are no earlier versions except unidentified ones
    return "";
}

# solar::upgrade --
#
# Purpose:  Upgrade model files and scripts
#
# Usage:    upgrade modelname
#           upgrade scriptname.tcl
#
# Notes:   If successful, the new file will replace the original one.
#             The old file is saved with ".old" tacked on to the end of
#             the name (e.g. amodel.mod.old).
#
#          If an error is reported, the original file remains unchanged.
#
#          If the file is a model, the ".mod" extension is assumed even if
#             not specified.  Solar always tacks on ".mod" to the end of
#             model filenames.
#
#          If the file is a script, it must end with the ".tcl" extension,
#             and the extension must be specified in the command as shown.
#             Upgrade looks for this, and if found it assumes that a script
#             is being upgraded.
#
#          solartcl appears at the top of all upgraded script files.  SOLAR
#             programmers are encoured to use solartcl as well.
#
#
# -

proc upgrade {givenname} {
    set scriptmode 0
    set modelname $givenname
#
# Check for .tcl and .mod extensions
#
    set glength [string length $givenname]
    if {$glength > 4} {
	if {![string compare ".tcl" [string range $givenname \
		[expr $glength - 4] end]]} {
	    set scriptmode 1
	} elseif {[string compare ".mod" [string range $givenname \
		[expr $glength - 4] end]]} {
	    set modelname "$givenname.mod"
	}
    }
#
# Open model file and working output file
#
    set oldfile [open $modelname r]
    set newfile [open $modelname.new w]
#
# Skip over leading comments
#   (might be #!solar statement)
#
    while {[string length [set nextline [gets $oldfile]]]} {
	if {"#" != [string range $nextline 0 0]} {
	    break
	}
	puts $newfile $nextline
    }
#
# Output new version statement
#
    set newversion [lindex [solarversion] 0]
    set newbeta [lindex [solarversion] 1]
    if {$scriptmode} {
        puts $newfile "solartcl $newversion $newbeta"
    } else {
	puts $newfile "solarmodel $newversion $newbeta"
    }
#
# Set preliminary version (gets overwritten if solarmodel statment
#
    set oldversion 0
#
# Read and convert each line
#
    set parameter_count 0
    while {[string length $nextline]} {
	set token1 ""
	set token2 ""
	set token3 ""
	scan $nextline "%s %s %s" token1 token2 token3
#
# If this is a solarmodel or solartcl statement, record actual version
#
    if {"solarmodel" == $token1 || \
	    "solartcl" == $token1} {
	scan $token2 "%f" junk oldversion
#
# Convert phenfile line (That's going way back)
#
    } elseif {"phenfile" == $token1 || \
		"phenfile" == $token2} {
	    puts $newfile "phenotypes load token3"
#
# Convert old-style covariate line
#
	} elseif {"covariate" == $token1} {
	    set varname $token2
	    set xsex 0
	    set xexp ""
	    for {set i 2} {$i < [llength $nextline]} {incr i} {
		set modifier [lindex $nextline $i]
		if {"sex" == $modifier} {
		    set xsex 1
		} elseif {"exp" == $modifier} {
		    incr i
		    set xexp [lindex $nextline $i]
		}
	    }
	    if {0==[llength $xexp] || ![string compare $xexp 1]} {
		puts $newfile "covariate $varname"
		if {$xsex} {
		    puts $newfile "covariate $varname*sex"
		}
	    } elseif {![string compare $xexp 2]} {
		puts $newfile "covariate $varname^2"
		if {$xsex} {
		    puts $newfile "covariate $varname^2*sex"
		}
	    } elseif {![string compare $xexp 12]} {
		puts $newfile "covariate $varname"
		if {$xsex} {
		    puts $newfile "covariate $varname*sex"
		}
		puts $newfile "covariate $varname^2"
		if {$xsex} {
		    puts $newfile "covariate $varname^2*sex"
		}
	    } else {
		error "Invalid covariate statement: $nextline"
	    }
#
# Convert parameter names
#
	} elseif {"parameter" == $token1 && !$scriptmode} {
	    incr parameter_count
	    if {2 < $parameter_count} {
		set name $token2
		set firstch [string range $name 0 0]
#
#   Convert 2 to ^2
#
		if {"b" == $firstch || "m" == $firstch || "f" == $firstch} {
		    if {0==[string compare "2" \
			    [string range $name end end]]} {
			set pentul [expr [string length $name] - 2]
			set name "[string range $name 0 $pentul]^2"
		    }
		}
#
#   Convert 'm' and 'f' covariate prefixes
#
		if {"m" == $firstch} {
		    set name "b[string range $name 1 end]"
		} elseif {"f" == $firstch} {
		    set name "b[string range $name 1 end]*sex"
		}
		set nextline [lreplace $nextline 1 1 $name]
	    }
	    puts $newfile $nextline
#
# Check for 'multipoint -f' which is now 'multipoint -overwrite'
#
	} elseif {("multipoint" == $token1 || "scanloci" == $token1) && \
		-1 != [lsearch -exact $nextline "-f"]} {
	    set position [lsearch -exact $nextline "-f"]
	    set nextline [lreplace $nextline $position $position "-overwrite"]
	    set nextline [lreplace $nextline 0 0 "multipoint"]
	    puts $newfile $nextline
#
# Default: just write old line out
#
	} else {
	    puts $newfile $nextline
	}
	set nextline [gets $oldfile]
    }
    close $newfile
    close $oldfile
    eval file rename -force $modelname $modelname.old
    eval file rename -force $modelname.new $modelname
}

# solar::read_arglist --
#
# Purpose:  Read hyphenated optional arguments and argument-value pairs
#
# Usage:    read_arglist arglist [identifier_name value_var]+
#           value_var := varname | {statement block}
#
# Notes:
#          This provides a general way of handling argument lists of the
#          form:
#
#          [[arg] | [-identifier [value]]]+
#
#          Which is to say that there may be any number of "regular"
#          arguments and "hyphenated" arguments.  The "hyphenated"
#          arguments may be stand-alone or paired with values.  (Unlike
#          typical Unix syntax, stand-alone hyphenated arguments MAY NOT be
#          strung together, and hyphenated arguments with values must be
#          separated by space and not with some other character such as =).
#
#          The "regular" arguments (those not hyphenated or part of
#          a hyphenated pair) are put into a list which is returned by
#          this procedure.
#
#          Hyphenated arguments may either require following "value"
#          arguments or not allow them (in which case the hyphenated
#          argument acts like a switch).  Value arguments must be separated
#          from the hyphenated argument by a space (as is typical in Tcl).
#          For example
#
#              bar -height 1.5
#
#          There are two ways read_arglist can handle a hyphenated argument.
#
#          (1) The first, specified by the 'varname' expansion of value_var,
#          performs an assignment of the "value" argument to the caller's
#          varname variable.  For example:
#
#              read_arglist $args -height h
#
#          If $args contains "-height 1.5", then 1.5 will be assigned to the
#          caller's 'h' variable.  Note that this method always requires
#          a value argument and so does not work for switch arguments.
#
#          (2) The second, specified by the '{statement block}' expansion
#          of value_var executes an arbitrary set of expressions in
#          the caller's context.  This allows a simple switch or more
#          complex list-building.  The the statement block contains the
#          token VALUE, a value argument is required and the token
#          VALUE is replaced by the actual value argument.  Substitution
#          is performed only once and only for the first occurance of
#          VALUE.
#
#          A simple switch is implemented like this:
#
#              read_arglist $args -bottom {set bottom 1}
#
#          If $args contains "-bottom," bottom (in the caller's context) is
#          set to 1.  A value argument is neither required nor allowed.
#
#          A list-building argument is implemented like this:
#
#              read_arglist $args -include {lappend ilist VALUE}
#
#          If $args contains "-include foo" then "lappend ilist foo" is
#          executed in the caller's context.
#
#          NOTE that in the {statement block} form, the statement block
#          IS REQUIRED to have more than one list element.  A llength is
#          done to determine which form is being used.  Thus, you cannot
#          have:
#
#              read_arglist $args -exit {exit}  ;# *** BAD ***
#
#          but you could have
#
#              read_arglist $args -exit {eval exit}
#
#          If -* is used as an identifier_value, it matches any argument
#          in the argument list and causes that argument do be added to
#          the return list.  Normally -* should be the last identifier
#          value; all following identifier values will be ignored.
#          Also, the value_var (or statement block) following -* is never
#          assigned or executed and so can be anything.  This is intended
#          as an escape to permit only certain arguments to be processed
#          leaving other variables for processing by a later procedure.
#
#    More notes:
#
#          It is the responsibility of the caller to assign default
#          values before calling this procedure.
#
#          Hyphenated arguments may not have hyphenated strings for values.
#          However, hyphenated arguments may have negative numbers (e.g.
#          -1.2e5) for values.  If the value does not parse as an integer
#          or floating point number, it must not begin with hyphen.  If
#          the token following a hyphenated argument begins with - but is
#          not a number, it is considered to be another hyphenated argument
#          (which may cause the preceding argument to be flagged as having
#          a missing value).
#
#          Hyphenated argument names must not be numbers (integer or floating
#          point).  For example, you may not have "-1" or "-1.2e5" as a
#          hyphenated argument.
#          
#          Hyphenated arguments which do not match any of the templates
#          given raise the "Invalid argument %s".
#
#          The identifier matching rule is case insensitive.
#
# -

proc read_arglist {arg_list args} {
    set case_insensitive 1

    set pname [lindex [info level [expr [info level] - 1]] 0]
    set target_pairs $args
    set return_list {}
    for {set i 0} {$i < [llength $arg_list]} {incr i} {
	set identifier [lindex $arg_list $i]
	if {"-" != [string range $identifier 0 0]} {
	    lappend return_list $identifier
	    continue
	}
	if {$case_insensitive} {
	    set identifier [string tolower $identifier]
	}
	set found 0
	for {set j 0} {$j < [llength $target_pairs]} {incr j 2} {
	    set target [lindex $target_pairs $j]
	    if {"-*" == $target} {
		set found 1
		lappend return_list $identifier
		break
	    }
	    if {$case_insensitive} {
		set target [string tolower $target]
	    }
	    if {![string compare $target $identifier]} {
		set varname [lindex $target_pairs [expr $j + 1]]
		if {[llength $varname] > 1} {
#
# statement evaluation specified
#
		    if {-1 != [string first VALUE $varname]} {
#
# expression has VALUE substitution so value is required
#
			incr i
			if {[llength $arg_list] < [expr $i + 1]} {
			error \
		   "Missing value for argument $identifier to procedure $pname"
			}
			set value [lindex $arg_list $i]
			if {"-" == [string range $value 0 0] && \
			    ![is_float [lindex $value 0]]} {
			    error \
		   "Missing value for argument $identifier to procedure $pname"
			}
#
# perform ONE substitution
#
			set position [string first VALUE $varname]
			set length [string length $varname]
			set left [expr $position - 1]
			set right [expr $position + $length]
			set block \
"[string range $varname 0 $left]$value[string range $varname $right end]"
                    } else {
			set block $varname
		    }
		    uplevel $block
#
# "simple" assignment specified ; argument always required
#
		} else {
		    incr i
		    if {[llength $arg_list] < [expr $i + 1]} {
			error \
		   "Missing value for argument $identifier to procedure $pname"
		    }
		    set value [lindex $arg_list $i]
		    if {"-" == [string range $value 0 0] && \
			    ![is_float [lindex $value 0]]} {
			error \
		   "Missing value for argument $identifier to procedure $pname"
		    }
		    upvar $varname callervar
		    set callervar $value
		}
		set found 1
		break
	    }
	}
	if {!$found} {
	    error "Invalid argument [lindex $arg_list $i] to procedure $pname"
	}
    }
    return $return_list
}

proc test_read_arglist {args} {
    set cutoff 6
    set h2r_factor 1.1
    set restart 0
    set force 0
    set include_list {}

    read_arglist $args -cutoff cutoff -h2rf h2r_factor -restart {restart 1} \
	    -f {set force 1} \
	    -include {lappend include_list VALUE}

    puts "cutoff:     $cutoff"
    puts "h2r_factor: $h2r_factor"
    puts "restart:    $restart"
    puts "force:      $force"
    puts "include_list: $include_list"
}    

# solar::markertest --
#
# Purpose:  Test markerfile for discrepancies; list blankable ID's
#
# Usage:    markertest <markerfile> [<marker>]* [<option>]
#           <option> := -1 | -many | -ped | -fam <famfile> | -2
#
#           <markerfile> is either exact filename or pattern including *
#           <marker> (optional) is either exact name or pattern including *
#           If no markers are specified, all markers in markerfile are tested.
#           Each marker is tested individually.
#
#           Results are recorded in markertest.out in current directory.
#           During normal operation, many error and warning messages may
#           appear.  Ignore these messages until markertest has finished.
#
#           If no options are specified, a flexible procedure is used that
#           should work in nearly all cases.  It is the same as markertest -1
#           followed by markertest -many if necessary.  IMPORTANT: Read the 
#           following two paragraphs to understand how those options work.
#
#           -1     Blank one individual at a time.  Report all individual
#                  blankings (if any) that fix discrepancy.  If this succeeds
#                  only one of the reported individuals needs to be blanked
#                  and it is up to user to pick the best one.  However, this
#                  procedure is good (if it works) because it will list all
#                  the possibilities, and it is relatively fast.  But if it
#                  is necessary to blank more than one individual AT THE 
#                  SAME TIME, this procedure will fail, so it is frequently
#                  inadequate by itself.
#
#           -many  Blank the first individual, and, if that doesn't fix
#                  discrepancy, blank the second individual, and then the
#                  third, and so on, until the discrepancy is fixed.  Then, 
#                  unblank all the individuals that can be unblanked without
#                  a discrepancy returning.  The result is one set of
#                  individuals that ALL NEED TO BE BLANKED AT THE SAME TIME.
#                  It is not necessarily the only such set of individuals,
#                  or the smallest set.  This procedure should always succeed
#                  in finding a set of blankable individuals.  (This
#                  option used to be named -r.)
#
#           -ped    Rather than blanking only one ID at a time, blank
#                   whole "extended pedigrees" at a time.  Blankable
#                   pedigrees are identified by pedindex.out PEDNO
#                   (the index created by "load pedigree") and
#                   by the first ID found in that pedigree.  This procedure
#                   is the fastest, and is useful in identifying errant
#                   pedigrees, provided there is only one errant pedigree.
#
#           -fam    Rather than blanking only one ID at a time, blank
#                   nuclear families (or other groups) identified by
#                   "famno."  The "famfile" must contain records
#                   including "ID" (permanent ID) and "FAMNO" (other
#                   fields will be ignored).  There may be more than
#                   one record for each ID.  Records may not use
#                   "FAMID" to disambiguate ID's.
#
#           -2      Try blanking combinations of 2 ID's until one such pair
#                   of blankings fixes the discrepancy.  Because this is an
#                   N x N problem, it may take a VERY LONG TIME to finish, but
#                   if you are convinced there is one pair that needs to be
#                   blanked, this procedure will find it.
#
# Notes:    Pedigree file must already have been loaded
#
#           Markerfile must have ID and and marker(s) only.  Each marker is
#           analyzed separately.  Results for all markers are reported in
#           markertest.out, which is divided into one section for each
#           marker.
#
#           Output is written to markertest.out which is divided into one
#           section for each marker.
# -

# proc markertest is now a selector which selects the appropriate subroutines

proc markertest {args} {
    set markertest1 0
    set markertest2 0
    set markertestmany 0
    set markertestped 0
    set markertestfam 0
    set famnofile ""

# Clean out old markertest.out

    set ofile [open markertest.out w]
    close $ofile

# Parse arguments

    set moreargs [read_arglist $args \
	    -1 {set markertest1 1} \
	    -2 {set markertest2 1} \
	    -many {set markertestmany 1} \
	    -r {set markertestmany 1} \
	    -ped {set markertestped 1} \
	    -fam famnofile]

# If famnofile, ensure that it exists

    if {"" != $famnofile} {
	if {![file exists $famfilename]} {
	    error "File $famfilename not found"
	}
	set markertestfam 1
    }

# Ensure that more than one option hasn't been specified

    set optioncount [expr $markertest1 + $markertest2 + $markertestmany + \
	    $markertestped + $markertestfam]
    if {$optioncount > 1} {
	error "Only one option may be specified at a time or use default"
    }

# First non-option argument is markerfilename(s)

    if {0==[llength $moreargs]} {
	error "Usage: markertest <markerfile> \[<marker>\]* \[<option>\]"
    }
    set markerfiles [lindex $moreargs 0]
    if {-1 != [string first * $markerfiles]} {
	set markerfiles [glob $markerfiles]
    }
    if {{} == $markerfiles || ![file exists [lindex $markerfiles 0]]} {
	error "No such file(s): $markerfiles"
    }

# Following arguments, if any, are markername(s)

    set markernames [lrange $moreargs 1 end]

# MAIN LOOP (for each markerfile)

    set markers_tested 0
    set discrep_found 0
    set discrep_solved 0
    foreach markerfile $markerfiles {
	set mfile [solarfile open $markerfile]
	if {![solarfile $mfile test_name ID]} {
	    putsteer markertest.out \
	      "Error!  Markerfile $markerfile does not have ID or EGO field\n"
	    solarfile $markerfile close
	    continue
	}
	set mfnames [lrange [solarfile $mfile names] 1 end]

# Make a list of markernames to test from this markerfile

	if {{} == $markernames} {
	    set testmarkers $mfnames
	} else {
	    set testmarkers {}
	    foreach markername $markernames {
		foreach mfname $mfnames {
		    if {-1 == [string first * $markername]} {
			if {[string_imatch $mfname $markername]} {
			    lappend testmarkers $mfname
			}
		    } else {
			set mf [string tolower $mfname]
			set m [string tolower $markername]
			if {[string match $m $mf]} {
			    lappend testmarkers $mfname
			}
		    }
		}
	    }
	}
	putsteer markertest.out \
		"\nMarkers to test in file $markerfile:\n$testmarkers"

# Build temporary file(s) with just one marker

	foreach marker $testmarkers {
	    incr markers_tested
	    putsteer markertest.out \
"\n   **********   Testing $marker   ******************************\n"

            solarfile $mfile rewind
	    solarfile $mfile start_setup
	    solarfile $mfile setup ID
	    solarfile $mfile setup $marker
	    set ofile [open markertest.$marker.tmp w]
	    puts $ofile "ID,$marker"
	    while {{} != [set record [solarfile $mfile get]]} {
		puts $ofile "[lindex $record 0],[lindex $record 1]"
	    }
	    close $ofile

# See if there are discrepancies for this marker

	    if {[catch {load marker markertest.$marker.tmp}]} {
		error "Error loading markerfile generated for $marker"
	    }
	    if {![catch {marker discrep}]} {
		putsteer markertest.out \
			"\nNo discrepancies found for $marker\n"
	    } else {
		incr discrep_found
		putsteer markertest.out \
			"Analyzing discrepancies found for $marker...\n"

# Apply user-selected method

		if {$markertestfam} {
		  set status [markertest_fam markertest.$marker.tmp $famnofile]
		} elseif {$markertestmany} {
		   set status [markertest_r markertest.$marker.tmp]
		} elseif {$markertest2} {
		    set status [markertest_2 markertest.$marker.tmp]
		} elseif {$markertest1} {
		    set status [markertest_1 markertest.$marker.tmp]
		} elseif {$markertestped} {
		    set status [markertest_ped markertest.$marker.tmp]
		} else {

# Apply default method: first do markertest_1, if that fails, try _r

		    putsteer markertest.out \
			    "First, try blanking one person at a time...\n"
		    set status [markertest_1 markertest.$marker.tmp]
		    if {$status} {
			putsteer markertest.out \
				"\nCould not find any one-person solutions"
			putsteer markertest.out \
				"Searching for a many-person solution...\n"
			set status [markertest_r markertest.$marker.tmp]
		    }
		}
		if {$status} {
		    putsteer markertest.out \
	       "No solutions could be found for discrepancy in marker $marker"
		} else {
		    incr discrep_solved
		}
		file delete markertest.$marker.tmp
	    }
	}
	solarfile $mfile close
    }
    putsteer markertest.out \
"\n************************* Summary of Results ***************************\n"

    putsteer markertest.out \
	    "$markers_tested markers tested"
    putsteer markertest.out \
	    "$discrep_found discrepancies found"
    if {$discrep_found} {
	putsteer markertest.out "$discrep_solved discrepancies solved"
    }
    puts "\nResults written to file markertest.out"
}


#
# Subroutine to fix discrepancy by blanking one person at a time
#
proc markertest_1 {markerfilename} {

    set DEBUG 1

# Open markerfile and setup ID,MARKER

    set mfile [solarfile open $markerfilename]
    set names [solarfile $mfile names]
    solarfile $mfile start_setup
    solarfile $mfile setup ID
    set markername [lindex [solarfile $mfile names] 1]
    solarfile $mfile setup $markername

# Read marker file into memory

    set Id {}
    set Marker {}
    set count 0
    while {{} != [set record [solarfile $mfile get]]} {
	lappend Id [lindex $record 0]
	lappend Marker [lindex $record 1]
	incr count
    }
    solarfile $mfile close

    if {$DEBUG} {puts "Read $count records from $markerfilename"}

# MAIN LOOP

    set Bad_Id {}
    for {set i 0} {$i < $count} {incr i} {
	if {"" == [lindex $Marker $i]} {
	    continue
	} else {

# Write temporary marker file

	    if {$DEBUG} {puts "Blanking [lindex $Id $i]..."}

	    set tfile [open "markertest.tmp" w]
	    puts $tfile "id,$markername"
	    for {set j 0} {$j < $count} {incr j} {
		if {$j != $i} {
		    puts $tfile "[lindex $Id $j],[lindex $Marker $j]"
		} else {
		    puts $tfile "[lindex $Id $j],"
		}
	    }
	    close $tfile

# Load temporary marker file and check for discrepancies

	    load marker markertest.tmp
	    if {![catch {marker discrep}]} {

# Write suspect ID to file and add to list

		putsteer markertest.out \
	       "Blanking this one ID fixes marker $markername: [lindex $Id $i]"
		lappend Bad_Id [lindex $Id $i]
	    }
	}
    }

    file delete markertest.tmp

# See if we got anything

    if {0==[llength $Bad_Id]} {
	return 1
    }
    return 0

}


#
# Subroutine to fix discrepancy by blanking everyone until fixed
#
proc markertest_r {markerfilename} {

    set DEBUG 1

# Open markerfile and setup ID,MARKER

    set mfile [solarfile open $markerfilename]
    set names [solarfile $mfile names]
    solarfile $mfile start_setup
    solarfile $mfile setup ID
    set markername [lindex [solarfile $mfile names] 1]
    solarfile $mfile setup $markername

# Read marker file into memory

    set Id {}
    set Marker {}
    set count 0
    while {{} != [set record [solarfile $mfile get]]} {
	lappend Id [lindex $record 0]
	lappend Marker [lindex $record 1]
	incr count
    }
    solarfile $mfile close

    if {$DEBUG} {puts "Read $count records from $markerfilename"}

# MAIN LOOP 1
#   Remove each ID until problem goes away

    for {set i 0} {$i < $count - 1} {incr i} {
	if {"" == [lindex $Marker $i]} {
	    continue  ;# It's already blank, no need to test
	} else {

# Write temporary marker file

	    puts "Blanking [lindex $Id $i]..."

	    set tfile [open "markertest.tmp" w]
	    puts $tfile "id,$markername"
	    for {set j 0} {$j < $count} {incr j} {
		if {$j > $i} {
		    puts $tfile "[lindex $Id $j],[lindex $Marker $j]"
		} else {
		    puts $tfile "[lindex $Id $j],"
		}
	    }
	    close $tfile

# Load temporary marker file and check for discrepancies

	    load marker markertest.tmp
	    if {![catch {marker discrep}]} {

# This final straw fixed the problem.  Break here.
		
		break
	    }
	}

# Didn't fix problem.  Keep on going.

    }

# If we removed more than first ID (more than likely) we must continue

    set Bad_Index 0   ;# If $i==0, this is answer
    if {$i > 0} {
	set Bad_Index {}

# MAIN LOOP 2
#   Add back in all that can be added back in

	set last_removed $i
	for {set i 0} {$i <= $last_removed} {incr i} {
	    
# Skip over already blank markers

	    if {"" == [lindex $Marker $i]} {
		continue
	    }

# Write temporary marker file

	    puts "Unblanking [lindex $Id $i]..."

	    set tfile [open "markertest.tmp" w]
	    puts $tfile "id,$markername"
	    for {set j 0} {$j < $count} {incr j} {

		set removed_list {}

		if {"" == [lindex $Marker $j]} {
		    puts $tfile "[lindex $Id $j],"
		} else {
		    if {$j <= $i && -1 == [lsearch $Bad_Index $j]} {
			puts $tfile "[lindex $Id $j],[lindex $Marker $j]"
		    } elseif {$j > $last_removed} {
			puts $tfile "[lindex $Id $j],[lindex $Marker $j]"
		    } else {
			puts $tfile "[lindex $Id $j],"
			lappend removed_list $j
		    }
		}
	    }
	    close $tfile

	    if {$DEBUG} {puts "Removed: $removed_list"}

# Load temporary marker file and check for discrepancies

	    load marker markertest.tmp
	    if {[catch {marker discrep}]} {

# This individual brough problem back.  Add to Bad_Index list

		lappend Bad_Index $i
	    }
	}
    }

    if {0==[llength $Bad_Index]} {
	putsteer markertest.out \
	       "Error 1 in markertest algorithm; contact solar@txbiomedgenetics.org"
	exit
    }

    file delete markertest.tmp

    if {$count == [llength $Bad_Index]} {
	putsteer markertest.out \
		"All ID's must be blanked for marker $markername !!!"
	return 1
    }

# Write output file
    
    putsteer markertest.out \
	    "All of the following ID's must be blanked for marker $markername:"
    foreach i $Bad_Index {
	putsteer markertest.out \
	     "        [lindex $Id $i]"
    }
    putsteer markertest.out " "
    return 0
}


proc markertest_ped {args} {

    set DEBUG 1

# Open markerfile and setup ID,MARKER

    set mfile [solarfile open $markerfilename]
    set names [solarfile $mfile names]
    solarfile $mfile start_setup
    solarfile $mfile setup ID
    set markername [lindex [solarfile $mfile names] 1]
    solarfile $mfile setup $markername

# Read marker file into memory

    set Id {}
    set Marker {}
    set count 0
    while {{} != [set record [solarfile $mfile get]]} {
	lappend Id [lindex $record 0]
	lappend Marker [lindex $record 1]
	incr count
    }
    solarfile $mfile close

    if {$DEBUG} {puts "Read $count records from $markerfilename"}

# Read pedindex.out and setup Pedno and First_ID associative arrays

    set pfile [solarfile open pedindex.out]
    solarfile $pfile setup ID
    solarfile $pfile setup PEDNO
    set highest_pedno 0
    while {{} != [set record [solarfile $pfile get]]} {
	set this_pedno [lindex $record 1]
	set Pedno([lindex $record 0]) $this_pedno
	if {$highest_pedno < $this_pedno} {
	    set highest_pedno $this_pedno
	    set First_ID($this_pedno) [lindex $record 0]
	}
    }
    solarfile $pfile close

# MAIN LOOP

    set Bad_Pedno {}
    for {set i 1} {$i < $highest_pedno} {incr i} {

# Write temporary marker file

	if {$DEBUG} {puts "Blanking pedindex.out PEDNO $i ..."}
	
	set tfile [open "markertest.tmp" w]
	puts $tfile "id,$markername"
	for {set j 0} {$j < $count} {incr j} {
	    if {$Pedno([lindex $Id $j]) != $i} {
		puts $tfile "[lindex $Id $j],[lindex $Marker $j]"
	    } else {
		puts $tfile "[lindex $Id $j],"
	    }
	}
	close $tfile
	
# Load temporary marker file and check for discrepancies

	load marker markertest.tmp
	if {![catch {marker discrep}]} {

# Write suspect PEDNO to file and add to list

	    putsteer markertest.out \
 "Fixed by blanking pedindex PEDNO [format %s $i] with first ID: $First_ID($i)"
	    lappend Bad_Pedno $i
	}
    }

    file delete markertest.tmp

# See if we got anything

    if {0==[llength $Bad_Pedno]} {
	return 1
    }
    return 0
}


proc markertest_fam {markerfilename famfilename} {

    set DEBUG 1

# Open markerfile and setup ID,MARKER

    set mfile [solarfile open $markerfilename]
    set names [solarfile $mfile names]
    solarfile $mfile start_setup
    solarfile $mfile setup ID
    set markername [lindex [solarfile $mfile names] 1]
    solarfile $mfile setup $markername

# Read marker file into memory

    set Id {}
    set Marker {}
    set count 0
    while {{} != [set record [solarfile $mfile get]]} {
	lappend Id [lindex $record 0]
	lappend Marker [lindex $record 1]
	incr count
    }
    solarfile $mfile close

    if {$DEBUG} {puts "Read $count records from $markerfilename"}

# Read famfile and setup Members (array of lists of members for each famno)
# and Families_Found (list of famnos)

    set ffile [solarfile open $famfilename]
    solarfile $ffile setup ID
    solarfile $ffile setup FAMNO
    set Families_Found {}
    while {{} != [set record [solarfile $ffile get]]} {
	set this_id [lindex $record 0]
	set this_fam [lindex $record 1]
	if {[catch {set memlist $Members($this_fam)}]} {
	    set Members($this_fam) $this_id
	} else {
	    lappend memlist $this_id
	    set Members($this_fam) $memlist
	}
	if {-1==[lsearch $Families_Found $this_fam]} {
	    lappend Families_Found $this_fam
	}
    }
    solarfile $ffile close

# MAIN LOOP

    set Bad_Famno {}
    foreach famno $Families_Found {

# Write temporary marker file

	if {$DEBUG} {puts "Blanking members of Famno $famno..."}
	set memlist $Members($famno)
#	puts $memlist
	
	set tfile [open "markertest.tmp" w]
	puts $tfile "id,$markername"
	for {set j 0} {$j < $count} {incr j} {
	    set this_person [lindex $Id $j]
	    if {-1==[lsearch $memlist $this_person]} {
		puts $tfile "$this_person,[lindex $Marker $j]"
	    } else {
		puts $tfile "$this_person,"
	    }
	}
	close $tfile
	
# Load temporary marker file and check for discrepancies

	load marker markertest.tmp
	if {![catch {marker discrep}]} {

# Write suspect FAMNO to file and add to list

	    set ofile [open markertest.out a]
	    puts $ofile "FAMNO $famno"
	    close $ofile
	    lappend Bad_Famno $famno
	}
    }

    file delete markertest.tmp

# See if we got anything

    if {0==[llength $Bad_Famno]} {
	return 1
    }
    return 0
}


proc markertest_2 {markerfilename} {

    set DEBUG 1

# Open markerfile and setup ID,MARKER

    set mfile [solarfile open $markerfilename]
    set names [solarfile $mfile names]
    solarfile $mfile start_setup
    solarfile $mfile setup ID
    set markername [lindex [solarfile $mfile names] 1]
    solarfile $mfile setup $markername

# Read marker file into memory

    set Id {}
    set Marker {}
    set count 0
    while {{} != [set record [solarfile $mfile get]]} {
	lappend Id [lindex $record 0]
	lappend Marker [lindex $record 1]
	incr count
    }
    solarfile $mfile close

    if {$DEBUG} {puts "Read $count records from $markerfilename"}

# MAIN LOOP

    set Bad_Id {}
    for {set i 0} {$i < $count} {incr i} {
	for {set j [expr $i + 1]} {$j < $count} {incr j} {
	    if {$i == $j || "" == [lindex $Marker $i] || \
		    "" == [lindex $Marker $j]} {
		continue
	    } else {

# Write temporary marker file

		if {$DEBUG} {puts "Blanking [lindex $Id $i] and [lindex $Id $j]"}

		set tfile [open "markertest.tmp" w]
		puts $tfile "id,$markername"
		for {set out_i 0} {$out_i < $count} {incr out_i} {
		    if {$out_i != $i && $out_i != $j} {
			puts $tfile "[lindex $Id $out_i],[lindex $Marker $out_i]"
		    } else {
			puts $tfile "[lindex $Id $out_i],"
		    }
		}
		close $tfile

# Load temporary marker file and check for discrepancies

		load marker markertest.tmp
		if {![catch {marker discrep}]} {

# Write suspect ID to file and add to list
		    putsteer markertest.out \
       "Blanking this pair fixes $markername: [lindex $Id $i],[lindex $Id $j]"
		    lappend Bad_Id [list [lindex $Id $i] [lindex $Id $j]]
		}
	    }
	}
    }

# See if we got anything

    file delete markertest.tmp

    if {0==[llength $Bad_Id]} {
	return 1
    }
    return 0
}

# End of markertest and subroutines


#
# General SOLAR Utilities...little programs could be useful to many scripts
#
# (Unfortunately this section wasn't created earlier, so general SOLAR
#  utilities are actually scattered throughout the code.)
#

proc wordcap {word} {
    return [catenate [string toupper [string index $word 0]] \
		[string range $word 1 end]]
}

proc append_mod {given_name} {return [append_extension $given_name ".mod"]}

proc append_extension {given_name extension} {
    if {-1 == [string first $extension $given_name]} {
	return [format "%s%s" $given_name $extension]
    } else {
	return $given_name
    }
}
	
# solar::if_parameter_exists
#
# Purpose:  Check if a parameter exists without creating it
#
# Usage:    if_parameter_exists <parameter_name>
#
#           Returns 1 if parameter exists, 0 otherwise.
#
# Notes:    This is used in scripts in a "if" statement.  For example:
#
#           if {[if_parameter_exists h2q1]} {
#               constraint e2 + h2r + h2q1 = 0
#           }
#
# -

proc if_parameter_exists {pname} {
    if {[catch {parameter $pname start}]} {
	return 0
    }
    return 1
}


# solar::startclock
# solar::stopclock
#
# Purpose: simple local timer (elapsed time)
#
# Usage: startclock;<timed command>;stopclock
#
# See Also: timediff
#-

proc startclock {} {
    global SOLAR_startclock
    set SOLAR_startclock [exec date]
    puts $SOLAR_startclock
    return $SOLAR_startclock
}

proc stopclock {} {
    global SOLAR_startclock
    set outstring "[timediff $SOLAR_startclock [exec date]] seconds"
    puts $outstring
    return $outstring
}

# solar::timediff --
#
# Purpose:  Calculate seconds between two system time strings
#
# Usage: timediff <start-time> <end-time>
#
# See Also: startclock, stopclock
#
# set starttime [exec date]
#  ... procedure to be timed
# set endtime [exec date]
# return "seconds: [timediff $starttime $endtime]"
#

proc timediff {first last} {
    set firstsub [findcolons $first]
    set lastsub [findcolons $last]
    set flist [split $firstsub :]
    set llist [split $lastsub :]
    if {3 != [llength $flist]} {
	error "invalid time specifier: $firstsub"
    }
    if {3 != [llength $llist]} {
	error "invalid time specifier $lastsub"
    }
    set fseconds [makenumber [lindex $flist 2]]
    set lseconds [makenumber [lindex $llist 2]]

    set fseconds [expr $fseconds + 60 * [makenumber [lindex $flist 1]]]
    set lseconds [expr $lseconds + 60 * [makenumber [lindex $llist 1]]]

    set fseconds [expr $fseconds + 3600 * [makenumber [lindex $flist 0]]]
    set lseconds [expr $lseconds + 3600 * [makenumber [lindex $llist 0]]]

    if {$lseconds < $fseconds} {
	set lseconds [expr $lseconds + 3600*24]
    }

    set total [expr $lseconds - $fseconds]

    return $total
}

proc findcolons {alist} {
    foreach term $alist {
	if {-1!=[string first : $term]} {
	    return $term
	}
    }
    error "No time found in $alist"
}

proc makenumber {string} {
    if {0==[string first 0 $string]} {
	set string [string range $string 1 end]
    }
    return $string
}

# solar::d2e2 --
# solar::d2e --
#
# Purpose: convert Fortran D style exponents to E form
#
# Usage: d2e <inputfilename> <outputfilename>
#        d2e2 <inputfilename> <outputfilename>
#
# d2e2 starts with line 2, so as not to disturb D's in the header line of
# comma delimited files.  d2e is more suitable for pedsys files.
#
# SOLAR now understands Fortran D style in phenotypes files in most
# circumstances anyway, so this conversion is not generally needed.  You
# will know you need this if you see error messages.
# -

proc d2e2 {infilename outfilename} {

    set infile [open $infilename]
    set outfile [open $outfilename w]

    set linecount 0
    while {-1 != [gets $infile line]} {
	set outline $line
	if {$linecount} {
	    catch {
		set outline [regsub -all D $line E]
	    }
	}
	incr linecount
	puts $outfile $outline
    }
    close $infile
    close $outfile
}

proc d2e {infilename outfilename} {

    set infile [open $infilename]
    set outfile [open $outfilename w]

    while {-1 != [gets $infile line]} {
	set outline $line
	catch {
	    set outline [regsub -all D $line E]
	}
	puts $outfile $outline
    }
    close $infile
    close $outfile
}




# solar::read_output  --
#
# Purpose:  Read variable statistics from maximization output file
#
# Usage:    read_output <outfile> <varname> [-mean | -min | -max | -std]
#
#           -mean Get variable mean (default)
#           -min  Get variable minimum
#           -max  Get variable maximum
#           -std  Get variable standard deviation
#           -d    1 if discrete, 0 otherwise
#
# Note:     If outfile is not full pathname, current trait/outdir is assumed.
#           Statistics pertain to actual sample used in maximization.
#
# Example:  read_output null1.out q4 -std
# -

proc read_output {outfile varname {getfield -mean}} {
    return [getvar $getfield $outfile $varname]
}


proc getvar {mean_min_max_std outfile varname} {

# Set index for kind of value required

    if {![string compare $mean_min_max_std -mean]} {
	set get_index 1
    } elseif {![string compare $mean_min_max_std -min]} {
	set get_index 3
    } elseif {![string compare $mean_min_max_std -max]} {
	set get_index 4
    } elseif {![string compare $mean_min_max_std -std]} {
	set get_index 2
    } elseif {![string compare $mean_min_max_std -d]} {
	set get_index 0
    } else {
	error "getvar needs -mean -min -max -std -d"
    }

# Expand pathname if not specified

    if {-1 == [string first / $outfile]} {
	set outfile [full_filename $outfile]
    }

# Check for file

    if {![file exists $outfile]} {
	error "No such file $outfile"
    }

# Normalize varname to lowercase

    set varname [string tolower $varname]

# Scan output file for variable statistics

    set ofile [open $outfile r]
    set found_variables {}
    set found_variable_means 0
    while {-1 != [gets $ofile line]} {
	if {-1 != [string first "Descriptive Statistics for the Variables" \
		$line]} {
	    gets $ofile line      ;# blank
	    gets $ofile line      ;# headings
	    set found_variable_means 1
	    break
	}
    }
    if {!$found_variable_means} {
	close $ofile
	error "Did not find variable means in output file $outfile"
    }

# Scan for the variable we want (or variable* because * means discrete)

    while {-1 != [gets $ofile line]} {
	if {6 != [llength $line]} {break}
	set thisname [string tolower [lindex $line 0]]
	if {![string compare $varname $thisname] || \
		![string compare $varname* $thisname]} {

# return the mean value

	    close $ofile
	    if {$get_index} {
		return [lindex $line $get_index]
	    } else {
		if {[string index $thisname end] != "*"} {
		    return 0
		} else {
		    return 1
		}
	    }
	}
    }
    close $ofile
    error "Did not find variable $varname in output file $outfile"
}
    
# solar::selectrecords --
#
# Purpose:  Select records from a file and copy them to a new file
#
# Usage:    selectrecords <infile> [<outfile>] [{<conditions>}]*
#
#           If not specified, <outfile> defaults to selectrecords.out
#
#           Each <condition> is a Tcl conditional expression which includes
#           field names in the file preceded by dollar sign $.
#           Field names are case insensitive (you need not match
#           capitalization used in file itself).  Each condition
#           must be enclosed in curly braces and spaced from other conditions
#           if any.  
#
#           Conditions may also include actual Tcl variables, preceded by $$
#           Tcl variables are Case Sensitive.
#
#           Simple examples are shown, but any valid Tcl expression operators
#           and functions may be used, and expressions may be arbitrarily
#           complex...they are evaluated by the Tcl expression parser, with
#           the exception of special pre-substitution of $$ variables.
#           Internally, $$ variables are upvar'd into a local variables having
#           leading capital S.
#
#           If a condition includes a non-existant field, it will never be
#           satisfied, producing an empty result file.  (In future, error
#           detection may be added.)  If a condition includes a undefined $$
#           tcl variable, an error will result.
#
#           Input file may be either PEDSYS or Comma Delimited format.
#           Output file is comma delimited.
#
#           If the first condition does not include any dollar signs,
#           it must include spaces (for example, {1 == 1}).  No such
#           requirement is made for subsequent conditions.  It seems pointless
#           to have condition without dollar signs anyway; if no condition
#           is given you get all records (the "null condition" is always true).
#               
# Example:  selectrecords phen.dat out.dat {$bmi > 0.3} {$famid == 10}
#
#           for {set F 1} {$F < 100} {incr F} {
#               selectrecords phen.dat out$F.dat {$bmi > 0.3} {$famid == $$F}
#           }
#
# Note:     Records are "selected" when they match ALL conditions given (unless
#           condition includes a non-existing field or has other error).
#
# -

proc selectrecords {args} {
#
# one of the tricks here is that file fields ultimately become
# variables.  This is fine unless the user specifies a variable that
# doesn't exist as a field, then it *may* inherit the value of a variable
# within this procedure (!).  To prevent this, all local variables have
# capitalization, never starting with capital S (used for Tcl variables).
#
    set inConditions {}
    set Files {}

# we used to test for presence of spaces, just to determine whether 2nd
# argument was filename or not.  Problem is, we can't test directly for
# braces.  But we can test for dollar sign, which would also be proof that
# argument was enclosed in braces (otherwise any dollar signs would be
# pre-applied in caller's context and not visible here).
#
# Now we permit either test, so spaces are not required except in unusual
# constant case like {1 == 1}, and only for first condition, so now it is
# highly unlikely for non-documentation-reading user (like most) to make error.
#
    if {[llength $args] == 0} {
	error "usage: selectrecords <input> [<output>] [<conditions>]*"
    }

    for {set I 0} {$I < [llength $args]} {incr I} {
	set Arg [lindex $args $I]
	if {$I == 0} {
	    lappend Files $Arg

	} elseif {$I == 1} {
	    if {[llength $Arg] > 1 || -1 != [string first "\$" $Arg]} {
		lappend inConditions $Arg
	    } else {
		lappend Files $Arg
	    }
	} else {
	    lappend inConditions $Arg
	}
    }

    set Conditions [string tolower $inConditions]

    set inFilename [lindex $Files 0]
    set outFilename selectrecords.out
    if {2<=[llength $Files]} {
	set outFilename [lindex $Files 1]
    }

    set iFile [tablefile open $inFilename]
    set Names [string tolower [tablefile $iFile names]]
    set oFile [open $outFilename w]

    tablefile $iFile start_setup
    set Header ""
    set Name_count 0
    foreach Name $Names {
	if {![unique $Name $Names]} {
	    if {[string tolower $Name] != "blank"} {
		puts "field Name $Name not included because it is not unique"
	    }
	} else {
	    tablefile $iFile setup $Name
	    incr Name_count
	    if {"" == $Header} {
		set Header $Name
	    } else {
		set Header "$Header,$Name"
	    }
	}
    }
#
# This needs more work, intended to
# give warning if Conditions use non-existing fieldName
#
#    set testc $Conditions
#    foreach icon $testc {
#	set con $icon
#	while {1} {
#	    set fdd [string first "\$" $con]
#	    if {$fdd < 0} break
#	    set termstart [expr $fdd + 1]
#	    set termend [expr [string wordend $con $termstart] - 1]
#	    set Name [string range $con $termstart $termend]
#	    puts "fieldName is $Name"
#	    if {-1 == [lsearch $Names $Name]} {
#		puts "warning: $Name not present in file"
#		puts "this condition will be ignored: $icon"
#		break
#	    }
#	    set con [string range $con [expr $termend + 1] end]
#	}
#    }
	

# permit Tcl variables using $$
# insert modified case-sensitive version back into case lowered string

    set Debugvar 0
    set Varlist {}
    while {1} {
	set fDD [string first "\$\$" $inConditions]
	if {$fDD < 0} break
	set Termstart [expr $fDD + 2]
	set Termend [expr [string wordend $inConditions $Termstart] - 1]
	set Name [string range $inConditions $Termstart $Termend]
	if {$Debugvar} {puts "variable Name is $Name"}
	if {-1 == [lsearch $Varlist $Name]} {
	    lappend Varlist $Name
	    if {[catch {upvar $Name S$Name}]} {
		error "Missing Tcl variable: $Name"
	    }
	    if {[catch {eval set fooBAR \$S$Name}]} {
		error "Missing tcl variable: $Name"
	    }
	    set preConditions [string range $Conditions 0 \
				   [expr $fDD - 1]]
	    set postConditions [string range $Conditions \
				    [expr $Termend + 1] end]
	    set Conditions "$preConditions\$S$Name$postConditions"
	    set preinConditions [string range $inConditions 0 \
				     [expr $fDD - 1]]
	    set postinConditions [string range $inConditions \
				      [expr $Termend + 1] end]
	    set inConditions "$preinConditions\$S$Name$postinConditions"
	} else {
	    error "did find Name"
	}
    }
    if {$Debugvar} {puts "Conditions is $Conditions"}
    if {$Debugvar} {puts "inConditions is $inConditions"}

    puts $oFile $Header

    set Written 0
    while {{} != [set Line [tablefile $iFile get]]} {
	for {set i 0} {$i < $Name_count} {incr i} {
	    set Name [lindex $Names $i]
	    set Value [lindex $Line $i]
	    eval set $Name \$Value
	}
	set tEST 1
	foreach CoN $Conditions {
#	    puts "CoN is $CoN"
	    set test 0
	    set foOBAR [catch {eval set tEST \[expr $CoN \] }]
#	    puts "foOBAR is $foOBAR"
#	    puts "tEST is $tEST"
	    if {$foOBAR || !$tEST} {
		set tEST 0
		break
	    }
	}
	if {$tEST} {
	    incr Written
	    puts $oFile [join $Line ,]
	}
    }

    tablefile $iFile close
    close $oFile

    return "$Written records written"
}

# solar::ped2csv
#
# Purpose:  Convert Pedsys format file to comma delimited format
#
# Usage:    ped2csv <pedfilename> [<outfilename>]
#
#           If <outfile> is not specified, filename is <pedfile>.csv
#
# Notes:    BLANK fields are removed.  Duplicate field names will cause an
#           error.
#
#           This command uses the "selectrecords" command, which makes it
#           very easy: "selectrecords <pedfilename> <outfilename>".  Since
#           no condition is specified, all records are selected, and since
#           selectrecords uses the tablefile command, it can read pedsys files.
# -

proc ped2csv {args} {
    set nargs [llength $args]
    set inname [lindex $args 0]

    if {$nargs == 0 || $nargs > 2} {
	error "invalid arguments to ped2csv"
    }
    if {$nargs == 1} {
	set outname [lindex $args 0].csv
    } else {
	set outname [lindex $args 1]
    }
    return [selectrecords $inname $outname]
}


# solar::selectfields --
#
# Purpose:  Select fields (columns) from data file(s) and copy to a new file
#
# Usage:    selectfields [-noid] [<infile>]* [.] [-np] [<field-name>]* 
#                        [-o <outfile>]  [-sample] [-list filename] [-noid]
#                                        
#           A optional period (aka dot) ends the list of filenames and starts
#           the list of field names.  If there is no dot, the first argument
#           is assumed to be the one and only data filename.  The currently
#           loaded phenotypes files are automatically included at the end of
#           the list of files.  If nothing precedes the dot, only the
#           phenotypes files are used.  Fields found in multiple files default
#           to the first file in which they are found, however a warning is
#           given when this happens.  The -np argument forces the loaded
#           phenotypes files to be ignored.  The -sample argument forces
#           only the inclusion of individuals having all field values
#           defined.  Otherwise, a record is written for every ID encountered
#           in the file(s) from which data is read, however one or more
#           data value(s) might be blank.  
#
#           -list filename   Use all the field names in this file, listed
#                            one per line.  These are appended to the list
#                            of field names given in the command line, if
#                            any.
#
#           If the -noid switch is given, the old version of selectfiles
#           is used.  This takes one and only one <infile> followed by a
#           list of fieldnames, with no dot in between.  The only other
#           option allowed is -o.  No ID field is required in the input
#           file, and no ID field is written unless included in the list
#           of fieldnames.  The loaded phenotypes file is not used unless
#           that is the one file named.
#
#           If not specified, <outfile> defaults to selectfields.out
#
#           <field-names> follow rules for phenotypes files and are also
#             affected by field command specifications.  For example,
#             if you specify "ID" as field name, this would also match a
#             field name "EGO" in the file.
#
#           Input file may be either PEDSYS or Comma Delimited format.
#           Output file is comma delimited.
#
# Example:  selectfields phen.dat out.dat ID AGE -o age.dat
#
# -

proc selectfields {args} {

    set outfilename selectfields.out
    set loadedfiles [phenotypes -files]
    set sample 0
    set listfile ""
    set noid 0
    set phenfiles ""

    set argsonly [read_arglist $args -out outfilename -o outfilename \
		      -list listfile -noid {set noid 1} \
		      -np {set loadedfiles ""} -sample {set sample 1}]

    if {$noid} {
	return [eval oldselectfields $args]
    }

    if {-1 != [set dashpos [lsearch $argsonly "."]]} {
	set phenfiles [lrange $argsonly 0 [expr $dashpos - 1]]
	set fields [lrange $argsonly [expr $dashpos + 1] end]
    } else {
	set phenfiles [lindex $argsonly 0]
	set fields [lrange $argsonly 1 end]
    }
    foreach loadedfile $loadedfiles {
	setappend phenfiles $loadedfile
    }
    if {[llength $phenfiles] < 1} {
	error "selectfields: No input files selected"
    }
    if {"" != $listfile} {
	set listfilelist [listfile $listfile]
	set fields [concat $fields $listfilelist]
    }

    if {[llength $fields] < 1} {
	error "selectfields: No fields selected"
    }

# remove redundant files phenfiles -> usefiles
# open all phen files as phenf(i) for field scanning

    set usefiles {}
    set j 0
    for {set i 0} {$i < [llength $phenfiles]} {incr i} {
	set preceding [lrange $phenfiles 0 [expr $i - 1]]
	set this [lindex $phenfiles $i]
	if {-1 != [lsearch $preceding $this]} {
	    puts "Warning: file $this listed twice"
	} else {
	    set phenf($j) [solarfile open $this]
	    set file_needed($j) 0
	    lappend usefiles $this 
	    incr j
	}
    }

# Fill Fields array for each file used
# set file_needed

    catch {unset Fields}
    catch {unset Data}
    for {set i 0} {$i < [llength $fields]} {incr i} {
	set field [lindex $fields $i]
	set found ""
	set reported_dup 0
	for {set j 0} {$j < [llength $usefiles]} {incr j} {
	    if {[solarfile $phenf($j) test_name $field]} {
		set filename [lindex $usefiles $j]
		if {$found==""} {
		    set found $filename
		    set file_needed($j) 1
		    if {[catch {set fields_here $Fields($filename)}]} {
			set fields_here ""
		    }
		    lappend fields_here $field
		    set Fields($filename) $fields_here
		} else {
		    if {!$reported_dup} {
			set reported_dup 1
			puts "Warning.  Using $field from file $found"
		    }
		        puts "          Ignoring $field in file $filename"
		}
	    }
	}
	if {$found==""} {
	    for {set j 0} {$j < [llength $usefiles]} {incr j} {
		solarfile $phenf($j) close
	    }
	    error "selectfields: Could not find field $field in any of the files:\n$usefiles"
	}
    }

# Close files not needed
# or Read data from each file into Data arrays

    set ids {}
    for {set j 0} {$j < [llength $usefiles]} {incr j} {
	if {!$file_needed($j)} {
	    solarfile $phenf($j) close
	} else {
	    set filename [lindex $usefiles $j]
#	    puts "Reading $filename"
	    set fields_here $Fields($filename)
	    solarfile $phenf($j) start_setup
	    solarfile $phenf($j) setup ID
	    foreach field_here $fields_here {
		solarfile $phenf($j) setup $field_here
	    }
	    set linecount 0
	    while {{} != [set record [solarfile $phenf($j) get]]} {
		incr linecount
		set id [lindex $record 0]
		if {-1 == [lsearch $ids $id]} {
		    lappend ids $id
		}
		set fieldno 0
		foreach field_here $fields_here {
		    incr fieldno ;# starts at 1
		    set Data($field_here,$id) \
			[lindex $record $fieldno]
		}
	    }
	}
    }

# Data is read, now close remaining input files

    for {set j 0} {$j < [llength $usefiles]} {incr j} {
	if {$file_needed($j)} {
	    solarfile $phenf($j) close
	}
    }

# Output 1 record for any ID found

    set written 0
    set outfile [open $outfilename w]
    puts $outfile ID,[join $fields ,]
    foreach id $ids {
	set record "$id"
	set skip 0
	foreach field $fields {
	    set datum ""
	    catch {set datum $Data($field,$id)}
	    if {$datum == "" && $sample} {
		set skip 1
	    }
	    set record "$record,$datum"
	}
	if {$skip} {continue}
	puts $outfile $record
	incr written
    }
    close $outfile
    return "$written records written OK"
}


proc oldselectfields {args} {

    set outfilename selectfields.out
    set argsonly [read_arglist $args -out outfilename -o outfilename \
		      -noid {set foobar 0} ]
    if {[llength $argsonly] < 2} {
	error "selectfields requires <infile> and <field-name>"
    }
    set inFilename [lindex $argsonly 0]
    set columns [lrange $argsonly 1 end]

    if {![file exists $inFilename]} {
	error "selectfields: file not found: $inFilename"
    }
    set tfile [solarfile open $inFilename]
    solarfile $tfile start_setup
    set header ""
    foreach col $columns {
	set header "$header,$col"
	if {![solarfile $tfile test_name $col]} {
	    solarfile $tfile close
	    error "selectfields: Missing field $col"
	}
	solarfile $tfile setup $col
    }
    set outfile [open $outfilename w]
    set header [string range $header 1 end]
    puts $outfile $header
    set count 0
    while {{} != [set line [solarfile $tfile get]]} {
	set outline [join $line ,]
	puts $outfile $outline
	incr count
    }
    close $outfile
    solarfile $tfile close
    return "$count records written"
}

# solar::joinfiles --
#
# Purpose:  Join files horizontally based on ID's
#
# Usage:    joinfiles [-all] [<filename>]* [-out <filename>] [-key <keylist>]
#                     -list <filename> -chunk <chunksize>
#
#           -out <filename>  Write joined records to this file (default is
#                            joinfiles.out in the working directory)
#           -key <keylist>   Base join on these key(s) (default is ID or EGO,
#                            and also FAMID if FAMID is present in all files)
#           -all             Filenames may be patterns with wildcards
#                            (including *  to match any sequence of characters
#                            and ? to match any one character) and/or names of
#                            directories whose files will be included.
#                            (Files in subdirectories are not included.)
#                            When using -all, no system limit on open files
#                            is applicable.
#            -list <filename> Include all files listed in <filename>, which
#                             has one filename in each line, which may be
#                             a pattern with wildcards.  Only one -list
#                             may be used.  When using -list, no system limit
#                             on open files is applicable.
#             -chunk <chunksize>  The chunk size used in joining files under
#                                 -all and -list options.  By joining only one
#                                 chunk of files at a time, the system limit
#                                 on open files is bypassed.  The default is
#                                 100.
#
#           Some additional esoteric options are described below in Note 7.
#
# Notes:
#
# 1)  Each file may either be Comma Delimited or Pedsys, and sucessive files
#     may use different formats.
#
# 2)  The output file will be Comma Delimited, thus this command also serves
#     to translate one or more Pedsys files to Comma Delimited format.
#
# 3)  Any field mapping of ID and FAMID to some other name through the 
#     "field" command will be applied if the keys are defaulted.  Key
#     matching is case insensitive, so the key might be "ID" in one file
#     and "id" in the next.
#
# 4)  Records will be output for every ID encountered regardless of whether
#     that ID is found in all files.
#
# 5)  If keys are specified, you'd better know what you are doing.
#     No field name mapping or testing of whether FAMID is required
#     will be done.  However, whether defaulted or not, the availability
#     of keys in every file will be tested.
#
# 6)  If the same filename is repeated in the list of files to join, the
#     repeats are ignored (for technical reasons).  If you must join the
#     same file to itself for some legitimate reason (???), copy
#     to a another filename first.
#
# 7)  If the same field name(s), other than the key(s), are found in more
#     than one file, the default action is to rename them in the output
#     file in a way so as to be unique.  The following format is used:
#
#     <field name>.<filename>[.<number>]
#
#     If adding the filename makes the field name unique, that is all that
#     is done, which makes for a nice new name.  For example:
#
#         q4.qaw10.phen    (phenotype q4 in file gaw10.phen)
#
#     Otherwise, <number> is applied, using the first number (starting
#     from 2 and ending with 30002) that makes the field name unique.
#     Unless there are more than 30000 matching field names, this will
#     guarantee that a unique name will be found and  used.  Also, 
#     with reasonably short names, it is likely that the resulting name
#     will be unique within 18 characters, which is currently required
#     for trait and covariate names.  However, uniqueness within 18 
#     characters is not guaranteed as that would require ugly renaming
#     and it's quite possible the 18 character limit may be removed
#     eventually anyway.  Uniqueness testing is case insensitive.
#
#     There are two other optional ways of handling field names which are
#     not unique.  These option specifiers may be used anywhere after the
#     command name but apply globally to all files.
#
#           -uniqueonly    Remove fields which are not unique among files
#                          (except keys).
#
#           -norename      Don't rename fields that are not unique, just
#                          include them.  (Note: If this option is applied,
#                          the resulting file may cause problems with
#                          various SOLAR commands.  For example, the
#                          "residual" command won't like it even if the
#                          duplicated field is NOT used as a trait or
#                          covariate.)
#
# 8)  If the same fieldname is repeated in one file, that field is
#     not included in the output.  (Such fields could not be selected
#     as traits or covariates in SOLAR anyway.)  This typically occurs
#     when there is a field named BLANK to separate columns in a Pedsys
#     file.  Also, fields with the "null" name (zero characters or all
#     blanks) are not included.
#-

proc joinfiles {args} {
#
# Define variables and defaults
#
    set debug 0                      ;# local debugging
    set keylist {}                   ;# list of keys (see note below)
    set allnames {}                  ;# all fieldnames
    set oname joinfiles.out          ;# output filename
    set default_keys 0               ;# default key name rules
    set xfile tablefile              ;# tablefile or solarfile
    set make_unique rename           ;# rename, only, norename
    set listfile ""
    set many ""
    set chunksize 100
#
# Note: Herein I use "key" to refer to the name(s) of field indentifiers which
#       must match in order to "join" records.  I use "id" as the name of
#       the "data" in a particular "key" field or fields.  However, user is
#       permitted to use the name "-id" to refer to "key" since "id" is
#       otherwise meaningless at user level.
#
# Read arguments
#
    set nuinames [read_arglist $args \
		    -output oname -out oname -o oname \
		    -ids keylist -id keylist \
		    -keys keylist -key keylist \
		    -uniqueonly {set make_unique only} \
		    -norename {set make_unique norename} \
		    -all {set many many} \
		    -list listfile \
		    -chunk chunksize \
		    -clump chunksize \
		   ]
    file delete $oname

    if {0 == [llength $keylist]} {
	set keylist {id famid}
	set default_keys 1
	set xfile solarfile
    }

#
# Remove non-unique files 
#
    set inames {}
    foreach iname $nuinames {
	setappend inames $iname
    }
#
# Handle -all and -list options
# These options work by re-calling joinfiles
#
    if {"" != $listfile || "" != $many} {
	set patternglob ""
	if {"" != $inames} {
	    foreach iname $inames {
		set pnames [glob -nocomplain $iname]
		if {{} == $pnames} {
		    error "joinfiles: File(s) $iname not found"
		}
		set patternglob [concat $patternglob $pnames]
	    }
	}
	if {"" != $listfile} {
	    set biglist [listfile $listfile]
	    foreach line $biglist {
		set pnames [glob -nocomplain $line]
		if {{} == $pnames} {
		    error "joinfiles: Listed file(s) not found: $line"
		}
		set patternglob [concat $patternglob $pnames]
	    }
	}
	if {"" == $patternglob} {
	    error "joinfiles: No files found"
	}

	set options [read_arglist $args -list ignore -all ignore \
			 -o ignore -out ignore -output ignore \
			 -* foo]
	foreach in $inames {
	    set options [remove_from_list $options $in]
	}
	if {"" != $options} {
	    puts "joinfile options are $options"
	}
	
	set j1 [file dirname $oname]/solar.joinfiles.alllist.tmp
	set j2 [file dirname $oname]/solar.joinfiles.alllist.temp
	file delete -force $j1
	file delete -force $j2
	set first 1
	set finalglob {}
	foreach patordir $patternglob {
	    if {[file isdirectory $patordir]} {
		set pat [glob -nocomplain $patordir/*]
		if {"" == $pat} {
		    puts "joinfiles: Warning.  Directory $patordir is empty."
		    continue
		}
	    } else {
		set pat $patordir
	    }
	    set finalglob [concat $finalglob $pat]
	}
	set finalset {}
	foreach ele $finalglob {
	    if {[file isdirectory $ele]} {
		continue
	    }
	    if {-1 == [lsearch -exact $finalset $ele]} {
		lappend finalset $ele
	    } else {
		puts "joinfiles: Warning.  File will only be included once:\n$ele\n"
	    }
	}
	if {{} == $finalset} {
	    error "joinfiles: Error.  No files found"
	}
	puts "Joining files $finalset\n"
	set chunk_include [expr $chunksize - 1]
	set first 1
	while {{} != $finalset} {
	    set thisbatch [lrange $finalset 0 $chunk_include]
	    set finalset [lrange $finalset $chunksize end]
	    foreach rfile $thisbatch {
		puts "Joining $rfile"
	    }
	    if {$first} {
		set returnout [eval joinfiles $thisbatch -o $j2 $options]
		set first 0
	    } else {
		set returnout [eval joinfiles $j2 $thisbatch -o $j1 $options]
		file rename -force $j1 $j2
	    }
	}
	file rename -force $j2 $oname
	set returnout "[lrange $returnout 0 end-1] $oname"
	file delete -force $j1
	file delete -force $j2
	return $returnout
    }
#
# Open each file as a tablefile into array tfile(iname)
# Get list of field names: allnames
# and per-file list: fnames(iname)
#    
    foreach iname $inames {
	set tf [set tfile($iname) [eval $xfile open $iname]]

# Get names, removing repeats and nulls

	set goodnames {}
	set tempnames [eval $xfile $tf names]
	set badnames {}  ;# already seen twice, don't try adding again
	foreach tempname $tempnames {
	    if {{} != $tempname} {
		if {-1 == [lsearch -exact [string tolower $badnames] \
			       [string tolower $tempname]]} {
		    if {-1 == [lsearch -exact [string tolower $goodnames] \
				   [string tolower $tempname]]} {
			lappend goodnames $tempname
		    } else {
			set goodnames [remlist $goodnames $tempname]
			lappend badnames $tempname
		    }
		}
	    }
	}
	set allnames [concat $allnames [set fnames($iname) $goodnames]]
	if {$default_keys} {
	    if {![eval $xfile $tf test_name famid]} {
		set keylist id
	    }
	}
	foreach key $keylist {
	    if {![eval $xfile $tf test_name $key]} {
		foreach jname $inames {
		    catch {eval $xfile $tfile($jname) close}
		}
		error "joinfiles:  Key $key not found in file $iname"
	    }
	}
    }
    if {$debug} {puts "allnames is $allnames"}
#
# Handle keys and duplicate names from each set
# creating "unique names": unames
#   (Note: case insensitivity here)
#	
    set nkeys [llength $keylist]

    set lkeylist [string tolower $keylist]      ;# "l" for lower case testing
    set lallnames [string tolower $allnames]
    foreach iname $inames {
	if {$xfile == "solarfile"} {
	    set lkeylist {}
	    foreach keyname $keylist {
		lappend lkeylist [string tolower [eval $xfile $tfile($iname) \
						      establish_name $keyname]]
	    }
	}
	set names $fnames($iname)
	set ns ""
	set uns ""
	foreach name $names {
	    set lname [string tolower $name]
	    if {-1 == [lsearch -exact $lkeylist $lname]} {
		if {[string_imatch $make_unique norename]} {
		    lappend ns $name
		    lappend uns $name
		} elseif {[unique $lname $lallnames]} {
		    lappend ns $name
		    lappend uns $name
		} elseif {[string_imatch $make_unique rename]} {
		    set test "$name.$iname"
		    set ltest [string tolower $test]
		    set lt_allnames [concat $lallnames $ltest]
		    if {[unique $ltest $lt_allnames]} {
			lappend ns $name
			lappend uns $test
			lappend allnames $test
			lappend lallnames $ltest
		    } else {
			for {set ii 2} {$ii < 30002} {incr ii} {
			    set test $name.$iname.$ii
			    set ltest [string tolower $test]
			    set lt_allnames [concat $lallnames $ltest]
			    if {[unique $ltest $lt_allnames]} {
				lappend ns $name
				lappend uns $test
				lappend allnames $test
				lappend lallnames $ltest
				break
			    }
			}
		    }
		}
	    }
	    set unames($iname) $uns
	    set fnames($iname) $ns
	}
    }
#
# Read elements of each file into an associative array (hashtable)
# named data(iname,id) where "id" is comma delimited key string of id's
# Also create set/list of all id's: allids
#
# Detect same id's used twice in same file as an error
#
    set allids {}
    set header [join $keylist ,]
    foreach iname $inames {
	set tf $tfile($iname)
	eval $xfile $tf start_setup
	foreach key $keylist {
	    eval $xfile $tf setup $key
	}
	foreach fname $fnames($iname) {
	    eval $xfile $tf setup $fname
	}
	foreach uname $unames($iname) {
	    set header "$header,$uname"
	}
	while {{} != [set record [eval $xfile $tf get]]} {
	    set id [lindex $record 0]
	    for {set i 1} {$i < $nkeys} {incr i} {
		set id "$id,[lindex $record $i]"
	    }
	    setappend allids $id
	    set fullid "$iname,$id"
	    if {![catch {set foo $data($fullid)}]} {
		foreach jname $inames {
		    catch {eval $xfile $tfile($jname) close}
		}
		error "joinfiles:  Found multiple records with key <$id> in file $iname"
	    }
	    set data($fullid) [lrange $record $nkeys end]
	}
    }
#
# WRITE OUTPUT FILE, starting with header
# then, for each id found, output composite record from all files
#
    putsout -q -d. $oname $header
    set count 0
    foreach id $allids {
	set record $id
	foreach iname $inames {
	    if {0 == [llength $fnames($iname)]} continue
	    if {[catch {set record "$record,[join $data($iname,$id) ,]"}]} {
		foreach n $fnames($iname) {
		    set record "$record,"
		}
	    }
	}
	putsout -q -d. $oname $record
	incr count
    }
    foreach jname $inames {
	catch {eval $xfile $tfile($jname) close}
    }
    return "$count records written to $oname"
}

proc unique {name list} {
    if {-1 != [set start [lsearch -exact $list $name]]} {
	if {-1 != [lsearch -exact [lrange $list [expr $start + 1] end] $name]} {
	    return 0
	}
    }
    return 1
}

# solar::invert
#
#
# invert from version 6 has been renamed "transpose" in version 7
# see "help transpose" for more information
#-

# solar::transpose
#
# Purpose:: transpose on MathMatrix or comma delimited file
#
# Usage: transpose <MathMatrix>  ;# returns id of transposed MathMatrix
#        transpose <infile> <outfile>  ;# transposes CSV file
#
# Note: All records must have same length.  First record is treated like all
# others.  To invert Pedsys file, use ped2csv first.  Memory usage for
# extremely large files (>100mb) could be a problem.  If memory is exhausted
# while caching the file in memory, solar might crash to the shell prompt.
# -

#
# this transpose below stores original file in an array of lists
# each list element must therefore be accessed through a lindex
# in theory, that might lead to more compute time than using an array
# to store each element, rather than each line.  However, this approach
# uses far less overall memory, and actually seems to run faster for large
# files.  The original invert based on element storage is 
# now called oldinvert and retained below.
#

proc transpose {args} {

    set nargs [llength $args]
    if {$nargs == 1} {
	return [ctranspose $args]
    } elseif {$nargs != 2} {
    error "Usage: transpose <csvinput> <csvoutput>  OR  transpose <MathMatrix>"
    }
    return [eval transpose_csv $args]
}

proc transpose_csv {infilename outfilename} {
    set infile [open $infilename]
    set outfile [open $outfilename w]

    set last_maxj -1
    for {set i 0} {1} {incr i} {
	if {-1 == [gets $infile inline]} {
	    break
	}
	set inlist [split $inline ,]
	set maxj [llength $inlist]
	if {$last_maxj == -1} {
	    set last_maxj $maxj
	} else {
	    if {$maxj != $last_maxj} {
		puts "Length of record $i is $maxj"
		puts "Length of previous records was $last_maxj"
		close $infile
		close $outfile
		error "invert: Error! All records must be same length!"
	    }
	}
	set mlist($i) $inlist
    }
    close $infile
    set maxi $i

    for {set j 0} {$j < $maxj} {incr j} {
	set first 0
	for {set i 0} {$i < $maxi} {incr i} {
	    if {$first==0} {
		set outline "[lindex $mlist($i) $j]"
		set first 1
	    } else {
		set outline "$outline,[lindex $mlist($i) $j]"
	    }
	}
	puts $outfile $outline
    }
    close $outfile
    return ""
}

proc oldtranspose {infilename outfilename} {
    set infile [open $infilename]
    set outfile [open $outfilename w]

    set last_maxj -1
    for {set i 0} {1} {incr i} {
	if {-1 == [gets $infile inline]} {
	    break
	}
	set inlist [split $inline ,]
	set maxj [llength $inlist]
	if {$last_maxj == -1} {
	    set last_maxj $maxj
	} else {
	    if {$maxj != $last_maxj} {
		puts "Length of record $i is $maxj"
		puts "Length of previous records was $last_maxj"
		close $infile
		close $outfile
		error "invert: Error! All records must be same length!"
	    }
	}
	for {set j 0} {$j < $maxj} {incr j} {
	    set atom [lindex $inlist $j]
	    set matrix($i,$j) $atom
	}
    }
    close $infile
    set maxi $i

    for {set j 0} {$j < $maxj} {incr j} {
	set first 0
	for {set i 0} {$i < $maxi} {incr i} {
	    if {$first==0} {
		set outline "$matrix($i,$j)"
		set first 1
	    } else {
		set outline "$outline,$matrix($i,$j)"
	    }
	}
	puts $outfile $outline
    }
    close $outfile
    return ""
}




#
# General Purpose Utilities...could be useful outside SOLAR
#
# (Unfortunately this section wasn't created earlier, so general purpose
#  utilities are actually scattered throughout the code.)
#


proc remove_from_list {alist ename} {
    set epos [lsearch -exact $alist $ename]
    return [lreplace $alist $epos $epos]
}

# solar::remlist
#
# Purpose:  Remove element from list by name
#
# Usage:    remlist <list> <element>
#
# Notes:    Input list is not modified, but new list is returned.
#
#           Only first matching element is removed.  This works well
#           with setappend for set-like behavior: use setappend to add
#           elements to "set" and remlist to remove
#           elements from set.
#
#           Match testing is case insensitive.
#
#           No error is raised if thre is no matching element; input
#           list is returned unchanged.
#
# See Also: setappend
#
# -

proc remlist {list element} {
    return [remove_from_list_if_found $list $element]
}

proc remove_from_list_if_found {alist ename} {
    set llist [string tolower $alist]
    set lname [string tolower $ename]
    set epos [lsearch -exact $llist $lname]
    if {$epos >= 0} {
	return [lreplace $alist $epos $epos]
    }
    return $alist
}


proc listfile {filename} {
    set retlist {}
    set infile [open $filename]
    while {-1 != [gets $infile line]} {
	if {"" != $line} {
	    lappend retlist [string trim $line]
	}
    }
    close $infile
    return $retlist
}

proc line_index {text lineno} {
    for {set i 0} {$i <= $lineno } {incr i} {
	set nindex [string first "\n" $text]
	if {$nindex < 0} {
	    if {$i == $lineno} {
		return $text
	    } else {
		return ""
	    }
	}
	if {$i == $lineno} {
	    return [string range $text 0 [expr $nindex - 1]]
	}
	set text [string range $text [expr $nindex + 1] end]
    }
    return ""
}

    
# solar::remove_global --
#
# Purpose:  Remove a global variable (so it no longer exists)
#
# Usage:    remove_global <variable_name>
#
# Notes:    It is not necessary to declare variable as global first,
#           and there is no error if no such global actually exists.
#
# See Also: if_global_exists
# -

proc remove_global {name} {
    purge_global $name
}

proc purge_global {name} {
    if {0 != [llength [info globals $name]]} {
	global $name
	unset $name
    }
}

# solar::is_nan --
#
# Purpose:  Check if value is NaN (Not a Number)
#
# Usage:    is_nan <number>
#
#           Returns 1 if number is NaN, 0 otherwise.
#
# Notes:    This is most useful in scripts, when getting the likelihood or
#           other value using read_model, you should check to be sure it
#           is not NaN due to maximization convergence error.
# -


proc is_nan {numstr} {
    if {![string compare "NAN" [string toupper [string range $numstr 0 2]]]} {
	return 1
    }
    return 0
}

proc is_integer {string} {
    if {1 == [scan $string "%d%s" test junk]} {
	return 1
    }
    return 0
}

proc is_float {string} {
    if {1 == [scan $string "%g%s" test junk]} {
	return 1
    }
    return 0
}

proc ensure_integer {string} {
    if {![is_integer $string]} {
	error "Invalid integer: $string"
    }
    return {}
}

proc ensure_float {string} {
    if {![is_float $string]} {
	error "Invalid floating point number: $string"
    }
    return {}
}

proc is_digit {ch} {
    return [string match {[0-9]} $ch]
}

proc is_alpha {ch} {
    return [string match {[A-Za-z]} $ch]
}

proc is_alnum {ch} {
    return [string match {[A-Za-z0-9]} $ch]
}


# solar::fformat -- 
#
# Purpose:  Replace Tcl format with fixed width fields for numbers
#
# Usage:    fformat <spec>+ <value1>+
#
#           <spec>   format specifier(s) as for Tcl format command.
#                    f, e, or g format required for "fixed width"
#                    operation, like this:
#
#                   %[--][W][.P]T  where T is e, f, g, or y
#                      default right justification
#                   -  specifies left justification
#                   --  specifies center justification
#                   W is desired width
#                   P is desired precision (before and after decimal)
#                   T is format type:
#                     f is floating decimal
#                     e is exponential
#                     g is floating decimal if suitable, then exponential
#                     y same as g, except that exponential format is not
#                       used until the output would otherwise be 0.0 or
#                       nearly so for a non-zero value.  At least one
#                       significant digit is preserved for P 1-4, two 
#                       digits for P 4-6, and three digits for P  7-*.
#                       This is more consistent with readability, 
#                       retaining the fixed format nearly as long as
#                       possible.  Sometimes, more space will be used than W,
#                       but this is much less likely than with the standard
#                       G format.  However, unlike F format, the the result
#                       will not go to zero unless it is zero.  When possible,
#                       allow more space in "width" than the precision seems
#                       to require.  That way, under special circumstances,
#                       there is extra space for signs, "e", decimal point,
#                       etc.
#                     z same as y, except resulting string is trimmed to
#                     minimum space for csv files
#
#                    Note: For fractional numbers, make width at least 2
#                    than precision, to allow for leading "0."  Then allow
#                    one more for - sign, if that is possible.
#
#          This is intended as a drop-in replacement for the Tcl "format"
#          command, modifying "minimum width" to "fixed width" for
#          the f, e, and g formats ("fixed width" makes for more
#          readable columns) and adding a center justification option.
# -

proc fformat {spec args} {

    set result ""
    set argindex 0

    set speclen [string length $spec]
    for {set i 0} {$i < $speclen} {incr i} {
	set newchar [string index $spec $i]
	set nextchar [string index $spec [expr $i + 1]]

	if {"%" == $newchar} {
	    if {"%" == $nextchar} {
		incr i
		set result [catenate $result % ]
		continue
	    }
	    set subspec "%"
	    for {incr i} {$i < $speclen} {incr i} {
		set subchar [string index $spec $i]
		set subspec [catenate $subspec $subchar]
		if {"." == $subchar || "-" == $subchar || \
			"+" == $subchar || "l" == $subchar || \
			[is_digit $subchar]} {
		    continue
		}
		set newresult [fformat1 $subspec [lindex $args $argindex]]
		incr argindex
		set result [catenate $result $newresult]
		break
	    }

	} else {
	    set result [catenate $result $newchar]
	}
    }
    return $result
}


# solar::fformat1 -- private
#
# Purpose: Subroutine used by fformat to format individual numbers
#
# Usage:   fformat <spec> <value>
#
#           See fformat for details.
# -

proc fformat1 {spec value} {

    if {"%" != [string index $spec 0]} {
	error "invalid spec: must start with %"
    }

    set prefix [string range $spec 1 [expr [string length $spec] - 2]]
    set letter [string range $spec end end]


# Parse leading + and -

    set justify right
    set signed ""
    set minuscount 0

    while {1} {
	if {"-" == [string index $prefix 0]} {
	    incr minuscount
	} elseif {"+" == [string index $prefix 0]} {
	    set signed "+"
	} else {
	    break
	}
	set prefix [string range $prefix 1 end]
    }

    if {$minuscount == 1} {
	set justify left
    } elseif {$minuscount > 1} {
	set justify center
    }

# Parse width, if any

    set width 0
    set gotwidth [scan $prefix %i width]

    set segment ""
    if {$gotwidth > 0 || "y" == $letter || "z" == $letter} {
	set output [fformat1w $signed $width $prefix $letter $value]
    } else {
	set output [format %$signed$prefix$letter $value]
    }

# Perform justification or centering  (only if width specified)

    if {$width != 0} {
	set s s
	if {"left" == $justify} {
	    set output [string trim $output]
	    set output [format %-$width$s $output]
	} elseif {"center" == $justify} {
	    set output [string trim $output]
	    set next left
	    while {[string length $output] < $width} {
		if {"left" == $next} {
		    set output " $output"
		    set next right
		} else {
		    set output "$output "
		    set next left
		}
	    }
	} ;# no need to do anything for right justification
    }
    return $output
}


proc fformat1w {signed width prefix letter value} {

    set yformat 0
    set zformat 0
    if {"y" == $letter} {
	set yformat 1
	set letter f
    } elseif {"z" == $letter} {
	set yformat 1
	set letter f
	set zformat 1
    }
    set default [format %$signed$prefix$letter $value]
    set new $default


# "s" format truncates to precision, which is dangerous
# (this is not the case for i or d format)

    if {"s" == $letter} {
	return $default
    }

    set catchval [catch {

# Get precision

	set dotpos [string first . $prefix]
	if {$dotpos == -1} {
	    set prec $width
	} else {
	    set prefixlen [string length $prefix]
	    set prec [string range $prefix [expr $dotpos + 1] [expr $prefixlen - 1]]
	    if {"" == $prec} {
		set prec $width
	    }
	}

	
#   puts "width >$width<   prec  >$prec<  letter  >$letter<"


# For "y" format, use "f" format if rules satisfied, else use "g"

	if {$yformat} {
	    set minsig 1
	    if {[string length $new] <= $width} {
		if {$value == 0} {
		    return $new
		}
		if {$prec < 5} {
		    if {$new != 0} {
			return $new
		    }
		} elseif {$prec < 7} {
		    set minsig 2
		    set target [string trimright $new]
		    set masklen [expr [string length $target] - 2]
		    set masked [string range $target 0 $masklen]
		    if {$masked != 0} {
			return $new
		    }
		} else {
		    set minsig 3
		    set target [string trimright $new]
		    set masklen [expr [string length $target] - 3]
		    set masked [string range $target 0 $masklen]
		    if {$masked != 0} {
			return $new
		    }
		}
	    }
	    set letter e
#	    puts "CALLING FORMAT %$signed$prefix$letter"
	    set new [format "%$signed$prefix$letter" $value]
	}

	if {[string length $new] <= $width} {
	    return $new
	}
	
	set minprec 0
	if {$yformat} {
	    set minprec [expr $minsig - 1]
	}

	for {set test $prec} {$test >= $minprec} {incr test -1} {
	    set newprefix "%$signed$width.$test$letter"
	    set new [format $newprefix $value]
	    if {[string length $new] <= $width} {
		return $new
	    }
	}
	return $new  ;# OK, this appears to be the best we can do

    } catchstring ]
    

# see if we just did a "return" from catch block
    if {$catchval == 2} {
	if {$zformat} {
	    set catchstring [string trim $catchstring]
	}
	return $catchstring
    }
	
# Otherwise, use default
    return $default
}


# solar::if_global_exists --
#
# Purpose:  Check if a Tcl global variable exists
#
# Usage:    if_global_exists <global_name>
#
#           Returns 1 if global exists, 0 otherwise.
#
# Notes:    This is used in scripts in an "if" statement.  For example:
#
#           if {[if_global_exists SPECIAL_CASE]} {
#               global SPECIAL_CASE
#               puts "This is case $SPECIAL_CASE"
#           }
#
#           You do not need to declare the variable "global" before
#           calling if_global_exists.  However, you will need to declare it
#           global before setting or using it in a script.  Note that all
#           variables declared at the interpreter level (at the solar>
#           prompt) are automatically global.  Global variables should
#           not be confused with "shell" variables such as SOLAR_BIN
#           (though, all shell variables may be found in the global
#           array "env", for example, $env(SOLAR_BIN)).
#
#           Global variables are a convenient way of passing variables
#           through many levels of procedure calls without rewriting all
#           the intervening procedures, or across commands on an ad hoc basis.
#           Use of global variables is considered "bad style" by programming
#           purists and other bores.  But if they're so smart, why aren't they
#           writing your program?  It is true, however, that use of global
#           variables can sometimes introduce bugs and other unintended
#           consequences.
#
#           Globals variables prefixed with SOLAR_ are reserved for use by
#           the standard SOLAR procedures defined in solar.tcl.  But solar.tcl
#           might also use unprefixed globals, so it is recommended that users
#           use their own unique prefix to be safe.
#
# See Also: remove_global
#
# -

proc if_global_exists {name} {
    if {0 == [llength [info globals $name]]} {
	return 0
    } else {
	return 1
    }
}

# solar::stringsub --
#
# Purpose:  Simple verbatim string substitution (not regsub)
#
# Usage:    stringsub <original> <target> <replacement>
#
# -

proc stringsub {original target replacement} {
    set out $original
    if {-1 != [set bpos [string first $target $original]]} {
	set out [catenate [string range $original 0 [expr $bpos - 1]] \
		$replacement \
		[string range $original \
		[expr $bpos + [string length $target]] end]]
    }
    return $out
}

# Replace element in pre-formatted string without changing format
# whitespace between each element assumed
proc replace_using_format {line formats index newitem} {
    set items {}
    set line_length [llength $line]
    for {set i 0} {$i < $line_length} {incr i} {
	if {$i == $index} {
	    lappend items $newitem
	} else {
	    lappend items [lindex $line $i]
	}
    }
    return [eval format \$formats $items]
}    

# solar::catenate --
#
# Purpose:  Concatenate strings
#
# Usage:    catenate [<string>]*
#
# Example:  set modelname [catenate $basename 0 .mod]
#
# -

proc catenate {args} {
    set base ""
    foreach arg $args {
	set base $base$arg
    }
    return $base
}

# solar::string_imatch --
#
# Purpose:  Case insensitive string match testing
#
# Usage:    string_imatch <string1> <string2>
#
#           Returns 1 for case insensitive match, 0 otherwise.
#
# Note:     Useful in SOLAR scripts.
#
# -

proc string_imatch {string1 string2} {
    if {[string compare [string tolower $string1] [string tolower $string2]]} {
	return 0
    }
    return 1
}

# Trimming

proc is_blank {testchar} {
    if {" " == $testchar || "\t" == $testchar} {
	return 1
    }
    return 0
}

proc trim_left {istring} {
    set leftchar [string range $istring 0 0]
    while {[is_blank $leftchar]} {
	set istring [string range $istring 1 end]
	set leftchar [string range $istring 0 0]
    }
    return $istring
}

	

proc remove_whitespace {istring} {
    set slength [string length $istring]
    set result ""
    for {set i 0} {$i < $slength} {incr i} {
	set char [string index $istring $i]
	if {[string compare $char " "] && \
		[string compare $char "\t"] && \
		[string compare $char "\n"] && \
		[string compare $char "\r"] && \
		[string compare $char "\v"] && \
		[string compare $char "\f"]} {
	    set result $result$char
	}
    }
    return $result
}

# solar::setappend --
#
# Purpose:  Append only new elements to a list (keeping it like a set)
#
# Usage:    setappend <listname> element
#
# Note:     The list is identified by name, and may be modified, as with
#           lappend.
#
# Example:  set friends "John Tom Laura Jeff"
#           setappend friends Harald
#
# See Also: remlist
#
# -

# Append only new atoms to list
proc setappend {aset atom} {
    upvar $aset a
    if {-1 == [lsearch -exact $a $atom]} {
	lappend a $atom
    }
    return $a
}


# solar::setxor --
#
# Purpose:  Perform exclusive-or (xor) on two sets (Tcl lists)
#
# Usage:    setxor aset bset
#
# Note:     If element appears multiple times in one list, but not in other,
#           it will appear multiple times in output.
# -

proc setxor {list1 list2} {
    set xor {}
    foreach e $list1 {
	if {-1 == [lsearch -exact $list2 $e]} {
	    lappend xor $e
	}
    }
    foreach e $list2 {
	if {-1 == [lsearch -exact $list1 $e]} {
	    lappend xor $e
	}
    }
    return $xor
}


proc lowest {args} {
    if {0 == [llength $args]} {
	error "No arguments to lowest"
    }
    set low [lindex $args 0]
    foreach arg $args {
	if {$arg < $low} {
	    set low $arg
	}
    }
    return $low
}

proc highest {args} {
    if {0 == [llength $args]} {
	error "No arguments to highest"
    }
    set high [lindex $args 0]
    foreach arg $args {
	if {$arg > $high} {
	    set high $arg
	}
    }
    return $high
}

proc sum_of_squares {list} {
    set sum 0.0
    foreach l $list {
	set sum [expr $sum + ($l * $l)]
    }
    return $sum
}

proc sum {args} {
    set sum 0
    foreach arg $args {
	set sum [expr $sum + $arg]
    }
    return $sum
}

# procedure to make absolute pathname
proc make_absolute_pathname {name} {
    if {"absolute" == [file pathtype $name]} {
	return $name
    }
    if {"." != $name} {
	set newname [file join [pwd] $name]
    } else {
	set newname [pwd]
    }
    return $newname
}

#proc to center a line in an (assumed) 80 column line
proc centerline {instring {linewidth 80}} {
    set slength [string length $instring]
    if {$slength > [expr $linewidth - 1]} {return $instring}
    set needs [expr ($linewidth - $slength) / 2]
    set width [expr $needs + $slength]
    return [format %[catenate $width s] $instring]
}

# proc to count occurrances of a particular character in a string
proc char_count {string target_char} {
    set slist [split $string ""]
    set count 0
    foreach char $slist {
	if {0 == [string compare $char $target_char]} {
	    incr count
	}
    }
    return $count
}

# proceedure to use global if it exists or altvalue otherwise
proc use_global_if_defined {globalname altvalue} {
    if {0 != [llength [info globals $globalname]]} {
	upvar #0 $globalname g
	return $g
    } else {
	return $altvalue
    }
}

# Procedure to forcibly delete arbitrary number of files
# (Tcl's "file delete -force" doesn't like the null file)
proc delete_files_forcibly {args} {
    if {{} == $args} {
	return ""
    } else {
	eval file delete -force $args
    }
}

# solar::countfields
#
# Purpose: determine consistency of number of columns in a comma delimited file
#
# Usage: countfields <filename>
#
# An information report is returned like this:
#
# longest: 8 (#1) x 1497    shortest: 8 (#1) x 1497
#
# This means that the longest record had 8 fields, the first such record was
# #1, and it was followed by 1497 others of same length in the file.
#
# As it happens, the shortest record also had 8 fields, it was #1, and followed
# by 1497 of the same length in the file.
# -

proc countfields {filename} {
    set ifile [open $filename]
    set shortest 0
    set shortcount 0
    set shortest_line 1
    set longest_line 1
    set longcount 0
    set longest 0
    set count 0
    while {-1 != [gets $ifile line]} {
	set rlist [split $line , ]
	set len [llength $rlist]
	incr count
	if {$count == 1} {
	    set shortest $len
	    set longest $len
	} else {
	    if {$len < $shortest} {
		set shortest $len
		set shortest_line $count
		set shortcount 1
	    } elseif {$shortest == $len} {
		incr shortcount
	    }
		
	    if {$len > $longest} {
		set longest $len
		set longest_line $count
		set longcount 1
	    } elseif {$longest == $len} {
		incr longcount
	    }
	}
    }
    close $ifile
    return "longest: $longest (\#$longest_line) x $longcount    shortest: $shortest (#$shortest_line) x $shortcount"

}
# solar::copybin --
#
# Purpose:  Install new executable file without disturbing current users
#
# Usage:    copybin <filename> <directory>
#
# Note:    The way this works is quite simple.  The original version of the
#          file is not overwritten or deleted, but instead renamed.  
#          Running processes continue to access the original version
#          through the inode, regardless of the name change, while new
#          processes will access the new version.  The renaming scheme
#          simply appends dot followed by a number to the filename.
#          The first available number starting from 1 is used.  For
#          example, the old "solarmain" becomes "solarmain.1" or
#          "solarmain.2" if a "solarmain.1" already exists, etc.  At some
#          point you might want to clear out some of the older versions, but
#          that is up to you, and it would lead to numbering that is not
#          sequential, since copybin always takes the first available
#          number.
#
#          This is similar in design to the Unix "install -f" command.
#          It lacks some of install's checking features, but in one way
#          is much more capable: it allows any number of new versions to
#          be installed without disturbing users of the first or any other
#          previous version.  The Unix install command only has one level
#          of backup since it merely prepends "OLD" to the original name.
#          If you do two install's in a row over a timespan in which jobs
#          are continuing to run (as, unfortunately, is often required)
#          copies of the original version are lost and users are likely
#          to get a memory mapping error of some kind.
#
#          This seems to work across NFS mounted filesystems, but it might
#          not work for you, so be wary.  Actually, in ancient Unix days this
#          command might not have been necessary, but now that memory mapping
#          is used to load image files, it is necessary now.
# -


proc copybin {newfile dir} {

    if {![file exists $newfile]} {
	error "No such file $newfile"
    }
    if {![file exists $dir]} {
	error "No such directory $dir"
    }
    if {[file isdirectory $newfile]} {
	error "File $newfile is a directory"
    }
    if {![file isdirectory $dir]} {
	error "File $dir is not a directory"
    }

    set filename [file tail $newfile]
    if {[file exists [file join $dir $filename]]} {
	set index 1
	while {[file exists [file join $dir $filename.$index]]} {
	    incr index
	}
	puts "Moving [file join $dir $filename] to [file join $dir $filename.$index]"
	exec mv [file join $dir $filename] [file join $dir $filename.$index]
    }

    puts "Copying $newfile to $dir"
    exec cp $newfile $dir
}

proc copybini {newfile dir} {

    if {![file exists $newfile]} {
	error "No such file $newfile"
    }
    if {![file exists $dir]} {
	error "No such directory $dir"
    }
    if {[file isdirectory $newfile]} {
	error "File $newfile is a directory"
    }
    if {![file isdirectory $dir]} {
	error "File $dir is not a directory"
    }

    exec /usr/sbin/install -f $dir -o $newfile
}

	
	
# procedure to make command aliases
#   Note: by design the longname can be longer than (or not same as) procname
#   But, shortname must be a right truncation of longname (only its length

proc make_shortcut {shortname longname {procname ""}} {
    if {"" == $procname} {set procname $longname}
    if {$longname == $shortname} {set longname [catenate $longname _]}
    set shortlen [string length $shortname]
    while {[string length $longname] >= $shortlen} {
	set newname $longname
	set longname [string range $longname 0 \
		[expr [string length $longname] - 2]]
	if {0==[string compare $newname $procname]} {continue}
	eval proc $newname args \{return \[eval $procname \$args\]\}
    }
    return ""
}

# solar::showproc --
#
# Purpose:  Show SOLAR procedure or write to a file
#
# Usage:    showproc <procname> [<filename>]
#
# If <filename> is not specified, procedure is displayed on terminal using
# the 'more' pager.  If <filename> is specified, renamed proc is written
# to that file.
#
# This procedure will show any SOLAR procedure (script), whether built-in
# or user-defined.  Some, but not all, built-in SOLAR commands
# are implemented as scripts, and can be shown by this command.  Other
# SOLAR commands are implemented in C++ and FORTRAN, and cannot be shown
# by this command.
#
# User-defined scripts must be used once before they can be shown.
#
# The formatting shown by showproc may not be as pretty as it actually is
# in the source file because it will concatenate lines which are extended
# using backslash.  showproc is based on the Tcl command "info body" which
# has this "feature."
#
# To protect built-in procedures from being accidentally superceded
# through the use of this command, the procedure name is suffixed with
# ".copy".  If you choose to edit the script, IT IS RECOMMENDED THAT
# YOU DO NOT RENAME IT TO HAVE THE SAME NAME AS THE ORIGINAL PROCEDURE
# UNLESS YOU REALLY KNOW WHAT YOU ARE DOING.  If you do that anyway,
# it would probably be ignored.  SOLAR prevents you from overriding
# built-in procedures by putting the directory containing the active
# solar.tcl file to the front of the auto-load list.  Normally, that
# directory is the SOLAR_BIN directory defined when SOLAR starts up.
# Even if you did have a copy of the solar.tcl file in your working
# directory when SOLAR started up, procedures might be resolved either
# to the solar.tcl file or to separate script files in your working
# directory, depending on which appears earlier in an alphabetical
# list.
#
# Before new procedures can be used in SOLAR you must restart SOLAR or give
# the newtcl command.
#
#-

proc writeproc {args} {
    error "Use showproc instead"
}

proc showproc {procname {filename ""}} {
    set display 0
    if {"" == $filename} {
	set filename /tmp/[catenate $procname _proc].[pid]
	set display 1
    }
    set procbody [info body $procname]
    set procargs [info args $procname]
    set ofile [open $filename w]
    puts $ofile "proc [catenate $procname .copy] \{$procargs\} \{"
    puts $ofile $procbody
    puts $ofile "\}"
    close $ofile
    if {$display} {
	exec more $filename >&@stdout
	delete_files_forcibly $filename
    }
}
#
# Solar command shortcuts
#

# solar::shortcut --
#
# Purpose:  Show command shortcuts legal in scripts
#
# Usage:    shortcut <command>
#
# -

proc shortcut {command} {
    global Solar_Shortcut_List
    set foundshort ""
    set foundlong ""
    foreach sc $Solar_Shortcut_List {
	set shortname [lindex $sc 0]
	set longname [lindex $sc 1]
	set shortend [expr [string length $shortname] - 1]
	set commandend [expr [string length $command] - 1]
	if {$commandend < $shortend} continue
	if {0==[string compare $shortname \
		[string range $command 0 $shortend]]} {
	    if {0==[string compare $command \
		    [string range $longname 0 $commandend]]} {
		set foundshort $shortname
		set foundlong $longname
		break
	    }
	}
    }
    if {"" == $foundshort} {
	error "Shortcuts for $command not found"
    }
    return "Shortcuts: $foundshort - $foundlong"
}

proc make_solar_shortcut {args} {
    global Solar_Shortcut_List
    set shortname [lindex $args 0]
    set longname [lindex $args 1]
    lappend Solar_Shortcut_List [list $shortname $longname]
    eval make_shortcut $args
    return ""
}

proc make_solar_aliases {} {
    global Solar_Shortcut_List
    set Solar_Shortcut_List {}
    make_solar_shortcut abou        about
    make_solar_shortcut allc        allcovars              allcovar
    make_solar_shortcut alnorm      alnorm
    make_solar_shortcut analy       analysis-examples      analysis-example
    make_solar_shortcut ascer       ascertainment
    make_solar_shortcut autom       automodel
    make_solar_shortcut bayesa      bayesavg
    make_solar_shortcut bayesm      bayesmodel             bayesmod
    make_solar_shortcut beni        benice
    make_solar_shortcut bou         boundary               boundary
    make_solar_shortcut boundari    boundaries             boundary
    make_solar_shortcut chang       change-notes
    make_solar_shortcut chi         chi
    make_solar_shortcut chro        chromosomes            chromosome
    make_solar_shortcut clod        clod
    make_solar_shortcut comb        combinations
    make_solar_shortcut cons        constraints            constraint
    make_solar_shortcut cov         covariates             covariate
    make_solar_shortcut discrete-note  discrete-notes
    make_solar_shortcut doc         doc
    make_solar_shortcut dran        drand
    make_solar_shortcut e2l         e2lower
    make_solar_shortcut e2s         e2squeeze
    make_solar_shortcut epista      epistasis
    make_solar_shortcut examp       example
    make_solar_shortcut excl        exclude
    make_solar_shortcut fie         fields                 field
    make_solar_shortcut file-f      file-freq
    make_solar_shortcut file-map    file-map
    make_solar_shortcut file-mar    file-markers           file-marker
    make_solar_shortcut file-pe     file-pedigrees         file-pedigree
    make_solar_shortcut file-ph     file-phenotypes
    make_solar_shortcut fine        finemap
    make_solar_shortcut fix         fix
    make_solar_shortcut fre         freq
    make_solar_shortcut getvar      getvar
    make_solar_shortcut grid        grid
    make_solar_shortcut h2qf        h2qf
    make_solar_shortcut h2rf        h2rf
    make_solar_shortcut hel         help
    make_solar_shortcut help-d      help-document
    make_solar_shortcut helpa       helpadd
    make_solar_shortcut hou         houses                 house
    make_solar_shortcut ibd         ibd
    make_solar_shortcut ibdd        ibddir
    make_solar_shortcut ibdo        ibdoptions             ibdoption
    make_solar_shortcut ibs         ibs
    make_solar_shortcut interv      intervals              interval
    make_solar_shortcut key         key
    make_solar_shortcut linkm       linkmodel              linkmod
    make_solar_shortcut linkmod2    linkmod2p
    make_solar_shortcut load        load
    make_solar_shortcut loadk       loadkinship            loadkin
    make_solar_shortcut lod         lod
    make_solar_shortcut lodn        lodn
    make_solar_shortcut logl        loglikelihood          loglike
    make_solar_shortcut madj        madj
    make_solar_shortcut map         map
    make_solar_shortcut mark        markers                marker
    make_solar_shortcut markert     markertest
    make_solar_shortcut matr        matrixes               matrix
    make_solar_shortcut maxi        maximize
    make_solar_shortcut mem         memory
    make_solar_shortcut mibd        mibd
    make_solar_shortcut mibdd       mibddir
    make_solar_shortcut minipl      miniplot
    make_solar_shortcut model       model
    make_solar_shortcut mu          mu
    make_solar_shortcut mul         multipoint
    make_solar_shortcut needk       needk2
    make_solar_shortcut newt        newtcl
    make_solar_shortcut newm        newmodels              newmod
    make_solar_shortcut no_ch       no_check_os
    make_solar_shortcut null        null
    make_solar_shortcut nulln       nulln
    make_solar_shortcut oldm        oldmodel
    make_solar_shortcut ome         omega
    make_solar_shortcut opt         options                option
    make_solar_shortcut outd        outdir
    make_solar_shortcut par         parameters             parameter
    make_solar_shortcut ped         pedigrees              pedigree
    make_solar_shortcut pedli       pedlike
    make_solar_shortcut pedlo       pedlod
    make_solar_shortcut perd        perdelta
    make_solar_shortcut perturb     perturb
    make_solar_shortcut phen        phenotypes
    make_solar_shortcut plo         plotmulti              plot
    make_solar_shortcut polyg       polygenic
    make_solar_shortcut polym       polymodel              polymod
    make_solar_shortcut quadrat     quadratic
    make_solar_shortcut qtn         qtnmarker              qtnm
    make_solar_shortcut regi        register
    make_solar_shortcut relat       relatives
    make_solar_shortcut resi        residuals              residual
    make_solar_shortcut sav         save
    make_solar_shortcut scree       screencov
    make_solar_shortcut shortc      shortcuts              shortcut
    make_solar_shortcut simin       siminf
    make_solar_shortcut simqt       simqtl
    make_solar_shortcut slod        slod
    make_solar_shortcut solarm      solarmodel
    make_solar_shortcut solart      solartcl
    make_solar_shortcut solarv      solarversion
    make_solar_shortcut spor        spormodel               spormod
    make_solar_shortcut stringp     stringplot
    make_solar_shortcut tabl        tablefile
    make_solar_shortcut tclg        tclgr
    make_solar_shortcut tdis        tdist
    make_solar_shortcut trai        traits                  trait
    make_solar_shortcut twop        twopoint
    make_solar_shortcut upg         upgrade
    make_solar_shortcut usag        usages                  usage
    make_solar_shortcut usor        usort
    make_solar_shortcut verb        verbosity
    make_solar_shortcut mgas        mgassociation           mgassoc
    return ""
}

#
# The following section documents commands implemented in C++
#
# solar::model --
#
# Purpose:  Describe, save, or load a model
# 
# Usage:    save model <modelname>     ; save current model to a file
#           load model <modelname>     ; load model from a file
#           model                      ; display model on terminal
#           model new                  ; reset to new empty model
# 
# Notes:    An extension .mod is automatically appended if not specified.
#           You must specify directory path if you want to save model
#           in a subdirectory of the current directory.
# - 

# solar::load --
#
# Purpose:  Load a user data file (pedigree, phenotype, marker, etc.)
#
# Usage:    load <object-type> [<options>] <arguments>
#
#           load pedigree <filename>
#           load phenotypes <filename>
#           load matrix [-sample | -allow] <filename> <name1> [<name2>]
#           load matrix [-cols <tcl-list>] <filename> ;# MathMatrix
#           load model <filename>
#           load freq [-nosave] <filename>
#           load marker [-xlinked] <filename>
#           load map [-haldane | -kosambi] <filename>
#
# Notes:    There is much more information available for each variant of
#           the load command.  See the documentation for the
#           particular object type, for example, "help pedigree".
#           For information about a particular file format, see the
#           applicable file-* documentation, for example, "help file-pedigree".
#-

# solar::save --
#
# Purpose:  save <object-type> <arguments>
#
#           save model <filename>
#
#           More information is available under "help model"
#-

proc load {args} {

    if {[llength $args] < 1} {
	error "Usage: load <object-type> [<options>] <args>"
    }
    return [eval [lindex $args 0] load [lrange $args 1 end]]
}

proc save {args} {

    if {[llength $args] < 1} {
	error "Usage: save <object-type> [<options>] <args>"
    }
    return [eval [lindex $args 0] save [lrange $args 1 end]]
}


# solar::mu --
#
# Purpose:  Set or Display the Mu equation (trait value estimator)
#
#           Usually the covariate command is used to set this automatically,
#           but the mu command provides more advanced capabilities, such as
#           using log or sine functions.
# 
# Usage:    mu                        ; displays Mu equation
#           mu = mu + <expression>    ; add new terms to Mu equation
#           mu = <expression>         ; replaces Mu equation (SEE DISCUSSION!)
#           mu reset                  ; restores default Mu equation
#
#           <expression> may include mathematical operators (+-*/^), 
#           constants, parentheses, any mathematical functions defined in
#           the C programming language, any phenotypic variables included
#           in the analysis, sex, and for any variable "var" x_var (the
#           sample mean), min_var (the sample minimum), and max_var
#           (the sample maximum).  Parameters whose names include
#           any erstwhile operators including parentheses, *, or ^ must
#           be enclosed in angle brackets <> to prevent being parsed as
#           functions; note this always happens for bivariate models, or when
#           there are interaction covariates such as age*sex, or squared
#           covariates such as age^2.  For bivariate models, you can also
#           include "t1" and "t2": t1 is 1 if the mu is being evaluated
#           for the first trait, and 0 otherwise, and t2 has this behavior
#           for the second trait.  All variables included in the mu will be
#           required in the sample.
#
#           Also it is possible for the mu to include inequality operators
#           (such as >=) and the "print" function (for debugging purposes).
#           In these regards, the mu expression is like the omega expression.
#           See "help omega" for more about inequalities and print, and
#           a complete listing of the mathematical functions available.
#           
# Discussion:
#
#  The "mu" embodies the estimation of the trait value for any individual
#  based on the sample mean and their covariate values.  It does not
#  normally include genetic effects.  The difference from this estimation
#  and the actual value is used to determine genetic and other
#  intra-individual effects.  Thus, "mu" is evaluated in the context of
#  of a single individual, and NOT a pair of individuals as with "omega".
#
#  You can get examples of many possible "mu" commands by using the
#  mu command to display the current mu equation for different
#  models.  For example:
#
#  solar> model new
#  solar> trait q1
#  solar> covar age
#  solar> mu
#  mu = \{Mean+bage*(age-x_age)\}
#
#  First notice that the entire body of this default mu equation is
#  delimited by \{ and \} characters.  This is the portion which is
#  automatically generated by SOLAR and will be changed automatically
#  if your covariates are changed.  You should not normally edit this
#  portion of the mu.  If you need to change the mu, you can either
#  augment this portion with an additional expression, or replace the
#  mu altogether with a new expression, in which case you must leave
#  out the \{ and \} delimiters.  If you replace the mu altogether with
#  a new expression, you are then responsible for including terms for
#  covariates (if any) and it is not necessary to use the "covariate"
#  command at all.
#
#  The Mean and bage terms refer to parameters in the model, the age term
#  refers to a data variable "age" found in the phenotypes file, and the
#  term x_age refers to the average age of all individuals in this sample.
#  You may include similar terms in any new mu expression.
#
#  Adding To The mu
#
#  You can add additional terms either by appending them onto the mu shown
#  by the mu command (using terminal cut and paste makes this convenient)
#  or using the "mu = mu + ..." shorthand.  For example, using the
#  shorthand, you could add a new term for log(weight-100) as follows:
#
#  solar> mu = mu + log(weight-100)
#
#  OR by entering the following:
#
#  solar> mu = \{Mean+bage*(age-x_age)\} + log(weight-100)
#
#  in either case, the result would be the same:
#
#  solar> mu
#  mu = \{Mean+bage*(age-x_age)\} + log(weight-100)
#
#  If you then added another covariate, that would be included automatically
#  in the default portion of the mu:
#
#  solar> covar sex
#  solar> mu
#  mu = \{Mean+bage*(age-x_age)+bsex*Female\} + log(weight-100)
#
#  Notice here that the variable "Female" changes according to the sex.
#  It is "0" for male and "1" for female.
#
#  Replacing the Mu
#
#  You can also replace the Mu altogether, removing the "default portion."
#  If you remove the specially delimited "default portion" from the mu,
#  your covariate commands will have no effect on the mu, and you will
#  either have to write the beta parameters into the mu yourself or
#  remove the covariates altogether.  All phenotypic variables you
#  write into the model will be required for all individuals to be
#  included in the sample.
#
#  Continuing our example:
#
#  solar> covariate delete_all
#  solar> mu
#  mu = \{Mean]}
#  solar> mu = Mean + log(weight-min_weight)
#  solar> mu
#  mu =  Mean + log(weight-min_weight)
#
#  The Mu can be as elaborate as you like, including any mathematical
#  functions defined in the "C" programming language.  It need not include
#  the "Mean" parameter (in fact you do not even need a Mean parameter in
#  SOLAR anymore).
#
#  If you removed the default mu by mistake and need to restore it,
#  use the "mu reset" command.
#
#  Bivariate Mu
#
#  solar> model new
#  solar> trait q1 q2
#  solar> covar sex
#  solar> mu
# mu = \{t1*(<Mean(q1)>+<bsex(q1)>*Female) + t2*(<Mean(q2)>+<bsex(q2)>*Female)\}
#
#  Notice that the mu for this bivariate model has separte terms for the first
#  and second traits, which are identified by "t1" and "t2".  (The variable
#  "t1" is true if the first trait is being estimated, and false if the
#  second trait is being estimated.  If you replace the mu, any terms not
#  multiplied by "t1" or "t2" will be applied to the estimation of both
#  traits, and you may have as many (or as few) t1 and/or t2 terms as you
#  need.
#
# Additional Notes:
#
#  (1) Use the "mu = mu + <expression>" as described above instead of
#      the now obsolescent "mu = <expression> + mu" to add to the mu.
#      Also, you may notice that square brackets are no longer used
#      to delimit the default mu.  They did not work as had been intended.
#      The default portion of the mu is now delimited by \{ and \} which
#      may be included in a user specified mu.  Everything within the
#      delimiters is maintained by SOLAR and editing this portion will
#      have no effect.  It is simply displayed for informational purposes.
#      If the mu is defaulted, models will be saved with a mu "comment"
#      for informational purposes only; the actual mu is determined by
#      the covariates.
#
#  (2) As terms in the mu equation, you may use any constant, any
#      parameter, Sex, Mean, and any Phenotypic variable.  There are
#      also predefined terms for any phenotype named 'var': x_var
#      (the sample mean), min_var (the sample min), and max_var (the
#      sample max).  Any math operator (+,-,*,/) and function defined 
#      in the C programming language may be used.  Also, the ^ character
#      may be used to indicate exponentiation.
#
#  (3) Parameter names which include * or ^ should be enclosed in
#      <> angle brackets to prevent the names from being interpreted
#      as multiplication and/or exponentiation expressions:
#
#           mu = Mean + <bage*sex>*(age-x_age)*Female
#
#  (4) The default mu expression will display all variables as being
#      adjusted to their mean (e.g. age-x_age).  However, during
#      maximization, if a variable is found to be binary, the
#      variable is adjusted to its minimum (e.g. diabet-min_diabet)
#      instead.  This will be reflected after the first maximization.
#      User-created mu equations must always correctly specify
#      either the mean (x_) or min (min_) variable as required.
#-

# solar::option --
#
# Purpose:  Set or read the value of model-specific options.
# 
# Usage:    option <option name> <value>    ; sets option value
#           option <option name>            ; reads option value
#           option                          ; shows all option values
#
# Notes:    ibd-specific options are set by ibdoption.
#
#           Most options control fairly obscure aspects of SOLAR operation 
#           and are not normally changed directly by SOLAR users.  Many are 
#           automatically controlled by other SOLAR commands during
#           normal operation.
#
#           Model-specific options are stored in saved model files.
#           Starting a new 