unit csaread;
interface
// Extract vital details from the Siemens CSA header that is contained within a DICOM file.
//   This is DICOM group:element (0029:1010) [CSA Image Header Info]
//   These values are crucial converting 2D mosaics to 3D images and computing DTI vectors
//see http://nipy.sourceforge.net/nibabel/dicom/siemens_csa.html
//    This is a port of John Ashburners' spm_dicom_headers.m Matlab code
uses SysUtils, dialogsx, define_types;

type
  TBytearray =  array of byte;
  TCSA = record
     Slices,MosaicX,MosaicY: longword;
     Bvalue,DTIv1,DTIv2,DTIv3, SliceNormV1, SliceNormV2,SliceNormV3: double;
   end;

function DecodeCSA2 (lFilename: string; lCSAImageHeaderInfoPos, lCSAImageHeaderInfoSz: integer; var lCSA: TCSA): boolean;

function GetCSAImageHeaderInfoDTI (lFilename: string; lStart,lLength: integer; var lBval: integer; var ldti1,ldti2,ldti3: double): boolean;
function GetCSAImageHeaderInfo (lFilename: string; lStart,lLength: integer; var lMosaicSlices,lMosaicX,lMosaicY: integer; var lv1,lv2,lv3: double): boolean;


implementation

function GetCSAImageHeaderInfoDTI (lFilename: string; lStart,lLength: integer; var lBval: integer; var ldti1,ldti2,ldti3: double): boolean;
var
   lCSA: TCSA;
begin
     //lBval := -1;//imposibble - read error
     result := DecodeCSA2 (lFilename, lStart,lLength, lCSA);
     if not result then exit;
     lBval := round(lCSA.bvalue);
     ldti1 := lCSA.DTIv1;
     ldti2 := lCSA.DTIv2;
     ldti3 := lCSA.DTIv3;
end;

function GetCSAImageHeaderInfo (lFilename: string; lStart,lLength: integer; var lMosaicSlices,lMosaicX,lMosaicY: integer; var lv1,lv2,lv3: double): boolean;
var
   lCSA: TCSA;
begin
     //lMosaicSlices := -1;//imposibble - read error
     result := DecodeCSA2 (lFilename, lStart,lLength, lCSA);
     if not result then exit;
     lMosaicSlices := lCSA.Slices;
     lMosaicX := lCSA.MosaicX;
     lMosaicY := lCSA.MosaicY;
     lv1 := lCSA.SliceNormV1;
     lv2 := lCSA.SliceNormV2;
     lv3 := lCSA.SliceNormV3;  //5/5/2013
end;

function DecodeCSA2 (lFilename: string; lCSAImageHeaderInfoPos, lCSAImageHeaderInfoSz: integer; var lCSA: TCSA): boolean;
//provided with DICOM file as well as location and size of CSA header, this code returns the Siemens CSA header information

const
  kMaxItem = 4; // if you only need first 3 values, set to 4 so if an item has 6 values the final ones will overwrite 4th item
type
  TCSAtag = record
     name : string[64];
     vm: longint;
     vr123: string[3];
     vr4: string[1];
     syngodt ,nitems,xx : longint;
   end;
  TCSAitem = record
     xx1, xx2_Len, xx3_77, xx4: longint; // [ x L 77 x] L is length
     value: string;
   end;
var
    lFile    : File;
    lVers: string;
    lData : array of byte;
    lnTag,lPos,lI,lT,lIbound: integer;
    lTag :  TCSAtag;
    lItem : array [1..kMaxItem] of TCSAitem;
function SafeStr2Num (lStr: string): boolean; //for some reason, many fMRI images have bvalue = 'X1_01_001
var
  lP,lL: integer;
begin
  result := false;
  lL := length(lStr);
  if lL < 1 then exit;
  for lP := 1 to lL do
    if not(lStr[lP] in ['+','-','0'..'9','.','e','E']) then
      exit;
  result := true;
end;//nested func SafeStr2Num
function RightStr2Num (lStr: string): integer; //e.g. Siemens AcquisitionMatrixText "104p*96" -> "96"
var
  lL: integer;
  lS: string;
  lDone: boolean;
begin
  result := -1;
  lS := '';
  lL := length(lStr);
  if lL < 1 then exit;
  lDone := false;
  while (lL >= 1) and (not lDone) do begin
    if (lStr[lL] in ['+','-','0'..'9','.','e','E']) then
      lS := lStr[lL]+lS
    else if lS <> '' then
      lDone := true;
    dec(lL);
  end;
  if lS = '' then exit;
  result := strtoint(lS);
end; //nested func RightStr2Num
function LeftStr2Num (lStr: string): integer; //e.g. Siemens AcquisitionMatrixText "104p*96" -> "104"
var
  lP,lL: integer;
  lS: string;
  lDone: boolean;
begin
  result := -1;
  lS := '';
  lL := length(lStr);
  if lL < 1 then exit;
  lP := 1;
  lDone := false;
  while (lP <= lL) and (not lDone) do begin
    if (lStr[lP] in ['+','-','0'..'9','.','e','E']) then
      lS := lS + lStr[lP]
    else if lS <> '' then
      lDone := true;
    inc(lP);
  end;
  if lS = '' then exit;
  result := strtoint(lS);
end; //nested func LeftStr2Num
function freadStr(len: integer): string;
var
  i: integer;
begin
  if (len+lPos) >= lCSAImageHeaderInfoSz then
    Raise Exception.CreateFmt('csaread: corrupt file ', [lFilename]);
  result := '';
  i := 0;
  while (i < len) and (lData[i+lPos] <> 0) and (lData[i+lPos] <> $20)  do begin
      result := result + chr(lData[i+lPos]);
      inc(i);
  end;
  lPos := lPos + len;
end; //nested func freadStr
function freaduint32: longword; overload; //uint32
begin
  if (4+lPos) >= lCSAImageHeaderInfoSz then
    Raise Exception.CreateFmt('csaread: corrupt file ', [lFilename]);
  result := (lData[lPos+3] shl 24)+(lData[lPos+2] shl 16)+(lData[lPos+1] shl 8)+lData[lPos];
  lPos := lPos + 4;
end;//nested func freaduint32
function freadint32: longint; overload; //uint32
begin
  if (4+lPos) >= lCSAImageHeaderInfoSz then
    Raise Exception.CreateFmt('csaread: corrupt file ', [lFilename]);
  result := (lData[lPos+3] shl 24)+(lData[lPos+2] shl 16)+(lData[lPos+1] shl 8)+lData[lPos];
  lPos := lPos + 4;
end;//nested func freadint32
function freadTag: TCSAtag;
begin
    result.name := freadStr(64);
    result.vm:= freadint32;
    result.vr123:= freadStr(3);
    result.vr4:= freadStr(1);
    result.syngodt := freadint32;
    result.nitems := freadint32;
    result.xx := freadint32;
end;//nested func freadTag
function freadItem: TCSAitem;
begin
  result.xx1:= freadint32;
  result.xx2_Len:= freadint32;
  result.xx3_77:= freadint32;
  result.xx4:= freadint32;
  result.value := freadStr(result.xx2_len);
  lPos := lPos + ((4-(result.xx2_len) mod 4  )mod 4) ;
end;//nested func freadItem
begin  //main function DecodeCSA2
   result := false;
   lCSA.Bvalue := -1;
   if (lCSAImageHeaderInfoSz < 1) then
    exit;
   if FSize(lFilename) <= (lCSAImageHeaderInfoPos+lCSAImageHeaderInfoSz) then
        exit;
   if lCSAImageHeaderInfoSz < 118 then
    exit; //Too short to be a CSA header - Perhaps Philips or GE is using this tag
   setlength(lData, lCSAImageHeaderInfoSz);
   lPos := 0;
   FileMode := fmOpenRead;
   AssignFile(lFile, lFilename);
   Reset(lFile, 1);   // Now we define one record as 1 byte
   Seek(lFile, lCSAImageHeaderInfoPos);    // Records start from 0
   BlockRead(lFile, lData[0], lCSAImageHeaderInfoSz);
   CloseFile(lFile);
   lVers := freadStr(4);
   if lVers = 'SV10' then begin
      //read header
      lPos := lPos + 4; //skip 8 bytes of data, spm_dicom_headers refers to these as unused1 and unused2
      lnTag := freaduint32;
      if (lnTag < 1) or (lnTag > 1024) then begin
        showmsg('Error reading CSA header');
        exit;
      end;
      if (lData[lPos] <> 77) then showmsg('warning: strange CSA2 header');
      lPos := lPos + 4; // skip the four bytes 77 00 00 00
      //read tags
      for lT := 1 to lnTag do begin
        lTag := freadTag;
        if lTag.nitems > 0 then begin
          for lI := 1 to lTag.nitems do begin //read items
            if lI > kMaxItem then
              lIbound := kMaxItem //out of range
            else
              lIbound := lI;
            lItem[lIbound] := freadItem;
          end; //for each item
          if (lTag.name = 'NumberOfImagesInMosaic') then
            lCSA.Slices := round(strtofloat(lItem[1].value)) ;
          if (lTag.name = 'AcquisitionMatrixText') then begin  //'96p*96 -> X= 96 and Y= 96
            lCSA.MosaicX := LeftStr2Num(lItem[1].value);
            lCSA.MosaicY := RightStr2Num(lItem[1].value);
          end;
          if (lTag.name = 'B_value') and (SafeStr2Num(lItem[1].value)) then
            lCSA.Bvalue := strtofloat(lItem[1].value);
          if (lTag.name = 'DiffusionGradientDirection') and (SafeStr2Num(lItem[1].value)) and (SafeStr2Num(lItem[2].value)) and (SafeStr2Num(lItem[3].value))  then begin
            lCSA.DTIv1 := strtofloat(lItem[1].value);
            lCSA.DTIv2 := strtofloat(lItem[2].value);
            lCSA.DTIv3 := strtofloat(lItem[3].value);
          end;
          if (lTag.name = 'SliceNormalVector') and (SafeStr2Num(lItem[1].value)) and (SafeStr2Num(lItem[2].value)) and (SafeStr2Num(lItem[3].value))  then begin
            lCSA.SliceNormV1 := strtofloat(lItem[1].value);
            lCSA.SliceNormV2 := strtofloat(lItem[2].value);
            lCSA.SliceNormV3 := strtofloat(lItem[3].value);
          end;
        end;//at least one item
      end; //for each tag
      result := true;
      //showmsg('Success DecodeCSA2');
   end else begin
      msg('CSAread Warning: '+ lFilename +' at byte '+inttostr(lCSAImageHeaderInfoPos)+' reports version "'+lVers+'": only "SV10" format is supported: image is either corruprted, very old or new. See if a new version of this software is available.');
   end;
  lData := nil;
end;// func DecodeCSA2

end.
 
