/**
 * Java Image Science Toolkit (JIST)
 *
 * Image Analysis and Communications Laboratory &
 * Laboratory for Medical Image Computing &
 * The Johns Hopkins University
 * 
 * http://www.nitrc.org/projects/jist/
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 */
package edu.jhu.ece.iacl.jist.pipeline;

import gov.nih.mipav.model.scripting.ImageVariableTable;
import gov.nih.mipav.model.scripting.ParsedActionLine;
import gov.nih.mipav.model.scripting.Parser;
import gov.nih.mipav.model.scripting.ParserEngine;
import gov.nih.mipav.model.scripting.ParserException;
import gov.nih.mipav.model.scripting.ScriptRunner;
import gov.nih.mipav.model.scripting.ScriptableActionInterface;
import gov.nih.mipav.model.scripting.ScriptableActionLoader;
import gov.nih.mipav.model.scripting.VOITable;
import gov.nih.mipav.view.MipavUtil;
import gov.nih.mipav.view.Preferences;

import java.util.Vector;

// TODO: Auto-generated Javadoc
/**
 * Overload MIPAV's script runner.
 * 
 * @author Blake Lucas (bclucas@jhu.edu)
 */
public class MipavScriptRunner extends ScriptRunner {
	
	/**
	 * Parses and runs the commands contained within a given script file.
	 * 
	 * @param scriptFile The full path to the script file on disk which should be
	 * parsed.
	 * 
	 * @throws ParserException If a problem is encountered while parsing the script.
	 */
	public static final void runScript(String scriptFile) throws ParserException {
		// check that all externally-specified (not generated by the script
		// actions) images have been setup
		String[] requiredImages = Parser.getImageVarsUsedInScript(scriptFile);
		for (int i = 0; i < requiredImages.length; i++) {
			if (!ScriptRunner.getReference().getImageTable().isImageVariableSet(requiredImages[i])) {
				throw new ParserException(scriptFile, 0, "Required image not found: " + requiredImages[i]);
			}
		}
		ParserEngine parser = new ParserEngine(scriptFile, true);
		while (parser.hasMoreLinesToParse()) {
			ParsedActionLine parsedLine = parser.parseNextLine();
			if (parsedLine != null) {
				//FIXME Should not use Preferences.NON-STATIC-METHODS
				Preferences.debug("parser:\t\t" + parsedLine.convertToString() + "\n", Preferences.DEBUG_SCRIPTING);
				ScriptableActionInterface scriptAction = ScriptableActionLoader.getScriptableAction(parsedLine
						.getAction());
				//FIXME Should not use Preferences.NON-STATIC-METHODS				
				Preferences.debug("parser:\tRunning action:\t" + scriptAction.getClass().getName() + "\n",
						Preferences.DEBUG_SCRIPTING);
				try {
					scriptAction.scriptRun(parsedLine.getParameterTable());
				} catch (Exception e) {
					String message = "\n\n" + e.getClass().getName()
							+ "\n\n(see console or debugging window output for details)";
					ParserException exception = new ParserException(scriptFile, parser.getCurrentLineNumber(), message);
					exception.initCause(e);
					throw exception;
				}
			}
		}
		// if the -hide argument was specified, we require an Exit command
		// somewhere in the script. otherwise mipav
		// will appear to "hang"
		// Do NOT need to exit if MAPS is running plug-in
		/*
		 * if (!MipavViewUserInterface.getReference().isAppFrameVisible() &&
		 * !hasExitAction(scriptFile)) { new ActionExit().scriptRun(new
		 * ParameterTable()); }
		 */
	}

	/**
	 * Instantiates a new mipav script runner.
	 */
	public MipavScriptRunner() {
		super();
		ScriptRunner.singletonReference = this;
	}

	/**
	 * Execute a script, using a set of images.
	 * 
	 * @param file The path to the script file we want to run.
	 * @param imageNameList The list of the names of images to use while executing the
	 * script.
	 * @param voiPathList A list of VOI paths, in the order returned by
	 * Parser.getImageVarsUsedInScript().
	 * 
	 * @return <code>True</code> if execution of the script was successful,
	 * <code>false</code> otherwise.
	 */
	public /*synchronized*/ boolean runScript(String file, Vector imageNameList, Vector voiPathList) {
		if (isRunning()) {
			MipavUtil.displayError("A script is already being executed.");
			return false;
		}
		//FIXME Should not use Preferences.NON-STATIC-METHODS
		Preferences.debug("script runner:\tStarting script execution:\t" + file + "\n", Preferences.DEBUG_SCRIPTING);
		setRunning(true);
		scriptFile = file;
		imageTable = new ImageVariableTable();
		// ScriptThread thread;
		try {
			fillImageTable(imageNameList);
			voiTable = new VOITable(scriptFile, voiPathList);
			// thread = new ScriptThread(scriptFile);
			// thread.start();
			runScript(scriptFile);
			//FIXME Should not use Preferences.NON-STATIC-METHODS
			Preferences.debug("script runner:\tFinished script execution:\t" + scriptFile + "\n",
					Preferences.DEBUG_SCRIPTING);
			setRunning(false);
		} catch (ParserException pe) {
			handleParserException(pe);
			return false;
		}
		return true;
	}
}
