// Copyright 2018 The gVisor Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

//go:build linux
// +build linux

package memutil

import (
	"fmt"
	"unsafe"

	"golang.org/x/sys/unix"
)

// CreateMemFD creates a memfd file and returns the fd.
func CreateMemFD(name string, flags int) (int, error) {
	p, err := unix.BytePtrFromString(name)
	if err != nil {
		return -1, err
	}
	fd, _, e := unix.Syscall(unix.SYS_MEMFD_CREATE, uintptr(unsafe.Pointer(p)), uintptr(flags), 0)
	if e != 0 {
		if e == unix.ENOSYS {
			return -1, fmt.Errorf("memfd_create(2) is not implemented. Check that you have Linux 3.17 or higher")
		}
		return -1, e
	}
	return int(fd), nil
}
